package com.sonicsw.mf.framework.directory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.INamedPayload;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.config.impl.NameMapper;

public class SubscriptionRegistry
{
    private static final boolean DEBUG = false;
    
    private HashMap m_subscribers;

    private HashMap m_topics;

    private long m_subscriptionDuration;

    private Thread m_exiprator;

    private volatile boolean m_closed;

    private IPersistSubscribers m_storage;

    private LogicalSubscribersMap m_logicalMap;
    
    private static final String MF_RUNTIME_PREFIX = IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_RUNTIME_DIR + IMFDirectories.MF_DIR_SEPARATOR;

    // Unit tests
    public static void main(String[] args)
    throws Exception
    {
        SubscriptionRegistry registry = new SubscriptionRegistry(40, null);

        registry.subscribe("S1", "x");
        registry.subscribe("S2", "x");
        registry.subscribe("S3", "x");

        String E1 = "/a1/b1/l1";
        String E2 = "/a1/b1/l2";
        String E3 = "/a1/b2/l3";

        registry.addLogicalSubscriber(E1, "S1");
        registry.addLogicalSubscriber(E1, "S2");
        registry.addLogicalSubscriber(E2, "S2");
        registry.addLogicalSubscriber(E3, "S3");

        String oldEr = "/a1";
        String newEr = "/x1";

        System.out.println();
        System.out.println("Interested in " + oldEr + " to " + newEr + " renaming:");
        HashSet interestedSubs = registry.renameLogicalPath(oldEr, newEr);
        Iterator iterator = interestedSubs.iterator();
        while (iterator.hasNext())
        {
            System.out.println(iterator.next());
        }
    }

    SubscriptionRegistry(long subscriptionDuration, IPersistSubscribers storage)
    {
        m_closed = false;
        m_subscriptionDuration = subscriptionDuration;
        m_subscribers = new HashMap();
        m_topics = new HashMap();
        m_storage = storage;
        m_logicalMap = new LogicalSubscribersMap();

        m_exiprator = new Thread(DSComponent.GLOBAL_ID + " - Subscription Expirer")
        {
            @Override
            public void run()
            {
                while (!m_closed)
                {
                    try
                    {
                        Thread.sleep(m_subscriptionDuration);
                    }
                    catch (InterruptedException e)
                    {
                    }

                    // Expire old subscriptions. Expire also resets the current subscribers persistent list
                    if (!m_closed)
                    {
                        SubscriptionRegistry.this.expire();
                    }
                }
            }
        };
        m_exiprator.start();
    }

    synchronized void addLogicalSubscriber(String path, String subscriber)
    {
        m_logicalMap.addSubscriber(path, subscriber);
    }

    synchronized void deleteLogicalPath(String path)
    {
        m_logicalMap.removePath(path);
    }

    synchronized HashSet renameLogicalPath(String oldPath, String newPath)
    {
        HashSet interestedSubscribes = m_logicalMap.rename(oldPath, newPath);

        if (interestedSubscribes == null)
        {
            return new HashSet();
        }

        interestedSubscribes.retainAll(m_subscribers.keySet());
        
        return interestedSubscribes;
    }

    synchronized void close()
    {
        m_closed = true;
        m_exiprator.interrupt();
    }

    // Subscribe to a topic
    synchronized void subscribe(String subscriber, String topic)
    {
        if (subscribeInternal(subscriber, topic))
        {
            m_storage.addSubscriber(subscriber);
        }
    }

    // Subscribe to topics
    synchronized void subscribe(String subscriber, String[] topics)
    {
        boolean subscribersListChanged = false;
        for (int i = 0; i < topics.length; i++)
        {
            if (subscribeInternal(subscriber, topics[i]))
            {
                subscribersListChanged = true;
            }
        }

        if (subscribersListChanged)
        {
            m_storage.addSubscriber(subscriber);
        }
    }

    // Subscribe to a topic. returns true if m_subscribers changed
    private boolean subscribeInternal(String subscriber, String topic)
    {
        if (topic.startsWith(MF_RUNTIME_PREFIX))
        {
            return false;
        }
        
        long currentTime = System.currentTimeMillis();
        Subscription subscription = new Subscription(subscriber, currentTime);
        
        Object oldSubscription = m_subscribers.put(subscriber, subscription);
        
        HashMap topicSubscribers = (HashMap)m_topics.get(topic);
        if (topicSubscribers == null)
        {
            topicSubscribers = new HashMap();
            m_topics.put(topic, topicSubscribers);
        }
        
        topicSubscribers.remove(subscriber);
        topicSubscribers.put(subscriber, subscription);

        return oldSubscription == null;
    }

    // Get all the subscribes to a specific topic
    synchronized String[] getSubscribers(String topic)
    {
        HashMap topicSubscribers = (HashMap)m_topics.get(topic);
        if (topicSubscribers == null)
        {
            return IEmptyArray.EMPTY_STRING_ARRAY;
        }
        
        return (String[])topicSubscribers.keySet().toArray(IEmptyArray.EMPTY_STRING_ARRAY);
    }

    // Get all the subscribes (to any topic)
    synchronized String[] getSubscribers()
    {
        return (String[])m_subscribers.keySet().toArray(IEmptyArray.EMPTY_STRING_ARRAY);
    }

    // Expire subscribers that didn't renew on time
    synchronized void expire()
    {
        if (DEBUG)
        {
            System.out.println("SubscriptionRegistry.expire()");
        }
        
        long expiredSubscriptionTime = System.currentTimeMillis() - m_subscriptionDuration;
        
        // expire old topic subscriptions
        Iterator topicEntryIterator = m_topics.entrySet().iterator();
        while (topicEntryIterator.hasNext())
        {
            Map.Entry topicEntry = (Map.Entry)topicEntryIterator.next();
            HashMap subscriptions = (HashMap)topicEntry.getValue();
            
            Iterator subscriptionsEntryIterator = subscriptions.entrySet().iterator();
            while (subscriptionsEntryIterator.hasNext())
            {
                Map.Entry subscriptionEntry = (Map.Entry)subscriptionsEntryIterator.next();
                Subscription subscription = (Subscription)subscriptionEntry.getValue();
                
                if (subscription.m_subscriptionTime < expiredSubscriptionTime)
                {
                    if (DEBUG)
                    {
                        System.out.println("Removing subscription for topic [" + topicEntry.getKey() + "] by subscriber [" + subscription.m_subscriber + "]");
                    }
                    subscriptionsEntryIterator.remove();
                }
            }

            if (subscriptions.isEmpty())
            {
                if (DEBUG)
                {
                    System.out.println("Removing subscriptions map for topic [" + topicEntry.getKey() + "]");
                }
                topicEntryIterator.remove();
            }
        }
        
        ArrayList expiredSubscribers = new ArrayList();

        // expire old subscribers
        Iterator subscriptionEntryIterator = m_subscribers.entrySet().iterator();
        while (subscriptionEntryIterator.hasNext())
        {
            Map.Entry subscriptionEntry = (Map.Entry)subscriptionEntryIterator.next();
            Subscription subscription = (Subscription)subscriptionEntry.getValue();
            if (subscription.m_subscriptionTime < expiredSubscriptionTime)
            {
                if (DEBUG)
                {
                    System.out.println("Removing subscriber [" + subscription.m_subscriber + "]");
                }
                subscriptionEntryIterator.remove();
                expiredSubscribers.add(subscriptionEntry.getKey());
            }
        }
        
        // now update the storage to remove the expired subscribers
        if (!expiredSubscribers.isEmpty())
        {
            m_storage.removeSubscribers((String[])expiredSubscribers.toArray(IEmptyArray.EMPTY_STRING_ARRAY));
        }
    }

    private class Subscription
    {
        private String m_subscriber;

        private long m_subscriptionTime;

        Subscription(String subscriber, long time)
        {
            this.m_subscriber = subscriber;
            this.m_subscriptionTime = time;
        }
    }

    private class LogicalSubscribersMap
    {
        private NameMapper m_map;

        LogicalSubscribersMap()
        {
            m_map = new NameMapper();
        }

        void addSubscriber(String path, String subscriber)
        {
            EntityName pathE = getEntityName(path);
            HashSet subscribers = (HashSet)m_map.get(pathE);
            if (subscribers == null)
            {
                subscribers = new HashSet();
            }
            subscribers.add(subscriber);
            m_map.set(pathE, subscribers);
        }

        void removePath(String path)
        {
            m_map.remove(getEntityName(path));
        }

        HashSet rename(String oldPath, String newPath)
        {
            INamedPayload[] list = m_map.rename(getEntityName(oldPath), getEntityName(newPath));

            if (list == null)
            {
                return null;
            }

            HashSet allSubscribersSet = new HashSet();

            for (int i = 0; i < list.length; i++)
            {
                allSubscribersSet.addAll((HashSet)list[i].getPayload());
            }
            return allSubscribersSet;
        }

        private EntityName getEntityName(String path)
        {
            try
            {
                return new EntityName(path);
            }
            catch (ConfigException e)
            {
                throw new RuntimeException(e.toString());
            }
        }
    }
}
