/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.directory.impl;

import java.util.HashMap;

import com.sonicsw.security.pass.client.IPasswordUser;
import com.sonicsw.security.pass.mf.ConnectionException;
import com.sonicsw.security.pass.mf.IEvent;
import com.sonicsw.security.pass.mf.IEventDisconnected;
import com.sonicsw.security.pass.mf.IEventGroupsAdded;
import com.sonicsw.security.pass.mf.IEventGroupsDeleted;
import com.sonicsw.security.pass.mf.IEventGroupsModified;
import com.sonicsw.security.pass.mf.IEventUsersAdded;
import com.sonicsw.security.pass.mf.IEventUsersDeleted;
import com.sonicsw.security.pass.mf.IEventUsersModified;
import com.sonicsw.security.pass.mf.IGroup;
import com.sonicsw.security.pass.mf.IManagement;
import com.sonicsw.security.pass.mf.INotificationListener;
import com.sonicsw.security.pass.mf.InvalidConfigurationException;

import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.directory.IDebuggingMasks;
import com.sonicsw.mf.framework.directory.ILogger;

final class AuthSource implements IDebuggingMasks
{
    public static final long  TRY_AGAIN_INTERVAL = 10000;

    private IManagement m_managementInstance;
    private String m_domainName;
    private ILogger m_logger;
    private boolean m_isClosing = false;
    private HashMap m_connectionParameters;
    private Connector m_connector;
    private IAuthListener m_internalListener;

    AuthSource(IManagement managementInstance, String domainName, HashMap connectionParameters, ILogger logger) throws Throwable
    {
        m_internalListener = null;
        m_domainName = domainName;
        m_managementInstance = managementInstance;
        m_logger = logger;
        m_connectionParameters = connectionParameters;
    }

    void connect() throws Throwable
    {
        boolean alreadyConnected = false;
        try
        {
            m_logger.trace(TRACE_EXTERNAL_AUTH, "Connecting to external source of " + m_domainName);

            m_managementInstance.connect(m_connectionParameters);

            m_logger.logMessage("Connected to the authentication source of domain \"" + m_domainName + "\"", Level.TRACE);

            m_logger.trace(TRACE_EXTERNAL_AUTH, "Connected to external source of " + m_domainName);

            alreadyConnected = true;
        }
        catch (Throwable throwable)
        {
            m_logger.trace(TRACE_EXTERNAL_AUTH, "SPI connect failure, trace follows... ", throwable);
            // No point in retrying if InvalidConfigurationException (the connection parameters are wrong) or
            // if there is a bug (only ConnectionException should normally be thrown).
            if ((throwable instanceof InvalidConfigurationException) || !(throwable instanceof ConnectionException))
            {
                if (throwable instanceof ConnectionException)
                {
                    m_logger.logMessage("Failed to connect to the authentication source of domain \"" + m_domainName + "\", trace follows...", throwable, Level.WARNING);
                }
                else
                {
                    m_logger.logMessage("Failed to connect to the authentication source of domain \"" + m_domainName + "\", trace follows...", throwable, Level.WARNING);
                }
                throw throwable;
            }
            else
            {
                m_logger.logMessage("Failed to connect to the authentication source of domain \"" + m_domainName + "\": " + throwable.getMessage() + " - retrying...", Level.WARNING);
            }
        }

        m_connector = new Connector (alreadyConnected,  m_domainName, m_logger);
        m_connector.start();

    }

    IPasswordUser[] getUsers(long timeout) throws TimeoutException
    {
        m_logger.trace(TRACE_EXTERNAL_AUTH, "Getting users for " + m_domainName);

        IPasswordUser[] users = (IPasswordUser[])getPrincipals(timeout, true);

        m_logger.trace(TRACE_EXTERNAL_AUTH, "Got " + users.length + " users.");

        return users;
    }

    IGroup[] getGroups(long timeout) throws TimeoutException
    {
        m_logger.trace(TRACE_EXTERNAL_AUTH, "Getting groups for " + m_domainName);

        IGroup[] groups = (IGroup[])getPrincipals(timeout, false);

        m_logger.trace(TRACE_EXTERNAL_AUTH, "Got " + groups.length + " groups.");

        return groups;
    }

    Object getPrincipals(long timeout, boolean getUsers) throws TimeoutException
    {
        short tryAgain = 2;
        while (tryAgain > 0)
        {
            if (!(m_connector.isConnected()))
            {
                tryAgain--;
                if (!m_connector.waitUntilConnected(timeout))
                {
                    throw new TimeoutException();
                }

                if (m_isClosing)
                {
                    throw new TimeoutException();
                }
            }

            if (m_isClosing)
            {
                throw new TimeoutException();
            }

            try
            {
                if (getUsers)
                {
                    return m_managementInstance.getUsers();
                }
                else
                {
                    return m_managementInstance.getGroups();
                }
            }
            catch (Throwable throwable)
            {
                m_logger.trace(TRACE_EXTERNAL_AUTH, "SPI failure, trace follows...", throwable);
                if (m_isClosing)
                {
                    throw new TimeoutException();
                }

                if (!(throwable instanceof ConnectionException))
                {
                    m_logger.logMessage("SPI failure, trace follows...", throwable, Level.WARNING);
                }

                tryAgain--;
                if (m_connector.isConnected())
                {
                    m_connector.disconnected(throwable.getMessage(), null);
                }
            }
        }

        throw new TimeoutException();
    }

    private void disconnected(String message, Integer errorNum)
    {
        m_connector.disconnected(message, errorNum);
    }

    private void reconnected()
    {
        if (m_internalListener != null)
        {
            m_internalListener.connectionRecovered();
        }
    }

    void disconnect()
    {
        m_isClosing = true;
        m_connector.close();
        try
        {
            m_logger.trace(TRACE_EXTERNAL_AUTH, "Disconnecting from external source of " + m_domainName);

            m_managementInstance.disconnect();

            m_logger.trace(TRACE_EXTERNAL_AUTH, "Disconnected from external source of " + m_domainName);
        }
        catch (Throwable throwable)
        {
            m_logger.trace(TRACE_EXTERNAL_AUTH, "SPI disconnect failure, trace follows..." + throwable);
            throwable.printStackTrace();
        }
    }

    boolean isConnected()
    {
        return m_connector.isConnected();
    }

    boolean registerListener(IAuthListener internalListener)
    {
         m_internalListener = internalListener;

         m_logger.trace(TRACE_EXTERNAL_AUTH, "Setting listener to external source of " + m_domainName);

         boolean result = m_managementInstance.setNotificationListener(new ExternalListener(m_internalListener));

         m_logger.trace(TRACE_EXTERNAL_AUTH, "Listener set - " + result);

         return result;
    }

    private final class ExternalListener implements INotificationListener
    {
        private IAuthListener m_internalListener;

        ExternalListener(IAuthListener internalListener)
        {
            m_internalListener = internalListener;
        }

        @Override
        public void onNotification(IEvent event)
        {
            if (event == null)
            {
                m_logger.logMessage("INotificationListener got null event", Level.WARNING);
                return;
            }

            m_logger.trace(TRACE_EXTERNAL_AUTH, "Got " + m_domainName + " event: " + event.getClass().getName());

            if (event instanceof IEventUsersAdded)
            {
                m_internalListener.updateUsers(((IEventUsersAdded)event).getUsersAdded());
            }
            else if (event instanceof IEventUsersModified)
            {
                m_internalListener.updateUsers(((IEventUsersModified)event).getUsersModified());
            }
            else if (event instanceof IEventUsersDeleted)
            {
                IPasswordUser[] deletedUsers = ((IEventUsersDeleted)event).getUsersDeleted();
                String[] userNames = new String[deletedUsers.length];
                for (int i = 0; i < deletedUsers.length; i++)
                {
                    userNames[i] = deletedUsers[i].getName();
                }
                m_internalListener.deletePrincipals(userNames, true);
            }
            else if (event instanceof IEventGroupsAdded)
            {
                m_internalListener.updateGroups(((IEventGroupsAdded)event).getGroupsAdded());
            }
            else if (event instanceof IEventGroupsModified)
            {
                m_internalListener.updateGroups(((IEventGroupsModified)event).getGroupsModified());
            }
            else if (event instanceof IEventGroupsDeleted)
            {
                IGroup[] deletedGroups = ((IEventGroupsDeleted)event).getGroupsDeleted();
                String[] groupNames = new String[deletedGroups.length];
                for (int i = 0; i < deletedGroups.length; i++)
                {
                    groupNames[i] = deletedGroups[i].getName();
                }
                m_internalListener.deletePrincipals(groupNames, false);
            }
            else if (event instanceof IEventDisconnected)
            {
                IEventDisconnected discEvent = (IEventDisconnected)event;
                AuthSource.this.disconnected(discEvent.getErrorMessage(), new Integer(discEvent.getErrorCode()));
            }

        }
    }

    private final class Connector extends Thread
    {
        boolean m_connected;
        boolean m_closed;
        String m_domainName;
        ILogger m_logger;

        Connector (boolean alreadyConnected, String domainName, ILogger logger)
        {
            super("AuthSource.Connector for domain " + domainName);
            m_connected = alreadyConnected;
            m_domainName = domainName;
            m_closed = false;
            m_logger = logger;
        }

        private synchronized boolean finished()
        {
            if (m_closed)
            {
                return true;
            }

            while (m_connected)
            {
                if (m_closed)
                {
                    return true;
                }
                try
                {
                    wait();
                }
                catch (InterruptedException e){}

            }
            return false;
        }

        synchronized void close()
        {
            m_closed = true;
            notifyAll();
        }

        synchronized void disconnected(String message, Integer errorNum)
        {
            String printMessage = (message == null || message.length() == 0) ? "." : (": " + message);
            if (errorNum != null)
            {
                printMessage += " Error number: " + errorNum.toString();
            }
            m_logger.logMessage("Disconnected from authentication source of domain \"" + m_domainName + "\"" + printMessage, Level.INFO);
            m_connected = false;
            notifyAll();
        }

        boolean isConnected()
        {
            return m_connected;
        }

        // Wait until connected or closed.
        // Return true if connected.
        synchronized boolean waitUntilConnected(long timeout)
        {
            if (m_closed)
            {
                return false;
            }

            if (m_connected)
            {
                return true;
            }

            try
            {
                long startTime = System.currentTimeMillis();
                if (timeout <= 0)
                {
                    while (!m_connected && !m_closed)
                    {
                        wait();
                    }
                }
                else
                {
                    long waitTime = timeout;
                    while (!m_connected && !m_closed && waitTime > 0)
                    {
                        wait(waitTime);
                        waitTime = timeout - (System.currentTimeMillis() - startTime);
                    }
                }
            }
            catch (InterruptedException e)
            {
                return m_connected;
            }
            
            return m_connected;
        }

        @Override
        public void run()
        {
            while (true)
            {
                 synchronized (this)
                 {
                      if (finished())
                    {
                        return;
                    }

                      try
                      {
                          m_logger.trace(TRACE_EXTERNAL_AUTH, "Connecting to external source of " + m_domainName);

                          m_managementInstance.connect(m_connectionParameters);

                          m_logger.trace(TRACE_EXTERNAL_AUTH, "Connected to external source of " + m_domainName);

                          m_connected = true;
                          m_logger.logMessage("Connected to the authentication source of domain \"" + m_domainName + "\"", Level.INFO);
                          AuthSource.this.reconnected();
                          notifyAll();
                      }
                      catch (Throwable throwable)
                      {
                          m_logger.trace(TRACE_EXTERNAL_AUTH, "SPI connect failure, trace follows... ", throwable);
                          try
                          {
                              wait(TRY_AGAIN_INTERVAL);
                          }
                          catch (InterruptedException e){}
                      }
                 }
            }
        }
    }
}
