package com.sonicsw.mf.framework.directory.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;

import com.sonicsw.mx.util.ServiceMaintainer;
import com.sonicsw.mx.util.ServiceMaintenance;
import com.sonicsw.security.pass.client.IPasswordUser;
import com.sonicsw.security.pass.mf.IGroup;
import com.sonicsw.security.pass.mf.IManagement;
import com.sonicsw.util.debug.Debug;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.directory.IDebuggingMasks;
import com.sonicsw.mf.framework.directory.IDirectoryService;
import com.sonicsw.mf.framework.directory.ILogger;
import com.sonicsw.mf.mgmtapi.config.constants.IAuthenticationDomainConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IAuthenticationGroupConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IAuthenticationUserConstants;

// Manages security domains of this DS
final class AuthenticationConfigManager implements ILogger, IDebuggingMasks
{
    public static final String MF_REFRESH_ELEMENT = "_MFRefreshTime";
    public static final char[] USER_RESTRICTED_CHARS = {'*','#','$','/','\\'};
    public static final char[] GROUP_RESTRICTED_CHARS = {'*','#','$','/','\\','.','=',','};

    private static final long INITIALIZE_DOMAIN_TIMEOUT = 30000;
    private static final long ASYNC_UPDATE_TIMEOUT = 60000;
    private static final int DEFAULT_UPDATE_FREQUENCY_SECS = 600; // 10 minutes in seconds
    private static final int DEFAULT_REFRESH_FREQUENCY_SECS = 2400; // 40 minutes (1 hour guaranteed if all is ok)

    static final String DOMAINS_DIRECTORY = "/authentication/domains";
    static final String DOMAIN_DESCRIPTOR = "_MFDomainDescriptor";

    static final String DOMAIN_DESC_TYPE  = "MF_AUTHENTICATION_DOMAIN";
    private static final String EXTERNAL_USER_TYPE  = "MF_AUTHENTICATION_USER";
    private static final String EXTERNAL_GROUP_TYPE  = "MF_AUTHENTICATION_GROUP";
    private static final String REFRESH_TYPE  = "MF_REFRESH_TIME";
    private static final String SPI_DESC_TYPE  = "MF_MANAGEMENT_SPI";
    private static final String USER_GROUP_MEMBER_TYPE = "user";
    private static final String MEMBER_TYPE = "MEMBER_TYPE";
    private static final String MEMBER_NAME = "MEMBER_NAME";

    // MF_AUTHENTICATION_DOMAIN attributes
    private static final String DOMAIN_NAME_ATT = "DOMAIN_NAME";
    static final String DOMAIN_MGMT_SPI_ATT = "MGMT_SPI";
    static final String DOMAIN_EXTERNAL_ATT = "EXTERNAL";
    private static final String DOMAIN_CONNECTION_PARAM_ATT = "MGMT_SPI_CONNECTION_PARAMETERS";

    // EXTERNAL_USER_TYPE attributes
    private static final String EXTERNAL_USER_NAME_ATT = "USER_NAME";
    private static final String EXTERNAL_USER_PASSWORD_ATT = "PASSWORD";        //NOSONAR field change is not required.

    // EXTERNAL_GROUP_TYPE attributes
    private static final String EXTERNAL_GROUP_NAME_ATT = "GROUP_NAME";
    private static final String EXTERNAL_GROUP_MEMBERS_ATT = "GROUP_MEMBERS";


    // MF_PASS_MGMT_SPI attributes
    private static final String SPI_SOURCE_TYPE_ATT = "MGMT_SPI_NAME";
    private static final String SPI_CLASS_NAME_ATT = "CLASS_NAME";
    private static final String SPI_CLASSPATH_ATT = "CLASSPATH";

   // MF_REFRESH_TIME attributes
    private static final String REFRESH_TIME_ATT = "RECENT_REFRESH_TIME";


    //External authentication info
    static final String EXTERNAL_DIR = "external";
    static final String EXTERNAL_USERS_DIR = EXTERNAL_DIR + IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_USERS_DIR;
    static final String EXTERNAL_GROUPS_DIR = EXTERNAL_DIR + IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_GROUPS_DIR;


    private DirectoryService m_ds;
    private DSUpdater m_dsUpdater;
    private HashMap m_domains;
    private HashMap m_spis;
    private ILogger m_logger;
    private IDirIdentity[] m_domainIDs;
    private HashMap m_forbiddenDirs;
    private ServiceMaintainer m_DSManagedUpdateThread;
    private boolean m_isStopped = true;
    private boolean m_isClosing = false;
    private long m_updateFrequency;
    private long m_refreshFrequency;

    AuthenticationConfigManager(DirectoryService ds, ILogger logger, boolean isRestrictedBackupDS) throws DirectoryServiceException
    {
        m_ds = ds;
        m_dsUpdater = new DSUpdater();
        m_logger = logger;
        
        int updateFrequencyInSeconds = Integer.getInteger(IDirectoryService.AUTH_UPDATE_FREQUENCY_PROPERTY, DEFAULT_UPDATE_FREQUENCY_SECS);
        int refreshFrequencyInSeconds = Integer.getInteger(IDirectoryService.AUTH_REFRESH_FREQUENCY_PROPERTY, DEFAULT_REFRESH_FREQUENCY_SECS);          
        m_ds.logMessage("Setting external security domain update frequency to " + updateFrequencyInSeconds + " seconds", Level.INFO);
        m_ds.logMessage("Setting external security domain refresh frequency to " + refreshFrequencyInSeconds + " seconds", Level.INFO);

        m_updateFrequency = updateFrequencyInSeconds * 1000;
        m_refreshFrequency = refreshFrequencyInSeconds * 1000;       

        getDomainsFromDS();
        ArrayList needInitDomains =  getNeedUpdateDomains();
        initDomains(INITIALIZE_DOMAIN_TIMEOUT, needInitDomains);
        if (!isRestrictedBackupDS)
        {
            // don't do async updates if the DS is a read-only backup
            createAsyncUpdates(m_updateFrequency);
        }
    }

    @Override
    public void logMessage(String message, int severityLevel)
    {
        m_ds.logMessage(message, severityLevel);
    }
    
    @Override
    public void logMessage(String message, Throwable throwable, int severityLevel)
    {
        m_ds.logMessage(message, throwable, severityLevel);
    }

    @Override
    public void trace(int mask, String message)
    {
        m_ds.trace(mask, message);
    }
    
    @Override
    public void trace(int mask, String message, Throwable throwable)
    {
        m_ds.trace(mask, message, throwable);
    }

    // Can handle on the fly only the modification of connection parameters
    public synchronized void modifyDomainConnectionParameters(IElement domainDescriptorElement)
    {
        if (m_isClosing)
        {
            return;
        }

        EntityName elementName = null;
        try
        {
            elementName = new EntityName(domainDescriptorElement.getIdentity().getName());
        }
        catch (ConfigException e) // Should never happen
        {
            m_ds.logMessage("Failure to get security domain configuration identity, trace follows...", e, Level.SEVERE);
            return;
        }

        // The ID of the domain is the root directory of the domain
        String domainID = elementName.getParent();
        DomainConfiguration domainConf = (DomainConfiguration)m_domains.get(domainID);
        if (domainConf == null)
        {
            return;
        }

        if (!domainConf.m_external)
        {
            return;
        }

        // It is very unlikely that this could happen - only if the connection parameters are modified and
        // concurrently the DS is entered into a backup state.
        if (m_isStopped)
        {
            m_ds.logMessage("Ignoring new connection parameters of security domain \"" + domainConf.m_name + "\", with configuration id \"" + domainConf.m_domainID + "\" since the Directory Service is in a read-only state.", Level.WARNING);
            return;
        }

        // Get the new connection parameters
        IAttributeSet descAttributes = domainDescriptorElement.getAttributes();
        IAttributeSet connectionAttributes = (IAttributeSet)descAttributes.getAttribute(DOMAIN_CONNECTION_PARAM_ATT);
        HashMap newConnectionParameters = (connectionAttributes == null) ? null : connectionAttributes.getAttributes();
        if (!validConnectionParameters(newConnectionParameters))
        {
            m_ds.logMessage("The new connection parameters of security domain \"" + domainConf.m_name + "\", with configuration id \"" + domainConf.m_domainID + "\" are invalid; the new parameters are ignored.", Level.WARNING);
            return;
        }

        // PASS
        // Rajiv: This line has been commented so that any change in the domainConf
        // will cause reload. For complete implementation details, check with Ehud.

        // if (equalConnectionParam(newConnectionParameters, domainConf.m_connectionParameters))
        // return;

        stopAsyncUpdate(domainConf);

        close(domainConf);
        domainConf.m_connectionParameters = newConnectionParameters;

        createAuthSource(domainConf, m_logger);
        if (domainConf.m_externalSource == null)
        {
            m_ds.logMessage("Failed to connect with new connection parameters of security domain \"" + domainConf.m_name + "\", with configuration id \"" + domainConf.m_domainID + "\"", Level.WARNING);
            m_ds.logMessage("Security domain \"" + domainConf.m_name + "\" is not being updated from the external source", Level.WARNING);
            m_domains.remove(domainID);
            return;
        }

        // When there is an event listener we don't have an update thread so we need to initialize with fresh data
        if (domainConf.m_eventListener != null)
        {
            try
            {
                updateAll(domainConf, ASYNC_UPDATE_TIMEOUT, true);
            }
            catch (Exception e)
            {
                if (!(e instanceof TimeoutException)) // If it's a timeout - we hope the connection will be recovered -
                                                      // so we don't giveup
                {
                    m_ds.logMessage("Failed to update security domain \"" + domainConf.m_name + "\", trace follows...", e, Level.WARNING);
                    close(domainConf);
                    m_domains.remove(domainID);
                }
            }
        }

        startExternalAsyncUpdate(domainConf, m_updateFrequency);
    }

    private static boolean containsChar(String src, char[] chars)
    {
        if (src == null)
        {
            return false;
        }

        for (int i = 0; i < chars.length; i++)
        {
            if (src.indexOf(chars[i]) != -1)
            {
                return true;
            }
        }

        return false;
    }

    private boolean equalConnectionParam(HashMap newParams, HashMap oldParams)
    {
        if (newParams == null && oldParams == null)
        {
            return true;
        }
        else if (newParams == null)
        {
            return false;
        }
        else if (oldParams == null)
        {
            return false;
        }

        if (newParams.size() != oldParams.size())
        {
            return false;
        }

        Iterator iterator = newParams.keySet().iterator();

        while (iterator.hasNext())
        {
            String key = (String)iterator.next();
            String newVal = (String)newParams.get(key);
            String oldVal = (String)oldParams.get(key);
            if (oldVal == null)
            {
                return false;
            }
            if (!newVal.equals(oldVal))
            {
                return false;
            }
        }
        return true;
    }

    private boolean validConnectionParameters(HashMap paramTable)
    {
         if (paramTable == null)
        {
            return true;
        }

         Iterator params = paramTable.values().iterator();
         while (params.hasNext())
        {
            if(!(params.next() instanceof String))
            {
                return false;
            }
        }

         return true;
    }

    private void updateAll(DomainConfiguration domainConfig, long timeout, boolean updateRefreshElement) throws DirectoryServiceException, TimeoutException
    {
        updateExternalUsers(domainConfig, timeout);
        updateExternalGroups(domainConfig, timeout);
        if (updateRefreshElement)
        {
            updateRefreshElement(domainConfig);
        }
    }

    private void updateExternalUsers(DomainConfiguration domainConfig, long timeout)
        throws TimeoutException
    {
        IPasswordUser[] users = domainConfig.m_externalSource.getUsers(timeout);
        if (users == null)
        {
            users = new IPasswordUser[0];
        }
        String domainUsersDir = domainConfig.m_domainID + "/" + EXTERNAL_USERS_DIR;
        IDirElement[] userElements = externalUsersToElements(users, domainUsersDir, domainConfig.m_name);
        try
        {
            m_dsUpdater.importFromList(userElements, new String[]{domainUsersDir});
        }
        catch (Throwable t)
        {
            m_ds.logMessage("Failed to update external users for security domain \"" + domainConfig.m_name + "\", trace follows...", t, Level.WARNING);
        }
    }

    private void appendExternalUsers(IPasswordUser[] users, DomainConfiguration domainConfig)
    {
        if (users == null)
        {
            users = new IPasswordUser[0];
        }
        String domainUsersDir = domainConfig.m_domainID + "/" + EXTERNAL_USERS_DIR;
        IDirElement[] userElements = externalUsersToElements(users, domainUsersDir, domainConfig.m_name);
        try
        {
            m_dsUpdater.importFromList(userElements, null);
        }
        catch (Throwable t)
        {
            m_ds.logMessage("Failed to append external users for security domain \"" + domainConfig.m_name + "\", trace follows...", t, Level.WARNING);
        }
    }

    private void updateExternalGroups(DomainConfiguration domainConfig, long timeout)
        throws TimeoutException
    {
        IGroup[] groups = domainConfig.m_externalSource.getGroups(timeout);
        if (groups == null)
        {
            groups = new IGroup[0];
        }
        String domainGroupsDir = domainConfig.m_domainID + "/" + EXTERNAL_GROUPS_DIR;
        IDirElement[] groupElements = externalGroupsToElements(groups, domainGroupsDir, domainConfig.m_name);

        try
        {
            m_dsUpdater.importFromList(groupElements, new String[]{domainGroupsDir});
        }
        catch (Throwable t)
        {
            m_ds.logMessage("Failed to update external groups for security domain \"" + domainConfig.m_name + "\", trace follows...", t, Level.WARNING);
        }
    }

    private void appendExternalGroups(IGroup[] groups, DomainConfiguration domainConfig)
    {
        if (groups == null)
        {
            groups = new IGroup[0];
        }
        String domainGroupsDir = domainConfig.m_domainID + "/" + EXTERNAL_GROUPS_DIR;
        IDirElement[] groupElements = externalGroupsToElements(groups, domainGroupsDir, domainConfig.m_name);

        try
        {
            m_dsUpdater.importFromList(groupElements, null);
        }
        catch (Throwable t)
        {
            m_ds.logMessage("Failed to append external groups for security domain \"" + domainConfig.m_name + "\", trace follows...", t, Level.WARNING);
        }
    }

    private void deleteExternalPrincipals(String[] princNames, DomainConfiguration domainConfig, String dirName)
    {
        try
        {
            m_dsUpdater.deleteElements(externalPrincipalToElementNames(princNames, domainConfig.m_domainID + "/" + dirName));
        }
        catch (Throwable t)
        {
            m_ds.logMessage("Failed to delete external principals for security domain \"" + domainConfig.m_name + "\", trace follows...", t, Level.WARNING);
        }
    }

    // This routine works for both users and groups so we call it principal
    private String[] externalPrincipalToElementNames(String[] principalNames, String domainUsersDir)
    {
        String[] result = new String[principalNames.length];
        for (int i = 0; i < principalNames.length; i++)
        {
            result[i] = domainUsersDir + "/" + principalNames[i];
        }

        return result;
    }

    private IDirElement[] externalGroupsToElements(IGroup[] groups, String domainGroupsDir, String domainName)
    {
        ArrayList groupsList = new ArrayList();
        HashMap duplicationDetection = new HashMap();

        for (int i = 0; i < groups.length; i++)
        {
            IPasswordUser[] groupUsers= groups[i].getGroupUsers();
            if (groupUsers == null)
            {
                groupUsers = new IPasswordUser[0];
            }
            String groupName = groups[i].getName();
            
            if (groupName == null || groupName.trim().length() == 0)
            {
                m_ds.logMessage("An empty group name was received from external source of domain \"" +
                                 domainName + "\" - ignoring this group (name \"" + groupName + "\", idx " + i + ")", Level.WARNING);
                continue;
            }
            
            if (duplicationDetection.put(groupName, Boolean.TRUE) != null)
            {
                m_ds.logMessage("A duplicate group \"" + groupName + "\"  was received from external source of domain \"" +
                                 domainName + "\" - ignoring the second one", Level.WARNING);
                continue;
            }

            if (containsChar(groupName, GROUP_RESTRICTED_CHARS))
            {
                m_ds.logMessage("Group name \"" + groupName + "\"  contains one of the following restricted characters \"" +
                                 new String(GROUP_RESTRICTED_CHARS) + "\" - ignoring this group", Level.WARNING);
                continue;
            }

            String elementName = domainGroupsDir + "/" + groupName;
            IDirElement groupElement = ElementFactory.createElement(elementName, EXTERNAL_GROUP_TYPE, IAuthenticationGroupConstants.DS_C_VERSION);
            IAttributeSet groupAttributes = groupElement.getAttributes();
            try
            {
                groupAttributes.setStringAttribute(EXTERNAL_GROUP_NAME_ATT, groupName);


                IAttributeSet membersSet = groupAttributes.createAttributeSet(EXTERNAL_GROUP_MEMBERS_ATT);
                for (int j = 0; j < groupUsers.length; j++)
                {
                    String userName = groupUsers[j].getName();
                    IAttributeSet member = membersSet.createAttributeSet(userName);
                    member.setStringAttribute(MEMBER_NAME, userName);
                    member.setStringAttribute(MEMBER_TYPE, USER_GROUP_MEMBER_TYPE);

                    if (Debug.TRACE) {
                        Debug.trace(MEMBER_NAME + ": " + userName + " , " + MEMBER_TYPE + ": " + USER_GROUP_MEMBER_TYPE);
                    }

                }

                groupsList.add(groupElement.doneUpdate());
            }
            catch (Throwable t)
            {
                throwError(t); // Should never happen
            }
        }
        
        IDirElement[] result = new IDirElement[groupsList.size()];
        groupsList.toArray(result);
        return result;
    }

    private IDirElement[] externalUsersToElements(IPasswordUser[] users, String domainUsersDir, String domainName)
    {
        ArrayList usersList = new ArrayList();
        HashMap duplicationDetection = new HashMap();

        for (int i = 0; i < users.length; i++)
        {
            String userName = users[i].getName();
            if (duplicationDetection.put(userName, Boolean.TRUE) != null)
            {
                m_ds.logMessage("A duplicate User \"" + userName + "\"  was received from external source of domain \"" +
                                 domainName + "\" - ignoring the second one", Level.WARNING);
                continue;
            }

            if (containsChar(userName, USER_RESTRICTED_CHARS))
            {
                m_ds.logMessage("User name \"" + userName + "\"  contains one of the following restricted characters \"" +
                                 new String(USER_RESTRICTED_CHARS) + "\" - ignoring this user", Level.WARNING);
                continue;
            }

            byte[] password = users[i].getPassword();
            String elementName = domainUsersDir + "/" + userName;
            IDirElement userElement = ElementFactory.createElement(elementName, EXTERNAL_USER_TYPE, IAuthenticationUserConstants.DS_C_VERSION);
            IAttributeSet userAttributes = userElement.getAttributes();
            try
            {
                userAttributes.setStringAttribute(EXTERNAL_USER_NAME_ATT, userName);
                userAttributes.setBytesAttribute(EXTERNAL_USER_PASSWORD_ATT, password);

                if (Debug.TRACE) {
                    Debug.trace(EXTERNAL_USER_NAME_ATT + ": " + userName + " , " + EXTERNAL_USER_PASSWORD_ATT + ": " + password);
                }

                usersList.add(userElement.doneUpdate());
            }
            catch (Throwable t)
            {
                throwError(t); // Should never happen
            }
        }
        
        IDirElement[] result = new IDirElement[usersList.size()];
        usersList.toArray(result);
        return result;
    }

    private IBasicElement createRefreshElement(String elementName)
    {
         IDirElement element = ElementFactory.createElement(elementName, REFRESH_TYPE, IAuthenticationDomainConstants.DS_C_VERSION);
         IAttributeSet attributes = element.getAttributes();
         try
         {
             attributes.setLongAttribute(REFRESH_TIME_ATT, new Long(0));
             return element.doneUpdate();
         }
         catch (Throwable t)
         {
             throwError(t); // Should never happen
             return null;
         }
    }

    private long getRefreshTime(IElement element)
    {
        IAttributeSet attributes = element.getAttributes();
        return ((Long)attributes.getAttribute(REFRESH_TIME_ATT)).longValue();
    }

    private void updateRefreshElement(DomainConfiguration domainConfig)
    {
         try
         {
             IDirElement element = m_ds.getElement(domainConfig.m_domainID + "/" + MF_REFRESH_ELEMENT, true);
             IAttributeSet attributes = element.getAttributes();
             attributes.setLongAttribute(REFRESH_TIME_ATT, new Long(System.currentTimeMillis()));
             m_dsUpdater.setElement(element.doneUpdate());
         }
         catch (Throwable t)
         {
             // That could happen if both the thread and the event try to update concurrently - harmless
             if (t instanceof VersionOutofSyncException)
            {
                return;
            }

             m_ds.logMessage("Failed to update refresh element for security domain \"" + domainConfig.m_name + "\", trace follows...", t, Level.WARNING);
         }
    }

    //Doomsday scenario - should never happen
    private void throwError(Throwable t)
    {
        m_ds.logMessage("Security domain management failure, trace follows...", t, Level.WARNING);
        Error e = new Error("Security domain management failure, see cause");
        e.initCause(t);
        throw e;
    }

    /**
     * <b>NOTE</b>:This method is to be called from constructor only as it
     * initializes the m_domains and m_spis HashMap
     *
     * @throws DirectoryServiceException
     */
    private void getDomainsFromDS() throws DirectoryServiceException
    {
        m_domains = new HashMap();
        m_spis = new HashMap();

        m_domainIDs = new IDirIdentity[0];
        try
        {
            m_domainIDs = m_ds.listDirectories(DOMAINS_DIRECTORY);
        }
        catch (DirectoryServiceException e)
        {
            m_ds.trace(TRACE_EXTERNAL_AUTH,"AuthenticationConfigManager: No \"" + DOMAINS_DIRECTORY + "\" directory.");

            return;
        }

        for (int i = 0; i < m_domainIDs.length; i++)
        {
            // Added for bug Sonic00017238
            String domainID = m_domainIDs[i].getName();
            // getDomainsFromDS(domainID, false);
            getDomainsFromDS(domainID);
        }
    }

    /**
     * Creates and addes MgmtSPIConfiguration into the m_spis. It alos creates a
     * domain configuration and addes it to the m_domains.
     * <p>
     * @param domainID
     * <p>
     * @throws DirectoryServiceException
     * <p>
     */
    // Added for bug Sonic00017238
    // private void getDomainsFromDS(final String domainID, final boolean reloadExternalAuthenticationDomain) throws DirectoryServiceException
    private void getDomainsFromDS(final String domainID) throws DirectoryServiceException
    {
        String descriptorName = domainID + "/" + DOMAIN_DESCRIPTOR;
        IElement domainDescriptor = m_ds.getElement(descriptorName, false);

        if (domainDescriptor == null)
        {
            m_ds.logMessage("Could not find the domain decriptor for the \"" + domainID + "\" domain", Level.WARNING);
            return;
        }

        DomainConfiguration domainConf = getDomainConfFromElement(domainDescriptor, domainID);

        if (domainConf == null)
        {
            return;
        }

        // Get from the DS the management SPI if not in m_spis already
        MgmtSPIConfiguration spiConfig =  (MgmtSPIConfiguration)m_spis.get(domainConf.m_mgmtSpiRef);
        if (domainConf.m_external && spiConfig == null)
        {
            IElement spiDescriptor = m_ds.getElement(domainConf.m_mgmtSpiRef, false);
            if (spiDescriptor == null)
            {
                m_ds.logMessage("Could not find the \"" + domainConf.m_mgmtSpiRef + "\" configuration", Level.WARNING);
                logDomainFailure(domainConf.m_name);
                return;
            }

            spiConfig = getSPIConfFromElement(spiDescriptor);
            if (spiConfig == null)
            {
                logDomainFailure(domainConf.m_name);
                return;
            }

            spiConfig.m_spiClass = getSPIClass(spiConfig.m_className, spiConfig.m_classpath);
            if (spiConfig.m_spiClass == null)
            {
                logDomainFailure(domainConf.m_name);
                return;
            }

            m_spis.put(domainConf.m_mgmtSpiRef, spiConfig);

        }

        if (domainConf.m_external)
        {
            domainConf.m_managementInstance = getSPIInstance(spiConfig.m_spiClass);
            if (domainConf.m_managementInstance == null)
            {
                logDomainFailure(domainConf.m_name);
                return;
            }
            createAuthSource(domainConf, m_logger);
            if (domainConf.m_externalSource == null)
            {
                logDomainFailure(domainConf.m_name);
                return;
            }
        }

        m_domains.put(domainID, domainConf);
    }


    /**
     * <b>NOTE</b>:This method is to be called from constructor only.
     * <p>
     * Get the domain that need update with fresh data and also create
     * directories and refreshElement for new domains
     * <p>
     * @return list of domain IDs to be updated
     * <p>
     * @throws DirectoryServiceException
     */
    ArrayList getNeedUpdateDomains() throws DirectoryServiceException
    {
        m_forbiddenDirs = new HashMap();

        // Domains that need update with fresh data are either new domain or
        // domains that have an event listener and therefore don't have an update thread
        ArrayList needUpdateDomains = new ArrayList();

        for (int i = 0; i < m_domainIDs.length; i++)
        {
            // Added for bug Sonic00017238
            String domainID = m_domainIDs[i].getName();
            String domainIDToUpdate = getNeedUpdateDomains(domainID, false);

            if (domainIDToUpdate != null)
            {
                needUpdateDomains.add(domainIDToUpdate);
            }
        }

        return needUpdateDomains;
    }

    void createExternalDirectories(String domainID) throws DirectoryServiceException
    {
         String externalDir = domainID + "/" + EXTERNAL_DIR;
         String domainUsers = domainID + "/" + EXTERNAL_USERS_DIR;
         String domainGroups = domainID + "/" + EXTERNAL_GROUPS_DIR;

         m_forbiddenDirs.put(domainUsers, Boolean.TRUE);
         m_forbiddenDirs.put(domainGroups, Boolean.TRUE);

         if (!m_ds.directoryExists(externalDir))
        {
            m_ds.createDirectory(externalDir);
        }

         if (!m_ds.directoryExists(domainUsers))
        {
            m_ds.createDirectory(domainUsers);
        }

         if (!m_ds.directoryExists(domainGroups))
        {
            m_ds.createDirectory(domainGroups);
        }
    }


    /**
     *
     * @param domainID
     * <p>
     * @return the domain ID if it has to be updated.
     * <p>
     * @throws DirectoryServiceException
     */
    // Added for bug Sonic00017238
    private String getNeedUpdateDomains(String domainID, boolean reloadExternalAuthenticationDomain) throws DirectoryServiceException
    {
        DomainConfiguration domainConf = (DomainConfiguration)m_domains.get(domainID);

        if (domainConf == null || !domainConf.m_external)
        {
            return null;
        }

        if (domainConf.m_external)
        {
            createExternalDirectories(domainID);
        }

        // Create a MF_REFRESH_ELEMENT if doesn't exist yet
        String refreshElementName = domainID + "/" + MF_REFRESH_ELEMENT;
        IElement refreshElement = m_ds.getElement(refreshElementName, false);
        if (refreshElement == null)
        {
            if (reloadExternalAuthenticationDomain)
            {
                m_dsUpdater.setElement(createRefreshElement(refreshElementName));
            }
            else
            {
                m_ds.setElement(createRefreshElement(refreshElementName), null);
            }

            return domainID;
        }
        else if (getRefreshTime(refreshElement) == 0)
        {
            return domainID;
        }
        else if (domainConf.m_eventListener != null)
        {
            return domainID;
        }

        return null;
    }


    void initDomains(long timeout, ArrayList newDomains)
    {
        ArrayList failedDomains = new ArrayList();
        for (int i = 0; i < newDomains.size(); i++)
        {
            String domainID = (String)newDomains.get(i);
            DomainConfiguration domainConf = (DomainConfiguration)m_domains.get(domainID);
            if (domainConf == null || !domainConf.m_external)
            {
                continue;
            }

            try
            {
                updateAll(domainConf, timeout, true);
            }
            catch (Exception e)
            {
                if (e instanceof TimeoutException)
                {
                    m_ds.logMessage("Failed to initialize security domain \"" + domainConf.m_name + "\" due to timeout", Level.WARNING);
                }
                else
                {
                    m_ds.logMessage("Failed to initialize security domain \"" + domainConf.m_name + "\", trace follows...", e, Level.WARNING);
                }
                failedDomains.add(domainID);
            }
        }

        // Remove from the list domains that couldn't be initialized.
        for (int i = 0; i < failedDomains.size(); i++)
        {
            DomainConfiguration domainConfig = (DomainConfiguration)m_domains.remove(failedDomains.get(i));
            close(domainConfig);
        }

    }

    private void createAuthSource(DomainConfiguration domainConf, ILogger logger)
    {
        try
        {
            domainConf.m_externalSource = new AuthSource(domainConf.m_managementInstance, domainConf.m_name, domainConf.m_connectionParameters, this);
            EventListener listener = new EventListener(domainConf);
            if (domainConf.m_externalSource.registerListener(listener))
            {
                domainConf.m_eventListener = listener;
            }
            else
            {
                domainConf.m_eventListener = null;
            }
            domainConf.m_externalSource.connect();
        }
        catch (Throwable throwable)
        {
             m_ds.trace(TRACE_EXTERNAL_AUTH, "SPI failure, trace follows...", throwable);
            // AuthSource constructor already logged the problem
            domainConf.m_externalSource = null;
            domainConf.m_eventListener = null;
        }
    }

    private void logDomainFailure(String domainName)
    {
        m_ds.logMessage("Could not load the \"" + domainName + "\" domain", Level.WARNING);
    }

    private Class getSPIClass(String className, IAttributeList classpathList)
    {
        try
        {
            return m_ds.loadClass(className, DirectoryService.listToClasspath(classpathList));
        }
        catch (Throwable t)
        {
            m_ds.logMessage("Could not load SPI authentication management class \"" + className + "\", trace follows...", t, Level.SEVERE);
            return null;
        }
    }

    private IManagement getSPIInstance(Class spiClass)
    {
        try
        {
            Object SPIInstance = spiClass.newInstance();
            if (!(SPIInstance instanceof IManagement))
            {
                m_ds.logMessage(spiClass.getName() +  " does not implement the authentication management SPI interface", Level.SEVERE);
                return null;
            }
            return (IManagement)SPIInstance;
        }
        catch (Throwable t)
        {
            m_ds.logMessage("Could not create an instance of an authentication management class \"" + spiClass.getName() + "\", trace follows...", t, Level.SEVERE);
            return null;
        }

    }

    private ServiceMaintainer startRefreshElementThread()
    {
        ServiceMaintenance dsMaintenance = new ServiceMaintenance()
        {
            @Override
            public Exception doMaintenance()
            {
                updateRefreshElements();
                return null;
            }

            @Override
            public void onAccessibilityChange(boolean isAccessible)
            {
            }
        };

        return new ServiceMaintainer("Element Refresher", dsMaintenance, m_refreshFrequency, false);
    }

    private void updateRefreshElements()
    {
         Iterator domains = m_domains.values().iterator();
         while (domains.hasNext())
         {
            DomainConfiguration domainConf = (DomainConfiguration)domains.next();

            // The refresh element of DS managed domains and domains that support events need to be refreshed by
            // a thread since there is no active threads that constantly do that
            if (domainConf.m_eventListener != null && domainConf.m_externalSource.isConnected())
            {
                updateRefreshElement(domainConf);
            }
         }
    }

    String[] getExternalDomainsDescriptors()
    {
         ArrayList externalDescNames = new ArrayList();
         Iterator domains = m_domains.values().iterator();
         while (domains.hasNext())
         {
            DomainConfiguration domainConf = (DomainConfiguration)domains.next();

            if (domainConf.m_external)
            {
                externalDescNames.add(domainConf.m_domainID + "/" + DOMAIN_DESCRIPTOR);
            }
         }

         String[] result = new String[externalDescNames.size()];
         externalDescNames.toArray(result);
         return result;
    }

    void startClosingDomain(String domainID)
    {
         m_dsUpdater.registerThread();
         DomainConfiguration domainConf = (DomainConfiguration) m_domains.get(domainID);
         if (domainConf == null || !domainConf.m_external)
        {
            return;
        }

         stopAsyncUpdate(domainConf);
         close(domainConf);
         m_domains.remove(domainID);
    }

    void endClosingDomain()
    {
         m_dsUpdater.unregisterThread();
    }


    private void startExternalAsyncUpdate(DomainConfiguration domainConf, long frequency)
    {
        if (domainConf.m_eventListener != null)
        {
            domainConf.m_eventListener.stopDelay();
            return;
        }

        ServiceMaintenance dsMaintenance = new ServiceMaintenance()
        {
            @Override
            public Exception doMaintenance(String domainID)
            {
                return updateExternalDomain((DomainConfiguration)m_domains.get(domainID));
            }

            // This will never be called
            @Override
            public Exception doMaintenance()
            {
                m_ds.logMessage("startExternalAsyncUpdate.doMaintenance() shold never be called", Level.SEVERE);
                return null;
            }

            @Override
            public void onAccessibilityChange(boolean isAccessible)
            {
                //The m_externalSource object already logged the change of status so we do it only if TRACE is on
                if (isAccessible)
                {
                    m_ds.trace(TRACE_EXTERNAL_AUTH, "External authentication information update started");
                }
                else
                {
                    m_ds.trace(TRACE_EXTERNAL_AUTH, "External authentication information update failed - retrying...");
                }
            }
        };

        domainConf.m_externalUpdateThread = new ServiceMaintainer(dsMaintenance, frequency, false, domainConf.m_domainID);
    }

    private Exception updateExternalDomain(DomainConfiguration domainConfig)
    {
        try
        {
            long timeNow = System.currentTimeMillis();
            long timeSinceLastUpdate = timeNow - domainConfig.m_lastUpdateThreadRefresh;
            boolean needRefresh =  timeSinceLastUpdate >= m_refreshFrequency;
            updateAll(domainConfig, ASYNC_UPDATE_TIMEOUT, needRefresh);
            if (needRefresh)
            {
                domainConfig.m_lastUpdateThreadRefresh = timeNow;
            }
        }
        catch (Exception e)
        {
            return e;
        }
        return null; // Everything is ok
    }

    synchronized void stopAsyncUpdates()
    {
         m_isStopped = true;
         if (m_DSManagedUpdateThread != null)
         {
             m_DSManagedUpdateThread.close();
             try { m_DSManagedUpdateThread.join(); } catch (InterruptedException e){}

             m_DSManagedUpdateThread = null;
         }
         Iterator domains = m_domains.values().iterator();
         while (domains.hasNext())
        {
            stopAsyncUpdate((DomainConfiguration)domains.next());
        }

    }

    synchronized void stopAsyncUpdate(DomainConfiguration domainConfig)
    {
         if (domainConfig.m_externalUpdateThread != null)
         {
             domainConfig.m_externalUpdateThread.close();
             try { domainConfig.m_externalUpdateThread.join(); } catch (InterruptedException e){}
             domainConfig.m_externalUpdateThread = null;
         }
         else if (domainConfig.m_eventListener != null)
        {
            domainConfig.m_eventListener.startDelay();
        }

    }

    void createAsyncUpdates()
    {
        createAsyncUpdates(m_updateFrequency);
    }

    private synchronized void createAsyncUpdates(long frequency)
    {
         m_isStopped = false;
         Iterator domains = m_domains.values().iterator();
         boolean needRefreshThred = false;
         while (domains.hasNext())
         {
            DomainConfiguration domainConf = (DomainConfiguration)domains.next();

            if (domainConf.m_external)
            {
                startExternalAsyncUpdate(domainConf, frequency);
                if (domainConf.m_eventListener != null)
                {
                    needRefreshThred = true;
                }
            }
         }

         if (needRefreshThred)
        {
            m_DSManagedUpdateThread = startRefreshElementThread();
        }
    }


    synchronized void close()
    {
         m_isClosing = true;
         stopAsyncUpdates();
         Iterator domains = m_domains.values().iterator();
         while (domains.hasNext())
        {
            close((DomainConfiguration)domains.next());
        }
    }

    private void close(DomainConfiguration domainConfig)
    {
        if (!domainConfig.m_external)
        {
            return;
        }
        if (domainConfig.m_externalSource != null)
        {
            domainConfig.m_externalSource.disconnect();
        }
        domainConfig.m_externalSource = null;
    }

    @Override
    public String toString()
    {
        String result = "";

        if (!m_spis.isEmpty())
        {
            result += "Management SPIs:" + IContainer.NEWLINE;
        }
        Iterator spis = m_spis.values().iterator();
        while (spis.hasNext())
        {
            MgmtSPIConfiguration spiConfig = (MgmtSPIConfiguration)spis.next();
            result += spiConfig.m_sourceType +
                      " " +
                      spiConfig.m_className +
                      IContainer.NEWLINE;
        }

       if (!m_domains.isEmpty())
    {
        result += IContainer.NEWLINE + "Domains:" + IContainer.NEWLINE;
    }

        Iterator domains = m_domains.values().iterator();
        while (domains.hasNext())
        {
            DomainConfiguration domainConfig = (DomainConfiguration)domains.next();
            result += domainConfig.m_name +
                      (domainConfig.m_external ? " 'external' " : " 'DS managed' ")  +
                      ((domainConfig.m_mgmtSpiRef != null) ? domainConfig.m_mgmtSpiRef : "")  +
                      " " +
                      IContainer.NEWLINE;
        }

       if (!m_forbiddenDirs.isEmpty())
    {
        result += IContainer.NEWLINE + "Restricted Directories:" + IContainer.NEWLINE;
    }

        Iterator fDirs = m_forbiddenDirs.keySet().iterator();
        while (fDirs.hasNext())
        {
            result += (String)fDirs.next() + IContainer.NEWLINE;
        }

        return result;
    }

    private DomainConfiguration getDomainConfFromElement(IElement descElement, String domainID)
    {
        IElementIdentity descID = descElement.getIdentity();

        if (!descID.getType().equals(DOMAIN_DESC_TYPE))
        {
            m_ds.logMessage("\"" + descID.getName() + "\" has the wrong type. Should be \"" + DOMAIN_DESC_TYPE + "\"", Level.WARNING);
            return null;
        }


        IAttributeSet descAttributes = descElement.getAttributes();
        DomainConfiguration domainConfig = new DomainConfiguration();
        domainConfig.m_domainID = domainID;

        String name = (String)descAttributes.getAttribute(DOMAIN_NAME_ATT);
        domainConfig.m_name = (name == null) ? "?" : name;


        Reference spiRef = (Reference)descAttributes.getAttribute(DOMAIN_MGMT_SPI_ATT);
        domainConfig.m_mgmtSpiRef = (spiRef == null) ? null : spiRef.getElementName();

        Boolean external = (Boolean)descAttributes.getAttribute(DOMAIN_EXTERNAL_ATT);
        domainConfig.m_external = (external == null) ? false : external.booleanValue();

        IAttributeSet connectionAttributes = (IAttributeSet)descAttributes.getAttribute(DOMAIN_CONNECTION_PARAM_ATT);
        domainConfig.m_connectionParameters = (connectionAttributes == null) ? null : connectionAttributes.getAttributes();

        if (!validConnectionParameters(domainConfig.m_connectionParameters))
        {
            m_ds.logMessage("The connection parameters of domain \"" + domainConfig.m_name + "\", with configuration id \"" +
                             descID.getName() + "\" are not valid - must be of type String", Level.WARNING);
            return null;
        }

        if (domainConfig.m_external && domainConfig.m_mgmtSpiRef == null)
        {
            m_ds.logMessage("Domain \"" + domainConfig.m_name + "\", with configuration id \"" + descID.getName() +
                            "\" is external. It does not contain reference to Management SPI class", Level.WARNING);
            return null;
        }

        return domainConfig;
    }

    private MgmtSPIConfiguration getSPIConfFromElement(IElement descElement)
    {
        IElementIdentity descID = descElement.getIdentity();

        if (!descID.getType().equals(SPI_DESC_TYPE))
        {
            m_ds.logMessage("\"" + descID.getName() + "\" has the wrong type. Should be \"" + SPI_DESC_TYPE + "\"", Level.WARNING);
            return null;
        }


        IAttributeSet descAttributes = descElement.getAttributes();
        MgmtSPIConfiguration spiConfig = new MgmtSPIConfiguration();

        String type = (String)descAttributes.getAttribute(SPI_SOURCE_TYPE_ATT);
        spiConfig.m_sourceType= (type == null) ? "?" : type;

        spiConfig.m_className = (String)descAttributes.getAttribute(SPI_CLASS_NAME_ATT);

        spiConfig.m_classpath = (IAttributeList)descAttributes.getAttribute(SPI_CLASSPATH_ATT);

        return spiConfig;
    }


    void okToModify(EntityName name) throws DirectoryServiceException
    {
        String parentDir = name.getParentEntity().getName();
        Thread thisThread = Thread.currentThread();
        if (!(thisThread instanceof ServiceMaintainer) && !m_dsUpdater.isAuthorized(thisThread))
        {
            if (m_forbiddenDirs.get(parentDir) != null)
            {
                throw new DirectoryServiceException("Elements under the \"" + parentDir + "\" directory cannot be modified.");
            }
            if (name.getBaseName().equals(MF_REFRESH_ELEMENT))
            {
                throw new DirectoryServiceException("\"" + MF_REFRESH_ELEMENT + "\" cannot be modified.");
            }
        }
    }

    void okToDeleteDir(EntityName nameE) throws DirectoryServiceException
    {
        String name = nameE.getName();
        if (name.equals("/") || name.equals(DirectoryService.SYSTEM_DIRECTORY_PATH))
        {
            throw new DirectoryServiceException("\"" + name + "\" cannot be deleted.");
        }

        if (m_forbiddenDirs.get(name) != null && !m_dsUpdater.isAuthorized(Thread.currentThread()))
        {
            throw new DirectoryServiceException("The \"" + name + "\" directory cannot be modified.");
        }

    }

    void okToModify(String name) throws DirectoryServiceException
    {
        try
        {
            okToModify(new EntityName(name));
        }
        catch (ConfigException e)
        {
            throw new DirectoryServiceException(e.getMessage());
        }
    }

    // PASS
    // TODO rajiv doc this
    /**
     * @see DirectoryService#reloadExternalAuthenticationDomain(String)
     */
    Boolean reloadExternalAuthenticationDomain(final String domainID) throws DirectoryServiceException
    {
        // if the thread has reached here, then it has been aready been tested for
        // external domain.

        getDomainsFromDS(domainID);

        // Domains that need update with fresh data are either new domain or
        // domains that have an event listener and therefore don't have an update thread
        ArrayList needUpdateDomains = new ArrayList();

        String domainIDToUpdate = getNeedUpdateDomains(domainID, true);

        if (domainIDToUpdate != null)
        {
            needUpdateDomains.add(domainIDToUpdate);
        }

        initDomains(INITIALIZE_DOMAIN_TIMEOUT, needUpdateDomains);

        return Boolean.TRUE;
    }


    private final class DSUpdater
    {
        Hashtable m_authrizedThreads = new Hashtable();

        boolean isAuthorized(Thread thread)
        {
            return m_authrizedThreads.containsKey(thread);
        }

        void registerThread()
        {
             m_authrizedThreads.put(Thread.currentThread(), Boolean.TRUE);
        }

        void unregisterThread()
        {
             m_authrizedThreads.remove(Thread.currentThread());
        }

        void importFromList (IDirElement[] elements, String[] deleteList) throws DirectoryServiceException
        {
            try
            {
                registerThread();
                m_ds.importFromList(elements, deleteList);
            }
            finally
            {
                unregisterThread();
            }
        }

        void deleteElements (String[] names) throws DirectoryServiceException
        {
            try
            {
                registerThread();
                m_ds.deleteElements(names);
            }
            finally
            {
                unregisterThread();
            }

        }

        void setElement(IBasicElement element) throws DirectoryServiceException
        {
            try
            {
                registerThread();
                m_ds.setElement(element, null);
            }
            finally
            {
                unregisterThread();
            }

        }

    }

    private final class EventListener implements IAuthListener
    {
       final short UPDATE_USERS = 1;
       final short UPDATE_GROUPS = 2;
       final short DELETE_USERS = 3;
       final short DELETE_GROUPS = 4;
       final short CONNECTION_RECOVERED = 5;

        private DomainConfiguration m_domainConfig;
        private boolean m_delayMode;
        private ArrayList m_events;

        EventListener(DomainConfiguration domainConfig)
        {
            m_domainConfig = domainConfig;
            m_delayMode = true;
            m_events = new ArrayList();
        }

        synchronized void stopDelay()
        {
            m_delayMode = false;
            applyDelayedEvents();
        }

        synchronized void startDelay()
        {
            m_delayMode = true;
        }

        @Override
        synchronized public void connectionRecovered()
        {
            if (m_delayMode)
            {
                m_events.add(new Event(CONNECTION_RECOVERED));
                return;
            }

            startDelay(); // Start a delay since we don't want updateAll() collide with update events
            try
            {
                updateAll(m_domainConfig, ASYNC_UPDATE_TIMEOUT, true);
            }
            catch (Exception e)
            {
                if (!(e instanceof TimeoutException))
                {
                    // If it's a timeout - we hope the connection will be recovered -
                    // so we don't giveup
                    m_ds.logMessage("Failed to update domain \"" + m_domainConfig.m_name + "\" from external source, trace follows...", e, Level.WARNING);
                }
            }
            stopDelay();
        }

        @Override
        synchronized public void updateUsers(IPasswordUser[] users)
        {
            if (m_delayMode)
            {
                m_events.add(new Event(users, UPDATE_USERS));
            }
            else
            {
                appendExternalUsers(users, m_domainConfig);
                updateRefreshElement(m_domainConfig);
            }
        }

        @Override
        synchronized public void updateGroups(IGroup[] groups)
        {
            if (m_delayMode)
            {
                m_events.add(new Event(groups, UPDATE_GROUPS));
            }
            else
            {
                appendExternalGroups(groups, m_domainConfig);
                updateRefreshElement(m_domainConfig);
            }
        }

        @Override
        synchronized public void deletePrincipals(String[] principals, boolean isUsers)
        {
           if (m_delayMode)
        {
            m_events.add(new Event(principals, isUsers ? DELETE_USERS : DELETE_GROUPS));
        }
        else
           {
               deleteExternalPrincipals(principals, m_domainConfig, isUsers ? EXTERNAL_USERS_DIR : EXTERNAL_GROUPS_DIR);
               updateRefreshElement(m_domainConfig);
           }
        }

        private void applyDelayedEvents()
        {
            for (int i = 0; i < m_events.size(); i++)
            {
                Event event = (Event)m_events.get(i);
                switch (event.m_type)
                {
                    case CONNECTION_RECOVERED: connectionRecovered();
                                       break;
                    case UPDATE_USERS: appendExternalUsers((IPasswordUser[])event.m_data, m_domainConfig);
                                       break;
                    case UPDATE_GROUPS: appendExternalGroups((IGroup[])event.m_data, m_domainConfig);
                                       break;
                    case DELETE_USERS: deleteExternalPrincipals((String[])event.m_data, m_domainConfig, EXTERNAL_USERS_DIR );
                                       break;
                    case DELETE_GROUPS: deleteExternalPrincipals((String[])event.m_data, m_domainConfig, EXTERNAL_GROUPS_DIR );
                                       break;
                }
            }
            if (!m_events.isEmpty())
            {
                m_events = new ArrayList();
                updateRefreshElement(m_domainConfig);
            }
        }

        private final class Event
        {
            Object m_data;
            short m_type;

            Event (Object data, short type)
            {
                m_data = data;
                m_type = type;
            }

            Event (short type)
            {
                m_data = null;
                m_type = type;
            }


        }

    }

    private final class DomainConfiguration
    {
        String  m_name;
        String m_domainID;
        String m_mgmtSpiRef;
        boolean m_external;
        IManagement m_managementInstance = null;
        HashMap m_connectionParameters = null;
        AuthSource m_externalSource = null;
        ServiceMaintainer m_externalUpdateThread = null; // Used if there is no EventListener (no event support by the source)
        EventListener m_eventListener = null;
        long m_lastUpdateThreadRefresh = 0; // Used only if there is an active m_externalUpdateThread
    }

    private final class MgmtSPIConfiguration
    {
        String  m_sourceType;
        String m_className;
        IAttributeList m_classpath;
        Class m_spiClass;
    }



}
