package com.sonicsw.mf.framework.directory.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.IValidationElementChange;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.impl.DeltaElement;
import com.sonicsw.mf.common.config.impl.Element;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.config.impl.VersionMisMatchException;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.common.view.ILogicalNameSpace;
import com.sonicsw.mf.framework.directory.storage.PackedDirUtil;

public class BackReferenceMgr
{
    protected DirectoryService m_ds;

    protected IDirElement m_backRefTypeListElement;
    private boolean m_isRestrictedBackupDS;

    public static final String BACK_REF_TYPE_LIST_ELEMENT =  "back_ref_type_list";
    public static final String BACK_REF_DIR =  "back_refs";

    private static final String BACK_REF_TYPE_LIST_ELEMENT_PATH = IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_SCHEMA_DIR + IMFDirectories.MF_DIR_SEPARATOR + BACK_REF_TYPE_LIST_ELEMENT;
    private static final String BACK_REF_DIR_PATH = IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_SYSTEM_DIR + IMFDirectories.MF_DIR_SEPARATOR + BACK_REF_DIR;


    public BackReferenceMgr(DirectoryService ds, boolean isRestrictedBackupDS) throws DirectoryServiceException
    {
        m_ds = ds;
        m_isRestrictedBackupDS = isRestrictedBackupDS;
        setBackReferenceTypes(IEmptyArray.EMPTY_STRING_ARRAY, false);
    }

    public void setBackReferenceTypes(String[] typeList) throws DirectoryServiceException
    {
        setBackReferenceTypes(typeList, false);
    }

    public void setBackReferencesTypes(Element backRefElement) throws DirectoryServiceException
    {
        IAttributeSet typeAttrs = backRefElement.getAttributes();
        Set typeKeys = typeAttrs.getAttributes().keySet();
        String[] typeNames = new String[typeKeys.size()];
        typeKeys.toArray(typeNames);
        setBackReferenceTypes(typeNames, true); // deletes whatever types were there before; element
                                                // replaces the old element completely
    }

    static String getBackRefElementPath()
    {
        return BACK_REF_TYPE_LIST_ELEMENT_PATH;
    }

    private void setBackReferenceTypes(String[] typeList, boolean deleteOld) throws DirectoryServiceException
    {
        try
        {
            IDirElement element = m_ds.getElement(BACK_REF_TYPE_LIST_ELEMENT_PATH, false);
            
            // we can't do any DS updates so stick with what's there and get out
            if (this.m_isRestrictedBackupDS)
            {
                m_backRefTypeListElement = element;
                return;
            }
            
            HashMap existingTypes = null;
            if (element != null)
            {
                 existingTypes = element.getAttributes().getAttributes();
                 m_ds.deleteElement(BACK_REF_TYPE_LIST_ELEMENT_PATH, null);
            }

            m_backRefTypeListElement = ElementFactory.createElement(BACK_REF_TYPE_LIST_ELEMENT_PATH, "backRefTypeList", DirectoryService.MF_CONFING_VERSION);

            IAttributeSet refTypes = m_backRefTypeListElement.getAttributes();

            // Put existing types in
            if (existingTypes != null && !deleteOld)
            {
                Iterator iter = existingTypes.keySet().iterator();
                while (iter.hasNext())
                {
                    String type = (String)iter.next();
                    refTypes.setStringAttribute(type, type);
                }
            }

            // add new types
            for (int i = 0; i < typeList.length; i++)
            {
                String[] typesNotSupported = PackedDirUtil.ELEMENT_PACKING_TYPES;
                for (int j = 0; j < typesNotSupported.length; j++)
                {
                    if (typeList[i].equals(typesNotSupported[j]))
                    {
                        throw new DirectoryServiceException("The " + typeList[i] + " type cannot be defined for back reference support.");
                    }
                }
                refTypes.setStringAttribute(typeList[i], typeList[i]);
            }

            m_ds.setElement(m_backRefTypeListElement.doneUpdate(), null);
         }
         catch (ConfigException e)
         {
              throw new Error(e.toString(), e); // Should never happen
         }
    }

    public String[] getBackReferenceTypes()
    throws DirectoryServiceException
    {
        return (String[]) m_backRefTypeListElement.getAttributes().getAttributes().keySet().toArray(IEmptyArray.EMPTY_STRING_ARRAY);
    }

    public void resetBackReferences()
    throws DirectoryServiceException
    {
        setBackReferenceTypes(IEmptyArray.EMPTY_STRING_ARRAY, true);
    }

    void deleteBackRefDir()
    throws DirectoryServiceException
    {
        deleteDirectory(BACK_REF_DIR_PATH);
    }

    void scanDSandPopulateBackRefTree()
    throws DirectoryServiceException
    {
         IDirElement[] elements = m_ds.getAllElements("/", false);
         for (int i = 0; i < elements.length; i++)
         {
            elementCreated(elements[i]);
         }

         // Look in sub directories
         IDirIdentity[] dids = m_ds.listDirectories("/");
         for (int i = 0; i < dids.length; i++)
         {
            if (!dids[i].getName().startsWith("_MFSystem"))
            {
                scanDSandPopulateBackRefTree(dids[i].getName());
            }
         }
    }

    void scanDSandPopulateBackRefTree(String dirPath)
    throws DirectoryServiceException
    {
         IDirElement[] elements = m_ds.getAllElements(dirPath, false);
         for (int i = 0; i < elements.length; i++)
         {
            elementCreated(elements[i]);
         }

         // Look in sub directories
         IDirIdentity[] dids = m_ds.listDirectories(dirPath);
         for (int i = 0; i < dids.length; i++)
         {
            scanDSandPopulateBackRefTree(dids[i].getName());
         }
    }

    public AttributeName[] getReferences(String elementPath)
    throws DirectoryServiceException
    {
        try
        {
            IDirElement backRefElement = getBackRefElement(elementPath, true);
            if (backRefElement == null)
            {
                return new AttributeName[0];
            }

            Collection backRefs = backRefElement.getAttributes().getAttributes().values();
            AttributeName[] references = new AttributeName[backRefs.size()];
            Iterator it = backRefs.iterator();
            for (int i = 0; it.hasNext(); i++)
            {
                byte[] bytes  = (byte[]) it.next();
                ByteArrayInputStream in = new ByteArrayInputStream(bytes);
                ObjectInputStream objectIn = new ObjectInputStream(in);
                references[i] = (AttributeName) objectIn.readObject();
            }

            return references;
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to references to " + elementPath, e);
        }
    }

    public void addBackReference(String referToElement, AttributeName attrName)
    throws DirectoryServiceException
    {
        try
        {
            IDirElement backRefElement = getBackRefElement(referToElement);

            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ObjectOutputStream objectOut = new ObjectOutputStream(out);
            objectOut.writeObject(attrName);
            byte[] bytes = out.toByteArray();
            objectOut.close();

            backRefElement.getAttributes().setBytesAttribute(attrName.toString(), bytes);

            setBackRefElement(referToElement, backRefElement);
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to add back reference '" + attrName + "' to " + referToElement, e);
        }
    }

    public void deleteBackReference(String referToElement, AttributeName attrName)
    throws DirectoryServiceException
    {
        try
        {
            IDirElement backRefElement = getBackRefElement(referToElement);

            backRefElement.getAttributes().deleteAttribute(attrName.toString());

            setBackRefElement(referToElement, backRefElement);
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to delete back reference '" + attrName + "' to " + referToElement, e);
        }
    }

    public void updateBackReferences(IValidationElementChange[] changes)
    throws DirectoryServiceException
    {
        try
        {
            for (int i = 0; i < changes.length; i++)
            {
                switch (changes[i].getChangeType())
                {
                case IValidationElementChange.ELEMENT_ADDED:
                    elementCreated((IDirElement) changes[i].getElement());
                    break;
                case IValidationElementChange.ELEMENT_UPDATED:
                    IDirElement currentElement = (IDirElement) changes[i].getBeforeImage();
                    IDeltaElement delta = (IDeltaElement) changes[i].getElement();
                    elementModified(currentElement, delta);
                    break;
                case IValidationElementChange.ELEMENT_DELETED:
                    IDirElement deletedElement = (IDirElement) changes[i].getBeforeImage();
                    if (deletedElement != null)
                    {
                        elementDeleted(deletedElement);
                    }
                    break;
                }
            }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to update back references", e);
        }
    }

    public void elementCreated(IDirElement element)
    throws DirectoryServiceException
    {
        try
        {
            updateBackReferences(null, element);
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to create back references", e);
        }

    }

    public void elementModified(IDirElement currentElement, IDeltaElement delta)
    throws DirectoryServiceException
    {
        try
        {
            // Create temporary updated element.
            IDirElement updatedElement = (IDirElement) currentElement.createWritableClone();
            ((Element)updatedElement).doApplyDelta(((DeltaElement)delta).createClone());

            updateBackReferences(currentElement, updatedElement);
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to update back references", e);
        }

    }

    public void elementDeleted(IDirElement element)
    throws DirectoryServiceException
    {
        try
        {
            deleteBackReferences(element);
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to delete back references", e);
        }

    }

    public void elementDeleted(String elementName)
    throws DirectoryServiceException
    {
        elementDeleted(m_ds.getElementAsIs(elementName, true));
    }

    public void updateBackReferences(IDirElement currentElement, IDirElement updatedElement)
    throws DirectoryServiceException
    {
        try
        {
            if (!tracksReferences(updatedElement))
            {
                return;
            }

            HashMap addedRefs = new HashMap();
            HashMap deletedRefs = new HashMap();

            getReferenceChanges(currentElement, updatedElement, addedRefs, deletedRefs);

            // Add back refs for new references
            Iterator it = addedRefs.keySet().iterator();
            while (it.hasNext())
            {
                AttributeName addedRefAttr = (AttributeName) it.next();
                String referToElement = (String)addedRefs.get(addedRefAttr);
                if (referToElement.indexOf(ILogicalNameSpace.NO_STORAGE_LABEL) != -1)
                {
                    continue;
                }
                addBackReference(referToElement, addedRefAttr);
            }

            // Remove back refs for deleted references
            it = deletedRefs.keySet().iterator();
            while (it.hasNext())
            {
                AttributeName deletedRefAttr = (AttributeName) it.next();
                String referToElement = (String)deletedRefs.get(deletedRefAttr);
                if (referToElement.indexOf(ILogicalNameSpace.NO_STORAGE_LABEL) != -1)
                {
                    continue;
                }
                deleteBackReference(referToElement, deletedRefAttr);
            }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to update back references", e);
        }
    }

    public void deleteBackReferences(IDirElement element)
    throws DirectoryServiceException
    {
        try
        {
            if (!tracksReferences(element))
            {
                return;
            }

            // Determine references in updated element.
            HashMap references = new HashMap();
            findReferences(element, references);

            // Remove back refs for references in element
            Iterator it = references.keySet().iterator();
            while (it.hasNext())
            {
                AttributeName deletedRefAttr = (AttributeName) it.next();
                String referToElement = (String)references.get(deletedRefAttr);
                if (referToElement.indexOf(ILogicalNameSpace.NO_STORAGE_LABEL) != -1)
                {
                    continue;
                }
                deleteBackReference(referToElement, deletedRefAttr);
            }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to delete back references", e);
        }
    }



    public static void getReferenceChanges(IDirElement currentElement, IDirElement updatedElement,
                                          HashMap addedRefs, HashMap deletedRefs)
    throws VersionMisMatchException
    {
        // Determine references in updated element.
        HashMap updatedReferences = new HashMap();
        findReferences(updatedElement, updatedReferences);

        // Check if this is a new element.
        if (currentElement == null)
        {   // New element: all references in updated element are new.
            addedRefs.putAll(updatedReferences);
            return;
        }

        // Determine references in current element.
        HashMap currentReferences = new HashMap();
        findReferences(currentElement, currentReferences);

        // Determine Deleted and Modified References
        Iterator it = currentReferences.keySet().iterator();
        while (it.hasNext())
        {
            AttributeName currentRefAttr = (AttributeName) it.next();

            if (updatedReferences.containsKey(currentRefAttr))
            {   // Reference still exists in updated element.
                // Determine if reference value has changed
                if (!currentReferences.get(currentRefAttr).equals(updatedReferences.get(currentRefAttr)))
                {   // Reference has changed.
                    addedRefs.put(currentRefAttr, updatedReferences.get(currentRefAttr));
                    deletedRefs.put(currentRefAttr, currentReferences.get(currentRefAttr));
                }
            }
            else
            {   // Reference has been deleted from updated element.
                deletedRefs.put(currentRefAttr, currentReferences.get(currentRefAttr));
            }
        }

        // Determine New References
        it = updatedReferences.keySet().iterator();
        while (it.hasNext())
        {
            AttributeName updatedRefAttr = (AttributeName) it.next();
            if (!currentReferences.containsKey(updatedRefAttr))
            {   // Reference is only in updated element therefore new.
                addedRefs.put(updatedRefAttr, updatedReferences.get(updatedRefAttr));
            }
        }
    }

    public static void findReferences(IDirElement element, HashMap references)
    {
        boolean subclassOnly = element.getSuperElementName() != null;

        findReferences(element.getAttributes(), references,
                       new AttributeName(element.getIdentity().getName(), true));
    }

    public static void findReferences(IAttributeSet attSet, HashMap references,
                                      AttributeName myName)
    {

        HashMap map = attSet.getAttributes();
        Set keys = map.keySet();
        Iterator iter = keys.iterator();
        while( iter.hasNext() )
        {
            String key = (String)iter.next();
            Object value = map.get(key);
            if (value instanceof IAttributeSet)
            {
                findReferences((IAttributeSet)value, references,
                                   myName.createClone().setNextComponent(key));
            }
            else if (value instanceof IAttributeList)
            {
                findReferences((IAttributeList)value, references,
                                  myName.createClone().setNextComponent(key));
            }
            else if (value instanceof Reference)
            {
                references.put(myName.createClone().setNextComponent(key), ((Reference)value).getElementName());
            }
            else
            {
                continue;
            }
        }

    }

    public static void findReferences(IAttributeList attList, HashMap references,
                                      AttributeName myName)
    {
       for (int i = 0; i < attList.getCount(); i++)
       {
           Object value = attList.getItem(i);
           if (value instanceof IAttributeSet)
        {
            findReferences((IAttributeSet)value, references,
                              myName.createClone().setNextComponent(i));
        }
        else if (value instanceof IAttributeList)
        {
            findReferences((IAttributeList)value, references,
                              myName.createClone().setNextComponent(i));
        }
        else if (value instanceof Reference)
            {
                references.put(myName.createClone().setNextComponent(i), ((Reference)value).getElementName());
            }
        else
        {
            continue;
        }
       }
    }

    IDirElement getBackRefElement(String elementName)
    throws DirectoryServiceException
    {
        return getBackRefElement(elementName, false);
    }

    IDirElement getBackRefElement(String elementName, boolean doNotCreate)
    throws DirectoryServiceException
    {
        try
        {
            IDirElement element = null;
            EntityName name = new EntityName(BACK_REF_DIR_PATH + elementName);
            try
            {
                element = m_ds.getStorage().getElement(name);
            }
            catch (Exception e)
            {
                // Element does not exist
            }

            if (element == null && !doNotCreate)
            {
                element =  ElementFactory.createElement(BACK_REF_DIR_PATH + elementName, "backRef", DirectoryService.MF_CONFING_VERSION);
                m_ds.getStorage().setElement(name, element, true);
            }

            return element;
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to get back references for " + elementName, e);
        }
    }

    void setBackRefElement(String elementName, IDirElement element)
    throws DirectoryServiceException
    {
        try
        {
            EntityName name = new EntityName(BACK_REF_DIR_PATH + elementName);

            if (element.getAttributes().getAttributes().size() == 0)
            {
                m_ds.getStorage().deleteElement(name);
            }
            else
            {
                m_ds.getStorage().setElement(name, element, true);
            }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to set back references for " + elementName, e);
        }
    }

    void deleteBackRefElement(String elementName)
    throws DirectoryServiceException
    {
        try
        {
            EntityName name = new EntityName(BACK_REF_DIR_PATH + elementName);
            m_ds.getStorage().deleteElement(name);
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to delete back references for " + elementName, e);
        }
    }

    boolean tracksReferences(IDirElement element)
    {
        Object tmp = m_backRefTypeListElement.getAttributes().getAttribute(element.getIdentity().getType());
        if (tmp == null)
        {
            return false;
        }
        else
        {
            return true;
        }
    }

    void deleteDirectory(String path)
    throws DirectoryServiceException
    {
        try
        {
            EntityName name = new EntityName(path);

            if (!m_ds.getStorage().directoryExists(name))
            {
                return;
            }

            IElementIdentity[] ids = m_ds.getStorage().listElements(name);
            for (int i = 0; i < ids.length; i++)
            {
                m_ds.getStorage().deleteElement(new EntityName(ids[i].getName()));
            }

            IDirIdentity[] dirIds = m_ds.getStorage().listDirectories(name);
            for (int i = 0; i < dirIds.length; i++)
            {
                deleteDirectory(dirIds[i].getName());
            }

            m_ds.getStorage().deleteDirectory(name);

        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Failed to delete directory '" + path + "'", e);
        }
    }
}
