package com.sonicsw.mf.framework.directory.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.NotModifiedAttException;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.framework.IAuditManager;
import com.sonicsw.mf.framework.security.AuditManager;

class ChangeAuditor
{
    private DirectoryService m_ds;
    private IAuditManager m_auditManager;
    private ArrayList m_auditRecords;

    private class AuditRecord
    {
        String logicalPath;
        String storagePath;
        int action;
        String detail;
    }

    ChangeAuditor(DirectoryService ds, IAuditManager auditManager)
    {
        m_ds = ds;
        m_auditManager = auditManager;
        m_auditRecords = new ArrayList();
    }

    private String auditAttribute(String name, Object value)
    {
        return auditAttribute(name, value, null);
    }

    private String auditAttribute(String name, Object value, String action)
    {
        String valueString = null;
        if (value instanceof IAttributeSet)
        {
            valueString = auditAttributeSet((IAttributeSet)value);
            if (valueString.length() == 0)
             {
                return ""; // Don't audit empty AttributeSets.
            }
        }
        else
        {
            valueString = value.toString();
        }
        StringBuffer sb = new StringBuffer();
        sb.append("<").append(name);
        if (action != null)
        {
            sb.append(" action=\"").append(action).append("\"");
        }
        sb.append(">");
        sb.append(valueString);
        sb.append("</").append(name).append(">");
        return sb.toString();
    }

    private String auditAttributeSet(IAttributeSet set)
    {
        StringBuffer sb = new StringBuffer();
        HashMap map = set.getAttributes();
        Iterator iter = map.entrySet().iterator();
        while (iter.hasNext())
        {
            Map.Entry entry = (Map.Entry)iter.next();
            String name = (String) entry.getKey();
            Object value = entry.getValue();
            sb.append(auditAttribute(name, value));
        }
        return sb.toString();
    }

    private String auditDeltaAttributeSet(IDeltaAttributeSet set, AttributeName setName, IElement beforeImage)
    {
        StringBuffer sb = new StringBuffer();
        String[] delAttrs = set.getDeletedAttributesNames();
        String[] newAttrs = set.getNewAttributesNames();
        String[] modAttrs = set.getModifiedAttributesNames();

        if (delAttrs != null)
        {
            for (int i=0; i<delAttrs.length; i++)
            {
                String name = delAttrs[i];
                AttributeName fullName = setName.createClone().setNextComponent(name);
                Object value = beforeImage.getAttribute(fullName);
                sb.append(auditAttribute(name, value, "delete"));
            }
        }

        if (newAttrs != null)
        {
            for (int i=0; i<newAttrs.length; i++)
            {
                String name = newAttrs[i];
                Object value = null;
                try {
                    value = set.getNewValue(name);
                } catch (NotModifiedAttException e) {}
                sb.append(auditAttribute(name, value, "create"));
            }
        }

        if (modAttrs != null)
        {
            for (int i=0; i<modAttrs.length; i++)
            {
                String name = modAttrs[i];
                AttributeName fullName = setName.createClone().setNextComponent(name);
                Object newValue = null;
                try {
                    newValue = set.getNewValue(name);
                } catch (NotModifiedAttException e) {}
                if (newValue instanceof IDeltaAttributeSet)
                {
                    String attrs = auditDeltaAttributeSet((IDeltaAttributeSet)newValue, fullName, beforeImage);
                    if (attrs.length() > 0)
                    {
                        sb.append(auditAttribute(name, attrs, "update"));
                    }
                }
                else
                {
                    Object oldValue = beforeImage.getAttribute(fullName);
                    sb.append(auditAttribute(name, oldValue, "delete"));
                    sb.append(auditAttribute(name, newValue, "create"));
                }
            }
        }
        return sb.toString();
    }

    private String auditLogicalName(String storageName)
    {
        String logicalName = null;
        try
        {
            logicalName = m_ds.storageToLogical(storageName);
        }
        catch (Exception e)
        {
            logicalName = null;
        }
        return logicalName;
    }

    private String auditContents(IElement element)
    {
        String type = element.getIdentity().getType();
        IAttributeSet attributes = element.getAttributes();
        StringBuffer sb = new StringBuffer();
        sb.append("<").append(type).append(">");
        sb.append(auditAttributeSet(attributes));
        sb.append("</").append(type).append(">");
        return sb.toString();
    }

    private String auditDelta(IDeltaElement delta, IElement beforeImage)
    {
        String type = delta.getIdentity().getType();
        IDeltaAttributeSet set = (IDeltaAttributeSet)delta.getDeltaAttributes();
        StringBuffer sb = new StringBuffer();
        sb.append("<").append(type).append(">");
        sb.append(auditDeltaAttributeSet(set, new AttributeName(), beforeImage));
        sb.append("</").append(type).append(">");
        return sb.toString();
    }

    void auditCreate(IElement element)
    {
        AuditRecord rec = new AuditRecord();
        rec.storagePath = element.getIdentity().getName();
        rec.logicalPath = auditLogicalName(rec.storagePath);
        rec.action = AuditManager.CREATE_ACTION;
        rec.detail = auditContents(element);
        m_auditRecords.add(rec);
    }

    void auditDelete(IElement element, String logicalName)
    {
        AuditRecord rec = new AuditRecord();
        rec.storagePath = element.getIdentity().getName();
        rec.logicalPath = logicalName;
        rec.action = AuditManager.DELETE_ACTION;
        rec.detail = auditContents(element);
        m_auditRecords.add(rec);
    }

    void auditUpdate(IDeltaElement delta, IElement beforeImage)
    {
        AuditRecord rec = new AuditRecord();
        rec.storagePath = delta.getIdentity().getName();
        rec.logicalPath = auditLogicalName(rec.storagePath);
        rec.action = AuditManager.UPDATE_ACTION;
        rec.detail = auditDelta(delta, beforeImage);
        m_auditRecords.add(rec);
    }

    void auditFolderCreate(String path)
    {
        AuditRecord rec = new AuditRecord();
        rec.storagePath = null;
        rec.logicalPath = path;
        rec.action = AuditManager.CREATE_ACTION;
        rec.detail = "<MF_PATH>" + path + "</MF_PATH>";
        m_auditRecords.add(rec);
    }

    void auditFolderDelete(String path)
    {
        AuditRecord rec = new AuditRecord();
        rec.storagePath = null;
        rec.logicalPath = path;
        rec.action = AuditManager.DELETE_ACTION;
        rec.detail = "<MF_PATH>" + path + "</MF_PATH>";
        m_auditRecords.add(rec);
    }

    void auditRename(String oldPath, String newPath, String storagePath)
    {
        AuditRecord rec = new AuditRecord();
        rec.storagePath = storagePath;
        rec.logicalPath = newPath;
        rec.action = AuditManager.RENAME_ACTION;
        StringBuffer sb = new StringBuffer();
        sb.append("<MF_PATH action=\"delete\">").append(oldPath).append("</MF_PATH>");
        sb.append("<MF_PATH action=\"create\">").append(newPath).append("</MF_PATH>");
        rec.detail = sb.toString();
        m_auditRecords.add(rec);
    }

    void recordAudit()
    {
        // filter out internal events we are not interested in recording
        for (int i=0; i<m_auditRecords.size(); i++)
        {
            AuditRecord rec = (AuditRecord) m_auditRecords.get(i);
            if ((rec.storagePath != null) && rec.storagePath.startsWith("/_MFRuntime"))
            {
                m_auditRecords.remove(i);
                i--;
            }
        }
        // record the events
        int size = m_auditRecords.size();
        if (size == 0)
         {
            return; // nothing to record
        }
        String[] logicalPaths = new String[size];
        String[] storagePaths = new String[size];
        int[] actions = new int[size];
        String[] details = new String[size];
        for (int i=0; i<size; i++)
        {
            AuditRecord rec = (AuditRecord) m_auditRecords.get(i);
            logicalPaths[i] = rec.logicalPath;
            storagePaths[i] = rec.storagePath;
            actions[i] = rec.action;
            details[i] = rec.detail;
        }
        m_auditManager.recordConfigureEvent(logicalPaths, storagePaths, actions, details);
    }
}
