package com.sonicsw.mf.framework.directory.impl;

import java.util.ArrayList;

import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.config.impl.DSTransaction;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.directory.RepairNoStorageReferences;
import com.sonicsw.mf.framework.directory.SharedPermissionChecks;

public final class ClientTransaction
{
    ArrayList m_actions;
    DirectoryService m_ds;
    SharedPermissionChecks m_permissionChecks;
    int m_namingFailedIndex;  // this gets initialized in generateStorageNames
    ArrayList m_hierarchicalCreates = new ArrayList();
    ArrayList m_hierarchicalDeletes = new ArrayList();

    ClientTransaction (DSTransaction transaction, DirectoryService ds)
    {
        m_ds = ds;
        m_actions = transaction.getActions();
    }
    
    ClientTransaction(DSTransaction transaction, DirectoryService ds, IFrameworkComponentContext context)
    {
        this(transaction, ds);
        m_permissionChecks = new SharedPermissionChecks(context, ds);
    }

    INextVersionToken performActions() throws DirectoryServiceException
    {
        if (m_actions.isEmpty())
        {
            return null;
        }

        if (m_actions.size() == 1 && m_actions.get(0) instanceof DSTransaction.CreateElements)
        {
            DSTransaction.CreateElements createElements =  (DSTransaction.CreateElements)m_actions.get(0);
            if (m_ds.canBeDoneInBulk(createElements.m_elements))
            {
                try
                {
                    m_ds.m_lock.writeLock();
                    if (m_permissionChecks != null)
                    {
                        m_permissionChecks.createFSElementsCheck(createElements.m_elements);
                    }
                    return m_ds.createFSElements(createElements.m_elements, m_ds.assignStorageNames(createElements.m_elements, true));
                }
                finally
                {
                    m_ds.m_lock.releaseLock();
                }
            }
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        INextVersionToken token = null;
        boolean startedRememberingIDS = false;
        try
        {
            m_ds.m_lock.writeLock();
            startedRememberingIDS = m_ds.rememberNewIds();
            m_ds.m_trManager.join();
            joinedTransaction = true;

            // generate required storage names and execute the actions. If a name generation fails,
            // execute all the actions up to the action corresponding to the failed name generation,
            // then retry the naming generation from the index where it failed, and then
            // continue with the remaining actions. Multiple failures are handled with the same logic
            // through this recusive method

            ArrayList danglingRefList = new ArrayList();
            searchForHierarchicalConfigs();
            nameAndActionLoop(0, danglingRefList);
            if (!danglingRefList.isEmpty())
            {
                try
                {
                    RepairNoStorageReferences.repairReferences(m_ds, danglingRefList);
                }
                catch (Exception e)
                {
                    if (e instanceof DirectoryServiceException)
                    {
                        throw (DirectoryServiceException)e;
                    }
                    else
                    {
                        throw new DirectoryServiceException(e.toString());
                    }
                }
            }

            transactOK = true;
        }
        finally
        {
            try
            {
                if (startedRememberingIDS)
                {
                    token = m_ds.createNextVersionToken(true);
                }

                if (joinedTransaction)
                {
                    m_ds.m_trManager.leave(transactOK);
                }
            }
            finally
            {
                m_ds.m_lock.releaseLock();
            }
        }

        return token;

    }

    private void nameAndActionLoop(int startAt,  ArrayList danglingRefList) 
    throws DirectoryServiceException
    {
        try
        {
            generateStorageNames(startAt);
        }
        catch (DirectoryServiceException e)
        {
            if (startAt == m_namingFailedIndex)
             {
                // this is a naming error we can't recover from. Throw the exception
                   throw e;
                // else, we continue
            }
        }
        if (m_namingFailedIndex < m_actions.size())
        {
            performActionsInternal(startAt, m_namingFailedIndex - 1, danglingRefList);
            nameAndActionLoop(m_namingFailedIndex, danglingRefList);
        }
        else
        {
            performActionsInternal(startAt, m_actions.size() - 1, danglingRefList);
        }
    }

    private void performActionsInternal(int startIndex, int endIndex, ArrayList danglingRefList)
    throws DirectoryServiceException
    {
    	String actionName = null; // we'll throw a meaningful exception that tells
    	                          // the caller what transaction operation failed.
        try 
        {
			for (int i = startIndex; i <= endIndex; i++) 
			{
				if (m_actions.get(i) instanceof DSTransaction.UpdateElement) 
				{
					DSTransaction.UpdateElement action = (DSTransaction.UpdateElement) m_actions.get(i);
                    actionName = "UpdateElement";
					// If the delte is empty and we are instructed not to
					// perform the update for empty deltas then
					// We just verify that version of the delta matches the
					// version of the element
					if (action.m_doNotUpdateIfEmpty && action.m_element.emptyDelta()) 
					{
						IElementIdentity elementID = action.m_element.getIdentity();
						IElementIdentity dsID = m_ds.getFSIdentity(elementID.getName());
						if (dsID == null || !dsID.equalVersion(elementID)) 
						{
							String dsVersion = (dsID == null) ? null : new Long(dsID.getVersion()).toString();
							m_ds.throwVersionException(elementID.getName(), "Directory version: " + dsVersion
											+ "  Delta version: " + elementID.getVersion());
						}
					} 
					else 
					{
						if (m_permissionChecks != null)
                        {
                            m_permissionChecks.updateFSElementCheck(action.m_element.getIdentity().getName());
                        }
						m_ds.updateFSElement(action.m_element, danglingRefList);
					}
				}

				else if (m_actions.get(i) instanceof DSTransaction.SetAttributes) 
				{
					actionName = "SetAttributes";
					DSTransaction.SetAttributes action = (DSTransaction.SetAttributes) m_actions.get(i);
					if (m_permissionChecks != null) 
					{
						String attrObjectName = action.m_name;
						String possibleParent = m_ds.getHierarchicalPath(attrObjectName, m_hierarchicalCreates);
						// if this action isn't part of a create element of a
						// hierarchical config, do the regular check
						if (possibleParent == null)
                        {
                            m_permissionChecks.setMetaAttributesCheck(attrObjectName);
                        }
					}
					m_ds.setMetaAttributes(action.m_name, action.m_attributes);
				}

				else if (m_actions.get(i) instanceof DSTransaction.CreateElement) 
				{
					actionName = "CreateElement";
					DSTransaction.CreateElement action = (DSTransaction.CreateElement) m_actions.get(i);
					if (m_permissionChecks != null) 
					{
						String createElementName = action.m_element.getIdentity().getName();
						String possibleParent = m_ds.getHierarchicalPath(createElementName, m_hierarchicalCreates);
						// if this action is not part of creating a hierarchical
						// config, or this is the
						// parent hierarchical config, do the permission check.
						if (possibleParent == null || possibleParent.equals(createElementName))
                        {
                            m_permissionChecks.createFSElementsCheck(new IDirElement[] { action.m_element });
                        }
					}
					m_ds.createFSElement(action.m_element,action.m_newStorageName, danglingRefList);
				}

				else if (m_actions.get(i) instanceof DSTransaction.CreateElements) 
				{
					actionName = "CreateElements";
					DSTransaction.CreateElements action = (DSTransaction.CreateElements) m_actions.get(i);
					if (m_permissionChecks != null)
                    {
                        m_permissionChecks.createFSElementsCheck(action.m_elements);
                    }
					m_ds.createFSElements(action.m_elements,action.m_newStorageNames);
				}

				else if (m_actions.get(i) instanceof DSTransaction.CreateFolder) 
				{
					actionName = "CreateFolder";
					DSTransaction.CreateFolder action = (DSTransaction.CreateFolder) m_actions.get(i);
					if (m_permissionChecks != null) 
					{
						String possibleParent = m_ds.getHierarchicalPath(action.m_folderName, m_hierarchicalCreates);
						if (possibleParent == null)
                        {
                            m_permissionChecks.createFolderCheck(action.m_folderName, action.m_existingOk);
                        }
					}
					m_ds.createFolder(action.m_folderName, action.m_existingOk);
				}

				else if (m_actions.get(i) instanceof DSTransaction.DeleteFolder) 
				{
					actionName = "DeleteFolder";
					DSTransaction.DeleteFolder action = (DSTransaction.DeleteFolder) m_actions.get(i);
					if (m_permissionChecks != null) 
					{
						String possibleParent = m_ds.getHierarchicalPath(action.m_folderName, m_hierarchicalDeletes);
						// hierarchical parent configs are deleted as the folder
						// when the delete comes
						// from the SMC and config API, so we need to do the
						// permissions check if the folder name is
						// the hierarchical config (or if it's a folder outside
						// of a hierarchical config,
						// obviously)
						if (possibleParent == null || possibleParent.equals(action.m_folderName))
                        {
                            m_permissionChecks.deleteFolderCheck(action.m_folderName);
                        }
					}
					m_ds.deleteFolder(action.m_folderName);
				}
				// mrd 12/22/2004 local DSTransactions will still use AttachBlob
				// and the local DS will use the stream object to chunk to
				// storage
				// old client DSProxy transactions will still use the AttachBlob
				// with a byte[]
				else if (m_actions.get(i) instanceof DSTransaction.AttachBlob) 
				{
					actionName = "AttachBlob";
					DSTransaction.AttachBlob action = (DSTransaction.AttachBlob) m_actions.get(i);
					if (m_permissionChecks != null)
                    {
                        m_permissionChecks.attachFSBlobCheck(action.m_element.getIdentity().getName());
                    }
					if (action.m_stream != null)
                    {
                        m_ds.attachFSBlobInternal(action.m_element, action.m_stream, action.m_newStorageName, danglingRefList);
                    }
                    else
                    {
                        m_ds.attachFSBlobInternal(action.m_element, action.m_blob, action.m_newStorageName, danglingRefList);
                    }
				}
				// mrd 12/22/2004 proxy DSTransaction will use AppendBlob in the
				// transaction for the last chunk of the blob.
				else if (m_actions.get(i) instanceof DSTransaction.AppendBlob) 
				{
					actionName = "AppendBlob";
					DSTransaction.AppendBlob action = (DSTransaction.AppendBlob) m_actions.get(i);
					if (m_permissionChecks != null)
                    {
                        m_permissionChecks.appendFSBlobCheck(action.m_element.getIdentity().getName());
                    }
					m_ds.appendFSBlobInternal(action.m_element, action.m_blob, action.m_src, action.m_newStorageName, danglingRefList);
				}

				else if (m_actions.get(i) instanceof DSTransaction.SubclassElement) 
				{
					actionName = "SubclassElement";
					DSTransaction.SubclassElement action = (DSTransaction.SubclassElement) m_actions.get(i);
					if (m_permissionChecks != null)
                    {
                        m_permissionChecks.subclassFSElementCheck(action.m_delta.getIdentity().getName(), action.m_newElementPath);
                    }
					m_ds.subclassFSElementInternal(action.m_delta, action.m_newElementPath, action.m_newStorageName);
				}

				else if (m_actions.get(i) instanceof DSTransaction.CloneElement0) 
				{
					actionName = "CloneElement0";
					DSTransaction.CloneElement0 action = (DSTransaction.CloneElement0) m_actions.get(i);
					if (m_permissionChecks != null)
                    {
                        m_permissionChecks.cloneFSElementCheck(action.m_delta.getIdentity().getName(),action.m_newElementPath);
                    }
					m_ds.cloneFSElementInternal(action.m_delta, null, action.m_newElementPath, action.m_createTemplate, action.m_newStorageName);
				}

				else if (m_actions.get(i) instanceof DSTransaction.CloneElement1) 
				{
					actionName = "CloneElement1";
					DSTransaction.CloneElement1 action = (DSTransaction.CloneElement1) m_actions.get(i);
					if (m_permissionChecks != null)
                    {
                        m_permissionChecks.cloneFSElementCheck(action.m_sourcePath, action.m_newElementPath);
                    }
					m_ds.cloneFSElementInternal(null, action.m_sourcePath, action.m_newElementPath, action.m_createTemplate, action.m_newStorageName);
				}

				else if (m_actions.get(i) instanceof DSTransaction.DeleteElement) 
				{
					actionName = "DeleteElement";
					DSTransaction.DeleteElement action = (DSTransaction.DeleteElement) m_actions.get(i);
					if (m_permissionChecks != null) 
					{
						String possibleParent = m_ds.getHierarchicalPath(action.m_elementName, m_hierarchicalDeletes);
						// if this action is not part of deleting a hierarchical
						// configuration, perform the permission check.
						// When a hierarchical config is deleted from the SMC
						// and the config API, the common action in the
						// transaction is the DeleteFolder for the folder of the
						// parent hierarchical config. Because of this,
						// we let the DeleteFolder action do the permission
						// check for the parent hierarchical config.
						if (possibleParent == null)
                        {
                            m_permissionChecks.deleteFSElementCheck(action.m_elementName);
                        }
					}
					m_ds.deleteFSElement(action.m_elementName);
				}

				else if (m_actions.get(i) instanceof DSTransaction.Rename) 
				{
					actionName = "Rename";
					DSTransaction.Rename action = (DSTransaction.Rename) m_actions.get(i);
					if (m_permissionChecks != null)
                    {
                        m_permissionChecks.renameCheck(action.m_oldName, action.m_newName);
                    }
					m_ds.rename(action.m_oldName, action.m_newName);
				}

				else if (m_actions.get(i) instanceof DSTransaction.DetachBlob) 
				{
					actionName = "DetachBlob";
					DSTransaction.DetachBlob action = (DSTransaction.DetachBlob) m_actions.get(i);
					if (m_permissionChecks != null)
                    {
                        m_permissionChecks.detachFSBlobCheck(action.m_delta.getIdentity().getName());
                    }
					m_ds.detachFSBlob(action.m_delta);
				}

			}
		} 
        catch (DirectoryServiceException dirE) 
		{
			String previousMessage = dirE.getMessage();
			String newMessage = "";
			if (actionName != null)
			{
				newMessage = actionName + " action failed in ClientTransaction";
				if (previousMessage != null && previousMessage.length() > 0)
                {
                    newMessage = newMessage + ": " + previousMessage;
                }

			}
            else
            {
                newMessage = previousMessage;
            }
			Exception linked = dirE.getLinkedException();
			DirectoryServiceException newE = new DirectoryServiceException(newMessage);
			newE.setLinkedException(linked);
            throw newE;
		}

    }

    //Here we track down the hierarchical configurations being created or deleted, so that we 
    // apply the same permission check to the sub configurations that get created or deleted. Though 
    // configurations can also be created through the clone and subclassing operations, neither the SMC 
    // nor the config API use the transaction clone and subclassing operations - the SMC uses create 
    // for the copied and subclassed elements, and the config API currently disallows clone of 
    // hierarchical configurations and the use of templates/subclasses. So here, we're only tracking
    // the CreateElement or the DeleteElement of a hierarchical config. We also keep track of DeleteFolder
    // because the SMC deletes hierarchical configs by deleting the folder and all folders and
    // configs under it. So, the delete of a broker from the SMC doesn't include a DeleteElement
    // for the broker itself, but a DeleteFolder. The API will have a DeleteElement for the broker.


    void searchForHierarchicalConfigs()
        throws DirectoryServiceException
    {
        if (m_permissionChecks != null && m_permissionChecks.isPermissionsCheckingEnabled())
        {
            for (int i = 0; i < m_actions.size(); i++)
            {

                if (m_actions.get(i) instanceof DSTransaction.CreateElement)
                {
                    DSTransaction.CreateElement action = (DSTransaction.CreateElement)m_actions.get(i);
                    IElementIdentity id = action.m_element.getIdentity();
                    if (m_ds.isHierarchicalConfig(id.getReleaseVersion(), id.getType()))
                        // add both the name with the suffix and the name without the suffix,
                        // so we can correctly identify the folder creation name and the element creation
                        // name as the hierarchical config and perform the correct permission check
                    {
                        String idString = id.getName();
                        m_hierarchicalCreates.add(idString);
                        m_hierarchicalCreates.add(idString.substring(0, idString.lastIndexOf(IMFDirectories.MF_DIR_SEPARATOR)));
                    }
                }
               /* else if (m_actions.get(i) instanceof DSTransaction.DeleteElement)
                {
                    DSTransaction.DeleteElement action = (DSTransaction.DeleteElement)m_actions.get(i);
                    String idString = action.m_elementName;
                    if (m_ds.isHierarchicalPath(idString))
                    {
                        m_hierarchicalDeletes.add(idString);
                        m_hierarchicalDeletes.add(idString.substring(0, idString.lastIndexOf(IMFDirectories.MF_DIR_SEPARATOR)));
                    }
                }*/
                else if (m_actions.get(i) instanceof DSTransaction.DeleteFolder)
                {
                    DSTransaction.DeleteFolder action = (DSTransaction.DeleteFolder)m_actions.get(i);
                    String idString = action.m_folderName;
                    if (m_ds.isHierarchicalPath(idString))
                    {
                        m_hierarchicalDeletes.add(idString);
                    }
                }
            }
        }
    }
    
    void generateStorageNames(int startIndex) throws DirectoryServiceException
    {
        for (int i = startIndex; i < m_actions.size(); i++)
        {
            try
            {
                if (m_actions.get(i) instanceof DSTransaction.CreateElement)
                {
                    DSTransaction.CreateElement action = (DSTransaction.CreateElement)m_actions.get(i);
                    action.m_newStorageName = m_ds.assignStorageName(action.m_element);
                }
                else if (m_actions.get(i) instanceof DSTransaction.CreateElements)
                {
                    DSTransaction.CreateElements action = (DSTransaction.CreateElements)m_actions.get(i);
                    action.m_newStorageNames = m_ds.assignStorageNames(action.m_elements, false);
                }
                else if (m_actions.get(i) instanceof DSTransaction.SubclassElement)
                {
                    DSTransaction.SubclassElement action = (DSTransaction.SubclassElement)m_actions.get(i);
                    action.m_newStorageName = m_ds.assignStorageName(action.m_newElementPath,action.m_delta.getIdentity().getType());
                }
                else if (m_actions.get(i) instanceof DSTransaction.CloneElement0)
                {
                    DSTransaction.CloneElement0 action = (DSTransaction.CloneElement0)m_actions.get(i);
                    action.m_newStorageName = m_ds.assignStorageName(action.m_newElementPath,action.m_delta.getIdentity().getType());
                }
                else if (m_actions.get(i) instanceof DSTransaction.CloneElement1)
                {
                    DSTransaction.CloneElement1 action = (DSTransaction.CloneElement1)m_actions.get(i);
                    IElementIdentity id = m_ds.getFSIdentity(action.m_sourcePath);
                    if (id == null)
                    {
                        // well, it might be that a previous action will define this element. throw a DirectoryServiceException
                        // and we can try to perform all the actions to here. Found this while testing the internal AppendBlob
                        // actions
                        throw new DirectoryServiceException("CloneElement source " + action.m_sourcePath + " has not been defined yet");
                    }
                    String type = id.getType();
                    action.m_newStorageName = m_ds.assignStorageName(action.m_newElementPath,type);
                }
                else if (m_actions.get(i) instanceof DSTransaction.AttachBlob)
                {
                    DSTransaction.AttachBlob action = (DSTransaction.AttachBlob)m_actions.get(i);
                    // No need to assign a new name if action.m_element is a delta - the element already exists
                    if ((action.m_element instanceof IDirElement) && (!m_ds.isHandledByHandler(action.m_element)))
                    {
                        action.m_newStorageName = m_ds.assignStorageName(action.m_element);
                    }
                }
                else if (m_actions.get(i) instanceof DSTransaction.AppendBlob)
                {
                    DSTransaction.AppendBlob action = (DSTransaction.AppendBlob)m_actions.get(i);
                    // No need to assign a new name if action.m_element is a delta - the element already exists
                    // If this append follows a series of incomplete appends, the storage name has been created already
                    // but has been kept in a temporary structure in memory and not in the view, so that a failed
                    // transaction doesn't endup with a corrupt view that points to a non existing element.
                    // if the action will be handled by a handler, don't assign a storage name
                    if (!m_ds.isHandledByHandler(action.m_element))
                    {
                        if ((action.m_element instanceof IDirElement) && (action.m_src == 0))
                        {
                            action.m_newStorageName = m_ds.assignStorageName(action.m_element);
                        }
                        else if ((action.m_element instanceof IDirElement) && (action.m_src != 0))
                        {
                            action.m_newStorageName = m_ds.tempBlobLogicalToStorage(action.m_element.getIdentity().getName());
                        }
                    }
                }
            }
            catch (DirectoryServiceException e)
            {
                m_namingFailedIndex = i;
                throw e;
            }
        }
        m_namingFailedIndex = m_actions.size();
    }

}
