/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */
package com.sonicsw.mf.framework.directory.impl;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.StringReader;
import java.net.URL;
import java.net.URLClassLoader;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import com.sonicsw.mx.config.util.SonicFSFile;
import com.sonicsw.mx.util.IEmptyArray;
import com.sonicsw.util.debug.Debug;

import com.sonicsw.mf.common.IComponent;
import com.sonicsw.mf.common.IDSTransaction;
import com.sonicsw.mf.common.IDirectoryCacheService;
import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IDSHandler;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.IElementChangeHandler;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IEnvelope;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.INameChangeHandler;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.config.IValidationElementChange;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.impl.Blob;
import com.sonicsw.mf.common.config.impl.CannotAdjustToSuperModificationException;
import com.sonicsw.mf.common.config.impl.DSTransaction;
import com.sonicsw.mf.common.config.impl.DeltaElement;
import com.sonicsw.mf.common.config.impl.DirIdentity;
import com.sonicsw.mf.common.config.impl.Element;
import com.sonicsw.mf.common.config.impl.ElementIdentity;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.config.impl.EnvelopeElement;
import com.sonicsw.mf.common.config.impl.ICanReplaceRef;
import com.sonicsw.mf.common.config.impl.IReplaceRef;
import com.sonicsw.mf.common.config.impl.Identity;
import com.sonicsw.mf.common.config.impl.NextVersionToken;
import com.sonicsw.mf.common.config.impl.VersionMisMatchException;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.config.query.ElementComparator;
import com.sonicsw.mf.common.config.query.EqualExpression;
import com.sonicsw.mf.common.config.query.From;
import com.sonicsw.mf.common.config.query.FromDirectory;
import com.sonicsw.mf.common.config.query.FromElementList;
import com.sonicsw.mf.common.config.query.FromElementType;
import com.sonicsw.mf.common.config.query.FromFolder;
import com.sonicsw.mf.common.config.query.OrderedBy;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.config.query.QueryBatch;
import com.sonicsw.mf.common.config.query.Select;
import com.sonicsw.mf.common.config.query.Where;
import com.sonicsw.mf.common.config.query.impl.ElementEvaluation;
import com.sonicsw.mf.common.config.query.impl.ElementSelection;
import com.sonicsw.mf.common.config.query.impl.ReferenceReplacer;
import com.sonicsw.mf.common.dirconfig.BackupAlreadyInProgress;
import com.sonicsw.mf.common.dirconfig.BackupPathExists;
import com.sonicsw.mf.common.dirconfig.BackupStatus;
import com.sonicsw.mf.common.dirconfig.DirectoryDoesNotExistException;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceClosedException;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.ElementInPathException;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.common.dirconfig.InvalidBackupPath;
import com.sonicsw.mf.common.dirconfig.InvalidPathException;
import com.sonicsw.mf.common.dirconfig.InvalidXMLException;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;
import com.sonicsw.mf.common.runtime.IBackupStatus;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.security.ConfigurePermissionDeniedException;
import com.sonicsw.mf.common.security.IConfigurePermissionBits;
import com.sonicsw.mf.common.security.IConfigureScopeBits;
import com.sonicsw.mf.common.security.IManagePermissionBits;
import com.sonicsw.mf.common.security.IManageScopeBits;
import com.sonicsw.mf.common.security.IManagementPermission;
import com.sonicsw.mf.common.security.InvalidManagementPermissionException;
import com.sonicsw.mf.common.security.ManagementPermissionDeniedException;
import com.sonicsw.mf.common.security.ManagementPermissionFactory;
import com.sonicsw.mf.common.view.IDeltaView;
import com.sonicsw.mf.common.view.ILink;
import com.sonicsw.mf.common.view.ILogicalNameSpace;
import com.sonicsw.mf.common.view.INamingListener;
import com.sonicsw.mf.common.view.IView;
import com.sonicsw.mf.common.view.ViewException;
import com.sonicsw.mf.common.view.impl.DeltaView;
import com.sonicsw.mf.common.view.impl.LogicalNameSpace;
import com.sonicsw.mf.common.view.impl.View;
import com.sonicsw.mf.common.view.impl.ViewConstants;
import com.sonicsw.mf.common.xml.DirectoryBuilder;
import com.sonicsw.mf.common.xml.ElementBuilder;
import com.sonicsw.mf.common.xml.ElementListBuilder;
import com.sonicsw.mf.common.xml.Validator;
import com.sonicsw.mf.common.xml.XMLStringWriter;
import com.sonicsw.mf.framework.IAuditManager;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.IPermissionsManager;
import com.sonicsw.mf.framework.agent.TaskScheduler;
import com.sonicsw.mf.framework.directory.Convert5to6;
import com.sonicsw.mf.framework.directory.DSComponent;
import com.sonicsw.mf.framework.directory.IClassLoaderUtility;
import com.sonicsw.mf.framework.directory.IDebuggingMasks;
import com.sonicsw.mf.framework.directory.IDirectoryMFService;
import com.sonicsw.mf.framework.directory.IDirectoryService;
import com.sonicsw.mf.framework.directory.IFSStorage;
import com.sonicsw.mf.framework.directory.ILogger;
import com.sonicsw.mf.framework.directory.IPersistSubscribers;
import com.sonicsw.mf.framework.directory.RepairNoStorageReferences;
import com.sonicsw.mf.framework.directory.SizedArrayList;
import com.sonicsw.mf.framework.directory.storage.DSEncryptionException;
import com.sonicsw.mf.framework.directory.storage.IStorage;
import com.sonicsw.mf.framework.directory.storage.PackedDirUtil;
import com.sonicsw.mf.framework.directory.storage.ParentDirDoesNotExistException;
import com.sonicsw.mf.framework.directory.storage.StorageException;
import com.sonicsw.mf.framework.directory.storage.fs.FSStorage;
import com.sonicsw.mf.framework.directory.storage.pse.PSEStorage;
import com.sonicsw.mf.framework.elementversion.IArrayElementNotificationListener;
import com.sonicsw.mf.framework.elementversion.INotificationConsumer;
import com.sonicsw.mf.framework.elementversion.NotificationConsumer;
import com.sonicsw.mf.mgmtapi.config.constants.IAuthenticationGroupConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IAuthenticationUserConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IBackupDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDomainConstants;

public class DirectoryService
implements IDirectoryService, IPersistSubscribers, IDirectoryCacheService, IDebuggingMasks, ILogger
{
    public static final String SYSTEM_DIRECTORY_PATH = IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_SYSTEM_DIR;

    public static final int DS_STORAGE_STRUCTURE_VERSION = 8;

    public static final int DS_API_VERSION = 4;

    private static final String VIRTUAL_DIRECTORY = "_MFVirtual"; // To access data that should be retrieved from

    // storage

    private static final String VIRTUAL_DIRECTORY_PATH = IMFDirectories.MF_DIR_SEPARATOR + VIRTUAL_DIRECTORY;

    public static final String DOMAIN_PATH = IMFDirectories.MF_DIR_SEPARATOR + "domain";

    public static final String PERMISSIONS_PATH = IMFDirectories.MF_DIR_SEPARATOR + "permissions";

    private static final String DS_VERSION_INFO = "ds_version_info";

    private static final String BLOB_COPIES_DIR = "blob_copies";

    private static final String BLOB_COPIES_PREFIX = "_MFCP";

    static final String MF_CONFING_VERSION = "100";

    public static final String LOCK_ELEMENT = "lock";

    public static final String BACKUP_ELEMENT = "backup_state";

    public static final String OPEN_ELEMENT = "dsisopen";

    public static final String SUBSCRIBERS_ELEMENT = "subscribers";

    public static final String SUBSCRIBERS_COLLECTION_NAME = "ds_subscribers";

    public static final String BACKUP_VERSION_ELEMENT = "backup_version";

    public static final String BACKUP_STATUS_ELEMENT = "backup_status";

    public static final String BACKUP_STATUS_ELEMENT_PATH = SYSTEM_DIRECTORY_PATH + IMFDirectories.MF_DIR_SEPARATOR + BACKUP_STATUS_ELEMENT;

    public static final String BACKUP_VERSION_ELEMENT_PATH = SYSTEM_DIRECTORY_PATH + IMFDirectories.MF_DIR_SEPARATOR + BACKUP_VERSION_ELEMENT;

    public static final String DS_VERSION_INFO_PATH = VIRTUAL_DIRECTORY_PATH + IMFDirectories.MF_DIR_SEPARATOR + DS_VERSION_INFO;

    public static final String VERSION_ELEMENT = "version";

    public static final String VERSION_ELEMENT_PATH = SYSTEM_DIRECTORY_PATH + IMFDirectories.MF_DIR_SEPARATOR + VERSION_ELEMENT;

    public static final String VIEW_ELEMENT = IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_SCHEMA_DIR + IMFDirectories.MF_DIR_SEPARATOR + "view";

    static final String STORAGE_HINTS_ELEMENT = IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_SCHEMA_DIR + IMFDirectories.MF_DIR_SEPARATOR + "storage_hints";

    private static final String HINTS_ATT = "HINTS_ATTRIBUTE";

    private static final String LOCK_ELEMENT_PATH = SYSTEM_DIRECTORY_PATH + IMFDirectories.MF_DIR_SEPARATOR + LOCK_ELEMENT;

    private static final String BACKUP_ELEMENT_PATH = SYSTEM_DIRECTORY_PATH + IMFDirectories.MF_DIR_SEPARATOR + BACKUP_ELEMENT;

    private static final String OPEN_ELEMENT_PATH = SYSTEM_DIRECTORY_PATH + IMFDirectories.MF_DIR_SEPARATOR + OPEN_ELEMENT;

    public static final String IDCACHE_ELEMENT = SYSTEM_DIRECTORY_PATH + IMFDirectories.MF_DIR_SEPARATOR + "idcache";

    private static final String SUBSCRIBERS_ELEMENT_PATH = SYSTEM_DIRECTORY_PATH + IMFDirectories.MF_DIR_SEPARATOR + SUBSCRIBERS_ELEMENT;

    private static final String SUSPEND_NOTIFICATIONS_ELEMENT_PATH = SYSTEM_DIRECTORY_PATH + IMFDirectories.MF_DIR_SEPARATOR + "suspend_notifications";

    private static final String RESERVED_CHARACTERS = "\\|";

    private static final String DOMAIN_ELEMENT_PATH = DOMAIN_PATH + IMFDirectories.MF_DIR_SEPARATOR + "domain";

    private static final String MANAGE_PERMISSIONS_PATH = PERMISSIONS_PATH + IMFDirectories.MF_DIR_SEPARATOR + "manage";

    private static final String CONFIGURE_PERMISSIONS_PATH = PERMISSIONS_PATH + IMFDirectories.MF_DIR_SEPARATOR + "configure";

    private static EntityName VIEW_ELEMENT_ENTITY_NAME = null;

    private static EntityName SYSTEM_DIRECTORY_PATH_ENTITY_NAME = null;

    private static EntityName LOCK_ELEMENT_ENTITY_NAME = null;

    private static EntityName BACKUP_ELEMENT_ENTITY_NAME = null;

    private static EntityName OPEN_ELEMENT_ENTITY_NAME = null;

    private static EntityName VERSION_ELEMENT_ENTITY_NAME = null;

    private static EntityName IDCACHE_ELEMENT_ENTITY_NAME = null;

    private static EntityName SUBSCRIBERS_ELEMENT_ENTITY_NAME = null;

    private static EntityName BACKUP_VERSION_ELEMENT_ENTITY_NAME = null;

    private static EntityName BACKUP_STATUS_ELEMENT_ENTITY_NAME = null;

    private static EntityName DOMAIN_PATH_ENTITY_NAME = null;

    private static EntityName DOMAIN_ELEMENT_ENTITY_NAME = null;

    private static EntityName PERMISSIONS_PATH_ENTITY_NAME = null;

    private static EntityName MANAGE_PERMISSIONS_ELEMENT_ENTITY_NAME = null;

    private static EntityName CONFIGURE_PERMISSIONS_ELEMENT_ENTITY_NAME = null;

    private static final int PERMISSION_CONFIGURE_FOLDER = 0;

    private static final int PERMISSION_CONFIGURE_ELEMENT = 1;

    private static final int PERMISSION_MANAGE_FOLDER = 2;

    private static final int PERMISSION_MANAGE_CONTAINER = 3;

    private static final int PERMISSION_MANAGE_COMPONENT = 4;

    private static EntityName SUSPEND_NOTIFICATIONS_ENTITY_NAME = null;

    // for the permissions API
    private static final String GLOBAL_PERMISSIONS_GROUP_NAME = "Administrators";

    private static final String ROOT_DIRECTORY = "/";

    private static final String PERMISSIONS_VALUE_ATTRIBUTE_NAME = "VALUE";

    private static final String DOMAIN_ELEMENT_NAME = "/domain/domain";

    static
    {
        try
        {
            VIEW_ELEMENT_ENTITY_NAME = new EntityName(VIEW_ELEMENT);
            SYSTEM_DIRECTORY_PATH_ENTITY_NAME = new EntityName(SYSTEM_DIRECTORY_PATH);
            LOCK_ELEMENT_ENTITY_NAME = new EntityName(LOCK_ELEMENT_PATH);
            BACKUP_ELEMENT_ENTITY_NAME = new EntityName(BACKUP_ELEMENT_PATH);
            OPEN_ELEMENT_ENTITY_NAME = new EntityName(OPEN_ELEMENT_PATH);
            VERSION_ELEMENT_ENTITY_NAME = new EntityName(VERSION_ELEMENT_PATH);
            IDCACHE_ELEMENT_ENTITY_NAME = new EntityName(IDCACHE_ELEMENT);
            SUBSCRIBERS_ELEMENT_ENTITY_NAME = new EntityName(SUBSCRIBERS_ELEMENT_PATH);
            BACKUP_VERSION_ELEMENT_ENTITY_NAME = new EntityName(BACKUP_VERSION_ELEMENT_PATH);
            BACKUP_STATUS_ELEMENT_ENTITY_NAME = new EntityName(BACKUP_STATUS_ELEMENT_PATH);
            DOMAIN_PATH_ENTITY_NAME = new EntityName(DOMAIN_PATH);
            DOMAIN_ELEMENT_ENTITY_NAME = new EntityName(DOMAIN_ELEMENT_PATH);
            PERMISSIONS_PATH_ENTITY_NAME = new EntityName(PERMISSIONS_PATH);
            MANAGE_PERMISSIONS_ELEMENT_ENTITY_NAME = new EntityName(MANAGE_PERMISSIONS_PATH);
            CONFIGURE_PERMISSIONS_ELEMENT_ENTITY_NAME = new EntityName(CONFIGURE_PERMISSIONS_PATH);
            SUSPEND_NOTIFICATIONS_ENTITY_NAME = new EntityName(SUSPEND_NOTIFICATIONS_ELEMENT_PATH);
        }
        catch (ConfigException e)
        {
            e.printStackTrace();
            throw new Error(e.toString());
        }
    }

    // For v5 to v6 conversion
    private static final String POLICIES_DIRECTORY = "/policies";
    private static final ThreadLocal<SimpleDateFormat> DATE_PARSER_THREAD_LOCAL = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat("yy/MM/dd HH:mm:ss");
        }
    };

    private String m_domain;

    private File m_domainDir;

    private File m_blobCopiesDir;

    private IStorage m_storage;

    private IStorage m_blobStorage;

    private IStorage m_systemStorage;

    private Hashtable m_startup_properties;

    private boolean m_open;

    private IDCache m_idCache;

    private com.sonicsw.mf.common.ILogger m_logger;

    private Object m_subscribers_lock = new Object();

    private ImportManager m_importManager = null;

    private NotificationManager m_notificationManager = null;

    private long m_backupVersion = 0;

    private String m_hostDir;

    private TriggerValidator m_validator;

    private DSHandlers m_dsHandlers;

    private IClassLoaderUtility m_loader;

    private IFrameworkComponentContext m_context = null;

    private AuthenticationConfigManager m_authentication = null;

    private int m_traceMask;

    private LocalListener m_localListener = null;

    private boolean m_noContainer;

    private boolean m_ciFirstPhase;

    private LogicalNameSpace m_logicalNameSpace;

    private boolean m_FSInterfaceIsUsed;

    private long m_copySequenceNum;

    private String m_storageType;

    private boolean m_useFS_STORAGE, m_usePSE_STORAGE;

    TransactionManager m_trManager = null;

    ReadWriteLock m_lock;

    BackReferenceMgr m_backRefMgr = null;

    BackupStatus m_backupStatus = new BackupStatus();

    HashMap m_hierarchicalTypes = null;

    HashMap m_suspendNotifications = null;

    private int m_zipFileCounter = 0;
    
    private static volatile StandaloneURLStreamHandlerFactoryImpl m_URLHandlerFactory = null;
    private static Object LOCK_OBJ = new Object();

    public DirectoryService(Hashtable properties, String domain, com.sonicsw.mf.common.ILogger logger)
    throws DirectoryServiceException
    {
        
        this(properties, domain, logger, true, null, null);
    }

    public DirectoryService(Hashtable properties, String domain, com.sonicsw.mf.common.ILogger logger,
                            boolean noContainer, IClassLoaderUtility loader, IFrameworkComponentContext context)
    throws DirectoryServiceException
    {
        m_startup_properties = properties;
        m_domain = domain;

        Boolean isRestrictedBackupDSProperty = (Boolean)properties.get(IDirectoryMFService.IS_RESTRICTED_BACKUP);
        boolean isRestrictedBackupDS = isRestrictedBackupDSProperty == null ? false : isRestrictedBackupDSProperty.booleanValue();
        
        Boolean tmp = (Boolean)properties.get(IDirectoryMFService.CI_FIRST_PHASE_ATTRIBUTE);
        if (tmp != null && tmp.booleanValue())
        {
            m_ciFirstPhase = true;
        }
        else
        {
            m_ciFirstPhase = false;
        }
        m_lock = new ReadWriteLock(isRestrictedBackupDS);
        m_notificationManager = new NotificationManager(this);
        m_logger = logger;
        m_loader = loader;
        m_context = context;
        m_noContainer = noContainer;
        m_logicalNameSpace = null;
        m_FSInterfaceIsUsed = false;
        m_copySequenceNum = 0;
        Integer traceMaskObject = (Integer)properties.get(IDirectoryService.TRACE_MASK_ATTRIBUTE);
        m_traceMask = (traceMaskObject == null) ? 0 : traceMaskObject.intValue();
        m_storageType = (String)properties.get(IDirectoryService.STORAGE_TYPE_ATTRIBUTE);
        m_useFS_STORAGE = m_storageType.equals(IDirectoryService.FS_STORAGE);
        m_usePSE_STORAGE = m_storageType.equals(IDirectoryService.PSE_STORAGE);
        Boolean testAuthentication = (Boolean)properties.get(IDirectoryService.TEST_AUTHENTICATION_ATTRIBUTE);
        if (m_storageType == null || (!m_useFS_STORAGE && !m_usePSE_STORAGE))
        {
            throw new DirectoryServiceException("Only FILE_SYSTEM_STORAGE and PSE_STORAGE storage types are currently supported.");
        }

        String hostDirDepricated = (String)properties.get(IFSStorage.FS_HOST_DIRECTORY_ATTRIBUTE);
        String hostDir = (String)properties.get(IFSStorage.HOST_DIRECTORY_ATTRIBUTE);
        trace(TRACE_STARTUP, "STARTUP: " + properties.toString());

        if (hostDirDepricated != null)
        {
            logMessage("The HOST_DIRECTORY attribute under the FILE_SYSTEM_STORAGE attribute set is deprecated", Level.WARNING);
            if (hostDir != null)
            {
                throw new DirectoryServiceException("Found two HOST_DIRECTORY attributes. Please remove the one under " + "the FILE_SYSTEM_STORAGE attribute set.");
            }
            else
            {
                hostDir = hostDirDepricated;
            }
        }

        m_hostDir = hostDir;

        createDomainDir(hostDir, domain, noContainer);
        cleanupBlobCopies();
        String password = (String)properties.get(IFSStorage.PASSWORD_ATTRIBUTE);
        try
        {
            if (m_useFS_STORAGE)
            {
                if ((password != null) && (password.length() != 0))
                {
                    m_storage = new FSStorage(hostDir, domain, IMFDirectories.MF_DATA_DIR, password, true);
                }
                else
                {
                    m_storage = new FSStorage(hostDir, domain, IMFDirectories.MF_DATA_DIR, true);
                }

                // Having a seperate FSStorage object allows us to use it concurrently with regular data
                // transactions. Note that m_systemStorage is not passed to the transaction manager so it doesn't
                // participate in transactions.
                m_systemStorage = new FSStorage(hostDir, domain, IMFDirectories.MF_DATA_DIR);
                m_blobStorage = new FSStorage(hostDir, domain, IMFDirectories.MF_BLOBS_DIR, true);
            }

            else
            // PSEStorage only for now
            {
                // extract fault tolerant attributes for PSEStorage
                //
                HashMap ftProps = new HashMap(m_startup_properties);
                if ((password != null) && (password.length() != 0))
                {
                    m_storage = new PSEStorage(hostDir, domain, IMFDirectories.MF_DATA_DIR, password, ftProps);
                }
                else
                {
                    m_storage = new PSEStorage(hostDir, domain, IMFDirectories.MF_DATA_DIR, null, ftProps);
                }
                m_systemStorage = new PSEStorage((PSEStorage)m_storage);
                m_systemStorage.createCollectionIfNotCreated(SUBSCRIBERS_COLLECTION_NAME);
                m_blobStorage = m_storage;
            }

            m_storage.setLogger(this);
            m_systemStorage.setLogger(this);
            m_blobStorage.setLogger(this);
            openStorage();

            m_open = true;
            createSchemaElements(isRestrictedBackupDS);
            createJNDIContext();
            initLogicalNameSpace();
            initSuspendNotifications();
        }
        catch (StorageException e)
        {
            close();
            throw convertException(e);
        }
        catch (DirectoryServiceException e)
        {
            boolean normalClose = (e instanceof DirLockedException || e instanceof DecryptionException) ? false : true;
            close(normalClose);

            throw e;
        }

        if (!m_ciFirstPhase)
        {
            m_validator = new TriggerValidator(this);
            // cleanupBlobCopies has to be called ahead of this, so the handler for
            // the blob copies directory can be initialized properly.
            m_dsHandlers = new DSHandlers(this);
        }
        trace(TRACE_STARTUP, "Created the trigger validator.");

        if (!noContainer || (testAuthentication != null && testAuthentication.booleanValue()))
        {
            m_authentication = new AuthenticationConfigManager(this, this, isRestrictedBackupDS);
            trace(TRACE_STARTUP, "Created the authentication manager.");
        }
        
        if (noContainer) // if no container we still need a URL stream handler factory for sonicrn:/// handling
        {
         // The URL handler can only be set once in a jvm, but SDM opens and closes instances of the DS
            // several times within one JVM. Therefore, we need to refresh the DS instance of the handler without
            // resetting the handler. 
            try
            {
                if (m_URLHandlerFactory == null)
                {
                	synchronized (LOCK_OBJ) 
                	{
                        if (m_URLHandlerFactory == null)
                        {
                        	m_URLHandlerFactory = new StandaloneURLStreamHandlerFactoryImpl(this);
                        	URL.setURLStreamHandlerFactory(m_URLHandlerFactory);
                        }
					}
                }
                else
                {
                    m_URLHandlerFactory.setDirectoryService(this);
                }
                    
            }
            catch(Throwable e)
            {
                logMessage("Failed to set URLStreamHandlerFactory, trace follows...", e, Level.SEVERE);
            }
        }
    }

    // This routine is designed to prevent any possibility of skipping the m_lock.releaseLock() call
    private void leaveTransactionAndReleaseLock(boolean joinedTransaction, boolean transactOK)
    throws DirectoryServiceException
    {
        DirectoryServiceException commitException = null;
        Throwable otherThrowable = null;

        if (joinedTransaction)
        {
            try
            {
                m_trManager.leave(transactOK);
            }
            catch (DirectoryServiceException e)
            {
                commitException = e;
            }
            catch (Throwable t)
            {
                otherThrowable = t;
            }
        }

        m_lock.releaseLock();
        if (commitException != null)
        {
            throw commitException;
        }
        if (otherThrowable != null)
        {
            if (otherThrowable instanceof Error)
            {
                throw (Error)otherThrowable;
            }
            if (otherThrowable instanceof RuntimeException)
            {
                throw (RuntimeException)otherThrowable;
            }
            else
            {
                // Cannot really happen but we don't want to take any risk of "loosing" and exception
                throw new Error(otherThrowable.toString());
            }
        }
    }

    // Silent coverion of complex configuration links.
    private void convertLogicalNamesV5toV6()
    {
        try
        {
            IView view = getView();
            boolean viewMod1 = convertLogicalNamesV5toV6(view, AuthenticationConfigManager.DOMAINS_DIRECTORY, AuthenticationConfigManager.DOMAIN_DESCRIPTOR);
            boolean viewMod2 = convertLogicalNamesV5toV6(view, POLICIES_DIRECTORY, null);

            if (viewMod1 || viewMod2)
            {
                setView(view.doneUpdate());
            }
        }
        catch (Exception e) // Should never happen
        {
            trace(TRACE_ALL_DS_ACCESS, "convertLogicalNamesV5toV6 failed, trace follows...", e);
        }
    }

    private boolean convertLogicalNamesV5toV6(IView view, String topDirName, String descriptorBaseName)
    throws Exception
    {
        if (directoryExists(topDirName))
        {
            IDirIdentity[] domainIDs = listDirectories(topDirName);
            boolean viewHasModified = false;
            for (int i = 0; i < domainIDs.length; i++)
            {
                EntityName dirNameE = new EntityName(domainIDs[i].getName());
                if (descriptorBaseName == null)
                {
                    descriptorBaseName = dirNameE.getBaseName();
                }
                String descriptorName = dirNameE.getName() + IMFDirectories.MF_DIR_SEPARATOR + descriptorBaseName;

                String[] logicalNames = view.lookupLink(descriptorName);
                if (logicalNames.length == 0)
                {
                    continue;
                }

                // logicalNames can contain only a single item
                ILink link = (ILink)view.getElement(logicalNames[0]);

                // Allready converted ?
                if (link.isComplex())
                {
                    return false;
                }
                else
                {
                    viewHasModified = true;
                    link.setComplex();
                }
            }
            return viewHasModified;

        }
        else
        {
            return false;
        }
    }

    private void initLogicalNameSpace()
    throws DirectoryServiceException
    {
        m_logicalNameSpace = new LogicalNameSpace();
        m_logicalNameSpace.init(this, STORAGE_HINTS_ELEMENT, HINTS_ATT);
        m_trManager.setLogicalNameSpace(m_logicalNameSpace);
    }

    @Override
    public int getDirectoryServiceVersion()
    {
        return DS_API_VERSION;
    }

    @Override
    public String getDirectoryServiceReleaseVersion()
    {
        return com.sonicsw.mf.common.Version.getReleaseName();
    }

    @Override
    public void setTraceMask(int traceMask)
    {
        m_traceMask = traceMask;
    }

    void runValidationTriggers(ArrayList modificationList, boolean isImportTransaction)
    throws DirectoryServiceException
    {
      
        if (m_validator != null)
        {
            m_validator.validate(modificationList, isImportTransaction);
        }

        IValidationElementChange[] changes = (IValidationElementChange[])modificationList.toArray(new IValidationElementChange[0]);
        if (m_backRefMgr != null)
        {
            m_backRefMgr.updateBackReferences(changes);
        }
    }

    void runOnDeleteTriggers(ArrayList modificationList)
    throws DirectoryServiceException
    {
        if (m_validator != null)
        {
            m_validator.runOnDeleteTriggers(modificationList);
        }
    }
    
    void runAfterDeleteTriggers(ArrayList modificationList)
    throws DirectoryServiceException
    {
        if (m_validator != null)
        {
            m_validator.runAfterDeleteTriggers(modificationList);
        }
    }
    
    void runOnUpdateTriggers(ArrayList modificationList) throws DirectoryServiceException
    {
        if (m_validator != null)
        {
            m_validator.runOnUpdateTriggers(modificationList);
        }
    }
    
    void runAfterUpdateTriggers(ArrayList modificationList) throws DirectoryServiceException
    {
        if (m_validator != null)
        {
            m_validator.runAfterUpdateTriggers(modificationList);
        }
    }
    
    void runOnCreateTriggers(ArrayList modificationList) throws DirectoryServiceException
    {
        if (m_validator != null)
        {
            m_validator.runOnCreateTriggers(modificationList);
        }
    }

    void runAfterCreateTriggers(ArrayList modificationList) throws DirectoryServiceException
    {
        if (m_validator != null)
        {
            m_validator.runAfterCreateTriggers(modificationList);
        }
    }

    boolean hasConsumer()
    {
        return m_notificationManager.hasConsumer();
    }

    // Create a domain directory if one doesn't exist
    private void createDomainDir(String hostDirName, String domainName, boolean noContainer)
    throws DirectoryServiceException
    {
        if (domainName == null || domainName.length() == 0)
        {
            throw new DirectoryServiceException("A domain name is missing.");
        }

        if (hostDirName == null || hostDirName.length() == 0)
        {
            throw new DirectoryServiceException("The HOST_DIRECTORY attribute is missing.");
        }

        File hostDir = new File(hostDirName);
        if (!hostDir.exists())
        {
            throw new DirectoryServiceException("Directory \"" + hostDirName + "\", does not exist.");
        }

        m_domainDir = new File(hostDir, domainName);

        if (!m_domainDir.exists())
        {
            if (!noContainer || m_ciFirstPhase)
            {
                throw new DirectoryServiceException("Domain directory \"" + m_domainDir.getPath() + "\" was not found.");
            }
            if (!m_domainDir.mkdir())
            {
                throw new DirectoryServiceException("Cannot create the root directory \"" + m_domainDir.getPath() + "\".");
            }
        }

        if (!m_domainDir.canWrite())
        {
            throw new DirectoryServiceException("No write permission in directory \"" + m_domainDir.getPath() + "\".");
        }
    }

    // Cleanup copies of blob files used by the DS the session before. Those files are typically validator
    // or DSHandler JAR files - the DS cannot use these stored blob directly becuase applications will not be able
    // to remove them, if needed, while used.
    private void cleanupBlobCopies()
    throws DirectoryServiceException
    {
        m_blobCopiesDir = new File(m_domainDir, BLOB_COPIES_DIR);
        if (!m_blobCopiesDir.exists())
        {
            if (!m_blobCopiesDir.mkdir())
            {
                throw new DirectoryServiceException("Cannot create the directory \"" + getCanonicalPath(m_blobCopiesDir) + "\".");
            }
        }

        File[] deletionList = m_blobCopiesDir.listFiles();
        if (deletionList == null)
        {
            throw new DirectoryServiceException(getCanonicalPath(m_blobCopiesDir) + " is corrupt. Remove it and restart the program.");
        }
        for (int i = 0; i < deletionList.length; i++)
        {
            deletionList[i].delete();
        }
    }

    @Override
    public String getDomainDirectoryName()
    {
        if (m_domainDir != null)
        {
            return getCanonicalPath(m_domainDir);
        }
        else
        {
            return "?";
        }
    }

    String getBlobCopiesDir()
    {
        return getCanonicalPath(m_blobCopiesDir);
    }

    @Override
    public String getDomain()
    {
        return m_domain;
    }

    // Subclasses elements are realized by applying the subclassing delta onto the super element. Super eelements
    // are returned without the deltas of their subclassed elements
    @Override
    public IDirElement getElement(String elementName, boolean forUpdate, boolean getSubclassingDelta)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getElement " + elementName + " for update " + forUpdate);

        if (elementName.startsWith(VIRTUAL_DIRECTORY_PATH))
        {
            if (forUpdate)
            {
                throw new DirectoryServiceException("Cannot update virtual data.");
            }

            String nameComp[] = validateName(elementName).getNameComponents();
            if (nameComp.length > 1 && nameComp[0].equals(VIRTUAL_DIRECTORY) && nameComp[1].equals(DS_VERSION_INFO))
            {
                IDirElement infoElement = ElementFactory.createElement(DS_VERSION_INFO_PATH, "ds_info", MF_CONFING_VERSION);
                IAttributeSet attributes = infoElement.getAttributes();
                try
                {
                    attributes.setLongAttribute(IDirectoryMFService.BACKUP_VERSION_ATTR, new Long(m_backupVersion));
                }
                // These exceptions should never happen
                catch (ReadOnlyException e)
                {
                    throw new Error(e.toString());
                }
                catch (AttributeSetTypeException e)
                {
                    throw new Error(e.toString());
                }
                catch (ConfigException e)
                {
                    throw new Error(e.toString());
                }

                return infoElement;
            }
            else
            {
                return null;
            }

        }

        return getElementInternal(elementName, forUpdate, false, getSubclassingDelta);

    }

    @Override
    public IDirElement getElement(String elementName, boolean forUpdate)
    throws DirectoryServiceException
    {
        return getElement(elementName, forUpdate, false);
    }

    // Get the element as stored - not sublassing transformations applied
    public IDirElement getElementAsIs(String elementName, boolean forUpdate)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getElementAsIs " + elementName + " for update " + forUpdate);

        return getElementInternal(elementName, forUpdate, true, false);
    }

    @Override
    public IDirElement[] getElements(Query query, boolean forUpdate)
    throws DirectoryServiceException
    {
        return getElements(query, forUpdate, false);
    }

    @Override
    public IDirElement[] getElements(Query query, boolean forUpdate, boolean getSubclassingDelta)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getElements " + query + " for update " + forUpdate);

        if (query == null)
        {
            throw new DirectoryServiceException("The query parameter of the getElements call cannot be null.");
        }

        if ((m_traceMask & IComponent.TRACE_VERBOSE) > 0)
        {
            trace(TRACE_ALL_DS_ACCESS, query.toString());
        }

        Select select = query.getSelect();

        if (select != null && forUpdate)
        {
            throw new DirectoryServiceException("Elements from a query that contains a SELECT clause must be fetched for read-only.");
        }

        From from = query.getFrom();
        if (from == null)
        {
            throw new DirectoryServiceException("The query must contain a FROM clause.");
        }

        try
        {
            m_lock.readLock();

            String[] elementList = null;

            if (from instanceof FromDirectory)
            {
                IElementIdentity[] ids = listElements(((FromDirectory)from).getDirectoryName());
                elementList = new String[ids.length];
                for (int i = 0; i < elementList.length; i++)
                {
                    elementList[i] = ids[i].getName();
                }
            }
            else if (from instanceof FromElementType)
            {
                throw new DirectoryServiceException("'From Type' queries are not supported by the IDirectoryAdminService interface.");
            }
            else
            {
                elementList = ((FromElementList)from).getElementNames();
            }

            Where where = query.getWhere();

            ArrayList qualifiedElement = new ArrayList();
            HashMap alreadyConsideredTable = new HashMap();
            ElementSelection selector = null;
            if (select != null)
            {
                selector = new ElementSelection(select);
            }
            for (int i = 0; i < elementList.length; i++)
            {
                // Maybe was fetched before together with another candidate
                if (alreadyConsideredTable.get(elementList[i]) != null)
                {
                    continue;
                }

                // If we select then we'll need to modify the element to remove unwanted attributes
                boolean getForUpdate = select != null ? true : forUpdate;

                // Check the elementList[i] candidate and any other candidate that can be retrieved without
                // extra I/O
                IDirElement[] candidates = getElements(elementList[i], getForUpdate, getSubclassingDelta);

                for (int j = 0; j < candidates.length; j++)
                {
                    alreadyConsideredTable.put(candidates[j].getIdentity().getName(), Boolean.TRUE);
                    addQueryCandidate(candidates[j], qualifiedElement, where, selector);
                }
            }

            IDirElement[] result = new IDirElement[qualifiedElement.size()];

            for (int i = 0; i < qualifiedElement.size(); i++)
            {
                result[i] = (IDirElement)qualifiedElement.get(i);
            }

            if ((m_traceMask & IComponent.TRACE_VERBOSE) > 0)
            {
                for (int i = 0; i < result.length; i++)
                {
                    trace(TRACE_ALL_DS_ACCESS, result[i].toString());
                }
            }

            return result;

        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    private void addQueryCandidate(IDirElement candidate, ArrayList selectedList, Where where, ElementSelection selector)
    {
        if (where == null || (where != null && ElementEvaluation.evaluate(candidate, where)))
        {
            if (selector != null)
            {
                try
                {
                    selector.removeUnselectedAttributes(candidate);
                }
                catch (Exception e) // Should never happen
                {
                    logMessage("Failed, trace follows...", e, Level.WARNING);
                    throw new Error(e.toString());
                }
                candidate = (IDirElement)((Element)candidate).createClone(); // to remove delta history
                ((Element)candidate).setReadOnly(true);
            }
            selectedList.add(candidate);
        }
    }

    @Override
    public HashMap getStorageToLogicalMap()
    {
        trace(TRACE_ALL_DS_ACCESS, "getStorageToLogicalMap");

        try
        {
            m_lock.readLock();
            return m_logicalNameSpace.getStorageToLogicalMap();

        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    @Override
    public void newTriggerValidator()
    {
        trace(TRACE_ALL_DS_ACCESS, "newTriggerValidator");

        try
        {
            m_lock.readLock();
            m_validator = new TriggerValidator(this);

        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    @Override
    public void newDSHandlers()
    {
        trace(TRACE_ALL_DS_ACCESS, "newDSHandlers");

        try
        {
            m_lock.readLock();
            m_dsHandlers = new DSHandlers(this);

        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    // Used only by version upgrade utilities
    @Override
    public void upgrade5to6(String configType, String configLogicalPath)
    throws DirectoryServiceException
    {

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            Convert5to6.doUpgrade5to6(this, configType, configLogicalPath);
            transactOK = true;

        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.getMessage());
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    // Used only by version upgrade utilities
    @Override
    public void setElementReleaseVersion(String elementName, String newVersion)
    throws DirectoryServiceException
    {
        EntityName eName = validateName(elementName);

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            IDirElement element = m_storage.getElement(eName);
            ElementIdentity id = (ElementIdentity)element.getIdentity();
            id.setType(id.getType(), newVersion);
            storeInternal(eName, element);

            // Add the newVersion the the element's SYSTEM_ATTRIBUTES to generate a notification
            String logicalName = null;
            try
            {
                logicalName = m_logicalNameSpace.logicalFromStorage(eName.getName());
                // Silently fail to update (and generate a notification) if the element doesn't have a logical
                // name since tools only care about elements with logical names
                if (logicalName != null)
                {
                    Element logicalElement = (Element)getFSElement(logicalName, true);
                    logicalElement.addReleaseVersion(newVersion);
                    updateFSElement((IDeltaDirElement)logicalElement.doneUpdate());
                }
            }
            catch (Exception e)
            {
            }

            transactOK = true;

        }
        catch (StorageException e)
        {
            throw convertException(e);
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    private IDirElement getElementInternal(String elementName, boolean forUpdate, boolean asIs, boolean attachDelta)
    throws DirectoryServiceException
    {
        try
        {
            m_lock.readLock();

            validateOpen();
            EntityName eName = validateName(elementName);
            if (m_dsHandlers != null)
            {
                IDSHandler handler = m_dsHandlers.getHandler(eName);
                if (handler != null)
                {
                    return m_dsHandlers.getElement(handler, elementName);
                }
            }

            // First check in memory, if the element does not exist then we don't even check the storage
            // hence being more efficient in cases many clients are checking for elements that might not
            // exist
            if (m_idCache.get(eName) == null)
            {
                return null;
            }

            try
            {
                return prepareForShipment(m_storage.getElement(eName), forUpdate, asIs, attachDelta);
            }
            catch (StorageException e)
            {
                throw convertException(e);
            }
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    @Override
    public IDirElement[] getElements(String elementName, boolean forUpdate)
    throws DirectoryServiceException
    {
        return getElements(elementName, forUpdate, false);
    }

    public IDirElement[] getElements(String elementName, boolean forUpdate, boolean getSubclassingDelta)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getElements " + elementName + " for update " + forUpdate);

        validateOpen();
        EntityName eName = validateName(elementName);
        if (PackedDirUtil.underPackedDir(eName))
        {
            return getElementsAsIs(eName, forUpdate);
        }
        else
        {
            IDirElement element = getElement(elementName, forUpdate, getSubclassingDelta);
            if (element != null)
            {
                return new IDirElement[]
                { element };
            }
            else
            {
                return new IDirElement[0];
            }
        }
    }

    private DirectoryServiceException convertException(StorageException se)
    {
        DirectoryServiceException directoryException = se instanceof ParentDirDoesNotExistException ? new DirectoryDoesNotExistException("Parent directory problem: see cause") : new DirectoryServiceException("Storage exception: see cause");
        directoryException.initCause(se);
        return directoryException;
    }

    // Returns the element and other elements that can be retrieved without any extra I/O
    private IDirElement[] getElementsAsIs(EntityName eName, boolean forUpdate)
    throws DirectoryServiceException
    {
        try
        {
            m_lock.readLock();

            try
            {
                IDirElement[] elements = m_storage.getElements(eName);
                if (elements == null)
                {
                    return new IDirElement[0];
                }
                for (int i = 0; i < elements.length; i++)
                {
                    ((Element)elements[i]).setReadOnly(!forUpdate);
                }
                return elements;
            }
            catch (StorageException e)
            {
                throw convertException(e);
            }
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    public boolean isExpandableBlob(EntityName eName)
    {
        m_lock.readLock();
        try
        {
            File blobFile = m_blobStorage.blobToFile(eName);
            // this assumes the file has been copied to local sotrage. The file will be opened to find
            // the manifest or a sonic.xml file
            boolean expandable =  Blob.isExpandableFile(blobFile);
            if (m_blobStorage instanceof PSEStorage)
            {
                blobFile.delete();
            }
            return expandable;
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    private IDirElement prepareForShipment(IDirElement element, boolean forUpdate, boolean asIs, boolean attachDelta)
    throws DirectoryServiceException
    {

        if (element == null)
        {
            return null;
        }

        if (!asIs)
        {
            String superReference = element.getSuperElementName();
            boolean isSuper = ((Element)element).isSuperElement();
            if (superReference != null)
            {
                element = realizeSubClassedElement(element.getIdentity(), superReference, attachDelta);
            }
            else if (isSuper)
            {
                element = (IDirElement)cleanupSuperElement((Element)element, false, false);
            }
        }

        ((Element)element).setReadOnly(!forUpdate);

        return element;

    }

    private Element cleanupSuperElement(Element element, boolean deleteAll, boolean clearTemplate)
    {
        element.setReadOnly(false);
        element.cleanupSuperAttributes(deleteAll, clearTemplate);
        return (Element)element.createClone(); // to remove delta history
    }

    // Create a subclassed element by applying the delta that defines it to the super element
    private IDirElement realizeSubClassedElement(IElementIdentity id, String superReference, boolean attachDelta)
    throws DirectoryServiceException
    {

        Element superElement = (Element)getElementAsIs(superReference, false);

        if (superElement == null)
        {
            throw new DirectoryServiceException('"' + superReference + "\" was deleted.");
        }

        return realizeSubClassedElement(superElement, id, attachDelta);
    }

    private IDirElement realizeSubClassedElement(Element superElement, IElementIdentity id, boolean attachDelta)
    {
        byte[] deltaBytes = superElement.getSubclassedDelta(id.getName());
        Element realizedElement = doRealizeSub(superElement, attachDelta ? deltaBytes : null, DeltaElement.fromBytes(deltaBytes), id.getName(), id);
        return (IDirElement)realizedElement;
    }

    @Override
    public IIdentity getIdentity(String name)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getIdentity " + name);
        try
        {
            m_lock.readLock();
            validateOpen();

            return m_idCache.get(validateName(name));
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public IDirElement[] getAllElements(String dirName, boolean forUpdate)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getAllElements " + dirName + " for update " + forUpdate);

        try
        {
            m_lock.readLock();
            validateOpen();
            EntityName dirNameE = validateName(dirName);

            if (m_dsHandlers != null)
            {
                IDSHandler handler = m_dsHandlers.getHandler(dirNameE);
                if (handler != null)
                {
                    return m_dsHandlers.getAllElements(handler, dirName);
                }
            }

            IDirElement[] elements = m_storage.getAllElements(dirNameE);

            for (int i = 0; i < elements.length; i++)
            {
                elements[i] = prepareForShipment(elements[i], forUpdate, false, false);
            }

            return elements;
        }
        catch (StorageException e)
        {
            throw convertException(e);
        }
        finally
        {
            m_lock.releaseLock();
        }
    }
    @Override
    public IDirElement[] getAllElementsCompressed(String dirName, boolean forUpdate)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getAllElementsCompressed " + dirName + " for update " + forUpdate);
        try
        {
            m_lock.readLock();
            return getAllElements(dirName, forUpdate);
        }
        finally
        {
            m_lock.releaseLock();
        }
    }


    boolean directoryExists(String dirName)
    throws DirectoryServiceException
    {
        try
        {
            m_lock.readLock();
            validateOpen();
            Object id = null;

            // If the parent directory of dirName does not exist then DirectoryServiceException will be thrown
            // and, in that case, id should remain null
            try
            {
                id = m_idCache.get(validateName(dirName));
            }
            catch (DirectoryServiceException e)
            {
            }

            if (id == null)
            {
                return false;
            }

            if (!(id instanceof IDirIdentity))
            {
                return false;
            }

            return true;

        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public IIdentity[] listAll(String dirName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "listAll " + dirName);
        try
        {
            m_lock.readLock();
            validateOpen();

            EntityName dirNameE = validateName(dirName);

            if (m_dsHandlers != null)
            {
                IDSHandler handler = m_dsHandlers.getHandler(dirNameE);
                if (handler != null)
                {
                    return m_dsHandlers.listAll(handler, dirName);
                }
            }

            return m_idCache.list(dirNameE, true, true);
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public IDirIdentity[] listDirectories(String dirName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "listDirectories " + dirName);
        try
        {
            m_lock.readLock();
            validateOpen();

            EntityName dirNameE = validateName(dirName);

            if (m_dsHandlers != null)
            {
                IDSHandler handler = m_dsHandlers.getHandler(dirNameE);
                if (handler != null)
                {
                    return m_dsHandlers.listDirectories(handler, dirName);
                }
            }

            IIdentity[] ids = m_idCache.list(dirNameE, true, false);
            IDirIdentity[] dirIds = new IDirIdentity[ids.length];
            for (int i = 0; i < ids.length; i++)
            {
                dirIds[i] = (IDirIdentity)ids[i];
            }
            return dirIds;
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public IElementIdentity[] listElements(String dirName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "listElements " + dirName);
        try
        {
            m_lock.readLock();
            validateOpen();

            EntityName dirNameE = validateName(dirName);
            if (m_dsHandlers != null)
            {
                IDSHandler handler = m_dsHandlers.getHandler(dirNameE);
                if (handler != null)
                {
                    return m_dsHandlers.listElements(handler, dirName);
                }
            }

            IIdentity[] ids = m_idCache.list(dirNameE, false, true);
            IElementIdentity[] elementIds = new IElementIdentity[ids.length];
            for (int i = 0; i < ids.length; i++)
            {
                elementIds[i] = (IElementIdentity)ids[i];
            }
            return elementIds;
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    // Efficient bulk storage of elements: All the elements must be under the same directory
    @Override
    public void setElements(IDirElement[] elements)
    throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "setElements: " + elements.length + " are set.");
        if ((m_traceMask & IComponent.TRACE_VERBOSE) > 0)
        {
            for (int i = 0; i < elements.length; i++)
            {
                trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, elements[i].toString());
            }
        }

        validateOpen();

        if (elements.length == 0)
        {
            return;
        }

        if (!m_notificationManager.inThrowAway())
        {
            throw new DirectoryServiceException("setElements() must be called from the import manager.");
        }

        EntityName firstName = validateName(elements[0].getIdentity().getName());
        String parentDir = firstName.getParent();
        boolean packerDir = PackedDirUtil.underPackedDir(firstName);

        if (m_authentication != null)
        {
            m_authentication.okToModify(firstName);
        }

        try
        {
            m_lock.writeLock();

            EntityName[] eNames = new EntityName[elements.length];
            for (int i = 0; i < elements.length; i++)
            {
                eNames[i] = validateName(elements[i].getIdentity().getName());

                if (!eNames[i].getParent().equals(parentDir))
                {
                    throw new DirectoryServiceException("All the elements must be under the same directory.");
                }

            }

            boolean transactOK = false;
            try
            {
                m_trManager.join();

                if (packerDir)
                {
                    m_storage.setElements(elements, false);

                    for (int i = 0; i < eNames.length; i++)
                    {
                        m_idCache.addElement(eNames[i], elements[i].getIdentity());
                    }
                }
                else
                {
                    for (int i = 0; i < elements.length; i++)
                    {
                        storeInternal(eNames[i], elements[i]);
                    }
                }

                transactOK = true;

            }
            catch (StorageException e)
            {
                throw convertException(e);
            }
            finally
            {

                m_trManager.leave(transactOK);
            }

        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    //Efficient bulk deletetion: All the elements must be under the same directory
    @Override
    public void deleteElements(String[] elementNames)
        throws DirectoryServiceException
    {
        commonDeleteElements(elementNames, false);
    }

    public void importDeleteElements(String[] elementNames)
        throws DirectoryServiceException
    {
        commonDeleteElements(elementNames, true);
    }

    private void commonDeleteElements(String[] elementNames, boolean importDelete)
    throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "deleteElements: " + elementNames.length + " are deleted.");

        validateOpen();

        if (elementNames.length == 0)
        {
            return;
        }

        EntityName firstName = validateName(elementNames[0]);
        String parentDir = firstName.getParent();
        boolean packedDir = PackedDirUtil.underPackedDir(firstName);

        try
        {
            m_lock.writeLock();

            ArrayList nameList = new ArrayList();

            for (int i = 0; i < elementNames.length; i++)
            {
                EntityName eName = validateName(elementNames[i]);

                if (!eName.getParent().equals(parentDir))
                {
                    throw new DirectoryServiceException("All deleted elements must be under the same directory.");
                }

                Object id = m_idCache.get(eName);
                if (id == null)
                {
                    continue;
                }

                if (!(id instanceof IElementIdentity))
                {
                    throw new DirectoryServiceException('"' + elementNames[i] + "\" is not an element.");
                }

                nameList.add(eName);

            }

            if (nameList.isEmpty())
            {
                return;
            }

            EntityName[] eNames = new EntityName[nameList.size()];
            nameList.toArray(eNames);

            boolean transactOK = false;
            try
            {
                m_trManager.join();

                if (packedDir)
                {
                    ArrayList modifications = new ArrayList();

                    for (int i = 0; i < eNames.length; i++)
                    {
                        IElementIdentity id = (IElementIdentity) m_idCache.get(eNames[i]);
                        Element beforeImage = (Element)getElementAsIs(id.getName(), false);
                        Element deletedElement = (Element)Element.createDeletedElement(id);
                        String logicalName = storageToLogical(id.getName());
                        ModificationItem modItem = new ModificationItem(beforeImage, deletedElement, logicalName, true);
                        modifications.add(modItem);
                    }

                    m_storage.deleteElements(eNames);

                    m_notificationManager.addGroupNotification(modifications, true);

                    for (int i = 0; i < eNames.length; i++)
                    {
                        m_idCache.remove(eNames[i]);
                    }
                }
                else
                {
                    for (int i = 0; i < elementNames.length; i++)
                    {
                        commonDeleteElement(elementNames[i], null, importDelete);
                    }
                }

                transactOK = true;

            }
            catch (StorageException e)
            {
                throw convertException(e);
            }
            finally
            {

                m_trManager.leave(transactOK);
            }

        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    private IElementIdentity commonDeleteElement(String elementName, IDeltaView view, boolean importDelete)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "deleteElement " + elementName + " view " + view);
        try
        {
            m_lock.writeLock();
            validateOpen();

            IDirElement viewElement = null;
            BeforeAfterPair pair = null;
            if (view != null)
            {
                pair = prepareToSet(((DeltaView)view).getElement(), null);
                viewElement = pair.m_after;
            }

            EntityName eName = validateName(elementName);

            if (m_authentication != null)
            {
                m_authentication.okToModify(eName);
            }

            IIdentity id = m_idCache.get(eName);

            if (id == null)
            {
                return null;
            }

            if (!(id instanceof IElementIdentity))
            {
                throw new DirectoryServiceException('"' + elementName + "\" is not an element.");
            }

            IElementIdentity elementId = (IElementIdentity)id;

            Element oldElement = (Element)getElementAsIs(eName.getName(), false);

            ModificationItem modItem = null;
            Element oldRealized = null;
            Element beforeImage = oldElement;
            String logicalName = null;
            boolean transactOK = false;
            try
            {
                m_trManager.join();

                if (oldElement.isSuperElement())
                {
                    throw new DirectoryServiceException('"' + elementName + "\" is a super element, it can be deleted only after " + "its sub-classed elements are unSubclassed or deleted.");
                }
                else if (oldElement.isSubclassedElement())
                {
                    oldRealized = (Element)getElement(id.getName(), false);
                    beforeImage = oldRealized;
                    modItem = removeSubclassedFromSuper(oldElement.getSuperElementName(), eName.getName());
                }
                // if we're in the middle of an import, keep the blob envelope elements, but make it look
                // like they've been deleted
                boolean importingBlob = false;
                if (importDelete)
                {
                    IAttributeSet topSet = oldElement.getAttributes();
                    IAttributeSet systemAttrs = (IAttributeSet)topSet.getAttribute(Blob.SYSTEM_ATTRIBUTES);
                    if (systemAttrs != null)
                    {
                        Integer state = (Integer)systemAttrs.getAttribute(Blob.LARGE_FILE_STATE);
                        if (state != null)
                        {
                            importingBlob = true;
                        }
                    }
                }
                if (!importingBlob)
                {
                    m_storage.deleteElement(eName);
                }
                m_idCache.remove(eName);
                if (m_logicalNameSpace != null) // we can get here while the DS is booting up
                {
                    logicalName = storageToLogical(elementName);
                    if (logicalName != null)
                     {
                        removePermissions(logicalName);
                       // do not remove the root permissions if there is only a storage name for this element
                    }
                }
                if (viewElement != null)
                {
                    storeInternal(VIEW_ELEMENT_ENTITY_NAME, viewElement);
                }
                transactOK = true;

            }
            catch (StorageException e)
            {
                throw convertException(e);
            }
            finally
            {
                // Announce the modification
                if (modItem != null)
                {
                    notifyMods(modItem, null);
                }
                Element deletedElement = (Element)Element.createDeletedElement(elementId);
                ModificationItem modification = new ModificationItem(beforeImage, deletedElement, logicalName, false);
                notifyMods(modification, createViewMod(view, pair));

                m_trManager.leave(transactOK);
            }

            return elementId;
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    public IElementIdentity importDeleteElement(String elementName, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException
    {
        return commonDeleteElement(elementName, view, true);
    }

    @Override
    public IElementIdentity deleteElement(String elementName, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return commonDeleteElement(elementName, view, false);
    }

    private ModificationItem createViewMod(IDeltaView view, BeforeAfterPair pair)
    {
        if (view == null)
        {
            return null;
        }

        return createViewMod(view, (Element)pair.m_before);
    }

    private ModificationItem createViewMod(IDeltaView view, Element beforeImage)
    {
        if (view == null)
        {
            return null;
        }

        return new ModificationItem((DeltaElement)((DeltaView)view).getElement(), beforeImage);
    }

    @Override
    public void createDirectory(String dirName)
    throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "createDirectory " + dirName);

        boolean transactionOk = false;
        boolean joinedTransaction = false;
        try
        {

            m_lock.writeLock();

            validateOpen();

            if (EntityName.containsAnyChar(dirName, RESERVED_CHARACTERS))
            {
                throw new DirectoryServiceException("Invalid name \"" + dirName + "\". " + "Directory names cannot contain the characters: " + RESERVED_CHARACTERS);
            }

            EntityName dName = validateName(dirName);

            if (m_dsHandlers != null && m_dsHandlers.getHandler(dName) != null)
            {
                throw new DirectoryServiceException(dirName + " is associated with a DS handler: " + " Cannot create " + dirName + " - DS handlers are read-only.");
            }

            IIdentity id = m_idCache.get(dName);

            if (id != null)
            {
                throw new DirectoryServiceException('"' + dirName + "\" already exists.");
            }

            try
            {

                m_trManager.join();
                joinedTransaction = true;
                m_storage.createDirectory(dName);
                m_idCache.addDirectory(dName);
                transactionOk = true;
            }
            catch (StorageException e)
            {
                throw convertException(e);
            }
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactionOk);
        }
    }

    @Override
    public void deleteDirectory(String dirName, IDeltaView view)
    throws DirectoryServiceException
    {
        deleteDirectory(dirName, view, true);
    }

    private void deleteDirectory(String dirName, IDeltaView view, boolean topLevel)
    throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "deleteDirectory " + dirName + " view " + view);

        validateOpen();
        EntityName dName = validateName(dirName);

        try
        {
            if (m_authentication != null)
            {
                m_authentication.okToDeleteDir(dName);

                // Noop if not a directory domain
                if (topLevel)
                {
                    m_authentication.startClosingDomain(dName.getName());
                }
            }

            m_lock.writeLock();

            IDirElement viewElement = null;
            BeforeAfterPair pair = null;
            if (view != null)
            {
                pair = prepareToSet(((DeltaView)view).getElement(), null);
                viewElement = pair.m_after;
            }

            boolean transactOK = false;
            try
            {
                m_trManager.join();

                IIdentity[] ids = listAll(dirName);
                ArrayList elementsToDelete = new ArrayList();
                ArrayList dirsToDelete = new ArrayList();

                // First, delete all the elements under dirName
                for (int i = 0; i < ids.length; i++)
                {
                    if (ids[i] instanceof IElementIdentity)
                    {
                        elementsToDelete.add(ids[i].getName());
                    }
                    else
                    {
                        dirsToDelete.add(ids[i].getName());
                    }

                }
                String[] elementNames = new String[elementsToDelete.size()];
                elementsToDelete.toArray(elementNames);
                deleteElements(elementNames);

                // Second, deletes all the directories under dirName
                for (int i = 0; i < dirsToDelete.size(); i++)
                {
                    deleteDirectory((String)dirsToDelete.get(i), null, false);
                }

                // Third, deletes dirName
                deleteDirectory(dirName);

                // Fourth, store the new view
                if (viewElement != null)
                {
                    storeInternal(VIEW_ELEMENT_ENTITY_NAME, viewElement);
                }
                if (view != null)
                {
                    notifyMods(createViewMod(view, pair), null);
                }

                transactOK = true;

            }
            finally
            {
                m_trManager.leave(transactOK);

            }

        }
        finally
        {
            m_lock.releaseLock();
            if (m_authentication != null && topLevel)
            {
                m_authentication.endClosingDomain();
            }
        }

    }

    @Override
    public void deleteDirectory(String dirName)
    throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "deleteDirectory " + dirName);
        boolean transactionOk = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            validateOpen();
            EntityName dName = validateName(dirName);

            IIdentity[] ids = listAll(dirName);
            if (ids.length > 0)
            {
                throw new DirectoryServiceException(dirName + " is not empty.");
            }

            try
            {
                m_trManager.join();
                joinedTransaction = true;
                m_storage.deleteDirectory(dName);
                m_idCache.remove(dName);
                transactionOk = true;
            }
            catch (StorageException e)
            {
                throw convertException(e);
            }
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactionOk);
        }
    }

    NextVersionToken createNextVersionToken(boolean replaceToLogicalNames)
    {
        HashMap tokenTable = null;
        if (m_notificationManager != null)
        {
            tokenTable = m_notificationManager.getAndResetNewIds();
        }
        else
        {
            tokenTable = new HashMap();
        }

        NextVersionToken token = new NextVersionToken(tokenTable);

        if (replaceToLogicalNames)
        {
            return token.replaceReferences(m_logicalNameSpace.createNameReplacer(false));
        }
        else
        {
            return token;
        }
    }

    @Override
    public INextVersionToken setElements(IBasicElement[] elements, String[] deleteList, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return setElements(elements, deleteList, view, false);
    }

    private IDirElement[] canBeDoneInBulk(IBasicElement[] elements)
    throws DirectoryServiceException
    {
        boolean bulkImport = true;
        IDirElement[] dirElements = null;

        if (elements != null && elements.length > 0)
        {
            EntityName firstName = validateName(elements[0].getIdentity().getName());
            String parentDir = firstName.getParent();
            boolean packerDir = PackedDirUtil.underPackedDir(firstName);

            if (packerDir)
            {
                if (m_authentication != null)
                {
                    m_authentication.okToModify(firstName);
                }

                dirElements = new IDirElement[elements.length];
                for (int i = 0; i < elements.length; i++)
                {
                    EntityName eName = validateName(elements[i].getIdentity().getName());

                    if (!(elements[i] instanceof IDirElement) || !eName.getParent().equals(parentDir))
                    {
                        bulkImport = false;
                        break;
                    }
                    else
                    {
                        dirElements[i] = (IDirElement)((Element)elements[i]).createClone();
                    }
                }
            }
            else
            {
                bulkImport = false;
            }
        }
        else
        {
            bulkImport = false;
        }

        if (bulkImport)
        {
            return dirElements;
        }
        else
        {
            return null;
        }

    }

    boolean canBeDoneInBulk(IDirElement[] elements)
    throws DirectoryServiceException
    {

        if (elements == null || elements.length == 0)
        {
            return false;
        }

        EntityName firstName = validateName(elements[0].getIdentity().getName());
        String parentDir = firstName.getParent();
        boolean packerDir = PackedDirUtil.underPackedDir(firstName);

        if (!packerDir)
        {
            return false;
        }

        for (int i = 0; i < elements.length; i++)
        {
            EntityName eName = validateName(elements[i].getIdentity().getName());

            if (!eName.getParent().equals(parentDir))
            {
                return false;
            }
        }

        return true;
    }

    INextVersionToken setElements(IBasicElement[] elements, String[] deleteList, IDeltaView view, boolean noBulk)
    throws DirectoryServiceException, VersionOutofSyncException
    {

        // Look wheteher we can optimize and set all the elements efficiently in bulk. We do that
        // If there are no elements in the deletion list, all the elements are new and all are under the same
        // packed directory.
        IDirElement[] dirElements = null;
        if (!noBulk && (deleteList == null || deleteList.length == 0))
        {
            dirElements = canBeDoneInBulk(elements);
        }
        if (dirElements != null)
        {
            HashMap idList = importFromList(dirElements, null, true);
            INextVersionToken viewToken = null;
            if (view != null)
            {
                viewToken = setViewGetToken(view);
            }
            return new NextVersionToken(idList, (NextVersionToken)viewToken);
        }

        boolean transactOK = false;
        NextVersionToken token = null;
        boolean startedRememberIDs = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();

            if (m_notificationManager != null)
            {
                startedRememberIDs = m_notificationManager.rememberNewIds();
            }

            m_trManager.join();
            joinedTransaction = true;

            if (deleteList != null)
            {
                for (int i = 0; i < deleteList.length; i++)
                {
                    if (deleteList[i] != null)
                    {
                        deleteElement(deleteList[i], null);
                    }
                }
            }

            if (elements != null)
            {
                for (int i = 0; i < elements.length; i++)
                {
                    if (elements[i] != null)
                    {
                        setElement(elements[i], null, true);
                    }
                }
            }

            if (view != null)
            {
                setView(view);
            }

            transactOK = true;
        }
        finally
        {
            if (startedRememberIDs)
            {
                token = createNextVersionToken(false);
            }
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }

        return token;

    }


    @Override
    public INextVersionToken setElement(IBasicElement element, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return setElement(true, element, view);
    }

    INextVersionToken setElement(boolean fireTriggers, IBasicElement element, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        NextVersionToken token = null;

        boolean startedRememberIDs = false;

        try
        {
            m_lock.writeLock();

            if (m_notificationManager != null)
            {
                startedRememberIDs = m_notificationManager.rememberNewIds();
            }
            setElement(element, view, true, fireTriggers);
        }
        finally
        {
            if (startedRememberIDs)
            {
                token = createNextVersionToken(false);
            }
            m_lock.releaseLock();
        }

        return token;
    }

    private void setElement(IBasicElement element, IDeltaView view, boolean notifySubclassSetting)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        setElement(element, view, notifySubclassSetting, true);
    }

    private void setElement(IBasicElement element, IDeltaView view, boolean notifySubclassSetting, boolean fireTriggers)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        validateOpen();
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "setElement " + element + " view " + view);
        if (element == null)
        {
            throw new DirectoryServiceException("The element must not be null.");
        }

        try
        {
            m_lock.writeLock();

            if (m_authentication != null)
            {
                m_authentication.okToModify(element.getIdentity().getName());
            }

            IDirElement oldElement = null;
            boolean isSubClassed = false;
            boolean isSuper = false;
            if (element instanceof IDirElement)
            {
                // The isSuperElement and isSubclassedElement should never happen the way we plan to do
                // xml import.
                if (((Element)element).isSuperElement() || ((Element)element).isSubclassedElement())
                {
                    throw new Error("should never happen.");
                }

                EntityName nameE = validateName(element.getIdentity().getName());

                if (m_dsHandlers != null && m_dsHandlers.getHandler(nameE) != null)
                {
                    throw new DirectoryServiceException(nameE.getName() + " is associated with a DS handler: " + " Cannot set " + nameE.getName() + " - DS handlers are read-only.");
                }

                IIdentity id = m_idCache.get(nameE);

                if (id != null && (id instanceof DirIdentity))
                {
                    throw new DirectoryServiceException('"' + element.getIdentity().getName() + "\" directory exists.");
                }

                if (((Element)element).getSuperToSubclassFrom() != null)
                {
                    subclassOnDSSide((Element)element, view);
                    return;
                }

            }
            else
            {
                oldElement = getElementAsIs(element.getIdentity().getName(), true);
                if (oldElement != null)
                {
                    IElementIdentity deltaIdentity = element.getIdentity();
                    IElementIdentity oldIdentity = oldElement.getIdentity();
                    if (!oldIdentity.equalEntity(deltaIdentity))
                    {
                        throwVersionException(deltaIdentity.getName(), "Element \"" + deltaIdentity.getName() + "\" with creation timestamp " + deltaIdentity.getCreationTimestamp() + " was not found." + " The creation timestamp in the directory is "
                                                                       + oldIdentity.getCreationTimestamp());
                    }

                    isSuper = ((Element)oldElement).isSuperElement();
                    isSubClassed = ((Element)oldElement).isSubclassedElement();
                }
            }

            if (isSubClassed)
            {
                setSubclassedElement((DeltaElement)element, (Element)oldElement, view, notifySubclassSetting, fireTriggers);
            }
            else if (isSuper)
            {
                boolean transactOK = false;
                HashMap allNotifications = null;
                try
                {
                    m_trManager.join();
                    allNotifications = setSuperElement((DeltaElement)element, (Element)oldElement, view);
                    transactOK = true;
                }
                finally
                {
                    m_trManager.leave(transactOK);
                }

                // Notify all the modifications
                m_notificationManager.addNotifications(allNotifications);
            }
            else
            {
                setElementInternal(element, oldElement, view, true, fireTriggers);
            }
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    private void setSubclassedElement(DeltaElement delta, Element oldElement, IDeltaView view, boolean notify, boolean fireTriggers)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        if (delta.isDeleted())
        {
            throw new Error("deleteElement should be used for deleting elements.");
        }

        String superName = oldElement.getSuperElementName();
        Element superElement = (Element)getElementAsIs(superName, true);
        IElementIdentity deltaIdentity = delta.getIdentity();
        IElementIdentity oldIdentity = oldElement.getIdentity();

        if (superElement == null)
        {
            throw new DirectoryServiceException("The super element \"" + superName + "\" of \"" + deltaIdentity.getName() + "\" is missing.");
        }

        byte[] deltaBytes = superElement.getSubclassedDelta(deltaIdentity.getName());
        if (deltaBytes == null)
        {
            throw new DirectoryServiceException("The super element \"" + superName + "\" of \"" + deltaIdentity.getName() + "' does not contain the sub-classing information.");
        }
        DeltaElement subclassingDelta = DeltaElement.fromBytes(deltaBytes);
        if (deltaIdentity.getVersion() != oldIdentity.getVersion())
        {
            throwVersionException(deltaIdentity.getName(), deltaIdentity.getName() + " was modified by another application.");
        }

        Element realizedOld = (Element)realizeSubClassedElement(superElement, oldIdentity, false);

        // When the DS runs in the same process with the application we must clone the delta to prevent side effects
        // on the application
        if (m_noContainer && delta != null)
        {
            delta = delta.createClone();
        }

        subclassingDelta.adjustToSubclassedModification(delta);
        superElement.addSubclassedElement(deltaIdentity.getName(), subclassingDelta.toBytes());

        boolean transactOK = false;
        BeforeAfterPair pair = null;
        try
        {
            m_trManager.join();
            // We use storeInterna to prevent the version# of superElement from being increments - this is a system
            // update
            storeInternal(validateName(superName), superElement);

            setElementAsIs(oldElement.doneUpdate(), null, false); // This increments the version
            if (view != null)
            {
                pair = prepareToSet(((DeltaView)view).getElement(), null);
                storeInternal(VIEW_ELEMENT_ENTITY_NAME, pair.m_after);
            }
            transactOK = true;
        }
        catch (ReadOnlyException e)
        {
            throw new Error(e.toString()); // This should not happen
        }
        finally
        {
            if (notify)
            {
                notifyMods(new ModificationItem(delta, realizedOld), createViewMod(view, pair));
            }
            m_trManager.leave(transactOK, fireTriggers);
        }

    }

    private HashMap setSuperElement(DeltaElement superDelta, Element originalSuperElement, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        BeforeAfterPair pair = prepareToSet((IBasicElement)superDelta.createClone(), (IDirElement)originalSuperElement);
        Element newSuperElement = (Element)pair.m_after;
        String[] subList = originalSuperElement.getSubclassedList();
        HashMap allNotifications = new HashMap();
        for (int i = 0; i < subList.length; i++)
        {
            String subclassedName = subList[i];
            Element subclassed = (Element)getElementAsIs(subclassedName, true);
            if (subclassed == null)
            {
                throw new DirectoryServiceException("Subclassed element \"" + subclassedName + "\" is missing.");
            }
            try
            {

                // When the DS runs in the same process with the application we must clone the delta to prevent side
                // effects
                // on the application
                DeltaElement delta = superDelta;
                if (m_noContainer && superDelta != null)
                {
                    delta = superDelta.createClone();
                }

                ModificationItem modItem = updateSubClassed(originalSuperElement, newSuperElement, delta, subclassed);
                if (!modItem.m_delta.emptyDelta())
                {
                    allNotifications.put(subclassedName, modItem);
                }
            }
            catch (CannotAdjustToSuperModificationException e)
            {
                String problematicAttribute = e.getFailingAttribute();
                throwVersionException(originalSuperElement.getIdentity().getName(), "Cannot modify super element \"" + originalSuperElement.getIdentity().getName() + "\" " + "because of the \"" + problematicAttribute
                                                                                    + "\" attribute of subclassed element \"" + subclassedName + "\". Please unSubclass \"" + subclassedName + "\" first.");
            }

        }

        EntityName superName = validateName(newSuperElement.getIdentity().getName());
        storeInternal(superName, (IDirElement)newSuperElement);

        allNotifications.put(originalSuperElement.getIdentity().getName(), new ModificationItem(superDelta, (Element)pair.m_before));

        if (view != null)
        {
            BeforeAfterPair viewPair = prepareToSet(((DeltaView)view).getElement(), null);
            storeInternal(VIEW_ELEMENT_ENTITY_NAME, viewPair.m_after);
            allNotifications.put(VIEW_ELEMENT_ENTITY_NAME.getName(), new ModificationItem((DeltaElement)((DeltaView)view).getElement(), (Element)viewPair.m_before));
        }

        return allNotifications;
    }

    // Update the subclassing delta based on the modifications of the superElement and return the notification delta of
    // the subElement
    private ModificationItem updateSubClassed(Element originalSuperElement, Element newSuperElement, DeltaElement superDelta, Element subclassed)
    throws CannotAdjustToSuperModificationException, DirectoryServiceException, VersionOutofSyncException
    {
        IElementIdentity subclassedID = subclassed.getIdentity();
        String subClassedName = subclassedID.getName();
        DeltaElement oldSubclassingDelta = DeltaElement.fromBytes(originalSuperElement.getSubclassedDelta(subClassedName));
        Element oldRealized = doRealizeSub(originalSuperElement, null, oldSubclassingDelta, subClassedName, subclassedID);
        DeltaElement newSubclassedDelta = oldSubclassingDelta.createClone();
        newSubclassedDelta.adjustToSuperclassModification(superDelta);
        Element newRealized = doRealizeSub(newSuperElement, null, newSubclassedDelta, subClassedName, subclassed.getIdentity());
        newSuperElement.addSubclassedElement(subClassedName, newSubclassedDelta.toBytes());
        DeltaElement notificationDelta = oldRealized.createDelta(newRealized);
        if (!notificationDelta.emptyDelta())
        {
            try
            {
                setElementAsIs(subclassed.doneUpdate(), null, false); // This increments the version
            }
            catch (ReadOnlyException e)
            {
                throw new Error(e.toString()); // This should not happen
            }
        }

        return new ModificationItem(notificationDelta, oldRealized);
    }

    private void setElementAsIs(IBasicElement element, IDeltaView view, boolean notifyMods)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        validateOpen();
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "setElementAsIs " + element + " view " + view + "notify " + notifyMods);
        try
        {
            m_lock.writeLock();

            setElementInternal(element, null, view, notifyMods);
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    private void setElementInternal(IBasicElement element, IDirElement oldElement, IDeltaView view, boolean notifyMods)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        setElementInternal(element, oldElement, view, notifyMods, true);
    }

    private void setElementInternal(IBasicElement element, IDirElement oldElement, IDeltaView view, boolean notifyMods, boolean fireTriggers)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        IDirElement viewElement = null;
        BeforeAfterPair viewPair = null;
        if (view != null)
        {
            viewPair = prepareToSet(((DeltaView)view).getElement(), null);
            viewElement = viewPair.m_after;
        }

        BeforeAfterPair pair = prepareToSet(element, oldElement);
        IDirElement newElement = pair.m_after;

        boolean transactOK = false;
        try
        {
            m_trManager.join();
            storeInternal(validateName(element.getIdentity().getName()), newElement);
            if (viewElement != null)
            {
                storeInternal(VIEW_ELEMENT_ENTITY_NAME, viewElement);
            }
            transactOK = true;
        }
        finally
        {
            if (notifyMods)
            {
                // Announce the modification
                ModificationItem modItem = null;
                if (element instanceof IDirElement)
                {
                    modItem = new ModificationItem((Element)newElement);
                }
                else
                {
                    modItem = new ModificationItem((DeltaElement)element, (Element)pair.m_before);
                }
                notifyMods(modItem, createViewMod(view, viewPair));
            }
            m_trManager.leave(transactOK, fireTriggers);
        }

    }

    @Override
    public IDirElement[] getUpdatedList(IElementIdentity[] identities, HashMap deletedConfs, long callerBackupVersion)
    throws DirectoryServiceException
    {
        trace(TRACE_CONTAINER_ACCESS + TRACE_ALL_DS_ACCESS, "getUpdatedList " + identities.length + " identities");

        try
        {
            m_lock.readLock();

            boolean backupVersionMismatch = callerBackupVersion != 0 && callerBackupVersion != m_backupVersion;
            if (backupVersionMismatch)
            {
                trace(TRACE_CONTAINER_ACCESS + TRACE_ALL_DS_ACCESS, "caller backup version: " + callerBackupVersion + " DS backup version: " + m_backupVersion);
            }

            SizedArrayList resultList = new SizedArrayList();

            return buildUpdateList(resultList, backupVersionMismatch, identities, deletedConfs);
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    private void markListUnfinished(SizedArrayList resultList)
    {
        Element firstElement = (Element)resultList.get(0);
        if (!(firstElement instanceof IEnvelope))
        {
            firstElement = new EnvelopeElement((IDirElement)firstElement);
            resultList.set(0, firstElement);
        }
        ((IEnvelope)firstElement).setProperty(IEnvelope.RECONCILIATION_NOT_FINISHED_LABEL, "true");
    }

    @Override
    public IDirElement[] getNewElements(String dirName, IElementIdentity[] identities)
    throws DirectoryServiceException
    {
        trace(TRACE_CONTAINER_ACCESS + TRACE_ALL_DS_ACCESS, "getNewElements from " + dirName + " - " + identities.length + " identities ");

        try
        {
            m_lock.readLock();

            SizedArrayList resultList = new SizedArrayList();

            IDirElement[] elements = buildNewList(resultList, SizedArrayList.MAXIMUM_CONTENT_SIZE, identities, new String[]
            { dirName });
            if (elements == null)
            {
                elements = new IDirElement[0];
            }

            return elements;
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    // Return a single element array if the caller doesn't have the correct version. Returns null if the caller has
    // has the correct element. Return an array with a single null element if the element is not found.
    @Override
    public IDirElement[] getElementIfUpdated(long callerBackupVersion, String elementName, IElementIdentity id)
    throws DirectoryServiceException
    {
        // If id is null, the element is not yet in the container's cache or if
        // the backup version has changed - get the element from storage
        if (id == null || (callerBackupVersion != 0 && callerBackupVersion != m_backupVersion))
        {
            return new IDirElement[]
            { getElement(elementName, false) };
        }

        EntityName elementNameE = validateName(elementName);
        IIdentity cacheObj = null;
        try
        {
            cacheObj = m_idCache.get(elementNameE);
        }
        catch (DirectoryDoesNotExistException e)
        {
        } // Equivalent to not findig the element

        // If the element is not found, or the name does not point to an elementnthen it's deleted
        if (cacheObj == null || !(cacheObj instanceof IElementIdentity))
        {
            return new IDirElement[1];
        }

        IElementIdentity cacheID = (IElementIdentity)cacheObj;

        if (cacheID.equalVersion(id))
        {
            return null; // The caller has the correct version
        }
        else {
            return new IDirElement[]
            { getElement(elementName, false) }; // The caller does not have the correct version
        }

    }

    @Override
    public IDirElement[][] getCacheElements(long callerBackupVersion, IElementIdentity[] identities, HashMap deletedConfigurations, String dirNames[])
    throws DirectoryServiceException
    {
        trace(TRACE_CONTAINER_ACCESS + TRACE_ALL_DS_ACCESS, "getCacheElements " + identities.length + " identities, " + (deletedConfigurations == null ? "<null>" : Integer.toString(deletedConfigurations.size())) + " deletedConfigurations, "
                                                                      + (dirNames == null ? "<null>" : Integer.toString(dirNames.length)) + " dirNames");

        IDirElement[][] elements = new IDirElement[2][];

        try
        {
            m_lock.readLock();

            boolean backupVersionMismatch = callerBackupVersion != 0 && callerBackupVersion != m_backupVersion;
            if (backupVersionMismatch)
            {
                trace(TRACE_CONTAINER_ACCESS + TRACE_ALL_DS_ACCESS, "caller backup version: " + callerBackupVersion + " DS backup version: " + m_backupVersion);
            }

            validateOpen();

            SizedArrayList resultList = null;
            int estimatedContentSize = 0;

            if (deletedConfigurations != null)
            {
                resultList = new SizedArrayList();
                elements[0] = buildUpdateList(resultList, backupVersionMismatch, identities, deletedConfigurations);

                // we stop in the middle since the content of the result got too large - we will tell the
                // caller and the caller will make another getCacheElements() call
                estimatedContentSize = resultList.getEstimatedContentSize();
                if (estimatedContentSize >= SizedArrayList.MAXIMUM_CONTENT_SIZE)
                {
                    elements[1] = new IDirElement[0];
                    return elements;
                }
            }

            if (dirNames != null)
            {
                resultList = new SizedArrayList();
                // for each element that has been replaced we need to update the identity so that we
                // don't also get a new element because the creation timestamp has been changed
                int elementsLength = elements[0] != null ? elements[0].length : 0;
                for (int i = 0; i < elementsLength; i++)
                {
                    if (elements[0][i] instanceof IEnvelope && ((IEnvelope)elements[0][i]).getProperty(IEnvelope.DELETED_BUT_RECREATED_LABEL) != null)
                    {
                        IElementIdentity oldID = ((IEnvelope)elements[0][i]).getEnvelopedElement().getIdentity();
                        // the element after the deleted but recreated will be the replacement (with the new id)
                        IElementIdentity newID = ((IEnvelope)elements[0][++i]).getEnvelopedElement().getIdentity();
                        for (int j = 0; j < identities.length; j++)
                        {
                            if (identities[j].equalEntity(oldID))
                            {
                                identities[j] = newID;
                                break;
                            }
                        }
                    }
                }
                elements[1] = buildNewList(resultList, SizedArrayList.MAXIMUM_CONTENT_SIZE - estimatedContentSize, identities, dirNames);
            }
        }
        finally
        {
            m_lock.releaseLock();
        }

        return elements;
    }

    private IDirElement[] buildNewList(SizedArrayList resultList, int availableElementsContentSize, IElementIdentity[] identities, String[] dirNames)
    throws DirectoryServiceException
    {
        try
        {
            for (int i = 0; i < dirNames.length; i++)
            {
                // Lists all the elements in the directory
                IElementIdentity[] elementIds = null;
                try
                {
                    elementIds = listElements(dirNames[i]);
                }
                catch (DirectoryDoesNotExistException e) // Equivalent to not findig the element
                {
                    elementIds = new IElementIdentity[0];
                }

                // Put the old identities in a table for fast access by name. We only care
                // about entities in the dirName directory.
                HashMap oldIds = new HashMap();
                EntityName dName = new EntityName(dirNames[i]);
                for (int j = 0; j < identities.length; j++)
                {
                    EntityName eName = new EntityName(identities[j].getName());

                    if (dName.isParent(eName))
                    {
                        oldIds.put(eName.getName(), identities[j]);
                    }
                }

                // Find the identities of new elements
                ArrayList newIds = new ArrayList();
                for (int j = 0; j < elementIds.length; j++)
                {
                    IElementIdentity oldId = (IElementIdentity)oldIds.get(elementIds[j].getName());

                    // If the caller already have this element do nothing
                    if (oldId != null && oldId.equalEntity(elementIds[j]))
                    {
                        continue;
                    }

                    newIds.add(elementIds[j]);
                }

                // We have the list of ids of new elements, now we prepare the list of elements
                for (int j = 0; j < newIds.size(); j++)
                {
                    if (resultList.getEstimatedContentSize() > availableElementsContentSize)
                    {
                        // We tell the caller that we didn't finish. Note that there is a small chance that
                        // resultList.getEstimatedContentSize() is larger than SizedArrayList.MAXIMUM_CONTENT_SIZE
                        // even if we finished - but that not problem since we'll just get an extra redundant
                        // getUpdatedList call
                        markListUnfinished(resultList);
                        break;
                    }

                    IElementIdentity id = (IElementIdentity)newIds.get(j);
                    resultList.add(getElement(id.getName(), false));
                }
            }

            int resultSize = resultList.size();
            if (resultSize == 0)
            {
                return null;
            }

            IDirElement[] result = new IDirElement[resultSize];
            resultList.toArray(result);

            if ((m_traceMask & IComponent.TRACE_VERBOSE) > 0)
            {
                for (int i = 0; i < result.length; i++)
                {
                    trace(TRACE_CONTAINER_ACCESS + TRACE_ALL_DS_ACCESS, result[i].toString());
                }
            }

            return result;
        }
        catch (ConfigException e)
        {
            throw new DirectoryServiceException(e.toString());
        }
    }

    private IDirElement[] buildUpdateList(SizedArrayList resultList, boolean backupVersionMismatch, IElementIdentity[] identities, HashMap deletedConfs)
    throws DirectoryServiceException
    {
        validateOpen();

        for (int i = 0; i < identities.length; i++)
        {
            // We stop in the middle since the content of the result got too large - we will tell the
            // caller and the caller will make another getUpdatedList() call.

            if (resultList.getEstimatedContentSize() >= SizedArrayList.MAXIMUM_CONTENT_SIZE)
            {
                break;
            }

            EntityName elementName = validateName(identities[i].getName());
            IIdentity cacheObj = null;
            try
            {
                cacheObj = m_idCache.get(elementName);
            }
            catch (DirectoryDoesNotExistException e)
            {
            } // Equivalent to not findig the element

            // If the element is not found then it's deleted and we tell the caller
            // by sending back a deleted element with the same identity
            if (cacheObj == null)
            {
                resultList.add(Element.createDeletedElement(identities[i]));
                continue;
            }

            if (!(cacheObj instanceof IElementIdentity))
            {
                throw new Error(elementName.getName() + " is not an element.");
            }

            IElementIdentity cacheID = (IElementIdentity)cacheObj;

            // The caller has the correct version
            if (cacheID.equalVersion(identities[i]) && !backupVersionMismatch)
            {
                continue;
            }

            // If the creation timestamp is not the same as the caller's then it's deleted and recreated.
            // by sending back a deleted element with the same identity inside an envelope that indicates
            // that the element was later recreated.
            boolean wasDeleted = false;
            if (!cacheID.equalEntity(identities[i]) || backupVersionMismatch)
            {
                wasDeleted = true;
                IEnvelope envelope = new EnvelopeElement(Element.createDeletedElement(identities[i]));
                envelope.setProperty(IEnvelope.DELETED_BUT_RECREATED_LABEL, "true");
                resultList.add(envelope);
            }

            // The caller doesn't have the correct version, we have to send it
            IDirElement elementToSend = getElement(elementName.getName(), false);
            if (wasDeleted) // Deleted and recreated - we send in an envelope marked REPLACEMENT
            {
                elementToSend = new EnvelopeElement(elementToSend); // Put elementToSend in an envelope
                ((IEnvelope)elementToSend).setProperty(IEnvelope.REPLACEMENT_LABEL, "true");
            }

            resultList.add(elementToSend);
        }

        // Add to the list elements which where deleted and recreated
        Iterator iterator = deletedConfs.keySet().iterator();
        while (iterator.hasNext() && resultList.getEstimatedContentSize() <= SizedArrayList.MAXIMUM_CONTENT_SIZE)
        {
            IDirElement newElement = getElement((String)iterator.next(), false);
            if (newElement != null)
            {
                resultList.add(newElement);
            }
        }

        // We tell the caller that we didn't finish. Note that there is a small chance that
        // resultList.getEstimatedContentSize() is larger than SizedArrayList.MAXIMUM_CONTENT_SIZE
        // even if we finished - but that not problem since we'll just get an extra redundant
        // getUpdatedList call
        if (resultList.getEstimatedContentSize() > SizedArrayList.MAXIMUM_CONTENT_SIZE)
        {
            markListUnfinished(resultList);
        }

        int resultSize = resultList.size();

        // One more slot to send the new version
        if (backupVersionMismatch)
        {
            resultSize++;
        }

        IDirElement[] result = new IDirElement[resultSize];
        for (int i = 0; i < resultList.size(); i++)
        {
            result[i] = (IDirElement)resultList.get(i);
        }

        // Pass back to the caller the new backup version
        if (backupVersionMismatch)
        {
            result[resultSize - 1] = getElement(DS_VERSION_INFO_PATH, false);
        }

        if ((m_traceMask & IComponent.TRACE_VERBOSE) > 0)
        {
            for (int i = 0; i < result.length; i++)
            {
                trace(TRACE_CONTAINER_ACCESS + TRACE_ALL_DS_ACCESS, result[i].toString());
            }
        }

        return result;
    }

    // Used by containers to reconcile their storage-logical map. Receives a storage-to-logical map and reports back
    // those logical names that were modified and those names that were deleted (puts null for the logical name of
    // those).
    @Override
    public HashMap getUpdatedLogicalNames(HashMap names)
    {
        trace(TRACE_CONTAINER_ACCESS + TRACE_ALL_DS_ACCESS, "getUpdatedLogicalNames: " + names.size() + " names");

        try
        {
            m_lock.readLock();
            HashMap correctionMap = new HashMap();
            Iterator iterator = names.keySet().iterator();
            while (iterator.hasNext())
            {
                String storageName = (String)iterator.next();
                String logicalName = null;
                try
                {
                    logicalName = m_logicalNameSpace.logicalFromStorage(storageName);
                }
                catch (Exception e)
                {
                }
                if (logicalName != null && ((String)names.get(storageName)).equalsIgnoreCase(logicalName))
                {
                    continue;
                }
                else
                {
                    correctionMap.put(storageName, logicalName);
                }
            }
            return correctionMap;
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    public void unsubscribeNaming()
    {
        m_logicalNameSpace.unsubscribe();
    }

    @Override
    public void subscribeNaming(INamingListener listener)
    {
        m_logicalNameSpace.subscribe(listener);
        m_dsHandlers.subscribe(listener);
    }

    @Override
    public INotificationConsumer subscribeAll(IArrayElementNotificationListener listener)
    {
        NotificationConsumer consumer = new NotificationConsumer(listener);
        m_notificationManager.setConsumer(consumer);
        return consumer;
    }

    @Override
    public IView getView()
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getView");
        try
        {
            m_lock.readLock();
            return new View((IDirElement)getElement(VIEW_ELEMENT, true));
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public IView storeViewInternal(IView view)
    throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "setView");
        try
        {
            storeInternal(VIEW_ELEMENT_ENTITY_NAME, ((View)view).getConfigElement(), true);
            return new View((IDirElement)m_storage.getElement(VIEW_ELEMENT_ENTITY_NAME));
        }
        catch (StorageException e)
        {
            throw new DirectoryServiceException(e.toString());
        }
    }

    @Override
    public IView setView(IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "setView");
        try
        {
            m_lock.writeLock();
            setElement(((DeltaView)view).getElement(), null);
            return getView();
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public INextVersionToken setViewGetToken(IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "setViewGetToken");

        NextVersionToken token = null;
        boolean startedRememberIDs = false;

        try
        {
            m_lock.writeLock();
            if (m_notificationManager != null)
            {
                startedRememberIDs = m_notificationManager.rememberNewIds();
            }
            setElement(((DeltaView)view).getElement(), null);
        }
        finally
        {
            if (startedRememberIDs)
            {
                token = createNextVersionToken(false);
            }
            m_lock.releaseLock();
        }

        return token;
    }

    @Override
    public IDirElement cloneElement(String elementName, String newName, boolean createTemplate, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return cloneElementInternal(null, elementName, newName, createTemplate, view);
    }

    @Override
    public IDirElement cloneElement(IBasicElement delta, String newName, boolean createTemplate, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return cloneElementInternal(delta, null, newName, createTemplate, view);
    }

    @Override
    public IDirElement cloneElement(String elementName, String newName, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return cloneElementInternal(null, elementName, newName, false, view);
    }

    @Override
    public IDirElement cloneElement(IBasicElement delta, String newName, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return cloneElementInternal(delta, null, newName, false, view);
    }

    private IDirElement cloneElementInternal(IBasicElement delta, String oldName, String newName, boolean createTemplate, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        String elementName = (delta != null) ? delta.getIdentity().getName() : oldName;
        if (delta != null && !(delta instanceof IDeltaElement))
        {
            throw new DirectoryServiceException("The delta parameter must be the modified result of IDirElement.doneUpdate().");
        }
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "cloneElement from " + elementName + " to " + newName);
        try
        {
            m_lock.writeLock();

            if (m_authentication != null)
            {
                m_authentication.okToModify(newName);
            }

            Element original = (Element)getElementAsIs(elementName, false);

            if (original == null)
            {
                throw new DirectoryServiceException('"' + elementName + "\" was deleted.");
            }

            // When the DS runs in the same process with the application we must clone the delta to prevent side effects
            // on the application
            DeltaElement deltaClone = null;
            if (m_noContainer && delta != null)
            {
                deltaClone = ((DeltaElement)delta).createClone();
            }
            else
            {
                deltaClone = (DeltaElement)delta;
            }

            if (original.isSubclassedElement())
            {
                if (createTemplate)
                {
                    throw new DirectoryServiceException("Cannot create a template from sub-classed element \"" + elementName + "\".");
                }
                return cloneSubclassedElement(original, deltaClone, newName, view);
            }

            boolean cloningSuper = original.isSuperElement();
            boolean isTemplate = original.isTemplate();

            Element newElement = (Element)original.createWritableClone();
            if (deltaClone != null)
            {
                newElement.doApplyDelta(deltaClone);

                if (createTemplate && !isTemplate)
                {
                    newElement.setTemplate();
                }

                if (newElement.getSuperElementName() != null)
                {
                    throw new DirectoryServiceException("doneUpdateForSubclassing() was called rather than doneUpdate().");
                }
                // Parts of the element copied from the delta might be read-only - we have to make it writable
                newElement.setReadOnly(false);
            }
            else if (createTemplate && !isTemplate)
            {
                newElement.setTemplate();
            }

            Identity id = (Identity)newElement.getIdentity();
            id.setName(newName);
            ((ElementIdentity)id).setVersion(IElementIdentity.INITIAL_VERSION);
            if (cloningSuper)
            {
                newElement = cleanupSuperElement(newElement, true, false);
            }
            setElement(newElement, view);
            return getElement(newName, true);
        }
        catch (VersionMisMatchException e)
        {
            throwVersionException(elementName, elementName + " was modified by another application.");
            return null;
        }

        finally
        {
            m_lock.releaseLock();
        }
    }

    // clone a subclassed element in two stages: 1) Use the delta of the original element to subclass again from the
    // same
    // super. 2) Update the new clone with the cloning delta (if not null).
    private IDirElement cloneSubclassedElement(Element original, DeltaElement delta, String newName, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        IElementIdentity originalID = original.getIdentity();

        Element oldView = (Element)getElement(VIEW_ELEMENT, false);

        String originalName = originalID.getName();

        if (delta != null && delta.getIdentity().getVersion() != originalID.getVersion())
        {
            throwVersionException(originalName, originalName + " was modified by another application.");
        }

        String superName = original.getSuperElementName();

        Element superElement = (Element)getElementAsIs(superName, true);

        if (superElement == null)
        {
            throw new DirectoryServiceException("The super element \"" + superName + "\" of \"" + originalName + "\" is missing.");
        }

        byte[] deltaBytes = superElement.getSubclassedDelta(originalName);
        if (deltaBytes == null)
        {
            throw new DirectoryServiceException("The super element \"" + superName + "\" of \"" + originalName + "\" does not contain the sub-classing information.");
        }

        DeltaElement subclassingDelta = DeltaElement.fromBytes(deltaBytes);
        DeltaElement subclassingDeltaCopy = subclassingDelta.createClone();

        // versions must much for a successful subclassElement() call below
        ((ElementIdentity)subclassingDeltaCopy.getIdentity()).setVersion(superElement.getIdentity().getVersion());

        if (delta == null)
        {
            return (Element)subclassElement(subclassingDeltaCopy, newName, view, true);
        }

        boolean transactOK = false;
        Element realizedElement = null;
        try
        {
            m_trManager.join();
            Element realizedFirstPhase = (Element)subclassElement(subclassingDeltaCopy, newName, view, false);

            DeltaElement deltaCopy = delta.createClone();
            deltaCopy.setIdentity(((ElementIdentity)realizedFirstPhase.getIdentity()).createClone());
            setElement(deltaCopy, null, false);
            transactOK = true;
        }
        finally
        {
            if (transactOK)
            {
                realizedElement = (Element)getElement(newName, false);

                // Because of our two phases the version inrealizedElement is 2, but since it is a new element,
                // skipping one version should not cause problems at the local container cache
                notifyMods(new ModificationItem(realizedElement), createViewMod(view, oldView));
            }

            m_trManager.leave(transactOK);
        }

        return (realizedElement != null ? (IDirElement)realizedElement.createWritableClone() : null);
    }

    private Element doRealizeSub(Element superElement, byte[] deltaBytes, DeltaElement delta, String newName, IElementIdentity subID)
    {
        try
        {
            return doRealizeSub0(superElement, deltaBytes, delta, newName, subID);
        }
        catch (VersionOutofSyncException e)
        {
            throw new Error(e.toString()); // Should not happen because we are realizing using delta that we already
            // established
        }
    }

    private Element doRealizeSub(Element superElement, DeltaElement delta, String newName)
    throws VersionOutofSyncException
    {
        return doRealizeSub0(superElement, null, delta, newName, null);
    }

    private Element doRealizeSub0(Element superElement, byte[] deltaBytes, DeltaElement delta, String newName, IElementIdentity subID)
    throws VersionOutofSyncException
    {
        // When the DS runs in the same process with the application we must clone the delta to prevent side effects
        // on the application
        if (m_noContainer)
        {
            delta = delta.createClone();
        }

        try
        {
            Element newElement = (Element)superElement.createWritableClone();
            newElement = cleanupSuperElement(newElement, true, true);
            Identity id = (Identity)newElement.getIdentity();

            // When we apply the delta in order to realize a subclassed element the versions might not match
            // because the super element might have been changed since we originally created the delta. That is ok
            // Since we always adjust the delta to the super modifications (or the modification occur in system
            // attributes
            // irrelevant to the subclassed item). So we adjust the version of the super to prevent the doApplyDelta
            // from
            // failing.
            if (subID != null)
            {
                ((ElementIdentity)id).setVersion(delta.getIdentity().getVersion());
            }

            newElement.doApplyDelta(delta);
            id.setName(newName);
            if (deltaBytes != null)
            {
                newElement.setSubclassingDelta(deltaBytes);
            }

            // It's a new sub element - just created
            if (subID == null)
            {
                initializeElementVersion((ElementIdentity)id);
            }

            // The identity is of the sub element, not the super
            if (subID != null)
            {
                newElement.setNewIdentity(((ElementIdentity)subID).createClone());
            }

            return newElement;

        }
        catch (VersionMisMatchException e)
        {
            throwVersionException(delta.getIdentity().getName(), delta.getIdentity().getName() + " was modified by another application.");
            return null;
        }

    }

    static void initializeElementVersion(ElementIdentity id)
    {
        id.setVersion(IElementIdentity.INITIAL_VERSION + 1);
        id.setCreationTimestamp(System.currentTimeMillis());
    }

    @Override
    public final void close()
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "close");
        close(true);
    }

    @Override
    public final void close(boolean normalClose)
    throws DirectoryServiceException
    {
        close(normalClose, false);
    }

    // This version is directly called only from TIDE tests with close the
    // DS with normalClose=false and then delete the lock file right after.
    // Because deleting the lock file doesn't do anything with the PSEStorage
    // implementation, we added this method that will delete the element.
    public synchronized void close(boolean normalClose, boolean forceDeleteLock)
    throws DirectoryServiceException
    {
        if (m_storage == null)
        {
            return;
        }

        // Must be done before we lock since the thread that imports external users might be waiting
        // on the lock while holding a monitor which will prevent us from closing it - that will cause a
        // deadlock
        if (m_authentication != null)
        {
            m_authentication.close();
        }
        boolean storageClosed = false;
        boolean systemClosed = false;
        try
        {
            // If we are in backup state we can close without getting a arite lock
            if (!m_lock.hasNoWriteLock())
            {
                m_lock.writeLock();
            }
            try
            {
                if (m_trManager != null)
                {
                    m_trManager.close();
                }
                if (m_usePSE_STORAGE && (normalClose || forceDeleteLock))
                {
                    m_storage.startTransaction();
                }
                if (normalClose)
                {
                    // if we're doing a dsadmin dump, we want to leave the
                    // id cache the way we found it
                    if (!Boolean.getBoolean("DSDUMP"))
                    {
                        m_storage.deleteElement(IDCACHE_ELEMENT_ENTITY_NAME, true);
                        if (m_idCache != null)
                        {
                            m_storage.setElement(IDCACHE_ELEMENT_ENTITY_NAME, m_idCache.wrapAsElement(IDCACHE_ELEMENT));
                        }
                    }

                    m_storage.deleteElement(BACKUP_ELEMENT_ENTITY_NAME, true);
                    if (m_useFS_STORAGE)
                    {
                        m_storage.deleteElement(LOCK_ELEMENT_ENTITY_NAME, true);
                    }
                    m_storage.deleteElement(OPEN_ELEMENT_ENTITY_NAME, true);
                }

                if (!normalClose && forceDeleteLock && m_useFS_STORAGE)
                {
                    m_storage.deleteElement(LOCK_ELEMENT_ENTITY_NAME, true);
                }

                if (m_usePSE_STORAGE && (normalClose || forceDeleteLock))
                {
                    m_storage.commitTransaction();
                }
                	m_storage.close();
                storageClosed = true;
                if (m_useFS_STORAGE && m_blobStorage != null)
                {
                    m_blobStorage.close();
                }

                if (m_systemStorage != null)
                {
                    m_systemStorage.close();
                }

                systemClosed = true;
            }
            catch (StorageException e) // Should never happen
            // mrd 05/10/06 it does happen with the
            // encryption test in TIDE
            {
                logMessage("Close failed, trace follows...", e, Level.WARNING);

                String addErrorMessage = "";
                try
                {
                    if (!storageClosed)
                    {
                        m_storage.close();
                    }
                    if (!systemClosed)
                    {
                        m_systemStorage.close();
                    }
                }
                catch (Exception e2)
                {
                    addErrorMessage = " Also unable to close the underlying DS storage: " + e2.toString();
                }
                Error error = new Error();
                error.initCause(e);
            }

            m_open = false;
            m_notificationManager.setConsumer(null);
            m_localListener = null;
            m_storage = null;
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    private void closeFiles()
    throws DirectoryServiceException
    {
        try
        {
            if (m_useFS_STORAGE)
            {
                if (m_trManager != null)
                {
                    m_trManager.closeFiles();
                }
                if (m_blobStorage != null)
                {
                    m_blobStorage.closeFiles();
                }
            }
            if (m_storage != null)
            {
                m_storage.closeFiles();
            }
        }
        catch (StorageException e)
        {
            throw convertException(e);
        }
    }

    private void openFiles()
    throws DirectoryServiceException
    {
        try
        {
            if (m_useFS_STORAGE)
            {
                if (m_trManager != null)
                {
                    m_trManager.openFiles();
                }
                if (m_blobStorage != null)
                {
                    m_blobStorage.openFiles();
                }
            }
            if (m_storage != null)
            {
                m_storage.openFiles();
            }
        }
        catch (StorageException e)
        {
            throw convertException(e);
        }
    }

    IStorage getStorage()
    {
        return m_storage;
    }

    static EntityName validateName(String name)
    throws DirectoryServiceException
    {
        try
        {
            return new EntityName(name);
        }
        catch (ConfigException e)
        {
            throw new DirectoryServiceException(e.toString());
        }
    }

    private void validateOpen()
    throws DirectoryServiceException
    {
        if (!m_open)
        {
            throw new DirectoryServiceClosedException("The Directory Service object is closed.");
        }
    }

    private void openStorage()
    throws DirectoryServiceException
    {
        boolean firstTime = false;
        trace(TRACE_ALL_DS_ACCESS, "Checking system elements...");
        try
        {
            if (!m_storage.directoryExists(SYSTEM_DIRECTORY_PATH_ENTITY_NAME))
            {
                if (m_usePSE_STORAGE)
                {
                    m_storage.startTransaction();
                }
                m_storage.createDirectory(SYSTEM_DIRECTORY_PATH_ENTITY_NAME);
                if (m_usePSE_STORAGE)
                {
                    m_storage.commitTransaction();
                }
                firstTime = true;
            }
        }
        catch (StorageException e)
        {
            throw convertException(e);
        }

        // Checks the version
        try
        {
            IDirElement versionElement = m_storage.getElement(VERSION_ELEMENT_ENTITY_NAME);
            if (versionElement == null)
            {
                versionElement = ElementFactory.createElement(VERSION_ELEMENT_PATH, "version", MF_CONFING_VERSION);
                IAttributeSet attributes = versionElement.getAttributes();
                attributes.setIntegerAttribute("VERSION", new Integer(DS_STORAGE_STRUCTURE_VERSION));
                if (m_usePSE_STORAGE)
                {
                    m_storage.startTransaction();
                }
                m_storage.setElement(VERSION_ELEMENT_ENTITY_NAME, (IDirElement)versionElement.doneUpdate());
                if (m_usePSE_STORAGE)
                {
                    m_storage.commitTransaction();
                }
            }
            else
            {
                IAttributeSet attributes = versionElement.getAttributes();
                Integer dsVersion = (Integer)attributes.getAttribute("VERSION");
                if (dsVersion.intValue() != DS_STORAGE_STRUCTURE_VERSION)
                {
                    throw new DirectoryServiceException("The directory service storage version of domain \"" + m_domain + "\" mismatches the " + "version of the software.\nStorage version is: " + dsVersion + ". Software version is: "
                                                        + DS_STORAGE_STRUCTURE_VERSION + ".");
                }
            }

        }

         catch (DSEncryptionException e)
         {
             throw new DecryptionException(e.getMessage());
         }

        catch (StorageException e)
        {
            throw convertException(e);
        }
        catch (ReadOnlyException e)
        {
            throw new Error(e.toString());
        }
        catch (AttributeSetTypeException e)
        {
            throw new Error(e.toString());
        }
        catch (ConfigException e)
        {
            throw new Error(e.toString());
        }

        // Checks the lock and the backup state
        try
        {
            IDirElement backupElement = m_storage.getElement(BACKUP_ELEMENT_ENTITY_NAME);
            if (backupElement != null)
            {
                StringBuffer msg = new StringBuffer();
                msg.append("The domain storage reflects that either the store is currently being backed up or it is a backup copy that must be prepared for use.");
                msg.append(IContainer.NEWLINE);
                msg.append("You must either shutdown the currently running Directory Service (if appropriate) or run the dsAdmin script with the prepare_backup_image_for_restore option before restarting this container.");
                throw new DirLockedException(msg.toString());
            }

            if (m_useFS_STORAGE)
            {
                IDirElement lockElement = m_storage.getElement(LOCK_ELEMENT_ENTITY_NAME);
                if (lockElement == null)
                {
                    lockElement = ElementFactory.createElement(LOCK_ELEMENT_PATH, "lock", MF_CONFING_VERSION);
                    if (m_usePSE_STORAGE)
                    {
                        m_storage.startTransaction();
                    }
                    m_storage.setElement(LOCK_ELEMENT_ENTITY_NAME, (IDirElement)lockElement.doneUpdate());
                    if (m_usePSE_STORAGE)
                    {
                        m_storage.commitTransaction();
                    }
                }
                else
                {
                    StringBuffer msg = new StringBuffer();
                    msg.append("A lock was found on the \"").append(m_domain).append("\" domain storage.");
                    msg.append(IContainer.NEWLINE);
                    msg.append("There is either a Directory Service running already or the previous session was not properly terminated. ");
                    msg.append("If you are sure there is no other Directory Service running then you may remove the ");
                    msg.append((getCanonicalPath(m_domainDir) + IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_DATA_DIR + LOCK_ELEMENT_ENTITY_NAME.getName()).replace('/', File.separatorChar));
                    msg.append(" file and restart this program.");
                    throw new DirLockedException(msg.toString());
                }
            }
        }
        catch (StorageException e)
        {
            throw convertException(e);
        }
        catch (ReadOnlyException e)
        {
            throw new Error(e.toString());
        }

        // Create an initial backup version if does not exist

        try
        {
            // The storage update operations are inside a storage transaction
            // in the case of PSEStorage. They cannot be inside a storage transaction
            // for FSStorage because the commit would potentially commit a transaction
            // that's waiting to get rolledback after the storage is opened (this method)
            IDirElement backupVersionElement = m_storage.getElement(BACKUP_VERSION_ELEMENT_ENTITY_NAME);
            if (backupVersionElement == null)
            {
                backupVersionElement = ElementFactory.createElement(BACKUP_VERSION_ELEMENT_PATH, "backup_version", MF_CONFING_VERSION);
                IAttributeSet attributes = backupVersionElement.getAttributes();
                m_backupVersion = System.currentTimeMillis();
                trace(TRACE_STARTUP, "Creating an initial backup version " + m_backupVersion);
                attributes.setLongAttribute(IDirectoryMFService.BACKUP_VERSION_ATTR, new Long(m_backupVersion));
                if (m_usePSE_STORAGE)
                {
                    m_storage.startTransaction();
                }
                m_storage.setElement(BACKUP_VERSION_ELEMENT_ENTITY_NAME, (IDirElement)backupVersionElement.doneUpdate());
                if (m_usePSE_STORAGE)
                {
                    m_storage.commitTransaction();
                }

            }
            else
            {
                IAttributeSet attributes = backupVersionElement.getAttributes();
                m_backupVersion = ((Long)attributes.getAttribute(IDirectoryMFService.BACKUP_VERSION_ATTR)).longValue();
                trace(TRACE_STARTUP, "The backup version is " + m_backupVersion);
            }

            // create an initial last backup status element if it does not exist

            IDirElement backupStatus = m_storage.getElement(BACKUP_STATUS_ELEMENT_ENTITY_NAME);

            if (backupStatus == null)
            {
                backupStatus = ElementFactory.createElement(BACKUP_STATUS_ELEMENT_PATH, "1", MF_CONFING_VERSION);
                saveBackupStatusAttrs(backupStatus);
            }
            else
            {
                readBackupStatusAttrs(backupStatus);
            }

        }
        catch (StorageException e)
        {
            throw convertException(e);
        }
        catch (ReadOnlyException e)
        {
            throw new Error(e.toString());
        }
        catch (AttributeSetTypeException e)
        {
            throw new Error(e.toString());
        }
        catch (ConfigException e)
        {
            throw new Error(e.toString());
        }

        // Need recovery?
        IDirElement cacheElement = null;
        m_idCache = null;
        try
        {
            IDirElement openElement = m_storage.getElement(OPEN_ELEMENT_ENTITY_NAME);
            if (openElement == null)
            {
                // It appears the DS was closed properly - mark it as open
                openElement = ElementFactory.createElement(OPEN_ELEMENT_PATH, "dsisopen", MF_CONFING_VERSION);
                if (m_usePSE_STORAGE)
                {
                    m_storage.startTransaction();
                }
                m_storage.setElement(OPEN_ELEMENT_ENTITY_NAME, (IDirElement)openElement.doneUpdate());
                if (m_usePSE_STORAGE)
                {
                    m_storage.commitTransaction();
                }
                // Can we get the ID cache element and extract the cache from it?
                try
                {
                    cacheElement = m_storage.getElement(IDCACHE_ELEMENT_ENTITY_NAME);
                    m_idCache = new IDCache(cacheElement);
                }
                catch (Throwable t)
                {
                    // When we create the DS for the first time there is no cache element and it's ok
                    if (!firstTime)
                    {
                        logMessage("The Directory Service storage is damaged - automatic recovery will be performed", Level.WARNING);
                    }
                }

            }
            else
            {
                logMessage("Prior shutdown did not execute correctly - automatic recovery will be performed", Level.WARNING);
            }

            boolean needsRecovery = false;
            if (m_idCache == null && !firstTime)
            {
                needsRecovery = true;
            }

            if (needsRecovery)
            {
                logMessage("Starting Directory Service recovery...", Level.INFO);
            }

            // Must be done before new IDCache(this) so the cache get the correct stuff - after the transaction recovery
            if (m_usePSE_STORAGE)
            {
                m_trManager = new TransactionManager(m_domainDir, new IStorage[]
                { m_storage }, true, m_notificationManager, m_storageType);
            }
            else
            {
                // FS_STORAGE
                m_trManager = new TransactionManager(m_domainDir, new IStorage[]
                { m_storage, m_blobStorage }, true, m_notificationManager, m_storageType);
            }
            trace(TRACE_STARTUP, "Created the transaction manager");

            // m_idCache will be an empty element if we're opening a restored backup for
            // the first time. It is saved as an empty element to the backed up storage
            // by the online backup.
            // 2013-01-08 We need to go through every element when doing a dsadmin dumpto find corruption, 
            // so re-generate the cache
            if (m_idCache == null ||m_idCache.isEmpty() || Boolean.getBoolean("DSDUMP"))
            {
                trace(TRACE_STARTUP, "reading the ID cache...");
                m_idCache = new IDCache(this);
                trace(TRACE_STARTUP, "...reading of ID cache done");
            }

            m_trManager.setCache(m_idCache);

            if (needsRecovery)
            {
                logMessage("...Directory Service recovery complete", Level.INFO);
            }
        }
        catch (StorageException e)
        {
            throw convertException(e);
        }
        catch (ReadOnlyException e)
        {
            throw new Error(e.toString());
        }

    }


    @Override
    public String[] getCollectionSubscribers() throws DirectoryServiceException
    {
        try
        {

            synchronized (m_subscribers_lock)
            {
                return m_systemStorage.collectionToStringArray(SUBSCRIBERS_COLLECTION_NAME);
            }

        }
        catch (Exception e)
        {
            e.printStackTrace();
            throw new DirectoryServiceException(e.getMessage());
        }
    }

    @Override
    public void removeSubscribers(String[] subscribers)
    {
        try
        {

            synchronized (m_subscribers_lock)
            {
                m_systemStorage.removeFromCollection(SUBSCRIBERS_COLLECTION_NAME, subscribers);
            }
        }
        catch (Exception e)
        {
            logMessage("Failed to remove subscribers, trace follows...", e, Level.WARNING);
        }
    }

    @Override
    public void addSubscriber(String subscriber)
    {
        try
        {

            synchronized (m_subscribers_lock)
            {
                m_systemStorage.addToCollection(SUBSCRIBERS_COLLECTION_NAME, subscriber);
            }

        }
        catch (Exception e)
        {
            logMessage("Failed to add subscriber, trace follows...", e, Level.WARNING);
            e.printStackTrace();
        }
    }


    @Override
    public String[] getSubscribers()
    throws DirectoryServiceException
    {
        try
        {
            // There is no reason to use m_lock for the SUBSCRIBERS_ELEMENT which is accessed
            // only by the system
            synchronized (m_subscribers_lock)
            {
                IDirElement subscribersElement = m_systemStorage.getElement(SUBSCRIBERS_ELEMENT_ENTITY_NAME);

                if (subscribersElement == null)
                {
                    return IEmptyArray.EMPTY_STRING_ARRAY;
                }

                return (String[])unwrapFromElement(subscribersElement);
            }

        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString());
        }
    }

    @Override
    public void setSubscribers(String[] subList)
    throws DirectoryServiceException
    {
        try
        {
            // There is no reason to use m_lock for the the SUBSCRIBERS_ELEMENT which is accessed
            // only by the system
            synchronized (m_subscribers_lock)
            {
                m_systemStorage.deleteElement(SUBSCRIBERS_ELEMENT_ENTITY_NAME);
                m_systemStorage.setElement(SUBSCRIBERS_ELEMENT_ENTITY_NAME, wrapObjectAsElement(subList, SUBSCRIBERS_ELEMENT_PATH, "subscribers"));
            }

        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString());
        }
    }

    private void createSchemaElements(boolean isRestrictedBackupDS)
    throws DirectoryServiceException
    {
        if (isRestrictedBackupDS)
        {
            m_backRefMgr = new BackReferenceMgr(this, isRestrictedBackupDS);
            return;
        }
            
        try
        {
            m_lock.writeLock();
            if (getIdentity(IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_SCHEMA_DIR) == null)
            {
                createDirectory(IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_SCHEMA_DIR);
            }
            m_backRefMgr = new BackReferenceMgr(this, false);

            try
            {
                if (getIdentity(VIEW_ELEMENT) == null)
                {
                    trace(TRACE_STARTUP, "Creating the initial view element");

                    IDirElement viewElement = ElementFactory.createElement(VIEW_ELEMENT, ViewConstants.VIEW_TYPE, ViewConstants.VIEW_VERSION);
                    setElement(viewElement.doneUpdate(), null);
                }
            }
            // The following exceptions mean a bug in the code
            catch (ReadOnlyException e)
            {
                throw new Error(e.toString());
            }
            catch (VersionOutofSyncException e)
            {
                throw new Error(e.toString());
            }
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    // Creates the hints element id doesn't already exist
    private void createHintsElement()
    throws DirectoryServiceException
    {
        if (getIdentity(STORAGE_HINTS_ELEMENT) != null)
        {
            return;
        }
        try
        {
            trace(TRACE_STARTUP, "Creating the initial storage hints element");

            IDirElement storageHints = ElementFactory.createElement(STORAGE_HINTS_ELEMENT, "storage_hints", MF_CONFING_VERSION);
            IAttributeSet atts = storageHints.getAttributes();
            atts.createAttributeSet(HINTS_ATT);
            setElement(storageHints.doneUpdate(), null);
        }
        // The following exceptions mean a bug in the code
        catch (ReadOnlyException e)
        {
            throw new Error(e.toString());
        }
        catch (VersionOutofSyncException e)
        {
            throw new Error(e.toString());
        }
        catch (AttributeSetTypeException e)
        {
            throw new Error(e.toString());
        }
        catch (ConfigException e)
        {
            throw new Error(e.toString());
        }
    }

    private void createJNDIContext()
    throws DirectoryServiceException
    {
        trace(TRACE_STARTUP, "Creating the initial JNDI context");

        if (getIdentity(IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_JNDI_DIR) == null)
        {
            try
            {
                m_lock.writeLock();
                createDirectory(IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_JNDI_DIR);
            }
            finally
            {
                m_lock.releaseLock();
            }
        }
    }

    private void storeInternal(EntityName name, IDirElement element)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        storeInternal(name, element, false);
    }

    private void storeInternal(EntityName name, IDirElement element, boolean byLogicalViewer)
    throws DirectoryServiceException, VersionOutofSyncException
    {

        // Fix 00016691
        // Note that we could make m_logicalNameSpace consistent by calling m_logicalNameSpace.reset(). But
        // that will be very expensive if there is a rogue client that keeps updating the view directly. Also
        // clients that use the FS API will not get correct notifications. So if there is allready an FS
        // client we throw an exception. If not then we mark m_logicalNameSpace as inconsistent and will throw
        // an exception if somebody tries to use it before it's reconciled.
        // Mari Davila 6/23/04 - If the upgrade is happening, let it update the view without this error
        if (name.getName().equals(VIEW_ELEMENT) && !byLogicalViewer && m_logicalNameSpace != null && (!System.getProperty("MQ_UPGRADE", "false").equals("true")))
        {
            if (m_FSInterfaceIsUsed && !m_noContainer)
            {
                throw new DirectoryServiceException("The DS admin API cannot be used to update the view concurrently with FS API clients.");
            }
            m_logicalNameSpace.setNotConsistent();
        }

        try
        {
            m_storage.setElement(name, element);
            m_idCache.addElement(name, element.getIdentity());
        }
        catch (StorageException e)
        {
            throw convertException(e);
        }

    }

    void notifyMods(Element element, boolean noBeforeImage)
    {
        notifyMods(new ModificationItem(element, noBeforeImage), null);
        m_notificationManager.doNotify();
    }

    void notifyMods(Element element)
    {
        notifyMods(new ModificationItem(element), null);
        m_notificationManager.doNotify();
    }

    void groupNotifyMods(ArrayList group, boolean hasDelete)
    {
        m_notificationManager.addGroupNotification(group, hasDelete);
        m_notificationManager.doNotify();
    }

    // modItem0 should never be null
    void notifyMods(ModificationItem modItem0, ModificationItem modItem1)
    {
        IBasicElement mod0 = modItem0.getModification();
        IBasicElement mod1 = null;
        if (modItem1 != null)
        {
            mod1 = modItem1.getModification();
        }

        if (mod0 instanceof IDirElement)
        {
            ((Element)mod0).setReadOnly(true);
        }

        if (mod1 != null && (mod1 instanceof IDirElement))
        {
            ((Element)mod1).setReadOnly(true);
        }

        int numMods = (modItem1 == null) ? 1 : 2;
        ModificationItem[] modList = new ModificationItem[numMods];

        modList[0] = modItem0;

        if (modItem1 != null)
        {
            modList[1] = modItem1;
        }

        m_notificationManager.addNotifications(modList);

    }

    private BeforeAfterPair prepareToSet(IBasicElement element, IDirElement oldElement0)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        String elementName = element.getIdentity().getName();

        if (elementName.startsWith(SYSTEM_DIRECTORY_PATH))
        {
            throw new DirectoryServiceException("It is illegal to modify the \"" + SYSTEM_DIRECTORY_PATH + "\" directory.");
        }

        EntityName eName = validateName(elementName);

        long elementVersion = element.getIdentity().getVersion();
        Element oldElement = (Element)oldElement0;
        if (oldElement == null)
        {
            try
            {
                oldElement = (Element)m_storage.getElement(eName);
                if (element instanceof Element && oldElement != null)
                {
                    // check if this is the second phase import of a blob envelope element.
                    // In this case we want to ignore the element we've left in storage.
                    // The envelope element would have been specially marked with
                    // BLOB_ENVELOPE_ELEMENT_IMPORT by the import manager
                    Boolean blobImport = (Boolean)Blob.getBlobState((Element)element, IBlob.BLOB_ENVELOPE_ELEMENT_IMPORT);
                    if (blobImport != null && blobImport.booleanValue())
                    {
                        oldElement = null;
                    }
                }
            }
            catch (StorageException e)
            {
                throw convertException(e);
            }
        }
        else
        {
            oldElement = (Element)oldElement.createClone();
            oldElement.setReadOnly(true);
        }

        IDeltaDirElement delta = null;

        if (oldElement != null && element instanceof IDirElement)
        {
            throwVersionException(elementName, "Element \"" + elementName + "\" already exists.");
        }

        if (oldElement == null && element instanceof IDeltaDirElement)
        {
            throwVersionException(elementName, "Element \"" + elementName + "\" does not exist.");
        }

        boolean oldElementIsSubclassed = oldElement != null && oldElement.getSuperElementName() != null;

        if (element instanceof IDirElement && elementVersion != IElementIdentity.INITIAL_VERSION)
        {
            throwVersionException(elementName, "Element \"" + elementName + "\" is new but " + "its version is " + elementVersion);
        }
        if (element instanceof IDeltaElement)
        {
            // When the DS runs in the same process with the application we must clone the delta to prevent side effects
            // on the application
            if (m_noContainer)
            {
                delta = ((DeltaElement)element).createClone();
            }
            else
            {
                delta = (DeltaElement)element;
            }

            if (delta.isDeleted())
            {
                throw new Error("deleteElement should be used for deleting elements.");
            }

            IElementIdentity deltaIdentity = delta.getIdentity();
            IElementIdentity oldIdentity = oldElement.getIdentity();

            if (!oldIdentity.equalEntity(deltaIdentity))
            {
                throwVersionException(deltaIdentity.getName(), "Element \"" + deltaIdentity.getName() + "\" with creation timestamp " + deltaIdentity.getCreationTimestamp() + " was not found." + " The creation timestamp in the directory is "
                                                               + oldIdentity.getCreationTimestamp());
            }

        }

        IDirElement newElement = null;
        if (element instanceof IDirElement)
        {
            newElement = (IDirElement)((IDirElement)element).createWritableClone();
            ElementIdentity id = (ElementIdentity)newElement.getIdentity();
            initializeElementVersion(id);
        }
        else
        {
            newElement = (IDirElement)oldElement.createWritableClone();
            try
            {
                ((Element)newElement).doApplyDelta(delta);
                boolean newElementIsSubclassed = newElement.getSuperElementName() != null;

                // doneUpdateForSubclassing() was called instead of doneUpdate()
                if (newElementIsSubclassed && !oldElementIsSubclassed)
                {
                    throw new DirectoryServiceException("doneUpdateForSubclassing() was called rather than doneUpdate().");
                }

                // Parts of the element copied from the delta might be read-only - we have to make it writable
                ((Element)newElement).setReadOnly(false);
            }
            catch (VersionMisMatchException e)
            {
                throwVersionException(oldElement.getIdentity().getName(), "Directory version: " + oldElement.getIdentity().getVersion() + "  Delta version: " + delta.getIdentity().getVersion());
            }
        }
        return new BeforeAfterPair(oldElement, newElement);
    }
    
    @Override
    public final void logMessage(String message, int severityLevel)
    {
        if (m_logger == null)
        {
            System.err.println("(" + Level.LEVEL_TEXT[severityLevel] + ") " + message);
        }
        else
        {
            m_logger.logMessage(message, severityLevel);
        }
    }
    
    @Override
    public final void logMessage(String message, Throwable throwable, int severityLevel)
    {
        if (m_logger == null)
        {
            System.err.println("(" + Level.LEVEL_TEXT[severityLevel] + ") " + message);
            throwable.printStackTrace();
        }
        else
        {
            m_logger.logMessage(message, throwable, severityLevel);
        }
    }
    
    @Override
    public final void trace(int mask, String message)
    {
        boolean trace = (m_traceMask & mask) != 0;
        if (!trace)
        {
            return;
        }
        
        if (m_logger == null)
        {
            System.err.println("(" + Level.LEVEL_TEXT[Level.TRACE] + ") " + message);
        }
        else
        {
            m_logger.logMessage(message, Level.TRACE);
        }
    }
    
    @Override
    public void trace(int mask, String message, Throwable throwable)
    {
        boolean trace = (m_traceMask & mask) != 0;
        if (!trace)
        {
            return;
        }
        
        if (m_logger == null)
        {
            System.err.println("(" + Level.LEVEL_TEXT[Level.TRACE] + ") " + message);
            throwable.printStackTrace();
        }
        else
        {
            m_logger.logMessage(message, throwable, Level.TRACE);
        }
    }

    void throwVersionException(String elementName, String message)
    throws VersionOutofSyncException
    {
        throw new VersionOutofSyncException("[" + elementName + "] " + message);
    }

    private Object unwrapFromElement(IDirElement element)
    throws DirectoryServiceException
    {
        try
        {
            byte[] cacheData = (byte[])element.getAttributes().getAttribute("DATA_ATTRIBUTE");
            ByteArrayInputStream in = new ByteArrayInputStream(cacheData);
            ObjectInputStream objectIn = new ObjectInputStream(in);
            return objectIn.readObject();

        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString());
        }
    }

    private IDirElement wrapObjectAsElement(Object o, String elementName, String elementType)
    {
        try
        {
            IDirElement cacheElement = ElementFactory.createElement(elementName, elementType, MF_CONFING_VERSION);
            IAttributeSet attributes = cacheElement.getAttributes();

            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ObjectOutputStream objectOut = new ObjectOutputStream(out);
            objectOut.writeObject(o);
            byte[] bytes = out.toByteArray();
            objectOut.close();

            attributes.setBytesAttribute("DATA_ATTRIBUTE", bytes);
            return (IDirElement)cacheElement.doneUpdate();

        }
        catch (Exception e)
        {
            logMessage("Failed to wrap an object as an element, trace follows...", e, Level.WARNING);
            return null;
        }
    }

    public String exportElementsToXML(IDirElement[] elements)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "exportElementsToXML");
        try
        {
            m_lock.readLock();
            ElementListBuilder builder = new ElementListBuilder(elements);
            builder.setDirectoryService(this);
            builder.init();
            return builder.getXMLString();
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public String exportDSBootFileString(String elementName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "exportDSBootFileString " + elementName);
        try
        {
            m_lock.readLock();
            IDirElement el = getElement(elementName, false);
            if (el == null)
            {
                throw new DirectoryServiceException(elementName + " doesn't exist.");
            }
            // if we're exporting the DS element, as when we write a DS boot file,
            // we must
            // 1) for a backup DS, export the attributes it shares with the primary
            // 2) export the replication connections element for an FT DS (primary or backup)
            // It could be that we don't want to do this everytime the DS element is exported,
            // but only when a boot file is being written. In that case, we'll need to
            // define a different method to write the DS boot file.
            IDirElement[] ftDSElements = checkFTDS(el);
            if (ftDSElements != null)
            {
                for (int i = 0; i < ftDSElements.length; i++)
                {
                    ftDSElements[i] = removeAttributesNotNeededForBoot(ftDSElements[i]);
                }
                return exportElementsToXML(ftDSElements);
            }

            ElementBuilder builder = new ElementBuilder(el, (XMLStringWriter)null);
            builder.setDirectoryService(this);
            builder.init();
            return builder.getXMLString();
        }
        finally
        {
            m_lock.releaseLock();
        }
    }


    private static IDirElement removeAttributesNotNeededForBoot(IDirElement srcElement)
    {
        if (srcElement == null)
        {
            return null;
        }

        try
        {
            IDirElement writableCopy = (IDirElement)srcElement.createWritableClone();
            ((Element)writableCopy).removeSystemAttributes();
            writableCopy.getAttributes().deleteAttribute(IDirectoryServiceConstants.CONFIG_ELEMENT_REFERENCES_ATTR);
            writableCopy.doneUpdate();
            return writableCopy;
        }
        catch (ReadOnlyException e) // Should never happen
        {
            throw new Error(e.toString());
        }
        catch (ConfigException e) // Should never happen
        {
            throw new Error(e.toString());
        }

    }


    @Override
    public String exportElementToXML(String elementName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "exportElementToXML " + elementName);
        try
        {
            m_lock.readLock();
            IDirElement el = getElement(elementName, false);
            if (el == null)
            {
                throw new DirectoryServiceException(elementName + " doesn't exist.");
            }
            ElementBuilder builder = new ElementBuilder(el, (XMLStringWriter)null);
            builder.setDirectoryService(this);
            builder.init();
            return builder.getXMLString();
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    private IDirElement[] checkFTDS(IDirElement el)
    throws DirectoryServiceException
    {
        IDirElement replConnectionsEl = null;
        boolean ftDS = false;
        // check to see if we're exporting the backup DS. The shared attributes must be copied
        // from the primary
        String elType = el.getIdentity().getType();
        if (elType.equals(IBackupDirectoryServiceConstants.DS_TYPE))
        {
            // make the changes in memory and pass the changed element to the XML dumper
            el = getElement(el.getIdentity().getName(), true);
            copySharedAttributes(el);
        }

        String replConnections = getReplicationConnectionsElementID(el);
        if (replConnections != null)
        {
            replConnectionsEl = getElement(replConnections, false);
            if (replConnectionsEl == null)
            {
                throw new DirectoryServiceException("Could not find the replication connections element while exporting the DS element " + el.getIdentity().getName());
            }
            ftDS = true;
        }

        if (ftDS)
        {
            return new IDirElement[]
            { el, replConnectionsEl };
        }
        else
        {
            return null;
        }
    }

    private String getReplicationConnectionsElementID(IDirElement dsElement)
    throws DirectoryServiceException
    {
        IAttributeSet dsAttributes = dsElement.getAttributes();
        IAttributeSet references = (IAttributeSet)dsAttributes.getAttribute(IDirectoryServiceConstants.CONFIG_ELEMENT_REFERENCES_ATTR);
        if (references != null)
        {
            if (dsElement.getIdentity().getType().equals(IBackupDirectoryServiceConstants.DS_TYPE))
            {
                // the replication connections reference for the backup has to be extracted from the
                // primary
                Reference primary = (Reference)references.getAttribute(IBackupDirectoryServiceConstants.PRIMARY_CONFIG_ELEMENT_REF_ATTR);
                if (primary != null)
                {
                    IAttributeSet primaryAttrs = getElement(primary.getElementName(), false).getAttributes();
                    references = (IAttributeSet)primaryAttrs.getAttribute(IDirectoryServiceConstants.CONFIG_ELEMENT_REFERENCES_ATTR);
                }
            }
        }
        if (references != null)
        {
            Reference ref = (Reference)references.getAttribute(IDirectoryServiceConstants.REPLICATION_CONNECTIONS_ELEMENT_REF_ATTR);
            if (ref != null)
            {
                return ref.getElementName();
            }
        }
        return null;
    }

    private void copySharedAttributes(IDirElement backupDS)
    throws DirectoryServiceException
    {
        String backupId = backupDS.getIdentity().getName();
        IAttributeSet backupAttrs = backupDS.getAttributes();
        IAttributeSet primaryAttrs;
        IAttributeSet references = (IAttributeSet)backupAttrs.getAttribute(IBackupDirectoryServiceConstants.CONFIG_ELEMENT_REFERENCES_ATTR);
        if (references == null)
        {
            throw new DirectoryServiceException("Backup DS configuration " + backupId + " does not reference a primary configuration");
        }
        else
        {
            Reference primaryRef = (Reference)references.getAttribute(IBackupDirectoryServiceConstants.PRIMARY_CONFIG_ELEMENT_REF_ATTR);
            if (primaryRef == null)
            {
                throw new DirectoryServiceException("Backup DS configuration " + backupId + " does not reference a primary configuration");
            }
            else
            {
                IElement primaryEl = getElement(primaryRef.getElementName(), false);
                if (primaryEl != null)
                {
                    primaryAttrs = primaryEl.getAttributes();
                    try
                    {
                        IAttributeSet primaryReplAttrs = (IAttributeSet)primaryAttrs.getAttribute(IDirectoryServiceConstants.REPLICATION_PARAMETERS_ATTR);
                        IAttributeSet backupReplAttrs = backupAttrs.createAttributeSet(IDirectoryServiceConstants.REPLICATION_PARAMETERS_ATTR);

                        // Create a temporary empty primaryReplAttrs if non exists in the primary config to simplify the code
                        if (primaryReplAttrs == null)
                        {
                            IElement tmpPrimaryEl = primaryEl.createWritableClone();
                            IAttributeSet tmpPrimaryAtts = tmpPrimaryEl.getAttributes();
                            primaryReplAttrs = tmpPrimaryAtts.createAttributeSet(IDirectoryServiceConstants.REPLICATION_PARAMETERS_ATTR);
                        }

                        com.sonicsw.mf.common.config.MergeUtil.mergeAddAndSet(backupReplAttrs, primaryReplAttrs);
                        backupAttrs.setStringAttribute(IDirectoryServiceConstants.DOMAIN_NAME_ATTR, (String)primaryAttrs.getAttribute(IDirectoryServiceConstants.DOMAIN_NAME_ATTR));
                        IAttributeSet primaryReferences = (IAttributeSet)primaryAttrs.getAttribute(IDirectoryServiceConstants.CONFIG_ELEMENT_REFERENCES_ATTR);
                        if (primaryReferences != null)
                        {
                            Object attrValue = primaryReferences.getAttribute(IDirectoryServiceConstants.REPLICATION_CONNECTIONS_ELEMENT_REF_ATTR);
                            references.setReferenceAttribute(IDirectoryServiceConstants.REPLICATION_CONNECTIONS_ELEMENT_REF_ATTR, (Reference)attrValue);
                        }
                        IAttributeSet fileSystem = backupAttrs.createAttributeSet(IDirectoryServiceConstants.FILE_SYSTEM_STORAGE_ATTR);
                        IAttributeSet primaryFileSystem = (IAttributeSet)primaryAttrs.getAttribute(IDirectoryServiceConstants.FILE_SYSTEM_STORAGE_ATTR);
                        String hostDeprecated = (String)primaryFileSystem.getAttribute(IDirectoryServiceConstants.HOST_DIRECTORY_ATTR);
                        String password = (String)primaryFileSystem.getAttribute(IDirectoryServiceConstants.PASSWORD_ATTR);
                        fileSystem.setStringAttribute(IDirectoryServiceConstants.HOST_DIRECTORY_ATTR, hostDeprecated);
                        fileSystem.setStringAttribute(IDirectoryServiceConstants.PASSWORD_ATTR, password);

                    }
                    catch (Exception e)
                    {
                        e.printStackTrace();
                        throw new DirectoryServiceException("Unable to copy the replication attributes onto the backup DS configuration " + backupId + ": " + e.toString());
                    }
                }
                else
                {
                    throw new DirectoryServiceException("Unable to find primary configuration for " + backupId);
                }
            }
        }
    }

    @Override
    public String exportDirectoryToXML(String dirName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "exportDirectoryToXML " + dirName);
        try
        {
            m_lock.readLock();
            IIdentity dir = getIdentity(dirName);
            if (dir == null)
            {
                throw new DirectoryServiceException(dirName + " doesn't exist.");
            }
            DirectoryBuilder builder = new DirectoryBuilder(dirName);
            builder.setDirectoryService(this);
            builder.init();
            return builder.getXMLString();
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public void dumpContentsToXML()
    {
        trace(TRACE_ALL_DS_ACCESS, "dumpContentsToXML");
        Thread dumper = new Thread("DS dumper")
        {
            @Override
            public void run()
            {
                // work out the next file name to use
                File cwd = new File(".");
                int newVersion = 0;
                final String prefix = "dump." + m_domain + '.';
                final String suffix = ".xml";
                File[] existingDumps = cwd.listFiles(new FilenameFilter()
                {
                    @Override
                    public boolean accept(File dir, String name)
                    {
                        return (name.startsWith(prefix) && name.endsWith(suffix));
                    }
                });
                if (existingDumps.length > 0)
                {
                    int latestVersion = -1;
                    int prefixLength = prefix.length();
                    int suffixLength = suffix.length();
                    for (int i = 0; i < existingDumps.length; i++)
                    {
                        String tmp = existingDumps[i].getName().substring(prefixLength);
                        tmp = tmp.substring(0, tmp.length() - suffixLength);
                        if (tmp.length() < 1)
                        {
                            continue;
                        }
                        try
                        {
                            int version = Integer.parseInt(tmp);
                            if (version > latestVersion)
                            {
                                latestVersion = version;
                            }
                        }
                        catch (NumberFormatException e)
                        {
                            continue;
                        }
                    }
                    if (latestVersion > -1)
                    {
                        newVersion = ++latestVersion;
                    }
                }
                File dumpFile = new File(cwd, prefix + newVersion + suffix);

                try
                {
                    // get the contents from the root
                    String contents = DirectoryService.this.exportDirectoryToXML("/");

                    // now write the new dump file out

                    FileOutputStream fos = new FileOutputStream(dumpFile);
                    fos.write(contents.getBytes());
                    fos.close();
                    DirectoryService.this.logMessage("Dumped DS contents to " + dumpFile.getCanonicalPath(), Level.INFO);
                }
                catch (Exception e)
                {
                    DirectoryService.this.logMessage("Failed to dump DS contents to " + getCanonicalPath(dumpFile) + ", trace follows...", e, Level.WARNING);
                }
            }
        };
        dumper.setDaemon(true);
        dumper.start();
    }

    private Validator createParser(String XMLDocument, boolean validate)
    {
        Validator xmlValidator = new Validator(new StringReader(XMLDocument));
        xmlValidator.createSAXParser();
        xmlValidator.setDirectoryService(this);
        if (validate)
        {
            xmlValidator.setValidation(true);
        }
        return xmlValidator;
    }

    @Override
    public void importFromList(IDirElement[] list, String[] deletionDirectories)
    throws DirectoryServiceException
    {
        importFromList(list, deletionDirectories, false);
    }

    private HashMap importFromList(IDirElement[] list, String[] deletionDirectories, boolean noReplace)
    throws DirectoryServiceException
    {

        try
        {
            m_lock.writeLock();

            if (deletionDirectories == null)
            {
                deletionDirectories = IEmptyArray.EMPTY_STRING_ARRAY;
            }

            m_importManager = new ImportManager(this, deletionDirectories, noReplace);
            m_notificationManager.startThrowAway(); // The import manager will send notifications at the end.

            boolean transactOK = false;
            try
            {
                m_trManager.join();

                // phase1 - see ImportManager
                for (int i = 0; i < list.length; i++)
                {
                    m_importManager.importedElement((Element)list[i]);
                }
                m_importManager.nextPhase();

                // phase2 - see ImportManager
                for (int i = 0; i < list.length; i++)
                {
                    m_importManager.importedElement((Element)list[i]);
                }
                m_importManager.nextPhase();

                // phase3 - see ImportManager
                for (int i = 0; i < list.length; i++)
                {
                    m_importManager.importedElement((Element)list[i]);
                }

                HashMap restList = m_importManager.createAllPackedElements();

                transactOK = true;
                return restList;
            }
            finally
            {
                m_notificationManager.stopThrowAway();
                m_trManager.leave(transactOK, m_importManager, true);
            }

        }
        catch (VersionOutofSyncException e) // should never happen
        {
            logMessage("Failed import, trace follows...", e, Level.WARNING);
            Error error = new Error();
            error.initCause(e);
            
            throw e;
        }
        finally
        {
            m_notificationManager.stopThrowAway();
            m_importManager = null;
            m_lock.releaseLock();
        }

    }

    @Override
    public void importFromXML(String XMLDocument)
    throws DirectoryServiceException, InvalidXMLException
    {

        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "importFromXML");
        try
        {
            m_lock.writeLock();
            m_importManager = new ImportManager(this);
            m_notificationManager.startThrowAway(); // The import manager will send notifications at the end.

            if (XMLDocument.length() == 0 || XMLDocument == null)
            {
                throw new InvalidXMLException("XML file can't be empty");
            }

            boolean transactOK = false;
            try
            {
                m_trManager.join();

                // phase1 - see ImportManager
                Validator xmlValidator = createParser(XMLDocument, true);
                xmlValidator.setContentHandler(getDomain());
                xmlValidator.parseData();
                m_importManager.nextPhase();

                // phase2 - see ImportManager
                xmlValidator = createParser(XMLDocument, false); // we already validated - create a non validating
                // parser
                xmlValidator.setContentHandler(getDomain());
                xmlValidator.parseData();
                m_importManager.nextPhase();

                // phase3 - see ImportManager
                xmlValidator.resetXMLData(new StringReader(XMLDocument));
                xmlValidator.parseData();

                m_importManager.createAllPackedElements();

                transactOK = true;
            }
            finally
            {
                m_notificationManager.stopThrowAway();
                m_trManager.leave(transactOK, m_importManager, true);
            }

        }
        catch (VersionOutofSyncException e) // should never happen
        {
            logMessage("Failed to import from XML, trace follows...", e, Level.WARNING);
        }
        finally
        {
            m_notificationManager.stopThrowAway();
            if (m_importManager.wasLogicalNameSpaceModified())
            {
                m_logicalNameSpace.reset();
            }

            m_importManager = null;
            m_lock.releaseLock();
        }
    }

    // This method should be used only in DS 'standalone' mode - from the DirectorySeeder, for example
    // MRD 12/07/2004 modified to deal with large files
    @Override
    public void attachBlob(IBasicElement element, java.io.InputStream stream, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        if (stream == null)
        {
            throw new DirectoryServiceException("The blob stream object cannot be null.");
        }
        try
        {
            BufferedInputStream bStream = new BufferedInputStream(stream, IBlob.BLOB_CHUNK_SIZE);
            ByteArrayOutputStream oStream;
            int src = 0;

            byte[] blobPiece;
            int readIn = 0;
            while (readIn != -1)
            {
                int chunkIndex = 0;
                oStream = new ByteArrayOutputStream();
                while ((chunkIndex < IBlob.BLOB_CHUNK_SIZE) && (readIn != -1))
                {
                    readIn = bStream.read();
                    if (readIn != -1)
                    {
                        oStream.write(readIn);
                        chunkIndex = chunkIndex + 1;
                    }
                }
                oStream.close();
                blobPiece = oStream.toByteArray();
                appendBlob(element, blobPiece, src, (readIn == -1), view);
                src = src + blobPiece.length;
            }
        }
        catch (Exception e) // Could happen only because lack of memory
        {
            throw new DirectoryServiceException(e.toString());
        }
    }

    @Override
    public IBlob getBlob(String elementName, boolean forUpdate, int offset)
    throws DirectoryServiceException
    {
        return getBlob(elementName, forUpdate, offset, null);
    }

    // mrd 12/07/2004 New to deal with large file attachments in pieces
    private IBlob getBlob(String elementName, boolean forUpdate, int offset, String logicalPath)
    throws DirectoryServiceException
    {
        validateOpen();
        EntityName eName = validateName(elementName);

        try
        {
            m_lock.readLock();
            IDirElement element = getElement(elementName, forUpdate);
            if (Blob.isIncompleteBlob(element))
            {
                throw new DirectoryServiceException("File " + (logicalPath != null ? logicalPath : elementName) + " is incomplete - it must be restored.");
            }
            byte[] fileByteArray = null;
            Integer blobState;
            long blobSize = m_blobStorage.getBlobSize(eName);
            if (blobSize != 0) // the blobFile will be null if the blob is partially saved or no blob has been attached
            {
                if ((offset + IBlob.BLOB_CHUNK_SIZE) >= blobSize)
                {
                    blobState = IBlob.END;
                }
                else
                {
                    blobState = IBlob.PARTIAL;
                }
                fileByteArray = m_blobStorage.getBlob(eName, offset, IBlob.BLOB_CHUNK_SIZE);
                Blob.markBlobState((Element)element, blobState, IBlob.BLOB_TRANSFER_STATE);
                ((Element)element).setReadOnly(!forUpdate);
                // the BLOB_TRANSFER_STATE is not persisted? Shoult it be?
            }
            return new Blob(element, fileByteArray, this);
        }
        catch (Exception e)
        {
            if (e instanceof DirectoryServiceException)
            {
                throw (DirectoryServiceException)e;
            }
            else
            {
                DirectoryServiceException dirE = new DirectoryServiceException("Unable to get blob " + elementName);
                dirE.initCause(e);
                throw dirE;
            }
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    // implemented from IDirectoryMFService, internal MF method
    @Override
    public IBlob getEntireBlob(String elementName, boolean forUpdate)
    throws DirectoryServiceException
    {
        validateOpen();
        EntityName eName = validateName(elementName);

        try
        {
            m_lock.readLock();
            IDirElement element = getElement(elementName, forUpdate);
            if (Blob.isIncompleteBlob(element))
            {
                throw new DirectoryServiceException("File " + elementName + " is incomplete - it must be restored.");
            }
            byte[] fileByteArray = null;
            Integer blobState = IBlob.END;
            long blobSize = m_blobStorage.getBlobSize(eName);
            if (blobSize != 0) // the blobFile will be null if the blob is partially saved or no blob has been attached
            {
            	// get the entire blob in the byte array
                fileByteArray = m_blobStorage.getBlob(eName, 0, (int)blobSize);
                Blob.markBlobState((Element)element, blobState, IBlob.BLOB_TRANSFER_STATE);
                ((Element)element).setReadOnly(!forUpdate);
                // the BLOB_TRANSFER_STATE is not persisted? Shoult it be?
            }
            return new Blob(element, fileByteArray, this);
        }
        catch (Exception e)
        {
            if (e instanceof DirectoryServiceException)
            {
                throw (DirectoryServiceException)e;
            }
            else
            {
                throw new DirectoryServiceException(e.toString());
            }
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    // mrd 12/07/2004 modified to deal with large files. This method returns the first IBlob.BLOB_CHUNK_SIZE
    // piece of blob or the entire blob, if it's size is < IBLob.BLOB_CHUNK_SIZE
    @Override
    public IBlob getBlob(String elementName, boolean forUpdate)
    throws DirectoryServiceException
    {
        return getBlob(elementName, forUpdate, 0);
    }

    // mrd 12/07/2004 modified to deal with large files. If the blob has not been completely saved to disk, this
    // method returns null
    private File blobToFile(String blobName)
    throws DirectoryServiceException
    {
        validateOpen();
        EntityName eName = validateName(blobName);

        try
        {
            m_lock.readLock();
            IDirElement element = getElement(blobName, false);
            if (Blob.isIncompleteBlob(element))
            {
                throw new DirectoryServiceException("File " + element.getIdentity().getName() + " + is incomplete - it must be restored.");
            }
            return copyBlob(eName.getName());
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    // The copies are removed, the next time the DS is started
    // mrd 12/07/2004 not modified to deal with large files because it already deals with it in chunks of 4096 bytes!!
    private File copyBlob(String blobName)
    throws DirectoryServiceException
    {
        try
        {

            IBlob blob = getBlob(blobName, false);
            InputStream is = blob.getBlobStream();
            // Create a copy with a unique name
            File copyFile = new File(m_blobCopiesDir, BLOB_COPIES_PREFIX + new Long(System.currentTimeMillis()).toString() + "_" + m_copySequenceNum++);
            FileOutputStream fos = new FileOutputStream(copyFile);
            byte[] bytes = new byte[4096]; // 4096 is an arbitrary reasonable buffer size
            while (true)
            {
                int bytesRead = is.read(bytes);
                if (bytesRead < 1)
                {
                    break;
                }
                fos.write(bytes, 0, bytesRead);
            }
            is.close();
            fos.close();
            return copyFile;
        }
        catch (IOException e)
        {
            throw new DirectoryServiceException("Failed to create copy for: " + blobName + " :" + e.toString());
        }
    }

    @Override
    public void detachBlob(IDeltaDirElement delta, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {

        validateOpen();
        EntityName eName = validateName(delta.getIdentity().getName());

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            setElement(delta, view);

            m_blobStorage.deleteBlob(eName);
            transactOK = true;

        }
        catch (StorageException e)
        {
            throw convertException(e);
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }

    }

    // This is implemented as a "shortcut" - it is more efficient than attachBlob(IBasicElement, InputStream, IDeltaView
    // )
    // but there is a size limit (determined by the amount of memory the JVM has) - we will stop using it if we'll need
    // to support blobs larger than 10M and have communication support for that.
    // mrd 12/07/2004 not modified to deal with large blobs since it's already passed in a blob by the caller
    // This method is declared public in the interface but described as being "for internal use" only. When we decide to
    // completely
    // remove it, we need to remove its use in vobs_qa_mgmt, in directory.BlobTest and cache.CacheTest,
    // configuration.CachingTests
    // and directory.FileSystemProxyTest
    @Override
    public void attachBlob(IBasicElement element, byte[] blob, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        if (blob == null)
        {
            throw new DirectoryServiceException("The blob object cannot be null.");
        }

        validateOpen();
        EntityName eName = validateName(element.getIdentity().getName());

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();

            m_trManager.join();
            joinedTransaction = true;

            m_blobStorage.setBlob(eName, blob);
            boolean expandable = isExpandableBlob(eName);
            // if this is a new element and it is an expandable jar file, mark the element
            // before storing it.
            if (element instanceof IElement)
            {
                Blob.markBlobState((Element)element, new Boolean(expandable), IBlob.EXPAND_IN_CACHE);
            }
            setElement(element, view);
            // If element is a DeltaElement, it doesn't contain the system attributes
            // If it is expandable, fetch it from storage and mark it
            if (expandable && element instanceof IDeltaElement)
            {
                IDirElement storedElement = m_storage.getElement(eName);
                Blob.markBlobState((Element)storedElement, IBlob.EXPANDABLE, IBlob.EXPAND_IN_CACHE);
                storeInternal(eName, storedElement);
            }

            transactOK = true;

        }
        catch (StorageException e)
        {
            throw convertException(e);
        }
        catch (DirectoryServiceException dirE)
        {
            throw dirE;
        }
        catch (Exception ex)
        {
            throw new DirectoryServiceException(ex.toString());
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    // mrd 12/07/2004 New to deal with large file attachments in pieces.
    @Override
    public void appendBlob(IBasicElement element, byte[] blob, int from, boolean last, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        if (blob == null)
        {
            throw new DirectoryServiceException("The blob object cannot be null.");
        }

        validateOpen();

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            EntityName eName = validateName(element.getIdentity().getName());
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;
            // For an existing element, mark the element as INCOMPLETE until the last piece of the blob
            // comes in.
            if (element instanceof IDeltaElement)
            {
                IDirElement storedElement = m_storage.getElement(eName);
                Blob.markBlobState((Element)storedElement, last ? IBlob.COMPLETE : IBlob.INCOMPLETE, IBlob.LARGE_FILE_STATE);
                storeInternal(eName, storedElement);
            }
            m_blobStorage.appendBlob(eName, blob, from);
            // for a new element, mark it complete on the last blob append
            if ((last) && (element instanceof IElement))
            {
                Blob.markBlobState((Element)element, IBlob.COMPLETE, IBlob.LARGE_FILE_STATE);
            }
            if (last)
            {
                // if this blob is a jar file with a manifest, test to see if it should be expanded when cached and
                // mark the envelope element as expandable
                boolean expandable = isExpandableBlob(eName);
                if (element instanceof IDeltaElement)
                {
                    IDirElement storedElement = m_storage.getElement(eName);
                    Blob.markBlobState((Element)storedElement, new Boolean(expandable), IBlob.EXPAND_IN_CACHE);
                }
                else
                {
                    Blob.markBlobState((Element)element, new Boolean(expandable), IBlob.EXPAND_IN_CACHE);
                }
                setElement(element, view);
            }
            transactOK = true;

        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString());
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }

    }

    private IDirElement subclassOnDSSide(Element newInstance0, IDeltaView view)
    throws DirectoryServiceException
    {
        String newInstanceName = newInstance0.getIdentity().getName();
        String superName = newInstance0.getSuperToSubclassFrom();
        Element superElement = (Element)getElementAsIs(superName, true);

        if (superElement == null)
        {
            throw new DirectoryServiceException("Tried to subclass " + newInstanceName + " from non existent template " + superName);
        }
        if (!superElement.isTemplate())
        {
            throw new DirectoryServiceException("Tried to subclass " + newInstanceName + " from a non template element " + superName);
        }

        Element newInstance = (Element)newInstance0.createWritableClone();
        try
        {
            newInstance.removeSystemAttributes();
            superElement.removeSystemAttributes();
        }
        catch (Exception e)
        {
            throw new Error(e.toString()); // Should never happen
        }

        newInstance.addSuperAttribute(superName, false);

        DeltaElement delta = superElement.createDelta(newInstance);
        ((ElementIdentity)delta.getIdentity()).setVersion(superElement.getIdentity().getVersion());

        return subclassElement(delta, newInstanceName, view);

    }

    @Override
    public IDirElement subclassElement(IBasicElement delta, String newName, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return subclassElement(delta, newName, view, true);
    }

    @Override
    public void registerElementChangeHandler(IElementChangeHandler handler)
    {
        if (!m_open)
        {
            throw new RuntimeException("The DS is closed.");
        }

        createLocalListener();

        m_localListener.registerElementChangeHandler(handler);
    }

    @Override
    public void registerElementChangeHandler(String elementName, IElementChangeHandler handler)
    {
        if (!m_open)
        {
            throw new RuntimeException("The DS is closed.");
        }

        createLocalListener();

        m_localListener.registerElementChangeHandler(elementName, handler);
    }

    @Override
    public void unregisterElementChangeHandler(IElementChangeHandler handler)
    {
        if (m_localListener == null)
        {
            return;
        }

        m_localListener.unregisterElementChangeHandler(handler);
    }

    @Override
    public void unregisterElementChangeHandler(String elementName, IElementChangeHandler handler)
    {
        if (m_localListener == null)
        {
            return;
        }

        m_localListener.unregisterElementChangeHandler(elementName, handler);
    }

    @Override
    public void registerNameChangeHandler(INameChangeHandler handler)
    {
        if (!m_open)
        {
            throw new RuntimeException("The DS is closed.");
        }

        createLocalFSListener();

        ((LocalFSListener)m_localListener).registerNamingHandler(handler);
    }

    @Override
    public void unregisterNameChangeHandler(INameChangeHandler handler)
    {
        if (m_localListener == null)
        {
            return;
        }

        ((LocalFSListener)m_localListener).unregisterNamingHandler(handler);
    }

    @Override
    public void registerFSElementChangeHandler(IElementChangeHandler handler)
    {
        if (!m_open)
        {
            throw new RuntimeException("The DS is closed.");
        }

        createLocalFSListener();

        m_localListener.registerElementChangeHandler(handler);
    }

    @Override
    public void registerFSElementChangeHandler(String elementName, IElementChangeHandler handler)
    {
        if (!m_open)
        {
            throw new RuntimeException("The DS is closed.");
        }

        createLocalFSListener();

        m_localListener.registerElementChangeHandler(elementName, handler);
    }

    @Override
    public void unregisterFSElementChangeHandler(IElementChangeHandler handler)
    {
        if (m_localListener == null)
        {
            return;
        }

        m_localListener.unregisterElementChangeHandler(handler);
    }

    @Override
    public void unregisterFSElementChangeHandler(String elementName, IElementChangeHandler handler)
    {
        if (m_localListener == null)
        {
            return;
        }

        m_localListener.unregisterElementChangeHandler(elementName, handler);
    }

    private void createLocalListener()
    {
        if (m_localListener != null)
        {
            return;
        }

        if (hasConsumer())
        {
            throw new RuntimeException("Cannot have more than a single listener.");
        }

        m_localListener = new LocalListener();
        subscribeAll(m_localListener);
    }

    private void createLocalFSListener()
    {
        if (m_localListener != null)
        {
            return;
        }

        if (hasConsumer())
        {
            throw new RuntimeException("Cannot have more than a single listener.");
        }

        m_localListener = new LocalFSListener();
        subscribeAll(m_localListener);
        subscribeNaming((LocalFSListener)m_localListener);
    }

    private IDirElement subclassElement(IBasicElement delta, String newName0, IDeltaView view, boolean notifyModifications)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        validateOpen();
        if (delta == null || newName0 == null)
        {
            throw new DirectoryServiceException("A super element delta object and the name for the subclassed element must be specified.");
        }

        EntityName newNameE = validateName(newName0);

        if (m_authentication != null)
        {
            m_authentication.okToModify(newNameE);
        }

        String newName = newNameE.getName();

        if (!(delta instanceof IDeltaElement))
        {
            throw new DirectoryServiceException("The delta parameter must be the modified result of IDirElement.doneUpdateForSubclassing().");
        }

        EntityName deltaNameE = validateName(delta.getIdentity().getName());
        String deltaName = deltaNameE.getName();

        if (deltaName.equals(VIEW_ELEMENT))
        {
            throw new DirectoryServiceException("The view cannot be sub-classed.");
        }

        // Super and subclassed must be in the same directory
        if (!deltaNameE.getParent().equals(newNameE.getParent()))
        {
            throw new DirectoryServiceException("The super element and the sub-classed element must reside in the same directory.");
        }

        if (PackedDirUtil.underPackedDir(deltaNameE))
        {
            throw new DirectoryServiceException("Packed element \"" + deltaName + "\" cannot be sub-classed.");
        }

        try
        {
            m_lock.writeLock();

            IIdentity newID = m_idCache.get(newNameE);
            if (newID != null)
            {
                throw new DirectoryServiceException("\"" + newName + "\" already exists.");
            }

            Element superElement = (Element)getElementAsIs(deltaName, true);
            if (superElement == null)
            {
                throw new DirectoryServiceException("\"" + deltaName + "\" was deleted.");
            }
            if (!superElement.isTemplate())
            {
                throw new DirectoryServiceException("\"" + deltaName + "\" is not a template - cannot be sub-classed.");
            }

            Element superElementOld = (Element)superElement.createClone();
            superElementOld = cleanupSuperElement(superElementOld, false, false);

            if (superElement.isSubclassedElement())
            {
                throw new DirectoryServiceException("\"" + deltaName + "\" is sub-classed from \"" + superElement.getSuperElementName() + "\". It cannot be sub-classed again.");
            }

            // Realize the new element and make sure returned from a doneUpdateForSubclassing() call.
            Element realizedSub = doRealizeSub(superElement, (DeltaElement)delta, newName);
            if (realizedSub.getSuperElementName() == null)
            {
                throw new DirectoryServiceException("The delta must be returned from a doneUpdateForSubclassing() call.");
            }

            // Serialize the delta and store in the super element
            byte[] serializedDelta = ((DeltaElement)delta).toBytes();

            // Create the delta of before and after the superElement.addSubclassedElement() call
            DeltaElement superDelta = Element.addSubToSuperDelta(newName, superElement);

            superElement.addSubclassedElement(newName, serializedDelta);

            boolean transactOK = false;
            BeforeAfterPair pair = null;
            try
            {
                m_trManager.join();
                setElementAsIs((IDeltaElement)superElement.doneUpdate(), null, false);

                Element newSubClassed = new Element(realizedSub.getIdentity());
                newSubClassed.addSuperAttribute(deltaName);
                storeInternal(newNameE, (IDirElement)newSubClassed.doneUpdate());

                if (view != null)
                {
                    pair = prepareToSet(((DeltaView)view).getElement(), null);
                    storeInternal(VIEW_ELEMENT_ENTITY_NAME, pair.m_after);
                }
                transactOK = true;
            }
            finally
            {
                if (notifyModifications)
                {
                    notifyMods(new ModificationItem(realizedSub), createViewMod(view, pair));
                }

                // Use a clone of superElement for before-image
                notifyMods(new ModificationItem(superDelta, superElementOld), null);

                m_trManager.leave(transactOK);
            }

            IDirElement newElement = (IDirElement)realizedSub.createWritableClone();
            ((Element)newElement).setSubclassingDelta(serializedDelta);
            return newElement;

        }
        catch (ReadOnlyException e) // Should never happen
        {
            throw new Error(e.toString());
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    @Override
    public void unSubclassElement(String elementName, IDeltaView view)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        EntityName nameE = validateName(elementName);
        try
        {
            m_lock.writeLock();

            // Replaces the reference-to-super with a realized one
            Element realizedElement = (Element)getElement(elementName, true);
            String superName = realizedElement.getSuperElementName();

            // Not a subclassed element
            if (superName == null)
            {
                throw new DirectoryServiceException("Element \"" + elementName + "\" is not sub-classed.");
            }

            BeforeAfterPair pair = null;
            ModificationItem modItem = null;
            boolean transactOK = false;
            try
            {
                m_trManager.join();
                modItem = removeSubclassedFromSuper(superName, nameE.getName());

                realizedElement.removeSuperElementName();
                storeInternal(nameE, realizedElement);

                if (view != null)
                {
                    pair = prepareToSet(((DeltaView)view).getElement(), null);
                    storeInternal(VIEW_ELEMENT_ENTITY_NAME, pair.m_after);
                }
                transactOK = true;
            }
            finally
            {
                notifyMods(modItem, createViewMod(view, pair));
                m_trManager.leave(transactOK);
            }

        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    // Removes the subclassed delta from the super element and return the notification delta for that operation.
    private ModificationItem removeSubclassedFromSuper(String superName, String subclassedName)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        try
        {
            Element superElement = (Element)getElementAsIs(superName, true);

            // Sanity check
            if (superElement == null)
            {
                throw new DirectoryServiceException("Super element \"" + superName + "\" is missing.");
            }

            // Sanity check
            if (superElement.getSubclassedDelta(subclassedName) == null)
            {
                throw new DirectoryServiceException("Element \"" + subclassedName + "\" definition is missing at \"" + superName + "\".");
            }

            Element superElementOld = (Element)superElement.createClone();
            superElementOld = cleanupSuperElement(superElementOld, false, false);

            String[] oldSubList = superElement.getSubclassedList();

            DeltaElement superDelta = Element.removeSubToSuperDelta(oldSubList, subclassedName, superElement.getIdentity());

            superElement.removeSubclassedDelta(subclassedName);

            setElementAsIs((IDeltaElement)superElement.doneUpdate(), null, false);

            return new ModificationItem(superDelta, superElementOld);
        }
        catch (ReadOnlyException e) // Should never happen
        {
            throw new Error(e.toString());
        }
    }

    @Override
    public void importedElement(IBasicElement element)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        m_importManager.importedElement((Element)element);
    }

    @Override
    public long getBackupTimestamp()
    {
        return m_backupVersion;
    }

    Class loadClass(String className, String classpath)
    throws Exception
    {
        if (m_loader != null)
        {
            return m_loader.loadClass(className, expandClasspath(classpath));
        }
        else
        {
            return loadClassInternal(className, classpath);
        }
    }

    private Class loadClassInternal(String className, String classpath)
    throws Exception
    {
        classpath = expandClasspath(classpath);
        URLClassLoader urlLoader = new URLClassLoader(getURLList(classpath));
        return Class.forName(className, false, urlLoader);
    }

    /**
     * Expands a classpath that may include specifiers to JAR files that reside in the DS.
     *
     * For each JAR file that resides in the DS it substitutes the local version.
     */
    private String expandClasspath(String origClasspath)
    throws Exception
    {
        if (origClasspath == null)
        {
            return null;
        }

        StringBuffer newClasspath = new StringBuffer();

        // else we need to break the classpath into tokens and examine each token
        StringTokenizer st = new StringTokenizer(origClasspath, IContainer.DS_CLASSPATH_DELIMITER);
        while (st.hasMoreTokens())
        {
            String classpathToken = st.nextToken();
            // is this a path to a JAR stored in the DS ?
            if (classpathToken.length() > IContainer.DS_CLASSPATH_PROTOCOL.length() && classpathToken.substring(0, IContainer.DS_CLASSPATH_PROTOCOL.length()).equalsIgnoreCase(IContainer.DS_CLASSPATH_PROTOCOL))
            {
                String logicalArchiveName = classpathToken.substring(IContainer.DS_CLASSPATH_PROTOCOL.length());
                String archive = null;
                File archiveFile = null;
                try
                {
                    archive = logicalToStorage(IMFDirectories.MF_DIR_SEPARATOR + logicalArchiveName);
                    archiveFile = blobToFile(archive);
                }
                catch (Exception e)
                {
                    // Do nothing - we'll just ignore this part of the classpath
                }
                if (archiveFile == null)
                {
                    continue;
                }
                classpathToken = archiveFile.toURL().toString();
            }
            if (newClasspath.length() > 0)
            {
                newClasspath.append(IContainer.DS_CLASSPATH_DELIMITER);
            }
            newClasspath.append(classpathToken);
        }

        return newClasspath.toString();
    }

    static String listToClasspath(IAttributeList pluginArchiveList)
    {
        String classpath = "";
        if (pluginArchiveList != null)
        {
            for (int j = 0; j < pluginArchiveList.getCount(); j++)
            {
                classpath += (String)pluginArchiveList.getItem(j);
                if (j + 1 < pluginArchiveList.getCount())
                {
                    classpath += IContainer.DS_CLASSPATH_DELIMITER;
                }
            }
        }
        return classpath;
    }

    private static URL[] getURLList(String path)
    throws Exception
    {
        if (path == null || path.length() == 0)
        {
            return new URL[0];
        }

        StringTokenizer st = new StringTokenizer(path, IContainer.DS_CLASSPATH_DELIMITER);
        URL[] urls = new URL[st.countTokens()];

        for (int i = 0; i < urls.length; i++)
        {
            String token = st.nextToken();
            File file = new File(token);
            urls[i] = file.exists() ? file.toURL() : new URL(token);
        }

        return urls;
    }

    private class BeforeAfterPair
    {
        IDirElement m_before;

        IDirElement m_after;

        BeforeAfterPair(IDirElement before, IDirElement after)
        {
            m_before = before;
            m_after = after;
        }
    }

    @Override
    public void newAuthenticationDescriptor(IElement element)
    {
        if (m_authentication != null)
        {
            m_authentication.modifyDomainConnectionParameters(element);
        }
    }

    @Override
    public String[] getExternalDomainsDescriptors()
    {
        if (m_authentication == null)
        {
            return IEmptyArray.EMPTY_STRING_ARRAY;
        }
        else
        {
            return m_authentication.getExternalDomainsDescriptors();
        }

    }

    // Provides for grouping of ranges of the list in addtional to regular ArrayList functionality
    private final class GroupArrayList
    extends ArrayList
    {
        ArrayList m_ranges;

        GroupArrayList()
        {
            super();
            m_ranges = new ArrayList();
        }

        void addSingletonRanges()
        {
            int firstIndex = 0;
            ArrayList allRanges = new ArrayList();
            for (int i = 0; i < m_ranges.size(); i++)
            {
                Range groupRange = (Range)m_ranges.get(i);
                if (groupRange.m_start > firstIndex)
                {
                    allRanges.add(new Range(firstIndex, groupRange.m_start - firstIndex, false));
                    allRanges.add(groupRange);
                    firstIndex = groupRange.m_start + groupRange.m_size;
                }
                else
                {
                    allRanges.add(groupRange);
                    firstIndex = groupRange.m_start + groupRange.m_size;
                }
            }

            if (firstIndex < size())
            {
                allRanges.add(new Range(firstIndex, size() - firstIndex, false));
            }

            m_ranges = allRanges;

        }

        void addGroup(ArrayList group)
        {
            if (group == null || group.isEmpty())
            {
                return;
            }

            m_ranges.add(new Range(size(), group.size(), true));

            for (int i = 0; i < group.size(); i++)
            {
                add(group.get(i));
            }
        }

        int rangesCount()
        {
            return m_ranges.size();
        }

        int rangeSize(int rangeNum)
        {
            return ((Range)m_ranges.get(rangeNum)).m_size;
        }

        boolean rangeIsGroup(int rangeNum)
        {
            return ((Range)m_ranges.get(rangeNum)).m_group;
        }

        Iterator rangeIterator(int rangeNum)
        {
            Range groupRange = (Range)m_ranges.get(rangeNum);
            return new RangeIterator(groupRange.m_start, groupRange.m_size);
        }

        private class RangeIterator
        implements Iterator
        {
            int m_currentIndex;

            int m_last;

            RangeIterator(int first, int size)
            {
                m_currentIndex = first;
                m_last = first + size - 1;
            }

            @Override
            public boolean hasNext()
            {
                return m_currentIndex <= m_last;
            }

            @Override
            public Object next()
            {
                if (!hasNext())
                {
                    throw new IllegalStateException();
                }

                return get(m_currentIndex++);
            }

            @Override
            public void remove()
            {
                throw new IllegalStateException("remove() is not supported");
            }

        }

        private final class Range
        {
            int m_start;

            int m_size;

            boolean m_group;

            Range(int start, int size, boolean group)
            {
                m_start = start;
                m_size = size;
                m_group = group;
            }
        }
    }

    boolean rememberNewIds()
    {
        if (m_notificationManager != null)
        {
            return m_notificationManager.rememberNewIds();
        }
        else
        {
            return false;
        }
    }

    @Override
    public boolean isFSInterfaceInUse()
    {
        return m_FSInterfaceIsUsed;
    }

    @Override
    public IBasicElement[] translateElementsToLogical(IBasicElement[] elements)
    {
        ArrayList translatedList = translateElementsToLogicalInternal(elements);
        IBasicElement[] result = new IBasicElement[translatedList.size()];
        translatedList.toArray(result);
        return result;
    }

    private IDirElement[] translateElementsToLogical(IDirElement[] elements)
    {
        ArrayList translatedList = translateElementsToLogicalInternal(elements);
        IDirElement[] result = new IDirElement[translatedList.size()];
        translatedList.toArray(result);
        return result;
    }

    private ArrayList translateElementsToLogicalInternal(IBasicElement[] elements)
    {

        IReplaceRef nameReplacer = m_logicalNameSpace.createNameReplacer(false);
        ArrayList result = new ArrayList();
        for (int i = 0; i < elements.length; i++)
        {
            // Logical deletion events are published through naming events (at this point we don't know
            // what was the logical name of the deleted element (see spec)
            if (elements[i] instanceof IDirElement && ((IDirElement)elements[i]).isDeleted())
            {
                continue;
            }

            // Elements are enveloped only from at the ImportManager - we currently don't create logical
            // notifications for ImportManager modifications (see spec)
            if (elements[i] instanceof IEnvelope)
            {
                continue;
            }

            IBasicElement clone = null;
            if (elements[i] instanceof Element)
            {
                clone = ((Element)elements[i]).createClone();
            }
            else if (elements[i] instanceof DeltaElement)
            {
                clone = ((DeltaElement)elements[i]).createClone();
            }

            if (clone != null) {
	            String storageName = clone.getIdentity().getName();
	
	            ((ICanReplaceRef)clone).replaceReferences(false, nameReplacer);
	            String logicalName = null;
	            try
	            {
	                logicalName = nameReplacer.replace(storageName);
	            }
	            // We don't want to report about changes of internal MF elements that don't have logical names
	            catch (Exception e)
	            {
	                continue;
	            }
	            setLogicalName(clone, logicalName);
	            result.add(clone);
            }
        }
        return result;
    }

    private final class NotificationManager
    implements IModificationManager
    {
        private GroupArrayList m_notificationList;

        private boolean m_throwAway;

        private NotificationConsumer m_consumer;

        private DirectoryService m_ds;

        private HashMap m_newIdsTable;

        private boolean m_inDeleteTriggers;

        private boolean m_hasDelete;
        
        private boolean m_hasUpdate;
        
        private boolean m_hasAdded;

        NotificationManager(DirectoryService ds)
        {
            m_ds = ds;
            m_notificationList = new GroupArrayList();
            m_throwAway = false;
            m_consumer = null;
            m_newIdsTable = null;
            m_inDeleteTriggers = false;
            m_hasDelete = false;
            m_hasUpdate = false;
            m_hasAdded = false;
        }

        boolean rememberNewIds()
        {
            if (m_newIdsTable != null)
            {
                return false;
            }

            m_newIdsTable = new HashMap();
            return true;
        }

        HashMap getAndResetNewIds()
        {
            HashMap result = m_newIdsTable;
            m_newIdsTable = null;
            return result;
        }

        void setConsumer(NotificationConsumer consumer)
        {
            m_consumer = consumer;
        }

        boolean hasConsumer()
        {
            return m_consumer != null;
        }

        void startThrowAway()
        {
            m_throwAway = true;
        }

        void stopThrowAway()
        {
            m_throwAway = false;
        }

        boolean inThrowAway()
        {
            return m_throwAway;
        }

        void addNotifications(HashMap modifications)
        {
            ModificationItem[] modList = new ModificationItem[modifications.size()];
            Iterator iterator = modifications.values().iterator();
            int i = 0;
            while (iterator.hasNext())
            {
                modList[i++] = (ModificationItem)iterator.next();
            }
            addNotifications(modList);
        }

        void addNotifications(ModificationItem[] modifications)
        {
            if (m_throwAway)
            {
                return;
            }
            for (int i = 0; i < modifications.length; i++)
            {
                if (modifications[i].getChangeType() == IElementChange.ELEMENT_DELETED)
                {
                    m_hasDelete = true;
                }
                if (modifications[i].getChangeType() == IElementChange.ELEMENT_ADDED)
                {
                    m_hasAdded = true;
                }
                if (modifications[i].getChangeType() == IElementChange.ELEMENT_UPDATED)
                {
                    m_hasUpdate = true;
                }
                m_notificationList.add(modifications[i]);
                if (m_newIdsTable != null)
                {
                    IElementIdentity id = modifications[i].getModificationID();
                    IBasicElement modification = modifications[i].getModification();

                    // We send only new ids not ids of deleted elements
                    if (modification != null && (modification instanceof IDirElement) && ((IDirElement)modification).isDeleted())
                    {
                        continue;
                    }

                    // The client should not get next version tokens for updates the occur in triggers since only
                    // if the client updated the element locally it has enough information to increment to the next
                    // version
                    if (!m_inDeleteTriggers && id != null)
                    {
                        m_newIdsTable.put(id.getName(), id);
                    }
                }
            }
        }

        // The modifications list SHOULD NOT CONTAIN deltas - only Elements (deleted Elements and Envelope Elements is
        // ok).
        void addGroupNotification(ArrayList modifications, boolean hasDelete)
        {
            if (m_throwAway)
            {
                return;
            }

            if (hasDelete)
            {
                m_hasDelete = true;
            }

            m_notificationList.addGroup(modifications);

        }

        @Override
        public void onDelete()
        throws DirectoryServiceException
        {
            // For performance - don't do anything if there is not delete
        	// don't call triggers while the DM upgrade is happening
            if (!m_hasDelete || System.getProperty("MQ_UPGRADE", "false").equals("true"))
            {
                return;
            }

            try
            {
                m_inDeleteTriggers = true;
                m_ds.runOnDeleteTriggers(m_notificationList);
            }
            finally
            {
                m_inDeleteTriggers = false;
            }
        }

        @Override
        public void afterDelete()
        throws DirectoryServiceException
        {
            // For performance - don't do anything if there is not delete
        	// don't call the triggers if the DM upgrade is happening
            if (!m_hasDelete || System.getProperty("MQ_UPGRADE", "false").equals("true"))
            {
                return;
            }

            try
            {
                m_inDeleteTriggers = true;
                m_ds.runAfterDeleteTriggers(m_notificationList);
            }
            finally
            {
                m_inDeleteTriggers = false;
            }
        }

        @Override
        public void validate()
        throws DirectoryServiceException
        {
            m_ds.runValidationTriggers(m_notificationList, false);
        }
        
        @Override
        public void onUpdate()
        throws DirectoryServiceException
        {
            // For performance - don't do anything if there is no update
        	// don't call the triggers while the DM upgrade is happening
            if (!m_hasUpdate || System.getProperty("MQ_UPGRADE", "false").equals("true"))
            {
                return;
            }

            try
            {
                m_inDeleteTriggers = true;  // using m_inDeleteTriggers to mean m_inTrigger, really
                m_ds.runOnUpdateTriggers(m_notificationList);
            }
            finally
            {
                m_inDeleteTriggers = false;
            }
        }
        
        @Override
        public void afterUpdate()
        throws DirectoryServiceException
        {
            // For performance - don't do anything if there is no update
        	// Don't call the triggers if the DM upgrade is happening
            if (!m_hasUpdate || System.getProperty("MQ_UPGRADE", "false").equals("true"))
            {
                return;
            }

            try
            {
                m_inDeleteTriggers = true;  // using m_inDeleteTriggers to mean m_inTrigger, really
                m_ds.runAfterUpdateTriggers(m_notificationList);
            }
            finally
            {
                m_inDeleteTriggers = false;
            }
        }
        
        @Override
        public void onCreate()
        throws DirectoryServiceException
        {
            // For performance - don't do anything if there is nothing added
        	// don't call triggers while the DM upgrade is happening
            if (!m_hasAdded || System.getProperty("MQ_UPGRADE", "false").equals("true"))
            {
                return;
            }

            try
            {
                m_inDeleteTriggers = true;  // using m_inDeleteTriggers to mean m_inTrigger, really
                m_ds.runOnCreateTriggers(m_notificationList);
            }
            finally
            {
                m_inDeleteTriggers = false;
            }
        }

        @Override
        public void afterCreate()
        throws DirectoryServiceException
        {
            // For performance - don't do anything if there is nothing added
        	// don't call the triggers if the DM upgrade is happening. 
            if (!m_hasAdded || System.getProperty("MQ_UPGRADE", "false").equals("true"))
            {
                return;
            }

            try
            {
                m_inDeleteTriggers = true;  // using m_inDeleteTriggers to mean m_inTrigger, really
                m_ds.runAfterCreateTriggers(m_notificationList);
            }
            finally
            {
                m_inDeleteTriggers = false;
            }
        }

        @Override
        public void audit()
        throws DirectoryServiceException
        {
            if (m_context == null)
            {
                return;
            }
            IAuditManager auditManager = m_context.getAuditManager();
            if (auditManager == null)
            {
                return;
            }
            if (!auditManager.configureAuditingEnabled())
            {
                return;
            }
            if (m_logicalNameSpace == null)
             {
                return; //Can happen during failover where m_context and auditManager are already set but DS not fully initialized
            }

            ChangeAuditor auditor = new ChangeAuditor(m_ds, auditManager);
            auditModifications(auditor, m_notificationList);
            auditNamingChanges(auditor, m_logicalNameSpace.getAuditRecords());
            auditor.recordAudit();
        }

        private void auditModifications(ChangeAuditor auditor, List modifications)
        throws DirectoryServiceException
        {
            Iterator it = modifications.iterator();
            while (it.hasNext())
            {
                ModificationItem modItem = (ModificationItem)it.next();
                switch (modItem.m_type)
                {
                    case ModificationItem.CREATE:
                    {
                        IElement newElement = modItem.m_newElement;
                        auditor.auditCreate(newElement);
                        break;
                    }
                    case ModificationItem.DELETE:
                    {
                        IElement beforeImage = modItem.m_beforeImage;
                        String logicalName = modItem.m_deletedLogicalName;
                        auditor.auditDelete(beforeImage, logicalName);
                        break;
                    }
                    case ModificationItem.DELETE_NO_BEFORE_IMAGE:
                    {
                        IElement deletedElement = modItem.m_deletedElement;
                        String logicalName = modItem.m_deletedLogicalName;
                        auditor.auditDelete(deletedElement, logicalName);
                        break;
                    }
                    case ModificationItem.MODIFY:
                    {
                        IDeltaElement delta = modItem.m_delta;
                        IElement beforeImage = modItem.m_beforeImage;
                        auditor.auditUpdate(delta, beforeImage);
                        break;
                    }
                    default:
                    {
                        throw new DirectoryServiceException("Audit: unknown modification type");
                    }
                }
            }
        }

        private void auditNamingChanges(ChangeAuditor auditor, List namingChanges)
        throws DirectoryServiceException
        {
            Iterator it = namingChanges.iterator();
            while (it.hasNext())
            {
                LogicalNameSpace.AuditRecord rec = (LogicalNameSpace.AuditRecord)it.next();
                switch (rec.getAction())
                {
                    case LogicalNameSpace.AuditRecord.CREATE_FOLDER:
                    {
                        auditor.auditFolderCreate(rec.getNewPath());
                        break;
                    }
                    case LogicalNameSpace.AuditRecord.DELETE_FOLDER:
                    {
                        auditor.auditFolderDelete(rec.getOldPath());
                        break;
                    }
                    case LogicalNameSpace.AuditRecord.RENAME:
                    {
                        auditor.auditRename(rec.getOldPath(), rec.getNewPath(), rec.getStoragePath());
                        break;
                    }
                    default:
                    {
                        throw new DirectoryServiceException("Audit: unknown logical name change");
                    }
                }
            }
        }

        @Override
        public void doNotify()
        {
            m_notificationList.addSingletonRanges();
            if (m_consumer != null)
            {
                for (int i = 0; i < m_notificationList.rangesCount(); i++)
                {
                    doNotifyRange(m_notificationList.rangeIterator(i), m_notificationList.rangeSize(i), m_notificationList.rangeIsGroup(i));
                }
            }
            reset();
        }

        public void doNotifyRange(Iterator iterator, int size, boolean isGroup)
        {
            IBasicElement[] modList = new IBasicElement[size];
            int i = 0;
            while (iterator.hasNext())
            {
                modList[i++] = ((ModificationItem)iterator.next()).getModification();
            }

            if (isGroup)
            {
                IDirElement[] elementList = new IDirElement[modList.length];
                System.arraycopy(modList, 0, elementList, 0, modList.length);
                if (m_FSInterfaceIsUsed && m_localListener instanceof LocalFSListener)
                {
                    elementList = translateElementsToLogical(elementList);
                }
                m_consumer.elementsChanged(elementList);
            }
            else
            {
                if (m_FSInterfaceIsUsed && m_localListener instanceof LocalFSListener)
                {
                    modList = translateElementsToLogical(modList);
                }
                m_consumer.elementsChanged(modList);
            }
        }

        @Override
        public void reset()
        {
            m_hasDelete = false;
            m_hasAdded = false;
            m_hasUpdate = false;
            m_notificationList = new GroupArrayList();
        }

    }

    /** *********************************************************************** */
    /** ****************** Name Space Manipulation Methods ******************** */
    /** *********************************************************************** */

    @Override
    public String logicalToStorage(String path)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "logicalToStorage " + path);
        m_FSInterfaceIsUsed = true;
        try
        {
            m_lock.readLock();
            return m_logicalNameSpace.storageFromLogical(path);
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    // Used in places where we're looking for the storage name of a blob that's in
    // the process of getting added to the DS in chunks, like from ClientTransaction.
    String tempBlobLogicalToStorage(String path) throws DirectoryServiceException
    {
        try
        {
            m_lock.readLock();
            return (String)m_logicalNameSpace.tempBlobStorageName(path);
        }
        catch (ConfigException configE)
        {
        	DirectoryServiceException dirE = new DirectoryServiceException(configE.toString());
        	dirE.initCause(configE);
        	throw dirE;
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public String storageToLogical(String storageName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "storageToLogical " + storageName);

        try
        {
            m_lock.readLock();
            String logicalName = m_logicalNameSpace.logicalFromStorage(storageName);
            if (logicalName != null)
            {
                m_FSInterfaceIsUsed = true;
            }
            return logicalName;
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public void defineFolderMetaAttributes(String[] attributeNames)
    throws DirectoryServiceException
    {

        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "defineFolderMetaAttributes");
        m_FSInterfaceIsUsed = true;

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            for (int i = 0; i < attributeNames.length; i++)
            {
                m_logicalNameSpace.defineFolderMetaAttribute(attributeNames[i]);
            }
            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    @Override
    public String[] getDefinedFolderMetaAttributes()
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getDefinedFolderMetaAttributes");
        m_FSInterfaceIsUsed = true;
        try
        {
            m_lock.readLock();

            return m_logicalNameSpace.getDefinedFolderMetaAttributes();
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public void defineElementMetaAttributes(String[] attributeNames)
    throws DirectoryServiceException
    {

        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "defineElementMetaAttributes");
        m_FSInterfaceIsUsed = true;

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            for (int i = 0; i < attributeNames.length; i++)
            {
                m_logicalNameSpace.defineElementMetaAttribute(attributeNames[i]);
            }
            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    @Override
    public String[] getDefinedElementMetaAttributes()
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getDefinedElementMetaAttributes");
        m_FSInterfaceIsUsed = true;
        try
        {
            m_lock.readLock();

            return m_logicalNameSpace.getDefinedElementMetaAttributes();
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public void renameFolder(String oldName, String newName)
        throws DirectoryServiceException
    {
        try
        {
           ExtendedSonicFSFileSystem extendedFS = new ExtendedSonicFSFileSystem(this, null, null);
           SonicFSFile oldFile = extendedFS.getFileDetails(extendedFS.getCanonical(oldName));
           if (oldFile == null || !oldFile.isDirectory())
        {
            throw new DirectoryServiceException("\"" + oldName + "\" is not a folder");
        }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString());
        }


       rename(oldName, newName);
    }

    @Override
    public void renameFile(String oldName, String newName)
        throws DirectoryServiceException
    {
        try
        {
           ExtendedSonicFSFileSystem extendedFS = new ExtendedSonicFSFileSystem(this, null, null);
           SonicFSFile oldFile = extendedFS.getFileDetails(extendedFS.getCanonical(oldName));
           if (oldFile == null || !oldFile.isFile())
        {
            throw new DirectoryServiceException("\"" + oldName + "\" is not a file");
        }
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString());
        }


       rename(oldName, newName);
    }

    @Override
    public void rename(String oldName, String newName)
        throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "rename " + oldName + " to " + newName);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(oldName);
            if (handler != null)
            {
                m_dsHandlers.rename(handler, oldName, newName);
                return;
            }
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();

            m_trManager.join();
            joinedTransaction = true;
            int pathType = 0;

            try
            {
                if (isPermissionsCheckingEnabled())
                {
                    pathType  = m_logicalNameSpace.getNameSpaceType(oldName);
                }
            }
            catch (Exception e)
            {
                throw new DirectoryServiceException("Unable to get the type of the original type to check permissions: " + e.toString());
            }
            m_logicalNameSpace.rename(oldName, newName);

            try
            {
                movePermissions(oldName, newName, pathType);
            }
            catch (InvalidManagementPermissionException invalid)
            {
                throw new DirectoryServiceException("Unable to move permissions after rename " + invalid.toString());
            }

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    @Override
    public void createFolder(String folderName)
    throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "createFolder " + folderName);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(folderName);
            if (handler != null)
            {
                m_dsHandlers.createFolder(handler, folderName);
                return;
            }
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            m_logicalNameSpace.createFolder(folderName);
            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }

    }

    @Override
    public void createFolder(String folderName, boolean existingFolderOk)
    throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "createFolder " + folderName + " existingFolderOk " + existingFolderOk);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(folderName);
            if (handler != null)
            {
                m_dsHandlers.createFolder(handler, folderName, existingFolderOk);
                return;
            }
        }

        HashMap test = getMetaAttributes(folderName);
        if ((test != null) && existingFolderOk)
        {
            return;
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            m_logicalNameSpace.createFolder(folderName);
            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }

    }

    @Override
    public void deleteFolder(String folderName)
        throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "deleteFolder " + folderName);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(folderName);
            if (handler != null)
            {
                m_dsHandlers.deleteFolder(handler, folderName);
                return;
            }
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            switch (m_logicalNameSpace.getNameSpaceType(folderName))
            {
                case ILogicalNameSpace.DOES_NOT_EXIST_TYPE: // Could still be a virtual folder
                    deleteDirectory(m_logicalNameSpace.folderToDirectory(folderName));
                    break;
                case ILogicalNameSpace.ELEMENT_TYPE:
                    throw new DirectoryServiceException("Folder " + folderName + " does not exist.");

                case ILogicalNameSpace.FOLDER_TYPE:
                    String[] list = m_logicalNameSpace.list(folderName);
                    if (list.length > 0)
                    {
                        throw new DirectoryServiceException("Folder " + folderName + " cannot be deleted since it's not empty.");
                    }
                    else
                    {
                        m_logicalNameSpace.deleteFolder(folderName);
                    }
                    break;

                case ILogicalNameSpace.COMPLEX_FOLDER_TYPE:
                    String dirName = m_logicalNameSpace.storageFromLogical(folderName);
                    // Delete folderName after deleting directory, so logical name is available for auditing.
                    deleteDirectory(dirName, null);
                    m_logicalNameSpace.deleteFolder(folderName);
                    break;

                default:
                    throw new Error(); // Should never happen
            }
            removePermissions(folderName);
            transactOK = true;

        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    @Override
    public IDirElement[] getFSElements(String folderName, boolean forUpdate)
    throws DirectoryServiceException
    {
        m_FSInterfaceIsUsed = true;
        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(folderName);
            if (handler != null)
            {
                return m_dsHandlers.getFSElements(handler, folderName, forUpdate);
            }
        }

        try
        {
            // The special case of a complex configuration
            String directoryName = m_logicalNameSpace.folderToDirectory(folderName);
            if (directoryName != null)
            {
                IDirElement[] elements = getAllElements(directoryName, forUpdate);
                IReplaceRef nameReplacer = m_logicalNameSpace.createNameReplacer(false);
                for (int i = 0; i < elements.length; i++)
                {
                    String storageName = elements[i].getIdentity().getName();
                    ((Element)elements[i]).replaceReferences(false, nameReplacer);
                    setLogicalName(elements[i], nameReplacer.replace(storageName));
                }
                return elements;

            }
            else
            {
                HashMap[] elementsAtt = listFSElements(folderName);
                ArrayList elementList = new ArrayList();
                for (int i = 0; i < elementsAtt.length; i++)
                {
                    IElementIdentity id = (IElementIdentity)elementsAtt[i].get(ILogicalNameSpace.ELEMENT_IDENTITY);
                    if (id != null)
                    {
                        elementList.add(getFSElement(id.getName(), forUpdate, false));
                    }
                }
                return (IDirElement[])elementList.toArray(new IDirElement[0]);
            }

        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    @Override
    public HashMap[] listFSElements(String folderName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "listFSElements " + folderName);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(folderName);
            if (handler != null)
            {
                return m_dsHandlers.listFSElements(handler, folderName);
            }
        }

        ArrayList result = listFSAll(folderName, false, true, null);
        return (HashMap[])result.toArray(new HashMap[0]);
    }

    @Override
    public HashMap[] listAllFolders()
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "listAllFolders");

        List result = new ArrayList();

        try
        {
            m_lock.readLock();

            Map rootAttrs = new HashMap();
            rootAttrs.put(ILogicalNameSpace.FOLDER_NAME, "/");
            result.add(rootAttrs);

            _listAllFolders("/", result);
        }
        finally
        {
            m_lock.releaseLock();
        }

        return (HashMap[])result.toArray(new HashMap[0]);
    }

    private void _listAllFolders(String path, List res)
    throws DirectoryServiceException
    {
        List child = listFSAll(path, true, false, null);

        for (int i = 0; i < child.size(); i++)
        {
            Map aChild = (Map)child.get(i);
            String name = (String)aChild.get(ILogicalNameSpace.FOLDER_NAME);

            res.add(aChild);

            _listAllFolders(name, res);
        }
    }

    @Override
    public HashMap[] listFolders(String folderName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "listFolders " + folderName);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(folderName);
            if (handler != null)
            {
                return m_dsHandlers.listFolders(handler, folderName);
            }
        }

        ArrayList result = listFSAll(folderName, true, false, null);
        return (HashMap[])result.toArray(new HashMap[0]);
    }

    @Override
    public HashMap[] listFSAll(String path) throws DirectoryServiceException
    {
        return listFSAll(path, true, null);
    }

    @Override
    public HashMap[] listFSAll(String path, boolean getFolders, String extension)
    throws DirectoryServiceException
    {
        ArrayList result = listFSAll(path, getFolders, true, extension);
        return (HashMap[])result.toArray(new HashMap[0]);
    }

    @Override
    public ArrayList<HashMap> listFSAll(String path, boolean getFolders, boolean getElements, String extension)
    throws DirectoryServiceException
    {
        m_FSInterfaceIsUsed = true;
        try
        {
            if (m_dsHandlers != null)
            {
                IDSHandler handler = m_dsHandlers.getHandler(path);
                if (handler != null)
                {
                    return m_dsHandlers.listFSAll(handler, path, getFolders, getElements, extension);
                }
            }
            m_lock.readLock();

            String directoryName = m_logicalNameSpace.folderToDirectory(path);

            // The special case of a complex configuration

            if (directoryName != null)
            {
                return listFSAllComplex(path, directoryName, getFolders, getElements, extension);
            }

            String realPath = m_logicalNameSpace.getCaseSensitiveName(path);
            EntityName pathE = validateName(realPath);
            String pathPrefix = pathE.isRoot() ? "" : pathE.getName();
            String[] subItems = m_logicalNameSpace.list(pathE.getName());
            ArrayList returnList = new ArrayList();
            for (int i = 0; i < subItems.length; i++)
            {
                String itemName = pathPrefix + IMFDirectories.MF_DIR_SEPARATOR + subItems[i];
                HashMap attrs = m_logicalNameSpace.getMetaAttributes(itemName);

                if (qualifyAttributes(itemName, attrs, getFolders, getElements, extension))
                {
                    returnList.add(attrs);
                }
            }
            return returnList;
        }
        catch (Exception ex)
        {
            throw new DirectoryServiceException(ex.getMessage());
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    // Qualifies the attributes of a logical entity with its type
    private boolean qualifyAttributes(String path, HashMap attrs, boolean getFolders, boolean getElements, String extension)
    throws DirectoryServiceException
    {
        int nameSpaceType = m_logicalNameSpace.getNameSpaceType(path);

        if (getFolders && (nameSpaceType == ILogicalNameSpace.FOLDER_TYPE || nameSpaceType == ILogicalNameSpace.COMPLEX_FOLDER_TYPE))
        {
            attrs.put(ILogicalNameSpace.FOLDER_NAME, path);
            if (nameSpaceType == ILogicalNameSpace.COMPLEX_FOLDER_TYPE)
            {
                attrs.put(ILogicalNameSpace.IS_COMPLEX, Boolean.TRUE);
            }
            return true;
        }
        else if (getElements && nameSpaceType == ILogicalNameSpace.ELEMENT_TYPE)
        {
            if (extension != null && !path.endsWith(extension))
            {
                return false;
            }
            attrs.put(ILogicalNameSpace.ELEMENT_IDENTITY, getFSIdentity(path));
            return true;
        }

        return false;

    }

    private ArrayList listFSAllComplex(String logicalPath, String dirName, boolean getFolders, boolean getElements, String extension)
    throws DirectoryServiceException, ViewException, ConfigException
    {
        IIdentity[] ids = listAll(dirName);
        ArrayList returnList = new ArrayList();
        for (int i = 0; i < ids.length; i++)
        {
            HashMap attrs = new HashMap();
            if (getElements && ids[i] instanceof IElementIdentity)
            {
                ElementIdentity id = ((ElementIdentity)ids[i]).createClone();
                String logicalName = m_logicalNameSpace.logicalFromStorage(ids[i].getName());
                if (extension == null || logicalName.endsWith(extension))
                {
                    setLogicalName((IElementIdentity)id, logicalName);
                    attrs.put(ILogicalNameSpace.ELEMENT_IDENTITY, id);

                    returnList.add(attrs);
                }
            }

            if (getFolders && ids[i] instanceof IDirIdentity)
            {
                attrs.put(ILogicalNameSpace.FOLDER_NAME, m_logicalNameSpace.getCaseSensitiveName(logicalPath) + IMFDirectories.MF_DIR_SEPARATOR + validateName(ids[i].getName()).getBaseName());
                returnList.add(attrs);
            }

        }
        return returnList;
    }

    @Override
    public void setMetaAttributes(String name, HashMap newAttributes)
    throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "setMetaAttributes for " + name);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(name);
            if (handler != null)
            {
                m_dsHandlers.setMetaAttributes(handler, name, newAttributes);
                return;
            }
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;
            // ignore _ELEMENT_DENTITY, _ILogicalNameSpace.FOLDER_NAME and _ILogicalNameSpace.IS_COMPLEX
            HashMap modNewAttributes = (HashMap)newAttributes.clone();

            // Cleanup first to make sure a potential client bug will not cause the DS to send a
            // wrong notification in m_logicalNameSpace.setMetaAttributes
            modNewAttributes.remove(ILogicalNameSpace.ELEMENT_IDENTITY);
            modNewAttributes.remove(ILogicalNameSpace.FOLDER_NAME);
            modNewAttributes.remove(ILogicalNameSpace.IS_COMPLEX);
            qualifyAttributes(name, modNewAttributes, true, true, null);

            m_logicalNameSpace.setMetaAttributes(name, modNewAttributes);

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    @Override
    public HashMap getMetaAttributes(String name)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getMetaAttributes for " + name);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(name);
            if (handler != null)
            {
                return m_dsHandlers.getMetaAttributes(handler, name);
            }
        }

        try
        {
            m_lock.readLock();

            int nameSpaceType = m_logicalNameSpace.getNameSpaceType(name);

            if (nameSpaceType == ILogicalNameSpace.DOES_NOT_EXIST_TYPE) // Could still be under a complex folder
            {
                EntityName nameE = new EntityName(name);
                String parentDir = m_logicalNameSpace.folderToDirectory(nameE.getParent());

                if (parentDir != null)
                {
                    IIdentity id = getIdentity(parentDir + '/' + nameE.getBaseName());
                    // entities under a complex configuration don't have meta attributes
                    if (id != null)
                    {
                        HashMap map = new HashMap();
                        if (id instanceof IDirIdentity)
                        {
                            map.put(ILogicalNameSpace.FOLDER_NAME, name);
                        }
                        else
                        {
                            map.put(ILogicalNameSpace.ELEMENT_IDENTITY, getFSIdentity(name));
                        }
                        return map;
                    }
                    else {
                        return null; // Does not exist
                    }
                }
                else {
                    return null; // The parent is a regular folder and 'name' indeed does not exist
                }
            }

            HashMap attrs = m_logicalNameSpace.getMetaAttributes(name);

            if (nameSpaceType == ILogicalNameSpace.FOLDER_TYPE || nameSpaceType == ILogicalNameSpace.COMPLEX_FOLDER_TYPE)
            {
                attrs.put(ILogicalNameSpace.FOLDER_NAME, name);
                if (nameSpaceType == ILogicalNameSpace.COMPLEX_FOLDER_TYPE)
                {
                    attrs.put(ILogicalNameSpace.IS_COMPLEX, Boolean.TRUE);
                }
            }

            if (nameSpaceType == ILogicalNameSpace.ELEMENT_TYPE)
            {
                attrs.put(ILogicalNameSpace.ELEMENT_IDENTITY, getFSIdentity(name));
            }

            return attrs;

        }
        catch (Exception e)
        {
            trace(TRACE_ALL_DS_ACCESS, "Failed getting meta attributes, trace follows...", e);
            return null;
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    /** ******************************* ************************************** */
    /** ************************** Storage & Back reference Hints ***************************** */
    /** ********************************************************************** */
    @Override
    public void setBackReferenceTypes(String[] typeList)
    throws DirectoryServiceException
    {
        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            m_backRefMgr.setBackReferenceTypes(typeList);

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    @Override
    public String[] getBackReferenceTypes()
    throws DirectoryServiceException
    {
        try
        {
            m_lock.readLock();
            return m_backRefMgr.getBackReferenceTypes();
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public void resetBackReferences()
    throws DirectoryServiceException
    {
        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            m_backRefMgr.resetBackReferences();
            m_backRefMgr.deleteBackRefDir();

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    @Override
    public void rebuildBackReferences()
    throws DirectoryServiceException
    {
        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            m_backRefMgr.deleteBackRefDir();

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }

        transactOK = false;
        joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            m_backRefMgr.scanDSandPopulateBackRefTree();

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    @Override
    public void setStorageHint(String elementType, String directoryName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "setStorageHint for type " + elementType + " at directory " + directoryName);
        setStorageHint(elementType, null, directoryName, false);
    }

    @Override
    public void setStorageHint(String elementType, String postfix, String directoryName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "setStorageHint for type " + elementType + "." + postfix + " at directory " + directoryName);
        setStorageHint(elementType, postfix, directoryName, false);
    }

    @Override
    public void setComplexStorageHint(String elementType, String directoryName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "setStorageHint complex for type " + elementType + " at directory " + directoryName);
        setStorageHint(elementType, null, directoryName, true);
    }

    private void setStorageHint(String elementType, String postfix, String directoryName, boolean complex)
    throws DirectoryServiceException
    {

        validateOpen();

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            createHintsElement();
            String hintID = LogicalNameSpace.createHintID(elementType, postfix);

            IDirElement storageHints = getElement(STORAGE_HINTS_ELEMENT, true);
            IAttributeSet atts = storageHints.getAttributes();
            IAttributeSet hints = (IAttributeSet)atts.getAttribute(HINTS_ATT);
            IAttributeSet hint = hints.createAttributeSet(hintID);

            hint.setStringAttribute(LogicalNameSpace.HINT_DIRECTORY_ATT, new EntityName(directoryName).getName());

            if (complex)
            {
                hint.setBooleanAttribute(LogicalNameSpace.HINT_COMPLEX_ATT, Boolean.TRUE);
            }
            setElement(storageHints.doneUpdate(), null);
            m_logicalNameSpace.resetStorageHints((IAttributeSet)storageHints.getAttributes().getAttribute(HINTS_ATT));

            transactOK = true;

        }
        catch (Exception e)
        {
            e.printStackTrace();
            throw new DirectoryServiceException(e.toString());
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }

    }

    /** ******************************* ************************************** */
    /** ****************** Element Manipulation Methods ********************** */
    /** ******************************* ************************************** */

    @Override
    public IDirElement[] getFSElements(Query query, boolean forUpdate)
    throws DirectoryServiceException
    {
        return getFSElements(query, forUpdate, false);
    }

    private String[] getComplexDescriptors(String dirName, String type)
    throws DirectoryServiceException
    {
        ArrayList resultList = new ArrayList();
        IDirIdentity[] topDirIDs = listDirectories(dirName);
        for (int i = 0; i < topDirIDs.length; i++)
        {
            IElementIdentity[] topElements = listElements(topDirIDs[i].getName());
            for (int j = 0; j < topElements.length; j++)
            {
                if (topElements[j].getType().equals(type))
                {
                    resultList.add(topElements[j].getName());
                }
            }
        }
        return (String[])resultList.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
    }

    @Override
    public IDirElement[] getFSElements(Query query, boolean forUpdate,
    		boolean getSubclassingDelta) throws DirectoryServiceException
    {
    	return getFSElements(query, forUpdate, getSubclassingDelta, null).getElements();
    }

    @Override
    public QueryBatch getFSElements(Query query, boolean forUpdate,
    		boolean getSubclassingDelta, QueryBatch batch)
    throws DirectoryServiceException
    {

        trace(TRACE_ALL_DS_ACCESS, "getFSElements");
        m_FSInterfaceIsUsed = true;
        IDirElement[] candidates = null;
        IDirElement[] batchResult = null;
        try
        {
            m_lock.readLock();

            From from = query.getFrom();
            if (from == null)
            {
                throw new DirectoryServiceException("The query must contain a FROM clause.");
            }

            if (from instanceof FromElementList)
            {
                candidates = getFSElementsInternal(query, forUpdate, getSubclassingDelta);
            }
            else if (from instanceof FromElementType && m_logicalNameSpace.complexDirForType(((FromElementType)from).getType()) != null)
            {
                String descriptorType = ((FromElementType)from).getType();
                String topDir = m_logicalNameSpace.complexDirForType(descriptorType);
                String[] elementNames = getComplexDescriptors(topDir, descriptorType);

                // Convert to logical names so we can use the logical query for further processing
                for (int i = 0; i < elementNames.length; i++)
                {
                    elementNames[i] = m_logicalNameSpace.logicalFromStorage(elementNames[i]);
                }

                Query q = new Query().setSelect(query.getSelect()).setFrom(new FromElementList(elementNames)).setWhere(query.getWhere());
                candidates = getFSElementsInternal(q, forUpdate, getSubclassingDelta);
            }
            else if (from instanceof FromElementType)
            {
                String type = ((FromElementType)from).getType();
                ArrayList dirList = m_logicalNameSpace.directoriesForType(type);
                if (dirList == null || dirList.isEmpty())
                {
                    throw new DirectoryServiceException("Element type " + type + " doesn't have a storage hint.");
                }
                ArrayList allElements = new ArrayList();
                for (int i = 0; i < dirList.size(); i++)
                {
                    FromDirectory dirFrom = new FromDirectory((String)dirList.get(i));
                    Query q = new Query().setSelect(query.getSelect()).setFrom(dirFrom).setWhere(query.getWhere());
                    IDirElement[] elements = getFSElementsInternal(q, forUpdate, getSubclassingDelta);
                    for (int j = 0; j < elements.length; j++)
                    {
                        allElements.add(elements[j]);
                    }
                }
                candidates = new IDirElement[allElements.size()];
                if (candidates.length > 0)
                {
                    allElements.toArray(candidates);
                }

            }
            else if (from instanceof FromFolder)
            {
                ArrayList elements = listFSAll(((FromFolder)from).getFolderName(), false, true, null);
                if (!elements.isEmpty())
                {
                    String[] elementNames = new String[elements.size()];
                    for (int i = 0; i < elements.size(); i++)
                    {
                        String name = ((IElementIdentity)((HashMap)elements.get(i)).get(ILogicalNameSpace.ELEMENT_IDENTITY)).getName();
                        elementNames[i] = name;
                    }
                    FromElementList fromList = new FromElementList(elementNames);
                    Query q = new Query().setSelect(query.getSelect()).setFrom(fromList).setWhere(query.getWhere());
                    candidates = getFSElementsInternal(q, forUpdate, getSubclassingDelta);
                }
                else
                {
                    candidates = new IDirElement[0];
                }
            }
            else
            {
                throw new DirectoryServiceException("The IDirectoryFileSystemService interface does not " + "support the 'from directory' clause.");
            }
            OrderedBy o = query.getOrderedBy();
            if (o != null)
            {
                quickSortElements(candidates, o.getComparator());
            }
            if (batch != null)
            {
            	 batch.next(candidates);
            	 return batch;
            }
            else
            {
                return new QueryBatch(candidates, null);
            }
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    // This is called after the FromElementType clause is replaced with a FromDirectory clause
    private IDirElement[] getFSElementsInternal(Query query0, boolean forUpdate, boolean getSubclassingDelta)
    throws DirectoryServiceException
    {
        // Change all names and refereces to storage names
        Query query = ReferenceReplacer.replaceReferences(query0, m_logicalNameSpace.createNameReplacer(true));
        IDirElement[] elements = getElements(query, forUpdate, getSubclassingDelta);
        IReplaceRef nameReplacer = m_logicalNameSpace.createNameReplacer(false);
        for (int i = 0; i < elements.length; i++)
        {
            String storageName = elements[i].getIdentity().getName();
            ((Element)elements[i]).replaceReferences(false, nameReplacer);
            setLogicalName(elements[i], nameReplacer.replace(storageName));
        }
        return elements;
    }

    @Override
    public IDirElement getFSElement(String elementName, boolean forUpdate)
    throws DirectoryServiceException
    {
        m_FSInterfaceIsUsed = true;
        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(elementName);
            if (handler != null)
            {
                return m_dsHandlers.getFSElement(handler, elementName, forUpdate);
            }
        }

        return getFSElement(elementName, forUpdate, false);
    }

    @Override
    public IDirElement getFSElement(String elementName, boolean forUpdate, boolean getSubclassingDelta)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getFSElement " + elementName + " for update " + forUpdate);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(elementName);
            if (handler != null)
            {
                return m_dsHandlers.getFSElement(handler, elementName, forUpdate, getSubclassingDelta);
            }
        }

        validateName(elementName);
        try
        {
            m_lock.readLock();
            String storageName = null;
            try
            {
                storageName = m_logicalNameSpace.createNameReplacer(true).replace(elementName);
            }
            catch (Exception e)
            {
                trace(TRACE_ALL_DS_ACCESS, "getFSElement " + elementName + " not found, trace follows...", e);
                return null;
            }

            IDirElement element = getElement(storageName, forUpdate, getSubclassingDelta);
            if (element == null)
            {
                throw new DirectoryServiceException("Could not find element " + elementName + " - storage name " + storageName + " is invalid.");
            }
            ((Element)element).replaceReferences(false, m_logicalNameSpace.createNameReplacer(false));
            setLogicalName(element, m_logicalNameSpace.logicalFromStorage(storageName));
            return element;
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    // Used by the container to find the element by logical name but return with all references and ID as storage names
    @Override
    public IElement getElementByLogicalName(String logicalName)
    throws DirectoryServiceException
    {
        return getElementByLogicalName(logicalName, true);
    }

    private IElement getElementByLogicalName(String logicalName, boolean throwExceptionIfMissing)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getElementByLogicalName " + logicalName);

        validateName(logicalName);
        String expandableBlobName = null;
        try
        {
            m_lock.readLock();
            String storageName = null;
            try
            {
                storageName = m_logicalNameSpace.storageFromLogical(logicalName);
            }
            catch (ElementInPathException elInPathE)
            {
                storageName = elInPathE.getStorageName();
                expandableBlobName = elInPathE.getLogicalName();
            }
            catch (Exception e)
            {
                trace(TRACE_ALL_DS_ACCESS, "getFSElement " + logicalName + " not found, trace follows...", e);
                return null;
            }
            IElement element = getElement(storageName, false, false);
            if (element == null && throwExceptionIfMissing)
            {
                throw new DirectoryServiceException("Could not find element " + logicalName + " - storage name " + storageName + " is invalid.");
            }
            if ((expandableBlobName != null) && (element != null))
            {
                // find out if we have an expandable blob before we return the element that we found along the way
                IAttributeSet topSet = element.getAttributes();
                IAttributeSet systemAttrs = (IAttributeSet)topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);

                if (systemAttrs != null)
                {
                    Boolean expandInCache = (Boolean)systemAttrs.getAttribute(IBlob.EXPAND_IN_CACHE);
                    if ((expandInCache != null) && expandInCache.equals(IBlob.EXPANDABLE))
                    {
                        // non-persistently tell the caller of the logical name currently mapped to this element.
                        try
                        {
                            Blob.markBlobState((Element)element, expandableBlobName, IBlob.ARCHIVE_NAME);
                        }
                        catch (Exception e)
                        {
                            element = null;
                        }
                    }
                    else
                    {
                        element = null;
                    }
                }
                else
                {
                    element = null;
                }
            }

            return element;
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    // This methods allows DirectoryService to implement the IBlobSource interface
    @Override
    public IBlob getBlobByLogicalName(String unused, String blobName)
    throws DirectoryServiceException
    {
        return getBlobByLogicalName(blobName);
    }

    @Override
    public IBlob getBlobByLogicalName(String logicalName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getBlobByLogicalName " + logicalName);

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(logicalName);
            if (handler != null)
            {
                return m_dsHandlers.getBlobByLogicalName(handler, logicalName);
            }
        }

        try
        {
            m_lock.readLock();
            IElement element = getElementByLogicalName(logicalName);
            if (element == null)
            {
                return null;
            }
            IBlob retBlob = getBlob(element.getIdentity().getName(), false, 0, logicalName);
            // check to see if there is an archive name system attribute set in the element returned by
            // getElementByLogicalName
            // this attribute is not persisted, so put it in the element included in the blob object
            IAttributeSet topSet = element.getAttributes();
            IAttributeSet systemAttrs = (IAttributeSet)topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);
            if (systemAttrs != null)
            {
                String archiveLogicalName = (String)systemAttrs.getAttribute(IBlob.ARCHIVE_NAME);
                if (archiveLogicalName != null)
                {
                    Blob.markBlobState((Element)retBlob.getElement(), archiveLogicalName, IBlob.ARCHIVE_NAME);
                }
            }
            return retBlob;
        }
        catch (DirectoryServiceException dirE)
        {
            throw dirE;
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString());
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    // Used by the container to find the elements by logical name but return with all references and ID as storage names
    @Override
    public IElement[] getElementsByLogicalNames(String[] logicalNames)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getElementsByLogicalNames");

        try
        {
            m_lock.readLock();
            IElement[] elements = new IElement[logicalNames.length];
            for (int i = 0; i < elements.length; i++)
            {
                elements[i] = getElementByLogicalName(logicalNames[i], false);
            }
            return elements;
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    private void setLogicalName(IElementIdentity id, String name)
    {
        try
        {
            ((Identity)id).setName(new EntityName(name).getName());
        }
        catch (ConfigException e)
        {
            e.printStackTrace();
            throw new Error(e.toString());
        }
    }

    private void setLogicalName(IBasicElement element, String name)
    {
        try
        {
            String canonicalName = name;
            if (!name.startsWith(ILogicalNameSpace.DELETED_LABEL))
            {
                canonicalName = new EntityName(name).getName();
            }
            ((Identity)element.getIdentity()).setName(canonicalName);
        }
        catch (ConfigException e)
        {
            e.printStackTrace();
            throw new Error(e.toString());
        }
    }

    @Override
    public AttributeName[] getReferences(String elementPath)
    throws DirectoryServiceException
    {
        try
        {
            m_lock.readLock();

            String storagePath = m_logicalNameSpace.createNameReplacer(true).replace(elementPath);
            AttributeName[] attributeNames = m_backRefMgr.getReferences(storagePath);

            IReplaceRef replacer = m_logicalNameSpace.createNameReplacer(false);
            for (int i = 0; i < attributeNames.length; i++)
            {
                try
                {
                    String logicalName = replacer.replace(attributeNames[i].getElementName());
                    attributeNames[i].setElementName(logicalName);
                }
                catch (VersionOutofSyncException e)
                {
                }
            }
            return attributeNames;
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public IElementIdentity getFSIdentity(String elementName)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getFSIdentity " + elementName);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(elementName);
            if (handler != null)
            {
                return m_dsHandlers.getFSIdentity(handler, elementName);
            }
        }

        try
        {
            m_lock.readLock();
            String storageName = m_logicalNameSpace.storageFromLogical(elementName);
            IIdentity id = getIdentity(storageName);
            if (id == null)
            {
                return null;
            }
            else if (id instanceof IElementIdentity)
            {
                id = ((ElementIdentity)id).createClone();
                // Must clone before setting the logical name top prevent side effects
                setLogicalName((IElementIdentity)id, m_logicalNameSpace.logicalFromStorage(storageName));
                return (IElementIdentity)id;
            }
            else
            {
                throw new DirectoryServiceException(elementName + " is not an element.");
            }

        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    @Override
    public IElementIdentity deleteFSElement(String elementName)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "deleteFSElement " + elementName);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(elementName);
            if (handler != null)
            {
                return m_dsHandlers.deleteFSElement(handler, elementName);
            }
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        IElementIdentity id = null;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            int nameSpaceType = m_logicalNameSpace.getNameSpaceType(elementName);

            if (nameSpaceType == ILogicalNameSpace.DOES_NOT_EXIST_TYPE) // Could still be under a complex folder
            {
                String storageName = null;
                try
                {
                    storageName = m_logicalNameSpace.storageFromLogical(elementName);
                    id = getFSIdentity(elementName);
                    if (id == null && storageName != null)
                    {
                        throw new DirectoryServiceException(elementName + " is not an element.");
                    }
                }
                catch (Exception e)
                {
                }
                if (storageName == null)
                {
                    // If the parent folder exist we return null, otherwise, throw an exception
                    EntityName elementNameE = null;
                    try
                    {
                        elementNameE = new EntityName(elementName);
                        if (getMetaAttributes(elementNameE.getParent()) != null)
                        {
                            return null;
                        }
                    }
                    catch (Exception e)
                    {
                    }
                    throw new DirectoryServiceException(elementNameE.getParent() + " does not exist.");
                }

                deleteElement(storageName, null);
                m_logicalNameSpace.deleteElement(elementName, true); // Nothing to delete but m_logicalNameSpace will
                // create the notification
                transactOK = true;
                return id;
            }
            else if (nameSpaceType != ILogicalNameSpace.ELEMENT_TYPE)
            {
                throw new DirectoryServiceException(elementName + " is not an element.");
            }

            // An ELEMENT_TYPE
            id = getFSIdentity(elementName);
            IElementIdentity storageID = deleteElement(m_logicalNameSpace.storageFromLogical(elementName), null);

            // If we got here and the storageID is null it means the element already gor its storage name
            // but was not created yet. That can happen since the transaction fist assigns name to all the new elements.
            // In that case we don't want to delete the name from the logica space
            if (storageID != null)
            {
                m_logicalNameSpace.deleteElement(elementName);
            }
            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
        return id;
    }

    @Override
    public INextVersionToken createFSElement(IDirElement logicalElement)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        m_FSInterfaceIsUsed = true;
        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(logicalElement.getIdentity().getName());
            if (handler != null)
            {
                return m_dsHandlers.createFSElement(handler, logicalElement);
            }
        }

        return createFSElement(logicalElement, null);
    }

    INextVersionToken createFSElement(IDirElement logicalElement, String storageName0)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return createFSElement(logicalElement, storageName0, null);
    }

    INextVersionToken createFSElement(IDirElement logicalElement, String storageName0, ArrayList elementsWithNoStorageRef)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "createFSElement " + logicalElement);
        m_FSInterfaceIsUsed = true;
        IDirElement element = logicalElement;
        if (m_noContainer)
        {
            element = (IDirElement)((Element)logicalElement).createClone();
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        INextVersionToken token = null;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            String storageName = storageName0 != null ? storageName0 : assignStorageName(element);

            boolean storageOK = ((Element)element).replaceReferences(false, m_logicalNameSpace.createNameReplacer(true));

            if (!storageOK && elementsWithNoStorageRef != null)
            {
                elementsWithNoStorageRef.add(logicalElement.getIdentity().getName());
            }

            ((Identity)element.getIdentity()).setName(storageName);

            token = setElement(element, null);

            if (m_authentication != null && logicalElement.getIdentity().getType().equals(AuthenticationConfigManager.DOMAIN_DESC_TYPE))
            {
                IAttributeSet descAttributes = logicalElement.getAttributes();
                Boolean external = (Boolean)descAttributes.getAttribute(AuthenticationConfigManager.DOMAIN_EXTERNAL_ATT);
                if (external != null && external.booleanValue())
                {
                    m_authentication.createExternalDirectories(validateName(storageName).getParent());
                }
            }

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }

        if (token != null)
        {
            return ((NextVersionToken)token).replaceReferences(m_logicalNameSpace.createNameReplacer(false));
        }
        else
        {
            return null;
        }
    }

    INextVersionToken createFSElements(IDirElement[] logicalElements, String[] storageNames)
    throws DirectoryServiceException, VersionOutofSyncException
    {

        m_FSInterfaceIsUsed = true;
        IDirElement[] storageElements = new IDirElement[logicalElements.length];
        for (int i = 0; i < logicalElements.length; i++)
        {
            if (m_noContainer)
            {
                storageElements[i] = (IDirElement)((Element)logicalElements[i]).createClone();
            }
            else
            {
                storageElements[i] = logicalElements[i];
            }

            ((Element)storageElements[i]).replaceReferences(false, m_logicalNameSpace.createNameReplacer(true));
            ((Identity)storageElements[i].getIdentity()).setName(storageNames[i]);
        }

        INextVersionToken token = setElements(storageElements, null, null);
        if (token != null)
        {
            return ((NextVersionToken)token).replaceReferences(m_logicalNameSpace.createNameReplacer(false));
        }
        else
        {
            return null;
        }
    }

    // used by ClientTransaction to find out if it has to assign storage names
    boolean isHandledByHandler(IBasicElement element) throws DirectoryServiceException
    {
        if (m_dsHandlers != null)
        {
            String elementName = element.getIdentity().getName();
            IDSHandler handler = m_dsHandlers.getHandler(elementName);
            return (handler != null);
        }
        return false;
    }

    // two new (10/16/08) assignStorageName methods called by the pre-existing ones
    // with modView=true as the pre-existing ones always modified the view when a
    // new storage name was computed. modView=false is used by appendFSBlob to get
    // a storage name for a file without modifying the view if it knows its chunks
    // transaction is not the last chunk.

    private String assignStorageName(IBasicElement element, boolean modView)
    throws DirectoryServiceException
    {
        // if modView == false, this could be a request for a storage name from
        // a blob chunk. Look in the temporary blob logical name map.
        IElementIdentity id = element.getIdentity();
        return assignStorageName(id.getName(), id.getType(), modView);
    }

    private String assignStorageName(String logicalName, String type, boolean modView)
    throws DirectoryServiceException
    {
        String storageName = m_logicalNameSpace.createElement(logicalName, type, modView);
        ensureParentDirExists(storageName);
        return storageName;
    }

    // The original (pre 10/16/08) assignStorageName methods use modView=true as
    // we always used to modify the view element as soon as a storage name was computed
    String assignStorageName(IBasicElement element)
    throws DirectoryServiceException
    {
        return assignStorageName(element, true);
    }

    String assignStorageName(String logicalName, String type)
    throws DirectoryServiceException
    {
        return assignStorageName(logicalName, type, true);
    }

    // This is needed for the first element created under a complex configuration
    private void ensureParentDirExists(String storageName)
    throws DirectoryServiceException
    {
        try
        {
            EntityName parentE = new EntityName(storageName).getParentEntity();
            if (m_idCache.get(parentE) == null)
            {
                createDirectory(parentE.getName());
            }
        }
        catch (ConfigException e)
        {
            throw new DirectoryServiceException(e.toString());
        }
    }

    String[] assignStorageNames(IDirElement[] elements, boolean fromBulk)
    throws DirectoryServiceException
    {
        String[] result = new String[elements.length];
        for (int i = 0; i < elements.length; i++)
        {
            IElementIdentity id = elements[i].getIdentity();
            result[i] = m_logicalNameSpace.createElement(id.getName(), id.getType());

            // We don't want to call ensureParentDirExists many times if we don't have to
            if (i == 0 || !fromBulk)
            {
                ensureParentDirExists(result[i]);
            }
        }

        return result;
    }

    @Override
    public INextVersionToken updateFSElement(IDeltaDirElement logicalDelta)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(logicalDelta.getIdentity().getName());
            if (handler != null)
            {
                return m_dsHandlers.updateFSElement(handler, logicalDelta);
            }
        }

        return updateFSElement(logicalDelta, null);
    }

    public INextVersionToken updateFSElement(IDeltaDirElement logicalDelta, ArrayList elementsWithNoStorageRef)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "updateFSElement " + logicalDelta);
        m_FSInterfaceIsUsed = true;
        IDeltaDirElement delta = logicalDelta;

        if (m_noContainer)
        {
            delta = (IDeltaDirElement)((DeltaElement)logicalDelta).createClone();
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        INextVersionToken token = null;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            String storageName = m_logicalNameSpace.storageFromLogical(logicalDelta.getIdentity().getName());
            boolean storageOK = ((DeltaElement)delta).replaceReferences(false, m_logicalNameSpace.createNameReplacer(true));

            if (!storageOK && elementsWithNoStorageRef != null)
            {
                elementsWithNoStorageRef.add(logicalDelta.getIdentity().getName());
            }

            ((Identity)delta.getIdentity()).setName(storageName);


            if (delta.getIdentity().getName().equals(DOMAIN_ELEMENT_NAME))
            {
                domainElementAttributeChecks(delta);
            }

            token = setElement(delta, null);
            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }

        if (token != null)
        {
            return ((NextVersionToken)token).replaceReferences(m_logicalNameSpace.createNameReplacer(false));
        }
        else
        {
            return null;
        }
    }

    @Override
    public void copyFiles(String srcPath, String trgtPath) throws DirectoryServiceException
    {

        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "copyFiles from " + srcPath + " to " + trgtPath);
        boolean transactOK = false;
        boolean joinedTransaction = false;

        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            ExtendedSonicFSFileSystem fs = new ExtendedSonicFSFileSystem(this, TaskScheduler.getCurrentUserID(), m_domainDir);
            fs.copyFiles(srcPath, trgtPath);


            transactOK = true;
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString());
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    @Override
    public IDirElement cloneFSBlob(String fromElementName, String toElementName)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "cloneFSBlob from " + fromElementName + " to " + toElementName);
        boolean transactOK = false;
        boolean joinedTransaction = false;
        IDirElement copied = null;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            copied = cloneFSElement(fromElementName, toElementName);
            // if the blob is not COMPLETED on disk, there is no blob attachment, so we're done
            IAttributeSet topSet = copied.getAttributes();
            Object systemAttrs = topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);
            Integer blobState = null;
            if (systemAttrs != null)
            {
                blobState = (Integer)((IAttributeSet)systemAttrs).getAttribute(IBlob.LARGE_FILE_STATE);
            }
            if ((blobState == null) || (blobState.equals(IBlob.COMPLETE)))
            {
                String fromStorageName = m_logicalNameSpace.storageFromLogical(fromElementName);
                String toStorageName = m_logicalNameSpace.storageFromLogical(toElementName);
                EntityName fromEntityName = validateName(fromStorageName);
                EntityName toEntityName = validateName(toStorageName);

                m_blobStorage.copyBlob(fromEntityName, toEntityName);
            }

            transactOK = true;
        }

        catch (StorageException stE)
        {
            throw convertException(stE);
        }
        catch (Exception otherE)
        {
            throw new DirectoryServiceException(otherE.toString());
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
        return copied;
    }

    @Override
    public IDirElement cloneFSElement(String elementName, String newName)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return cloneFSElementInternal(null, elementName, newName, false, null);
    }

    @Override
    public IDirElement cloneFSElement(IBasicElement delta, String newName)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return cloneFSElementInternal(delta, null, newName, false, null);
    }

    @Override
    public IDirElement cloneFSElement(String elementName, String newName, boolean createTemplate)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return cloneFSElementInternal(null, elementName, newName, createTemplate, null);
    }

    @Override
    public IDirElement cloneFSElement(IBasicElement delta, String newName, boolean createTemplate)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return cloneFSElementInternal(delta, null, newName, createTemplate, null);
    }

    IDirElement cloneFSElementInternal(IBasicElement logicalDelta, String oldName, String newName, boolean createTemplate, String storageName0)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "cloneFSElement " + newName);
        m_FSInterfaceIsUsed = true;

        if (newName == null)
        {
            throw new DirectoryServiceException("cloneFSElement: + the name of the new element must not be null.");
        }
        if (logicalDelta == null && oldName == null)
        {
            throw new DirectoryServiceException("cloneFSElement: + the name or delta of the original element must be passed.");
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        IDirElement newElement = null;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            IReplaceRef nameReplacer = m_logicalNameSpace.createNameReplacer(true);
            DeltaElement storageDelta = (DeltaElement)logicalDelta;
            String oldStorageName = null;
            String elementType = null;

            if (oldName != null)
            {
                oldStorageName = m_logicalNameSpace.storageFromLogical(oldName);
                try
                {
                    elementType = ((IElementIdentity)m_idCache.get(new EntityName(oldStorageName))).getType();
                }
                catch (ConfigException e) // Should never happen
                {
                    e.printStackTrace();
                    throw new Error(e.toString());
                }
            }
            else
            {
                elementType = logicalDelta.getIdentity().getType();
                oldStorageName = m_logicalNameSpace.storageFromLogical(logicalDelta.getIdentity().getName());
                if (m_noContainer)
                {
                    storageDelta = ((DeltaElement)logicalDelta).createClone();
                }
                ((DeltaElement)storageDelta).replaceReferences(false, nameReplacer);
                ((Identity)storageDelta.getIdentity()).setName(oldStorageName);
            }

            String newStorageName = storageName0 != null ? storageName0 : assignStorageName(newName, elementType);

            newElement = cloneElementInternal((IBasicElement)storageDelta, oldStorageName, newStorageName, createTemplate, null);
            newElement = (IDirElement)((Element)newElement).createClone();
            ((Element)newElement).replaceReferences(false, m_logicalNameSpace.createNameReplacer(false));
            setLogicalName(newElement, m_logicalNameSpace.logicalFromStorage(newStorageName));

            transactOK = true;

        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
        return newElement;

    }

    // mrd 12/07/2004 modified to deal with large files
    @Override
    public void attachFSBlob(IBasicElement element, java.io.InputStream blobStream)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        if (blobStream == null)
        {
            throw new DirectoryServiceException("The blob stream object cannot be null.");
        }

        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(element.getIdentity().getName());
            if (handler != null)
            {
                m_dsHandlers.attachFSBlob(handler, element, blobStream);
                return;
            }
        }

        try
        {
            BufferedInputStream bStream = new BufferedInputStream(blobStream, IBlob.BLOB_CHUNK_SIZE);
            ByteArrayOutputStream oStream;
            int src = 0;

            byte[] blobPiece;
            int readIn = 0;
            while (readIn != -1)
            {
                int chunkIndex = 0;
                oStream = new ByteArrayOutputStream();
                while ((chunkIndex < IBlob.BLOB_CHUNK_SIZE) && (readIn != -1))
                {
                    readIn = bStream.read();
                    if (readIn != -1)
                    {
                        oStream.write(readIn);
                        chunkIndex = chunkIndex + 1;
                    }
                }
                oStream.close();
                blobPiece = oStream.toByteArray();
                appendFSBlob(element, blobPiece, src, (readIn == -1));
                src = src + blobPiece.length;
            }
        }
        catch (Exception e) // Could happen only because lack of memory
        {
            throw new DirectoryServiceException(e.toString());
        }
    }

    @Override
    public void attachFSBlob(IBasicElement element, byte[] blob)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(element.getIdentity().getName());
            if (handler != null)
            {
                m_dsHandlers.attachFSBlob(handler, element, blob);
                return;
            }
        }

        attachFSBlobInternal(element, blob, null);
    }

    void attachFSBlobInternal(IBasicElement element0, byte[] blobBytes, String storageName0)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        attachFSBlobInternal(element0, blobBytes, storageName0, null);
    }

    void attachFSBlobInternal(IBasicElement element0, byte[] blobBytes, String storageName0, ArrayList elementsWithNoStorageRef)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "attachFSBlob " + element0.getIdentity().getName());
        m_FSInterfaceIsUsed = true;

        boolean transactOK = false;
        boolean joinedTransaction = false;
        IBasicElement element = element0;
        String storageName = null;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;
            if (m_noContainer)
            {
                if (element instanceof Element)
                {
                    element = ((Element)element0).createClone();
                }
                else
                {
                    element = ((DeltaElement)element0).createClone();
                }
            }

            if (element instanceof Element)
            {
                storageName = storageName0 != null ? storageName0 : assignStorageName(element);
            }
            else
            {
                storageName = m_logicalNameSpace.storageFromLogical(element.getIdentity().getName());
            }

            boolean storageOK = ((ICanReplaceRef)element).replaceReferences(false, m_logicalNameSpace.createNameReplacer(true));

            if (!storageOK && elementsWithNoStorageRef != null)
            {
                elementsWithNoStorageRef.add(element0.getIdentity().getName());
            }

            ((Identity)element.getIdentity()).setName(storageName);
            attachBlob(element, blobBytes, null);

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    // the following is used by the ClientTransaction class. It needs an internal method that takes a generated
    // storage name and the ArrayList of elements with no storage reference
    void attachFSBlobInternal(IBasicElement element0, InputStream blobStream, String storageName0, ArrayList elementsWithNoStorageRef)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "attachFSBlob " + element0.getIdentity().getName());
        m_FSInterfaceIsUsed = true;

        // this dispatch to a handler is here for the Eclipse handler and the arguments are those of the public
        // attachFSBlob
        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(element0.getIdentity().getName());
            if (handler != null)
            {
                m_dsHandlers.attachFSBlob(handler, element0, blobStream);
                return;
            }
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        IBasicElement element = element0;
        String storageName = null;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;
            if (m_noContainer)
            {
                if (element instanceof Element)
                {
                    element = ((Element)element0).createClone();
                }
                else
                {
                    element = ((DeltaElement)element0).createClone();
                }
            }

            if (element instanceof Element)
            {
                storageName = storageName0 != null ? storageName0 : assignStorageName(element);
            }
            else
            {
                storageName = m_logicalNameSpace.storageFromLogical(element.getIdentity().getName());
            }

            boolean storageOK = ((ICanReplaceRef)element).replaceReferences(false, m_logicalNameSpace.createNameReplacer(true));

            if (!storageOK && elementsWithNoStorageRef != null)
            {
                elementsWithNoStorageRef.add(element0.getIdentity().getName());
            }

            ((Identity)element.getIdentity()).setName(storageName);
            attachBlob(element, blobStream, null);

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    // the following is used by the ClientTransaction class. It needs an internal method that takes a generated
    // storage name and the ArrayList of elements with no storage reference. This is always the last piece of the
    // blob, so appendBlob is called with the true flag
    void appendFSBlobInternal(IBasicElement element0, byte[] blobPiece, int offset, String storageName0, ArrayList elementsWithNoStorageRef)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "appendFSBlobInternal " + element0.getIdentity().getName());
        m_FSInterfaceIsUsed = true;

        // this dispatch to a handler is here for the Eclipse handler and the arguments are those of the public
        // appendFSBlob
        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(element0.getIdentity().getName());
            if (handler != null)
            {
                m_dsHandlers.appendFSBlob(handler, element0, blobPiece, offset, true);
                return;
            }
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        IBasicElement element = element0;
        String storageName = null;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;
            if (m_noContainer)
            {
                if (element instanceof Element)
                {
                    element = ((Element)element0).createClone();
                }
                else
                {
                    element = ((DeltaElement)element0).createClone();
                }
            }
            String logicalName = element.getIdentity().getName();
            if ((element instanceof Element) && offset == 0)
            {
                storageName = storageName0 != null ? storageName0 : assignStorageName(element);
            }
            else
            {
                storageName = storageName0 != null ? storageName0 : m_logicalNameSpace.storageFromLogical(logicalName);
            }

            boolean storageOK = ((ICanReplaceRef)element).replaceReferences(false, m_logicalNameSpace.createNameReplacer(true));

            if (!storageOK && elementsWithNoStorageRef != null)
            {
                elementsWithNoStorageRef.add(element0.getIdentity().getName());
            }

            ((Identity)element.getIdentity()).setName(storageName);
            appendBlob(element, blobPiece, offset, true, null);
            // In this last chunk of the blob, finally set the logical name of the
            // file element in the view. If this last chunk (they're always the last
            // chunk in this method) is also the first chunk, we've already added the name to
            // the view through the call to assignStorageName above.
            if (element instanceof Element && offset != 0)
            {
                m_logicalNameSpace.addViewLink(logicalName, storageName);
            }

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    // mrd 12/07/2004 new to deal with large file attachments in pieces.
    @Override
    public void appendFSBlob(IBasicElement element0, byte[] blob, int offset, boolean last)
    throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "appendFSBlob " + element0.getIdentity().getName());
        m_FSInterfaceIsUsed = true;

        // this dispatch to a handler is here for the Eclipse handler and the arguments are those of the public
        // appendFSBlob
        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(element0.getIdentity().getName());
            if (handler != null)
            {
                m_dsHandlers.appendFSBlob(handler, element0, blob, offset, last);
                return;
            }
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        IBasicElement element = element0;
        String storageName = null;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;
            if (m_noContainer)
            {
                if (element instanceof Element)
                {
                    element = ((Element)element0).createClone();
                }
                else
                {
                    element = ((DeltaElement)element0).createClone();
                }
            }

            // try and find the logical name of this element. If it's a new element,
            // the very first time appendFSBlob is called there won't be a storage name for it
            // but it will get created here. Subsequent times the assigned storage name will be found

            try
            {
                storageName = m_logicalNameSpace.storageFromLogical(element.getIdentity().getName());
            }
            catch (DirectoryServiceException dirEx)
            { // if the element is a new element, it might have not been assigned a storage name yet
                // If the blob is a new blob and this is a chunk other thant the first one,
                // m_logicalNamespace cannot calculate its storage name yet even though it's been computed
                // because it has not been put in the view. The DS then keeps the map while the blob
                // is still new and being appended to.
                if (element instanceof Element)
                {
                    if (offset != 0)
                    {
                        storageName = tempBlobLogicalToStorage(element.getIdentity().getName());//(String)m_tempBlobLogicalMap.get(element.getIdentity().getName().toLowerCase());
                    }
                    else
                    {
                        storageName = assignStorageName(element, false);
                    }
                }
                else
                {
                    throw dirEx;
                }
            }
            String logicalName = element.getIdentity().getName();
            ((Identity)element.getIdentity()).setName(storageName);
            if (last)
            {
                ((ICanReplaceRef)element).replaceReferences(false, m_logicalNameSpace.createNameReplacer(true));
                // the DS has been keeping a temporary logical name for this element while
                // chunks are appended; add it now to the view in the transaction for the last
                // chunk. This method is called for the last chunk from DSProxy.attachFSBlob
                // but not from the DSProxy transaction. That will happen through appendFSBlobInternal
                if (element instanceof Element)
                {
                    m_logicalNameSpace.addViewLink(logicalName, storageName);
                }
            }
            appendBlob(element, blob, offset, last, null);

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    // mrd 12/07/2004 New to deal with large file attachments. Get a piece of the larger blob starting at offset
    @Override
    public IBlob getFSBlob(String elementName, boolean forUpdate, int offset)
    throws DirectoryServiceException
    {
        trace(TRACE_ALL_DS_ACCESS, "getFSBlob " + elementName);
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(elementName);
            if (handler != null)
            {
                return m_dsHandlers.getFSBlob(handler, elementName, forUpdate, offset);
            }
        }

        validateName(elementName);
        try
        {
            m_lock.readLock();

            String storageName = null;
            try
            {
                storageName = m_logicalNameSpace.storageFromLogical(elementName);
            }
            catch (Exception e)
            {
                trace(TRACE_ALL_DS_ACCESS, "getFSElement " + elementName + " not found: " + e.toString());
                return null;
            }
            Blob blob = (Blob)getBlob(storageName, forUpdate, offset, elementName);
            IDirElement element = blob.getElement();
            if (element == null)
            {
                throw new DirectoryServiceException("Could not find element " + elementName + " - storage name " + storageName + " is invalid.");
            }
            ((Element)element).replaceReferences(false, m_logicalNameSpace.createNameReplacer(false));
            blob.setLogical(true);
            setLogicalName(element, m_logicalNameSpace.logicalFromStorage(storageName));
            return blob;
        }
        finally
        {
            m_lock.releaseLock();
        }

    }

    // mrd 12/07/2004 Modified to deal with large file attachments - return the first piece IBlob.BLOB_CHUNK_SIZE
    // or smaller
    @Override
    public IBlob getFSBlob(String elementName, boolean forUpdate)
    throws DirectoryServiceException
    {
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(elementName);
            if (handler != null)
            {
                return m_dsHandlers.getFSBlob(handler, elementName, forUpdate);
            }
        }

        return getFSBlob(elementName, forUpdate, 0);
    }

    @Override
    public void detachFSBlob(IDeltaDirElement logicalDelta)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "detachFSBlob " + logicalDelta.getIdentity().getName());
        m_FSInterfaceIsUsed = true;

        if (m_dsHandlers != null)
        {
            IDSHandler handler = m_dsHandlers.getHandler(logicalDelta.getIdentity().getName());
            if (handler != null)
            {
                m_dsHandlers.detachFSBlob(handler, logicalDelta);
                return;
            }
        }

        IDeltaDirElement delta = logicalDelta;

        if (m_noContainer)
        {
            delta = (IDeltaDirElement)((DeltaElement)logicalDelta).createClone();
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            String storageName = m_logicalNameSpace.storageFromLogical(logicalDelta.getIdentity().getName());
            ((DeltaElement)delta).replaceReferences(false, m_logicalNameSpace.createNameReplacer(true));
            ((Identity)delta.getIdentity()).setName(storageName);
            detachBlob(delta, null);

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    @Override
    public void repairReferences(String[] exclusions)
    throws DirectoryServiceException
    {
        try
        {
            RepairNoStorageReferences.repairReferences(this, exclusions);
        }
        catch (Exception e)
        {
            DirectoryServiceException dse = new DirectoryServiceException("Failed to repair referenaces, see cause");
            dse.initCause(e);
            throw dse;
        }
    }

    @Override
    public IDirElement subclassFSElement(IBasicElement logicalDelta, String newName)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return subclassFSElementInternal(logicalDelta, newName, null);
    }

    IDirElement subclassFSElementInternal(IBasicElement logicalDelta, String newName, String storageName0)
    throws DirectoryServiceException, VersionOutofSyncException
    {

        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "subclassFSElement " + newName);
        m_FSInterfaceIsUsed = true;

        if (newName == null)
        {
            throw new DirectoryServiceException("subclassFSElement: + the name of the new element must not be null.");
        }
        if (logicalDelta == null)
        {
            throw new DirectoryServiceException("subclassFSElement: + the template delta must not be null.");
        }

        boolean transactOK = false;
        boolean joinedTransaction = false;
        IDirElement newElement = null;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            IReplaceRef nameReplacer = m_logicalNameSpace.createNameReplacer(true);
            DeltaElement storageDelta = (DeltaElement)logicalDelta;

            String oldStorageName = m_logicalNameSpace.storageFromLogical(logicalDelta.getIdentity().getName());
            if (m_noContainer)
            {
                storageDelta = ((DeltaElement)logicalDelta).createClone();
            }
            ((DeltaElement)storageDelta).replaceReferences(false, nameReplacer);
            ((Identity)storageDelta.getIdentity()).setName(oldStorageName);

            String newStorageName = storageName0 != null ? storageName0 : assignStorageName(newName, logicalDelta.getIdentity().getType());

            newElement = subclassElement(storageDelta, newStorageName, null);
            newElement = (IDirElement)((Element)newElement).createClone();
            ((Element)newElement).replaceReferences(false, m_logicalNameSpace.createNameReplacer(false));
            setLogicalName(newElement, m_logicalNameSpace.logicalFromStorage(newStorageName));

            transactOK = true;

        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
        return newElement;
    }

    @Override
    public void unSubclassFSElement(String elementName)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "unSubclassFSElement " + elementName);
        m_FSInterfaceIsUsed = true;

        boolean transactOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;
            String storageName = m_logicalNameSpace.storageFromLogical(elementName);
            unSubclassElement(storageName, null);

            transactOK = true;
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }
    }

    @Override
    public IDSTransaction createTransaction()
    {
        return new DSTransaction();
    }

    @Override
    public INextVersionToken executeTransaction(IDSTransaction transaction)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        return executeTransaction(transaction, null);
    }

    @Override
    public INextVersionToken executeTransaction(IDSTransaction transaction, IFrameworkComponentContext context)
    throws DirectoryServiceException, VersionOutofSyncException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "executeTransaction start...");
        m_FSInterfaceIsUsed = true;
        INextVersionToken token = new ClientTransaction((DSTransaction)transaction, this, context).performActions();
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "executeTransaction done");
        return token;
    }

    @Override
    public IDirElement revertToTemplate(String elementName, AttributeName[] attributes)
    throws DirectoryServiceException
    {
        trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "revertToTemplate " + elementName);
        m_FSInterfaceIsUsed = true;

        boolean transactOK = false;
        boolean joinedTransaction = false;
        Element sub = null;
        Element newSub = null;
        Element returnSub = null;
        Element subAsIs = null;

        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;

            // get the sub as it's stored on disk (smaller than the realized element)

            subAsIs = (Element)getElementAsIs(m_logicalNameSpace.storageFromLogical(elementName), true);

            String parentName = subAsIs.getSuperElementName();
            if (parentName == null)
             {
                throw new DirectoryServiceException(elementName + " is not a subclassed element");
            // get the parent
            }

            Element parent = (Element)getElementAsIs(parentName, true).createWritableClone();

            // get the full subclassed element before the modifications so we can create a delta
            // after the attributes are reverted

            sub = (Element)realizeSubClassedElement(parent, subAsIs.getIdentity(), true);

            parent.revertToTemplate(attributes, sub.getIdentity().getName()); // use the storage identity

            // get the full subclassed element after the modifications to create the delta

            newSub = (Element)realizeSubClassedElement(parent, sub.getIdentity(), true);

            // save the parent and the sub.

            setElementInternal((IDeltaElement)parent.doneUpdate(), null, null, false); // no notifications for the
            // parent

            setElementAsIs(subAsIs.doneUpdate(), null, false);
            returnSub = (Element)getElement(sub.getIdentity().getName(), true, true);
            returnSub.replaceReferences(false, m_logicalNameSpace.createNameReplacer(false));
            setLogicalName(returnSub, m_logicalNameSpace.logicalFromStorage(sub.getIdentity().getName()));
            DeltaElement delta = sub.createDelta(newSub);
            notifyMods(new ModificationItem(delta, sub), null);

            transactOK = true;
        }
        catch (ReadOnlyException e)
        {
            throw new Error(e.toString());
        } // shouldn't happen
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactOK);
        }

        return returnSub;
    }

    /**
     * This method is used by PASS
     * <p>
     *
     * @see com.sonicsw.mf.common.IDirectoryAdminService#listExternalDomainWithManagementSPI()
     */
    @Override
    public String[] listExternalDomainWithManagementSPI()
    throws DirectoryServiceException
    {

        IIdentity[] ids = listAll(AuthenticationConfigManager.DOMAINS_DIRECTORY);

        if (ids == null)
        {
            return IEmptyArray.EMPTY_STRING_ARRAY;
        }

        ArrayList mfDomainDescriptorList = new ArrayList();

        for (int i = 0; i < ids.length; i++)
        {
            IIdentity id = ids[i];

            String mfDomainDescriptor = id.getName() + IMFDirectories.MF_DIR_SEPARATOR + AuthenticationConfigManager.DOMAIN_DESCRIPTOR;

            if (Debug.TRACE)
            {
                Debug.trace("Checking domain descriptor: " + mfDomainDescriptor);
            }

            IElement element = getElement(mfDomainDescriptor, true);

            AttributeName attrName = new AttributeName(AuthenticationConfigManager.DOMAIN_EXTERNAL_ATT);

            Boolean b = (Boolean)element.getAttribute(attrName);

            // if it is an external domain
            if (b != null && b.booleanValue())
            {
                if (Debug.TRACE)
                {
                    Debug.trace("External domain? : " + (b != null ? b.booleanValue() : false));
                }

                attrName = new AttributeName(AuthenticationConfigManager.DOMAIN_MGMT_SPI_ATT);

                Reference ref = (Reference)element.getAttribute(attrName);

                if (ref != null) // this is having a management SPI
                {
                    // add the refrence
                    mfDomainDescriptorList.add(mfDomainDescriptor);

                    if (Debug.TRACE)
                    {
                        Debug.trace("Adding to mfDomainDescriptorList: " + mfDomainDescriptor);
                    }
                }
            }
        }

        String[] retVal = new String[mfDomainDescriptorList.size()];

        mfDomainDescriptorList.toArray(retVal);

        return retVal;
    }

    /**
     * This method is used by PASS
     * <p>
     *
     * @see com.sonicsw.mf.common.IDirectoryAdminService#reloadExternalAuthenticationDomain(java.lang.String)
     */
    @Override
    public Boolean reloadExternalAuthenticationDomain(final String mfDomainDescriptor)
    throws DirectoryServiceException
    {
        try
        {
            // log a debug message
            trace(TRACE_UPDATES + TRACE_ALL_DS_ACCESS, "reloadExternalAuthenticationDomain " + mfDomainDescriptor);

            if (mfDomainDescriptor == null || mfDomainDescriptor.trim().length() == 0)
            {
                return Boolean.FALSE;
            }

            validateOpen();

            IDirElement element = getElement(mfDomainDescriptor, true);

            AttributeName attrName = new AttributeName(AuthenticationConfigManager.DOMAIN_EXTERNAL_ATT);

            Boolean isExternalDomain = (Boolean)element.getAttribute(attrName);

            // If this value is null or it is false, return Boolean.FALSE
            if (isExternalDomain == null || !isExternalDomain.booleanValue())
            {
                return Boolean.FALSE;
            }

            // Check for existance of Management SPI confoguration
            attrName = new AttributeName(AuthenticationConfigManager.DOMAIN_MGMT_SPI_ATT);

            Reference ref = (Reference)element.getAttribute(attrName);

            if (ref == null)
            {
                return Boolean.FALSE;
            }

            // If the reload is called first time, that is, before restarting the
            // Directory Service. If it is called for the first time, it will not have
            // directory structure like
            // .../external/_MFUsers
            // .../external/_MFGroups
            // It will not have a MF_REFRESH_ELEMENT - _MFRefreshTime

            // Get the domain id
            int pos = mfDomainDescriptor.lastIndexOf(IMFDirectories.MF_DIR_SEPARATOR);

            if (pos == -1)
            {
                // TODO rajiv log an error
                return Boolean.FALSE;
            }

            String domainID = mfDomainDescriptor.substring(0, pos);

            // Rajiv added:
            // When we started to create the directory structure, we broke the
            // 'reloading first time' logic.
            // The code in this matched if condition is now stale.
            // See com.sonicsw.mf.framework.directory.impl.AuthenticationConfigManager.createExternalDirectories(String
            // domainID)
            /*
             * String externalDir = domainID + IMFDirectories.MF_DIR_SEPARATOR +
             * AuthenticationConfigManager.EXTERNAL_DIR; String domainUsers = domainID + IMFDirectories.MF_DIR_SEPARATOR +
             * AuthenticationConfigManager.EXTERNAL_USERS_DIR; String domainGroups = domainID +
             * IMFDirectories.MF_DIR_SEPARATOR + AuthenticationConfigManager.EXTERNAL_GROUPS_DIR;
             *
             *
             * if (!directoryExists(externalDir) || !directoryExists(domainUsers) || !directoryExists(domainGroups)) {
             * if (Debug.TRACE) Debug.trace("reloading first time");
             * // This is the first time after the config is created and // DS has not been restarted. return
             * m_authentication.reloadExternalAuthenticationDomain(domainID); } else { if (Debug.TRACE)
             * Debug.trace("reloading second time");
             * // get the list of the domain descriptor String[] domainDescriptors =
             * m_authentication.getExternalDomainsDescriptors();
             *
             * if (domainDescriptors == null) return Boolean.FALSE;
             * // match the domain descriptors. for (int i = 0; i < domainDescriptors.length; i++) { if
             * (domainDescriptors[i].equals(mfDomainDescriptor)) { // Match found! newAuthenticationDescriptor(element);
             * return Boolean.TRUE; } } }
             */

            // get the list of the domain descriptor
            String[] domainDescriptors = m_authentication.getExternalDomainsDescriptors();

            if (domainDescriptors == null)
            {
                if (Debug.TRACE)
                {
                    Debug.trace("domainDescriptors == null, returning FALSE");
                }

                return Boolean.FALSE;
            }
            else if (domainDescriptors.length == 0)
            {
                if (Debug.TRACE)
                {
                    Debug.trace("reloading first time: " + mfDomainDescriptor);
                }

                // This is the first time after the config is created and
                // DS has not been restarted.
                Boolean b = m_authentication.reloadExternalAuthenticationDomain(domainID);

                if (Debug.TRACE)
                {
                    Debug.trace("returning: " + ((b != null && b.booleanValue() == true) ? "TRUE" : "FALSE"));
                }

                return b;
            }
            else
            {
                if (Debug.TRACE)
                {
                    Debug.trace("reloading second time: " + mfDomainDescriptor);
                }

                // match the domain descriptors.
                for (int i = 0; i < domainDescriptors.length; i++)
                {
                    if (domainDescriptors[i].equals(mfDomainDescriptor))
                    {
                        // Match found!
                        newAuthenticationDescriptor(element);

                        if (Debug.TRACE)
                        {
                            Debug.trace("Match found, returning TRUE");
                        }

                        return Boolean.TRUE;
                    }
                }
            }
        }
        catch (VersionOutofSyncException e)
        {
            if (Debug.TRACE)
            {
                e.printStackTrace();
            }

            throw e;
        }
        catch (DirectoryServiceException e)
        {
            if (Debug.TRACE)
            {
                e.printStackTrace();
            }

            throw e;
        }
        catch (Exception e)
        {
            if (Debug.TRACE)
            {
                e.printStackTrace();
            }

            throw new DirectoryServiceException(e.getMessage());
        }

        if (Debug.TRACE)
        {
            Debug.trace("Default: Returning FALSE");
        }

        return Boolean.FALSE;
    }

    @Override
    public boolean isHandlerPath(String path)
    throws DirectoryServiceException
    {
        if (m_dsHandlers != null)
        {
            return (m_dsHandlers.getHandler(path) != null);
        }
        else
        {
            return false;
        }
    }

    // getCanonicalPath could, in theory, fail on some systems. We use getAbsolutePath
    // if getCanonicalPath fails. The only drawback is "uglier" log messages
    static String getCanonicalPath(File file)
    {
        try
        {
            return file.getCanonicalPath();
        }
        catch (Exception e)
        {
            return file.getAbsolutePath();
        }
    }

    // backup method used for PSEStorage based online backup. backupDir
    // does not include the domain name, it is added here
    @Override
    public void startBackup(String backupDir, boolean overwrite)
    throws DirectoryServiceException
    {
        synchronized (m_backupStatus)
        {
            if (m_backupStatus.isInProgress())
            {
                throw new BackupAlreadyInProgress("Last backup started on " + DATE_PARSER_THREAD_LOCAL.get().format(new Date(m_backupStatus.getStartTime())));
            }
            m_backupStatus.startBackup(backupDir);
            // this status is not saved to storage. The previous backup status
            // has been saved, and if the DS was to crash now, this backup hasn't
            // really done anything, and the last backup status persisted will reflect that.
        }
        // variables used in the spawned thread
        final String finalBackupDir = backupDir;
        final File domainDir = new File(backupDir);
        final File parentDir = domainDir.getParentFile();
        final String leafDir = domainDir.getName();
        final long startTime = m_backupStatus.getStartTime();
        ;
        try
        {
            if (parentDir == null) //they only specified a filename
            {
                failBackup(backupDir, backupDir + " is an invalid backup path. The path must specify the Domain Directory where the DS storage should be placed. Example:  ./Domain1.backup");
                throw new InvalidBackupPath(backupDir + " is an invalid backup path. The path must specify the Domain Directory where the DS storage should be placed. Example:  ./Domain1.backup");
            }
            if (!parentDir.exists())
            {
                failBackup(backupDir, parentDir.getAbsolutePath() + " does not exist");
                throw new InvalidBackupPath(parentDir.getAbsolutePath() + " does not exist");
            }

            if (domainDir.exists() && (new File(domainDir, IMFDirectories.MF_DATA_DIR + PSEStorage.DB_EXTENSION)).exists() && !overwrite)
            {
                failBackup(backupDir, backupDir + " exists and overwrite flag is false");
                throw new BackupPathExists(backupDir);
            }

            domainDir.mkdir();
            if (!domainDir.exists())
            {
                failBackup(backupDir, "Directory Service backup could not find/create the backup domain directory");
                throw new DirectoryServiceException("Directory Service backup could not find/create the backup domain directory");
            }
            try
            {
                if (domainDir.getCanonicalPath().equals(m_domainDir.getCanonicalPath()))
                {
                    failBackup(backupDir, "Cannot backup Directory Service into the original DS directory; please use a different directory");
                    throw new DirectoryServiceException("Cannot backup Directory Service into the original DS directory; please use a different directory");
                }
            }
            catch (IOException ioEx)
            {
                failBackup(backupDir, ioEx.toString());
                throw new DirectoryServiceException("Error while trying to backup Directory Service " + ioEx.toString());
            }
        }
        catch (DirectoryServiceException dirE)
        {
            throw dirE;
        }
        catch (Exception e)
        {
            // this would be an exception while trying to save the backup status
            // StorageException, AttributeSetTypeException or ConfigException
            String errorMessage = "Unable to persist last backup state for backup started on " + DATE_PARSER_THREAD_LOCAL.get().format(new Date(m_backupStatus.getStartTime())) + " at location " + m_backupStatus.getLocation() + ", trace follows...";
            logMessage(errorMessage, e, Level.SEVERE);
            throw new DirectoryServiceException(errorMessage);
        }

        Thread backupThread = new Thread()
        {

            @Override
            public void run()
            {
                File dataDir = new File(domainDir, IMFDirectories.MF_DATA_DIR);

                try
                {
                    m_storage.backup(getCanonicalPath(dataDir));
                    updateBackupTimestamp(parentDir.getCanonicalPath(), leafDir);
                    try
                    {
                        finishBackup(finalBackupDir);
                    }
                    catch (Exception e3)
                    {
                        logMessage("Unable to persist backup status data for successful backup started on " + DATE_PARSER_THREAD_LOCAL.get().format(new Date(m_backupStatus.getStartTime())) + " to location " + finalBackupDir + ", trace follows...", e3, Level.SEVERE);
                    }
                }
                catch (Throwable e)
                {
                    try
                    {
                        failBackup(finalBackupDir, e);
                    }
                    catch (Throwable e2)
                    {
                        logMessage("Unable to persist backup status data for failed backup started at " + DATE_PARSER_THREAD_LOCAL.get().format(new Date(m_backupStatus.getStartTime())) + " to location " + finalBackupDir + ", trace follows...", e2, Level.SEVERE);
                    }
                }
            }
        };

        backupThread.setDaemon(true);
        backupThread.start();
    }
    
    private void failBackup(String backupDir, Throwable t)
    throws StorageException, AttributeSetTypeException, ConfigException, DirectoryServiceException
    {
        synchronized (m_backupStatus)
        {
            logMessage("Backup to " + backupDir + " started on " + DATE_PARSER_THREAD_LOCAL.get().format(new Date(m_backupStatus.getStartTime())) + " has failed, trace follows...", t, Level.SEVERE);
            m_backupStatus.failBackup();
            IDirElement backupStatus = m_storage.getElement(BACKUP_STATUS_ELEMENT_ENTITY_NAME);
            saveBackupStatusAttrs(backupStatus);
        }
    }

    private void failBackup(String backupDir, String message)
    throws StorageException, AttributeSetTypeException, ConfigException, DirectoryServiceException
    {
        synchronized (m_backupStatus)
        {
            logMessage("Backup to " + backupDir + " started on " + DATE_PARSER_THREAD_LOCAL.get().format(new Date(m_backupStatus.getStartTime())) + " has failed: " + message, Level.SEVERE);
            m_backupStatus.failBackup();
            IDirElement backupStatus = m_storage.getElement(BACKUP_STATUS_ELEMENT_ENTITY_NAME);
            saveBackupStatusAttrs(backupStatus);
        }
    }

    private void finishBackup(String backupDir)
    throws StorageException, AttributeSetTypeException, ConfigException, DirectoryServiceException
    {
        synchronized (m_backupStatus)
        {
            logMessage("DS online backup started on " + DATE_PARSER_THREAD_LOCAL.get().format(new Date(m_backupStatus.getStartTime())) + " to directory " + backupDir + " succeeded", Level.INFO);
            m_backupStatus.finishBackup();
            IDirElement backupStatus = m_storage.getElement(BACKUP_STATUS_ELEMENT_ENTITY_NAME);
            saveBackupStatusAttrs(backupStatus);
        }
    }

    private void updateBackupTimestamp(String parentDir, String leafDir)
    throws DirectoryServiceException
    {
        // remove the dsisopen element and update the version timestamp
        PSEStorage backedUpStorage = null;
        // Strings for error messages
        File dsDir = new File(parentDir, leafDir);
        String dsDirCanonical = parentDir; // to start with, in case getCanonicalPath errors
        try
        {
            dsDirCanonical = dsDir.getCanonicalPath();
            String password = (String)m_startup_properties.get(IFSStorage.PASSWORD_ATTRIBUTE);
            if ((password != null) && (password.length() != 0))
            {
                backedUpStorage = new PSEStorage(parentDir, leafDir, IMFDirectories.MF_DATA_DIR, password, null);
            }
            else
            {
                backedUpStorage = new PSEStorage(parentDir, leafDir, IMFDirectories.MF_DATA_DIR, null, null);
            }
            IDirElement backupVersionElement = backedUpStorage.getElement(BACKUP_VERSION_ELEMENT_ENTITY_NAME);
            if (backupVersionElement == null)
            {
                backupVersionElement = ElementFactory.createElement(DirectoryService.BACKUP_VERSION_ELEMENT_PATH, "backup_version", "2.0");
            }
            else
            {
                ((Element)backupVersionElement).setReadOnly(false);
            }
            IAttributeSet attributes = backupVersionElement.getAttributes();
            attributes.setLongAttribute(IDirectoryMFService.BACKUP_VERSION_ATTR, new Long(System.currentTimeMillis()));
            ((Element)backupVersionElement).setReadOnly(true);
            backedUpStorage.startTransaction();
            backedUpStorage.setElement(BACKUP_VERSION_ELEMENT_ENTITY_NAME, backupVersionElement);
            backedUpStorage.deleteElement(OPEN_ELEMENT_ENTITY_NAME, true);
            backedUpStorage.deleteElement(IDCACHE_ELEMENT_ENTITY_NAME, true);
            if (m_idCache != null)
            {
                backedUpStorage.setElement(IDCACHE_ELEMENT_ENTITY_NAME, IDCache.createEmptyForOnlineBackup(IDCACHE_ELEMENT));
            }
            
            backedUpStorage.commitTransaction();
        }
        catch (ReadOnlyException readE)
        {
            // shouldn't happen, we just fetched it for update
            throw new Error("Unable to update the backup version element for backup DS in " + dsDirCanonical + ": " + readE.toString());
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Unable to update the backup version element for backup DS in " + dsDirCanonical + ": " + e.toString());
        }
        finally
        {
            if (backedUpStorage != null)
            {
                try
                {
                    backedUpStorage.close();
                }
                catch (Exception e)
                {
                    throw new DirectoryServiceException("Unable to close the backed up storage after updating the backup timestamp in " + dsDirCanonical);
                }
            }
        }
    }

    @Override
    public IBackupStatus getBackupStatus()
    {
        return m_backupStatus;
    }

    @Override
    public IDirElement getDomainElement(boolean forUpdate)
    throws DirectoryServiceException
    {
        // ensure its there
        initDomainDirectory();

        return getElement(DOMAIN_ELEMENT_PATH, forUpdate, false);
    }

    // NOTE: this code repeats in setDomainElement. If it's changed here it
    // should be changed there as well. The two were not consolidated because
    // setDomainElement does extra stuff that updateFSElement doesn't need. When
    // I tried to consolidate them by removing the copying of the delta in setDomainElement,
    // I observed some "cache reconciliation" messages in the manage permissions
    // test in migtest, so I decide to leave setDomainElement alone.
    private void domainElementAttributeChecks(IDeltaDirElement delta)
    {
    	IDeltaAttributeSet deltaAttributeSet = (IDeltaAttributeSet)delta.getDeltaAttributes();

        String[] attributeNames = deltaAttributeSet.getNewAttributesNames();
        if (attributeNames.length > 0)
        {
            for (int i = 0; i < attributeNames.length; i++)
            {
                if (attributeNames[i].equals(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR))
                {
                    administratorCheck();
                }
            }
        }

        attributeNames = deltaAttributeSet.getModifiedAttributesNames();
        if (attributeNames.length > 0)
        {
            for (int i = 0; i < attributeNames.length; i++)
            {
                if (attributeNames[i].equals(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR))
                {
                    administratorCheck();
                }
            }
        }

        attributeNames = deltaAttributeSet.getDeletedAttributesNames();
        if (attributeNames.length > 0)
        {
            for (int i = 0; i < attributeNames.length; i++)
            {
                if (attributeNames[i].equals(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR))
                {
                    administratorCheck();
                }
            }
        }
    }

    //NOTE: The permissions code from this method repeats in domainElementAttributeChecks.
    // If it's changed here it
    // should be changed there as well. The two were not consolidated because
    // setDomainElement does extra stuff that updateFSElement doesn't need. When
    // I tried to consolidate them by removing the copying of the delta in setDomainElement,
    // I observed some "cache reconciliation" messages in the manage permissions
    // test in migtest, so I decide to leave setDomainElement alone.
    @Override
    public void setDomainElement(IDeltaDirElement delta)
    throws DirectoryServiceException
    {
        try
        {
            boolean dirty = true;
            m_lock.writeLock();

            IDeltaAttributeSet deltaAttributeSet = (IDeltaAttributeSet)delta.getDeltaAttributes();

            IDirElement domainElement = getElement(DOMAIN_ELEMENT_PATH, true, false);

            IAttributeSet domainAttributes = domainElement.getAttributes();

            String[] attributeNames = deltaAttributeSet.getNewAttributesNames();
            if (attributeNames.length > 0)
            {
                // getting the latest version and applying the changes will minimize chance of
                // getting out of sync issues
                for (int i = 0; i < attributeNames.length; i++)
                {
                    if (attributeNames[i].equals(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR))
                    {
                        administratorCheck();
                    }

                    Object newValue = deltaAttributeSet.getNewValue(attributeNames[i]);

                    if (attributeNames[i].equals(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR) && newValue == null || newValue.equals(""))
                    {
                        continue;
                    }

                    domainAttributes.setObjectAttribute(attributeNames[i], newValue);
                }

                dirty = true;
            }

            attributeNames = deltaAttributeSet.getModifiedAttributesNames();
            if (attributeNames.length > 0)
            {
                // getting the latest version and applying the changes will minimize chance of
                // getting out of sync issues
                for (int i = 0; i < attributeNames.length; i++)
                {
                    if (attributeNames[i].equals(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR))
                    {
                        administratorCheck();
                    }
                    Object newValue = deltaAttributeSet.getNewValue(attributeNames[i]);

                    if (attributeNames[i].equals(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR) && newValue == null || newValue.equals(""))
                    {
                        domainAttributes.deleteAttribute(attributeNames[i]);
                    }
                    else
                    {
                        domainAttributes.setObjectAttribute(attributeNames[i], newValue);
                    }
                }

                dirty = true;
            }

            attributeNames = deltaAttributeSet.getDeletedAttributesNames();
            if (attributeNames.length > 0)
            {
                // getting the latest version and applying the changes will minimize chance of
                // getting out of sync issues
                for (int i = 0; i < attributeNames.length; i++)
                {
                    if (attributeNames[i].equals(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR))
                    {
                        administratorCheck();
                    }

                    domainAttributes.deleteAttribute(attributeNames[i]);
                }

                dirty = true;
            }

            if (dirty)
            {
                setElement(domainElement.doneUpdate(), null, true);
            }
        }
        catch (VersionOutofSyncException e)
        {
            throw new Error(e.toString());
        }
        catch (ConfigException e)
        {
            throw new Error(e.toString());
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    private void administratorCheck()
    {
        if (TaskScheduler.isExecutionThread())
        {
            String currentUser = TaskScheduler.getCurrentUserID();
            if (currentUser == null || currentUser.length() == 0)
            {
                throw new ManagementPermissionDeniedException("No user identity associated with mangement request, check management security settings");
            }
            if (!currentUser.equals(IManagementPermission.SUPER_USER_NAME))
            {
                throw new ConfigurePermissionDeniedException("Permission denied (not \"Administrator\"): user identity=" + currentUser, null, 0);
            }
        }
    }

    @Override
    public IManagementPermission[][] getManagementPermissions(String paths[], String type)
    throws DirectoryServiceException
    {
        return getManagementPermissions(paths, type, null, true);
    }


    @Override
    public IManagementPermission[][] getManagementPermissions(String[] paths, String type, IFrameworkComponentContext context, boolean checkPath)
    throws DirectoryServiceException
    {
        ManagementPermissionFactory factory = new ManagementPermissionFactory();
        IManagementPermission[][] permissions = new IManagementPermission[paths.length][];
        IDirElement permissionsEl = null;
        try
        {
            if (type.equals(IManagementPermission.CONFIGURE_TYPE))
            {
                permissionsEl = getElement(CONFIGURE_PERMISSIONS_PATH, false);
            }
            else
            {
                permissionsEl = getElement(MANAGE_PERMISSIONS_PATH, false);
            }
        }
        catch (DirectoryDoesNotExistException e)
        {
            // Sonic00039841 - if permissions checking was enabled without creating the default permissions then the first access
            //                 would fail because the directories had not been created .. so create them and try again!
            initPermissionsDirectory();
            return getManagementPermissions(paths, type, context, checkPath);
        }
        catch (Exception e)
        {
            DirectoryServiceException dse = new DirectoryServiceException("Unable to get the " + type + " permissions element");
            dse.initCause(e);
            throw dse;
        }
        IAttributeSet topSet = permissionsEl.getAttributes();
        for (int i = 0; i < paths.length; i++)
        {
            String path = paths[i];
            // check the validity of the path
            ManagementPermission mp = new ManagementPermission(path, type, checkPath);
            // if running inside a container, check that the caller has the right permission to view the info
            if ( context != null)
            {
                mp.getInformationCheck(context);
            }
            String getPermissionsPath = mp.getStoredPath();
            String escapedPath = getPermissionsPath.replaceAll(IPermissionsManager.PATH_DELIMITER, IPermissionsManager.ESCAPED_PATH_DELIMITER);
            escapedPath = escapedPath.replaceAll(IPermissionsManager.SPACE_CHARACTER, IPermissionsManager.ESCAPED_SPACE_CHARACTER);
            IAttributeSet pathPermissions = (IAttributeSet)topSet.getAttribute(escapedPath);
            if (pathPermissions == null)
            {
                permissions[i] = new IManagementPermission[0];
            }
            else
            {
                HashMap allPrincipalPerms = pathPermissions.getAttributes();
                Set principals = allPrincipalPerms.keySet();
                Iterator it = principals.iterator();
                int index = 0;
                IManagementPermission[] permissionArray = new IManagementPermission[principals.size()];
                while (it.hasNext())
                {
                    String principal = (String)it.next();
                    short principalType = IManagementPermission.UNKNOWN_PRINCIPAL_TYPE;
                    if (context != null && context.getPermissionsManager() != null)
                    {
                        principalType = context.getPermissionsManager().getPrincipalType(principal);
                    }
                    else
                    {
                        principalType = getPrincipalType(principal);
                    }
                    IAttributeSet perms = (IAttributeSet)allPrincipalPerms.get(principal);
                    String permsVal = (String)perms.getAttribute(PERMISSIONS_VALUE_ATTRIBUTE_NAME);
                    Integer scope = new Integer(permsVal.substring(0, permsVal.indexOf(":")));
                    Integer princPerms = new Integer(permsVal.substring(permsVal.indexOf(":") + 1));
                    IManagementPermission managementPerm = factory.createManagementPermission(principal, principalType, scope.intValue(), princPerms.intValue());
                    permissionArray[index++] = managementPerm;
                }
                permissions[i] = permissionArray;
            }
        }
        return permissions;
    }

    private short getPrincipalType(String principal) throws DirectoryServiceException
    {
        IDirElement domainElement = getDomainElement(false);
        if (domainElement == null)
        {
            return IManagementPermission.UNKNOWN_PRINCIPAL_TYPE;
        }
        IAttributeSet topSet = domainElement.getAttributes();
        Reference authDomainRef = (Reference)topSet.getAttribute(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR);
        if (authDomainRef == null)
        {
            return IManagementPermission.UNKNOWN_PRINCIPAL_TYPE;
        }
        String domainElName = authDomainRef.getElementName();
        EntityName en;
        try
        {
            en = new EntityName(domainElName);
        }
        catch (ConfigException ce)
        {
            throw new DirectoryServiceException("Unable to check principal type for " + principal + ": " + ce.toString());
        }

        String parentDir = en.getParent();
        Query findPrincipalQuery = new Query();
        findPrincipalQuery.setFrom(new FromDirectory(parentDir + IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_USERS_DIR));
        EqualExpression comparison = new EqualExpression(new AttributeName(IAuthenticationUserConstants.USER_NAME_ATTR), principal);
        Where nameWhere = new Where(new EqualExpression[] {comparison});
        findPrincipalQuery.setWhere(nameWhere);
        IDirElement[] users = getElements(findPrincipalQuery, false);
        if (users.length > 0)
        {
            return IManagementPermission.USER_PRINCIPAL_TYPE;
        }
        // try to find it in the groups
        findPrincipalQuery.setFrom(new FromDirectory(parentDir + IMFDirectories.MF_DIR_SEPARATOR + IMFDirectories.MF_GROUPS_DIR));
        comparison = new EqualExpression(new AttributeName(IAuthenticationGroupConstants.GROUP_NAME_ATTR), principal);
        nameWhere = new Where(new EqualExpression[] {comparison});
        findPrincipalQuery.setWhere(nameWhere);
        IDirElement[] groups = getElements(findPrincipalQuery, false);
        if (groups.length > 0)
        {
            return IManagementPermission.GROUP_PRINCIPAL_TYPE;
        }
        return IManagementPermission.UNKNOWN_PRINCIPAL_TYPE;
    }

    @Override
    public void setManagementPermissions(String[] paths, String type, IManagementPermission[][] permissions)
    throws DirectoryServiceException, InvalidManagementPermissionException
    {
        setManagementPermissions(paths, type, permissions, null);
    }

    @Override
    public void setManagementPermissions(String[] paths, String type, IManagementPermission[][] permissions, IFrameworkComponentContext context)
    throws DirectoryServiceException, InvalidManagementPermissionException
    {
        m_lock.writeLock();
        try
        {
            initPermissionsDirectory();
            if (paths.length != permissions.length)
            {
                throw new DirectoryServiceException("The number of paths and the number of permission arrays does not match");
            }

            // can't set permissions for super user
            for (int i = 0; i < permissions.length; i++)
            {
                for (int j = 0; j < permissions[i].length; j++)
                {
                   if (permissions[i][j].getPrincipal().equals(IManagementPermission.SUPER_USER_NAME))
                {
                    throw new InvalidManagementPermissionException("Permissions cannot be defined for the user \"" + IManagementPermission.SUPER_USER_NAME + "\"");
                }
                }
            }

            // assume the /permissions directory already exists. We'd probably check for it
            // and create it if necessary when security is enabled
            IDirElement permissionsEl = null;
            try
            {
                if (type.equals(IManagementPermission.CONFIGURE_TYPE))
                {
                    permissionsEl = getElement(CONFIGURE_PERMISSIONS_PATH, true);
                }
                else
                {
                    permissionsEl = getElement(MANAGE_PERMISSIONS_PATH, true);
                }
            }
            catch (Exception e)
            {
                throw new DirectoryServiceException("setManagementPermissions: Unable to get " + type + " permissions element: " + e.toString());
            }
            IAttributeSet topSet = permissionsEl.getAttributes();

            for (int i = 0; i < paths.length; i++)
            {
                String path = paths[i];
                try
                {
                    ManagementPermission mp = new ManagementPermission(path, type, true);
                    if (context != null)
                    {
                        mp.setPermissionsCheck(context);
                    }
                    IManagementPermission[] pathPermissions = permissions[i];
                    String storedPath = mp.getStoredPath();
                    String hierarchicalPath = getHierarchicalPath(storedPath);
                    if (hierarchicalPath != null && !hierarchicalPath.equals(storedPath))
                    {
                    	if (!isHierarchicalPath(storedPath))
                        {
                    	    //could be the parent hierachical path with the suffix on it
                            throw new InvalidManagementPermissionException("Cannot set permissions on subconfigurations of hierarchical configurations: " + storedPath);
                        }
                    }
                    String escapedPath = storedPath.replaceAll(IPermissionsManager.PATH_DELIMITER, IPermissionsManager.ESCAPED_PATH_DELIMITER);
                    escapedPath = escapedPath.replaceAll(IPermissionsManager.SPACE_CHARACTER, IPermissionsManager.ESCAPED_SPACE_CHARACTER);
                    IAttributeSet permissionsSet = (IAttributeSet)topSet.getAttribute(escapedPath);
                    if (permissionsSet == null)
                    {
                        permissionsSet = topSet.createAttributeSet(escapedPath);
                    }

                    for (int j = 0; j < pathPermissions.length; j++)
                    {
                        IManagementPermission perm = pathPermissions[j];
                        mp.setPermission(perm, permissionsSet, context);
                    }
                }
                catch (InvalidManagementPermissionException invalidPerm)
                {
                    throw invalidPerm;
                }
                catch (InvalidPathException invalidPath)
                {
                    throw invalidPath;
                }
                catch (ConfigurePermissionDeniedException denied)
                {
                    throw denied;
                }
                catch (Exception e)
                {
                    throw new DirectoryServiceException("Unable to set the " + type + " permissions on path " + path + ": " + e.toString());
                }
            }
            try
            {
                setElement(permissionsEl.doneUpdate(), null);
            }
            catch (Exception e)
            {
                throw new DirectoryServiceException("Unable to store the modified " + type + " permissions element: " + e.toString());
            }
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public void removeManagementPermissions(String[] paths, String type)
    throws DirectoryServiceException
    {
        removeManagementPermissions(paths, type, (IFrameworkComponentContext)null, true);
    }

    @Override
    public void removeManagementPermissions(String[] paths, String type, IFrameworkComponentContext context, boolean checkPath)
    throws DirectoryServiceException
    {
        m_lock.writeLock();
        try
        {
            boolean dirty = false;
            initPermissionsDirectory();
            IDirElement permissionsElement = null;
            try
            {
                if (type.equals(IManagementPermission.CONFIGURE_TYPE))
                {
                    permissionsElement = getElement(CONFIGURE_PERMISSIONS_PATH, true);
                }
                else
                {
                    permissionsElement = getElement(MANAGE_PERMISSIONS_PATH, true);
                }
            }
            catch (Exception getE)
            {
                throw new DirectoryServiceException("removeManagementPermissions: Unable to get " + type + " permissions element: " + getE.toString());
            }
            IAttributeSet permissionsAttrs = permissionsElement.getAttributes();
            for (int i = 0; i < paths.length; i++)
            {

                String path = paths[i];
                ManagementPermission permission = new ManagementPermission(path, type, checkPath); // check path
                String storedPath = permission.getStoredPath();
                String escapedPath = storedPath.replaceAll(IPermissionsManager.PATH_DELIMITER, IPermissionsManager.ESCAPED_PATH_DELIMITER);
                escapedPath = escapedPath.replaceAll(IPermissionsManager.SPACE_CHARACTER, IPermissionsManager.ESCAPED_SPACE_CHARACTER);
                IAttributeSet pathPermissions = (IAttributeSet)permissionsAttrs.getAttribute(escapedPath);
                if (pathPermissions != null)
                {
                    try
                    {
                        if (context != null)
                        {
                            permission.setPermissionsCheck(context);
                        }
                        permissionsAttrs.deleteAttribute(escapedPath);
                        dirty = true;
                    }
                    catch (ConfigException configE)
                    {
                        throw new DirectoryServiceException("Unable to remove permissions for path " + path + ": " + configE.toString());
                    }
                }
            }
            try
            {
                if (dirty)
                {
                    setElement(permissionsElement.doneUpdate(), null);
                }
            }
            catch (Exception e)
            {
                throw new DirectoryServiceException("Unable to set the modified " + type + " permissions element: " + e.toString());
            }
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    @Override
    public void removeManagementPermissions(String[] paths, String type, String[][] principals)
    throws DirectoryServiceException, InvalidManagementPermissionException
    {
        removeManagementPermissions(paths, type, principals, null, true);
    }

    @Override
    public void removeManagementPermissions(String[] paths, String type, String[][] principals, IFrameworkComponentContext context, boolean checkPath)
    throws DirectoryServiceException, InvalidManagementPermissionException
    {
        m_lock.writeLock();
        try
        {
            boolean dirty = false;
            initPermissionsDirectory();
            IDirElement permissionsElement = null;
            try
            {
                if (type.equals(IManagementPermission.CONFIGURE_TYPE))
                {
                    permissionsElement = getElement(CONFIGURE_PERMISSIONS_PATH, true);
                }
                else
                {
                    permissionsElement = getElement(MANAGE_PERMISSIONS_PATH, true);
                }
            }
            catch (Exception getE)
            {
                throw new DirectoryServiceException("removeManagementPermissions: Unable to get " + type + " permissions element: " + getE.toString());
            }
            IAttributeSet permissionsAttrs = permissionsElement.getAttributes();
            for (int i = 0; i < paths.length; i++)
            {
                String path = paths[i];
                ManagementPermission permission = new ManagementPermission(path, type, checkPath); // check the path
                if (context != null)
                {
                    permission.setPermissionsCheck(context);
                }
                String storedPath = permission.getStoredPath();
                String escapedPath = storedPath.replaceAll(IPermissionsManager.PATH_DELIMITER, IPermissionsManager.ESCAPED_PATH_DELIMITER);
                escapedPath = escapedPath.replaceAll(IPermissionsManager.SPACE_CHARACTER, IPermissionsManager.ESCAPED_SPACE_CHARACTER);
                IAttributeSet pathPermissions = (IAttributeSet)permissionsAttrs.getAttribute(escapedPath);
                if (pathPermissions != null)
                {
                    String[] toRemove = principals[i];
                    if (toRemove != null)
                    {
                        for (int j = 0; j < toRemove.length; j++)
                        {
                            String principal = toRemove[j];
                            try
                            {
                                permission.deletePermission(principal, pathPermissions);
                                dirty = true;
                            }
                            catch (ConfigException configE)
                            {
                                throw new DirectoryServiceException("Unable to remove permissions for path " + path + " and principal " + principal + ": " + configE.toString());
                            }
                        }
                    }
                    if (pathPermissions.getAttributes().size() == 0)
                    {
                        try
                        {
                            permissionsAttrs.deleteAttribute(escapedPath);
                            dirty = true;
                        }
                        catch (ConfigException configE)
                        {
                            throw new DirectoryServiceException("Unable to remove permissions for path " + path + ": " + configE.toString());
                        }
                    }
                }
                else
                {
                    throw new InvalidPathException("Path " + path + " has no " + type + " permissions set");
                }
            }
            try
            {
                if (dirty)
                {
                    setElement(permissionsElement.doneUpdate(), null);
                }
            }
            catch (Exception e)
            {
                throw new DirectoryServiceException("Unable to set the modified " + type + " permissions element: " + e.toString());
            }
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    // assume this method is called from inside a transaction
    private void movePermissions(String oldPath, String newPath, int pathType)
        throws DirectoryServiceException, InvalidManagementPermissionException
    {
        if (isPermissionsCheckingEnabled())
        {
            String oldPermissionsPath = oldPath;
            String newPermissionsPath = newPath;
            boolean isFolder = pathType == ILogicalNameSpace.FOLDER_TYPE;
            if (isFolder)
            {
                if (!oldPath.endsWith("/"))
                {
                    oldPermissionsPath = oldPath + "/";
                }
                if (!newPath.endsWith("/"))
                {
                    newPermissionsPath = newPath + "/";
                }
            }

            IManagementPermission[][] configurePerms = getManagementPermissions(new String[] {oldPermissionsPath}, IManagementPermission.CONFIGURE_TYPE, null, false);
            IManagementPermission[][] managePerms =getManagementPermissions(new String[] {oldPermissionsPath}, IManagementPermission.MANAGE_TYPE, null, false);
            if (configurePerms[0].length > 0)
            {
                setManagementPermissions(new String[] {newPermissionsPath}, IManagementPermission.CONFIGURE_TYPE, configurePerms);
            }

            if (managePerms[0].length > 0)
            {
                setManagementPermissions(new String[] {newPermissionsPath}, IManagementPermission.MANAGE_TYPE, managePerms);
            }
            removeManagementPermissions(new String[] {oldPermissionsPath}, IManagementPermission.CONFIGURE_TYPE, null, false);
            removeManagementPermissions(new String[] {oldPermissionsPath}, IManagementPermission.MANAGE_TYPE, null, false);
        }
    }

    void copyConfigurePermissions(String srcPath, String trgtPath, int pathType)
        throws DirectoryServiceException, InvalidManagementPermissionException
    {

        String srcPermissionsPath = srcPath;
        String trgtPermissionsPath = trgtPath;
        boolean isFsrcer = pathType == ILogicalNameSpace.FOLDER_TYPE;
        if (isFsrcer)
        {
            if (!srcPath.endsWith("/"))
            {
                srcPermissionsPath = srcPath + "/";
            }
            if (!trgtPath.endsWith("/"))
            {
                trgtPermissionsPath = trgtPath + "/";
            }
        }

        IManagementPermission[][] configurePerms = getManagementPermissions(new String[] {srcPermissionsPath}, IManagementPermission.CONFIGURE_TYPE, null, false);

        if (configurePerms[0].length > 0)
        {
            setManagementPermissions(new String[] {trgtPermissionsPath}, IManagementPermission.CONFIGURE_TYPE, configurePerms);
        }
    }

    @Override
    public void removeAllManagementPermissions()
    throws DirectoryServiceException
    {
        administratorCheck();
        initPermissionsDirectory();
        boolean transactionOK = false;
        boolean joinedTransaction = false;
        try
        {

            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;
            IDirElement permissions = getElement(MANAGE_PERMISSIONS_PATH, true);
            IAttributeSet topSet = permissions.getAttributes();
            HashMap attrs = topSet.getAttributes();
            Set keys = attrs.keySet();
            Iterator keysIT = keys.iterator();
            while (keysIT.hasNext())
            {
                topSet.deleteAttribute((String)keysIT.next());
            }
            setElement(permissions.doneUpdate(), null);
            permissions = getElement(CONFIGURE_PERMISSIONS_PATH, true);
            topSet = permissions.getAttributes();
            attrs = topSet.getAttributes();
            keys = attrs.keySet();
            keysIT = keys.iterator();
            while (keysIT.hasNext())
            {
                topSet.deleteAttribute((String)keysIT.next());
            }
            setElement(permissions.doneUpdate(), null);
            transactionOK = true;
        }
        catch (Exception storeE)
        {
            throw new DirectoryServiceException("Unable to remove all permissions: " + storeE.toString());
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactionOK);
        }
    }

    @Override
    public void setDefaultManagementPermissions()
    throws DirectoryServiceException
    {
        administratorCheck();
        initPermissionsDirectory();
        boolean transactionOK = false;
        boolean joinedTransaction = false;
        try
        {
            m_lock.writeLock();
            m_trManager.join();
            joinedTransaction = true;
            // manage permissions
            IDirElement permissions = getElement(MANAGE_PERMISSIONS_PATH, true);
            IAttributeSet topSet = permissions.getAttributes();
            // in this case, we'll remove any existing permission for Administrators,
            // so we're left with the default one
            String escapedPath = ROOT_DIRECTORY.replaceAll(IPermissionsManager.PATH_DELIMITER, IPermissionsManager.ESCAPED_PATH_DELIMITER);
            escapedPath = escapedPath.replaceAll(IPermissionsManager.SPACE_CHARACTER, IPermissionsManager.ESCAPED_SPACE_CHARACTER);
            IAttributeSet rootManagePath = (IAttributeSet)topSet.getAttribute(escapedPath);
            if (rootManagePath == null)
            {
                rootManagePath = topSet.createAttributeSet(escapedPath);
            }
            IAttributeSet adminPerm = (IAttributeSet)rootManagePath.getAttribute(GLOBAL_PERMISSIONS_GROUP_NAME);
            if (adminPerm != null)
            {
                rootManagePath.deleteAttribute(GLOBAL_PERMISSIONS_GROUP_NAME);
            }

            adminPerm = rootManagePath.createAttributeSet(GLOBAL_PERMISSIONS_GROUP_NAME);
            int adminManageScope = IManageScopeBits.ALL_FOLDERS_SCOPE | IManageScopeBits.ALL_COMPONENTS_SCOPE | IManageScopeBits.ALL_CONTAINERS_SCOPE;
            int adminManagePerms = IManagePermissionBits.ALLOW_PERFORM_ACTIONS | IManagePermissionBits.ALLOW_SET_ATTRIBUTES | IManagePermissionBits.ALLOW_NOTIFICATION_SUBSCRIPTION | IManagePermissionBits.ALLOW_LIFE_CYCLE_CONTROL
                                   | IManagePermissionBits.ALLOW_GET_INFORMATION | IManagePermissionBits.ALLOW_ENABLE_DISABLE_METRICS;

            adminPerm.setStringAttribute(PERMISSIONS_VALUE_ATTRIBUTE_NAME, new String(adminManageScope + ":" + adminManagePerms));
            setElement(permissions.doneUpdate(), null);
            // configure permissions
            permissions = getElement(CONFIGURE_PERMISSIONS_PATH, true);
            topSet = permissions.getAttributes();
            rootManagePath = (IAttributeSet)topSet.getAttribute(escapedPath);
            if (rootManagePath == null)
            {
                rootManagePath = topSet.createAttributeSet(escapedPath);
            }
            adminPerm = (IAttributeSet)rootManagePath.getAttribute(GLOBAL_PERMISSIONS_GROUP_NAME);
            if (adminPerm != null)
            {
                rootManagePath.deleteAttribute(GLOBAL_PERMISSIONS_GROUP_NAME);
            }
            adminPerm = rootManagePath.createAttributeSet("Administrators");
            int adminConfigurePerm = IConfigurePermissionBits.ALLOW_DELETE | IConfigurePermissionBits.ALLOW_READ | IConfigurePermissionBits.ALLOW_SET_PERMISSIONS | IConfigurePermissionBits.ALLOW_WRITE;
            int adminConfigureScope = IConfigureScopeBits.ALL_CONFIGURATIONS_AND_FILES_SCOPE | IConfigureScopeBits.ALL_FOLDERS_SCOPE | IConfigureScopeBits.THIS_FOLDER_SCOPE;

            adminPerm.setStringAttribute(PERMISSIONS_VALUE_ATTRIBUTE_NAME, new String(adminConfigureScope + ":" + adminConfigurePerm));
            setElement(permissions.doneUpdate(), null);
            transactionOK = true;
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Unable to add default management permissions: " + e.toString());
        }
        finally
        {
            leaveTransactionAndReleaseLock(joinedTransaction, transactionOK);
        }

    }

    private void initDomainDirectory()
    throws DirectoryServiceException
    {
        try
        {
            if (!m_storage.directoryExists(DOMAIN_PATH_ENTITY_NAME))
            {
                createDirectory(DOMAIN_PATH);
            }

            IDirElement domain = m_storage.getElement(DOMAIN_ELEMENT_ENTITY_NAME);
            if (domain == null)
            {
                domain = ElementFactory.createElement(DOMAIN_ELEMENT_PATH, "1.0", "MF_DOMAIN");

                // create default values
                IAttributeSet domainAttributes = domain.getAttributes();
                domainAttributes.setBooleanAttribute(IDomainConstants.AUDIT_CONFIGURE_EVENTS_ATTR, Boolean.FALSE);
                domainAttributes.setBooleanAttribute(IDomainConstants.AUDIT_MANAGE_EVENTS_ATTR, Boolean.FALSE);
                domainAttributes.setBooleanAttribute(IDomainConstants.ENABLE_CENTRALIZED_AUDIT_ATTR, Boolean.FALSE);
                domainAttributes.setStringAttribute(IDomainConstants.DEFAULT_MANAGEMENT_AUDIT_CONFIG_ATTR, "sonicfs:///Security/DefaultAuditDestinations.xml");

                setElement(domain.doneUpdate(), null);
            }
        }
        catch (AttributeSetTypeException e)
        {
            throw new Error(e.toString());
        }
        catch (StorageException e)
        {
            throw convertException(e);
        }
        catch (ReadOnlyException e)
        {
            throw new Error(e.toString());
        }
        catch (ConfigException e)
        {
            throw new Error(e.toString());
        }
    }

    private void initPermissionsDirectory()
    throws DirectoryServiceException
    {
        try
        {
            if (!m_storage.directoryExists(PERMISSIONS_PATH_ENTITY_NAME))
            {
                createDirectory(PERMISSIONS_PATH);
            }

            IDirElement permissions = m_storage.getElement(MANAGE_PERMISSIONS_ELEMENT_ENTITY_NAME);
            if (permissions == null)
            {
                permissions = ElementFactory.createElement(MANAGE_PERMISSIONS_PATH, "MF_MANAGEMENT_PERMISSIONS", "1.0");
                setElement(permissions.doneUpdate(), null);
            }
            permissions = m_storage.getElement(CONFIGURE_PERMISSIONS_ELEMENT_ENTITY_NAME);
            if (permissions == null)
            {
                permissions = ElementFactory.createElement(CONFIGURE_PERMISSIONS_PATH, "MF_MANAGEMENT_PERMISSIONS", "1.0" );
                setElement(permissions.doneUpdate(), null);
            }
            initDomainDirectory();
        }
        catch (StorageException e)
        {
            throw convertException(e);
        }
        catch (ReadOnlyException e)
        {
            throw new Error(e.toString());
        }
    }

    private class ManagementPermission
    {
        int m_pathType;

        String m_containerName;

        String m_componentName;

        String m_path;

        String m_principal;

        String m_permissionType;

        int m_scope;

        int m_permissions;

        String m_folderName = null; // we store folder names with traling slashes in the permissions elements

        ManagementPermission(String path, String type, boolean checkPath)
        throws DirectoryServiceException
        {
            m_path = path;
            m_permissionType = type;
            try
            {
                if (checkPath)
                {
                    if (type.equals(IManagementPermission.CONFIGURE_TYPE))
                    {
                        m_pathType = checkConfigurePath();
                    }
                    else
                    {
                        m_pathType = checkManagePath();
                    }
                }
            }
            catch (InvalidPathException invalidE)
            {
                throw invalidE;
            }
            catch (Exception e)
            {
                throw new DirectoryServiceException("Error while setting " + type + " permissions for " + path + ": " + e.toString());
            }
        }

        final int checkConfigurePath()
        throws DirectoryServiceException
        {
            // the path has to be a folder or an element to be able to set configure permissions on it

            int type = m_logicalNameSpace.getNameSpaceType(m_path);
            if (type == ILogicalNameSpace.DOES_NOT_EXIST_TYPE)
            {
                // it could be an authentication domain...
                if (m_path.endsWith("_MFDomainDescriptor") && getFSElement(m_path, false) != null)
                {
                    return PERMISSION_CONFIGURE_ELEMENT;
                }
                throw new InvalidPathException("Path " + m_path + " does not exist and thus cannot be used with the permissions API");
            }

            if (type == ILogicalNameSpace.ELEMENT_TYPE)
            {
                return PERMISSION_CONFIGURE_ELEMENT;
            }
            else if (type == ILogicalNameSpace.FOLDER_TYPE || type == ILogicalNameSpace.COMPLEX_FOLDER_TYPE)
            {
                //because the logical name of elements like brokers represents a folder,
                // we'll try getting the element as a test for elements, and not the type
                // returned from getNameSpaceType

                if (!isHierarchicalPath(m_path))
                    // it's really a folder
                {
                    if (m_path.charAt(m_path.length() - 1) != (IMFDirectories.MF_DIR_SEPARATOR))
                    {
                        m_folderName = m_path + IMFDirectories.MF_DIR_SEPARATOR;
                    }
                    else
                    {
                        m_folderName = m_path;
                    }
                    return PERMISSION_CONFIGURE_FOLDER;
                }
                else
                {
                    return PERMISSION_CONFIGURE_ELEMENT;
                }
            }
            else
            {
                throw new InvalidPathException("Path " + m_path + " is not a folder or an element and this cannot be used to set configure permissions");
            }
        }

        final int checkManagePath()
        throws DirectoryServiceException, ConfigException, StorageException
        {
            // the path has to be a folder, or a container or a component in a container
            int type = m_logicalNameSpace.getNameSpaceType(m_path);
            // if it's a folder, we're done
            if (type != ILogicalNameSpace.FOLDER_TYPE)
            {
                // if it's an element, it has to be a container
                if (type == ILogicalNameSpace.ELEMENT_TYPE)
                {
                    String storageName = m_logicalNameSpace.storageFromLogical(m_path);
                    IDirElement el = getElement(storageName, false);
                    if (!el.getIdentity().getType().equals(IContainerConstants.DS_TYPE))
                    {
                        throw new InvalidPathException("Cannot identify " + m_path + " as a folder, container or component to be used in setting a manage permission");
                    }
                    m_containerName = m_path;
                    return PERMISSION_MANAGE_CONTAINER;
                }
                if (type == ILogicalNameSpace.COMPLEX_FOLDER_TYPE)
                {
                    throw new InvalidPathException("Cannot identify " + m_path + " as a folder, container or component to be used in setting a manage permission");
                }
                if (type == ILogicalNameSpace.DOES_NOT_EXIST_TYPE)
                // look for the component inside the container
                {
                    int colonIndex = m_path.indexOf(":");
                    if (colonIndex == -1)
                    {
                        throw new InvalidPathException("Cannot identify " + m_path + " as a folder, container or component to be used in setting a manage permission");
                    }
                    int equalIndex = m_path.indexOf("=");
                    if (equalIndex == -1)
                    {
                        throw new InvalidPathException("Cannot identify " + m_path + " as a folder, container or component to be used in setting a manage permission");
                    }
                    EntityName pathName = new EntityName(m_path);
                    String parentName = pathName.getParent(); // folder of the container
                    String containerLogicalName = parentName + m_path.substring(parentName.length(), colonIndex);
                    String possibleContainerStorageName;
                    try
                    {
                        possibleContainerStorageName = m_logicalNameSpace.storageFromLogical(containerLogicalName);
                    }
                    catch (Exception ve)
                    {
                        throw new InvalidPathException("Cannot identify " + m_path + " as a folder, container or component to be used in setting a manage permission: " + ve.toString());
                    }

                    IDirElement possibleContainer = getElement(possibleContainerStorageName, false);
                    if (possibleContainer == null || !possibleContainer.getIdentity().getType().equals(IContainerConstants.DS_TYPE))
                    {
                        throw new InvalidPathException("Cannot identify " + m_path + " as a folder, container or component to be used in setting a manage permission");
                    }
                    // possibleContainer is a container
                    m_containerName = containerLogicalName;

                    String componentName = m_path.substring(equalIndex + 1);
                    IAttributeSet containerTopSet = possibleContainer.getAttributes();
                    IAttributeSet componentsSet = (IAttributeSet)containerTopSet.getAttribute(IContainerConstants.COMPONENTS_ATTR);
                    if (componentsSet == null)
                    {
                        throw new InvalidPathException("Cannot identify " + m_path + " as a folder, container or component to be used in setting a manage permission");
                    }
                    IAttributeSet componentSet = (IAttributeSet)componentsSet.getAttribute(componentName);
                    if (componentSet == null)
                    {
                        throw new InvalidPathException("Cannot identify " + m_path + " as a folder, container or component to be used in setting a manage permission");
                    }
                    // otherwise we found the component in the container
                    m_componentName = componentName;
                    return PERMISSION_MANAGE_COMPONENT;
                }
                else
                {
                    throw new InvalidPathException("Cannot identify " + m_path + " as a folder, container or component to be used in setting a manage permission");
                }
            }
            else
            {
                if (m_path.charAt(m_path.length() - 1) != (IMFDirectories.MF_DIR_SEPARATOR))
                {
                    m_folderName = m_path + IMFDirectories.MF_DIR_SEPARATOR;
                }
                else
                {
                    m_folderName = m_path;
                }
                return PERMISSION_MANAGE_FOLDER;
        }
        }

        void setPermissionsCheck(IFrameworkComponentContext context)
        {
            if (context != null && context.getPermissionsManager() != null)
            {
                IPermissionsManager manager = context.getPermissionsManager();
                switch (m_pathType)
                {
                    case PERMISSION_CONFIGURE_FOLDER:
                    case PERMISSION_MANAGE_FOLDER:
                        manager.configurePermissionCheck(context, m_folderName, true, IConfigurePermissionBits.ALLOW_SET_PERMISSIONS);
                        break;
                    case PERMISSION_CONFIGURE_ELEMENT:
                    case PERMISSION_MANAGE_CONTAINER:
                        manager.configurePermissionCheck(context, m_path, true, IConfigurePermissionBits.ALLOW_SET_PERMISSIONS);
                        break;
                    case PERMISSION_MANAGE_COMPONENT:
                        manager.configurePermissionCheck(context, m_containerName, true, IConfigurePermissionBits.ALLOW_SET_PERMISSIONS);

                        break;
                }
            }
        }

        void getInformationCheck(IFrameworkComponentContext context)
        {
            if (context != null && context.getPermissionsManager() != null)
            {
                IPermissionsManager manager = context.getPermissionsManager();
                switch (m_pathType)
                {
                    case PERMISSION_CONFIGURE_FOLDER:
                    case PERMISSION_MANAGE_FOLDER:
                        manager.configurePermissionCheck(context, m_folderName, true, IConfigurePermissionBits.ALLOW_READ);
                        break;
                    case PERMISSION_CONFIGURE_ELEMENT:
                    case PERMISSION_MANAGE_CONTAINER:
                        manager.configurePermissionCheck(context, m_path, true, IConfigurePermissionBits.ALLOW_READ);
                        break;

                    case PERMISSION_MANAGE_COMPONENT:
                        manager.configurePermissionCheck(context, m_containerName, true, IConfigurePermissionBits.ALLOW_READ);
                        break;

                }
            }
        }

        String getStoredPath()
        {
            if (m_folderName != null)
            {
                return m_folderName;
            }
            else
            {
                return m_path;
            }
        }

        void setPermission(IManagementPermission perm, IAttributeSet permissionsSet, IFrameworkComponentContext context)
        throws InvalidManagementPermissionException, ConfigException, DirectoryServiceException
        {
            IPermissionsManager manager = null;
            if (context != null)
            {
                manager = context.getPermissionsManager();
            }
            m_scope = perm.getScope();
            m_principal = perm.getPrincipal();
            m_permissions = perm.getPermissions();
            short principalType = IManagementPermission.UNKNOWN_PRINCIPAL_TYPE;
            checkPermissionScope();
            if (manager != null)
            {
                principalType = manager.getPrincipalType(m_principal);
            }
            else
            {
                principalType = getPrincipalType(m_principal);
            }
            if (principalType == IManagementPermission.UNKNOWN_PRINCIPAL_TYPE)
            {
                throw new InvalidManagementPermissionException(m_principal + " is not a valid principal in the domain's authentication domain");
            }
            //TODO: check that the principal is in the domain wide auth_domain
            IAttributeSet permSet = (IAttributeSet)permissionsSet.getAttribute(m_principal);
            //if the permission we're about to set is exactly what we had before, don't set it
            if (permSet != null)
            {
                String setting = (String)permSet.getAttribute(PERMISSIONS_VALUE_ATTRIBUTE_NAME);
                int setScope = (new Integer(setting.substring(0, setting.indexOf(":")))).intValue();
                int setPermission = (new Integer(setting.substring(setting.indexOf(":") + 1))).intValue();
                if (setScope == m_scope && setPermission == m_permissions)
                {
                    return;
                }
            }
            if (permSet != null)
            {
                permissionsSet.deleteAttribute(m_principal);
            }
            permSet = permissionsSet.createAttributeSet(m_principal);
            permSet.setStringAttribute(PERMISSIONS_VALUE_ATTRIBUTE_NAME, new String(m_scope + ":" + m_permissions));
        }

        void deletePermission(String principal, IAttributeSet permissionsSet)
        throws ConfigException, InvalidManagementPermissionException
        {
            if (principal != null)
            {
                IAttributeSet principalPermSet = (IAttributeSet)permissionsSet.getAttribute(principal);
                if (principalPermSet != null)
                {
                    permissionsSet.deleteAttribute(principal);
                }
                else
                {
                    throw new InvalidManagementPermissionException("Cannot remove permission for " + principal + " on path " + m_path + ": principal permissions does not exist");
                }
            }
        }

        private void checkPermissionScope()
        throws InvalidManagementPermissionException
        {
            switch (m_pathType)
            {
                case PERMISSION_CONFIGURE_FOLDER:
                    checkConfigureFolderScope();
                    break;
                case PERMISSION_CONFIGURE_ELEMENT:
                    checkConfigureElementScope();
                    break;
                case PERMISSION_MANAGE_FOLDER:
                    checkManageFolderScope();
                    break;
                case PERMISSION_MANAGE_CONTAINER:
                    checkManageContainerScope();
                    break;
                case PERMISSION_MANAGE_COMPONENT:
                    checkManageComponentScope();
                    break;
            }
        }

        private void checkConfigureFolderScope()
        throws InvalidManagementPermissionException
        {
            if ((m_scope & IConfigureScopeBits.THIS_CONFIGURATION_OR_FILE_SCOPE) == IConfigureScopeBits.THIS_CONFIGURATION_OR_FILE_SCOPE)
            {
                throw new InvalidManagementPermissionException("Permission scope " + m_scope + " includes incorrect permissions for folder " + m_path);
            }
        }

        private void checkConfigureElementScope()
        throws InvalidManagementPermissionException
        {
            if (m_scope != IConfigureScopeBits.THIS_CONFIGURATION_OR_FILE_SCOPE)
            {
                throw new InvalidManagementPermissionException("Permission scope " + m_scope + " includes incorrect permissions for element " + m_path);
            }
        }

        private void checkManageFolderScope()
        throws InvalidManagementPermissionException
        {
            if (((m_scope & IManageScopeBits.THIS_COMPONENT_SCOPE) == IManageScopeBits.THIS_COMPONENT_SCOPE) || ((m_scope & IManageScopeBits.THIS_CONTAINER_SCOPE) == IManageScopeBits.THIS_CONTAINER_SCOPE))
            {
                throw new InvalidManagementPermissionException("Permission scope " + m_scope + " includes incorrect permissions for runtime folder " + m_path);
            }
        }

        private void checkManageContainerScope()
        throws InvalidManagementPermissionException
        {
            if (((m_scope & IManageScopeBits.ALL_CONTAINERS_SCOPE) == IManageScopeBits.ALL_CONTAINERS_SCOPE) || ((m_scope & IManageScopeBits.ALL_FOLDERS_SCOPE) == IManageScopeBits.ALL_FOLDERS_SCOPE)
                || ((m_scope & IManageScopeBits.THIS_COMPONENT_SCOPE) == IManageScopeBits.THIS_COMPONENT_SCOPE))
            {
                throw new InvalidManagementPermissionException("Permission scope " + m_scope + " includes incorrect scope for managed container " + m_path);
            }
        }

        private void checkManageComponentScope()
        throws InvalidManagementPermissionException
        {
            if (m_scope != IManageScopeBits.THIS_COMPONENT_SCOPE)
            {
                throw new InvalidManagementPermissionException("Permission scope " + m_scope + " includes incorrect scope for component " + m_path);
            }
        }
    }

    // gets used by a DS updating the timestamp element in the backup DS
    private IStorage getSystemStorage()
    {
        return m_systemStorage;
    }

    private void saveBackupStatusAttrs(IDirElement backupStatus)
    throws AttributeSetTypeException, ConfigException, StorageException, DirectoryServiceException
    {
        ((Element)backupStatus).setReadOnly(false);
        IAttributeSet topSet = backupStatus.getAttributes();
        //topSet.setBooleanAttribute(BACKUP_STATUS_IN_PROGRESS_ATTR, new Boolean(m_backupStatus.inProgress));
        topSet.setStringAttribute(BackupStatus.LOCATION, m_backupStatus.getLocation());
        topSet.setLongAttribute(BackupStatus.START_TIME, new Long(m_backupStatus.getStartTime()));
        topSet.setLongAttribute(BackupStatus.COMPLETION_TIME, new Long(m_backupStatus.getCompletionTime()));
        try
        {
        	m_lock.writeLock();
            if (m_usePSE_STORAGE)
            {
                m_storage.startTransaction();
            }
            ((Element)backupStatus).setReadOnly(true);
            m_storage.setElement(BACKUP_STATUS_ELEMENT_ENTITY_NAME, backupStatus);
            if (m_usePSE_STORAGE)
            {
                m_storage.commitTransaction();
            }
        }
        finally
        {
        	m_lock.releaseLock();
        }
    }

    private void readBackupStatusAttrs(IDirElement backupStatus)
    {
        IAttributeSet topSet = backupStatus.getAttributes();
        m_backupStatus.setCompletionTime((Long)topSet.getAttribute(BackupStatus.COMPLETION_TIME));
        m_backupStatus.setStartTime((Long)topSet.getAttribute(BackupStatus.START_TIME));
        m_backupStatus.setLocation((String)topSet.getAttribute(BackupStatus.LOCATION));
    }

    @Override
    public int getNameSpaceType(String path)
    {
        return m_logicalNameSpace.getNameSpaceType(path);
    }

    // we assume this is called inside a transaction
    @Override
    public boolean isPermissionsCheckingEnabled() throws DirectoryServiceException
    {
        IDirElement domainEl = null;
        try
        {
            domainEl = getElement(DOMAIN_ELEMENT_PATH, false, false);
        }
        catch (Exception ex) {} // ok, no domain element
        if (domainEl != null)
        {
            IAttributeSet topset = domainEl.getAttributes();
            Reference authDomainRef = (Reference)topset.getAttribute(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR);
            return (authDomainRef != null && authDomainRef.getElementName() != null &&
                authDomainRef.getElementName().length() > 0);
        }
        return false;
    }


    @Override
    public String getPermissionsPath(String path)
    throws DirectoryServiceException
    {
        try
        {
            String hierarchicalPath = getHierarchicalPath(path);
            if (hierarchicalPath == null)
            {
                return path;
            }
            else
            {
                return hierarchicalPath;
            }
            }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Unable to verify permissions for " + path + ": " + e.toString());
        }
    }
    // Find a parent path in choices for the the given path. Used in ClientTransaction
    // to find hierarchical parent configurations for sub configurations being created
    // or deleted.
    // Find the exact match first, so that ClientTransaction will do the appropriate check for the
    // parent hierarchical configuration, or a match along the path second, which means
    // ClientTransaction will skip the check for the sub element.
    public String getHierarchicalPath(String path, ArrayList choices) throws DirectoryServiceException
    {
        try
        {
            if (path.length() == 0)
            {
                return null;
            }
            if (isPathInChoices(path, choices))
            {
                return path;
            }
            EntityName eName = new EntityName(path);
            String[] nameComponents = eName.getNameComponents();

            //String suffix = null;
            String parentPath = null;
            EntityName parentEName = null;
            for (int length = 1; length <= nameComponents.length; length++)
            {
                parentEName = new EntityName(nameComponents, length);
                parentPath = parentEName.toString();
                if (isPathInChoices(parentPath, choices))
                {
                    return parentPath;
                }
            }

            return null;
        }

        catch (Exception e)
        {
            throw new DirectoryServiceException("Unable to determine hierarchical path for " + path + ": " + e.toString());
        }
    }

    @Override
    public String getHierarchicalPath(String path) throws DirectoryServiceException
    {
        try
        {
            if (path.length() == 0)
            {
                return null;
            }
            EntityName eName = new EntityName(path);
            String[] nameComponents = eName.getNameComponents();

            //String suffix = null;
            String parentPath = null;
            EntityName parentEName = null;
            for (int length = 1; length <= nameComponents.length; length++)
            {
                parentEName = new EntityName(nameComponents, length);
                parentPath = parentEName.toString();
                if (isHierarchicalPath(parentPath))
                {
                    return parentPath;
                }
            }

            return null;
        }
        catch (DirectoryServiceException dirE)
        {
            throw dirE;
        }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Unable to determine hierarchical path for " + path + ": " + e.toString());
        }
    }

    public boolean isHierarchicalConfig(String version, String type)
    {
        if (m_hierarchicalTypes != null)
        {
            String hierarchicalType = (String)m_hierarchicalTypes.get(DSComponent.HIERARCHICAL_TYPES_PATH + version + IMFDirectories.MF_DIR_SEPARATOR + type);
            return (hierarchicalType != null);
        }
        else
        {
            return false;
        }
    }

    public boolean isPathInChoices(String path, ArrayList choices)
    {
        if (choices != null)
        {
            for (int i=0; i<choices.size(); i++)
            {
                if (path.equals((String)choices.get(i)))
                {
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public boolean isHierarchicalPath(String path) throws DirectoryServiceException
    {
        try
    {
        HashMap metaAttrs = getMetaAttributes(path);
            ElementIdentity id = null;
        if (metaAttrs == null)
        {
            return false;
        }
        String toolAttrs = (String)metaAttrs.get("TOOL_ATTRIBUTES");
        if (toolAttrs == null)
            {
                // We want this method to return true for the folder name and the config name with the suffix.
                // For all hierarchical types but complex types, both the folder and the config name with the
                // suffix have TOOL_ATTRIBUTES set. Complex type config names with the suffix do not, so, we look for
                // their type and version through the element identity, which is what is returned by getMetaAttributes for
                // complex configurations. This will not work until there is an element identity for the path, and
                // when these configs are created in a transaction, the element identity is set after the meta
                // attributes, so this test might not work in the middle of the transaction that's creating the element.
                // We're only depending on this method returning true for complex type config names
                // in the case where we're trying to delete the config, as the permission checked is different depending
                // on whether we're deleting the hirarchical config itself, or a sub config. See
                // SharedPermissionChecks.deleteFSElementCheck
                id = (ElementIdentity)metaAttrs.get("_ELEMENT_IDENTITY");
                if (id == null)
                {
                    return false;
                }
                else
                {
                    toolAttrs = "TYPE=" + id.getType() + ";CONFIG_VERSION=" + id.getReleaseVersion();
                }
            }
        int typeIndex = toolAttrs.indexOf("TYPE=");
        if (typeIndex == -1)
        {
            return false;
        }
        String type = toolAttrs.substring(typeIndex + 5);
        if (type == null)
        {
            return false;
        }
        int endOfType = type.indexOf(";");
        if (endOfType != -1)
        {
            type = type.substring(0, endOfType);
        }
        int versionIndex = toolAttrs.indexOf("CONFIG_VERSION=");
        if (versionIndex == -1)
        {
            return false;
        }
        String version = toolAttrs.substring(versionIndex + 15);
        if (version == null)
        {
            return false;
        }
        int endOfVersion = version.indexOf(";");
        if (endOfVersion != -1)
        {
            version = version.substring(0, endOfVersion);
        }
        // find the matching element
            return isHierarchicalConfig(version, type);
    }
        catch (Exception e)
        {
            throw new DirectoryServiceException("Unable to get the type and version from the meta attributes of " + path + ": " + e.toString());
        }
    }

    // we assume this is called from inside a transaction
    private void removePermissions(String path) throws DirectoryServiceException
    {
        if (isPermissionsCheckingEnabled())
        {
            boolean isFolder = getNameSpaceType(path) == ILogicalNameSpace.FOLDER_TYPE;
            String pathToRemove = isFolder ? path + "/" : path;
            removeManagementPermissions(new String[]{pathToRemove}, IManagementPermission.CONFIGURE_TYPE, null, false);
            removeManagementPermissions(new String[]{pathToRemove}, IManagementPermission.MANAGE_TYPE, null, false);
        }
    }

    @Override
    public void setHierarchicalTypes(HashMap map)
    {
        m_hierarchicalTypes = map;
    }

    private void createSuspendNotificationsElement()
    throws DirectoryServiceException
    {
        try
        {
        	IDirElement suspendEl = m_storage.getElement(SUSPEND_NOTIFICATIONS_ENTITY_NAME);
            if (suspendEl != null)
            {
                return;
            }
        	m_lock.writeLock();
            trace(TRACE_STARTUP, "Creating the initial suspend notifications element");

            suspendEl = ElementFactory.createElement(SUSPEND_NOTIFICATIONS_ELEMENT_PATH, "suspend_notifications", MF_CONFING_VERSION);
            m_storage.startTransaction();
            m_storage.setElement(SUSPEND_NOTIFICATIONS_ENTITY_NAME, (IDirElement)suspendEl.doneUpdate());
            m_storage.commitTransaction();

        }
        // The following exceptions mean a bug in the code
        catch (ReadOnlyException e)
        {
            throw new Error(e.toString());
        }
        catch (ConfigException e)
        {
            throw new Error(e.toString());
        }
        catch (StorageException storeE)
        {
        	DirectoryServiceException dirE = new DirectoryServiceException("Unable to create " + SUSPEND_NOTIFICATIONS_ELEMENT_PATH);
        	dirE.initCause(storeE);
        	throw dirE;
        }
        finally
        {
        	m_lock.releaseLock();
        }
    }

    @Override
    public void suspendChangeNotifications(String targetContainerPath, String[] allowTypes)
        throws DirectoryServiceException
    {
    	String containerName;
    	validateOpen();

    	trace(TRACE_CONFIGURATION_NOTIFICATIONS, "suspendChangeNotifications " + targetContainerPath);

        try
        {
            m_lock.writeLock();

            if (targetContainerPath == null)
            {
                throw new DirectoryServiceException("Container path cannot be null in suspendChangeNotifications");
            }
            containerName = targetContainerPath.substring(targetContainerPath.lastIndexOf(IMFDirectories.MF_DIR_SEPARATOR) + 1);
            // create the element if it doesn't exist
            createSuspendNotificationsElement();

            IDirElement suspendEl = m_storage.getElement(SUSPEND_NOTIFICATIONS_ENTITY_NAME);
            ((Element)suspendEl).setReadOnly(false);
            IAttributeSet atts = suspendEl.getAttributes();
            if (atts.getAttribute(containerName) != null)
            {
                atts.deleteAttribute(containerName);
            }

            if (allowTypes == null)
            {
                allowTypes = new String[]{};
            }
            IAttributeList containerList = atts.createAttributeList(containerName);
            for (int i=0; i< allowTypes.length; i++)
            {
                containerList.addStringItem(allowTypes[i]);
            }
            m_storage.startTransaction();
            m_storage.setElement(SUSPEND_NOTIFICATIONS_ENTITY_NAME, suspendEl);
            m_storage.commitTransaction();
            m_suspendNotifications.put(containerName, allowTypes);
        }
        catch (Exception e)
        {
            e.printStackTrace();
            throw new DirectoryServiceException(e.toString());
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

    private void initSuspendNotifications() throws DirectoryServiceException, StorageException
    {
    	m_suspendNotifications = new HashMap();
    	IDirElement suspendEl = m_storage.getElement(SUSPEND_NOTIFICATIONS_ENTITY_NAME);
    	if (suspendEl != null)
    	{
    		IAttributeSet topSet = suspendEl.getAttributes();
    		HashMap attrMap = topSet.getAttributes();
    		Set attrKeys = attrMap.keySet();
    		Iterator keysIT = attrKeys.iterator();
    		while (keysIT.hasNext())
    		{
    			String containerName = (String)keysIT.next();
    			IAttributeList allowAList = (IAttributeList)topSet.getAttribute(containerName);
    			ArrayList allowList = allowAList.getItems();
    			String[] allowArray = new String[allowList.size()];
    			allowList.toArray(allowArray);
    			m_suspendNotifications.put(containerName, allowArray);
    		}
    	}
    }

    @Override
    public boolean areNotificationsSuspended(String containerRuntimeID)
    {

    	String containerName = containerRuntimeID.substring(containerRuntimeID.indexOf(".") + 1);
    	boolean suspended = m_suspendNotifications.get(containerName) != null;
    	 if ((m_traceMask & IComponent.TRACE_VERBOSE) > 0)
        {
            trace(TRACE_CONFIGURATION_NOTIFICATIONS, "DirectoryService.areNotificationsSuspended returning " + suspended + " for container ID " + containerRuntimeID);
        }
    	return suspended;
    }

	@Override
    public void resumeChangeNotifications(String containerPath)
			throws DirectoryServiceException
	{
		String containerName = null;
		validateOpen();
        trace(TRACE_CONFIGURATION_NOTIFICATIONS, "DirectoryService.resumeChangeNotifications " + containerPath);
        try
        {
            m_lock.writeLock();

            if (containerPath == null)
            {
                throw new DirectoryServiceException("Container path cannot be null in resumeChangeNotifications");
            }
            containerName = containerPath.substring(containerPath.indexOf('.') + 1);

            IDirElement suspendEl = m_storage.getElement(SUSPEND_NOTIFICATIONS_ENTITY_NAME);
            if (suspendEl == null)
            {
            	logMessage("Notifications were never suspended for container " + containerPath, Level.WARNING);
            	return;
            }
            ((Element)suspendEl).setReadOnly(false);
            IAttributeSet atts = suspendEl.getAttributes();
            if (atts.getAttribute(containerName) != null)
            {
                atts.deleteAttribute(containerName);
            }
            m_storage.startTransaction();
            m_storage.setElement(SUSPEND_NOTIFICATIONS_ENTITY_NAME, suspendEl);
            m_storage.commitTransaction();
            m_suspendNotifications.remove(containerName);

        }
        catch (Exception e)
        {
            e.printStackTrace();
            throw new DirectoryServiceException(e.toString());
        }
        finally
        {
            m_lock.releaseLock();
        }
	}

	@Override
    public void resumeAllChangeNotifications() throws DirectoryServiceException
	{
		validateOpen();
		trace(TRACE_CONFIGURATION_NOTIFICATIONS, "DirectoryService.resumeAllChangeNotifications ");
		try
		{
			m_lock.writeLock();
			m_storage.startTransaction();
			m_storage.deleteElement(SUSPEND_NOTIFICATIONS_ENTITY_NAME);
			m_storage.commitTransaction();
			m_suspendNotifications = new HashMap();

		}
		catch (Exception e)
		{
			e.printStackTrace();
			throw new DirectoryServiceException(e.toString());
		}

		finally
		{
			m_lock.releaseLock();
		}
	}

	/**
	 * element sorter
	 * adapted from http://www.vogella.de/articles/JavaAlgorithmsQuicksort/article.html
	 */

	static private void quickSortElements(IDirElement[] els, ElementComparator<IDirElement> c)
	{
		// Check for empty or null array
		if (els ==null || els.length==0){
			return;
		}
		quicksort(els, 0, els.length - 1, c);

	}

	static void quicksort (IDirElement[] a, int low, int high, ElementComparator<IDirElement> c)
	{
		int i = low, j = high;
		// Get the pivot element from the middle of the list
		IDirElement pivot = a[low + (high-low)/2];

		// Divide into two lists
		while (i <= j) {
			// If the current value from the left list is smaller then the pivot
			// element then get the next element from the left list
			while (c.compare(a[i], pivot) < 0)
			{
				i++;
			}
			// If the current value from the right list is larger then the pivot
			// element then get the next element from the right list
			while (c.compare(a[j], pivot) > 0)
			{
				j--;
			}

			// If we have found a values in the left list which is larger then
			// the pivot element and if we have found a value in the right list
			// which is smaller then the pivot element then we exchange the
			// values.
			// As we are done we can increase i and j
			if (i <= j) {
				exchange(a, i, j);
				i++;
				j--;
			}
		}
		// Recursion
		if (low < j)
        {
            quicksort(a,low, j, c);
        }
		if (i < high)
        {
            quicksort(a,i, high, c);
        }

	}

	static private void exchange(IDirElement[] a, int i, int j) {
		IDirElement temp = a[i];
		a[i] = a[j];
		a[j] = temp;
	}

	@Override
    public IBlob getFiles(String path, boolean recurse, String extension) throws DirectoryServiceException
	{
	    trace(TRACE_ALL_DS_ACCESS, "getFiles ");
        m_FSInterfaceIsUsed = true;
        try
        {
            m_lock.readLock();
            ArrayList<HashMap> filesToGet = recursiveList(path, false, true, extension);
            String[] fileNames = new String[filesToGet.size()];
            int arrayIndex = 0;
            for (HashMap map : filesToGet)
           {
            // the recursive list should have returned only elements
                fileNames[arrayIndex++] = ((IElementIdentity)map.get(ILogicalNameSpace.ELEMENT_IDENTITY)).getName();
            }
            return getFiles(fileNames);
        }
        finally
        {
            m_lock.releaseLock();
        }
	}

	@Override
    public IBlob getFiles(String[] fileElementNames) throws DirectoryServiceException
	{
	    trace(TRACE_ALL_DS_ACCESS, "getFiles ");
        m_FSInterfaceIsUsed = true;

        try
        {
            m_lock.readLock();
	        return zipFiles(fileElementNames);
        }
        finally
        {
            m_lock.releaseLock();
        }
	}

	public IBlob zipFiles(String[] fileElementNames) throws DirectoryServiceException
	{
	    if (fileElementNames == null || fileElementNames.length == 0)
         {
            return null;  //cannot create a zip with no entries
        }
	    String baseName = generateZipFileName();
	    File zipFile = new File(m_blobCopiesDir, baseName);
	    String zipFileName = null;
	    FileOutputStream zipFileStream = null;
	    ZipOutputStream zipOutStream = null;
	    IBlob zipFileBlob = null;
	    boolean hasEntries = false;
	    try
	    {
	        zipFileName = zipFile.getCanonicalPath();
	        zipFileStream = new FileOutputStream(zipFile);
	        zipOutStream = new ZipOutputStream(zipFileStream);
	        for (final String elementName : fileElementNames )
	        {
	            IBlob blob = getFSBlob(elementName, false, 0);
	            if (blob == null)
                {
                    continue;
                }
	            zipOutStream.putNextEntry(new ZipEntry(elementName));
	            InputStream blobStream = blob.getBlobStream();
	            byte[] bytes = new byte[4096]; // 4096 is an arbitrary reasonable buffer size
	            while (true)
	            {
	                int bytesRead = blobStream.read(bytes);
	                if (bytesRead < 1)
                    {
                        break;
                    }
	                zipOutStream.write(bytes, 0, bytesRead);
	            }
	            blobStream.close();
	            zipOutStream.closeEntry();
	            hasEntries = true;
	        }
	        if (hasEntries)
	        {
	            zipOutStream.close();
	            String blobName = DiskFileDSHandler.getHandlerName() + IMFDirectories.MF_DIR_SEPARATOR + baseName;
	            return getFSBlob(blobName, false, 0);  // this will go through the DiskFileDSHandler
	                                                  // to construct the blob and provide chunking
	        }
	        else    // we cannot close a zip file with no entries, so return null
	        {
	            zipFileStream.close();
	            if (zipFile.exists())
                {
                    zipFile.delete();
                }
	            return null;
	        }
	    }
	    catch (Exception ex)
	    {
	        DirectoryServiceException dirE = new DirectoryServiceException("Unable to save zip file " + zipFileName);
	        dirE.initCause(ex);
	        throw dirE;

	    }
	}

	// same as the version in DSComponent, except is does not check management
	// permissions on the folders
	@Override
    public ArrayList<HashMap> recursiveList(String path, boolean getFolders, boolean getElements, String extension)
    throws DirectoryServiceException
    {
	    trace(TRACE_ALL_DS_ACCESS, "recursiveList " + path);
        m_FSInterfaceIsUsed = true;

        try
        {
            m_lock.readLock();
            ArrayList<HashMap> recursiveResult = new ArrayList<HashMap>();

            HashMap pathAttributes = getMetaAttributes(path);
            if (pathAttributes.get(ILogicalNameSpace.FOLDER_NAME) == null)
            {
                throw new DirectoryServiceException(path + " is not a folder name");
            }
            recursiveResult.add(pathAttributes);

            ArrayList<HashMap> localResult = listFSAll(path, getFolders, true, extension);

            Iterator childrenIT = localResult.iterator();
            while (childrenIT.hasNext())
            {
                HashMap childMap = (HashMap)childrenIT.next();
                String folderName = (String)childMap.get(ILogicalNameSpace.FOLDER_NAME);
                IElementIdentity elementID = (IElementIdentity)childMap.get(ILogicalNameSpace.ELEMENT_IDENTITY);
                if (folderName != null)
                {
                    recursiveResult.addAll(recursiveList(folderName, getFolders, getElements, extension));
                }
                else
                {
                    // it's an element
                    recursiveResult.add(childMap);
                }
            }
            return recursiveResult;
        }
        finally
        {
            m_lock.releaseLock();
        }
    }

	private synchronized String generateZipFileName()
	{
	    if (m_zipFileCounter + 1 == Integer.MAX_VALUE)
        {
            m_zipFileCounter = 0;
        }
	    return "filesZipFile_" + System.currentTimeMillis() + "_" + m_zipFileCounter++;
	}
}
