package com.sonicsw.mf.framework.directory.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.RandomAccessFile;

import com.sonicsw.mf.common.config.AbstractDSHandler;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IChunkedBlobStreamer;
import com.sonicsw.mf.common.config.IDSHandler;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.impl.Blob;
import com.sonicsw.mf.common.config.impl.Element;
import com.sonicsw.mf.common.config.impl.ElementIdentity;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.framework.IContainer;

public class DiskFileDSHandler extends AbstractDSHandler implements IDSHandler, IChunkedBlobStreamer
{
    private static final String DISK_FILE_TYPE = "MF_FILE";
    private static final String DISK_FILE_RELEASE_VERSION = "100";
    
    private static final String SIZE_ATTR = "SIZE";
    private static final String LAST_MODIFIED_TIME_ATTR = "LAST_MODIFIED_TIME";
    private static final String PERMISSIONS_ATTR = "PERMISSIONS";
    private static final String CREATION_TIME_ATTR = "CREATION_TIME";
    private static final String CREATED_BY_ATTR = "CREATED_BY";
    private static final String HANDLER_NAME = "/_MF_blob_copies";
    
    String m_diskLocation;
    
    DiskFileDSHandler(String diskLocation)
    {
        m_diskLocation = diskLocation;
    }
    
    public IDirElement craftNewElement(String logicalName, File zipFile) throws Exception
    {
        String baseName = zipFile.getName();
        IDirElement newElement = new Element(createElementIdentity(logicalName),false);
        IAttributeSet topSet = newElement.getAttributes();
        IAttributeSet systemAttrs = (IAttributeSet)topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);

        if (systemAttrs == null)
        {
            systemAttrs = topSet.createAttributeSet(IBlob.SYSTEM_ATTRIBUTES);
        }
        
        systemAttrs.setBooleanAttribute(IContainer.DO_NOT_CACHE_ATTR, Boolean.TRUE);
        
        topSet.setLongAttribute(LAST_MODIFIED_TIME_ATTR, new Long(zipFile.lastModified()));
        topSet.setLongAttribute(CREATION_TIME_ATTR, new Long(0));
        topSet.setStringAttribute(CREATED_BY_ATTR, "");
        topSet.setStringAttribute(PERMISSIONS_ATTR, "");
        FileInputStream fileStream = new FileInputStream(zipFile);
        topSet.setIntegerAttribute(SIZE_ATTR, new Integer(fileStream.available()));
        fileStream.close();
        
        return newElement;
    }

    
    @Override
    public IBlob getFSBlob(String blobName, boolean forUpdate, int offset) throws DirectoryServiceException
    {
        File diskFile = diskFile(blobName);
        // we don't expect this, since the only way to get here is from a 
        // blob that the DS created
        if (!diskFile.exists())
        {
            return null;
        }
        if (diskFile.isDirectory())
        {
            throw new DirectoryServiceException(blobName + " is not a file");
        }
        IBlob returnBlob = null;
        byte[] fileChunk = null;
        Integer blobTransferState;
        try
        {
            IDirElement blobEl = craftNewElement(blobName, diskFile);
            RandomAccessFile fStream = new RandomAccessFile(diskFile, "r");
            fStream.seek(offset);
            if ( (offset + IBlob.BLOB_CHUNK_SIZE) >= diskFile.length())
            {
                fileChunk = new byte[ (int) (diskFile.length() - offset)];
                blobTransferState = IBlob.END;
            }
            else
            {
                fileChunk = new byte[IBlob.BLOB_CHUNK_SIZE];
                blobTransferState = IBlob.PARTIAL;
            }
            fStream.read(fileChunk);
            fStream.close();
            returnBlob = new Blob(blobEl, fileChunk, this);
            Blob.markBlobState((Element)blobEl, blobTransferState, IBlob.BLOB_TRANSFER_STATE);
            returnBlob.setLogical(true);
            if (blobTransferState.equals(IBlob.END))
            {
                // client is done with the temporary file, so delete it
                diskFile.delete();
            }
        }
        catch (Exception e)
        {
            DirectoryServiceException dirE = new DirectoryServiceException("Could not fetch blob " + blobName);
            dirE.initCause(e);
            throw dirE;
        }
        return returnBlob;
    }
    
    static public String getHandlerName()
    {
        return HANDLER_NAME;
    }
    
    private File diskFile(String logicalName)
    {
        String baseName = logicalName.substring(logicalName.lastIndexOf(IMFDirectories.MF_DIR_SEPARATOR));
        File diskFile = new File(m_diskLocation, baseName);
        return diskFile;
    }
    
    private IElementIdentity createElementIdentity(String logicalName)
    {
        // assumes we only create elements for files
        ElementIdentity id = new ElementIdentity(logicalName, DISK_FILE_TYPE, DISK_FILE_RELEASE_VERSION);
        id.setCreationTimestamp(System.currentTimeMillis());
        return id;
    }

}
