/**
 * Copyright (c) 2009 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. (Confidential Information).  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sonic.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.mf.framework.directory.impl;

import java.io.File;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.StringTokenizer;

import com.sonicsw.mx.config.util.SonicFSException;
import com.sonicsw.mx.config.util.SonicFSFile;
import com.sonicsw.mx.config.util.SonicFSFileSystem;

import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.view.ILogicalNameSpace;



public class ExtendedSonicFSFileSystem extends SonicFSFileSystem
{
    private static String TEMP_FILE_NAME = "_tmp_ExtendedSonicFSFileSystem";

    private File m_tempDir;

    //Use DirectoryService.m_domainDir for temp directory
    public ExtendedSonicFSFileSystem(IDirectoryFileSystemService dfs, String user, File tempDir)
    {
        super(dfs, user);
        m_tempDir = tempDir;
    }

    @Override
    public SonicFSFile getFileDetails(String path0) throws SonicFSException
    {
        String path = getCanonical(path0);

        SonicFSFile file = null;
        try
        {
            file = getDetails(path);
        }
        catch (Exception e)
        {
            //path does not exist
        }

        return file;
    }


    public void copyFiles(String srcPath0, String trgtPath0) throws SonicFSException
    {
        String srcPath = getCanonical(srcPath0);
        String trgtPath = getCanonical(trgtPath0);

        if (trgtPath.equals(srcPath))
        {
            throw new SonicFSException("The source path and the target path are equal - \"" + srcPath + "\"");
        }


        SonicFSFile srcFile = getDetails(srcPath);

        SonicFSFile trgtFile = getFileDetails(trgtPath);
        if (trgtFile != null && trgtFile.isFile())
        {
            deleteFile(trgtFile.getFullName());
            trgtFile = null;
        }

        if (trgtFile != null)
        {
            copyInto(srcFile, trgtFile.getFullName());
        }
        else
        {
            String targetParent =  getParent(trgtPath);
            SonicFSFile targetParentFile = getDetails(targetParent);
            if (targetParentFile.isFile())
            {
                throw new SonicFSException("Cannot copy to \"" + trgtPath + "\" - \"" + targetParent + "\" is not a directory");
            }
            else if (srcFile.isFile())
            {
                copyFile(srcFile.getFullName(), trgtPath);
            }
            else
            {
                createDirectory(trgtPath);
                try
                {
                    ((DirectoryService)m_dfs).copyConfigurePermissions(srcPath, trgtPath, ILogicalNameSpace.FOLDER_TYPE);
                }
                catch(Exception e)
                {
                    throw new SonicFSException("Failed to copy permission(s) from \"" + srcPath + "\" - " + e.getMessage());
                }
                copyContent(srcFile.getFullName(), trgtPath);
            }

        }

    }

    public SonicFSFile[] deepListDetails(String srcPath) throws SonicFSException
    {
        ArrayList resultList = new ArrayList();
        SonicFSFile file = getDetails(srcPath);
        resultList.add(file);
        if (file.isDirectory())
        {
            listDirectoryContent(srcPath, resultList);
        }

        return (SonicFSFile[])resultList.toArray(new SonicFSFile[0]);
    }

    private void copyFile(String srcFile, String trgtPath) throws SonicFSException
    {
        File tempFile = null;
        try
        {
            SonicFSFile targetFile = getFileDetails(trgtPath);
            if (targetFile != null && targetFile.isFile())
            {
                deleteFile(trgtPath);
            }

            tempFile = getFile(srcFile);
            createFile(trgtPath, tempFile);
            try
            {
                ((DirectoryService)m_dfs).copyConfigurePermissions(srcFile, trgtPath, ILogicalNameSpace.ELEMENT_TYPE);
            }
            catch(Exception e)
            {
                throw new SonicFSException("Failed to copy permission(s) from \"" + srcFile + "\" - " + e.getMessage());
            }
        }
        finally
        {
            if (tempFile != null)
            {
                tempFile.delete();
            }
        }
    }

    private void listDirectoryContent(String srcPath, ArrayList resultList) throws SonicFSException
    {
        SonicFSFile[] list = listDetails(srcPath);
        for (int i = 0; i < list.length; i++)
        {
            SonicFSFile currentFile = list[i];
            resultList.add(currentFile);
            if (currentFile.isDirectory())
            {
                listDirectoryContent(currentFile.getFullName(), resultList);
            }
        }
    }

    private void copyContent(String srcPath, String trgtPath) throws SonicFSException
    {
        SonicFSFile[] list = listDetails(srcPath);
        for (int i = 0; i < list.length; i++)
        {
            copyInto(list[i], trgtPath);
        }
    }

    private void copyInto(SonicFSFile srcFile, String hostDir) throws SonicFSException
    {
        String srcName = srcFile.getName();
        String targetPath = createChildName(hostDir, srcName);
        SonicFSFile targetFile = getFileDetails(targetPath);

        if (srcFile.isFile())
        {
            copyFile(srcFile.getFullName(),  targetPath);
        }
        else
        {
            if (targetFile != null)
            {
                if (targetFile.isFile())
                {
                    deleteFile(targetPath);
                }
                else
                {
                    copyContent(srcFile.getFullName(), targetPath);
                    return;
                }
            }

            createDirectory(targetPath);
            try
            {
                ((DirectoryService)m_dfs).copyConfigurePermissions(srcFile.getFullName(), targetPath, ILogicalNameSpace.FOLDER_TYPE);
            }
            catch(Exception e)
            {
                throw new SonicFSException("Failed to copy permission(s) from \"" + srcFile.getFullName() + "\" - " + e.getMessage());
            }
            copyContent(srcFile.getFullName(), targetPath);
        }
    }

    private String createChildName(String parentPath, String childName)
    {
        String[] parentComponents = split(parentPath);
        String[] childComponents = new String[parentComponents.length + 1];
        for (int i = 0; i < parentComponents.length; i++)
        {
            childComponents[i] = parentComponents[i];
        }
        childComponents[parentComponents.length] = childName;
        return combine(childComponents, 0,  childComponents.length);
    }

    public String getParent(String path)
    {
        String[] components = split(path);
        return combine(components, 0, components.length - 1);
    }

    public String getCanonical(String path) throws SonicFSException
    {
        if (!path.startsWith("/"))
        {
            throw new SonicFSException("\"" + path + "\" does not start with /");
        }

        StringBuffer sb = new StringBuffer();

        int pathLength = path.length();
        boolean prevWasSeparator = false;
        for (int i = 0; i < pathLength; i++)
        {
            char c = path.charAt(i);
            if (prevWasSeparator && c == '/')
            {
                continue;
            }

            sb.append(c);

            prevWasSeparator = c == '/';

        }

        String canonical = sb.toString();

        int canonicalLength = canonical.length();

        //Drop the last '/' if exists and is not the root directory
        if (canonicalLength > 1 && canonical.charAt(canonicalLength - 1) == '/')
        {
            return canonical.substring(0, canonicalLength - 1);
        }
        else
        {
            return canonical;
        }


    }

    private static String[] split(String name)
    {
        StringTokenizer st = new StringTokenizer(name, "/");
        String[] res = new String[st.countTokens()];
        int i = 0;
        while (st.hasMoreTokens())
        {
            res[i++] = st.nextToken();
        }

        return res;
    }

    private static String combine(String[] s, int start, int stop)
    {
        StringBuffer sb = new StringBuffer("/");

        for (int i = start; i < stop; i++)
        {
            if (i != start)
            {
                sb.append('/');
            }

            sb.append(s[i]);
        }

        return sb.toString();
    }

    private File getFile(String path) throws SonicFSException
    {
        RandomAccessFile tempDataFile = null;
        File tempFile = null;

        try
        {
            // The get the blob attached to this file
            IBlob blob = m_dfs.getFSBlob(path, false);

            //Can use a fixed temporary name since there is only a single DS modifying transaction at a time
            tempFile = new File(m_tempDir, TEMP_FILE_NAME);
            tempFile.delete();
            tempDataFile = new RandomAccessFile(tempFile, "rw");

            InputStream is = blob.getBlobStream();
            byte[] buffer = new byte[1024];

            while (true)
            {
                int avail = is.read(buffer);

                if (avail == -1)
                {
                    break;
                }

                tempDataFile.write(buffer, 0, avail);
            }
            is.close();

            return tempFile;

        }
        catch(Exception e)
        {
            throw new SonicFSException("Failed to get file \"" + path + "\" - " + e.getMessage());
        }
        finally
        {
            if (tempDataFile != null)
            {
                try
                {
                    tempDataFile.close();
                }
                catch (Exception e)
                {
                }
            }
        }

    }



}