/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.directory.impl;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.impl.DirIdentity;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.DirectoryDoesNotExistException;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.framework.directory.storage.IStorage;
import com.sonicsw.mf.framework.directory.storage.StorageException;

class IDCache
{
    private final static String CACHE_DATA_ATT = "CAHCE_DATA";

    private HashMap m_root;
    private transient IStorage m_storage;
    private ArrayList m_events;

    IDCache(DirectoryService ds) throws DirectoryServiceException
    {
        m_events = null;
        m_root = new HashMap();
        m_storage = ds.getStorage();
        buildDir("/");
    }

    IDCache(IDirElement cacheElement) throws DirectoryServiceException
    {
        m_events = null;
        try
        {
            byte[] cacheData = (byte[])cacheElement.getAttributes().getAttribute(CACHE_DATA_ATT);
            ByteArrayInputStream in = new ByteArrayInputStream(cacheData);
            ObjectInputStream objectIn = new ObjectInputStream(in);
            m_root =  (HashMap)objectIn.readObject();

            m_storage = null;

        }
        catch (Exception e)
        {
            throw new DirectoryServiceException(e.toString());
        }

    }

    void startTransaction()
    {
        m_events = new ArrayList();
    }

    void commitTransaction()
    {
        m_events = null;
    }

    void rollBackTransaction()
    {
        ArrayList events = m_events;
        m_events = null;

        try
        {
            for (int i = events.size() - 1; i >= 0; i--)
            {
                Object event = events.get(i);
                if (event instanceof EntityName)
                {
                    remove((EntityName)event);
                }
                else
                {
                    IIdentity id = (IIdentity)event;
                    EntityName name = new EntityName(id.getName());
                    if (id instanceof IElementIdentity)
                    {
                        addElement(new EntityName(id.getName()), (IElementIdentity)id);
                    }
                    else
                    {
                        addDirectory(new EntityName(id.getName()));
                    }
                }
            }
        }
        catch (Exception e) // Should never happen
        {
            throw new IllegalStateException(e.toString());
        }
    }

    IDirElement wrapAsElement(String elementName)
    {
        try
        {
              IDirElement cacheElement =  ElementFactory.createElement(elementName, "idCache", DirectoryService.MF_CONFING_VERSION);
              IAttributeSet attributes = cacheElement.getAttributes();

              ByteArrayOutputStream out = new ByteArrayOutputStream();
              ObjectOutputStream objectOut = new ObjectOutputStream(out);
              objectOut.writeObject(m_root);
              byte[] bytes = out.toByteArray();
              objectOut.close();

              attributes.setBytesAttribute(CACHE_DATA_ATT, bytes);
              return (IDirElement)cacheElement.doneUpdate();

        }
        catch (Throwable e)
        {
            e.printStackTrace();
            throw new Error(e);
        }
    }
    
    static IDirElement createEmptyForOnlineBackup(String elementName)
    {
        try
        {
            IDirElement cacheElement = ElementFactory.createElement(elementName, "idCache", DirectoryService.MF_CONFING_VERSION);
            IAttributeSet attributes = cacheElement.getAttributes();
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ObjectOutputStream objectOut = new ObjectOutputStream(out);
            objectOut.writeObject(new HashMap());
            byte[] bytes = out.toByteArray();
            objectOut.close();
            attributes.setBytesAttribute(CACHE_DATA_ATT, bytes);
            return (IDirElement)cacheElement.doneUpdate();
        }
        catch (Throwable e)
        {
            e.printStackTrace();
            throw new Error(e);
        }
    }
    
    boolean isEmpty()
    {
        return m_root.isEmpty();
    }

    private void buildDir(String dirName) throws DirectoryServiceException
    {
        try
        {
            IIdentity[] ids = m_storage.listAll(new EntityName(dirName));
            for (int i = 0; i < ids.length; i++)
            {
                // Can happen if there is an incomplete blob element - see SNC00068249
                if (ids[i] == null)
                {
                    continue;
                }

                if (ids[i] instanceof IElementIdentity)
                {
                    addElement(new EntityName(ids[i].getName()), (IElementIdentity)ids[i]);
                }
                else  //Directory identity
                {
                    String subDirName = ids[i].getName();
                    addDirectory(new EntityName(subDirName));
                    buildDir(subDirName);
                }
            }
        }
        catch (ConfigException e)
        {
            throw new Error(e.toString());
        }
        catch (StorageException e)
        {
            throw new Error(e.toString());
        }

    }

    // Throwing exception is fatal since we call this after inserting into the storage and if that didn't fail
    // then inserting into the cache should not fail.
    void addElement(EntityName name, IElementIdentity id) throws DirectoryServiceException
    {
        addEntity(name, id);
    }

    // Throwing exception is fatal since we call this after inserting into the storage and if that didn't fail
    // then inserting into the cache should not fail.
    void addDirectory(EntityName name) throws DirectoryServiceException
    {
        addEntity(name, new HashMap());
    }

    private void addEntity(EntityName name, Object entity) throws DirectoryServiceException
    {
        Object oldEntity = getParentDirectory(name).put(name.getBaseName(), entity);
        if (m_events != null)
        {
            if (oldEntity == null)
            {
                m_events.add(name);
            }
            else if (oldEntity instanceof IElementIdentity)
            {
                m_events.add(oldEntity);
            }
        }
    }

    // An exception is thrown if the parent directory does not exist.
    // We call this after calling storage.delete so exception should normally not be thrown.
    void remove(EntityName name) throws DirectoryServiceException
    {
        Object entity = getParentDirectory(name).remove(name.getBaseName());
        if (m_events == null)
        {
            return;
        }
        if (entity instanceof IElementIdentity)
        {
            m_events.add(entity);
        }
        else
        {
            m_events.add(new DirIdentity(name.getName()));
        }

    }

    // Throws an exception if the parent directory does not exixts.
    // Returns null if there is no EntityName entity in the directory.
    IIdentity get(EntityName name) throws DirectoryServiceException
    {
        if (name.isRoot())
        {
            return new DirIdentity(name.getName());
        }

        Object node = getParentDirectory(name).get(name.getBaseName());
        if (node == null)
        {
            return null;
        }
        else if (node instanceof HashMap)
        {
            return new DirIdentity(name.getName());
        }
        else
        {
            return (IIdentity)node;
        }
    }

    // Throws an exception if the directory does not exist.
    IIdentity[] list(EntityName dirName, boolean directories, boolean elements) throws DirectoryServiceException
    {
        Object dirObject = entityForComponents(m_root, dirName.getNameComponents());
        if (dirObject == null)
        {
            throw new DirectoryDoesNotExistException("Directory '" + dirName.getName() + "' does not exist.");
        }
        if (!(dirObject instanceof HashMap))
        {
            throw new DirectoryServiceException("'" + dirName.getName() + "' is not a directory.");
        }

        Iterator keys = ((HashMap)dirObject).keySet().iterator();

        ArrayList idObjects = new ArrayList();
        while (keys.hasNext())
        {
            String currentKey = (String)keys.next();
            Object node = ((HashMap)dirObject).get(currentKey);
            if (node instanceof HashMap && directories)
            {
                idObjects.add(new DirIdentity(dirName.createKidName(currentKey)));
            }
            else if (node instanceof IElementIdentity && elements)
            {
                idObjects.add(node);
            }

        }

        IIdentity[] ids = new IIdentity[idObjects.size()];
        for (int i = 0; i < ids.length; i++)
        {
            ids[i] = (IIdentity)idObjects.get(i);
        }

        return ids;


    }

    private HashMap getParentDirectory(EntityName name) throws DirectoryServiceException
    {
        String[] parentComponents = name.getParentComponents();
        if (parentComponents == null)
        {
            throw new Error("Tried to get the parent of the root directory.");
        }

        Object lastInPath = entityForComponents(m_root, parentComponents);
        if (lastInPath == null || !(lastInPath instanceof HashMap))
        {
            throw new DirectoryDoesNotExistException("Parent directory of '" + name.getName() + "' does not exist.");
        }

        return (HashMap)lastInPath;
    }

   // Returns null if the last component is not found. Throws an exception if any of the components
   // on the way before the last are not found.
    private Object entityForComponents(HashMap initialDir, String[] components) throws DirectoryServiceException
    {
        if (components.length == 0)
        {
            return initialDir;
        }

        HashMap currentDir = initialDir;
        for (int i = 0; i < components.length; i++)
        {
            Object nextInPath = currentDir.get(components[i]);
            if (i + 1 == components.length)
            {
                return nextInPath;
            }
            if (nextInPath == null || !(nextInPath instanceof HashMap))
            {
                throw new DirectoryDoesNotExistException(components[i] + " not a directory.");
            }
            currentDir = (HashMap)nextInPath;
        }

        // We shouldn't get here
        throw new Error();

    }


}
