package com.sonicsw.mf.framework.directory.impl;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IEnvelope;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.impl.Blob;
import com.sonicsw.mf.common.config.impl.DeltaElement;
import com.sonicsw.mf.common.config.impl.Element;
import com.sonicsw.mf.common.config.impl.ElementIdentity;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.config.impl.EnvelopeElement;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;
import com.sonicsw.mf.framework.directory.storage.PackedDirUtil;

final class ImportManager implements IModificationManager
{

    private final static short INITIALIZATION_PHASE = 1;
    private final static short CREATE_SUPERS_PHASE = 2;
    private final static short CREATE_SUBS_PHASE = 3;

    private DirectoryService m_ds;


    private short m_phase = INITIALIZATION_PHASE;
    private HashMap m_oldIds;
    private HashMap m_importedSubs;
    private HashMap m_importedSupers;
    private HashMap m_importedRegulars;
    private HashGroups m_importedPacked;
    private HashMap m_modificationList;
    private HashMap m_deletionTable;
    private Element[] m_extraDeletionsList;
    private boolean m_noReplace;
    private boolean m_logicalNameSpaceModified;

    ImportManager(DirectoryService ds) throws DirectoryServiceException
    {
        this(ds, IEmptyArray.EMPTY_STRING_ARRAY, false);
    }

    ImportManager(DirectoryService ds, String[] deletionDirectories, boolean noReplace) throws DirectoryServiceException
    {
        m_ds = ds;
        m_noReplace = noReplace; // If true it's an error to replace an element with a new one
        createDeletionTable(deletionDirectories);
        reset();
    }

    void importedElement(Element element) throws DirectoryServiceException, VersionOutofSyncException
    {
        switch (m_phase)
        {
            case INITIALIZATION_PHASE:
                if (element.getIdentity().getName().equals(BackReferenceMgr.getBackRefElementPath()))
                {
                    m_ds.m_backRefMgr.setBackReferencesTypes(element);
                }
                else
                {
                    imported(element);
                }
                break;
            case CREATE_SUPERS_PHASE:
                if (element.getIdentity().getName().equals(BackReferenceMgr.getBackRefElementPath()))
                {
                    return;
                }
                if (element.getSuperElementName() == null)
                {
                    String[] subList = element.getSubclassedList();
                    if (subList.length > 0)
                    {
                        createSuper(element);
                    }
                    else
                    {
                        createRegular(element);
                    }
                    m_ds.m_backRefMgr.elementCreated(element);
                }
                break;
            case CREATE_SUBS_PHASE:
                if (element.getIdentity().getName().equals(BackReferenceMgr.getBackRefElementPath()))
                {
                    return;
                }
                if (element.getSuperElementName() != null)
                {
                    createSub(element);
                    m_ds.m_backRefMgr.elementCreated(element);
                }
                break;
            default:
                throw new Error();
        }         
    }

    private void createSub(Element element) throws DirectoryServiceException, VersionOutofSyncException
    {
        String elementName = element.getIdentity().getName();
        Boolean exists = (Boolean)m_importedSubs.get(elementName);
        m_modificationList.put(element.getIdentity(), exists);

        // Calculate the delta between the super and the sub and subclass the sub
        Element superElement = (Element)m_ds.getElement(element.getSuperElementName(), true);
        superElement.cleanupSuperAttributes(true, true);
        DeltaElement calculatedDelta = (DeltaElement)superElement.createDelta(element);
        calculatedDelta.setIdentity(((ElementIdentity)superElement.getIdentity()).createClone());
        m_ds.subclassElement(calculatedDelta, elementName, null);
    }

    private void createSuper(Element element) throws DirectoryServiceException, VersionOutofSyncException
    {
        String elementName = element.getIdentity().getName();
        Boolean exists = (Boolean)m_importedSupers.get(elementName);
        m_modificationList.put(element.getIdentity(), exists);

        // We have to eliminate the list since it will be created when we'll import the sub-classed dudes.
        element = (Element)element.createWritableClone();
        element.cleanupSuperAttributes(true, false);
        m_ds.setElement((IDirElement)element, null);
    }

    private void createRegular(Element element) throws DirectoryServiceException, VersionOutofSyncException
    {
        String elementName = element.getIdentity().getName();
        EntityName eName = getEntity(elementName);

        // We will create all the packed elements later, in bulk.
        if (PackedDirUtil.underPackedDir(eName))
        {
            ElementPair pair = (ElementPair)m_importedPacked.get(elementName);
            if (pair != null && !pair.m_canceled)
            {
                m_modificationList.put(pair.m_afterImage.getIdentity(), pair.m_beforeImage == null ? Boolean.FALSE : Boolean.TRUE);
            }

            return;
        }

        Boolean exists = (Boolean)m_importedRegulars.get(elementName);
        m_modificationList.put( element.getIdentity(), exists);

        // set a flag in the element if its a blob envelope element;
        // in this case, the first import pass has not deleted the element so
        // we can copy the blob onto the new element in the second phase
        // We mark the element with this flag so the DS will ignore the element
        // in storage when it checks for a pre-existing element when storing an IDirElement.
        // A pre-existing storage element would cause an exception from the DS.
        try
        {
            IAttributeSet topSet = element.getAttributes();
            if (topSet != null)
            {
                IAttributeSet systemAttrs = (IAttributeSet)topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);
                if (systemAttrs != null)
                {
                    Integer blobState = (Integer)systemAttrs.getAttribute(IBlob.LARGE_FILE_STATE);
                    if (blobState != null)
                    {
                        Blob.markBlobState(element, Boolean.TRUE, IBlob.BLOB_ENVELOPE_ELEMENT_IMPORT);
                    }
                }
            }
        }
        catch (Exception configE)
        {
            DirectoryServiceException dirE = new DirectoryServiceException("Unable to set system attributes during import " + configE.toString());
            dirE.setLinkedException(configE);
            throw dirE;
        }
        m_ds.setElement((IDirElement)element, null);
    }

    // Set packed elements efficiently
    HashMap createAllPackedElements() throws DirectoryServiceException
    {
        HashMap groups = m_importedPacked.getGroups();
        Iterator iterator = groups.keySet().iterator();
        HashMap newIdList = new HashMap();
        while (iterator.hasNext())
        {
            String parentDir = (String)iterator.next();
            ArrayList group = (ArrayList)groups.get(parentDir);
            ArrayList newList = new ArrayList();

            for (int i = 0; i < group.size(); i++)
            {
                ElementPair pair = (ElementPair)m_importedPacked.get(group.get(i));
                if (pair.m_afterImage != null && !pair.m_canceled)
                {
                    newList.add(pair.m_afterImage);
                }
            }

            IDirElement[] newElements = new IDirElement[newList.size()];
            newList.toArray(newElements);
            m_ds.setElements(newElements);
            for (int i = 0; i < newElements.length; i++)
            {
                IElementIdentity elementID = newElements[i].getIdentity();
                newIdList.put(elementID.getName(), elementID);
            }
        }
        return newIdList;

    }

    private void importedPacked(String name, Element element) throws DirectoryServiceException
    {
        DirectoryService.initializeElementVersion((ElementIdentity)element.getIdentity());
        ElementPair pair = (ElementPair)m_importedPacked.get(name);
        if (pair == null)
        {
            IDirElement[] elements =  m_ds.getElements(name, false);
            boolean wasPlaced = false;
            for (int i = 0; i < elements.length; i++)
            {
                String elementName = elements[i].getIdentity().getName();
                if (name.equals(elementName))
                {
                    pair = new ElementPair(elements[i], element);
                    wasPlaced = true;
                }
                else if (m_importedPacked.get(elementName) == null)
                {
                    pair = new ElementPair(elements[i], null);
                }

                if (pair != null)
                {
                    m_importedPacked.put(elementName, pair);
                }
            }
            if (!wasPlaced)
            {
                m_importedPacked.put(name, new ElementPair(null, element));
            }
        }
        else
        {
            pair.m_afterImage = element;
        }
    }

    boolean wasLogicalNameSpaceModified()
    {
        return m_logicalNameSpaceModified;
    }

    private void imported(Element element) throws DirectoryServiceException
    {
        String name = element.getIdentity().getName();
        EntityName eName = null;
        try
        {
            eName = getEntity(name);
        }
        catch (IllegalArgumentException e)
        {
            throw new DirectoryServiceException(e.getMessage());
        }

        if (PackedDirUtil.underPackedDir(eName))
        {
            importedPacked(name, element);
            return;
        }

        String canonicalName = eName.getName();
        if (canonicalName.equals(DirectoryService.VIEW_ELEMENT) || canonicalName.equals(DirectoryService.STORAGE_HINTS_ELEMENT))
        {
            m_logicalNameSpaceModified = true;
        }


        boolean exists = false;
        try
        {
           IDirElement oldElement = m_ds.getElement(name, false);
           if (oldElement != null)
           {
               exists = true;
               m_oldIds.put(name, oldElement.getIdentity());
           }
        }
        catch (DirectoryServiceException e) // Directory doesn't exit - not an error, exists remains 'false'
        {
        }

        boolean isSub = element.getSuperElementName() != null;
        boolean isSuper = element.getSubclassedList().length > 0;

        if (isSub)
        {
            m_importedSubs.put(name, new Boolean(exists));
        }
        else if (isSuper)
        {
            m_importedSupers.put(name, new Boolean(exists));
        }
        else
        {
            try
            {
                m_importedRegulars.put(name, new Boolean(exists));
            }
            catch (IllegalArgumentException e)
            {
                throw new DirectoryServiceException(e.getMessage());
            }
        }

    }

    void nextPhase() throws DirectoryServiceException, VersionOutofSyncException
    {
        switch (m_phase)
        {
            case INITIALIZATION_PHASE:
                validateSubsImported();
                cancelRedundantImportOfPackedElements();
                deleteOldElements();
                m_phase = CREATE_SUPERS_PHASE;
                break;
            case CREATE_SUPERS_PHASE:
                m_phase = CREATE_SUBS_PHASE;
                break;
            default: throw new Error(); // Should not happen
        }
    }

    // We cancel imports of packed elements when the new elements are identical to the old elements.
    // It's important to optimize that since there could be a very large number of packed elements and,
    // in particular, when we import external users we don't want to do anything with users that didn't change
    private void cancelRedundantImportOfPackedElements()
    {
        Iterator iterator = m_importedPacked.values().iterator();
        while (iterator.hasNext())
        {
            ElementPair pair = (ElementPair)iterator.next();
            if (pair != null && pair.m_afterImage != null && pair.m_beforeImage != null)
            {
                Element e1 = (Element)pair.m_afterImage;
                Element e2 = (Element)pair.m_beforeImage;
                DeltaElement calculatedDelta = (DeltaElement)e1.createDelta(e2);
                if (calculatedDelta.emptyDelta())
                {
                    pair.m_canceled = true;
                }
            }
        }
    }

    private void validateSubsImported() throws DirectoryServiceException
    {
        // Make sure all subclassed elements of an existing super element are imported
        Iterator iterator = m_importedSupers.keySet().iterator();
        while (iterator.hasNext())
        {
            String superName = (String)iterator.next();
            if (((Boolean)m_importedSupers.get(superName)).booleanValue())
            {
                Element oldSuperElement = (Element)m_ds.getElement(superName, false);
                String[] subList = oldSuperElement.getSubclassedList();
                for (int i = 0; i < subList.length; i++)
                {
                    if (m_importedSubs.get(subList[i]) == null)
                    {
                        throw new DirectoryServiceException("A sub-classed element of '" + superName + "' - '" +
                                                             subList[i] + "' is not imported. If a super element is imported " +
                                                             "then all the sub-classed elements that already exist in storage " +
                                                             "must be imported as well, or deleted or unSubclassed."   );
                    }
                }
            }

        }
     }

     private void createDeletionTable(String[] deletionList) throws DirectoryServiceException
     {
         m_deletionTable = new HashMap();
         for (int i = 0; i < deletionList.length; i++)
         {
             EntityName dirName = null;
             try
             {
                 dirName = getEntity(deletionList[i]);
             }
             catch (IllegalStateException e)
             {
                 throw new DirectoryServiceException(deletionList[i] + " is an illegal directory name.");
             }

             if (!PackedDirUtil.isPackedDir(dirName))
            {
                throw new DirectoryServiceException(deletionList[i] +  " is not a packed directory.");
            }

             HashMap dirList = new HashMap();
             m_deletionTable.put(deletionList[i], dirList);
             IIdentity[] list = m_ds.listElements(deletionList[i]);

             for (int j = 0; j < list.length; j++)
            {
                dirList.put(list[j].getName(), list[j]);
            }
         }
     }

     private void deleteOldElements() throws DirectoryServiceException
     {

        // Remove from the deletion table all the elements that didnt change -
        //  where pair.m_canceled is true. We also remove elements that will be overwritten anyhow
        // when we write the new elements.
        HashMap groups = m_importedPacked.getGroups();
        Iterator iterator = groups.keySet().iterator();
        while (iterator.hasNext())
        {
            String parentDir = (String)iterator.next();
            HashMap deletionGroup = (HashMap)m_deletionTable.get(parentDir);

            // Deletion was not requested for this parentDir
            if (deletionGroup == null)
            {
                continue;
            }

            ArrayList group = (ArrayList)groups.get(parentDir);

            for (int i = 0; i < group.size(); i++)
            {
                String elementName = (String)group.get(i);
                ElementPair pair = (ElementPair)m_importedPacked.get(elementName);
                if (pair.m_beforeImage != null && (pair.m_canceled || pair.m_afterImage != null))
                {
                    deletionGroup.remove(elementName);
                }
            }


        }

        // Delete all the elements left in the deletion table
        Iterator dirIterator = m_deletionTable.values().iterator();
        while (dirIterator.hasNext())
        {
            HashMap dirList = (HashMap)dirIterator.next();
            iterator = dirList.keySet().iterator();
            ArrayList group = new ArrayList();

            while (iterator.hasNext())
            {
                group.add(iterator.next());
            }

            String[] elementNames = new String[group.size()];
            group.toArray(elementNames);
            m_ds.importDeleteElements(elementNames);
        }

        // Create the list for later - for validations and notifications
        m_extraDeletionsList = deletionListFromTable();


        // Delete all the regular elements
        iterator = m_importedRegulars.keySet().iterator();
        while (iterator.hasNext())
        {
            String elementName = (String)iterator.next();
            if (((Boolean)m_importedRegulars.get(elementName)).booleanValue())
            {
                m_ds.m_backRefMgr.elementDeleted(elementName);
                m_ds.importDeleteElement(elementName, null);
            }
        }


        // Delete all existing sub-classed elements
        iterator = m_importedSubs.keySet().iterator();
        while (iterator.hasNext())
        {
            String elementName = (String)iterator.next();
            if (((Boolean)m_importedSubs.get(elementName)).booleanValue())
            {
                m_ds.m_backRefMgr.elementDeleted(elementName);
                m_ds.importDeleteElement(elementName, null);
            }
        }

        // Delete all super elements
        iterator = m_importedSupers.keySet().iterator();
        while (iterator.hasNext())
        {
            String elementName = (String)iterator.next();
            if (((Boolean)m_importedSupers.get(elementName)).booleanValue())
            {
                m_ds.m_backRefMgr.elementDeleted(elementName);
                m_ds.importDeleteElement(elementName, null);
            }
        }

    }


    private Element[] deletionListFromTable()
    {
        ArrayList deletedElementsList = new ArrayList();

        Iterator dirIterator = m_deletionTable.values().iterator();
        while (dirIterator.hasNext())
        {
            HashMap dirList = (HashMap)dirIterator.next();
            Iterator iterator = dirList.values().iterator();

            while (iterator.hasNext())
            {
                deletedElementsList.add(iterator.next());
            }

        }

        Element[] result = new Element[deletedElementsList.size()];
        for (int i = 0; i < deletedElementsList.size(); i++)
        {
            result[i] = (Element)Element.createDeletedElement((IElementIdentity)deletedElementsList.get(i));
        }

        return result;

    }

    @Override
    public void validate() throws DirectoryServiceException
    {

        // First create a modification list for the validator
        ArrayList list = new ArrayList();
        Iterator iterator = m_modificationList.keySet().iterator();
        while (iterator.hasNext())
        {
            IElementIdentity elementID = (IElementIdentity)iterator.next();
            String elementName = elementID.getName();
            boolean exists = ((Boolean)m_modificationList.get(elementID)).booleanValue();

            if (m_noReplace && exists)
            {
                throw new DirectoryServiceException(elementName + " already exists.");
            }

            ElementPair pair = (ElementPair)m_importedPacked.get(elementName);

            IDirElement element = null;
            if (pair == null)
            {
                element = m_ds.getElement(elementName, false);
            }
            else
            {
                if (pair.m_canceled)
                {
                    continue;
                }

                element = pair.m_afterImage;
            }

            // We report the delete if the element existed before the import
            if (exists)
            {
                IElementIdentity oldID = null;
                if (pair == null)
                {
                    oldID = (IElementIdentity)m_oldIds.get(elementID.getName());
                }
                else
                {
                    oldID = pair.m_beforeImage.getIdentity();
                }

                list.add(new ModificationItem((Element)Element.createDeletedElement(oldID), true));
            }

            // report the creation of a new element
            if (pair == null)
            {
                list.add(new ModificationItem(m_ds, (ElementIdentity)element.getIdentity()));
            }
            else
            {
                list.add(new ModificationItem((Element)element));
            }
        }

        // Add deleted elements for the the exclusive import case
        for (int i = 0; i < m_extraDeletionsList.length; i++)
        {
            list.add(new ModificationItem(m_extraDeletionsList[i], true));
        }


        // Now validate
        m_ds.runValidationTriggers(list, true);

    }

    @Override
    public void audit() throws DirectoryServiceException {}

    @Override
    public void doNotify() throws DirectoryServiceException
    {
        if (!m_ds.hasConsumer())
        {
            return;
        }

        Iterator iterator = m_modificationList.keySet().iterator();
        GroupNotification groupNotification = new GroupNotification();
        while (iterator.hasNext())
        {
            IElementIdentity elementID = (IElementIdentity)iterator.next();
            String elementName = elementID.getName();
            boolean exists = ((Boolean)m_modificationList.get(elementID)).booleanValue();
            ElementPair pair = (ElementPair)m_importedPacked.get(elementName);
            String parentName = null;

            IDirElement element = null;
            if (pair == null)
            {
                element = m_ds.getElement(elementName, false);
            }
            else
            {
                if (pair.m_canceled)
                {
                    continue;
                }

                element = pair.m_afterImage;
                parentName = pair.m_parent;
            }

            // We notify the delete if the element existed before the import
            if (exists)
            {
                IElementIdentity oldID = null;
                if (pair == null)
                {
                    oldID = (IElementIdentity)m_oldIds.get(elementID.getName());
                }
                else
                {
                    oldID = pair.m_beforeImage.getIdentity();
                }

                IEnvelope envelope = new EnvelopeElement(Element.createDeletedElement(oldID));
                envelope.setProperty(IEnvelope.DELETED_BUT_RECREATED_LABEL, "true");
                groupNotification.addNotification((Element)envelope, parentName, true);

                // Put the element in an envelope marked REPLACEMENT
                element = new EnvelopeElement(element);
                ((IEnvelope)element).setProperty(IEnvelope.REPLACEMENT_LABEL, "true");
            }

            // Notify the creation of a new element
            groupNotification.addNotification((Element)element, parentName, false);
        }

        // Notify deleted element for the exclusive import case
        for (int i = 0; i < m_extraDeletionsList.length; i++)
        {
            Element element = m_extraDeletionsList[i];
            EntityName eName = getEntity(element.getIdentity().getName());
            String parentName = eName.getParent();

            groupNotification.addNotification(element, parentName, true);
        }

        groupNotification.doNotify();

    }

    @Override
    public void reset()
    {
        m_oldIds = new HashMap();
        m_importedSubs = new HashMap();
        m_importedSupers = new HashMap();
        m_importedRegulars = new HashMap();
        m_importedPacked = new HashGroups();
        m_modificationList = new HashMap();
        m_logicalNameSpaceModified = false;
    }

    private class GroupNotification
    {
        HashMap m_groups;

        GroupNotification()
        {
            m_groups = new HashMap();
        }

        void addNotification(Element element, String parentName, boolean deleted)
        {
            if (parentName == null)
            {
                 if (deleted)
                {
                    m_ds.notifyMods(element, true);
                }
                else
                {
                    m_ds.notifyMods(element);
                }

                return;
            }

            ArrayList group = (ArrayList)m_groups.get(parentName);
            if (group == null)
            {
                group = new ArrayList();
                m_groups.put(parentName, group);
            }
            group.add(new Notification(element, deleted));

        }

        void doNotify()
        {
             Iterator iterator = m_groups.values().iterator();

             while (iterator.hasNext())
             {
                 ArrayList currentGroup = (ArrayList)iterator.next();
                 ArrayList modList = new ArrayList();
                 boolean hasDelete = false;
                 for (int i = 0; i < currentGroup.size(); i++)
                 {
                     Notification notification = (Notification)currentGroup.get(i);
                     notification.m_element.setReadOnly(true);
                     ModificationItem modItem = null;
                     if (notification.m_deleted)
                     {
                         hasDelete = true;
                         modItem = new ModificationItem(notification.m_element, true);
                     }
                    else
                    {
                        modItem = new ModificationItem(notification.m_element);
                    }
                     modList.add(modItem);
                 }
                 m_ds.groupNotifyMods(modList, hasDelete);
             }
        }

        private class Notification
        {
            Notification(Element element, boolean deleted)
            {
                m_element = element;
                m_deleted = deleted;
            }

            Element m_element;
            boolean m_deleted;
        }
    }

    private class ElementPair extends HashMap
    {
        IDirElement m_beforeImage;
        IDirElement m_afterImage;
        boolean m_canceled;
        String m_parent;

        ElementPair(IDirElement beforeImage, IDirElement afterImage)
        {
            m_beforeImage = beforeImage;
            m_afterImage = afterImage;
            m_canceled = false;
            m_parent = null;
        }
    }

    private class HashGroups extends HashMap
    {
        private HashMap m_groups;

        HashGroups()
        {
            super();
            m_groups = new HashMap();
        }

        @Override
        public Object put(Object key, Object value)
        {
            EntityName eName = getEntity((String)key);
            String parentName = eName.getParent();

            ArrayList group = (ArrayList)m_groups.get(parentName);
            if (group == null)
            {
                group = new ArrayList();
                m_groups.put(parentName, group);
            }
            group.add(key);

            ((ElementPair)value).m_parent = parentName;
            return super.put(key, value);
        }


        HashMap getGroups()
        {
            return m_groups;
        }

        @Override
        public Object get(Object key)
        {
            return super.get(key);
        }

        // Canceled is implemented rather than remove() to avoid an expensive search
        // in an ArrayList group to remove the key
        public void cancel(Object key)
        {
             ElementPair pair = (ElementPair)super.get(key);
             if (pair == null)
            {
                return;
            }
             pair.m_canceled = true;
        }

        @Override
        public Object remove(Object key)
        {
            throw new IllegalStateException("Not Supported");
        }

    }

    private static EntityName getEntity(String name)
    {
        EntityName entity = null;
        try{ entity = new EntityName(name); }
        catch(Exception e) { throw new  IllegalArgumentException(name + " is illegal.");}
        return entity;
    }

    @Override
    public void onCreate() throws DirectoryServiceException { }
    @Override
    public void afterCreate() throws DirectoryServiceException { }

    @Override
    public void onUpdate() throws DirectoryServiceException { }
    @Override
    public void afterUpdate() throws DirectoryServiceException { }

    @Override
    public void onDelete() throws DirectoryServiceException { }
    @Override
    public void afterDelete() throws DirectoryServiceException { }
}

