/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.directory.impl;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;

import com.sonicsw.mf.common.IDirectoryAdminNotifications;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.IElementChangeHandler;
import com.sonicsw.mf.common.config.IEnvelope;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.framework.elementversion.IArrayElementNotificationListener;


// Listen to DS modification notifications through the IArrayElementNotificationListener interface
// and report to listeners who registered through the IDirectoryAdminNotifications interface.
class LocalListener implements IDirectoryAdminNotifications, IArrayElementNotificationListener
{
    private HashSet m_globalChangeHandlers;
    private HashMap m_elementChangeHandlers;

    LocalListener()
    {
        m_globalChangeHandlers = new HashSet();
        m_elementChangeHandlers = new HashMap();
    }


    //
    // IDirectoryAdminNotifications implementation
    //

    @Override
    public void registerElementChangeHandler(IElementChangeHandler handler)
    {
        synchronized(m_globalChangeHandlers)
        {
            m_globalChangeHandlers.add(handler);
        }

    }
    @Override
    public void registerElementChangeHandler(String elementName, IElementChangeHandler handler)
    {
        synchronized(m_elementChangeHandlers)
        {
            HashSet handlers = (HashSet)m_elementChangeHandlers.get(elementName);
            if (handlers == null)
            {
                handlers = new HashSet();
            }
            handlers.add(handler);
            m_elementChangeHandlers.put(elementName, handlers);
        }

    }

    @Override
    public void unregisterElementChangeHandler(IElementChangeHandler handler)
    {
        synchronized(m_globalChangeHandlers)
        {
            m_globalChangeHandlers.remove(handler);
        }

    }
    @Override
    public void unregisterElementChangeHandler(String elementName, IElementChangeHandler handler)
    {
        synchronized(m_elementChangeHandlers)
        {
            HashSet handlers = (HashSet)m_elementChangeHandlers.get(elementName);
            if (handlers == null)
            {
                return;
            }
            handlers.remove(handler);

            if (handlers.isEmpty())
            {
                m_elementChangeHandlers.remove(elementName);
            }
            else
            {
                m_elementChangeHandlers.put(elementName, handlers);
            }
        }

    }



    //
    // IArrayElementNotificationListener implementation
    //

    @Override
    public void elementsChanged(IBasicElement[] elements)
    {
        handleElementsChanged(elements);
    }

    @Override
    public void elementsChanged(IDirElement[] elements)
    {
        handleElementsChanged(elements);
    }

    private void handleElementsChanged(IBasicElement[] elements)
    {
        for (int i = 0; i < elements.length; i++)
        {
            boolean reportChange = true;
            boolean replacement = false;
            IBasicElement currentElement = elements[i];
            if (currentElement instanceof IEnvelope)
            {
                 IEnvelope envelope = (IEnvelope)currentElement;
                 reportChange = envelope.getProperty(IEnvelope.DELETED_BUT_RECREATED_LABEL) == null ? true : false;
                 replacement = envelope.getProperty(IEnvelope.REPLACEMENT_LABEL) == null ? false : true;
                 currentElement = (IBasicElement)envelope.getEnvelopedElement();
            }

            if (!reportChange)
            {
                continue;
            }

            IElementChange elementChange = new ElementChange(currentElement, replacement);

            HashSet handlers = null;
            synchronized(m_globalChangeHandlers)
            {
                handlers = (HashSet)m_globalChangeHandlers.clone();
            }

            // get the individual element handlers and add them in
            String elementName = currentElement.getIdentity().getName();
            HashSet elementHandlers = null;
            synchronized(m_elementChangeHandlers)
            {
                elementHandlers = (HashSet)m_elementChangeHandlers.get(elementName);
            }

            if (elementHandlers != null)
            {
                Iterator iterator = elementHandlers.iterator();
                while (iterator.hasNext())
                {
                    handlers.add(iterator.next());
                }
            }

            Iterator iterator = handlers.iterator();
            while (iterator.hasNext())
            {
                IElementChangeHandler handler = (IElementChangeHandler)iterator.next();
                try
                {
                    handler.handleElementChange(elementChange);
                }
                catch (Throwable t) // Should never happen
                {
                    t.printStackTrace();
                }
            }
        }
    }

    private final class ElementChange
    implements IElementChange
    {
        IBasicElement m_element;
        boolean m_replacement;

        ElementChange(IBasicElement element, boolean replacement)
        {
            m_element = element;
            m_replacement = replacement;
        }

        @Override
        public IBasicElement getElement() { return m_element; }

        @Override
        public short getChangeType()
        {
            if (m_element instanceof IDeltaElement)
            {
                return IElementChange.ELEMENT_UPDATED;
            }
            if (((IElement)m_element).isDeleted())
            {
                return IElementChange.ELEMENT_DELETED;
            }
            if (m_replacement)
            {
                return IElementChange.ELEMENT_REPLACED;
            }
            return IElementChange.ELEMENT_ADDED;
        }
    }

}
