/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.framework.directory.impl;

import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IValidationElementChange;
import com.sonicsw.mf.common.config.impl.DeltaElement;
import com.sonicsw.mf.common.config.impl.Element;
import com.sonicsw.mf.common.config.impl.ElementIdentity;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;

final class ModificationItem
implements IValidationElementChange
{
    static final int CREATE = 1;
    static final int DELETE = 2;
    static final int MODIFY = 3;
    static final int DELETE_NO_BEFORE_IMAGE = 4;

    int m_type;
    DeltaElement m_delta;
    Element m_beforeImage;
    Element m_newElement;
    Element m_deletedElement;
    String m_deletedLogicalName;
    IBasicElement m_modification;
    DirectoryService m_ds;
    ElementIdentity m_elementID;
    boolean m_skipValidation = false;

    IBasicElement getModification()
    {
        if (m_modification != null)
        {
            return m_modification;
        }
        else if (m_elementID != null)
        {
            if (m_ds == null)
            {
                throw new IllegalStateException("m_ds should not be null.");
            }

            try
            {
                return m_ds.getElement(m_elementID.getName(), false);
            }
            catch (DirectoryServiceException e) // Should never happen
            {
                e.printStackTrace();
                throw new IllegalStateException(e.toString());
            }
        }
        else
        {
            return null;
        }
    }

    IElementIdentity getModificationID()
    {
        if (m_modification != null)
        {
            return m_modification.getIdentity();
        }
        else
        {
            return m_elementID;
        }

    }

    ModificationItem(DeltaElement delta, Element beforeImage)
    {
        m_type = MODIFY;
        m_modification = m_delta = delta;
        m_beforeImage = beforeImage;
        m_newElement = null;
        m_deletedElement = null;
        m_deletedLogicalName = null;
        m_elementID = null;
        m_ds = null;
    }

    ModificationItem(Element element)
    {
        m_type = CREATE;
        m_delta = null;
        m_beforeImage = null;
        m_modification = m_newElement = element;
        m_deletedElement = null;
        m_deletedLogicalName = null;
        m_elementID = null;
        m_ds = null;

    }

    // We don't want to save the element but rather save its name and fetch from the ds when needed.
    ModificationItem(DirectoryService ds, ElementIdentity elementID)
    {
        m_type = CREATE;
        m_delta = null;
        m_beforeImage = null;
        m_modification = m_newElement = null;
        m_deletedElement = null;
        m_deletedLogicalName = null;
        m_elementID = elementID;
        m_ds = ds;
    }

    ModificationItem(Element beforeImage, Element deletedElement, String deletedLogicalName, boolean skipValidation)
    {
        m_type = DELETE;
        m_delta = null;
        m_beforeImage = beforeImage;
        m_newElement = null;
        m_modification = m_deletedElement = deletedElement;
        m_deletedLogicalName = deletedLogicalName;
        m_elementID = null;
        m_ds = null;
        m_skipValidation = skipValidation;
    }

    ModificationItem(Element deletedElement, boolean noBefore)
    {
        if (!noBefore)
        {
            throw new Error();
        }

        m_type = DELETE_NO_BEFORE_IMAGE;
        m_delta = null;
        m_beforeImage = null;
        m_newElement = null;
        m_modification = m_deletedElement = deletedElement;
        m_deletedLogicalName = null;
        m_elementID = null;
        m_ds = null;
    }

    /********* IValidationElementChange implementation *********/
    @Override
    public IBasicElement getElement()
    {
        return getModification();
    }

    @Override
    public short getChangeType()
    {
        switch (m_type)
        {
            case CREATE: return (short)ELEMENT_ADDED;
            case DELETE: return (short)ELEMENT_DELETED;
            case MODIFY: return (short)ELEMENT_UPDATED;
            case DELETE_NO_BEFORE_IMAGE: return (short)ELEMENT_DELETED;
            default: throw new Error();
        }
    }

    @Override
    public IElement getBeforeImage()
    {
        return (IElement)m_beforeImage;
    }

    @Override
    public boolean skipValidation()
    {
        return m_skipValidation;
    }
}
