package com.sonicsw.mf.framework.directory.impl;

import java.util.LinkedList;

import com.sonicsw.mf.common.dirconfig.BackupStateException;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.InvalidRoleException;
import com.sonicsw.mf.framework.agent.TaskScheduler;
import com.sonicsw.mf.framework.directory.DSComponent;

final class ReadWriteLock
{
    private LinkedList m_waitersQueue = new LinkedList();
    private boolean m_isRestrictedBackupDS; // indicates the DS is an active backup that does not allow updates

    public ReadWriteLock(boolean isRestrictedBackupDS)
    {
        m_isRestrictedBackupDS = isRestrictedBackupDS;
        if (m_isRestrictedBackupDS)
        {
            noWriteLock();
        }
    }

    private int getFirstLockTypeIndx(int lockType) {
        if(m_waitersQueue.isEmpty())
        {
            return Integer.MAX_VALUE;
        }

        for(int index = 0; index< m_waitersQueue.size(); index++)
        {
           Caller caller = (Caller) m_waitersQueue.get(index);
            if(caller.state == lockType)
            {
                return index;
            }
        }
        return Integer.MAX_VALUE;
    }

    private int getIndex(Thread t) {

        if(m_waitersQueue.isEmpty())
        {
            return -1;
        }
        for(int index = 0 ; index < m_waitersQueue.size(); index++)
        {
            Caller caller = (Caller) m_waitersQueue.get(index);
            if (caller.t == t)
            {
                return index;
            }
        }
        return -1;
    }

    public synchronized void readLock() {
        Caller caller;
        Thread me = Thread.currentThread();
        int index = getIndex(me);
        if (index == -1) {
            caller = new Caller(me, Caller.READ);
            m_waitersQueue.addLast(caller);
        }
        else
        {
            caller = (Caller) m_waitersQueue.get(index);
        }

        while (getIndex(me) > getFirstLockTypeIndx(Caller.WRITE)) {
            try {
                wait();
            } catch (Exception e) {e.printStackTrace();}
        }
        caller.nAcquires++;
    }

       public synchronized boolean noWriteLock() {
                // We already have a NO_WRITE lock
                if (getFirstLockTypeIndx(Caller.NO_WRITE) != Integer.MAX_VALUE)
                {
                    return false;
                }

                Caller caller;
                Thread me = Thread.currentThread();
                int noWriteIndex = 0;
                if (getIndex(me) == -1)
                {
                     caller = new Caller(null, Caller.NO_WRITE);
                     m_waitersQueue.addLast(caller);
                     noWriteIndex = m_waitersQueue.size() -1;
                }
                else
                {
                    throw new Error("Cannot call noWriteLock while holding a lock");
                }

                while (noWriteIndex > getFirstLockTypeIndx(Caller.WRITE)) {
                        try {
                                wait();
                        } catch (Exception e) {e.printStackTrace();}
                }
                caller.nAcquires++;
                return true;
        }


        public synchronized void removeNoWriteLock()
        {
            int noWriteCallerIndex = getFirstLockTypeIndx(Caller.NO_WRITE);
            if (noWriteCallerIndex == Integer.MAX_VALUE)
            {
                throw new Error("No NO_WRITE lock");
            }
            m_waitersQueue.remove(noWriteCallerIndex);
            notifyAll();
        }

        public synchronized boolean hasNoWriteLock()
        {
            return getFirstLockTypeIndx(Caller.NO_WRITE) != Integer.MAX_VALUE;
        }

    public synchronized void writeLock() throws DirectoryServiceException {

        validateAdministratorRole();

        // a WRITE lock is not allowed when there there is a NO_WRITE lock
        if (getFirstLockTypeIndx(Caller.NO_WRITE) < Integer.MAX_VALUE)
        {
            if (m_isRestrictedBackupDS)
            {
                throw new com.sonicsw.mf.common.dirconfig.UpdateDisallowedOnFailoverException();
            }

            throw new BackupStateException("The BACKUP Directory Service is set to Failover Read-only - cannot be modified.");
        }

        Caller caller;
        Thread me = Thread.currentThread();
        int index = getIndex(me);
        if (index == -1) {
            caller = new Caller(me, Caller.WRITE);
            m_waitersQueue.addLast(caller);
        }
        else {
            caller = (Caller) m_waitersQueue.get(index);
            if (caller.state == Caller.READ)
            {
                throw new Error("Lock upgrade is invalid");
            }
            caller.state = Caller.WRITE;
        }
        while (getIndex(me) != 0) {
            try {
                wait();
            } catch (Exception e) {e.printStackTrace();}
        }
        caller.nAcquires++;
    }

    public synchronized void releaseLock() {
        Caller caller;
        Thread me = Thread.currentThread();
        int index;
        index = getIndex(me);

                // No lock to release by this thread
                if (index == -1)
                {
                    return;
                }

        if (index  > getFirstLockTypeIndx(Caller.WRITE))
        {
            throw new Error("Conflicting WRITE lock");
        }
        caller = (Caller) m_waitersQueue.get(index);
        caller.nAcquires--;
        if (caller.nAcquires == 0) {
            m_waitersQueue.remove(index);
            notifyAll();
        }
    }

   //////Wrapper for the caller thread
   //////////////////////////////////////////


    class Caller {
        static final int READ = 0;
        static final int WRITE = 1;
        static final int NO_WRITE = 2;
        Thread t;
        int state;
        int nAcquires;

        Caller(Thread t, int state) {
            this.t = t;
            this.state = state;
            nAcquires = 0;
        }
    }


    private void validateAdministratorRole() throws InvalidRoleException
    {
        String role = TaskScheduler.getCurrentRole();
        if (role != null && role.equals(DSComponent.READER_ROLE))
        {
            throw new InvalidRoleException("Request rejected: unauthorized request.");
        }
    }

   /////
    //////////////////////////////////////////////////////////////

    ///UnitTest

    public static class Test{

    DSResource resource = new DSResource();

        static class DSResource{

            ReadWriteLock lock = new ReadWriteLock(false);

            public void read(String reader)
            {
                try{
                    lock.readLock();
                    System.out.println("\t\t" + reader + "reading");
                    try{ Thread.currentThread().sleep(300);}
                    catch(InterruptedException ex){}
                    System.out.println("\t\t" + reader + "done");
                }
                finally{
                    lock.releaseLock();
                }

            }

            public void write(String writer) throws Exception
            {
                try{
                    lock.writeLock();
                    System.out.println("\t\t" + writer + "writing");
                    try{ Thread.currentThread().sleep(1000);}
                    catch(InterruptedException ex){}
                    System.out.println("\t\t" + writer + "done");
                }
                finally{
                    lock.releaseLock();
                }
            }
       }//end of DSResource


        class Reader extends Thread
        {
            private String name;
            Reader(String name){ this.name = name;}
            @Override
            public void run(){
                System.out.println("Starting " + name);
                resource.read(name);
                System.out.println("Stopping " + name);
            }
        }//end of Reader

        class Writer extends Thread
        {
            private String name;
            Writer(String name){ this.name = name;}
            @Override
            public void run(){
                System.out.println("Starting " + name);
                try
                {
                    resource.write(name);
                }
                catch (Exception e)
                {
                    e.printStackTrace();
                    throw new Error(e.toString());
                }
                System.out.println("Stopping " + name);
            }
        }//end of Writer



    public Test()
    {
        new Reader ("r/0").start();
        new Writer("w/0").start();
        new Reader("r/1").start();
        new Writer("w/1").start();
        new Writer("w/2").start();
        new Reader("r/2").start();
        new Reader("r/3").start();
        new Reader("r/4").start();
        new Reader("r/5").start();
        new Reader("r/6").start();
        new Writer("w/3").start();
    }

    static public void main( String[] args)
    {
        Test t = new Test();
    }

   }//end of Test class
}//end of ReadWriteLock class


