package com.sonicsw.mf.framework.directory.impl;
import java.io.File;
import java.io.FileDescriptor;
import java.io.RandomAccessFile;

import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.view.impl.LogicalNameSpace;
import com.sonicsw.mf.framework.directory.IDirectoryService;
import com.sonicsw.mf.framework.directory.storage.IStorage;
import com.sonicsw.mf.framework.directory.storage.StorageException;

// The TransactionManager is NOT thread safe. The DS locking mechanism guaratees that a single thread has
// has a transaction at any time. The join mechanism of the TransactionManager guaratees that that thread
// has only one transaction.
public final class TransactionManager
{
    public static boolean _for_testing_only_error_before_commit = false;
    public static boolean _for_testing_only_error_after_commit = false;

    public final static String TRANSACTION_FILE = "tr"; //not used for PSEStorage
    private final static int COMMIT_NOTE = 2;  // not used for PSEStorage
    private final static int ROLLBACK_NOTE = 1;  // not used for PSEStorage

    private File m_dsLocation;
    private IStorage[] m_storages;
    private int m_numParticipants = 0;
    private boolean m_doSync;  			// not used for PSEStorage
    private RandomAccessFile m_trFile;	// not used for PSEStorage
    private FileDescriptor m_trFileDesc;	// not used for PSEStorage
    private IModificationManager m_modManager;
    private IDCache m_idCache;
    private LogicalNameSpace m_logicalNameSpace;
    private String m_dsStorageType;

    TransactionManager(File dsLocation, IStorage[] storages, boolean doSync, IModificationManager modManager,
    		String dsStorageType)
        throws DirectoryServiceException
    {
        m_storages = storages;
        m_dsStorageType = dsStorageType;
        m_doSync = doSync;
        m_dsLocation = dsLocation;
        m_modManager = modManager;
       
        if (m_dsStorageType.equals(IDirectoryService.FS_STORAGE))
        {
        	createCommitFile(dsLocation);
        	if (hasCommitNote())
            {
                doCommit(true);
            }
            else
            {
                doRollback(true);
            }
        }
    }

    void setLogicalNameSpace(LogicalNameSpace nameSpace)
    {
        m_logicalNameSpace = nameSpace;
    }

    void setCache(IDCache idCache)
    {
        m_idCache = idCache;
    }

    // Join the transaction (starts a new one if was not started yet)
    void join() throws DirectoryServiceException
    {
        try
        {
             if (m_numParticipants++ == 0)
             {
                 m_idCache.startTransaction();
                 for (int i = 0; i < m_storages.length; i++)
                {
                    m_storages[i].startTransaction();
                }
             }
        }
        catch (StorageException e)
        {
            throw new DirectoryServiceException(e.toString(), e);
        }
    }

    void leave(boolean ok) throws DirectoryServiceException
    {
        leave(ok, null, true);
    }

    void leave(boolean ok, boolean fireTriggers) throws DirectoryServiceException
    {
        leave(ok, null, fireTriggers);
    }

    // Leaves the transaction.
    // The last to leave determines the success or failure of the transaction.
    void leave(boolean ok, IModificationManager modManager0, boolean fireTriggers)
    throws DirectoryServiceException
    {
        IModificationManager modManager = (modManager0 != null) ? modManager0 : m_modManager;

        if (m_numParticipants == 0)
        {
            throw new Error();
        }

        if (m_numParticipants == 1)
        {
            DirectoryServiceException exception = null;

            if (ok)
            {
                try
                {
                    if (fireTriggers)
                    {
                        modManager.onDelete();
                        modManager.onUpdate();
                        modManager.onCreate();
                        modManager.validate();
                    }
                    
                    if (_for_testing_only_error_before_commit == true && m_dsStorageType.equals(IDirectoryService.FS_STORAGE))
                    {
                        throw new Error("_for_testing_only_error_before_commit = true");
                    }
                    if (m_dsStorageType.equals(IDirectoryService.FS_STORAGE))
                    {
                        writeCommitNote();
                    }
                    if (_for_testing_only_error_after_commit == true && m_dsStorageType.equals(IDirectoryService.FS_STORAGE))
                    {
                        throw new Error("_for_testing_only_error_after_commit = true");
                    }
                }
                catch (DirectoryServiceException e)
                {
                    exception = e;
                    ok = false;
                }
                finally
                {
                    m_numParticipants = 0;
                }

                if (ok)
                {
                    modManager.audit();
                    if (m_logicalNameSpace != null)
                    {
                        m_logicalNameSpace.storeUpdates();
                    }

                    doCommit(false);

                    if (fireTriggers)
                    {
                        modManager.afterCreate();
                        modManager.afterUpdate();
                        modManager.afterDelete();
                    }

                    if (m_logicalNameSpace != null)
                    {
                        m_logicalNameSpace.doNotify();
                    }
                    modManager.doNotify();
                }
            }

            if (!ok)
            {
                try
                {
                    modManager.reset();
                    doRollback(false);
                    if (m_logicalNameSpace != null)
                    {
                        m_logicalNameSpace.reset();
                    }
                    if (exception != null)
                    {
                        throw exception;
                    }
                }
                finally
                {
                    m_numParticipants = 0;
                }
            }
        }
        else
        {
            m_numParticipants--;
        }
    }

    // Called to after online backup is done
   void openFiles() throws DirectoryServiceException
    {
	    if (m_dsStorageType.equals(IDirectoryService.FS_STORAGE))
        {
            createCommitFile(m_dsLocation);
        }
    } 

    // Called to allow online backup
    void closeFiles() throws DirectoryServiceException
    {
        if (m_dsStorageType.equals(IDirectoryService.FS_STORAGE))
        {
            closeCommitFile();
        }
    } 

    void close() throws DirectoryServiceException
    {
        if (m_dsStorageType.equals(IDirectoryService.FS_STORAGE))
        {
            closeCommitFile();
        }
        try
        {
             for (int i = 0; i < m_storages.length; i++)
            {
                m_storages[i].closeTransactionManager();
            }
        }
        catch (StorageException e)
        {
            throw new DirectoryServiceException(e.toString(), e);
        } 
    }

    private void doRollback(boolean recovery) throws DirectoryServiceException
    {
        if (!recovery)
        {
            m_idCache.rollBackTransaction();
        }

        try
        {
             for (int i = 0; i < m_storages.length; i++)
            {
                m_storages[i].rollbackTransaction();
            }
        }
        catch (StorageException e)
        {
            throw new DirectoryServiceException(e.toString(), e);
        }

    }

    private void doCommit(boolean recovery)  throws DirectoryServiceException
    {
    	if (_for_testing_only_error_before_commit == true && 
    			m_dsStorageType.equals(IDirectoryService.PSE_STORAGE))
        {
            throw new Error("_for_testing_only_error_before_commit = true");
        }
        if (!recovery)
        {
            m_idCache.commitTransaction();
        }
        try
        {
            for (int i = 0; i < m_storages.length; i++)
            {
                m_storages[i].commitTransaction();
            }
        }
        catch (StorageException e)
        {
            throw new DirectoryServiceException(e.toString(), e);
        }
        if (_for_testing_only_error_after_commit == true &&
        		m_dsStorageType.equals(IDirectoryService.PSE_STORAGE))
        {
            throw new Error("_for_testing_only_error_after_commit = true");
        }
        if  (m_dsStorageType.equals(IDirectoryService.FS_STORAGE))
        {
            writeRollbackNote();
        }
    }

    // used only for FS_STORAGE
    private void closeCommitFile() throws DirectoryServiceException
    {
        try
        {
             m_trFile.close();
        }
        catch (Exception e)
        {
             throw new DirectoryServiceException(e.toString(), e);
        }
    }
    // Used only for FS_STORAGE
    private void createCommitFile(File dsLocation) throws DirectoryServiceException
    {
        try
        {
             File trFilePath = new File(dsLocation, TRANSACTION_FILE);
             boolean fileExists = trFilePath.exists();
             m_trFile = new RandomAccessFile(trFilePath, "rw");
             m_trFileDesc = m_trFile.getFD();

             if (fileExists)
             {
                 m_trFile.seek(0);
                 return;
             }
            else
            {
                writeRollbackNote();
            }
        }
        catch (Exception e)
        {
             throw new DirectoryServiceException(e.toString(), e);
        }
    }

    // used only for FS_STORAGE
    private void writeCommitNote() throws DirectoryServiceException
    {
        writeNote(COMMIT_NOTE);
    }

    // used only for FS_STORAGE
    private void writeRollbackNote() throws DirectoryServiceException
    {
        writeNote(ROLLBACK_NOTE);
    }

    // used only for FS_STORAGE
    private void writeNote(int note) throws DirectoryServiceException
    {
        try
        {
             m_trFile.writeInt(note);
             if (m_doSync)
            {
                m_trFileDesc.sync();
            }
             m_trFile.seek(0);
        }
        catch (Exception e)
        {
             throw new DirectoryServiceException(e.toString(), e);
        }
    }

    // used only  for FS_STORAGE
    private boolean hasCommitNote() throws DirectoryServiceException
    {
        try
        {
             boolean hasCommit = m_trFile.readInt() == COMMIT_NOTE;
             m_trFile.seek(0);
             return hasCommit;
        }
        catch (Exception e)
        {
             throw new DirectoryServiceException(e.toString(), e);
        }
    } 

}
