package com.sonicsw.mf.framework.directory.impl;

import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IPostCommitTrigger;
import com.sonicsw.mf.common.config.ITriggerDSContext;
import com.sonicsw.mf.common.config.IValidationConstants;
import com.sonicsw.mf.common.config.IValidationDSContext;
import com.sonicsw.mf.common.config.IValidationElementChange;
import com.sonicsw.mf.common.config.IValidator;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceTriggerException;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceValidationException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.directory.IDebuggingMasks;


final class TriggerValidator implements IValidationConstants, IDebuggingMasks
{

    private static final String CONFIG_VERSION_101 = "101";
    private static final String CONFIG_VERSION_100 = "100";
    private static final String CONFIG_VERSION_99 = "99";

    private HashMap m_validationMethodsTable;
    private DirectoryService m_ds;
    private IValidator m_badValidator = null;

    TriggerValidator(DirectoryService ds)
    {
        m_ds = ds;
        m_validationMethodsTable = new HashMap();
        PluginInfo[] pluginList = getAllPlugins();
        for (int i = 0; i < pluginList.length; i++)
        {
            IValidator validator = null;
            try
            {
                validator = createValidatorPlugin(pluginList[i].m_className, pluginList[i].m_classpath);
                // Make sure the plugin validates the correct version
                String pluginReleaseVersion = validator.getReleaseVersion();
                if (pluginReleaseVersion == null || pluginReleaseVersion.length() == 0)
                {
                    throw new Exception("The validation plugin version is missing.");
                }
                if (!pluginReleaseVersion.equals(pluginList[i].m_releaseVersion))
                {
                    throw new Exception("Plugin version mismatch - " + pluginReleaseVersion);
                }
            }
            catch (Exception e)
            {
            	String upgradeFlag = System.getProperty("MQ_UPGRADE");
            	String loadingValidators = System.getProperty("loading_validators");
            	String trimmedClasspath = null;
            	if (pluginList[i].m_classpath != null && pluginList[i].m_classpath.length() > 11)
                 {
                    trimmedClasspath = pluginList[i].m_classpath.substring(11); // trim the sonicfs:///
                }
                 if (upgradeFlag == null || !upgradeFlag.equals("true"))
                 {
                	 if (loadingValidators == null || trimmedClasspath == null || loadingValidators.indexOf(trimmedClasspath) == -1)
                    {
                        m_ds.logMessage("Could not load validator " + pluginList[i].m_className + " for release version " + pluginList[i].m_releaseVersion + ", trace follows...", e, Level.WARNING);
                    }
                 }
                 if (m_badValidator == null)
                 {
                     m_badValidator = new BadValidator();
                     m_validationMethodsTable.put("_MF_MISSING_VALIDATOR", m_badValidator);
                 }
                 continue;
            }

            String[] elementTypes = validator.getElementTypesValidated();
            if (elementTypes == null)
            {
                elementTypes = IEmptyArray.EMPTY_STRING_ARRAY;
            }

            associateTypesWithValidator(elementTypes, pluginList[i].m_releaseVersion, validator);

            // CONFIG_VERSION_101 triggers can handle CONFIG_VERSION_99 and CONFIG_VERSION_100 elements as well
            if (pluginList[i].m_releaseVersion.equals(CONFIG_VERSION_101))
            {
                associateTypesWithValidator(elementTypes, CONFIG_VERSION_100, validator);
                associateTypesWithValidator(elementTypes, CONFIG_VERSION_99, validator);
            }
        }

    }

    private PluginInfo pluginFromElement(IElement pluginElement)
    {
        String validationClass = "";
        String releaseVersion = "";

        try
        {
            IAttributeSet pluginAttributes = pluginElement.getAttributes();
            validationClass = (String)pluginAttributes.getAttribute(VALIDATION_CLASS);
            if (validationClass == null)
            {
                return null;
            }

            releaseVersion = (String)pluginAttributes.getAttribute(RELEASE_VERSION);
            if (releaseVersion == null)
            {
                m_ds.logMessage("Element " + pluginElement.getIdentity().getName() + " does not have a release version attribute", Level.WARNING);
                releaseVersion = "?";
            }

            String classpath = DirectoryService.listToClasspath((IAttributeList)pluginAttributes.getAttribute(VALIDATOR_ARCHIVES));

            return new PluginInfo(validationClass, classpath.length() > 0 ? classpath : null, releaseVersion);
        }
        catch (Exception e)
        {
        	String upgradeFlag = System.getProperty("MQ_UPGRADE");
            if (upgradeFlag == null || !upgradeFlag.equals("true"))
            {
                m_ds.logMessage("Could not load validator " + validationClass + " for release version " + releaseVersion + ", trace follows...", e, Level.WARNING);
            }
            return null;
        }
    }

    private PluginInfo[] getAllPlugins()
    {
        IElementIdentity[] libraryElements = null;
        try
        {
            libraryElements = m_ds.listElements(MF_LIBRARY_DIR);
        }
        catch (DirectoryServiceException e) // No /_MFLibrary directory
        {
            return new PluginInfo[0];
        }

        ArrayList list = new ArrayList();
        for (int i = 0; i < libraryElements.length; i++)
        {
            IElement pluginElement = null;
            try
            {
                pluginElement = m_ds.getElement(libraryElements[i].getName(), false);
            }
            catch (DirectoryServiceException e)
            {
                throw new IllegalStateException(e.toString()); // Should never happen
            }

            if (!libraryElements[i].getType().equals(MF_VALIDATOR_TYPE))
            {
                continue;
            }

            PluginInfo pluginInfo = pluginFromElement(pluginElement);
            if (pluginInfo == null)
            {
                continue;
            }

            list.add(pluginInfo);
        }

        PluginInfo[] pluginList = new PluginInfo[list.size()];
        list.toArray(pluginList);
        return pluginList;
    }

    // Associate between the {elementType,releaseVersion} pair and the validating method
    private void associateTypesWithValidator(String[] elementTypes,  String releaseVersion, IValidator validator)
    {
        for (int j = 0; j < elementTypes.length; j++)
        {
        	ArrayList typeValidatorList = (ArrayList)m_validationMethodsTable.get(elementTypes[j] + releaseVersion);
        	if (typeValidatorList == null)
            {
                typeValidatorList = new ArrayList();
            }
        	typeValidatorList.add(validator);
            m_validationMethodsTable.put(elementTypes[j] + releaseVersion, typeValidatorList);
            m_ds.trace(TRACE_VALIDATION_TRIGGERS, "Validate: " + elementTypes[j] + "  version: " + releaseVersion +
                " by: " + validator.getClass().getName());

        }
    }

    void runOnCreateTriggers(ArrayList list) throws DirectoryServiceException
    {
     // Sort by validators
        HashMap validationGroups = new HashMap();
        HashMap validators = new HashMap();
        for (int i = 0; i < list.size(); i++)
        {
            ModificationItem modItem = (ModificationItem)list.get(i);
            if (modItem.getChangeType() != IElementChange.ELEMENT_ADDED)
            {
                continue;
            }

            ArrayList modValidators = getModValidators(modItem, false);
            
            if (modValidators.isEmpty())
            {
                continue;
            }
            prepareCreateTriggers(modItem, modValidators, validators, validationGroups);           
        }

        // Now run the trigger for each component
        Iterator iterator = validationGroups.keySet().iterator();

        while (iterator.hasNext())
        {
            String validatorID = (String)iterator.next();
            ArrayList vGroup = (ArrayList)validationGroups.get(validatorID);
            IValidator validator = (IValidator)validators.get(validatorID);

            IDirElement[] newImages = new IDirElement[vGroup.size()];
            vGroup.toArray(newImages);
            try
            {
                validator.onCreate(new TriggerDSContext(m_ds) , newImages);
            }
            catch (Throwable t)
            {
                throw new DirectoryServiceTriggerException(validatorID, t);
            }
        }
    }

    void runAfterCreateTriggers(ArrayList list) throws DirectoryServiceException
    {
        // Sort by validators
        HashMap validationGroups = new HashMap();
        HashMap validators = new HashMap();
        for (int i = 0; i < list.size(); i++)
        {
            ModificationItem modItem = (ModificationItem)list.get(i);
            if (modItem.getChangeType() != IElementChange.ELEMENT_ADDED)
            {
                continue;
            }

            ArrayList modValidators = getModValidators(modItem, true);
            
            if (modValidators.isEmpty())
            {
                continue;
            }
            prepareCreateTriggers(modItem, modValidators, validators, validationGroups);           
        }

        // Now run the trigger for each component
        Iterator iterator = validationGroups.keySet().iterator();

        while (iterator.hasNext())
        {
            String validatorID = (String)iterator.next();
            ArrayList vGroup = (ArrayList)validationGroups.get(validatorID);
            IPostCommitTrigger validator = (IPostCommitTrigger)validators.get(validatorID);

            IDirElement[] newImages = new IDirElement[vGroup.size()];
            vGroup.toArray(newImages);
            try
            {
                validator.afterCreate(new TriggerDSContext(m_ds) , newImages);
            }
            catch (Throwable t)
            {
                throw new DirectoryServiceTriggerException(validatorID, t);
            }
        }
    }
    
    void runOnUpdateTriggers(ArrayList list) throws DirectoryServiceException
    {
     // Sort by validators
        HashMap validationGroups = new HashMap();
        HashMap validationGroupsAfter = new HashMap();
        HashMap validators = new HashMap();
        for (int i = 0; i < list.size(); i++)
        {
            ModificationItem modItem = (ModificationItem)list.get(i);
            if (modItem.getChangeType() != IElementChange.ELEMENT_UPDATED)
            {
                continue;
            }

            ArrayList modValidators = getModValidators(modItem, false);
            
            if (modValidators.isEmpty())
            {
                continue;
            }
            prepareUpdateTriggers(modItem, modValidators, validators, validationGroups, validationGroupsAfter);           
        }

        // Now run the trigger for each component
        Iterator iterator = validationGroups.keySet().iterator();
        int iteratorIndex = 0;
        while (iterator.hasNext())
        {
            String validatorID = (String)iterator.next();
            ArrayList vGroup = (ArrayList)validationGroups.get(validatorID);
            ArrayList afterGroup = (ArrayList)validationGroupsAfter.get(validatorID);
            IValidator validator = (IValidator)validators.get(validatorID);
            IDirElement[] beforeImages = new IDirElement[vGroup.size()];
            IDeltaElement[] afterImages = new IDeltaElement[afterGroup.size()];
            vGroup.toArray(beforeImages);
            afterGroup.toArray(afterImages);
            
            try
            {
                
                validator.onUpdate(new TriggerDSContext(m_ds) , beforeImages, afterImages);
            }
            catch (Throwable t)
            {
                throw new DirectoryServiceTriggerException(validatorID, t);
            }
        }
    }
    
    void runAfterUpdateTriggers(ArrayList list) throws DirectoryServiceException
    {
        // Sort by validators
        HashMap validationGroups = new HashMap();
        HashMap validationGroupsAfter = new HashMap();
        HashMap validators = new HashMap();
        for (int i = 0; i < list.size(); i++)
        {
            ModificationItem modItem = (ModificationItem)list.get(i);
            if (modItem.getChangeType() != IElementChange.ELEMENT_UPDATED)
            {
                continue;
            }

            ArrayList modValidators = getModValidators(modItem, true);
            
            if (modValidators.isEmpty())
            {
                continue;
            }
            prepareUpdateTriggers(modItem, modValidators, validators, validationGroups, validationGroupsAfter);           
        }

        // Now run the trigger for each component
        Iterator iterator = validationGroups.keySet().iterator();
        int iteratorIndex = 0;
        while (iterator.hasNext())
        {
            String validatorID = (String)iterator.next();
            ArrayList vGroup = (ArrayList)validationGroups.get(validatorID);
            ArrayList afterGroup = (ArrayList)validationGroupsAfter.get(validatorID);
            IPostCommitTrigger validator = (IPostCommitTrigger)validators.get(validatorID);

            IDirElement[] beforeImages = new IDirElement[vGroup.size()];
            IDeltaElement[] afterImages = new IDeltaElement[afterGroup.size()];
            vGroup.toArray(beforeImages);
            afterGroup.toArray(afterImages);
            
            try
            {
                validator.afterUpdate(new TriggerDSContext(m_ds) , beforeImages, afterImages);
            }
            catch (Throwable t)
            {
                throw new DirectoryServiceTriggerException(validatorID, t);
            }
        }
    }
    
    void runOnDeleteTriggers(ArrayList list) throws DirectoryServiceException
    {
        // Sort the deletions by validators
        HashMap validationGroups = new HashMap();
        HashMap validators = new HashMap();
        for (int i = 0; i < list.size(); i++)
        {
            ModificationItem modItem = (ModificationItem)list.get(i);
            if (modItem.getChangeType() != IElementChange.ELEMENT_DELETED)
            {
                continue;
            }

            ArrayList modValidators = getModValidators(modItem, false);
            
            if (modValidators.isEmpty())
            {
                continue;
            }
            prepareDeleteTriggers(modItem, modValidators, validators, validationGroups);           
        }

        // Now run the trigger for each component
        Iterator iterator = validationGroups.keySet().iterator();

        while (iterator.hasNext())
        {
            String validatorID = (String)iterator.next();
            ArrayList vGroup = (ArrayList)validationGroups.get(validatorID);
            IValidator validator = (IValidator)validators.get(validatorID);

            IDirElement[] beforeImages = new IDirElement[vGroup.size()];
            vGroup.toArray(beforeImages);
            try
            {
                validator.onDelete(new TriggerDSContext(m_ds) , beforeImages);
            }
            catch (Throwable t)
            {
                throw new DirectoryServiceTriggerException(validatorID, t);
            }
        }
    }
    
    void runAfterDeleteTriggers(ArrayList list) throws DirectoryServiceException
    {
        // Sort the deletions by validators
        HashMap validationGroups = new HashMap();
        HashMap validators = new HashMap();
        for (int i = 0; i < list.size(); i++)
        {
            ModificationItem modItem = (ModificationItem)list.get(i);
            if (modItem.getChangeType() != IElementChange.ELEMENT_DELETED)
            {
                continue;
            }

            ArrayList modValidators = getModValidators(modItem, true);
            
            if (modValidators.isEmpty())
            {
                continue;
            }
            prepareDeleteTriggers(modItem, modValidators, validators, validationGroups);           
        }

        // Now run the trigger for each component
        Iterator iterator = validationGroups.keySet().iterator();

        while (iterator.hasNext())
        {
            String validatorID = (String)iterator.next();
            ArrayList vGroup = (ArrayList)validationGroups.get(validatorID);
            IPostCommitTrigger validator = (IPostCommitTrigger)validators.get(validatorID);

            IDirElement[] beforeImages = new IDirElement[vGroup.size()];
            vGroup.toArray(beforeImages);
            try
            {
                validator.afterDelete(new TriggerDSContext(m_ds) , beforeImages);
            }
            catch (Throwable t)
            {
                throw new DirectoryServiceTriggerException(validatorID, t);
            }
        }
    }

    private ArrayList getModValidators(ModificationItem modItem, boolean postCommitTriggersOnly)
    {
        ArrayList modValidators = new ArrayList();

        IElementIdentity modID = modItem.getModificationID();
        String validatorID = modID.getType() + modID.getReleaseVersion();

        ArrayList validatorsAL = (ArrayList)m_validationMethodsTable.get(validatorID);
        if (validatorsAL != null)
        {
            modValidators.addAll(validatorsAL);
        }
        validatorsAL = (ArrayList)m_validationMethodsTable.get("**");
        if (validatorsAL != null)
        {
            modValidators.addAll(validatorsAL);
        }
        validatorsAL = (ArrayList)m_validationMethodsTable.get("*" + modID.getReleaseVersion());
        if (validatorsAL != null)
        {
            modValidators.addAll(validatorsAL);
        }
        validatorsAL = (ArrayList)m_validationMethodsTable.get(modID.getType() + "*");
        if (validatorsAL != null)
        {
            modValidators.addAll(validatorsAL);
        }
        
        // take out any validators [triggers] that don't implement IPostCommitTrigger
        if (postCommitTriggersOnly)
        {
            Iterator iterator = modValidators.iterator();
            while (iterator.hasNext())
            {
                IValidator validator = (IValidator)iterator.next();
                if (!(validator instanceof IPostCommitTrigger))
                {
                    iterator.remove();
                }
            }
        }

        return modValidators;
    }
    
    private void prepareDeleteTriggers(ModificationItem modItem, ArrayList modValidators, HashMap validators, HashMap validationGroups)
    {
        Iterator validatorsIt = modValidators.iterator();
        while (validatorsIt.hasNext())
        {
            IValidator validator = (IValidator)validatorsIt.next();
            String validatorID = (String)constructValidatorID(validator);
            ArrayList vGroup = (ArrayList)validationGroups.get(validatorID);

            if (vGroup == null)
            {
                vGroup = new ArrayList();
                validationGroups.put(validatorID, vGroup);
                validators.put(validatorID, validator);
            }
            IDirElement beforeImage = (IDirElement)modItem.getBeforeImage();
            // beforeImage will be null elements in packed directories, like ACLs, USERS are deleted in a group
            // The current validator interface expects beforeImages for onDelete, and will not
            // be able to do much without a before image. If a specific validator for deletion of packed elements
            // in a group is required, the onDelete method of the interface will have to change so that
            // some other meaningful data can be passed to it (perhaps the entire ModificationItem)
            // Individual users, groups, acls, etc will have a beforeImage when they get here.
            if (beforeImage != null)
            {
                vGroup.add(beforeImage);
            }
        }
    }
    
    private void prepareCreateTriggers(ModificationItem modItem, ArrayList modValidators, HashMap validators, HashMap validationGroups)
    {
        Iterator validatorsIt = modValidators.iterator();
        while (validatorsIt.hasNext())
        {
            IValidator validator = (IValidator)validatorsIt.next();
            String validatorID = (String)constructValidatorID(validator);
            ArrayList vGroup = (ArrayList)validationGroups.get(validatorID);

            if (vGroup == null)
            {
                vGroup = new ArrayList();
                validationGroups.put(validatorID, vGroup);
                validators.put(validatorID, validator);
            }
            IDirElement newImage = (IDirElement)modItem.getModification();
           
            if (newImage != null)
            {
                vGroup.add(newImage);
            }
        }
    }
    
    private void prepareUpdateTriggers(ModificationItem modItem, ArrayList modValidators, HashMap validators, HashMap validationGroupsBefore, HashMap validationGroupAfter)
    {
        Iterator validatorsIt = modValidators.iterator();
        while (validatorsIt.hasNext())
        {
            IValidator validator = (IValidator)validatorsIt.next();
            String validatorID = (String)constructValidatorID(validator);
            ArrayList vGroup = (ArrayList)validationGroupsBefore.get(validatorID);

            if (vGroup == null)
            {
                vGroup = new ArrayList();
                validationGroupsBefore.put(validatorID, vGroup);
                validators.put(validatorID, validator);
            }
            IDirElement beforeImage = (IDirElement)modItem.getBeforeImage();
            
            if (beforeImage != null)
            {
                vGroup.add(beforeImage);
            }
            
            
            ArrayList after = (ArrayList)validationGroupAfter.get(validatorID);

            if (after == null)
            {
                after = new ArrayList();
                validationGroupAfter.put(validatorID, after);
                validators.put(validatorID, validator);
            }
            IBasicElement afterImage = (IBasicElement)modItem.getModification();
            
            if (afterImage != null)
            {
                after.add(afterImage);
            }
        }
    }

    void validate(ArrayList list, boolean isImportTransaction) throws DirectoryServiceException
    {
        // Sort the modifications by validators
        HashMap validationGroups = new HashMap();
        HashMap validators = new HashMap();
        
        for (int i = 0; i < list.size(); i++)
        {
            ModificationItem modItem = (ModificationItem)list.get(i);
            IElementIdentity modID = modItem.getModificationID();
            String validatorID = modID.getType() + modID.getReleaseVersion();
            
            
            ArrayList modValidators = new ArrayList();
            ArrayList validatorsAL = (ArrayList)m_validationMethodsTable.get(validatorID);
            if (validatorsAL != null)
            {
                modValidators.addAll(validatorsAL);
            }
            validatorsAL = (ArrayList)m_validationMethodsTable.get("**");
            if (validatorsAL != null)
            {
                modValidators.addAll(validatorsAL);
            }
            validatorsAL  = (ArrayList)m_validationMethodsTable.get("*" + modID.getReleaseVersion());
            if (validatorsAL != null)
            {
                modValidators.addAll(validatorsAL);
            }
            validatorsAL = (ArrayList)m_validationMethodsTable.get(modID.getType() + "*");
            if (validatorsAL != null)
            {
                modValidators.addAll(validatorsAL);
            }
            
            if (modValidators.isEmpty()) // No validation for this type of element
            {
                if (m_badValidator == null)
                {
                    continue;
                }
                else
                {
                    //validatorID = "_MF_MISSING_VALIDATOR";
                	modValidators.add(m_badValidator);
                }
            }

            Iterator validatorsIt = modValidators.iterator();
            while (validatorsIt.hasNext())
            {
            	IValidator validator = (IValidator)validatorsIt.next();
            	validatorID = (String)constructValidatorID(validator);
                ArrayList vGroup = (ArrayList)validationGroups.get(validatorID);

                if (vGroup == null)
                {
                    vGroup = new ArrayList();
                    validationGroups.put(validatorID, vGroup);
                    validators.put(validatorID, validator);
                }
                vGroup.add(modItem);
            }
        }

        // Now validate each validation group by its validating routine
        Iterator iterator = validationGroups.keySet().iterator();

        while (iterator.hasNext())
        {
            String validatorID = (String)iterator.next();
            ArrayList vGroup = (ArrayList)validationGroups.get(validatorID);
            IValidator validator = (IValidator)validators.get(validatorID);

            if (validator == null)
            {
                validator = m_badValidator;
                if (validator == null)
                {
                    throw new DirectoryServiceValidationException(validatorID, new Exception("No validator is available."));
                }
            }

            ModificationItem[] modifications = new ModificationItem[vGroup.size()];
            vGroup.toArray(modifications);
            try
            {
                validator.validate(new ValidationDSContext(m_ds, isImportTransaction), modifications);
            }
            catch (Throwable t)
            {
            	t.printStackTrace();
                throw new DirectoryServiceValidationException(validatorID, t);
            }

        }

    }
    // we need a unique ID when type and version are not enough - we might have more than one
    // validator which validates the same type/version
    private String constructValidatorID(IValidator validator)
    {
    	String[] types = validator.getElementTypesValidated();
    	String ID = "";
    	if (types != null)
    	{
	    	for (int i=0; i<types.length; i++)
            {
                ID = ID + types[i];
            }
    	}
    	ID = ID + validator.getReleaseVersion();
    	ID = ID + validator.getClass();
    	return ID;
    }

    private IValidator createValidatorPlugin(String className, String classpath) throws Exception
    {
        Class validatorClass = m_ds.loadClass(className, classpath);
        Constructor validatorContructor = null;
        try
        {
            validatorContructor = validatorClass.getConstructor(ITriggerDSContext.class);
        } catch (NoSuchMethodException e) { }
        
        Object validator = validatorContructor == null ? validatorClass.newInstance() : validatorContructor.newInstance(new TriggerDSContext(m_ds));
        if (!(validator instanceof IValidator))
        {
            throw new Exception(className +  " must implement the 'IValidator' interface.");
        }

        return (IValidator)validator;
    }

    private static class PluginInfo
    {
        String m_className;
        String m_classpath;
        String m_releaseVersion;

        PluginInfo (String className, String classpath, String releaseVersion)
        {
            m_className = className;
            m_classpath = classpath;
            m_releaseVersion = releaseVersion;
        }
    }

    static class BadValidator implements IValidator
    {
        @Override
        public String getReleaseVersion()
        {
            return "0";
        }

        @Override
        public void validate(IValidationDSContext dsContext, IValidationElementChange[] changes) throws Exception
        {
            //throw new Exception("Validation failed due to a failure to load one or more validation classes.");
        }
        @Override
        public String[] getElementTypesValidated()
        {
            return null;
        }

        @Override
        public void onDelete(ITriggerDSContext dsContext, IDirElement[] changes) throws DirectoryServiceException{}

      
        @Override
        public void onUpdate(ITriggerDSContext dsContext, IDirElement[] beforeImages, IDeltaElement[] afterImages)
        throws DirectoryServiceException
        {
            // TODO Auto-generated method stub
            
        }

    
        @Override
        public void onCreate(ITriggerDSContext dsContext, IDirElement[] afterImages)
        throws DirectoryServiceException
        {
            // TODO Auto-generated method stub
            
        }
    }


}
