package com.sonicsw.mf.framework.directory.storage;
import java.io.File;

import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.framework.directory.ILogger;

/**
  * This interface is implemented for each type of storage underlying the central repository of the directory
  * service.  In particular, we implement a JNDI IStorage class and a file-system IStorage class.
  */
public interface IStorage
{

 /**
  * Returns the domain name for this storage (which is the root directory in the file-system implementation).
  *
  * @return the domain name of this storage.
  *
  */
  public String getDomain();


 /**
  * Returns the element specified by name.
  *
  * @param elementName the full path from the root directory '/' to the element's name
  *
  * @return an element (or null if the elemnt is not found)
  *
  * @exception StorageException if the parent directory is not valid or the element is found but is not
  *             accessible or not valid.
  *
  */
  public IDirElement getElement(EntityName elementName) throws StorageException;

 /**
  * Returns the element specified by name and all the other elements that can be retrieved without any extra I/O.
  * Used by callers that populate a cache for maximum efficiency when most of the elements under the directory
  * are to be accessed.
  *
  * @param elementName the full path from the root directory '/' to the element's name
  *
  * @return  an array of elements (or an empty array if the element was not found)
  *
  * @exception StorageException if the parent directory is not valid or the element is found but is not
  *             accessible or not valid.
  *
  */
  public IDirElement[] getElements(EntityName elementName) throws StorageException;


 /**
  * Returns all the elements under the specified directory.
  *
  * @param dirName the full path from the root directory '/' to the directory name
  *
  * @return all the elements under the directory (can be 0 length array if no elements are found)
  *
  * @exception StorageException if the directory is not valid or an element is not accessible or not valid.
  *
  */
  public IDirElement[] getAllElements(EntityName dirName) throws StorageException;

 /**
  * Returns all the ids ofthe  elements under the specified directory.
  *
  * @param dirName the full path from the root directory '/' to the directory name
  *
  * @return all the identities of elements under the directory (can be 0 length array if no elements are found)
  *
  * @exception StorageException if the directory is not valid or an element is not accessible or not valid.
  *
  */
  public IElementIdentity[] listElements(EntityName dirName) throws StorageException;


 /**
  * Returns all the ids of the directories under the specified directory.
  *
  * @param dirName the full path from the root directory '/' to the directory name
  *
  * @return all the identities of directories under the directory (can be 0 length array if no directories
  *         are found)
  *
  * @exception StorageException if the directory is not valid
  *
  */
  public IDirIdentity[] listDirectories(EntityName dirName) throws StorageException;

 /**
  * Returns all the ids of the directories and elements under the specified directory.
  *
  * @param dirName the full path from the root directory '/' to the directory name
  *
  * @return all the identities under the directory (can be 0 length array if no elements or directories
  *         are found).
  *
  * @exception StorageException if the directory is not valid
  *
  */
  public IIdentity[] listAll(EntityName dirName) throws StorageException;

 /**
  * Deletes the element specified by name and returns its id.
  *
  * @param elementName the full path from the root directory '/', to the element's name
  *
  * @return the id of the deleted  element (or null if the element was not found)
  *
  * @exception StorageException if the directory is not valid or the file is corrupt
  *
  */
  public IElementIdentity deleteElement(EntityName elementName) throws StorageException;

 /**
  * Deletes the element specified by name and returns its id.
  *
  * @param elementName the full path from the root directory '/', to the element's name
  * @param force delete the file even if corrupt
  *
  * @return the id of the deleted  element (or null if the element was not found)
  *
  * @exception StorageException if the directory is not valid
  *
  */
  public IElementIdentity deleteElement(EntityName elementName, boolean force) throws StorageException;


 /**
  * Deletes the elements list efficiently (more efficient than the deletion of each element separately).
  * All the elements must belong to the same directory
  *
  * @param elementNames the list of element names to delete
  *
  * @return the id list of the deleted elements
  *
  * @exception StorageException if the directory is not valid
  *
  */
  public IElementIdentity[] deleteElements(EntityName[] elementNames) throws StorageException;

 /**
  * Creates a new directory.
  *
  * @param dirName the full path from the root directory '/' to the directory name
  *
  * @exception StorageException if the directory could not be created
  *
  */
  public void createDirectory(EntityName dirName) throws StorageException;

 /**
  * Deletes a directory.
  *
  * @param dirName the full path from the root directory '/' to the directory name
  *
  * @exception StorageException if the parent dierctory is not valid or the directory is found but could
  *            not be deleted
  *
  */
  public void deleteDirectory(EntityName dirName) throws StorageException;

 /**
  * Checks the existence of a directory.
  *
  * @param dirName the full path from the root directory '/' to the directory name
  *
  * @return true if the directory exists
  *
  */
  public boolean directoryExists(EntityName dirName);


 /**
  * Sets a new element or replaces an existing element (the parent directory must exist)
  *
  * @param elementName the full path from the root directory '/' to the element's name
  * @param element the element to be set
  *
  * @exception StorageException if the new element could not be set or the existing element could not be
  *            replaced
  *
  */
  public void setElement(EntityName elementName, IDirElement element) throws StorageException;

 /**
  * Sets a new element or replaces an existing elemen - creates the parent directory if needed
  *
  * @param elementName the full path from the root directory '/' to the element's name
  * @param element the element to be set
  * @param createParentDir if true the parent directory(s) should be automatically created
  *
  * @exception StorageException if the new element could not be set or the existing element could not be
  *            replaced
  *
  */
  public void setElement(EntityName elementName, IDirElement element, boolean createParentDir) throws StorageException;

 /**
  * Sets new elements or replaces an existing elements - creates the parent directory(s) if needed
  *
  * @param elements the elements to be set
  * @param createParentDir if true the parent directory(s) should be automatically created
  *
  * @exception StorageException if an element could not be set or an existing element could not be  replaced
  *
  */
  public void setElements(IDirElement[] elements, boolean createParentDir) throws StorageException;


 /**
  * Closes the storage and releases any resources it holds.
  *
  * @exception StorageException if failed to release resources or do a cleanup (e.g. flush buffers),
  *
  */
  public void close() throws StorageException;

 /**
  * Sets a new blob or replaces an existing one.
  *
  * @param blobName the name of the blob (the full path)
  * @param blob the blob's bytes
  *
  * @exception StorageException if the blob cannot be stored
  *
  */
  public void setBlob(EntityName blobName, byte[] blob) throws StorageException;

 /**
  * Returns a reference to the blob as a File. Used to optimize blob access to blobs in the
  * file system storage case when local access is needed (the blob is already stored as a file);
  * The existence of the file is not checked.
  *
  * @param blobName the name of the blob (the full path)
  * @return the File object of the file that stores the blob
  *
  */
  public File blobToFile(EntityName blobName);

 /**
  * Retrieves the blob
  *
  * @param blobName the name of the blob (the full path)
  * @return the blob's bytes (null if the blob is not there)
  *
  * @exception StorageException if the storage cannot be accessed
  *
  */
  public byte[] getBlob(EntityName blobName) throws StorageException;

 /**
  * Deletes the blob
  *
  * @param blobName the name of the blob (the full path)
  *
  * @exception StorageException if the blob cannot be deleted (it's ok if it was already deleted)
  *
  */
  public void deleteBlob(EntityName blobName) throws StorageException;

  /**
  * Append a piece of blob to an existing blob or create a new one.
  *
  * @param entityName the name of the blob (the full path)
  * @param blob the bytes of the piece of the blob to be appended
  * @param offset the location in the blob where the append operation will start
  *
  * @exception StorageException if the blob cannot be stored
  *
  */

  public void appendBlob(EntityName entityName, byte[] blobBytes, int offset) throws StorageException;

  /**
  * Retrieves a piece of the blob, starting at byte offset, and of length "length" or
  * the entire remaining blob starting at offset, if it is smaller
  *
  * @param entityName the name of the blob (the full path)
  * @param offset the starting index in the stream
  * @param length the number of bytes to return in the blob
  * @return the blob's bytes (null if the blob is not there)
  *
  * @exception StorageException if the storage cannot be accessed
  *
  */

  public byte[] getBlob(EntityName entityName, int offset, int length) throws StorageException;

  /**
  * Starts a transaction.
  *
  * @exception StorageException if fails to access the transaction logs.
  *
  */
  public void startTransaction() throws StorageException;

 /**
  * Commit the transaction.
  *
  * @exception StorageException if fails to access the transaction logs.
  *
  */
  public void commitTransaction() throws StorageException;

 /**
  * Rollback the transaction (rollbackTransaction might be called even if a transaction was never started).
  *
  * @exception StorageException if fails to access the transaction logs.
  *
  */
  public void rollbackTransaction() throws StorageException;

 /**
  * Close all the open files - used before online backup>
  *
  * @exception StorageException if fails to close the files
  *
  */
  public void closeFiles() throws StorageException;

 /**
  * Reopen files closedc by closeFiles
  *
  * @exception StorageException if fails to open the files
  *
  */
  public void openFiles() throws StorageException;

 /**
  * Sets a logger (optional)
  *
  * @param logger
  *
  */
  public void setLogger(ILogger logger);

  /**
   * Copy a blob from one element to another
   * @exception StorageException
   */

  public void copyBlob(EntityName fromBlobName, EntityName toBlobName) throws StorageException;

  /**
   * Return the size of a blob in bytes
   * @exception StorageException
   */

  public long getBlobSize(EntityName blobName) throws StorageException;

  /**
   * Do cleanup activities related to closing the transaction manager
   * @exception StorageException
   */

  public void closeTransactionManager() throws StorageException;

  /**
   * Backup the storage to another directory
   * @exception StorageException
   */

  public void backup(String backupDir);


  /*********** String Collection methods **********/

 /**
  * Returns the collection items in a String array
  * @param collectionName
  * @exception StorageException
  */
  public String[] collectionToStringArray(String collectionName) throws StorageException;

 /**
  * Removes a string item from the collection
  * @param collectionName
  * @param item to be removed
  * @exception StorageException
  */
  public void removeFromCollection(String collectionName, String item) throws StorageException;

 /**
  * Removes a string items from the collection
  * @param collectionName
  * @param items to be removed
  * @exception StorageException
  */
  public void removeFromCollection(String collectionName, String[] items) throws StorageException;

 /**
  * Adds a string item to the collection
  * @param collectionName
  * @param item to be removed
  * @exception StorageException
  */
  public void addToCollection(String collectionName, String item) throws StorageException;

 /**
  * Creates a new collection
  * @param collectionName
  * @exception StorageException
  */
  public void createCollectionIfNotCreated(String collectionName) throws StorageException;


}

