package com.sonicsw.mf.framework.directory.storage.fs;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.impl.DirIdentity;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.framework.directory.ILogger;
import com.sonicsw.mf.framework.directory.storage.IStorage;
import com.sonicsw.mf.framework.directory.storage.PackedDirUtil;
import com.sonicsw.mf.framework.directory.storage.ParentDirDoesNotExistException;
import com.sonicsw.mf.framework.directory.storage.StorageException;


/* FSStorage implements the file-system store for the Directory Service and the persistent-cache .
 * Some methods are synchronized for the persistent-cache usage since several threads might need to
 * access it synchronously
 */
public class FSStorage implements IStorage
{

    private File m_domainDir = null;
    private File m_dataDir = null;
    private FSTransactionManager m_trManager;
    private FileManager m_fileManager;
    private Mapper m_mapper;
    private ILogger m_logger;
    private static int COPY_CHUNK_SIZE = 1000000;


    public static final String TR_FILE_POSTFIX = "tr";


    public FSStorage (String hostDirName, String domainName, boolean doSync) throws StorageException
    {
        init(hostDirName, domainName, IMFDirectories.MF_DATA_DIR, null, doSync, true);
    }

    public FSStorage (String hostDirName, String domainName, String dataDirName, boolean doSync) throws StorageException
    {
        init(hostDirName, domainName, dataDirName, null, doSync, true);
    }

    public FSStorage (String hostDirName, String domainName, String dataDirName, String password, boolean doSync) throws StorageException
    {
        init (hostDirName, domainName, dataDirName, password, doSync, true);
    }

    public FSStorage (String hostDirName, String domainName, String dataDirName) throws StorageException
    {
        init (hostDirName, domainName, dataDirName, null, false, false);
    }

    @Override
    public void setLogger(ILogger logger)
    {
        m_logger = logger;
        if (m_fileManager != null)
        {
            m_fileManager.setLogger(logger);
        }
    }

    private void init (String hostDirName, String domainName, String dataDirName, String password, boolean doSync, boolean haveTrManager)
        throws StorageException
    {
        if (domainName == null || domainName.length() == 0)
        {
            throw new StorageException("A domain name is missing.");
        }

        if (hostDirName== null || hostDirName.length() == 0)
        {
            throw new StorageException("A host directory name is missing.");
        }

        File hostDir = new File (hostDirName);
        if (!hostDir.exists())
        {
            throw new StorageException("Directory '" + hostDirName +  ", does not exist.");
        }

        m_logger = null;
        m_domainDir = new File(hostDir, domainName);

        if (!m_domainDir.exists())
        {
            if (!m_domainDir.mkdir())
            {
                throw new StorageException("Cannot create the root directory '" + m_domainDir.getPath() + "'.");
            }
        }

        if (!m_domainDir.canWrite())
        {
            throw new StorageException("No write permission in directory '" + m_domainDir.getPath() + "'.");
        }

        m_dataDir = new File(m_domainDir, dataDirName);

        if (!m_dataDir.exists())
        {
            if (!m_dataDir.mkdir())
            {
                throw new StorageException("Cannot create the data directory '" + m_dataDir.getPath() + "'.");
            }
        }

        m_mapper = new Mapper();


        File transactionsFile = null;
        if (haveTrManager)
        {
            transactionsFile = new File(m_domainDir, dataDirName + "." + TR_FILE_POSTFIX);
        }
        else
        {
            transactionsFile = new File(m_domainDir, "_TR_FILE_DOES_NOT_EXIST");
        }
        m_trManager = new FSTransactionManager(transactionsFile, m_dataDir, doSync);


        try
        { 
            m_fileManager = new FileManager(password, m_trManager); 
        }
        catch(Exception e)
        {
            throw new StorageException(e.getMessage(), e); 
        }
    }

    @Override
    public String getDomain()
    {
        return m_domainDir.getName();
    }

    @Override
    public synchronized IDirElement getElement(EntityName elementName) throws StorageException
    {
        File elementFile = fileForName(elementName, true, m_mapper);
        if (!elementFile.exists() || elementFile.isDirectory())
        {
            return null;
        }

        return (IDirElement)m_fileManager.getElement(elementFile, elementName);
    }

    @Override
    public synchronized IDirElement[] getElements(EntityName elementName) throws StorageException
    {
        File elementFile = fileForName(elementName, true, m_mapper);
        if (!elementFile.exists() || elementFile.isDirectory())
        {
            return new IDirElement[0];
        }

        return m_fileManager.getElements(elementFile);
    }


    @Override
    public IDirElement[] getAllElements(EntityName dirName) throws StorageException
    {
        File[] kids = listKids(dirName);
        ArrayList elementObjects = new ArrayList();

        for (int i = 0; i < kids.length; i++)
        {
            if (kids[i].isFile())
            {
                Object [] elementsObj = m_fileManager.getElements(kids[i]);
                for (int j = 0; j < elementsObj.length; j++)
                {
                    elementObjects.add(elementsObj[j]);
                }
            }
        }

        return (IDirElement[])elementObjects.toArray(new IDirElement[elementObjects.size()]);
    }


    @Override
    public IElementIdentity[] listElements(EntityName dirName) throws StorageException
    {
        File[] kids = listKids(dirName);
        ArrayList elementIds = new ArrayList();

        for (int i = 0; i < kids.length; i++)
        {
            if (kids[i].isFile())
            {
                Object [] ids = m_fileManager.getIds(kids[i]);
                for (int j = 0; j < ids.length; j++)
                {
                    elementIds.add(ids[j]);
                }
            }
        }
        return (IElementIdentity[])elementIds.toArray(new IElementIdentity[elementIds.size()]);
    }

    @Override
    public IDirIdentity[] listDirectories(EntityName dirName) throws StorageException
    {
       File[] kids = listKids(dirName);
       ArrayList dirIdObjects = new ArrayList();

       for (int i = 0; i < kids.length; i++)
    {
        if (kids[i].isDirectory())
        {
            dirIdObjects.add(createDirID(dirName, kids[i].getName()));
        }
    }

       IDirIdentity[] dirIds = new IDirIdentity[dirIdObjects.size()];

       for (int i = 0; i < dirIdObjects.size(); i++)
    {
        dirIds[i] = (IDirIdentity)dirIdObjects.get(i);
    }

       return dirIds;
    }

    @Override
    public IIdentity[] listAll(EntityName dirName) throws StorageException
    {
       File[] kids = listKids(dirName);
       ArrayList kidObjects = new ArrayList();

       for (int i = 0; i < kids.length; i++)
    {
        if (kids[i].isDirectory())
        {
            kidObjects.add(createDirID(dirName, kids[i].getName()));
        }
        else if (kids[i].isFile())
           {
                Object [] ids = m_fileManager.getIds(kids[i]);
                for (int j = 0; j < ids.length; j++)
                {
                    kidObjects.add(ids[j]);
                }
           }
        else
        {
            throw new StorageException("Unknown file type for '" + kids[i].getName() + "' in directory '" + dirName.getName() + "'.");
        }
    }

       return  (IIdentity[])kidObjects.toArray(new IIdentity[kidObjects.size()]);
    }

    @Override
    public synchronized IElementIdentity deleteElement(EntityName elementName) throws StorageException
    {
        return deleteElement(elementName, false);
    }

    @Override
    public synchronized IElementIdentity deleteElement(EntityName elementName, boolean force) throws StorageException
    {
        File elementFile = fileForName(elementName, true, m_mapper);
        try
        {
            return m_fileManager.deleteElement(elementFile, elementName);
        }
        catch (StorageException e)
        {
            if (force)
            {
                elementFile.delete();
                return null;
            }
            else
            {
                throw e;
            }
        }
    }

    @Override
    public synchronized IElementIdentity[] deleteElements(EntityName[] elementNames)throws StorageException
    {
        if (elementNames.length == 0)
        {
            return new IElementIdentity[0];
        }

        HashMap groups = groupElementNames(elementNames);
        Iterator iterator = groups.keySet().iterator();
        ArrayList allDeletedElements = new ArrayList();
        while( iterator.hasNext())
        {
            File file = (File)iterator.next();
            ArrayList group = (ArrayList)groups.get(file);
            IElementIdentity[] deletedIDs = m_fileManager.deleteElements(file, (EntityName[])group.toArray(new EntityName[group.size()]));
            for (int i = 0; i < deletedIDs.length; i++)
            {
                allDeletedElements.add(deletedIDs[i]);
            }
        }
        IElementIdentity[] result = new IElementIdentity[allDeletedElements.size()];
        allDeletedElements.toArray(result);
        return result;

    }


    @Override
    public void createDirectory(EntityName dirName) throws StorageException
    {
        createDirectory(dirName, false);
    }

    private void createDirectory(EntityName dirName, boolean createParent) throws StorageException
    {
        createDirectory(m_dataDir, dirName, createParent, m_trManager);
    }

    @Override
    public boolean directoryExists(EntityName dirName)
    {
        return directoryExists(m_dataDir, dirName);
    }

    @Override
    public void deleteDirectory(EntityName dirName) throws StorageException
    {
        File dir = fileForName(dirName, true, null);

        if (!dir.exists())
        {
            return;
        }

        m_trManager.deleteDirectory(dirName.getName());
        if (!dir.delete())
        {
            File[] kids = listKids(dirName);
            if (kids.length > 0)
            {
                throw new StorageException("'" + dirName.getName() + "' is not empty - cannot be deleted.");
            }
            throw new StorageException("Cannot delete directory '" + dirName.getName() + "'.");
        }
    }

    @Override
    public synchronized void setElement(EntityName elementName, IDirElement element) throws StorageException
    {
        setElement(elementName, element, false);
    }

    @Override
    public synchronized void setElement(EntityName elementName, IDirElement element, boolean createPath) throws StorageException
    {
       File elementFile = null;
       try
       {
           elementFile = fileForName(elementName, true, m_mapper);
       }
       catch (StorageException pe)
       {
           if (!createPath || !(pe instanceof ParentDirDoesNotExistException))
        {
            throw pe;
        }

           try
           {
               createDirectory(new EntityName(elementName.getParent()), true);
           }
           catch (ConfigException e) 
           {
               throw new Error(e.toString(), e);
           }
           elementFile = fileForName(elementName, false, m_mapper);
       }

       m_fileManager.setElement(elementFile, element);

    }

    @Override
    public synchronized void setElements(IDirElement[] elements, boolean createPath) throws StorageException
    {
        if (elements.length == 0)
        {
            return;
        }

        EntityName checkName = getEntity(elements[0].getIdentity().getName());
        EntityName dirName = getEntity(checkName.getParent());

        if (!directoryExists(dirName))
        {
            if (!createPath)
            {
                throw new StorageException(dirName + " doesn't exist.");
            }
            else
            {
                createDirectory(dirName, true);
            }
        }

        HashMap groups = groupElements(checkName,  elements);
        Iterator i = groups.keySet().iterator();
        while( i.hasNext())
        {
            File file = (File)i.next();
            ArrayList group = (ArrayList)groups.get(file);
            m_fileManager.setElements(file, (IDirElement[])group.toArray(new IDirElement[group.size()]));
        }
    }

    @Override
    public void close() throws StorageException
    {
        if (m_trManager != null)
        {
            m_trManager.close();
        }
    }

    private File fileForName(EntityName entityName, boolean checkParent, Mapper mapper) throws StorageException
    {
        return fileForName(m_dataDir, entityName, checkParent, mapper);
    }

    private File[] listKids(EntityName dirName) throws StorageException
    {
        File dirFile = fileForName(dirName, false, null);
        if (!dirFile.exists())
        {
            throw new StorageException("Directory '" + dirName.getName() +  "' does not exist.");
        }

        if (!dirFile.isDirectory())
        {
            throw new StorageException("'" + dirName.getName() +  "' is not a directory.");
        }

        String[] kidNames = dirFile.list();
        File[] files = new File[kidNames.length];

        for (int i = 0; i < kidNames.length; i++)
        {
            files[i] = new File(dirFile, kidNames[i]);
        }

        return files;
    }

    private IDirIdentity createDirID(EntityName parent, String name) throws StorageException
    {
       String parentName = parent.getName();
       String prefix = parentName.length() == 1 ? parentName : parentName + "/";
       return new DirIdentity(prefix + name);
    }

    @Override
    public void setBlob(EntityName blobName, byte[] blob) throws StorageException
    {
       File elementFile = null;
       try
       {
           elementFile = fileForName(blobName, true, null);
       }
       catch (StorageException pe)
       {
           if (!(pe instanceof ParentDirDoesNotExistException))
        {
            throw pe;
        }

           try
           {
               createDirectory(new EntityName(blobName.getParent()), true);
           }
           catch (ConfigException e) 
           {
               throw new Error(e.toString(), e);
           }
           elementFile = fileForName(blobName, false, null);
       }

       if (elementFile.exists())
    {
        m_trManager.deleteElement(blobName);
    }
    else
    {
        m_trManager.newElement(blobName.getName());
    }

       try
       {
           FileOutputStream fileOut = new FileOutputStream(elementFile);
           fileOut.write(blob);
           fileOut.close();
       }
       catch (IOException e)
       {
            throw new StorageException("Could not write binary attachment '" + blobName.getName() + "':" + e.toString(), e);
       }

    }

    // mrd 12/07/2004 added to deal with large file attachments. See interface definition for details
    @Override
    public void appendBlob(EntityName entityName, byte[] blobBytes, int from) throws StorageException
    {
        appendBlob(entityName, blobBytes, 0, blobBytes.length, from);
    }
    /**
     * 
     * @param entityName name of the blobs
     * @param blobBytes bytes to write
     * @param blobOff where to start writing from within the byte[]
     * @param blobLen how many byes from the byte[]
     * @param from from where in the file
     * @throws StorageException
     */
    public void appendBlob(EntityName entityName, byte[] blobBytes, int blobOff, int blobLen, int from) throws StorageException
    {
        try
        {
            File blobFile = null;
            try
            {
                blobFile = fileForName(entityName, true, null);
            }
            catch (StorageException pe)
            {
                if (! (pe instanceof ParentDirDoesNotExistException))
                {
                    throw pe;
                }

                try
                {
                    createDirectory(new EntityName(entityName.getParent()), true);
                }
                catch (ConfigException e)
                {
                    throw new Error(e.toString(), e);
                }
                blobFile = fileForName(entityName, false, null);
            }
            if ((from == 0) &&(blobFile.exists()))
            {
                deleteBlob(entityName);
            }
            if ((from != 0) && (blobFile.length() != from))
            {
                throw new Error("Cannot append to a blob in the middle of a file");
            }
            FileOutputStream blobStream = new FileOutputStream(blobFile.getAbsolutePath(), true);
            blobStream.write(blobBytes, blobOff, blobLen);
            blobStream.close();
        }
        catch (Exception e)
        {
            e.printStackTrace();
            throw new StorageException(e.toString(), e);
        }
    }

    @Override
    public File blobToFile(EntityName blobName)
    {
       try
       {
           return fileForName(blobName, false, null);
       }
       catch (StorageException e)
       {
           throw new Error(e.toString(), e); // Should never happen since we don't check for the Files's existence
       }
    }

    @Override
    public long getBlobSize(EntityName blobName) throws StorageException
    {
    	File blobFile = fileForName(blobName, false, null);
    	if (blobFile != null)
        {
            return blobFile.length();
        }
        else
        {
            return 0;
        }
    }

    @Override
    public byte[] getBlob(EntityName blobName) throws StorageException
    {
      // We could make this obsolete because of large file handling. mrd 12/07/2004 Instead, if the
      // blob is less than the BLOB_CHUNK_SIZE, we can return the whole thing still.
       return getBlob(blobName, 0, COPY_CHUNK_SIZE);
    }

    // mrd 12/07/2004 added to deal with large file attachments. See interface definition for details
    @Override
    public byte[] getBlob(EntityName blobName, int offset, int length) throws StorageException
    {
        byte[] fileByteArray = null;
        try
        {
            File blobFile = blobToFile(blobName);
            if (!blobFile.exists())
            {
                return null;
            }
            RandomAccessFile blobInputStream = new RandomAccessFile(blobFile, "r");
            if ( (offset + length) >= blobFile.length())
            {
                fileByteArray = new byte[(int)(blobFile.length() - offset)];
            }
            else
            {
                fileByteArray = new byte[length];
            }
            blobInputStream.seek(offset);
            if (!(fileByteArray.length == 0))
            {
                blobInputStream.read(fileByteArray, 0, fileByteArray.length);
            }
            blobInputStream.close();
            return fileByteArray;
        }
        catch (Exception e)
        {
            throw new StorageException(e.toString(), e);
        }
    }

    @Override
    public void deleteBlob(EntityName blobName) throws StorageException
    {
      try
      {
        File blobFile = fileForName(blobName, true, null);
        m_trManager.deleteElement(blobName);
        blobFile.delete();
      }
      catch (StorageException e)
      {
        throw new StorageException("Unable to delete blob '" + blobName.getName() + "', blob may be in use - " + e.toString(), e);
      }
    }

   // mrd 12/07/2004 modified to deal with large file attachments in chunks
    @Override
    public void copyBlob(EntityName fromBlobName, EntityName toBlobName) throws StorageException
    {
      File blobFile = blobToFile(fromBlobName);
      if (blobFile.exists())
      {
          try
          {
              FileInputStream inS = new FileInputStream(blobFile);
              int available = inS.available();
              inS.close(); // opened it just to get the number of bytes available
              byte[] blobPiece;
              int src = 0;
              while ( (src + COPY_CHUNK_SIZE) < available)
              {
                  blobPiece = getBlob(fromBlobName, src, COPY_CHUNK_SIZE);
                  appendBlob(toBlobName, blobPiece, src);
                  src = src + IBlob.BLOB_CHUNK_SIZE;
              }
              blobPiece = getBlob(fromBlobName, src, COPY_CHUNK_SIZE);
              appendBlob(toBlobName, blobPiece, src);
          }
          catch (IOException ioEx)
          {
              throw new StorageException(ioEx.toString(), ioEx);
          }
      }
    }

    @Override
    public void startTransaction() throws StorageException
    {
        m_trManager.startTransaction();
    }

    @Override
    public void commitTransaction() throws StorageException
    {
        m_trManager.commit();
    }

    @Override
    public void rollbackTransaction() throws StorageException
    {
        m_trManager.rollback();
    }

    @Override
    public void closeFiles() throws StorageException
    {
        m_trManager.closeFiles();
    }

    @Override
    public void openFiles() throws StorageException
    {
        m_trManager.openFiles();
    }

    static File fileForName(File dataDir, EntityName entityName, boolean checkParent, Mapper mapper) throws StorageException
    {
        if (checkParent)
        {
            File parentDir = null;
            try
            {
                parentDir = fileForName(dataDir, new EntityName(entityName.getParent()), false, null);
            }
            catch (ConfigException e) 
            {
                throw new Error(e.toString(), e);
            }
            if (!parentDir.exists())
            {
                throw new ParentDirDoesNotExistException("Directory '" + parentDir.getName() +  "' does not exist.");
            }
        }
        if (mapper == null)
        {
            return new File(dataDir, entityName.getName().substring(1));
        }
        else
        {
            return mapper.getFile(dataDir, entityName);
        }
    }

    static boolean directoryExists(File dataDir, EntityName dirName)
    {
        File dir = null;
        try
        {
            dir = fileForName(dataDir, dirName, false, null);
        }
        catch (StorageException e)
        {
            return false;
        }

        if (dir.exists())
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    static void createDirectory(File dataDir, EntityName dirName, boolean createParent, FSTransactionManager trMngr) throws StorageException
    {
        File newDir = null;
        try
        {
            newDir = fileForName(dataDir, dirName, true, null);
        }
        catch (StorageException e)
        {
            if (!createParent)
            {
                throw e;
            }
            else
            {
                // Create the paranet directory first...
                try
                {
                    createDirectory(dataDir, new EntityName(dirName.getParent()), createParent, trMngr);
                }
                catch (ConfigException ce) 
                {
                    throw new Error(ce.toString(), ce);
                }

                //... And no can create the new directory file...
                newDir = fileForName(dataDir, dirName, false, null);
            }
        }

        if (trMngr != null)
        {
            trMngr.newDirectory(dirName.getName());
        }
        if (!newDir.mkdir())
        {
            throw new StorageException("Cannot create directory '" + dirName.getName() + "'.");
        }
    }

   //helper functions used for file packing
   //////////////////////////////////////////

    private EntityName getEntity(String name)
    {
        EntityName entity = null;
        try{ entity = new EntityName(name); }
        catch(Exception e) {}
        return entity;
    }

    private HashMap groupElements(EntityName checkName, IDirElement[] elements)
        throws StorageException
    {
        HashMap groups = new HashMap();
        String parentName = checkName.getParent();

        for (int i = 0; i < elements.length; i++)
        {
            ArrayList list = null;
            EntityName name = getEntity(elements[i].getIdentity().getName());
            if (!parentName.equals(name.getParent()))
            {
                throw new StorageException("All the elements must reside in the same directory.");
            }
            File file = m_mapper.getFile(m_dataDir, name);
            if (groups.containsKey(file))
            {
                list = (ArrayList)groups.get(file);
            }
            else
            {
                list = new ArrayList();
                groups.put(file, list);
            }
            list.add(elements[i]);

        }
        return groups;
    }

   private HashMap groupElementNames(EntityName[] names) throws StorageException
    {
        if (names.length == 0)
        {
            return new HashMap();
        }

        EntityName checkName = names[0];

        HashMap groups = new HashMap();
        String parentName = checkName.getParent();

        for (int i = 0; i < names.length; i++)
        {
            ArrayList list = null;
            if (!parentName.equals(names[i].getParent()))
            {
                throw new StorageException("All the elements must reside in the same directory.");
            }
            File file = m_mapper.getFile(m_dataDir, names[i]);
            if (groups.containsKey(file))
            {
                list = (ArrayList)groups.get(file);
            }
            else
            {
                list = new ArrayList();
                groups.put(file, list);
            }
            list.add(names[i]);

        }
        return groups;
    }


    private class Mapper
    {
        private int m_index;

        Mapper()
        {
            m_index = 64; // Optimal looks like 64 or 128
        }

        File getFile(File dataDir, EntityName entityName)
        {
            if (PackedDirUtil.underPackedDir(entityName))
            {
                 String path = entityName.getParentEntity().createChild(hashFunction(entityName.getBaseName())).getName();
                 return new File(dataDir, path.substring(1));
            }
            else
            {
                return new File(dataDir,entityName.getName().substring(1));
            }
        }

        private String hashFunction(String elementName)
        {
            int fileIndex = elementName.hashCode();

            if (fileIndex < 0)
            {
                fileIndex *= -1;
            }

            int a0 = fileIndex;
            int a1 = a0 / 256;
            int a2 = a1 / 256;
            int a3 = a2 / 256;

            fileIndex = ((a0 & 0x000000FF) ^ (a1 & 0x000000FF) ^ (a2 & 0x000000FF) ^ (a3 & 0x000000FF)) % m_index;

            return "_MF" + (new Integer(fileIndex)).toString();
        }

    }

    @Override
    public void closeTransactionManager()
    {

    }

    // method added when it was required for PSEStorage. Online backup
    // with FSStorage doesn't happen at this level, so nothing to do here.
    @Override
    public void backup(String backupDir)
    {

    }

  /**** Collection methods are not supported by the FS implementation of IStorage ****/
  @Override
public String[] collectionToStringArray(String collectionName) throws StorageException
  {
      throw new StorageException("Not Implemented");
  }

  @Override
public void removeFromCollection(String collectionName, String item) throws StorageException
  {
      throw new StorageException("Not Implemented");
  }

  @Override
public void removeFromCollection(String collectionName, String[] items) throws StorageException
  {
      throw new StorageException("Not Implemented");
  }

  @Override
public void addToCollection(String collectionName, String item) throws StorageException
  {
      throw new StorageException("Not Implemented");
  }

  @Override
public void createCollectionIfNotCreated(String collectionName) throws StorageException
  {
      throw new StorageException("Not Implemented");
  }
}

