package com.sonicsw.mf.framework.directory.storage.fs;
import java.io.EOFException;
import java.io.File;
import java.io.FileDescriptor;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.framework.directory.storage.StorageException;


// The File System Storage transaction manager is used by FSStorage to make sequence of operations (new elements creation,
// element replacement and element deletion) transactional.
final public class FSTransactionManager
{
    private final static String NOTES_FILE_NAME = "notes";
    private final static String BEFORE_IMAGE_DIR_NAME = "before_image";

    private File m_tManagerDir;
    private File m_dataDir;
    private File m_beforeImageDir;
    private TransactionNotes m_trNotes;
    private boolean m_doSync;
    private boolean m_transactionOn;

    // If tManagerDir does not exist it's eaither because it's the first time, or the FSStorgae using this FSTransactionManager
    // never uses transactions (when it is used for persistent cache, for example). 
    FSTransactionManager(File tManagerDir, File dataDir,  boolean doSync) throws StorageException
    {
        m_tManagerDir = tManagerDir;
        m_dataDir = dataDir;
        m_beforeImageDir = null;
        m_trNotes = null;
        m_doSync = doSync;
        m_transactionOn = false;
        m_beforeImageDir = new File(m_tManagerDir, BEFORE_IMAGE_DIR_NAME);
        if (m_tManagerDir.exists())
        {
            m_trNotes = new TransactionNotes(new File(m_tManagerDir, NOTES_FILE_NAME), m_doSync);
        }
    }

    void startTransaction() throws StorageException
    {
        if (m_transactionOn)
         {
            throw new Error(); // Should not happen
        }

        if (!m_tManagerDir.exists())
        {
            if (!m_tManagerDir.mkdir())
            {
                throw new StorageException("Cannot create the transaction manager directory '" + m_tManagerDir.getPath() + "'.");
            }
        }

        if (!m_beforeImageDir.exists())
        {
            if (!m_beforeImageDir.mkdir())
            {
                throw new StorageException("Cannot create the transaction manager directory '" + m_beforeImageDir.getPath() + "'.");
            }
        }

        if (m_trNotes == null)
        {
            m_trNotes = new TransactionNotes(new File(m_tManagerDir, NOTES_FILE_NAME), m_doSync);
        }

        m_transactionOn = true;
    }

    void newDirectory(String newDirName) throws StorageException
    {
        if (!m_transactionOn)
        {
            return;
        }

         m_trNotes.addNewDirectory(newDirName);
    }

    void deleteDirectory(String deletedDirName) throws StorageException
    {
        if (!m_transactionOn)
        {
            return;
        }

         m_trNotes.addDeletedDirectory(deletedDirName);
    }

    void newElement(String newElementName) throws StorageException
    {
        if (!m_transactionOn)
        {
            return;
        }

         m_trNotes.addNewElement(newElementName);
            
    }

    void deleteElement(EntityName deletedElementName) throws StorageException
    {
        if (!m_transactionOn)
        {
            return;
        }

        String elementName = deletedElementName.getName();

        // if we already have a note (and before image - if the element was deleted) for this element - we don't do anything
        if(!m_trNotes.addDeletedElement(elementName))
        {
            return;
        }

        String dirName = deletedElementName.getParent();
        EntityName dirNameE = getParsedName(dirName);
        
        if (!FSStorage.directoryExists(m_beforeImageDir, dirNameE))
        {
            FSStorage.createDirectory(m_beforeImageDir, dirNameE, true, null);
        }

        File dataImage = new File(m_dataDir, elementName.substring(1));
        if (!dataImage.exists())
        {
            return;
        }

        File beforeImage = new File(m_beforeImageDir, elementName.substring(1));
        try
        {
            if (!dataImage.renameTo(beforeImage))
            {
                throw new StorageException("Failed creating before image for '" + dataImage.getPath() + "'.");
            }
        }
        catch (Exception e)
        {
            throw new StorageException("Failed creating before image for '" + dataImage.getPath() + "'.", e);
        }

    }

    void commit() throws StorageException
    {
        String[] deletedElements = m_trNotes.getDeletedElements();

        // Deletes the before image files
        for (int i = 0; i < deletedElements.length; i++)
        {
            FSStorage.fileForName(m_beforeImageDir, getParsedName(deletedElements[i]), false, null).delete();
        }

        m_trNotes.reset();
        m_transactionOn = false;
    }

    void close() throws StorageException
    {
        if (m_trNotes != null)
        {
            m_trNotes.close();
        }
    }

    void closeFiles() throws StorageException
    {
        if (m_trNotes != null)
        {
            m_trNotes.closeFile();
        }
    }

    void openFiles() throws StorageException
    {
        if (m_trNotes != null)
        {
            m_trNotes.openFile();
        }
    }

    void rollback() throws StorageException
    {
        // rollback might be called even if m_trNotes was never initiated since the transaction manager
        // does not distingiush between NO-TRANSACTION and ROLLEDBACK-TRANSACTION
        if (m_trNotes == null)
        {
            return;
        }

        ArrayList dirEvents = m_trNotes.getDirectoryEvents();
        boolean hasDirEvents = dirEvents.size() > 0;

        // Re-create all the directories deleted by the transaction in reverse order
        if (hasDirEvents && !m_trNotes.hasDirectoryCreateNotes())
        {
            for (int i = dirEvents.size() - 1; i >= 0; i--)
            {
                 String dirName = (String)dirEvents.get(i);
                 FSStorage.fileForName(m_dataDir, getParsedName(dirName), false, null).mkdir();
            }
        }

        String[] newElements = m_trNotes.getNewElements();
        for (int i = 0; i < newElements.length; i++)
        {
            FSStorage.fileForName(m_dataDir, getParsedName(newElements[i]), false, null).delete();
        }

        String[] deletedElements = m_trNotes.getDeletedElements();
        for (int i = 0; i < deletedElements.length; i++)
        {
            File beforeImage = new File(m_beforeImageDir, deletedElements[i].substring(1));

            if (!beforeImage.exists())
            {
                continue;
            }

            File dataImage = new File(m_dataDir, deletedElements[i].substring(1));
            dataImage.delete();
            if (!beforeImage.renameTo(dataImage))
            {
                throw new StorageException("rollback: Failed restoring '" + dataImage.getPath() + "'.");
            }
        }

        // Delete all the directories created by the transaction in reverse order
        if (hasDirEvents && m_trNotes.hasDirectoryCreateNotes())
        {
            for (int i = dirEvents.size() - 1; i >= 0; i--)
            {
                 String dirName = (String)dirEvents.get(i);
                 FSStorage.fileForName(m_dataDir, getParsedName(dirName), false, null).delete();
            }
        }


        m_trNotes.reset();
        m_transactionOn = false;
    }

    private EntityName getParsedName(String name)
    {
         try
         {
              return new EntityName(name);
         }
         catch (ConfigException e)
         {
              throw new Error(); // Should never happen
         }
    }

    private class TransactionNotes
    {
        private final static char DELETED_ELEMENT_TYPE = 'd';
        private final static char DELETED_DIR_TYPE = 'e';
        private final static char NEW_ELEMENT_TYPE = 'n';
        private final static char NEW_DIR_TYPE = 'o';

        private File m_notesFilePath;
        private boolean m_doSync;
        private RandomAccessFile m_notesFile;
        private HashMap m_newElements;
        private HashMap m_deletedElements;
        private ArrayList m_dirEvents;
        private boolean m_createDirEvents;
        private FileDescriptor m_notesDescriptor;

        TransactionNotes(File notesFilePath, boolean doSync) throws StorageException
        {
            m_notesFilePath = notesFilePath;
            m_doSync = doSync;
            m_newElements = new HashMap();
            m_deletedElements = new HashMap();
            m_dirEvents = new ArrayList();
            m_createDirEvents = false;

            openFile();

            // Get all the notes
            while(true)
            {
                String note = readNextNote();
                if (note == null || note.length() == 0)
                {
                    break;
                }
                char noteType = note.charAt(0);
                if (noteType == NEW_ELEMENT_TYPE)
                {
                    m_newElements.put(note.substring(1), "");
                }
                else if (noteType == DELETED_ELEMENT_TYPE)
                {
                    m_deletedElements.put(note.substring(1), "");
                }
                else
                {
                    if (noteType == NEW_DIR_TYPE)
                    {
                        m_createDirEvents = true;
                    }
                   
                    m_dirEvents.add(note.substring(1));
                }
            }
        }

        void close()  throws StorageException
        {
            closeFile();
        }

        final void openFile() throws StorageException
        {
            try
            {
                m_notesFile = new RandomAccessFile(m_notesFilePath, "rw");
                m_notesDescriptor = m_notesFile.getFD();
            }
            catch (Exception e)
            {
                throw new StorageException(e.toString(), e);
            }
        }

        void closeFile()  throws StorageException
        {
            try
            {
                if (m_notesFile != null)
                {
                    m_notesFile.close();
                }

            }
            catch (Exception e)
            {
                throw new StorageException(e.toString(), e);
            }
        }


        void reset()  throws StorageException
        {
            m_newElements = new HashMap();
            m_deletedElements = new HashMap();
            m_dirEvents = new ArrayList();
            m_createDirEvents = false;


            try
            {
                m_notesFile.setLength(0);
                if (m_doSync)
                {
                    m_notesDescriptor.sync();
                }

            }
            catch (Exception e)
            {
                throw new StorageException(e.toString(), e);
            }
        }

        String[] getDeletedElements()
        {
            return getElements(m_deletedElements);
        }

        String[] getNewElements()
        {
            return getElements(m_newElements);
        }

        ArrayList getDirectoryEvents()
        {
            return m_dirEvents;
        }

        boolean hasDirectoryCreateNotes()
        {
            return m_createDirEvents;
        }

        private String[] getElements(HashMap notesTable)
        {
            String[] result = new String[notesTable.size()];
            Iterator iterator = notesTable.keySet().iterator();
            for(int i = 0; i < result.length; i++)
            {
                result[i] = (String)iterator.next();
            }
            return result;
        }


        // Returns null when there are no more notes
        private String readNextNote() throws StorageException
        {
             try
             {
                  return m_notesFile.readUTF();
             }
             catch (EOFException e)
             {
                  return null;
             }
             catch (IOException e)
             {
                 throw new StorageException(e.toString(), e);
             }
        }

        void addDeletedDirectory(String dirName) throws StorageException
        {
            addDirectory(dirName, DELETED_DIR_TYPE);
        }

        void addNewDirectory(String dirName) throws StorageException
        {
            addDirectory(dirName, NEW_DIR_TYPE);
        }

        void addDirectory(String dirName, char noteType) throws StorageException
        {
            if (noteType == NEW_DIR_TYPE)
            {
                if (m_dirEvents.size() > 0 && !m_createDirEvents)
                {
                    throw new IllegalStateException("Directory creation & deletion cannot be mixed in a single transaction.");
                }
                m_createDirEvents = true;
            }

            m_dirEvents.add(dirName);

            try
            {
                m_notesFile.writeUTF(noteType + dirName);
                if (m_doSync)
                {
                    m_notesDescriptor.sync();
                }
            }
            catch (IOException e)
            {
                throw new StorageException(e.toString(), e);
            }

        }

        boolean addDeletedElement(String elementName) throws StorageException
        {
            return addElement(elementName, DELETED_ELEMENT_TYPE);
        }

        boolean addNewElement(String elementName) throws StorageException
        {
            return addElement(elementName, NEW_ELEMENT_TYPE);
        }

        boolean addElement(String elementName, char noteType) throws StorageException
        {
            if (hasNoteFor(elementName))
            {
                return false;
            }

            if (noteType == DELETED_ELEMENT_TYPE)
            {
                m_deletedElements.put(elementName, "");
            }
            else
            {
                m_newElements.put(elementName, "");
            }

            try
            {
                m_notesFile.writeUTF(noteType + elementName);
                if (m_doSync)
                {
                    m_notesDescriptor.sync();
                }
            }
            catch (IOException e)
            {
                throw new StorageException(e.toString(), e);
            }

            return true;
        }

        private boolean hasNoteFor(String elementName)
        {
            return m_deletedElements.get(elementName) != null || m_newElements.get(elementName) != null;
        }

    }


    
}
