package com.sonicsw.mf.framework.directory.storage.fs;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.HashMap;

import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.agent.EncryptionException;
import com.sonicsw.mf.framework.agent.PBE;
import com.sonicsw.mf.framework.directory.ILogger;
import com.sonicsw.mf.framework.directory.storage.DSEncryptionException;
import com.sonicsw.mf.framework.directory.storage.StorageException;

final class FileManager
{
    private String m_password = null;
    private FSTransactionManager m_trManager = null;
    private ILogger m_logger;

    FileManager(String password, FSTransactionManager manager)
    throws Exception
    {
        m_password = password;
        m_trManager = manager;
        m_logger = null;
    }

    void setLogger(ILogger logger)
    {
        m_logger = logger;
    }


    /** Returns the element stored in the file.
    *   Return null if the file does not exist or the element is not in the file.
    */
    IDirElement getElement(File file, EntityName elementName) throws StorageException
    {
        IdsAndElements idsAndElements = readFile(file, false);
        return (IDirElement)idsAndElements.elements.get(elementName.getBaseName());
    }


    /** Returns all the elements in the file.
    *   Returns a zero-length array if the file is not there or the file is empty.
    */
    IDirElement[] getElements(File file) throws StorageException
    {
        IdsAndElements idsAndElements = readFile(file, false);
        return (IDirElement[])idsAndElements.elements.values().toArray(new IDirElement[idsAndElements.elements.size()]);
    }


    /** Returns the ids of all the elements in the file (reads only the the ids map)
    *   Returns a zero-length array if the file is not there or the file is empty.
    */
    IElementIdentity[] getIds(File file) throws StorageException
    {
        IdsAndElements idsAndElements = readFile(file, true);
        return (IElementIdentity[])idsAndElements.ids.values().toArray(new IElementIdentity[idsAndElements.ids.size() ]);
    }

    /** Deletes the element (if exists).
    *   NOOP if the element or the file does not exist
    */
    IElementIdentity deleteElement(File file, EntityName elementName) throws StorageException
    {
        String name = elementName.getBaseName();
        IdsAndElements idsAndElements = readFile(file, false);
        IElementIdentity id = (IElementIdentity)idsAndElements.ids.get(name);

        if (id == null)
        {
            return null;
        }

        idsAndElements.ids.remove(name);
        idsAndElements.elements.remove(name);

        updateFile(file, idsAndElements.ids, idsAndElements.elements, elementName.getParentEntity());
        return id;

    }

    /** Deletes the elements which exist
    */
    IElementIdentity[] deleteElements(File file, EntityName[] elementNames) throws StorageException
    {
        IdsAndElements idsAndElements = readFile(file, false);

        EntityName elementName = null;
        ArrayList deletedIds = new ArrayList();
        for (int i = 0; i < elementNames.length; i++)
        {
            String name = elementNames[i].getBaseName();

            if (idsAndElements.ids.get(name) == null)
            {
                continue;
            }

            // We need that for the updateFile() call
            elementName = elementNames[i];

            Object removedID = idsAndElements.ids.remove(name);
            if (removedID != null)
            {
                deletedIds.add(removedID);
            }
            idsAndElements.elements.remove(name);
        }

        // If elementName is null we didn't delete anything and the whole operation is a noop
        if (elementName != null)
        {
            updateFile(file, idsAndElements.ids, idsAndElements.elements, elementName.getParentEntity());
        }

        IElementIdentity[] result = new IElementIdentity[deletedIds.size()];
        deletedIds.toArray(result);
        return result;

    }


    /** Put the element in the file (create the file if doesn't exit).
    *   Replace the element if already in the file.
    */
    void setElement(File file, IDirElement element) throws StorageException
    {
       setElementsInternal(file, new IDirElement[] {element}, getParentDir(element));
    }


    /** Equivalent to multiple setElement. The file read and written only
    *   once (so it doesn't call setElement multiple times).
    */
    void setElements(File file, IDirElement[] elements) throws StorageException
    {
        if (elements.length == 0)
        {
            throw new IllegalArgumentException();
        }

        setElementsInternal(file, elements, getParentDir(elements[0]));
    }


    private void setElementsInternal(File file, IDirElement[] newElements, EntityName dirName)
        throws StorageException
    {

        IdsAndElements idsAndElements = readFile(file, false);

        for (int i = 0 ; i < newElements.length; i++)
        {
             IElementIdentity id = newElements[i].getIdentity();
             String name = null;
             try
             {
                 name = (new EntityName(id.getName())).getBaseName();
             }
             catch (Exception e)
             {
                 throw new Error(e.toString()); // Should never happen
             }


             idsAndElements.ids.put(name, id);
             idsAndElements.elements.put(name, newElements[i]);
        }

        updateFile(file, idsAndElements.ids, idsAndElements.elements, dirName);

    }

    //writes data to a file, as two HashMap objects-> table of IDS, table of Elements
    private void updateFile(File file, HashMap ids, HashMap elements, EntityName dirName) throws StorageException
    {
        try
        {
            EntityName fileName = dirName.createChild(file.getName());

            if(ids.isEmpty()) //only for delete mode
            {
                m_trManager.deleteElement(fileName);
                file.delete();
                return;
            }

            if (file.exists())
            {
                m_trManager.deleteElement(fileName);
            }
            else
            {
                m_trManager.newElement(fileName.getName());
            }

            FileOutputStream fos = new FileOutputStream(file);
            BufferedOutputStream fileOut = new BufferedOutputStream (fos);
            ObjectOutputStream objectOS = new ObjectOutputStream (fileOut);

            if (m_password == null)
            {
                objectOS.writeObject(ids);
                objectOS.writeObject(elements);
            }
            else
            {
                objectOS.writeObject(convertObjectToBytes(ids));
                objectOS.writeObject(convertObjectToBytes(elements));
            }
            objectOS.close();
        }
        catch (Exception e)
        {
            throw new StorageException("Could not write file'" + "':" + file.getPath() + " " + e.toString());
        }

    }

    private IdsAndElements readFile(File file, boolean idsOnly) throws StorageException
    {
        ObjectInputStream objectS = null;

        if (!file.exists() || file.isDirectory())
        {
            return new IdsAndElements(true);
        }

        try
        {
            IdsAndElements idsAndElements = new IdsAndElements();
            BufferedInputStream is = new BufferedInputStream(new FileInputStream(file));
            objectS = new ObjectInputStream(is);

            if (m_password == null)
            {
                Object idsObject = objectS.readObject();
                if (idsObject instanceof byte[])
                {
                    throw new DSEncryptionException("Could not access the Directory Service because " +
                                                     "of a wrong or a lacking encryption password.");
                }
                idsAndElements.ids = (HashMap)idsObject;
            }
            else
            {
                idsAndElements.ids = readEncryptedData(objectS);
            }

            if (idsOnly)
            {
                objectS.close();
                return idsAndElements;
            }

            idsAndElements.elements = m_password == null ? (HashMap)objectS.readObject() : readEncryptedData(objectS);
            objectS.close();
            return idsAndElements;
        }
        catch(Exception ex)
        {
            try { if (objectS != null) objectS.close(); } catch(Exception ioe) { }

            if (ex instanceof EncryptionException)
            {
                throw new DSEncryptionException("Failed to read encrypted data", (EncryptionException)ex);
            }

            String msg = "Could not read file '" + file.getName();

            // idsOnly is 'true' when the DS recovers and scans the entire tree. In that case
            // we want to keep going even if there are files it cannot read. On the other hand a failure
            // to read a specific file during operation should cause an exception. We no longer delete a file
            // we cannot read, assuming it's corrupt (Sonic00019077). There there are cases when we cannot
            // read a 'good' file because the process runs out of file descriptors, for example.
            if (!idsOnly)
            {
                StorageException storageException = new StorageException(msg);
                storageException.initCause(ex);
                throw storageException;
            }

            msg += ", trace follows...";
            if (m_logger != null)
            {
                m_logger.logMessage(msg, ex, Level.SEVERE);
            }
            else
            {
                 System.err.println(msg);
                 ex.printStackTrace();
            }

            return new IdsAndElements(true);
        }

    }

    EntityName getParentDir(IDirElement element)
    {
        EntityName name = null;
        String elmntName =  element.getIdentity().getName();
        try { name = new EntityName(elmntName);}
        catch (Exception ex){throw new Error();} //shouldn't happen
        return name.getParentEntity();
    }

    //decrypts Element's data
    private HashMap readEncryptedData(ObjectInputStream input)
    throws Exception
    {
        Object inS = input.readObject();
        if (inS instanceof HashMap)
        {
            throw new DSEncryptionException("Could not access the Directory Service because of a wrong or a lacking encryption password.");
        }

        byte[] bytes = PBE.decrypt((byte[])inS, m_password);
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        ObjectInputStream object = new ObjectInputStream(in);

        return (HashMap)object.readObject();
    }

    //encrypts Element's data
    private byte[] convertObjectToBytes(Object data)
    throws Exception
    {
         ByteArrayOutputStream out = new ByteArrayOutputStream();
         ObjectOutputStream objectOut = new ObjectOutputStream(out);
         objectOut.writeObject(data);

         byte[] encBytes = PBE.encrypt(out.toByteArray(), m_password);
         out.close();

         return encBytes;
    }

    private class IdsAndElements
    {
        HashMap ids;
        HashMap elements;

        IdsAndElements()
        {
            this (false);
        }

        IdsAndElements (boolean emptyTables)
        {
            ids = null;
            elements = null;
            if (emptyTables)
            {
                ids = new HashMap();
                elements = new HashMap();
            }
        }

    }
}

