/* Copyright (c) 2003-2004 Sonic Software Corporation. All Rights Reserved. */

package com.sonicsw.mf.framework.directory.storage.pse;

import com.odi.ClassInfo;
import com.odi.Database;
import com.odi.GenericObject;
import com.odi.IPersistent;
import com.odi.IPersistentHooks;
import com.odi.ObjectStore;
import com.odi.imp.ObjectReference;
import com.odi.util.HashPersistent;

/* Code and comments taken directly from progress.message.dbsc.pse.pc on 5/1/2006 */

/**
 * This is a wrapper class for primitive byte arrays, used to provide
 * a level of indirection between small "header" objects and large
 * "body" objects.  For example, a Message object may contain
 * meta-data fields such as id and size, and then also have a
 * reference to a separate BLOB object containing the "body" or
 * serialized contents.  Many queries, and even update operations, may
 * only need to access header info, and not need to materialize the
 * body contents in-memory from the db.
 *
 * PSE will not fetch the body from the db when only the header is
 * accessed.  So, unnecessary db access isn't an issue in this case.
 * However, because PSE does need to initialize the java reference
 * from the header to the body object, it must at least create a
 * hollow (uninitialized) body object to refer to.  That hollow
 * object will be the size of the actual body contents.  So, even
 * though the body won't be initialized, it could still consume a
 * large amount of memory relative to the size of a much smaller
 * header object.  This unnecessary memory consumption could become a
 * significant issue in the case of queries over large collections of
 * header objects.
 *
 * Use of this byte array wrapper class avoids the memory consumption
 * issue because a small header object can just refer to a tiny,
 * hollow wrapper object when the contents of the large byte array
 * aren't needed.  The indirection prevents the allocation of large,
 * empty byte array objects that might never be initialized.
 *
 * Note that the wrapper pattern does introduce the overhead of one
 * extra object per header/body representation.  Since PSE performance
 * can be significantly affected by the total number of persistent
 * objects accessed, there is a penalty associated with avoiding the
 * extra memory usage.  There is a tradeoff between using this wrapper
 * pattern versus simply using a direct reference between header and
 * byte array.  So, your mileage may vary depending on the specific
 * use cases.
 */
public class ByteArrayWrapper
    implements IPersistent, IPersistentHooks
{

    public ByteArrayWrapper(byte bytes[])
    {
        ODITheHashCode = HashPersistent.getNextHashCode();
        _Bytes = bytes;
    }

    public byte[] getBytes()
    {
        if(ODIObjectState < 0)
        {
            ObjectStore.fetch(this);
        }
        return _Bytes;
    }

    @Override
    public void postInitializeContents()
    {
        ObjectStore.fetch(_Bytes);
    }

    @Override
    public void preFlushContents()
    {
        if(!ObjectStore.isPersistent(this))
        {
            return;
        }
        if(!ObjectStore.isPersistent(_Bytes))
        {
            ObjectStore.migrate(_Bytes, Database.of(this), false);
        }
        ObjectStore.evict(_Bytes, 2);
    }

    @Override
    public void preDestroyPersistent()
    {
        if(ODIObjectState < 0)
        {
            ObjectStore.fetch(this);
        }
        if(_Bytes != null)
        {
            ObjectStore.destroy(_Bytes);
        }
        if((ODIObjectState & 2) != 0)
        {
            ObjectStore.dirty(this);
        }
        _Bytes = null;
    }

    @Override
    public ObjectReference ODIgetRef()
    {
        return ODIref;
    }

    @Override
    public void ODIsetRef(ObjectReference objectreference)
    {
        ODIref = objectreference;
    }

    @Override
    public byte ODIgetState()
    {
        return ODIObjectState;
    }

    @Override
    public void ODIsetState(byte byte0)
    {
        ODIObjectState = byte0;
    }

    @Override
    protected synchronized Object clone()
        throws CloneNotSupportedException
    {
        if(ODIObjectState < 0)
        {
            ObjectStore.fetch(this);
        }
        ByteArrayWrapper bytearraywrapper = (ByteArrayWrapper)super.clone();
        bytearraywrapper.ODITheHashCode = bytearraywrapper.ODIComputeHashCode();
        bytearraywrapper.ODIref = null;
        bytearraywrapper.ODIObjectState = 0;
        return bytearraywrapper;
    }

    @Override
    public void preClearContents()
    {
    }

    @Override
    public boolean equals(Object o) {
        return super.equals(o);
    }
    
    @Override
    public int hashCode()
    {
        if(ODIObjectState < 0)
        {
            ObjectStore.fetch(this);
        }
        return ODITheHashCode;
    }

    private int ODIComputeHashCode()
    {
        return HashPersistent.getNextHashCode();
    }

    @Override
    public void initializeContents(GenericObject genericobject)
    {
        ClassInfo classinfo = myOdiClassInfoInstance;
        ODITheHashCode = genericobject.getIntField(1, classinfo);
        _Bytes = (byte[])genericobject.getArrayField(2, classinfo);
    }

    @Override
    public void flushContents(GenericObject genericobject)
    {
        ClassInfo classinfo = myOdiClassInfoInstance;
        genericobject.setIntField(1, ODITheHashCode, classinfo);
        genericobject.setArrayField(2, _Bytes, classinfo);
    }

    @Override
    public void clearContents()
    {
        ODITheHashCode = 0;
        _Bytes = null;
    }

    public ByteArrayWrapper(ClassInfo classinfo)
    {
        ODITheHashCode = HashPersistent.getNextHashCode();
    }

    private static final ClassInfo getClassInfoInstance()
    {
        try
        {
            return ClassInfo.get(Class.forName("com.sonicsw.mf.framework.directory.storage.pse.ByteArrayWrapper"));
        }
        catch(ClassNotFoundException classnotfoundexception)
        {
            return null;
        }
    }

    private int ODITheHashCode;
    private byte _Bytes[];
    private transient ObjectReference ODIref;
    public transient byte ODIObjectState;
    static ClassInfo myOdiClassInfoInstance = getClassInfoInstance();

}