package com.sonicsw.mf.framework.directory.storage.pse;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import com.odi.Database;
import com.odi.DatabaseNotFoundException;
import com.odi.ObjectStore;
import com.odi.Session;
import com.odi.Transaction;
import com.odi.util.OSTreeSet;
import com.odi.util.query.Query;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.impl.Blob;
import com.sonicsw.mf.common.config.impl.DirIdentity;
import com.sonicsw.mf.common.config.impl.Element;
import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.directory.ILogger;
import com.sonicsw.mf.framework.directory.storage.IStorage;
import com.sonicsw.mf.framework.directory.storage.ParentDirDoesNotExistException;
import com.sonicsw.mf.framework.directory.storage.StorageException;


public class PSEStorage implements IStorage
{
	Database m_store;
	Session m_session;
	String m_domain;
	OSTreeSet m_data;
	OSTreeSet m_directories;
	String m_domainDir;
	String m_tempDir;
	String m_password = null;
	String m_dbPath = null;
	ILogger m_logger;
    HashMap m_collections;

	Transaction m_transaction = null;

	boolean m_sharedDB = false;
	boolean m_startsUpdateTransactions = false;

	public static String DB_EXTENSION = ".odb";
	String DB_DATA_ROOT_NAME = "data";
	String DB_DIRECTORIES_ROOT_NAME = "directories";
	String DB_ELEMENT_NAME_ATTRIBUTE = "getElementName()";
	String DB_PARENT_DIRECTORY_ATTRIBUTE = "getParentDirectory()";
	String DB_DIRECTORY_NAME_ATTRIBUTE = "getDirectoryName()";
	String DB_GET_PARENT_DIR_NAME_ATTRIBUTE = "getParentDirectoryName()";
	String DB_GROUP_ELEMENT_NAME = "getGroupName()";
	Class DB_ELEMENT_CLASS = DSElement.class;
	Class DB_DIRECTORY_CLASS = DSDirectory.class;

    String DB_COLLECTION_NAME_ATTRIBUTE = "getName()";
    Class DB_COLLECTION_CLASS = CollectionElement.class;

	private static int COPY_CHUNK_SIZE = 1000000;
	private static String TEMPFILES_DIR_NAME = "tempFiles";

	private static boolean DEBUG;
	private static boolean DEBUG_TRANSACTION;
	
	private static boolean DSDUMP = Boolean.getBoolean("DSDUMP");
	private static HashMap m_corruptIds = new HashMap();

	static
	{
		String debug = System.getProperty("PSE_DEBUG", "false");
		DEBUG = (new Boolean(debug)).booleanValue();
		debug = System.getProperty("PSE_DEBUG_TRANSACTION", "false");
		DEBUG_TRANSACTION = DEBUG || (new Boolean(debug)).booleanValue();
	}

	/*public static void main(String[]args)
	{
		PSEStorage store1 = null;
		PSEStorage store2 = null;
		try
		{
			store1 = new PSEStorage(".", "Domain1", "data", null, null);
			basicElementTest(store1);
			//blobTest(store1);
			//store2 = new PSEStorage(store1);
			//twoStoresOneDBTest(store1, store2);
			//testDBCleanupDirs(store1);
			//testDBCleanupElements(store1);
			File backupDir = new File(".", "Domain1Backup");
			backupDir.mkdirs();
			File backupDataDir = new File(backupDir, "data");
			store1.backup(backupDataDir.getCanonicalPath());
			store2 = new PSEStorage(".", "Domain1Backup", "data", null, null);
			IIdentity[] allIds = store2.listAll(new EntityName("/a/b"));
			System.out.println("Printing listAll(\"/a/b\") in the backup database");
			for (int i=0; i<allIds.length; i++)
				System.out.print(allIds[i].getName() + " ");
			System.out.println();
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
		finally
		{
			try {if (store1 != null) store1.close();} catch (Exception e2) {} // oh well
			try {if (store2 != null) store2.close();} catch (Exception e2) {}
		}
	}*/


	private static void testDBCleanupDirs(PSEStorage store) throws Exception
	{
		// print the contents of an empty DB
		store.showDB("EmptyDB.txt");
		store.startTransaction();
		store.createDirectory(new EntityName("/firstLevel"));
		store.createDirectory(new EntityName("/firstLevel/secondLevel"));
		store.commitTransaction();
		store.showDB("DBWithDirs.txt");
		store.startTransaction();
		store.deleteDirectory(new EntityName("/firstLevel/secondLevel"));
		store.deleteDirectory(new EntityName("/firstLevel"));
		store.commitTransaction();
		store.showDB("DBAfterDeleteDirectories.txt");
	}

	private static void testDBCleanupElements(PSEStorage store) throws Exception
	{
		store.showDB("EmptyDB.txt");
		store.startTransaction();
		IDirElement el = ElementFactory.createElement("/Element1", "TYPE", "1");
		IAttributeSet topSet = el.getAttributes();
		topSet.setStringAttribute("StringAttribute", "StringValue");
		store.setElement(new EntityName("/Element1"), el);
		store.commitTransaction();
		store.showDB("DBWithElement.txt");
		store.startTransaction();
		byte[] blobPiece = new byte[1000000];
		for (int i=0; i<1000000; i++)
        {
            blobPiece[i]= (byte)65;
        }
		// blob with 10 chunks
		for (int i=0; i < 10; i++)
        {
            store.appendBlob(new EntityName("/Element1"), (byte[])blobPiece.clone(), i*1000000);
        }
		store.commitTransaction();
		store.showDB("DBWithElementAndBlob.txt");
		store.startTransaction();
		System.out.println("Blob size == " + store.getBlobSize(new EntityName("/Element1")));
		store.deleteBlob(new EntityName("/Element1"));
		store.commitTransaction();
		store.showDB("DBAfterDeleteBlob.txt");
		store.startTransaction();
		store.deleteElement(new EntityName("/Element1"));
		store.commitTransaction();
		store.showDB("DBAfterDeletedElements.txt");
	}

	private void showDB(String fileName) throws IOException
	{
		beginOSTransaction("showDB", ObjectStore.READONLY);
		PrintStream dbPrintStream = new PrintStream(new FileOutputStream(m_dbPath + "\\" + fileName));
		m_store.show(dbPrintStream, true, true);
		dbPrintStream.close();
	}

	private static void basicElementTest(PSEStorage store) throws Exception
	{
//		 TEST: create the directories first
		store.startTransaction();
		store.createDirectory(new EntityName("/a"));
		store.createDirectory(new EntityName("/a/b"));
		store.commitTransaction();
		EntityName el1Name = new EntityName("/a/b/c");
		// TEST: negative test for getElement
		IDirElement fromStore = store.getElement(el1Name);
		if (fromStore == null)
        {
            System.out.println("Negative getElement test worked");
        }
        else
        {
            System.out.println("*****Negative getElement test did not work");
        }
		IDirElement el1 = ElementFactory.createElement("/a/b/c", "MF_CONTAINER", "3.0");
		// TEST: setElement
		store.startTransaction();
		store.setElement(el1Name, el1, true);
		store.commitTransaction();
		// TEST: getElement
		el1 = store.getElement(el1Name);
		if (el1 != null)
        {
            System.out.println("Element came back from PSEStorage correctly, element name == " +
					el1.getIdentity().getName());
        }
        else
        {
            System.out.println("******New element could not be retrieved from store!!!");
        }
		// TEST: directoryExists
		if (store.directoryExists(new EntityName("/a")))
        {
            System.out.println("Directory /a exists");
        }
        else
        {
            System.out.println("***** Directory /a not found!!!");
        }
		// TEST: getElements
		IDirElement[] grouped = store.getElements(el1Name);
		if (grouped.length == 1)
        {
            System.out.println("getElements returned 1 element");
        }
        else
        {
            System.out.println("*****getElements did not return the right number of elements!!");
        }
		// TEST: setElements
		store.startTransaction();
		store.createDirectory(new EntityName("/a/b/_MFUsers"));
		IDirElement el2 = ElementFactory.createElement("/a/b/_MFUsers/derek", "MF_CONTAINER", "3.0");
		IDirElement el3 = ElementFactory.createElement("/a/b/_MFUsers/derekal", "MF_CONTAINER", "3.0");
		store.setElements(new IDirElement[] {el2, el3}, false);
		store.commitTransaction();
		grouped = store.getElements(new EntityName(el2.getIdentity().getName()));
		grouped = store.getElements(new EntityName(el3.getIdentity().getName()));
		// TEST: listAll
		IIdentity[] allIds = store.listAll(new EntityName("/a/b"));
		System.out.println("Printing listAll(\"/a/b\")");
		for (int i=0; i<allIds.length; i++)
        {
            System.out.print(allIds[i].getName() + " ");
        }
		System.out.println();
		// TEST: deleteElement
		store.startTransaction();
		store.deleteElement(el1Name);
		store.commitTransaction();
		if (store.getElement(el1Name) == null)
        {
            System.out.println("Element was deleted correctly");
        }
        else
        {
            System.out.println("******Element was not deleted correctly!!!");
        }
	}

	private static void blobTest(PSEStorage store) throws Exception
	{
		// small blob
		store.startTransaction();
		store.createDirectory(new EntityName("/a/blobs"));
		byte[] smallBlob = (new String("A small blob")).getBytes();
		for (int i=0; i<smallBlob.length; i++)
        {
            System.out.print(smallBlob[i] + " ");
        }
		System.out.println();
		EntityName blobName = new EntityName("/a/blobs/blob1");
		store.appendBlob(blobName, smallBlob, 0);
		store.commitTransaction();
		// get it back and check the contents
		byte[] smallBlobAgain = store.getBlob(blobName);
		for (int i=0; i<smallBlobAgain.length; i++)
        {
            System.out.print(smallBlob[i] + " ");
        }
		System.out.println();
		if (!(new String(smallBlobAgain)).equals("A small blob"))
        {
            System.out.println("Small blob worked correctly");
        }
        else
        {
            System.out.println("********* Small blob failed: " + new String(smallBlob));
        }

		// large blob
		FileInputStream fileStream = new FileInputStream("C:\\mgmttest_sandbox\\sonic\\Archives\\MF\\3.0\\MFcontainer.car");
		int fileSize = fileStream.available();
		boolean done = false;
		blobName = new EntityName("/a/blobs/MFcontainer.car");
		int blobIndex = 0;
		byte[] blobChunk;
		while (!done)
		{
			if (fileStream.available() >=COPY_CHUNK_SIZE)
            {
                blobChunk = new byte[COPY_CHUNK_SIZE];
            }
            else
			{
				blobChunk = new byte[fileStream.available()];
				done = true;
			}
			fileStream.read(blobChunk);
			store.startTransaction();
			store.appendBlob(blobName, blobChunk, blobIndex*COPY_CHUNK_SIZE);
			store.commitTransaction();
			blobIndex = blobIndex + 1;
		}
		// get it back and test it
		blobChunk = store.getBlob(blobName, COPY_CHUNK_SIZE, COPY_CHUNK_SIZE);
		if (blobChunk.length != COPY_CHUNK_SIZE)
        {
            System.out.println("********* Chunk size of large blob is not correct " + blobChunk.length);
        }
        else
        {
            System.out.println("Was able to get chunk of large blob");
        }
		// test blobToFile and check the file size
		File turnToFile = store.blobToFile(blobName);
		if (turnToFile.length() != fileSize)
        {
            System.out.println("******** blobToFile did not return the same size " +
					turnToFile.length() + " vs " + fileSize);
        }
        else
        {
            System.out.println("blobToFile returned a good file");
        }
		// now copy the blob!!
		EntityName newBlobName = new EntityName("/a/blobs/copiedBlob");
		store.startTransaction();
		store.copyBlob(blobName, newBlobName);
		store.commitTransaction();
		if (store.getBlobSize(blobName) == store.getBlobSize(newBlobName))
        {
            System.out.println("Copying blob worked");
        }
        else
        {
            System.out.println("****** copied blob is not the same size: copied == " +
					store.getBlobSize(newBlobName) + " original == " + store.getBlobSize(blobName));
        }
		// copy it again, make sure it works
		store.startTransaction();
		store.copyBlob(blobName, newBlobName);
		store.commitTransaction();
		if (store.getBlobSize(blobName) == store.getBlobSize(newBlobName))
        {
            System.out.println("Copying blob the second time worked");
        }
        else
        {
            System.out.println("****** copied blob the second time is not the same size: copied == " +
					store.getBlobSize(newBlobName) + " original == " + store.getBlobSize(blobName));
        }
		// and delete the blob
		store.startTransaction();
		store.deleteBlob(newBlobName);
		store.commitTransaction();
		if (store.getBlob(newBlobName) != null)
        {
            System.out.println("****** Was able to get a blob that I just deleted!");
        }
        else
        {
            System.out.println("Blob was deleted correctly");
        }
	}

	static void twoStoresOneDBTest(PSEStorage store1,PSEStorage store2)
		throws Exception
	{
		/*store1.getElement(new EntityName("/a/b/c"));
		store2.getElement(new EntityName("/a/b/c"));
		store1.startTransaction();
		store2.commitTransaction(); */
		store1.startTransaction();
		store2.createDirectory(new EntityName("/_MFSystem"));
		store2.createDirectory(new EntityName("/_MFSystem/stuff"));
		store2.setElement(new EntityName("/_MFSystem/system_element"),ElementFactory.createElement("/_MFSystem/system_element", "MF_CONTAINER", "3.0"));
		store1.commitTransaction();
		System.out.println("Done with twoStoresOneDBTest");
	}

	// use the same database, but a separate session
	public PSEStorage(PSEStorage existingStorage) throws StorageException
	{
		m_password = existingStorage.m_password;
		m_domainDir = existingStorage.m_domainDir;
		m_tempDir = existingStorage.m_tempDir;
		m_domain = existingStorage.m_domain;
		m_dbPath = existingStorage.m_dbPath;
		HashMap params = new HashMap();

	    params.put("NO_READ_LOCK", Boolean.TRUE);
		try
		{
			setSession();
			joinSession();
			try
			{
			    m_store = Database.open(m_dbPath, ObjectStore.UPDATE, params);
			    if (DEBUG)
                {
                    System.out.println("PSEStorage opened database with path " + m_dbPath
							+ " in session " + m_session);
                }
			    beginOSTransaction("PSEStorage", ObjectStore.UPDATE);
			    m_data = (OSTreeSet)m_store.getRoot(DB_DATA_ROOT_NAME);
			    m_directories = (OSTreeSet)m_store.getRoot(DB_DIRECTORIES_ROOT_NAME);
			    m_sharedDB = true;
				m_startsUpdateTransactions = true;

			}

			catch (DatabaseNotFoundException ex)
			{
				throw new StorageException("PSEStorage unable to open database in separate session " + ex.toString(), ex);
			}
			commitOSTransaction("PSEStorage", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
			//writeSchemas();
		}
		finally
		{

		}
	}

	public PSEStorage(String hostDirName, String domainName, String dataDirName, String password,
			HashMap properties)
		throws StorageException
	{
		// TODO probably move this to an init method after it grows
		m_password = password;
		File dbDir = new File(hostDirName, domainName);
		File dbPath = new File(dbDir, dataDirName + DB_EXTENSION);

		try
		{
			m_domainDir = dbDir.getCanonicalPath();
			m_dbPath = dbPath.getCanonicalPath();
			File tempDir = new File(m_domainDir, TEMPFILES_DIR_NAME);
			if (!tempDir.exists())
			{
				boolean madeTempDir = tempDir.mkdir();
				if (!madeTempDir)
                {
                    throw new StorageException("PSEStorage is unable to create the temporary file directory");
                }
			}
			m_tempDir = tempDir.getCanonicalPath();

			setSession();
			joinSession();
			HashMap params;
			if (properties != null)
            {
                params = new HashMap(properties);
            }
            else
            {
                params = new HashMap();
            }
		    params.put("NO_READ_LOCK", Boolean.TRUE);

			try
			{
				m_domain = domainName;
				if (DEBUG)
                {
                    System.out.println("PSEStorage trying to open database with path " + m_dbPath
							+ " in session " + m_session);
                }
			    m_store = Database.open(m_dbPath, ObjectStore.UPDATE, params);
			    if (DEBUG)
                {
                    System.out.println("PSEStorage opened database with path " + dbPath.getCanonicalPath()
							+ " in session " + m_session);
                }
			    beginOSTransaction("PSEStorage", ObjectStore.UPDATE);
			    m_data = (OSTreeSet)m_store.getRoot(DB_DATA_ROOT_NAME);
			    m_directories = (OSTreeSet)m_store.getRoot(DB_DIRECTORIES_ROOT_NAME);
			}
			catch (DatabaseNotFoundException ex)
			{
				m_store = Database.create(m_dbPath, ObjectStore.OWNER_WRITE, params);
				if (DEBUG)
                {
                    System.out.println("PSEStorage created DB with path " + dbPath.getCanonicalPath()
							+ " in session " + m_session);
                }
				beginOSTransaction("PSEStorage" ,ObjectStore.UPDATE);
			    // Create the database roots and give them appropriate values
			    m_store.createRoot(DB_DATA_ROOT_NAME, m_data = new OSTreeSet(m_store, DB_ELEMENT_CLASS, DB_ELEMENT_NAME_ATTRIBUTE));
			    m_store.createRoot(DB_DIRECTORIES_ROOT_NAME, m_directories = new OSTreeSet(m_store, DB_DIRECTORY_CLASS, DB_DIRECTORY_NAME_ATTRIBUTE));
			    // add the root directory
			    m_directories.add(new DSDirectory("/"));

			    m_data.addIndex(DSElement.class, DB_PARENT_DIRECTORY_ATTRIBUTE);
			    m_data.addIndex(DSElement.class, DB_GROUP_ELEMENT_NAME);
			    m_directories.addIndex(DSDirectory.class, DB_GET_PARENT_DIR_NAME_ATTRIBUTE);
			}
			// clean the temporary files directory when the database is opened, in case
			// for some reason temporary files are still hanging around.
			// This is not expected.
			cleanTempDir();
			// workaround for multiple sessions accessing this database, trying to
			// write to the schema String simultaneously

			commitOSTransaction("PSEStorage", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
			//writeSchemas();
		}
		catch (IOException ioEx)
		{
			throw new StorageException("Unable to open database " + dbPath, ioEx);
		}
	}

	private void cleanTempDir()
	{
		File tempDir = new File(m_tempDir);
		if (tempDir.exists())
		{
		    File[] tempFiles = tempDir.listFiles();
		    for (int i=0; i< tempFiles.length; i++)
		    {
			    boolean deleted = tempFiles[i].delete();
			    if (!deleted && m_logger != null)
                {
                    // probably not bad enough to throw an exception, but log the
				// fact that we weren't able to delete the temporary file
				    m_logger.logMessage("PSEStorage unable to remove temporary file " + tempFiles[i].getName(), Level.WARNING);
                }
		    }
		}
	}

	private void writeSchemas() throws StorageException
	{
		//create one of every kind of object and get rid of it

		try
		{
			beginOSTransaction("writeSchemas", ObjectStore.UPDATE);
			m_directories.add(new DSDirectory("/workaround"));
		    DSElement el = new DSElement("/workaround/writeSchemas", null, null);
		    DSBlob header = new DSBlob();
		    DSBlobChunk chunk = new DSBlobChunk((new String("bytes for the blob")).getBytes());
		    header.addChunk(chunk);
		    m_data.add(el);
		    commitOSTransaction("writeSchemas", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);

		    // now get rid of the elements

		    beginOSTransaction("writeSchemas", ObjectStore.UPDATE);
		    m_directories.remove(m_directories.getFromPrimaryIndex("/workaround"));
		    m_data.remove(m_data.getFromPrimaryIndex("/workaround/writeSchemas"));
		    commitOSTransaction("writeSchemas", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);

		}
		catch (StorageException e)
		{
			throw e;
		}
		catch (Exception e2)
		{
            throw new StorageException("PSEStorage unable to workaround schema string problem, see cause", e2);
		}
	}

	@Override
    public String getDomain()
	{
		return m_domain;
	}

    @Override
    public IDirElement getElement(EntityName elementName)
    throws StorageException
    {
        joinSession();
        if (DEBUG)
        {
            System.out.println("PSEStorage.getElement " + elementName.getName());
        }
        beginOSTransaction("getElement", ObjectStore.READONLY);
        DSElement el = (DSElement)m_data.getFromPrimaryIndex(elementName.getName());
        if (el != null)
        {
            if (DEBUG)
            {
                System.out.println("PSEStorage.getElement " + elementName.getName() + " element was not null");
            }
            try
            {
                return el.getIDirElement(m_password);
            }
            catch (Throwable t)
            {
                if (DSDUMP)
                {
                    m_corruptIds.put(elementName.getName(),  new Object[] {t, t.getStackTrace()});
                }
                else if (t instanceof StorageException)
                {
                    throw (StorageException)t;
                }
                else if (t instanceof Exception)
                {
                    if (DEBUG)
                    {
                        System.out.println("PSEStorage.getElement caught exception, trace follows: ");
                        t.printStackTrace();
                    }
                    StorageException se = new StorageException("Unable to get element " + el.getElementName() + " from PSEStorage, see cause");
                    se.initCause(t);
                    throw se;
                }
                else if (t instanceof Error)
                {
                    throw ((Error)t);
                }
            }
        }
        if (DEBUG)
        {
            System.out.println("PSEStorage.getElement " + elementName.getName() + " returning null");
        }
        return null;
    }

	@Override
    public  IDirElement[] getElements(EntityName elementName) throws StorageException
	{
		joinSession();
		beginOSTransaction("getElements", ObjectStore.READONLY);
		DSElement el = (DSElement)m_data.getFromPrimaryIndex(elementName.getName());

		if (el != null)
		{
			try
			{
				/*if (PackedDirUtil.underPackedDir(elementName))
				{
					String indexValue = el.getGroupName();
					if (DEBUG) System.out.println("PSEStorage.getElements groupName == " + indexValue +
							" for element " + elementName.getName());
					Query groupQuery = new Query(DSElement.class, "getGroupName() == \"" +
							indexValue + "\"");
					Iterator inTheGroup = groupQuery.iterator(m_data);
					ArrayList elsList = new ArrayList();
					while (inTheGroup.hasNext())
					{
						DSElement dsEl = (DSElement)inTheGroup.next();
						elsList.add(dsEl.getIDirElement(m_password));
					}
					return (IDirElement [])elsList.toArray(new IDirElement[] {});
				}
				else */
				    return new IDirElement[] {el.getIDirElement(m_password)};
			}
            catch (Throwable t)
            {
                if (DSDUMP)
                {
                    m_corruptIds.put(elementName.getName(), new Object[]
                    { t, t.getStackTrace() });
                }
                else if (t instanceof Exception)
                {
                    if (DEBUG)
                    {
                        t.printStackTrace();
                    }
                    StorageException se = new StorageException("PSEStorage unable to getElements " + elementName.getName() + ", see cause");
                    se.initCause(t);
                    throw se;
                }
                else if (t instanceof Error)
                {
                    throw (Error)t;
                }
            }
		}
		return new IDirElement[0];
	}

	@Override
    public IDirElement[] getAllElements(EntityName dirName)
			throws StorageException
	{
		joinSession();
		beginOSTransaction("getAllElements", ObjectStore.READONLY);
		IDirElement[] els;
		try
		{
			els = listElementsInternal(dirName);
		}
		catch (Exception e)
		{
		    throw new StorageException("PSEStorage unable to getAllElements, see cause", e);
		}
		return els;
	}

	@Override
    public IElementIdentity[] listElements(EntityName dirName) throws StorageException
	{
		joinSession();
		beginOSTransaction("listElements", ObjectStore.READONLY);
		IElementIdentity[] ids;
		try
		{
			ids = listElementIdsInternal(dirName);
		}
		catch (Exception e)
		{
            throw new StorageException("PSEStorage unable to listElements, see cause", e);
		}
		return ids;
	}

	// to be called inside a read transaction
	private IElementIdentity [] listElementIdsInternal(EntityName dirName) throws Exception
	{
		if (internalGetDirectory(dirName) == null)
        {
            throw new StorageException("Directory " + dirName.getName() +
					" does not exist");
        }
		Query elQuery = new Query(DSElement.class, "getParentDirectory() == \"" + dirName.getName() +
				"\"");
		Iterator elsIterator = elQuery.iterator(m_data);
		ArrayList elsList = new ArrayList();
		int elIndex = 0; // used to create unique ids when reporting corruption 
		                 // and we cannot find the name of the element
		while (elsIterator.hasNext())
		{
		    DSElement pseEl = null;
		    try
		    {
		        pseEl = (DSElement)elsIterator.next();
		        elsList.add(pseEl.getElementIdentity(m_password));
		    }
            catch (Throwable t)
            {
                if (DSDUMP)
                {
                    if (pseEl != null)
                    {
                        try
                        {
                            m_corruptIds.put(pseEl.getElementName(), new Object[] {t, t.getStackTrace()});
                        }
                        catch (Throwable elNameT)
                        {
                            m_corruptIds.put(++elIndex + dirName.getName(), new Object[] {t.toString() + ", and a subsequent " + elNameT + " while listing directory " + dirName.getName(), t.getStackTrace()});
                        }
                    }
                    else
                    {
                        m_corruptIds.put(++elIndex + dirName.getName(), new Object[] {t.toString() + ", while listing directory " + dirName.getName(), t.getStackTrace()});
                    }
                }
                else if (t instanceof Error)
                {
                    throw (Error)t;
                }
            }
		}
		IElementIdentity[] idsArray = new IElementIdentity[elsList.size()];
		elsList.toArray(idsArray);
		return idsArray;
	}

//	 to be called inside a read transaction
	private IDirElement[] listElementsInternal(EntityName dirName) throws Exception
	{
		if (internalGetDirectory(dirName) == null)
        {
            throw new StorageException("Directory " + dirName.getName() +
					" does not exist");
        }
		Query elQuery = new Query(DSElement.class, "getParentDirectory() == \"" + dirName.getName() +
				"\"");
		Iterator elsIterator = elQuery.iterator(m_data);

		ArrayList elsList = new ArrayList();
		while (elsIterator.hasNext())
		{
		    DSElement pseEl = (DSElement)elsIterator.next();
		    IDirElement dirEl = pseEl.getIDirElement(m_password);
		    elsList.add(dirEl);
		}
		IDirElement [] elsArray = new IDirElement[elsList.size()];
		elsList.toArray(elsArray);
		return elsArray;
	}

	@Override
    public IDirIdentity[] listDirectories(EntityName dirName)
			throws StorageException
	{
		joinSession();
		beginOSTransaction("listDirectories", ObjectStore.READONLY);

		if (internalGetDirectory(dirName) == null)
        {
            throw new StorageException("PSEStorage listDirectories, directory does not exist " + dirName.getName());
        }
		return listDirectoriesInternal(dirName.getName());
	}

	// to be called inside of a read transaction
	private IDirIdentity[] listDirectoriesInternal(String parentDir)
	{
		if (DEBUG)
        {
            System.out.println("PSEStorage.listDirectoriesInternal " +parentDir);
        }
		Query dirQuery = new Query(DSDirectory.class, "getParentDirectoryName()  == \"" +
				parentDir + "\"");
		Iterator dirIterator = dirQuery.iterator(m_directories);
		ArrayList dirIDs = new ArrayList();
		while (dirIterator.hasNext())
		{
			DSDirectory dir = (DSDirectory)dirIterator.next();
			dirIDs.add(new DirIdentity(dir.getDirectoryName()));
		}
		IDirIdentity[] idsArray = new IDirIdentity[dirIDs.size()];
		dirIDs.toArray(idsArray);
		if (DEBUG)
        {
            System.out.println("PSEStorage.listDirectoriesInternal " +parentDir +
            		" returning array of " + idsArray.length + " directories");
        }
		return idsArray;
	}

	@Override
    public IIdentity[] listAll(EntityName dirName) throws StorageException
	{
		joinSession();
		beginOSTransaction("listAll", ObjectStore.READONLY);
		try
		{
			if (internalGetDirectory(dirName) == null)
            {
                throw new StorageException("PSEStorage listAll, directoy does not exist: " + dirName.getName());
            }
			IDirIdentity[] dirIDs = listDirectoriesInternal(dirName.getName());
			IElementIdentity[] elIDs = listElementIdsInternal(dirName);
			IIdentity[] allIds= new IIdentity[dirIDs.length + elIDs.length];
			System.arraycopy(dirIDs, 0, allIds, 0, dirIDs.length);
			System.arraycopy(elIDs, 0, allIds, dirIDs.length, elIDs.length);
			return allIds;
		}
		catch (Exception e)
		{
			if (e instanceof StorageException)
            {
                throw (StorageException)e;
            }
            else
			{
		        StorageException se = new StorageException("PSEStorage listAll could not list for directory " + dirName.getName() + ", see cause");
	            se.initCause(e);
	            throw se;
			}
		}
	}

	private void deleteElementInternal(DSElement el)
	{
		if (DEBUG)
        {
            System.out.println("PSEStorage.deleteElementInternal " + el.getElementName());
        }
		m_data.remove(el);
		ObjectStore.destroy(el);
	}

	@Override
    public IElementIdentity deleteElement(EntityName elementName)
			throws StorageException
	{
		joinSession();
		boolean startedTransaction = false;
		if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
		{
			beginOSTransaction("setElement", ObjectStore.UPDATE);
			startedTransaction = true;
		}
		DSElement el = (DSElement)m_data.getFromPrimaryIndex(elementName.getName());
		try
		{
			if (el != null)
			{
				IElementIdentity id = el.getElementIdentity(m_password);
				deleteElementInternal(el);
				return id;
			}
		}
		catch (Exception e)
		{
			if (e instanceof StorageException)
            {
                throw (StorageException)e;
            }
            else
			{
		        StorageException se = new StorageException("PSEStorage unable to delete element " + elementName.getName() + " m_password = " + m_password + ", see cause");
		        se.initCause(e);
		        throw se;
			}
		}
		finally
		{
			if (startedTransaction)
            {
                commitOSTransaction("deleteElement", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
            }
		}
		return null;
	}

	@Override
    public IElementIdentity deleteElement(EntityName elementName, boolean force)
			throws StorageException
	{
		// for now this is the same as deleteElement
		return deleteElement(elementName);
	}

	@Override
    public IElementIdentity[] deleteElements(EntityName[] elementNames)
			throws StorageException
	{
		joinSession();
		boolean startedTransaction = false;
		if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
		{
			beginOSTransaction("setElement", ObjectStore.UPDATE);
			startedTransaction = true;
		}
		if (DEBUG)
        {
            System.out.println("PSEStorage.deleteElements, array of " + elementNames.length + " elements");
        }
		if (elementNames.length == 0)
        {
            return new IElementIdentity[0];
        }
		ArrayList ids = new ArrayList();

		try
		{
			for (int i=0; i<elementNames.length; i++)
			{
				EntityName name = elementNames[i];
				DSElement el = (DSElement)m_data.getFromPrimaryIndex(name.getName());

				if (el != null)
				{
					m_data.remove(el);
					ids.add(el.getElementIdentity(m_password));
					ObjectStore.destroy(el);
				}
			}
			IElementIdentity[] idsArray = new IElementIdentity[ids.size()];
			ids.toArray(idsArray);
			return idsArray;
		}
		catch (Exception e)
		{
			if (e instanceof StorageException)
            {
                throw (StorageException)e;
            }
            else
			{
	            StorageException se = new StorageException("PSEStorage unable to delete elements, see cause");
                se.initCause(e);
                throw se;
			}
		}
		finally
		{
			if (startedTransaction)
            {
                commitOSTransaction("deleteElements", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
            }
		}
	}

	@Override
    public void createDirectory(EntityName dirName) throws StorageException
	{
		joinSession();
		boolean startedTransaction = false;
		if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
		{
			beginOSTransaction("createDirectory", ObjectStore.UPDATE);
			startedTransaction = true;
		}
		try
		{
		    createDirectoryInternal(dirName, false);
		}
		finally
		{
			if (startedTransaction)
            {
                commitOSTransaction("createDirectory", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
            }
		}
	}
	// to be called inside of a transaction. Parent directory checks are done
	// by the caller
	private void createDirectoryInternal(EntityName dirName, boolean createParent) throws StorageException
	{
		if (internalGetDirectory(dirName.getParentEntity()) == null)
		{
			if (createParent)
            {
                createDirectoryInternal(dirName.getParentEntity(), createParent);
            }
            else
            {
                throw new ParentDirDoesNotExistException("PSEStorage unable to create directory " +
						dirName.getName() + " because parent directory does not exist");
            }
		}
		if (DEBUG)
        {
            System.out.println("PSEStorage.createDirectoryInternal " +dirName.getName());
        }
		m_directories.add(new DSDirectory(dirName.getName()));
	}

	@Override
    public void deleteDirectory(EntityName dirName) throws StorageException
	{
		joinSession();
		boolean startedTransaction = false;
		if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
		{
			beginOSTransaction("setElement", ObjectStore.UPDATE);
			startedTransaction = true;
		}
		if (DEBUG)
        {
            System.out.println("PSEStorage.deleteDirectory " + dirName.getName());
        }
		DSDirectory dir = internalGetDirectory(dirName);
		if (dir != null)
		{
			Object[] ids;
			try
			{
				ids = listElementsInternal(dirName);
				if (ids.length > 0)
                {
                    throw new StorageException("PSEStorage cannot delete directory " +
							dirName.getName() + "because it's not empty");
                }
                else
				{
					m_directories.remove(dir);
					ObjectStore.destroy(dir);
				}
			}
			catch (Exception e)
			{
				if (e instanceof StorageException)
                {
                    throw (StorageException)e;
                }
                else
				{
		            StorageException se = new StorageException("Unable to delete directory " + dirName.getName() + ", see cause");
		            se.initCause(e);
		            throw se;
				}
			}
			finally
			{
				if (startedTransaction)
                {
                    commitOSTransaction("deleteDirectory", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
                }
			}
		}
	}

	@Override
    public boolean directoryExists(EntityName dirName)
	{
		joinSession();
		beginOSTransaction("directoryExists", ObjectStore.READONLY);
		boolean exists = internalGetDirectory(dirName) != null;
		return exists;
	}
	// call this if you're already in a transaction
	private DSDirectory internalGetDirectory(EntityName dirName)
	{
		return (DSDirectory)m_directories.getFromPrimaryIndex(dirName.getName());
	}

	@Override
    public void setElement(EntityName elementName, IDirElement element)
			throws StorageException
	{
		setElement(elementName, element, false);
	}

	@Override
    public void setElement(EntityName elementName, IDirElement element,
			boolean createParentDir) throws StorageException
	{
		joinSession();
		boolean startedTransaction = false;
		if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
		{
			beginOSTransaction("setElement", ObjectStore.UPDATE);
			startedTransaction = true;
		}
		try
		{
			setElementInternal(elementName, element, createParentDir);
		}
        catch (Exception ex)
        {
            m_logger.logMessage("Failed to store element, trace follows...", ex, Level.WARNING);
            StorageException se = new StorageException("Failed to store element, see cause");
            se.initCause(ex);
            throw se;
        }
		finally
		{
			if (startedTransaction)
            {
                commitOSTransaction("setElement", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
            }
		}
	}

	// to be called inside the transaction
	private void setElementInternal(EntityName elementName, IDirElement element, boolean createParentDir)
		throws Exception
	{
		if (internalGetDirectory(elementName.getParentEntity()) == null)
		{
			if (createParentDir)
            {
                createDirectoryInternal(elementName.getParentEntity(), createParentDir);
            }
            else
            {
                throw new StorageException("PSEStorage unable to set element " +
					elementName.getName() + " because parent directory does not exist");
            }
		}
		DSElement el = (DSElement)m_data.getFromPrimaryIndex(elementName.getName());
		DSBlob blob = null;
		if (el != null)
		{
			// remember if there is a blob with this name
			blob = el.getBlob();
            el.keepBlob(true);
			deleteElementInternal(el);
		}
		if (DEBUG)
        {
            System.out.println("PSEStorage.setElementInternal setting element " +
            		elementName.getName());
        }
        // If this is a blob envelope element specially marked so the DS would ignore the element
        // in storage during the second phase of the import, unmark it now
        Boolean envelopeImport = (Boolean)Blob.getBlobState((Element)element, Blob.BLOB_ENVELOPE_ELEMENT_IMPORT);
        if (envelopeImport != null && envelopeImport.booleanValue())
        {
            Blob.markBlobState((Element)element, Boolean.FALSE, IBlob.BLOB_ENVELOPE_ELEMENT_IMPORT);
        }
		DSElement newElement = new DSElement(new String(elementName.getName()), element, m_password);
        newElement.setBlob(blob);
		m_data.add(newElement);
	}

	@Override
    public void setElements(IDirElement[] elements, boolean createParentDir)
			throws StorageException
	{
		joinSession();
		boolean startedTransaction = false;
		if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
		{
			beginOSTransaction("setElement", ObjectStore.UPDATE);
			startedTransaction = true;
		}
		try
		{
			for (int i=0; i<elements.length; i++)
            {
                setElementInternal(new EntityName(elements[i].getIdentity().getName()),
						elements[i], createParentDir);
            }
		}
		catch (Exception e)
		{
			if (e instanceof StorageException)
            {
                throw (StorageException)e;
            }
            else
			{
                StorageException se = new StorageException("PSEStorage failed during setElements, see cause");
                se.initCause(e);
                throw se;
			}
		}
		finally
		{
			if (startedTransaction)
            {
                commitOSTransaction("setElements", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
            }
		}
	}

	@Override
    public void close() throws StorageException
	{
		joinSession();
		// The way the DS call IStorage.close, you could have failed a
		// transaction and a transaction would still be open for IStorage.
		// for this implementation, we must revert the transaction before
		// closing. See qa.mgmt.mf.directory.TransactionTest
		Transaction currentTr = null;
		try
		{
			currentTr = Transaction.current();
		}
		catch (Exception e) {}
		if (DEBUG)
        {
            System.out.println("PSEStorage.close in session " + Session.getCurrent() + " m_session == " + m_session +
            		" m_transaction " + m_transaction + " session.inTransaction == " + m_session.inTransaction() +
            		" Transaction.current() == " + currentTr);
        }
		if (m_transaction != null)
		{
			//if (m_transaction.getType() == ObjectStore.UPDATE)
				//System.out.println("PSEStorage.close transaction being aborted is UPDATE");
			m_transaction.abort();
			if (DEBUG)
            {
                System.out.println("PSEStorage.close aborted m_transaction");
            }
			m_transaction = null;
		}
		// we cannot close the DB if it was created/opened by another session

		m_store.close();
		m_session.terminate();

		if (DEBUG)
        {
            System.out.println("PSEStorage.close terminated session " + m_session +
            		" and closed database " + m_store);
        }

	}

	// this particular method is probably obsolete; however, it must work as expected.
	// After storing a blob this way, when we call getBlob, which expected chunks of
	// 1000000 bytes, that must not crash. We could probably store this blob as one
	// chunk, but getBlob wouldn't like it.
	@Override
    public void setBlob(EntityName blobName, byte[] blob)
			throws StorageException
	{
		joinSession();
		boolean startedTransaction = false;
		if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
		{
			beginOSTransaction("setElement", ObjectStore.UPDATE);
			startedTransaction = true;
		}
		if (DEBUG)
        {
            System.out.println("PSEStorage.setBlob " + blobName.getName()
            		+ ", size of the array " + blob.length + " and the first byte of the array = " +
            		blob[0]);
        }
		try
		{
			String bName = blobName.getName();
			boolean newElement = false;
			DSElement blobE = (DSElement)m_data.getFromPrimaryIndex(bName);
			if (blobE == null)
			{
				blobE = new DSElement(bName, null, null);
				newElement = true;
			}
			DSBlob header = blobE.getBlob();
			if (header != null)
            {
                deleteBlobInternal(blobE);
            }
			header = new DSBlob();
			for (int i=0; i<=(blob.length / COPY_CHUNK_SIZE); i++)
			{
				byte[] chunkBytes;
				if (blob.length - (i*COPY_CHUNK_SIZE) < COPY_CHUNK_SIZE)
                {
                    chunkBytes = new byte[blob.length - (i*COPY_CHUNK_SIZE)];
                }
                else
                {
                    chunkBytes = new byte[COPY_CHUNK_SIZE];
                }
				System.arraycopy(blob, i*COPY_CHUNK_SIZE, chunkBytes, 0, chunkBytes.length);
				DSBlobChunk chunk = new DSBlobChunk(chunkBytes);
				header.addChunk(chunk);
			}
			blobE.setBlob(header);
			if (newElement)
            {
                m_data.add(blobE);
            }
		}
		catch (Exception e)
		{
            throw new StorageException("PSEStorage.setBlob exception, see cause", e);
		}
	}

	@Override
    public long getBlobSize(EntityName blobName) throws StorageException
	{
		joinSession();
		beginOSTransaction("getBlobSize", ObjectStore.READONLY);
		DSElement blobE = (DSElement)m_data.getFromPrimaryIndex(blobName.getName());
		if (blobE != null)
		{
			DSBlob blob = blobE.getBlob();
			if (blob != null)
            {
                return blob.getBlobSize();
            }
		}
		return 0;

	}
	// In the past, the DS was lax about expecting to find a
	// file for the blobs. We can limit the assumption of blobs as files but not entirely.
	// Files are needed when we use the zip file API, and for the classpath of classloaders
	// when the DS is trying to find the validation triggers, SPI classes, etc.
	// Uses: 1) PersistentBlobCache (uses JAR API) 2) DS, to figure out if a blob is expandable, it
	// uses the Jar API to find the sonic.xml member, also for the files in classpaths for
	// validators, etc.
	// The API definition for this method doesn't throw an exception, but in this
	// implementation, errors could happen, so we throw runtime errors instead
	// mrd 10/23/07 This method will use the temporary file built while appendBlob
	// is called so it doesn't have to recreate the file from the storage when it's called
	// from isExpandableBlob. It will check that the file is the right size, as a test
	// that the file got created without error.
	@Override
    public File blobToFile(EntityName blobName)
	{
		joinSession();
		beginOSTransaction("blobToFile", ObjectStore.READONLY);
		String bName = blobName.getName();
		if (DEBUG)
        {
            System.out.println("PSEStorage.blobToFile " + bName);
        }
		try
		{
			File blobFile = new File(m_tempDir, blobName.getBaseName());
			DSElement blobE = (DSElement)m_data.getFromPrimaryIndex(bName);
			DSBlob header = null;
			if (blobE != null)
            {
                header = blobE.getBlob();
            }
			int blobLength = 0;

			if (header != null)
            {
				if (blobFile.exists() && blobFile.length() == header.getBlobSize())
                {
                    // if we were successfull in writing a temporary file
	                // while the blob was being put together, return it.
	                return blobFile;
                }

				blobLength = header.getBlobLength();
				FileOutputStream blobStream = new FileOutputStream(blobFile);
				for (int i=0; i<blobLength; i++)
				{
					DSBlobChunk chunk = (DSBlobChunk)header.getBlobChunk(i);
					byte[] chunkBytes = chunk.getBytes();
					blobStream.write(chunkBytes);
				}
				blobStream.close();
			}
			return blobFile;
		}
		catch (Exception e)
		{
		    throw new Error("PSEStorage blobToFile failed for " + blobName.getName() + ", see cause", e);
		}
	}

	@Override
    public byte[] getBlob(EntityName blobName) throws StorageException
	{
		return getBlob(blobName, 0, COPY_CHUNK_SIZE);
	}

	@Override
    public void deleteBlob(EntityName blobName) throws StorageException
	{
		joinSession();
		boolean startedTransaction = false;
		if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
		{
			beginOSTransaction("setElement", ObjectStore.UPDATE);
			startedTransaction = true;
		}
		try
		{
			DSElement blobE = (DSElement)m_data.getFromPrimaryIndex(blobName.getName());
			if (blobE != null)
             {
                deleteBlobInternal(blobE);
			// the element is modified. Need to delete and re-add? OR will
			// the transaction save the changes?
            }
		}
		catch (Exception e) // we're not expecting a StorageException here
		{
			e.printStackTrace();
            throw new StorageException("PSEStorage is unable to delete blob " + blobName.getName() + ", see cause", e);
		}
		finally
		{
			if (startedTransaction)
            {
                commitOSTransaction("deleteBlob", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
            }
		}
	}

	@Override
    public void appendBlob(EntityName entityName, byte[] blobBytes, int offset)
			throws StorageException
	{
		joinSession();
		if (DEBUG && blobBytes.length > 0)
        {
            System.out.println("PSEStorage.appendBlob " + entityName.getName()
				+ ", size of the array " + blobBytes.length + " at offset " + offset +
				" and the first byte in the array == " + blobBytes[0]);
        }
		try
        {
			if (internalGetDirectory(entityName.getParentEntity()) == null)
			{
				createDirectoryInternal(entityName.getParentEntity(), true);
			}
			String bName = entityName.getName();
			DSElement blobE = (DSElement)m_data.getFromPrimaryIndex(bName);
			DSBlob blobHeader = null;
			if (blobE != null)
			{
				blobHeader = blobE.getBlob();
				m_data.remove(blobE); // about to modify it
				if (DEBUG)
                {
                    System.out.println("PSEStorage.appendBlob DSElement was not null, so we removed it");
                }
				if (blobHeader == null)
                {
                    if (DEBUG)
                    {
                        System.out.println("PSEStorage.appendBlob existing DSElement did not have a blob header");
                    }
                }
			}
			else
			{
				blobE = new DSElement(bName, null, null);
				if (DEBUG)
                {
                    System.out.println("PSEStorage.appendBlob creating DSElement " + bName);
                }
			}
			// if we're starting at index 0, and the blob exists, delete the whole thing
			if (offset == 0 && (blobHeader != null))
			{
				if (DEBUG)
                {
                    System.out.println("PSEStorage.appendBlob deleting blob to start a new one");
                }
				deleteBlobInternal(blobE);
				blobHeader = null;
			}

			if (offset != 0)
			{
				if ((blobHeader != null) && (blobHeader.getBlobSize() != offset))
                {
                    throw new Error("PSEStorage cannot append to blob, index out of bounds. Size of blob = " +
                    		blobHeader.getBlobSize() + " and index for append = " + offset);
                }
                else if (blobHeader == null)
                {
                    throw new Error("PSEStorage cannot append to blob, index out of bounds. Blob is non existent and index for append = " + offset);
                }
			}
			if (blobHeader == null)
			{
				if (DEBUG)
                {
                    System.out.println("PSEStorage.appendBlob creating a new blobHeader and setting it in the element ");
                }
				blobHeader = new DSBlob();
				blobE.setBlob(blobHeader);
			}

			int newLength = blobHeader.getBlobLength() + 1;
			if (DEBUG)
            {
                System.out.println("PSEStorage.appendBlob new blobLength == " + newLength);
            }
			DSBlobChunk blobChunk = new DSBlobChunk(blobBytes);
			blobHeader.addChunk(blobChunk);
			if (DEBUG)
            {
                System.out.println("PSEStorage.appendBlob adding DSElement to m_data ");
            }
			m_data.add(blobE);
			appendToTemp(entityName, blobBytes, offset);
			if (DEBUG)
            {
                System.out.println("PSEStorage.appendBlob exiting");
            }

        }
        catch (Exception e)
        {
        	if (DEBUG)
            {
                e.printStackTrace();
            }
            if (e instanceof StorageException)
            {
                throw (StorageException)e;
            }
            StorageException se = new StorageException("PSEStorage unable to append blob " + entityName.getName() + ", see cause");
            se.initCause(e);
            throw se;
        }
	}


	// for now, deletes both the header and the chunks
	private void deleteBlobInternal(DSElement blobE)
	{
		String name = blobE.getElementName();
		if (DEBUG)
        {
            System.out.println("PSEStorage.deleteBlobInternal " + name);
        }
		blobE.setBlob(null);
	}

	private void appendToTemp(EntityName entityName, byte[] blobBytes, int offset)
	{
		File tempFile = new File(m_tempDir, entityName.getBaseName());
		if (tempFile.length() == offset)
		// If this condition is not true,
		// probably something weird happened in another append.
		// blobToFile will just have to do the work in the end
		{
			try
			{
				FileOutputStream tempStream = new FileOutputStream(tempFile, true);
				tempStream.write(blobBytes);
				tempStream.close();
			}
			catch (IOException ioEx)
			{
				// We don't want to throw an exception here, if the rest
				// of the appendBlob was successful.
				// if writing out a piece of the temporary file fails,
				// blobToFile will know to recreate the file. We'll loose
				// the performance improvement of having the file available
				// already
			}
		}

	}


	@Override
    public byte[] getBlob(EntityName entityName, int offset, int length)
			throws StorageException
	{
		joinSession();
		beginOSTransaction("getBlob", ObjectStore.READONLY);

		String name = entityName.getName();
		DSElement blobE = (DSElement)m_data.getFromPrimaryIndex(name);
		DSBlob header = null;
		if (blobE != null)
        {
            header = blobE.getBlob();
        }
		if (header == null)
        {
            return null;
        }
		int size = header.getBlobSize();

		if (offset >= size)
        {
            return new byte[0];
        }
		if (offset == 0 && length == size)
        {
            return getEntireBlob(header);
        }
		// for our applications, we might be able to assume that all chunks are the
		// same size.
		// we also assume offset will be at the beginning of a chunk, and
		// length will be equal to COPY_CHUNK_SIZE
		int blobChunkIndex = (offset / COPY_CHUNK_SIZE);
		// test that we're starting at the beginning of a chunk
		int offsetInChunk = offset % COPY_CHUNK_SIZE;
		if (offsetInChunk != 0)
        {
            throw new StorageException("PSEStorage cannot fetch a partial blob segment for " +
					name + " starting at index " + offset);
        }
		if ((length != COPY_CHUNK_SIZE) && ((blobChunkIndex+1) != header.getBlobLength()))
        {
            throw new StorageException("PSEStorage cannot fetch a partial blob segment for " +
					name + " of length != 1000000 bytes");
        }
		DSBlobChunk chunk = header.getBlobChunk(blobChunkIndex);
		if (DEBUG)
        {
            System.out.println("PSEStorage.getBlob " + name + " returning blob of size " + chunk.getBytes().length +
            		" and the first byte of the array == " + chunk.getBytes()[0]);
        }
		byte[] chunkBytes = chunk.getBytes();
		byte[] copyBytes = new byte[chunkBytes.length];
		System.arraycopy(chunkBytes, 0, copyBytes, 0, chunkBytes.length);
		return copyBytes;
	}

	private byte[] getEntireBlob(DSBlob header)
	{
		byte[] copyBytes = new byte[header.getBlobSize()];
		int copyBytesIndex = 0;
		for (int chunkIndex = 0; chunkIndex < header.getBlobLength(); chunkIndex++)
		{
		    DSBlobChunk chunk = header.getBlobChunk(chunkIndex);
		    if (DEBUG)
            {
                System.out.println("PSEStorage.getEntireBlob fetching chunk " + chunkIndex);
            }
		    byte[] chunkBytes = chunk.getBytes();
		    System.arraycopy(chunkBytes, 0, copyBytes, copyBytesIndex, chunkBytes.length);
		    copyBytesIndex = copyBytesIndex + chunkBytes.length;
		}
		return copyBytes;
	}

	@Override
    public void startTransaction() throws StorageException
	{
		joinSession();
		beginOSTransaction("startTransaction", ObjectStore.UPDATE);
	}

	@Override
    public void commitTransaction() throws StorageException
	{
		joinSession();
		commitOSTransaction("commitTransaction", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
	}

	@Override
    public void rollbackTransaction() throws StorageException
	{
		joinSession();
		if (m_transaction != null)
		{
			m_transaction.abort(ObjectStore.RETAIN_HOLLOW);
			if (DEBUG_TRANSACTION)
            {
                System.out.println("PSEStorage.rollbackTransaction " + m_transaction.toString());
            }
			m_transaction = null;
		}

	}

	@Override
    public void closeFiles() throws StorageException {
		// TODO Auto-generated method stub

	}

	@Override
    public void openFiles() throws StorageException {
		// TODO Auto-generated method stub

	}

	@Override
    public void setLogger(ILogger logger)
	{
		m_logger = logger;   // We don't use yet...
	}

	@Override
    public void copyBlob(EntityName fromBlobName, EntityName toBlobName)
			throws StorageException
	{
		joinSession();
		boolean startedTransaction = false;
		if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
		{
			beginOSTransaction("setElement", ObjectStore.UPDATE);
			startedTransaction = true;
		}
		String fBlobName = fromBlobName.getName();
		String tBlobName = toBlobName.getName();
		boolean newElement = false;
		if (DEBUG)
        {
            System.out.println("PSEStorage.copyBlob " + fBlobName + " " + tBlobName);
        }
		try
		{
			DSElement blobE = (DSElement)m_data.getFromPrimaryIndex(fBlobName);
			DSBlob header = null;
			if (blobE != null)
            {
                header = blobE.getBlob();
            }
			if (header != null)
			{
				// have to remove the old one if it was there
				// easier in the file implementation, just overwrite the file
				DSElement  newBlobE = (DSElement)m_data.getFromPrimaryIndex(tBlobName);
				DSBlob newHeader = null;
				if (newBlobE != null)
                {
                    newHeader = newBlobE.getBlob();
                }
                else
				{
					newBlobE = new DSElement(tBlobName, null, null);
					newElement = true;
				}
				if (newHeader != null)
				{
					deleteBlobInternal(newBlobE);
				}
				newHeader = new DSBlob();
				for (int i=0; i<header.getBlobLength(); i++)
				{
					DSBlobChunk fromChunk = header.getBlobChunk(i);
					byte[] fromChunkArray = fromChunk.getBytes();
					byte[] newArray = new byte[fromChunkArray.length];
					System.arraycopy(fromChunkArray, 0, newArray, 0, fromChunkArray.length );
					DSBlobChunk newChunk = new DSBlobChunk(newArray);
					newHeader.addChunk(newChunk);
				}
				newBlobE.setBlob(newHeader);
				if (newElement)
                {
                    m_data.add(newBlobE);
                }
			}
		}
		catch (Exception e)
		{
            throw new StorageException("PSEStorage.copyBlob from " + fBlobName + " to " + tBlobName + " exception, see cause", e);
		}
		finally
		{
			if (startedTransaction)
            {
                commitOSTransaction("copyBlob", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
            }
		}
	}

    private void joinSession()
	{
		boolean printed = false;
		Session currSession = Session.getCurrent();
		if (currSession != null && currSession != m_session)
		{
			if (DEBUG)
			{
				System.out.println("PSEStorage.joinSession thread " + Thread.currentThread() +
			          " leaving session " + currSession);
			    printed = true;
			}
			Session.leave();
		}


		if (currSession != m_session)
		{
            m_session.join();

			if (DEBUG)
			{
				System.out.println("PSEStorage.joinSession thread " + Thread.currentThread() + " just joined session: " +
						Session.getCurrent());
			    printed = true;
			}
		}
		if (DEBUG && !printed)
        {
            System.out.println("PSEStorage.joinSession didn't switch a thing, current session = " + Session.getCurrent());
        }

	}

	private void setSession()
	{
		m_session = Session.create(null, null);
		if (DEBUG)
        {
            System.out.println("PSEStorage.setSession creating a new session " + m_session);
        }
	}

	private synchronized void beginOSTransaction(String methodName, int access)
	{
		if (m_transaction != null)
		{
			if (access == ObjectStore.UPDATE)
			{
				// the DS does not allow nested UPDATE transactions,
				// but there could be a read transaction open. If so,
				// commit it and start the UPDATE transaction

				m_transaction.abort(ObjectStore.RETAIN_HOLLOW);
				m_transaction = Transaction.begin(access);

			}
		}
        else
        {
            m_transaction = Transaction.begin(access);
        }

		if (DEBUG_TRANSACTION)
		{
			System.out.println("PSEStorage.beginOSTransaction " + methodName + " in session " +
					m_session.toString().substring(m_session.toString().indexOf("Session")) +
					" transaction == " + m_transaction.toString().substring(m_transaction.toString().indexOf("Transaction")));
		}
	}

	private void commitOSTransaction(String methodName, int access, int commitFlag)
	{
		Transaction temp = m_transaction;
		m_transaction.commit(commitFlag);
		m_transaction = null;
		if (DEBUG_TRANSACTION)
        {
            System.out.println("PSEStorage.commitTransaction after transaction.commit in session " +
					m_session.toString().substring(m_session.toString().indexOf("Session")) +
					" for transaction " + temp.toString().substring(temp.toString().indexOf("Transaction")));
        }
	}

	@Override
    public void closeTransactionManager() throws StorageException
	{
		joinSession();
		if (m_transaction != null)
		{
			m_transaction.abort(ObjectStore.RETAIN_HOLLOW);
			m_transaction = null;
		}
	}

	@Override
    public void backup(String backupDir)
	{
		m_store.backup(backupDir + DB_EXTENSION);
	}

    private static final String[] EMPTY_STRING_ARRAY = new String[0];


    @Override
    public String[] collectionToStringArray(String collectionName) throws StorageException
    {
        if (m_collections == null)
        {
            throw new StorageException("Collection " + collectionName + " does not exist");
        }
        try
        {
            joinSession();
            beginOSTransaction("collectionToStringArray", ObjectStore.READONLY);
            OSTreeSet collection = (OSTreeSet)m_collections.get(collectionName);
            if (collection == null)
            {
                collection = (OSTreeSet)m_store.getRoot(collectionName);
                if (collection == null)
                {
                    throw new StorageException("Collection " + collectionName + " does not exist");
                }
                m_collections.put(collectionName, collection);
            }

            Iterator iter = collection.iterator();
            ArrayList list = new ArrayList();
            while (iter.hasNext())
            {
                CollectionElement el = (CollectionElement)iter.next();
                list.add(el.getName());
            }
            return (String[])list.toArray(EMPTY_STRING_ARRAY);
        }
        catch (Throwable t)
        {
            throw new StorageException(t.getMessage(), t);
        }
    }

    @Override
    public void removeFromCollection(String collectionName, String item) throws StorageException
    {
        removeFromCollection(collectionName, new String[]{item});
    }

    @Override
    public void removeFromCollection(String collectionName, String[] items) throws StorageException
    {
        if (m_collections == null)
        {
            throw new StorageException("Collection " + collectionName + " does not exist");
        }

        boolean startedTransaction = false;

        try
        {
            joinSession();

            if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
            {
                beginOSTransaction("removeFromCollection", ObjectStore.UPDATE);
                startedTransaction = true;
            }

            OSTreeSet collection = (OSTreeSet)m_collections.get(collectionName);
            if (collection == null)
            {
                collection = (OSTreeSet)m_store.getRoot(collectionName);
                if (collection == null)
                {
                    throw new StorageException("Collection " + collectionName + " does not exist");
                }
                m_collections.put(collectionName, collection);
            }

            for (int i = 0; i < items.length; i++)
            {
                CollectionElement el = (CollectionElement)collection.getFromPrimaryIndex(items[i]);
                collection.remove(el);
                ObjectStore.destroy(el);
            }

        }
        catch (Throwable t)
        {
            throw new StorageException(t.getMessage(), t);
        }
        finally
        {
            if (startedTransaction)
            {
                commitOSTransaction("createCollection", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
            }
        }
    }


    @Override
    public void addToCollection(String collectionName, String item) throws StorageException
    {
        if (m_collections == null)
        {
            throw new StorageException("Collection " + collectionName + " does not exist");
        }

        boolean startedTransaction = false;

        try
        {
            joinSession();

            if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
            {
                beginOSTransaction("addToCollection", ObjectStore.UPDATE);
                startedTransaction = true;
            }

            OSTreeSet collection = (OSTreeSet)m_collections.get(collectionName);
            if (collection == null)
            {
                collection = (OSTreeSet)m_store.getRoot(collectionName);
                if (collection == null)
                {
                    throw new StorageException("Collection " + collectionName + " does not exist");
                }
                m_collections.put(collectionName, collection);
            }

            if (collection.getFromPrimaryIndex(item) == null)
            {
                collection.add(new CollectionElement(item));
            }

        }
        catch (Throwable t)
        {
            throw new StorageException(t.getMessage(), t);
        }
        finally
        {
            if (startedTransaction)
            {
                commitOSTransaction("createCollection", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
            }
        }
    }

    @Override
    public void createCollectionIfNotCreated(String collectionName) throws StorageException
    {
        if (m_collections == null)
        {
            m_collections = new HashMap();
        }

        if (m_collections.get(collectionName) != null)
        {
            return;
        }

        boolean startedTransaction = false;

        try
        {
            joinSession();

            if (m_startsUpdateTransactions && (!m_session.inTransaction() || m_session.currentTransaction().getType() != ObjectStore.UPDATE))
            {
                beginOSTransaction("createCollection", ObjectStore.UPDATE);
                startedTransaction = true;
            }

            OSTreeSet collection = null;
            try
            {
                collection = (OSTreeSet)m_store.getRoot(collectionName);
            }
            catch (com.odi.DatabaseRootNotFoundException e){}

            if (collection == null)
            {
                collection = new OSTreeSet(m_store, DB_COLLECTION_CLASS, DB_COLLECTION_NAME_ATTRIBUTE);
                m_store.createRoot(collectionName, collection);

            }
            m_collections.put(collectionName, collection);
        }
        catch (Throwable t)
        {
            throw new StorageException(t.getMessage(), t);
        }
        finally
        {
            if (startedTransaction)
            {
                commitOSTransaction("createCollection", ObjectStore.UPDATE, ObjectStore.RETAIN_HOLLOW);
            }
        }
    }
    
    public static HashMap getCorruptIds()
    {
        return m_corruptIds;
    }
    
    public static boolean hasCorruptElements()
    {
        return m_corruptIds.size() > 0;
    }
}
