// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.mf.framework.logger;

import java.util.HashMap;
import java.util.Map;

import com.sonicsw.mf.common.metrics.IHistoricalMetric;
import com.sonicsw.mf.common.runtime.INotification;

/**
 * Implements {@link LogFormatter} providing delimited text formatting.
 */
class DelimitedTextLogFormatter
implements LogFormatter
{
    private static final String ARRAY_DELIMITER = ",";
    private static final String ARRAY_DELIMITER_ALTERNATIVE = ";";
    private static Map formatters = new HashMap();
    private final String textDelimiter;
    private final String arrayDelimiter;

    private DelimitedTextLogFormatter(String textDelimiter)
    {
        this.textDelimiter = textDelimiter;
        this.arrayDelimiter = ARRAY_DELIMITER.equals(textDelimiter) ? ARRAY_DELIMITER_ALTERNATIVE : ARRAY_DELIMITER;
    }

    /**
     * Returns a {@link DelimitedTextLogFormatter} instance for the specified text delimiter.
     * @param textDelimiter the text delimiter
     * @return a {@link DelimitedTextLogFormatter} instance
     */
    public static DelimitedTextLogFormatter getInstance(String textDelimiter)
    {
        synchronized (formatters)
        {
            DelimitedTextLogFormatter formatter = (DelimitedTextLogFormatter)formatters.get(textDelimiter);
            if (formatter == null)
            {
                formatter = new DelimitedTextLogFormatter(textDelimiter);
                formatters.put(textDelimiter, formatter);
            }
            return formatter;
        }
    }

    /* (non-Javadoc)
     * @see com.sonicsw.mf.framework.logger.LogFormatter#formatHistoricalMetric(com.sonicsw.mf.common.metrics.IHistoricalMetric)
     */
    @Override
    public Object formatHistoricalMetric(IHistoricalMetric metric)
    {
        StringBuffer buffer = new StringBuffer();

        // add the fully-qualified metric name
        buffer.append("source=");
        buffer.append(metric.getSource());

        // add the text delimiter
        buffer.append(textDelimiter);

        // add the metric ID
        buffer.append("id=");
        buffer.append(metric.getMetricIdentity().getName());

        // add the text delimiter
        buffer.append(textDelimiter);

        // add the metric value
        buffer.append("value=");
        buffer.append(metric.getValue());

        // add the text delimiter
        buffer.append(textDelimiter);

        // add the metric's original timestamp
        buffer.append("timestamp=");
        buffer.append(metric.getCurrencyTimestamp());

        return buffer.toString();
    }

    /* (non-Javadoc)
     * @see com.sonicsw.mf.framework.logger.LogFormatter#formatNotification(com.sonicsw.mf.common.runtime.INotification)
     */
    @Override
    public Object formatNotification(INotification notification)
    {
        StringBuffer buffer = new StringBuffer();

        // add the notification type
        buffer.append("type=").append(notification.getType());

        // add the delimiter
        buffer.append(textDelimiter);

        // add the notification source
        if (notification instanceof javax.management.Notification)
        {
            buffer.append("source=").append(((javax.management.Notification)notification).getSource());
        }
        else
        {
            buffer.append("source=").append(notification.getSourceIdentity().getCanonicalName());
        }

        // add the delimiter
        buffer.append(textDelimiter);

        // add the notification sequence number
        buffer.append("sequenceNumber=");
        buffer.append(notification.getSequenceNumber());

        // add the delimiter
        buffer.append(textDelimiter);

        // add the notification "UserData", if any, and a delimiter
        if (notification instanceof javax.management.Notification)
        {
            buffer.append("userData=");
            buffer.append(((javax.management.Notification)notification).getUserData());
            buffer.append(textDelimiter);
        }

        // add the original notification timestamp
        buffer.append("timestamp=").append(notification.getTimeStamp());

        // add any attached attributes [and any necessary delimiters]
        Object[] attributes = ((INotification)notification).getAttributes().entrySet().toArray();
        for (int i = 0; i < attributes.length; i++)
        {
            buffer.append(textDelimiter);
            Map.Entry entry = (Map.Entry) attributes[i];
            buffer.append(entry.getKey()).append('=');
            Object attribute = entry.getValue();
            if (attribute == null || !attribute.getClass().isArray())
            {
                buffer.append(attribute);
            }
            else
            {
                Object[] attributeArray = (Object[]) attribute;
                buffer.append("[");
                for (int j = 0; j < attributeArray.length; j++)
                {
                    if (j != 0)
                    {
                        buffer.append(arrayDelimiter);
                    }
                    buffer.append(attributeArray[j]);
                }
                buffer.append("]");
            }
        }

        return buffer.toString();
    }
}

