// Copyright (c) 2010 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.mf.framework.logger;

import java.net.URL;
import java.util.ArrayList;
import java.util.Date;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;
import javax.management.ObjectName;

import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.xml.DOMConfigurator;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.metrics.IHistoricalMetric;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.IValueType;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.common.metrics.manager.IMetricsRegistrar;
import com.sonicsw.mf.common.metrics.manager.IStatistic;
import com.sonicsw.mf.common.metrics.manager.StatisticsFactory;
import com.sonicsw.mf.common.runtime.ICanonicalName;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.IContainerState;
import com.sonicsw.mf.common.runtime.IMonitoredMetrics;
import com.sonicsw.mf.common.runtime.IMonitoredNotifications;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.AbstractFrameworkComponent;

public class LoggerComponent
extends AbstractFrameworkComponent
{
    public String m_configID = null;
    public ICanonicalName m_componentName = null;
    private IMetricsRegistrar m_metricsRegistrar;
    private String m_controlCode = null;
    private String m_collectionsMonitorRuntimeID = null;
    private String m_log4jConfigurationFilePath = null;
    private String m_logFormat = null;
    private String m_logTextDelimiter = null;
    private String[] m_componentCollectionIDs = null;
    private int m_collectionsMonitorPollInterval;  // units of poll interval are seconds
    private int m_maxLookbackPeriod;  // units of lookback period are minutes
    private Object m_lock = new Object();
    private LoggingThread m_loggingThread = null;  // LoggingThread is an inner class
    private boolean m_stopping = false;
    private Object m_changeLock = new Object();  // object used to synchronize changes to member data while Logger is running
    private static final String LOGGER_TRACE_MASK_VALUES = "16=logger failures,32=logger exceptions,64=logger invocations,128=logging cycle";
    public static final int TRACE_LOGGER_FAILURES = 16;
    public static final int TRACE_LOGGER_EXCEPTIONS = 32;
    public static final int TRACE_LOGGER_INVOCATIONS = 64;
    public static final int TRACE_LOGGING_CYCLE = 128;
    public static final int TRACE_DETAIL = 1;

    private static final ArrayList ATTRIBUTE_INFOS = new ArrayList();
    private static final ArrayList OPERATION_INFOS = new ArrayList();
    private static final ArrayList NOTIFICATION_INFOS = new ArrayList();

    private IStatistic m_metricsLoggedPerMinuteStatistic;
    private IStatistic m_notificationsLoggedPerMinuteStatistic;

    // Attribute name constants [these can be replaced when generated constants are available]
    private static final String PRODUCT_INFORMATION_ATTR = "PRODUCT_INFORMATION";
    private static final String CONTROL_NUMBER_ATTR = "CONTROL_NUMBER";
    private static final String EVALUATION_MODE_ATTR = "EVALUATION_MODE";
    private static final String COLLECTIONS_MONITOR_RUNTIME_ID_ATTR = "COLLECTIONS_MONITOR_RUNTIME_ID";
    private static final String COLLECTIONS_MONITOR_POLL_INTERVAL_ATTR = "COLLECTIONS_MONITOR_POLL_INTERVAL";
    private static final String MAX_LOOKBACK_PERIOD_ATTR = "MAX_LOOKBACK_PERIOD";
    private static final String LOG4J_CONFIGURATION_FILE_PATH_ATTR = "LOG4J_CONFIGURATION_FILE_PATH";
    private static final String COMPONENT_COLLECTIONS_ATTR = "COMPONENT_COLLECTIONS";
    private static final String LOG_FORMAT_ATTR = "LOG_FORMAT";
    private static final String L0G_TEXT_DELIMITER_ATTR = "L0G_TEXT_DELIMITER";

    private static final String LOGGER_NOTIFICATION_PREFIX = "sonicmf.notification.";
    private static final String LOGGER_METRIC_PREFIX = "sonicmf.metric.";

    private static final String DELIMITED_TEXT_LOG_FORMAT_TYPE = "DELIMITED TEXT";
    private static final String XML_LOG_FORMAT_TYPE = "XML";
    private static final String JAVA_OBJECT_LOG_FORMAT_TYPE = "JAVA OBJECT";

    private static final String LOG_TEXT_DELIMITER_DEFAULT = ",";

    private static final int COLLECTIONS_MONITOR_POLL_INTERVAL_DEFAULT = 30; // default is 30 seconds
    private static final int MAX_LOOKBACK_PERIOD_DEFAULT = 10; // default is 10 minutes

    public static final String GET_STORED_METRICS_METHOD_NAME = "getStoredMetrics";
    public static final String GET_STORED_NOTIFICATIONS_METHOD_NAME = "getStoredNotifications";

    public static final String NEWLINE = System.getProperty("line.separator");

    public static final com.sonicsw.mf.common.metrics.IMetricIdentity METRICS_LOGGED_PER_MINUTE_METRIC_ID = com.sonicsw.mf.common.metrics.MetricsFactory.createMetricIdentity(new String[]
    {
        "logger",
        "metrics",
        "LoggedPerMinute"
    });

    public static final com.sonicsw.mf.common.metrics.IMetricIdentity NOTIFICATIONS_LOGGED_PER_MINUTE_METRIC_ID = com.sonicsw.mf.common.metrics.MetricsFactory.createMetricIdentity(new String[]
    {
        "logger",
        "notifications",
        "LoggedPerMinute"
    });

    // static initializer [for attributes, operations, notifications and metrics]
    static
    {
        //
        // Attributes
        //
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("ControlNumber", String.class.getName(), "The LoggerComponent License Control Number [required in order to access Logger functionality].", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("EvaluationMode", Boolean.class.getName(), "Flag that indicates if LoggerComponent is running in evaluation-only mode [limited time use].", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("CollectionsMonitorRuntimeID", String.class.getName(), "The runtime ID of the CollectionsMonitor with which this LoggerComponent instance is associated.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("CollectionsMonitorPollInterval", Integer.class.getName(),"The amount of time that the LoggerComponent instance will wait between attempts to poll its associated CollectionMonitor for metrics and notifications data.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("MaxLookbackPeriod", Integer.class.getName(),"The maximum number of threads that the Agent Manager can create to poll containers for their state.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("Log4jConfigurationFilePath", String.class.getName(), "Path to log4j configuration file.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogFormat", String.class.getName(), "Format used to log data [Delimited Text, XML, or Java Object format].", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogTextDelimiter", String.class.getName(), "Character [or text] used to delimit log entries [applicable only when using the Delimited Text log format].", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("ComponentCollections", String[].class.getName(), "IDs of Component Collections in which this LoggerComponent is interested.", true, true, false));

        //
        // Operations
        //
        MBeanParameterInfo[] mbParamInfos = null;

        // operation to add a Component Collection ID to those in which the LoggerComponent is currently interested
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("componentCollectionID", String[].class.getName(), "The ID of a Component Collection to be added to the list of Component Collections in which the LoggerComponent is currently interested.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("addComponentCollectionID", "Adds a Component Collection ID to the list of Component Collections in which the LoggerComponent is currently interested.",
                                                   mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));

        // operation to remove a Component Collection ID from those in which the LoggerComponent is currently interested
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("componentCollectionID", String[].class.getName(), "The ID of a Component Collection to be removed from the list of Component Collections in which the LoggerComponent is currently interested.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("removeComponentCollectionID", "Removes a Component Collection ID from the list of Component Collections in which the LoggerComponent is currently interested. A return value of \"false\" indicates that the attempt was unsuccessful [A LoggerComponent must always be configured with at least one Component Collection ID, and an attempt to remove the only remaining ID will fail].",
                                                   mbParamInfos, Boolean.class.getName(), MBeanOperationInfo.ACTION));

        //
        // Notifications
        //

        // Logger does not currently have any notifications beyond those provided by the superclass
    }

    @Override
    public void init(IComponentContext context)
    {
        // invoke the superclass' init method...
        super.init(context);

        // get this component's name from the context
        m_componentName = super.m_context.getComponentName();

        // read in the Logger's configuration
        readConfiguration(context);

        // initial the log4j environment
        initLog4jEnvironment();

        // needs to be reset, in case it was set prior to initialization
        setTraceMask(super.getTraceMask());

        // initialize the metrics
        IMetricInfo[] loggerMetricsInfo = getMetricsInfo();
        m_metricsRegistrar = super.m_context.initMetricsManagement(loggerMetricsInfo);
        initMetrics();

        // create the logging thread [but do not start it]
        m_loggingThread = new LoggingThread();
        m_loggingThread.setDaemon(true);
    }

    @Override
    public synchronized void start()
    {
        if (super.m_state == IComponentState.STATE_ONLINE)
        {
            return;
        }

        super.start();

        // indicate that LoggerComponent is not in the process of being stopped...
        m_stopping = false;

        // start the logging thread (create it if it doesn't exist - allows stop/start w/o reload)
        if (m_loggingThread == null)
        {
            m_loggingThread = new LoggingThread();
            m_loggingThread.setDaemon(true);
        }
        m_loggingThread.start();
    }

    @Override
    public synchronized void stop()
    {
        if (m_stopping)
        {
            return;
        }

        m_stopping = true;  // indicate that "stop" is in progress...[no need to synch on "m_lock", since m_stopping is a boolean, and access is synchronous...]
        if (m_loggingThread != null)
        {
            synchronized(m_loggingThread.getLoggingThreadLockObj())
            {
                m_loggingThread.getLoggingThreadLockObj().notifyAll();
                m_loggingThread = null;
            }
        }

        // if this instance is already offline, simply return
        if (super.m_state == IComponentState.STATE_OFFLINE)
        {
            return;
        }

        super.stop();
    }

    @Override
    public void destroy()
    {
        // other operations first...
        LogManager.shutdown();

        // then...
        super.destroy();
    }

    //
    // getter/setter methods for Attributes
    //
    @Override
    public String getTraceMaskValues()
    {
        return (super.getTraceMaskValues() + "," + LOGGER_TRACE_MASK_VALUES);
    }

    @Override
    public void setTraceMask(Integer traceMask)
    {
        super.setTraceMask(traceMask);
    }

    public String getControlNumber()
    {
        return this.m_controlCode;
    }

    public void setControlNumber(String controlCode)
    {
        this.m_controlCode = controlCode;
    }

    public Boolean getEvaluationMode()
    {
        return Boolean.FALSE;
    }

    /*
    // This setting is (now) based strictly and solely on the license code
    public void setEvaluationMode(Boolean evaluationMode)
    {
        this.m_evaluationMode = evaluationMode.booleanValue();
    }
    */

    public String getCollectionsMonitorRuntimeID()
    {
        return this.m_collectionsMonitorRuntimeID;
    }

    public void setCollectionsMonitorRuntimeID(String collectionsMonitorRuntimeID)
    {
        this.m_collectionsMonitorRuntimeID = collectionsMonitorRuntimeID;
    }

    public Integer getCollectionsMonitorPollInterval()
    {
        return new Integer(this.m_collectionsMonitorPollInterval);
    }

    public void setCollectionsMonitorPollInterval(Integer collectionsMonitorPollInterval)
    {
        this.m_collectionsMonitorPollInterval = collectionsMonitorPollInterval.intValue();
    }

    public Integer getMaxLookbackPeriod()
    {
        return new Integer(this.m_maxLookbackPeriod);
    }

    public void setMaxLookbackPeriod(Integer maxLookbackPeriod)
    {
        this.m_maxLookbackPeriod = maxLookbackPeriod.intValue();
    }

    public String getLog4jConfigurationFilePath()
    {
        return this.m_log4jConfigurationFilePath;
    }

    public void setLog4jConfigurationFilePath(String log4jConfigurationFilePath)
    {
        this.m_log4jConfigurationFilePath = log4jConfigurationFilePath;
    }

    public String getLogFormat()
    {
        return this.m_logFormat;
    }

    public void setLogFormat(String logFormat)
    {
        if ( (logFormat.equalsIgnoreCase(DELIMITED_TEXT_LOG_FORMAT_TYPE)) ||
             (logFormat.equalsIgnoreCase(XML_LOG_FORMAT_TYPE)) ||
             (logFormat.equalsIgnoreCase(JAVA_OBJECT_LOG_FORMAT_TYPE)) )
        {
            this.m_logFormat = logFormat;
        }
    }

    public String getLogTextDelimiter()
    {
        return this.m_logTextDelimiter;
    }

    public void setLogTextDelimiter(String logTextDelimiter)
    {
        this.m_logTextDelimiter = logTextDelimiter;
    }

    public String[] getComponentCollections()
    {
        return this.m_componentCollectionIDs;
    }

    public void setComponentCollectionIDs(String[] componentCollectionIDs)
    {
        this.m_componentCollectionIDs = componentCollectionIDs;
    }

    //
    // methods for Operations
    //
    public void addComponentCollectionName(String componentCollectionID)
    {
        super.validateOnline();  // make sure component is online

        if (this.m_componentCollectionIDs == null)
        {
            this.m_componentCollectionIDs = new String[1];
            this.m_componentCollectionIDs[0] = componentCollectionID;
        }
        else
        {
            int size = this.m_componentCollectionIDs.length;
            String[] updatedComponentCollectionIDs = new String[size+1];
            for (int i = 0; i < size; i++)
            {
                updatedComponentCollectionIDs[i] = this.m_componentCollectionIDs[i];  // add the existing names
            }
            updatedComponentCollectionIDs[size] = componentCollectionID;  // add the new name
            this.m_componentCollectionIDs = updatedComponentCollectionIDs;
        }
    }

    public Boolean removeComponentCollectionName(String componentCollectionID)
    {
        super.validateOnline();  // make sure component is online

        if (this.m_componentCollectionIDs == null)
        {
            return new Boolean(false);
        }
        else
        {
            int size = this.m_componentCollectionIDs.length;

            // cannot remove the ID if it is the ONLY Component Collection ID [LoggerComponent must be monitoring at least one Component Collection]
            if ( (size == 1) && (this.m_componentCollectionIDs[0].equals(componentCollectionID)))
            {
                return new Boolean(false);
            }

            // check if name is in list of Component Collection IDs and remove it if so...
            boolean status = false;
            ArrayList tempList = new ArrayList(size);
            for (int i = 0; i < size; i++)
            {
                if (this.m_componentCollectionIDs[i].equals(componentCollectionID))
                {
                    status = true;  // found specified name - will be removed
                    continue;
                }
                else
                {
                    tempList.add(this.m_componentCollectionIDs[i]);
                }
            }

            // if the specified name was not found in the array of Component Collection names, simply return with a value of "false"...
            if (!status)
            {
                tempList.clear();  // don't need to use the list
                return new Boolean(false);
            }

            // create a new [compacted] array of Component Collection names, and then return
            Object[] tempArray = tempList.toArray();
            size = tempArray.length;
            String[] updatedComponentCollectionIDs = new String[size];
            for (int i = 0; i < size; i++)
            {
                updatedComponentCollectionIDs[i] = (String) tempArray[i];
            }
            this.m_componentCollectionIDs = updatedComponentCollectionIDs;
            return new Boolean(status);
        }
    }

    //
    // methods to dynamically handle configuration changes/updates
    //
    @Override
    public synchronized void handleElementChange(IElementChange elementChange)
    {
        // we are interested in changes to:
        //  - the control code
        //  - the CollectionsMonitor runtime ID
        //  - the CM poll interval
        //  - the maximum lookback period
        //  - the log format
        //  - the log text delimiter
        //  - the evalation mode flag
        //  - the list of Component Collections

        String configID = elementChange.getElement().getIdentity().getName();

        //  make sure that provided config ID matches that of this LoggerComponent instance
        if (!configID.equals(m_configID))
        {
            return;
        }

        if (elementChange.getChangeType() == IElementChange.ELEMENT_UPDATED)
        {
            IDeltaElement changeElement = (IDeltaElement)elementChange.getElement();
            IDeltaAttributeSet attrs = (IDeltaAttributeSet)changeElement.getDeltaAttributes();

            String [] mods = attrs.getModifiedAttributesNames();
            handleChangeLoggerAttrs(mods, attrs);

            mods = attrs.getNewAttributesNames();
            handleChangeLoggerAttrs(mods, attrs);

            mods = attrs.getDeletedAttributesNames();
            handleDeletedLoggerAttrs(mods);
        }

    }

    //
    // getter methods for Attributes Info, Operations Info and Notifications Info
    //
    @Override
    public MBeanAttributeInfo[] getAttributeInfos() { return (MBeanAttributeInfo[])ATTRIBUTE_INFOS.toArray(IEmptyArray.EMPTY_ATTRIBUTE_INFO_ARRAY); }

    @Override
    public MBeanOperationInfo[] getOperationInfos() { return (MBeanOperationInfo[])OPERATION_INFOS.toArray(IEmptyArray.EMPTY_OPERATION_INFO_ARRAY); }

    @Override
    public MBeanNotificationInfo[] getNotificationInfos() { return (MBeanNotificationInfo[])NOTIFICATION_INFOS.toArray(IEmptyArray.EMPTY_NOTIFICATION_INFO_ARRAY); }

    //
    // methods to provide Metrics support
    //
    @Override
    public synchronized void enableMetrics(IMetricIdentity[] ids)
    {
        for (int i = 0; i < ids.length; i++)
        {
            if (ids[i].equals(METRICS_LOGGED_PER_MINUTE_METRIC_ID))
            {
                if (m_metricsLoggedPerMinuteStatistic == null)
                {
                    m_metricsLoggedPerMinuteStatistic = StatisticsFactory.createStatistic(IStatistic.COUNTER_MODE, true, null, (short) 1);
                }
                m_metricsRegistrar.registerMetric(METRICS_LOGGED_PER_MINUTE_METRIC_ID, m_metricsLoggedPerMinuteStatistic);
            }
            if (ids[i].equals(NOTIFICATIONS_LOGGED_PER_MINUTE_METRIC_ID))
            {
                if (m_notificationsLoggedPerMinuteStatistic == null)
                {
                    m_notificationsLoggedPerMinuteStatistic = StatisticsFactory.createStatistic(IStatistic.COUNTER_MODE, true, null, (short) 1);
                }
                m_metricsRegistrar.registerMetric(NOTIFICATIONS_LOGGED_PER_MINUTE_METRIC_ID, m_notificationsLoggedPerMinuteStatistic);
            }
        }
    }

    @Override
    public synchronized void disableMetrics(IMetricIdentity[] ids)
    {
        for (int i = 0; i < ids.length; i++)
        {
            if (ids[i].equals(METRICS_LOGGED_PER_MINUTE_METRIC_ID))
            {
                // unregister the metric with the MetricsManager
                m_metricsRegistrar.unregisterMetric(ids[i]);
                m_metricsLoggedPerMinuteStatistic = null;
            }
            if (ids[i].equals(NOTIFICATIONS_LOGGED_PER_MINUTE_METRIC_ID))
            {
                // unregister the metric with the MetricsManager
                m_metricsRegistrar.unregisterMetric(ids[i]);
                m_notificationsLoggedPerMinuteStatistic = null;
            }
        }
    }

    public static IMetricInfo[] getMetricsInfo()
    {
        // create the infos
        IMetricInfo[] infos = new IMetricInfo[2];
        infos[0] = MetricsFactory.createMetricInfo(METRICS_LOGGED_PER_MINUTE_METRIC_ID, IValueType.PER_MINUTE_RATE,
                                                   "Rate at which metrics have been logged.",
                                                   null, false, true, true, false, "metrics logged per minute");
        infos[1] = MetricsFactory.createMetricInfo(NOTIFICATIONS_LOGGED_PER_MINUTE_METRIC_ID, IValueType.PER_MINUTE_RATE,
                                                   "Rate at which notifications have been logged.",
                                                   null, false, true, true, false, "notifications logged per minute");
        return infos;
    }

    private void readConfiguration(IComponentContext context)
    {
        // Get the configuration from the configuration element
        IElement loggerElement = context.getConfiguration(true);
        if (loggerElement != null)
        {
            m_configID = loggerElement.getIdentity().getName();
            IAttributeSet loggerAttributes = loggerElement.getAttributes();

            // get the logger's basic attributes
            readBasicAttributes(loggerAttributes, context);
        }
    }

    private void readBasicAttributes(IAttributeSet attributes, IComponentContext context)
    {
        Object obj = null;

        m_collectionsMonitorRuntimeID = (String) attributes.getAttribute(COLLECTIONS_MONITOR_RUNTIME_ID_ATTR); // required attribute
        m_log4jConfigurationFilePath = (String) attributes.getAttribute(LOG4J_CONFIGURATION_FILE_PATH_ATTR); // required attribute
        obj = attributes.getAttribute(COLLECTIONS_MONITOR_POLL_INTERVAL_ATTR); // optional attribute
        m_collectionsMonitorPollInterval = (obj != null) ? ( (Integer) obj).intValue() : COLLECTIONS_MONITOR_POLL_INTERVAL_DEFAULT;
        obj = attributes.getAttribute(MAX_LOOKBACK_PERIOD_ATTR); // optional attribute
        m_maxLookbackPeriod = (obj != null) ? ( (Integer) obj).intValue() : MAX_LOOKBACK_PERIOD_DEFAULT;
        m_logFormat = (String) attributes.getAttribute(LOG_FORMAT_ATTR); // required attribute
        if (m_logFormat == null)
        {
            m_logFormat = LoggerComponent.DELIMITED_TEXT_LOG_FORMAT_TYPE;
        }
        obj = attributes.getAttribute(L0G_TEXT_DELIMITER_ATTR); // optional attribute
        m_logTextDelimiter = (obj != null) ? (String) obj : LOG_TEXT_DELIMITER_DEFAULT;
        IAttributeSet productInformationAttributes = (IAttributeSet) attributes.getAttribute(PRODUCT_INFORMATION_ATTR);
        if (productInformationAttributes != null)
        {
            m_controlCode = (String) productInformationAttributes.getAttribute(CONTROL_NUMBER_ATTR);
            obj = productInformationAttributes.getAttribute(EVALUATION_MODE_ATTR);
            /*m_evaluationMode = (obj != null) ? ( (Boolean) obj).booleanValue() : false;*/  // now based solely on the license code
        }

        m_componentCollectionIDs = readComponentCollectionIDs();
    }
    
    private String[] readComponentCollectionIDs()
    {
        IElement loggerElement = super.m_context.getConfiguration(true);
        IAttributeSet loggerAttributes = loggerElement.getAttributes();
        IAttributeSet componentCollectionsSet = (IAttributeSet) loggerAttributes.getAttribute(COMPONENT_COLLECTIONS_ATTR);
        Object[] ids = (Object[]) componentCollectionsSet.getAttributes().values().toArray();

        String[] componentCollectionIDs = new String[ids.length];
        Reference ccRef = null;
        for (int i = 0; i < ids.length; i++)
        {
            ccRef = (Reference) ids[i];
            String ccCID = ccRef.getElementName() ;
            IElement configElem = super.m_context.getConfiguration(ccCID, true) ;  // interested in updates
            String name = configElem.getIdentity().getName();
            componentCollectionIDs[i] = name;
        }
        return componentCollectionIDs;
    }

    private void initMetrics()
    {
        // "metrics logged rate" statistic provider and statistic
        m_metricsLoggedPerMinuteStatistic = StatisticsFactory.createStatistic(IStatistic.COUNTER_MODE, true, null, (short)1);
        m_notificationsLoggedPerMinuteStatistic = StatisticsFactory.createStatistic(IStatistic.COUNTER_MODE, true, null, (short)1);
    }

    private void handleChangeLoggerAttrs(String[] attributeNames, IDeltaAttributeSet modifiedAttrs)
    {
        try
        {
            String collectionsMonitorRuntimeID = null;
            String[] componentCollectionIDs = null;
            Integer collectionsMonitorPollInterval = null;
            Integer maxLookbackPeriod = null;
            String logFormat = null;
            String logTextDelimiter = null;

            for (int i = 0; i < attributeNames.length; i++)
            {
                if (attributeNames[i].equals(LoggerComponent.COLLECTIONS_MONITOR_RUNTIME_ID_ATTR))
                {
                    collectionsMonitorRuntimeID = (String)modifiedAttrs.getNewValue(LoggerComponent.COLLECTIONS_MONITOR_RUNTIME_ID_ATTR);
                }
                else if (attributeNames[i].equals(LoggerComponent.COLLECTIONS_MONITOR_POLL_INTERVAL_ATTR))
                {
                    collectionsMonitorPollInterval = (Integer)modifiedAttrs.getNewValue(LoggerComponent.COLLECTIONS_MONITOR_POLL_INTERVAL_ATTR);
                }
                else if (attributeNames[i].equals(LoggerComponent.MAX_LOOKBACK_PERIOD_ATTR))
                {
                    maxLookbackPeriod = (Integer)modifiedAttrs.getNewValue(LoggerComponent.MAX_LOOKBACK_PERIOD_ATTR);
                }
                else if (attributeNames[i].equals(LoggerComponent.LOG_FORMAT_ATTR))
                {
                    logFormat = (String)modifiedAttrs.getNewValue(LoggerComponent.LOG_FORMAT_ATTR);
                }
                else if (attributeNames[i].equals(LoggerComponent.L0G_TEXT_DELIMITER_ATTR))
                {
                    logTextDelimiter = (String)modifiedAttrs.getNewValue(LoggerComponent.L0G_TEXT_DELIMITER_ATTR);
                }
                /*
                else if (attributeNames[i].equals(this.EVALUATION_MODE_ATTR))
                {
                    evaluationMode = (Boolean)modifiedAttrs.getNewValue(this.EVALUATION_MODE_ATTR);
                }
                */
                else if (attributeNames[i].equals(LoggerComponent.COMPONENT_COLLECTIONS_ATTR))
                {
                    componentCollectionIDs = readComponentCollectionIDs();
                }
            }

            // change the data members
            synchronized(m_changeLock)
            {
                if (collectionsMonitorRuntimeID != null)
                {
                    this.m_collectionsMonitorRuntimeID = collectionsMonitorRuntimeID;
                }
                if (componentCollectionIDs != null)
                {
                    this.m_componentCollectionIDs = componentCollectionIDs;
                }
                if (collectionsMonitorPollInterval != null)
                {
                    this.m_collectionsMonitorPollInterval = collectionsMonitorPollInterval.intValue();
                }
                if (maxLookbackPeriod != null)
                {
                    this.m_maxLookbackPeriod = maxLookbackPeriod.intValue();
                }
                if (logFormat != null)
                {
                    this.m_logFormat = logFormat;
                }
                if (logTextDelimiter != null)
                {
                    this.m_logTextDelimiter = logTextDelimiter;
                /*
                if (evaluationMode != null)
                    this.m_evaluationMode = evaluationMode.booleanValue();  // now based solely on the license code
                */
                }
            }
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Error modifying LoggerComponent runtime from configuration change, trace follows...", e, Level.WARNING);
        }
    }

    private void handleDeletedLoggerAttrs(String[] attributeNames)
    {
        try
        {
            synchronized (m_changeLock)
            {
                for (int i=0; i< attributeNames.length; i++)
                {
                    if (attributeNames[i].equals(LoggerComponent.COLLECTIONS_MONITOR_POLL_INTERVAL_ATTR))
                    {
                        this.m_collectionsMonitorPollInterval = LoggerComponent.COLLECTIONS_MONITOR_POLL_INTERVAL_DEFAULT;
                    }
                    else if (attributeNames[i].equals(LoggerComponent.MAX_LOOKBACK_PERIOD_ATTR))
                    {
                        this.m_maxLookbackPeriod = LoggerComponent.MAX_LOOKBACK_PERIOD_DEFAULT;
                    }
                    else if (attributeNames[i].equals(LoggerComponent.L0G_TEXT_DELIMITER_ATTR))
                    {
                        this.m_logTextDelimiter = LoggerComponent.LOG_TEXT_DELIMITER_DEFAULT;
                    }
                }
            }
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Error modifying LoggerComponent runtime from configuration change [deletion]", e, Level.SEVERE);
        }
    }

    private void initLog4jEnvironment()
    {
        // read properties from specified log4j configuration file
        URL log4jConfigurationFileURL = null;
        try
        {
            log4jConfigurationFileURL = new URL(this.m_log4jConfigurationFilePath);
        }
        catch (Exception e)
        {
            // TBD: handle this...
            this.m_context.logMessage("Exception occurred while attempting to load log4j configuration file " + this.m_log4jConfigurationFilePath, e, Level.SEVERE);
        }


        //TBD: probably need to do something else here if the
        //     sonicfs URL is being used - need to figure out what...

        // configure the logger
        if(this.m_log4jConfigurationFilePath.endsWith("xml"))
        {
            DOMConfigurator.configure(log4jConfigurationFileURL);
        }
        else
        {
            PropertyConfigurator.configure(log4jConfigurationFileURL);
        }
    }

    //
    // inner classes
    //

    private final class LoggingThread
    extends Thread
    {
        private boolean m_cmWaitingMessageLogged = false;
        private Exception m_retrievalFailure = null;
        private Exception m_metricLogFailure = null;
        private Exception m_notificationLogFailure = null;
        private Long m_lastMetricRetrievalTimestamp = null;
        private Long m_lastNotificationRetrievalTimestamp = null;
        private Object m_loggingThreadLockObj = new Object();

        private final Object[] EMPTY_OBJECT_ARRAY = new Object[0];
        private final String[] EMPTY_STRING_ARRAY = new String[0];

        private LoggingThread()
        {
            super("LoggingThread");
        }

        @Override
        public void run()
        {
            // don't even bother to do anything if LoggingComponent has already been "stopped"
            if (LoggerComponent.this.m_stopping)
            {
                return;
            }

            // creat a local copy of the Collections Monitor runtime ID [to avoid indeterminate behavior
            // in the event that the Logger's configuration is changed in the DS during runtime]
             String collectionsMonitorRuntimeID = LoggerComponent.this.m_collectionsMonitorRuntimeID;

            // check to make sure that CM runtime ID has been set.
            // Note that part of the validation process is to make sure that the specified CM has been started, and that
            // the method will wait [for the configured period] for the CM to come online if necessary.
            if (!performCollectionsMonitorValidation(collectionsMonitorRuntimeID))
            {
                synchronized(LoggerComponent.this.m_lock)
                {
                    LoggerComponent.this.stop();
                }
                return;
            }

            // check for a valid [7.0 or higher] CM runtime ID
            // [pre-SonicMQ7.0 CollectionsMonitor instances are not valid
            // for LoggerComponents, as those pre-7.0 CM's do not provide the necessary
            // API methods for collecting stored notifications and metrics data]
            if (!validateCollectionsMonitorVersion(collectionsMonitorRuntimeID))
            {
                LoggerComponent.this.m_context.registerErrorCondition("Invalid CollectionsMonitor Runtime ID specified in LoggerComponent's configuration",Level.SEVERE);
                synchronized(LoggerComponent.this.m_lock)
                {
                    LoggerComponent.this.stop();
                }
                return;
            }

            // have thread loop in order to get data from CM, sleeping for the designated polling interval
            // after retrieval of stored data
            while (!LoggerComponent.this.m_stopping)
            {
                // sleep for the configured poll interval period
                try
                {
                    synchronized(m_loggingThreadLockObj)
                    {
                        m_loggingThreadLockObj.wait(LoggerComponent.this.m_collectionsMonitorPollInterval * 1000); // convert from seconds to milliseconds
                        if (LoggerComponent.this.m_stopping)
                         {
                            break; // if thread component is being stopped, need to exit while loop...
                        }
                    }
                }
                catch (Exception e)
                {
                    if (LoggerComponent.this.m_stopping)
                     {
                        return;  // if thread component is being stopped, need to exit...
                    }
                }

                // Get the array of Component Collection IDs whose metrics/notifications are to be retrieved, and also check
                // to see if the CollectionsMonitor runtime ID has been [dynamically] changed
                String[] componentCollectionIDs = null;
                boolean cmChanged = false;
                synchronized(LoggerComponent.this.m_changeLock)
                {
                    componentCollectionIDs = LoggerComponent.this.m_componentCollectionIDs;

                    if (!collectionsMonitorRuntimeID.equals(LoggerComponent.this.m_collectionsMonitorRuntimeID))
                    {
                        cmChanged = true;
                        collectionsMonitorRuntimeID = LoggerComponent.this.m_collectionsMonitorRuntimeID;
                    }
                }

                // Validate the Collections Monitor, if the CollectionsMonitor runtime ID has been changed
                // dynamically [after the logging thread started].
                if (cmChanged)
                {
                    if (!performCollectionsMonitorValidation(collectionsMonitorRuntimeID))
                    {
                        // stop the Logger if the new Collections Monitor is invalid
                        synchronized(LoggerComponent.this.m_lock)
                        {
                            LoggerComponent.this.stop();
                        }
                        return;
                    }
                }

                // poll the CM for its stored metrics and log them
                logStoredMetrics(collectionsMonitorRuntimeID,componentCollectionIDs);

                // poll the CM for its stored notifications and log them
                logStoredNotifications(collectionsMonitorRuntimeID,componentCollectionIDs);

            } //while (!LoggerComponent.this.m_stopping)
        }

        private boolean isCollectionsMonitorOnline(String cmRuntimeID) throws Exception
        {
            boolean online = false;  // assume the worst - that the CM is not currently loaded/online....

            // query the AGENT [that is in the same container as the CM] to get the state of the components in the
            // container that hosts the CM...
            ObjectName cmObjName = new ObjectName(cmRuntimeID);
            String container = cmObjName.getDomain();
            ObjectName agentObjName = new ObjectName(container + ":ID=AGENT");
            boolean synchronous = true;

            IContainerState containerState = (IContainerState)LoggerComponent.this.m_frameworkContext.invoke(agentObjName.getCanonicalName(), "getContainerState", EMPTY_OBJECT_ARRAY, EMPTY_STRING_ARRAY, synchronous, 0); // "0"
            if (containerState == null)
            {
                return false;
            }
            IComponentState[] componentStates = containerState.getComponentStates();

            // find the state of the CM...
            IComponentState componentState = null;
            String id = cmObjName.getKeyProperty("ID");
            for (int i = 0; i < componentStates.length; i++)
            {
                if (((IComponentIdentity)componentStates[i].getRuntimeIdentity()).getComponentName().equals(id))
                {
                    componentState = componentStates[i];
                    break;
                }
            }

            // if CM is online, set return value accordingly
            if ((componentState != null) && (componentState.getState() == IComponentState.STATE_ONLINE))
            {
                online = true;
            }

            return online;
        }

        private boolean waitForCMToStart(String cmRuntimeID)
        {
            boolean online = false;
            while (!LoggerComponent.this.m_stopping)
            {
                try
                {
                    online = isCollectionsMonitorOnline(cmRuntimeID);
                    if (!online)
                    {
                        if (!this.m_cmWaitingMessageLogged)
                        {
                            LoggerComponent.this.m_frameworkContext.logMessage("CollectionsMonitor " + m_collectionsMonitorRuntimeID + " not yet in ONLINE state; LoggerComponent " + m_componentName.getComponentName() + " will continue to attempt to contact the CollectionsMonitor...", Level.INFO);
                            this.m_cmWaitingMessageLogged = true;
                        }

                        try  // sleep for the configured poll interval period
                        {
                            synchronized (m_loggingThreadLockObj)
                            {
                                m_loggingThreadLockObj.wait(LoggerComponent.this.m_collectionsMonitorPollInterval * 1000); // convert from seconds to milliseconds
                                if (LoggerComponent.this.m_stopping)
                                 {
                                    break; // if thread component is being stopped, need to exit while loop...
                                }
                            }
                        }
                        catch (Exception e)
                        {
                            if (LoggerComponent.this.m_stopping)
                             {
                                break; // if thread component is being stopped, need to exit while loop...
                            }
                        }
                    }
                    else if ((online) && (this.m_cmWaitingMessageLogged))
                    {
                        LoggerComponent.this.m_frameworkContext.logMessage("CollectionsMonitor " + m_collectionsMonitorRuntimeID + " is now ONLINE; LoggerComponent " + m_componentName.getComponentName() + " has connected to the CollectionsMonitor...", Level.INFO);
                        this.m_cmWaitingMessageLogged = false;
                    }
                    else {
                        break;  // out of "while (!LoggerComponent.this.m_stopping)"
                    }
                }
                catch (Exception e)
                {
                    // An exception thrown while attempting to get the CM's state from the framework
                    // is an indication of a severe problem - log the appropriate messages
                    if (!LoggerComponent.this.m_stopping)
                    {
                        LoggerComponent.this.m_frameworkContext.logMessage("Error contacting CollectionsMonitor [" + cmRuntimeID + "], trace follows...", e, Level.SEVERE);
                    }
                    break;  // out of "while (!LoggerComponent.this.m_stopping)"
                }
            } // "while (!LoggerComponent.this.m_stopping)"

            return online;
        }

        private boolean performCollectionsMonitorValidation(String cmRuntimeID)
        {
            boolean isValid = true;  // presume validity

            isValid = validateCollectionsMonitorID(cmRuntimeID);  // check for null/empty ID

            if (isValid)
             {
                isValid = validateCollectionsMonitorOnline(cmRuntimeID);  // check if CM is online [Important Note: this method will also wait for it to come online]
            }

            if (isValid)
             {
                isValid = validateCollectionsMonitorVersion(cmRuntimeID);  // make sure specified CM supports the method(s) necessary for obtaining stored notifications/metrics...
            }

            return isValid;
        }

        private boolean validateCollectionsMonitorID(String cmRuntimeID)
        {
            boolean isValid = true;  // presume that the specified ID is valid

            if ( (cmRuntimeID == null) || (cmRuntimeID.equals("")))
            {
                isValid = false;

                // log an error message, register the error condition
                LoggerComponent.this.m_context.logMessage("CollectionsMonitor Runtime ID not specified; please update the LoggerComponent's configuration with a valid CollectionsMonitor Runtime ID.", Level.SEVERE);
                LoggerComponent.this.m_context.registerErrorCondition("CollectionsMonitor Runtime ID not specified in LoggerComponent's configuration",Level.SEVERE);
            }

            return isValid;
        }

        private boolean validateCollectionsMonitorOnline(String cmRuntimeID)
        {
            // check that specified CM is online; if it is not, wait until it comes online...
            boolean cmOnline = waitForCMToStart(cmRuntimeID);

            // if the CM did not come online within the allowed time period, log an error message...
            if (!cmOnline)
            {
                if (LoggerComponent.this.m_stopping)
                {
                    return false;
                }
                LoggerComponent.this.m_context.registerErrorCondition("Error occurred during initial attempt to contact CollectionsMonitor with runtime ID = " + cmRuntimeID, Level.SEVERE);
            }

            return cmOnline;
        }

        private boolean validateCollectionsMonitorVersion(String cmRuntimeID)
        {
            boolean isValid = false;  // presume invalid

            // make sure that the configured CollectionsMonitor
            // supports the required "getStoredNotificationsAndMetricsData"
            // method [that was introduced in SonicMQ7.0]
            try
            {
                Object[] params = {};
                String[] signatures = {};
                boolean synchronous = true;
                MBeanOperationInfo[] operations = null;
                MBeanInfo info = (MBeanInfo) LoggerComponent.this.m_frameworkContext.invoke(cmRuntimeID, "getMBeanInfo", params, signatures, synchronous, 0);  // "0" to indicate that container's timeout value will be used
                if (info != null)
                {
                    operations = info.getOperations();
                }

                boolean methodFound = false;
                int len = 0;
                if (operations != null)
                {
                    len = operations.length;
                }
                for (int i = 0; i < len; i++)
                {
                    if (operations[i].getName().equals(LoggerComponent.GET_STORED_NOTIFICATIONS_METHOD_NAME))  // if the CM exposes the method to get stored notifications, it will also expose the method to get stored metrics, so there is no need to check for both...
                    {
                        methodFound = true;
                        break;
                    }
                }

                if (methodFound)
                {
                    isValid = true;
                }
            }
            catch (Exception e)
            {
                 // log debug message if tracing is enabled
                 if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGER_EXCEPTIONS) > 0)
                 {
                     if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                     {
                         LoggerComponent.this.m_context.logMessage("Exception while checking CollectionsMonitor for existence of \"getStoredMetricsAndNotificationsData\" method; likely that CollectionsMonitor instance is an older version that does not support this method, CollectionsMonitor runtimeID = " + cmRuntimeID,
                             e, Level.TRACE);
                     }
                     else
                     {
                         LoggerComponent.this.m_context.logMessage("Exception while checking CollectionsMonitor for existence of \"getStoredMetricsAndNotificationsData\" method; likely that CollectionsMonitor instance is an older version that does not support this method.",
                             e, Level.TRACE);
                     }
                 }
            }

            if (!isValid)
            {
               // provide trace info, if tracing is enabled...
                if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGER_FAILURES) > 0)
                {
                    if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                    {
                        LoggerComponent.this.m_context.logMessage("Invalid CollectionsMonitor specified for this Logger instance [likely that CollectionsMonitor instance is an older version that does not support \"getStoredMetricsAndNotificationsData\" method], CollectionsMonitor runtimeID = " + cmRuntimeID, Level.TRACE);
                    }
                }

                // log an error message, register the error condition, stop this LoggerComponent instance, and return
                LoggerComponent.this.m_context.logMessage("Invalid CollectionsMonitor Runtime ID specified; please update the LoggerComponent's configuration with the Runtime ID of an existing SonicMQ7.0 [or higher] CollectionsMonitor instance.", Level.SEVERE);

            }

            // return the result
            return isValid;
        }

        private boolean setCMStatusMsgFlags(String cmRuntimeID)
        {
            // Note: this method is only called if there has been an exception while trying to contact the CM

            boolean statusMsgLogged = false;
            try
            {
                boolean online = isCollectionsMonitorOnline(cmRuntimeID);
                if (!online)
                {
                    if (!this.m_cmWaitingMessageLogged)
                    {
                        // if the CM is not online, and if a message to that effect has not yet been logged, then
                        // log a message and set the flag to indicate that the message has been logged [so that there will be one and only one such message logged]
                        LoggerComponent.this.m_frameworkContext.logMessage("CollectionsMonitor not yet in ONLINE state; LoggerComponent will continue attempts to contact the CollectionsMonitor...", Level.INFO);
                        this.m_cmWaitingMessageLogged = true;
                    }
                }
                else  // if CM is online, but had been offline previously, log a message to that effect...
                {
                    // if the CM was previously off-line, and the flag was set to indicate that a message
                    // about the offline CM had been logged, then
                    // clear the "waiting" flag and log a message to indicate that the CM is back online...
                    if (this.m_cmWaitingMessageLogged)
                    {
                        LoggerComponent.this.m_frameworkContext.logMessage("CollectionsMonitor now in ONLINE state; LoggerComponent will poll CollectionsMonitor for stored notifications and metrics...", Level.INFO);
                        this.m_cmWaitingMessageLogged = false;
                    }

                    // Note that if CM is online and no CM offline message had previously been logged, then this method
                    // doesn't need to do anything...
                }

                statusMsgLogged = true;  // if CM offline and msg logged, or if CM online, then this was a success...

                this.m_retrievalFailure = null;
            }
            catch (Exception e)
            {
                // An exception thrown while attempting to get the CM's state from the framework
                // is an indication of a severe problem - log the appropriate messages, and return false (failure)...

                // log debug message if tracing is enabled
                if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGER_EXCEPTIONS) > 0)
                {
                    if (this.m_retrievalFailure != null && (!e.getClass().isInstance(this.m_retrievalFailure) || e.getMessage().equals(this.m_retrievalFailure.getMessage())))
                    {
                        if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                        {
                            LoggerComponent.this.m_context.logMessage( "Exception while attempting to repeatedly contact CollectionsMonitor after initial failure, CollectionsMonitor runtimeID = " + cmRuntimeID, e, Level.TRACE);
                            this.m_retrievalFailure = e;
                        }
                        else
                        {
                            LoggerComponent.this.m_context.logMessage( "Exception while attempting to repeatedly contact CollectionsMonitor after initial failure...", e, Level.TRACE);
                            this.m_retrievalFailure = e;
                        }
                    }
                }

                LoggerComponent.this.m_frameworkContext.logMessage("Error while attempting to contact CollectionsMonitor = " + cmRuntimeID, e, Level.SEVERE);
            }


            return statusMsgLogged;
        }

        /**
         *  Note carefully that this method may stop the LoggerThread [in the event of an exception while
         *  trying to contact the specfied CollectionsMonitor].
         * @param cmRuntimeID
         * @param componentCollectionIDs
         */
        private void logStoredMetrics(String cmRuntimeID, String[] componentCollectionIDs)
        {
            boolean getMoreMetrics = true;
            IMonitoredMetrics monitoredMetrics = null;

            // retrieve stored metrics...
            while (getMoreMetrics)
            {
                try
                {
                    // set the lookback period (do it here so that any changes to configured value will be picked up on each iteration)
                    long lookback;
                    synchronized (LoggerComponent.this.m_changeLock)
                    {
                        lookback = LoggerComponent.this.m_maxLookbackPeriod * 60000;  /*convert lookback period from minutes to milliseconds*/
                    }

                    // the lastRetrievalTimestamp will intentionally be "null" for the first invocation - this will force the full lookback period to be used by the CM
                    monitoredMetrics = null;
                    Object[] params = { componentCollectionIDs,
                                        m_lastMetricRetrievalTimestamp,
                                        new Long(lookback)
                                      };
                    String [] signatures = { "[Ljava.lang.String;",
                                             "java.lang.Long",
                                             "java.lang.Long"
                                           };
                    boolean synchronous = true;

                    long cmElapsedTime = System.currentTimeMillis();
                    monitoredMetrics = (IMonitoredMetrics) LoggerComponent.this.m_frameworkContext.invoke(cmRuntimeID, LoggerComponent.GET_STORED_METRICS_METHOD_NAME, params, signatures, synchronous, 0);  // "0" to indicate that container's timeout value will be used
                    cmElapsedTime = System.currentTimeMillis() - cmElapsedTime;

                    // process the results
                    long logElapsedTime = System.currentTimeMillis();
                    try
                    {
                        processRetrievedMetrics(monitoredMetrics);
                        logElapsedTime = System.currentTimeMillis() - logElapsedTime;

                        // clear flag for log failure log message
                        this.m_metricLogFailure = null;
                    }
                    catch (Exception e)
                    {
                        // this will only occur if the LoggerComponent instance has been configured using an
                        // evaluation license code AND the logged event count limit [as determined by the evaluation license code]
                        // has been exceeded; in that case, set the lastMetricReceivedTimestamp and break out of the "while" loop

                        //If no new metrics then use the previous m_lastMetricRetrievalTimestamp
                        if (monitoredMetrics.getMetrics().length > 0)
                        {
                            m_lastMetricRetrievalTimestamp = new Long(monitoredMetrics.getLatestTimestamp());
                        }

                        // log debug message if tracing is enabled
                        if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGER_EXCEPTIONS) > 0)
                        {
                            if (this.m_metricLogFailure == null || !e.getClass().isInstance(this.m_metricLogFailure) || e.getMessage().equals(this.m_metricLogFailure.getMessage()))
                            {
                                if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                                {
                                    LoggerComponent.this.m_context.logMessage("Failure while logging metric to log4j, trace follows...", e, Level.TRACE);
                                }
                                else
                                {
                                    LoggerComponent.this.m_context.logMessage("Failure while logging metric to log4j", Level.TRACE);
                                }
                            }

                            this.m_metricLogFailure = e;
                        }

                        break;
                    }


                    // check if there is more data to retrieve from the CollectionsMonitor
                    getMoreMetrics = monitoredMetrics.hasMoreDataToRetrieve();

                    if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGING_CYCLE) > 0)
                    {
                        StringBuffer traceMessage = new StringBuffer("Retrieve and log metrics: ");
                        traceMessage.append("retrieve duration (millis)=").append(cmElapsedTime).append(", log duration (millis)=").append(logElapsedTime);
                        if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                        {
                            traceMessage.append(", timestamp1=").append(m_lastNotificationRetrievalTimestamp).append(", lookback=").append(lookback);
                            traceMessage.append(", metric count=").append(monitoredMetrics.getMetrics().length).append(", timestamp2=").append(monitoredMetrics.getLatestTimestamp()).append(", more=").append(getMoreMetrics);
                        }
                        LoggerComponent.this.m_context.logMessage(traceMessage.toString(), Level.TRACE);
                    }

                    // update the lookback period for the next retrieval

                    //If no new metrics then use the previous m_lastMetricRetrievalTimestamp
                    if (monitoredMetrics.getMetrics().length > 0)
                    {
                        m_lastMetricRetrievalTimestamp = new Long(monitoredMetrics.getLatestTimestamp());
                    }

                    // clear flag for retrieval failure log message
                    this.m_retrievalFailure = null;
                }
                catch (Exception ex)
                {
                    // first of all, set the "getMoreMetrics" flag to false.
                    // This is because even if this exception occurred due to the CM temporarily going offline,
                    // we will want to wait for the next polling cycle in order to make sure that all of the
                    // stored data from the CM is retrieved [with the exception, we don't know how much of the
                    // data was actually returned for this polling cycle].
                    getMoreMetrics = false;

                    // leave the lastMetricRetrievedTimestamp as it was, since the retrieval attempt failed...

                    // if the thread is being stopped, simply return...
                    if (LoggerComponent.this.m_stopping)
                    {
                        return;
                    }

                    // Note: shouldn't encounter a "NoSuchMethodException", as a check has already been
                    // performed that specified CollectionsMonitor supports the required method(s).

                    // log debug message if tracing is enabled
                    if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGER_EXCEPTIONS) > 0)
                    {
                        if (this.m_retrievalFailure == null || !ex.getClass().isInstance(this.m_retrievalFailure) || ex.getMessage().equals(this.m_retrievalFailure.getMessage()))
                        {
                            if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                            {
                                LoggerComponent.this.m_context.logMessage("Exception while attempting to retrieve stored metrics by invoking CollectionsMonitor's \"" + GET_STORED_METRICS_METHOD_NAME + "\" method, CollectionsMonitor may not be ONLINE - CollectionsMonitor runtimeID = " + cmRuntimeID,
                                                                          ex, Level.TRACE);
                            }

                            this.m_retrievalFailure = ex;
                        }
                    }


                    // check CM's state; if not ONLINE, log [one and only one] message
                    // that the LoggerComponent will keep trying to contact the CM until
                    // it [the CM] is ONLINE.
                    if (!setCMStatusMsgFlags(cmRuntimeID))
                    {
                        LoggerComponent.this.m_context.registerErrorCondition("Error occurred while attempting to contact CollectionsMonitor = " + cmRuntimeID,Level.SEVERE);
                        synchronized (LoggerComponent.this.m_lock)
                        {
                            LoggerComponent.this.stop();
                        }
                        return;
                    }
                } //catch (Exception ex)
            } //while (getMoreMetrics)
        }

        private void logStoredNotifications(String cmRuntimeID, String[] componentCollectionIDs)
        {
            boolean getMoreNotifications = true;
            IMonitoredNotifications monitoredNotifications = null;

            // retrieve stored notifications...
            while (getMoreNotifications)
            {
                try
                {
                    // set the lookback period (do it here so that any changes to configured value will be picked up on each iteration)
                    long lookback;
                    synchronized (LoggerComponent.this.m_changeLock)
                    {
                        lookback = LoggerComponent.this.m_maxLookbackPeriod * 60000;  /*convert lookback period from minutes to milliseconds*/
                    }

                    // the lastRetrievalTimestamp will intentionally be "null" for the first invocation - this will force the full lookback period to be used by the CM
                    monitoredNotifications = null;
                    Object[] params = { componentCollectionIDs,
                                        m_lastNotificationRetrievalTimestamp,
                                        new Long(lookback)
                                      };
                    String [] signatures = { "[Ljava.lang.String;",
                                             "java.lang.Long",
                                             "java.lang.Long"
                                           };
                    boolean synchronous = true;

                    long cmElapsedTime = System.currentTimeMillis();
                    monitoredNotifications = (IMonitoredNotifications) LoggerComponent.this.m_frameworkContext.invoke(cmRuntimeID, LoggerComponent.GET_STORED_NOTIFICATIONS_METHOD_NAME, params, signatures, synchronous, 0);  // "0" to indicate that container's timeout value will be used
                    cmElapsedTime = System.currentTimeMillis() - cmElapsedTime;

                    // process the results
                    long logElapsedTime = System.currentTimeMillis();
                    try
                    {
                        processRetrievedNotifications(monitoredNotifications);
                        logElapsedTime = System.currentTimeMillis() - logElapsedTime;

                        // clear flag for log failure log message
                        this.m_notificationLogFailure = null;
                    }
                    catch (Exception e)
                    {
                        // this will only occur if the LoggerComponent instance has been configured using an
                        // evaluation license code AND the logged event count limit [as determined by the evaluation license code]
                        // has been exceeded; in that case, set the lastNotificationReceivedTimestamp and break out of the "while" loop

                        //If no new notifications then use the previous m_lastNotificationRetrievalTimestamp
                        if (monitoredNotifications.getNotifications().length > 0)
                        {
                            m_lastNotificationRetrievalTimestamp = new Long(monitoredNotifications.getLatestTimestamp());
                        }

                        // log debug message if tracing is enabled
                        if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGER_EXCEPTIONS) > 0)
                        {
                            if (this.m_notificationLogFailure == null || !e.getClass().isInstance(this.m_notificationLogFailure) || e.getMessage().equals(this.m_notificationLogFailure.getMessage()))
                            {
                                if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                                {
                                    LoggerComponent.this.m_context.logMessage("Failure while logging notification to log4j, trace follows...", e, Level.TRACE);
                                }
                                else
                                {
                                    LoggerComponent.this.m_context.logMessage("Failure while logging notification to log4j", Level.TRACE);
                                }
                            }

                            this.m_notificationLogFailure = e;
                        }

                        break;
                    }

                    // check if there is more data to retrieve from the CollectionsMonitor
                    getMoreNotifications = monitoredNotifications.hasMoreDataToRetrieve();

                    if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGING_CYCLE) > 0)
                    {
                        StringBuffer traceMessage = new StringBuffer("Retrieve and log notifications: ");
                        traceMessage.append("retrieve duration (millis)=").append(cmElapsedTime).append(", log duration (millis)=").append(logElapsedTime);
                        if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                        {
                            traceMessage.append(", timestamp1=").append(m_lastNotificationRetrievalTimestamp).append(", lookback=").append(lookback);
                            traceMessage.append(", notification count=").append(monitoredNotifications.getNotifications().length).append(", timestamp2=").append(monitoredNotifications.getLatestTimestamp()).append(", more=").append(getMoreNotifications);
                        }
                        LoggerComponent.this.m_context.logMessage(traceMessage.toString(), Level.TRACE);
                    }

                    // update the lookback period for the next retrieval

                    //If no new notifications then use the previous m_lastNotificationRetrievalTimestamp
                    if (monitoredNotifications.getNotifications().length > 0)
                    {
                        m_lastNotificationRetrievalTimestamp = new Long(monitoredNotifications.getLatestTimestamp());
                    }

                    // clear flag for retrieval failure log message
                    this.m_retrievalFailure = null;
                }
                catch (Exception ex)
                {
                    // first of all, set the "getMoreNotifications" flag to false.
                    // This is because even if this exception occurred due to the CM temporarily going offline,
                    // we will want to wait for the next polling cycle in order to make sure that all of the
                    // stored data from the CM is retrieved [with the exception, we don't know how much of the
                    // data was actually returned for this polling cycle].
                    getMoreNotifications = false;

                    // leave the lastNotificationRetrievalTimestamp value as it was, as the retrieval attempt failed...

                    // if the thread is being stopped, simply return...
                    if (LoggerComponent.this.m_stopping)
                    {
                        return;
                    }

                    // Note: shouldn't encounter a "NoSuchMethodException", as a check has already been
                    // performed that specified CollectionsMonitor supports the required method(s).

                    // log debug message if tracing is enabled
                    if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGER_EXCEPTIONS) > 0)
                    {
                        if (this.m_retrievalFailure == null || !ex.getClass().isInstance(this.m_retrievalFailure) || ex.getMessage().equals(this.m_retrievalFailure.getMessage()))
                        {
                            if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                            {
                                LoggerComponent.this.m_context.logMessage("Exception while attempting to retrieved stored notifications by invoking CollectionsMonitor's \"" + GET_STORED_NOTIFICATIONS_METHOD_NAME + "\" method, CollectionsMonitor may not be ONLINE - CollectionsMonitor runtimeID = " + cmRuntimeID,
                                                                          ex, Level.TRACE);
                            }

                            this.m_retrievalFailure = ex;
                        }
                    }

                    // check CM's state; if not ONLINE, log [one and only one] message
                    // that the LoggerComponent will keep trying to contact the CM until
                    // it [the CM] is ONLINE.
                    if (!setCMStatusMsgFlags(cmRuntimeID))
                    {
                        LoggerComponent.this.m_context.registerErrorCondition("Error occurred while attempting to contact CollectionsMonitor = " + cmRuntimeID,Level.SEVERE);
                        synchronized (LoggerComponent.this.m_lock)
                        {
                            LoggerComponent.this.stop();
                        }
                        return;
                    }
                } //catch (Exception ex)
            } //while (getMoreNotifications)
        }


        private void processRetrievedMetrics(IMonitoredMetrics monitoredMetrics) throws Exception
        {
            // process the retrieved metrics (if there are any)
            IHistoricalMetric[] metrics = monitoredMetrics.getMetrics();  // MonitoredMetrics never returns null for "getMetrics" - at worst, it returns an empty array
            for (int i=0, n = metrics.length; i < n; i++)
            {
                // logic to log the notification info...
                logMetric((IHistoricalMetric)metrics[i]);
            }
        }

        private void processRetrievedNotifications(IMonitoredNotifications monitoredNotifications) throws Exception
        {
            // process the retrieved metrics (if there are any)
            INotification[] notifications = monitoredNotifications.getNotifications();  // MonitoredNotifications never returns null for "getNotifications" - at worst, it returns an empty array
            for (int i=0, n = notifications.length; i < n; i++)
            {
                // logic to log the notification info...
                logNotification((INotification)notifications[i]);
            }
        }


        private void logNotification(INotification notification)
        {
            try
            {
                if ( (LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGER_INVOCATIONS) > 0)
                {
                    if ( (LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                    {
                        long timestamp = notification.getTimeStamp();
                        Date timestampDate = new Date(timestamp);
                        LoggerComponent.this.m_context.logMessage("Logging notification with event name: " + notification.getEventName() + ", event type: " + notification.getType() + ", source host: " + notification.getSourceHost() + ", source component: " + notification.getSourceIdentity().getCanonicalName() + ", and timestamp: " + timestampDate.toString(), Level.TRACE);
                    }
                    else
                    {
                        LoggerComponent.this.m_context.logMessage("Logging notification...", Level.TRACE);
                    }
                }

                // format the notification
                Object logEntry = formatNotification(notification);

                // get the log level from the notification its
                int severity = notification.getSeverity();

                // create a context string to identify which Logger to use
                // [notifications have the prefix so that they can be readily
                // distinguished (by script-based tools,etc.) from metrics
                // that are logged. This is a carry-over from the "NetMon" product.]
                String context = LoggerComponent.LOGGER_NOTIFICATION_PREFIX + notification.getType();  // e.g. "sonicmf.notification.broker.connections.count"

                // log the notification
                Logger logger = Logger.getLogger(context);
                if (severity == Level.SEVERE)
                {
                    logger.log(org.apache.log4j.Level.ERROR,logEntry);
                }
                else if (severity == Level.WARNING)
                {
                    logger.log(org.apache.log4j.Level.WARN,logEntry);
                }
                else
                {
                    logger.log(org.apache.log4j.Level.INFO,logEntry);
                }

                // update the notifications-logged-per-minute metric
                updateStatistic(LoggerComponent.this.m_notificationsLoggedPerMinuteStatistic,(long)1);  // increment the count of notifications logged per minute by one....

                // clear flag for log failure log message
                this.m_notificationLogFailure = null;
            }
            catch (RuntimeException e)
            {
                // log debug message if tracing is enabled
                if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGER_EXCEPTIONS) > 0)
                {
                    if (this.m_notificationLogFailure == null || !e.getClass().isInstance(this.m_notificationLogFailure) || e.getMessage().equals(this.m_notificationLogFailure.getMessage()))
                    {
                        if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                        {
                            LoggerComponent.this.m_context.logMessage("Failure while logging notification to log4j, trace follows...", e, Level.TRACE);
                        }
                        else
                        {
                            LoggerComponent.this.m_context.logMessage("Failure while logging notification to log4j", e, Level.TRACE);
                        }
                    }

                    this.m_notificationLogFailure = e;
                }

                throw e;
            }
        }

        private Object formatNotification(INotification notification)
        {
            return getLogFormatter().formatNotification(notification);
        }

        private void logMetric(IHistoricalMetric metric)
        {
            try
            {
                if ( (LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGER_INVOCATIONS) > 0)
                {
                    if ( (LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                    {
                        LoggerComponent.this.m_context.logMessage("Logging metric with metric name: " + metric.getMetricIdentity().getAbsoluteName() + ", value: " + metric.getValue() + ", source: " + metric.getSource() + ", and timestamp: " + metric.getCurrencyTimestamp(), Level.TRACE);
                    }
                    else
                    {
                        LoggerComponent.this.m_context.logMessage("Logging metric...", Level.TRACE);
                    }
                }

                // format the metric
                Object logEntry = formatMetric(metric);

                // convert the context from canonical name format into
                // a period-limited form that can be used to retrieve the
                // appropriate logger
                String periodDelimitedContext = LoggerComponent.LOGGER_METRIC_PREFIX + canonicalToLog4jFormat(metric.getSource());

                // log the metric
                Logger logger = Logger.getLogger(periodDelimitedContext);
                logger.log(org.apache.log4j.Level.INFO,logEntry);

                // update the metrics-logged-per-minute metric
                updateStatistic(LoggerComponent.this.m_metricsLoggedPerMinuteStatistic,(long)1);  // increment the count of metrics logged per minute by one....

                // clear flag for log failure log message
                this.m_metricLogFailure = null;
            }
            catch (RuntimeException e)
            {
                // log debug message if tracing is enabled
                if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_LOGGER_EXCEPTIONS) > 0)
                {
                    if (this.m_metricLogFailure == null || !e.getClass().isInstance(this.m_metricLogFailure) || e.getMessage().equals(this.m_metricLogFailure.getMessage()))
                    {
                        if ((LoggerComponent.this.m_traceMask & LoggerComponent.TRACE_DETAIL) > 0)
                        {
                            LoggerComponent.this.m_context.logMessage("Failure while logging metric to log4j, trace follows...", e, Level.TRACE);
                        }
                        else
                        {
                            LoggerComponent.this.m_context.logMessage("Failure while logging metric to log4j", e, Level.TRACE);
                        }
                    }

                    this.m_metricLogFailure = e;
                }

                throw e;
            }
        }

        private Object formatMetric(IHistoricalMetric metric)
        {
            return getLogFormatter().formatHistoricalMetric(metric);
        }

        private String canonicalToLog4jFormat(String canonicalName)
        {
            // Basically, this is to convert a string such as
            // "Domain1.Container1:ID=Broker1" into the string
            // "Domain1.Container1.Broker1".  The period-delimited
            // format is used by log4j when determining which
            // Appender(s) to use for logging.

            // get the first part of the context (this is the part before ":ID=")
            int colonIndex = canonicalName.indexOf(":ID=");
            String first = canonicalName.substring(0, colonIndex);

            // get the second part of the context (the part after ":ID=")
            String second = canonicalName.substring(colonIndex + 4);

            // Now build and return the full period-delimited notation
            return first + "." + second;
        }

        private void updateStatistic(IStatistic statistic, long value)
        {
            // grab a reference to the statistic in case another
            // thread sets the reference to null between the check and the update
            IStatistic s = statistic;
            if (s != null)
            {
                s.updateValue(value);
            }
        }
        
        public Object getLoggingThreadLockObj() {
            return m_loggingThreadLockObj;
        }
    } //LoggingThread inner class

    // REVISIT: this could be optimized by saving the formatter and only updating it when m_logFormat or
    //          m_logTextDelimiter changes.
    private LogFormatter getLogFormatter()
    {
        LogFormatter logFormatter = null;
        synchronized (m_changeLock)
        {
            if (LoggerComponent.DELIMITED_TEXT_LOG_FORMAT_TYPE.equals(m_logFormat))
            {
                logFormatter = DelimitedTextLogFormatter.getInstance(m_logTextDelimiter);
            }
            else if (LoggerComponent.XML_LOG_FORMAT_TYPE.equals(m_logFormat))
            {
                logFormatter = XmlLogFormatter.INSTANCE;
            }
            else // must be "Java Object" format...
            {
                logFormatter = PassthroughLogFormatter.INSTANCE;
            }
        }
        return logFormatter;
    }
}
