// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.mf.framework.manager;

import java.io.IOException;
import java.rmi.dgc.VMID;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.CompositeType;
import javax.management.openmbean.OpenDataException;
import javax.management.openmbean.OpenType;
import javax.management.openmbean.SimpleType;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.comm.IGlobalComponentListener;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.MFException;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.MFServiceNotActiveException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.impl.ElementIdentity;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.manager.IMetricsRegistrar;
import com.sonicsw.mf.common.runtime.ICollectiveOpStatus;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.IContainerExitCodes;
import com.sonicsw.mf.common.runtime.IContainerIdentity;
import com.sonicsw.mf.common.runtime.IContainerState;
import com.sonicsw.mf.common.runtime.IFaultTolerantState;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.IState;
import com.sonicsw.mf.common.runtime.IStateController;
import com.sonicsw.mf.common.runtime.IStateListener;
import com.sonicsw.mf.common.runtime.IStateManager;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.NonRecoverableStateChangeException;
import com.sonicsw.mf.common.runtime.RecoverableStateChangeException;
import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.common.runtime.impl.CollectiveOpStatus;
import com.sonicsw.mf.common.runtime.impl.ComponentState;
import com.sonicsw.mf.common.runtime.impl.ContainerIdentity;
import com.sonicsw.mf.common.runtime.impl.ContainerState;
import com.sonicsw.mf.framework.AbstractFrameworkComponent;
import com.sonicsw.mf.framework.IConnectorServer;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.IGlobalFrameworkComponent;
import com.sonicsw.mf.framework.INotificationHandler;
import com.sonicsw.mf.framework.IPermissionsManager;
import com.sonicsw.mf.framework.ITaskScheduler;
import com.sonicsw.mf.framework.util.NotificationForwarder;
import com.sonicsw.mf.framework.util.StateManager;
import com.sonicsw.mf.jmx.client.MFNotification;
import com.sonicsw.mf.mgmtapi.config.constants.IAgentManagerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IComponentCollectionConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerCollectionConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.runtime.IActivationDaemonProxy;
import com.sonicsw.mf.mgmtapi.runtime.IAgentManagerProxy;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;
import com.sonicsw.mf.mgmtapi.runtime.IDirectoryServiceProxy;

public final class AgentManager
extends AbstractFrameworkComponent
implements IGlobalFrameworkComponent, INotificationHandler, IGlobalComponentListener
{
    private static boolean DEBUG = false;

    private String m_configID;
    private String m_primaryConfigID;

    private NotificationForwarder m_stateNotificationForwarder;

    public static final String STARTUP_NOTIFICATION_TYPE = "Startup";
    public static final String UNREACHABLE_NOTIFICATION_TYPE = "Unreachable";
    public static final String FAILOVER_NOTIFICATION_TYPE = "Failover";

    // the following constant should end up in IAgentManagerConstants once the file gets
    // generated
    public static final String BACKUP_AGENT_MANAGER_TYPE = "MF_BACKUP_AGENT_MANAGER";
    public static final String CONFIG_ELEMENT_REFERENCES_ATTR = "CONFIG_ELEMENT_REFERENCES";
    public static final String PRIMARY_CONFIG_ELEMENT_REF_ATTR = "PRIMARY_AM_CONFIG_ELEMENT_REF";
    public static final String BACKUP_CONFIG_ELEMENT_REF_ATTR = "BACKUP_AM_CONFIG_ELEMENT_REF";

    private ITaskScheduler m_taskScheduler;

    private CurrentDomainState m_domainState;
    private DomainStateMonitor m_stateMonitor;

    private static final String LOGFAILURE_NOTIFICATION_TYPE = "Failure";
    private static final String LOGTHRESHOLD_NOTIFICATION_TYPE = "Threshold";

    private IPermissionsManager m_permissionsManager;

    private IMetricsRegistrar m_metricsRegistrar;

    private static final String MANAGER_TRACE_MASK_VALUES = "16=status polling,32=container unreachable,64=forwarded notifications,128=fault detection";
    public static final int TRACE_STATUS_POLLING = 16;
    public static final int TRACE_CONTAINER_UNREACHABLE = 32;
    public static final int TRACE_FORWARDED_NOTIFICATIONS = 64;
    public static final int TRACE_FAULT_DETECTION = 128; // Fault tolerance fault detection
    private static final String FAULT_TOLERANCE_ROLE_DEFAULT = "";
    private static final String FAULT_TOLERANCE_ROLE_PRIMARY = "PRIMARY";
    private static final String FAULT_TOLERANCE_ROLE_BACKUP = "BACKUP";

    private String m_uniqunessCallID;
    private Thread m_uniquenessCheckThread;
    private boolean m_uniquenessCallReceived;
    private boolean m_responseSent;

    private IConnectorServer m_connectorServer;
    private Object m_uniquenessCheckLockObj = new Object();
    private SdfMFTracingIntegration m_SdfMFTracingIntegration;

    //
    // fault tolerant support
    //
    private boolean m_isNotFaultTolerant;
    boolean m_isPrimary;
    boolean m_isBackup;

    private boolean m_allowFailover = true;
    private boolean m_abortFailover = false;
    private String m_faultToleranceRole = FAULT_TOLERANCE_ROLE_DEFAULT;
    private long m_faultDetectionInterval;
    private IStateManager m_faultToleranceStateManager;
    //VMID of this AgentManager, used as Key for calling leaseLock()
    private String m_vmID = new VMID().toString();
    private FaultDetector m_faultDetector;

    private CentralizedLogger m_centralizedLogger;
    public static final long LATENCY_LEASE_LOCK = 5000; // Latency time for leaseLock op
    public static final long LEASE_REQUEST_TIMEOUT_MIN = 10000; // minimal request timeout for leaseLock

    private static final String[] LEASELOCK_SIGNATURE = new String[]
    {
        String.class.getName(), // lockName
        String.class.getName(), // key
        Integer.class.getName() // leaseDuration
    };

    private static final ArrayList ATTRIBUTE_INFOS = new ArrayList();
    private static final ArrayList OPERATION_INFOS = new ArrayList();
    private static final ArrayList NOTIFICATION_INFOS = new ArrayList();

    private static CompositeType LOG_COMPOSITE_TYPE;
    private static String[] LOG_COMPOSITE_TYPE_ITEM_NAMES;

    static
    {
        //
        // Attributes
        //
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("FaultToleranceRole", String.class.getName(), "The fault tolerant role of the AM (primary or backup) or null if not fault tolerant.", true, false, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("FaultTolerantState", Short.class.getName(), "The current fault tolerant state of this Agent Manager.", true, false, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("FaultTolerantStateString", String.class.getName(), "The description of the current fault tolerant state of this Agent Manager.", true, false, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("AllowFailover", Boolean.class.getName(), "When this Agent Manager is in a standby state and this attribute is set to 'false', fault tolerant failover to active will not occur.", true, true, false));
        // Polling Threads
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("MaxPollingThreads", Integer.class.getName(),"The maximum number of threads that the Agent Manager can create to poll containers for their state.", true, false, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("MinPollingThreads", Integer.class.getName(),"The minimum number of threads that the Agent Manager will cache for reuse to poll containers for their state.", true, false, false));
        // Centralized log
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogFile", String.class.getName(), "The directory pathname of the centralized log file. Defaults to {<domain>.log} in the container's working directory.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogFileSize", Long.class.getName(), "The number of bytes written to the centralized log file. This value gets reset to 0 when the log is cleared or archived.", true, false, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogFileSizeThreshold", Long.class.getName(),"Once size of the the centralized log file reaches this threshold or for each time the size exceeds this threshold by a further 10%, the Agent Manager will send a warning notification.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("LogFileRolloverSizeThreshold", Long.class.getName(),"If the current centralized log file size equals or exceeds this threshold at midnight, then the Agent Manager will roll over the old contents to an archive file and create a new empty centralized log file.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("logFileRolloverTimeInterval", Integer.class.getName(),"Time interval to rollover the log file contents.", true, true, false));

        //
        // Operations
        //
        MBeanParameterInfo[] mbParamInfos = null;

        // collection invoke
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("configID", String.class.getName(), "The path to the component collection configuration (either storage or logical)."),
            new MBeanParameterInfo("operationName", String.class.getName(), "The name of the operation to invoke on each of the containers/components in the collection."),
            new MBeanParameterInfo("operationParams", Object[].class.getName(), "The parameters to the operation."),
            new MBeanParameterInfo("operationSignature", String[].class.getName(), "The signature of the operation."),
            new MBeanParameterInfo("sync", Boolean.class.getName(), "If false, aynchronously invoke the operation on the collection and return null, otherwise build and return a collective result."),
            new MBeanParameterInfo("timeout", Long.class.getName(), "The time in milliseconds to wait for all responses to be received. The value should be greater than the request timeout value. The value is ignored if asynchronous execution is specified.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("invoke", "Invokes an operation on the given component collection. If synchronous execution is requested, returns the return values or exceptions of the individual requests to member of the collection. The return object is an aggregation of return values and/or exceptions wrapped in a <a href=\"../../common/runtime/ICollectiveOpStatus.html\">ICollectiveOpStatus</a>. Returns null if asynchronous execution is requested.",
                                                   mbParamInfos, Object.class.getName(), MBeanOperationInfo.ACTION_INFO));

        // collection set attribute
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("configID", String.class.getName(), "The path to the component collection configuration (either storage or logical)."),
            new MBeanParameterInfo("attributeName", String.class.getName(), "The name of the attribute."),
            new MBeanParameterInfo("attributeValue", Object.class.getName(), "The value of the attribute."),
            new MBeanParameterInfo("sync", Boolean.class.getName(), "If false, aynchronously invoke the operation on the collection and return null, otherwise build and return a collective result."),
            new MBeanParameterInfo("timeout", Long.class.getName(), "The time in milliseconds to wait for all responses to be received. The value should be greater than the request timeout value. The value is ignored if asynchronous execution is specified.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("setAttribute", "Sets an attribute on the given component collection. If synchronous execution is requested, returns the return values or exceptions of the individual requests to member of the collection. Returns null if asynchronous execution is requested.",
                                                   mbParamInfos, ICollectiveOpStatus.class.getName(), MBeanOperationInfo.ACTION));

        // collection set attributes
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("configID", String.class.getName(), "The path to the component collection configuration (either storage or logical)."),
            new MBeanParameterInfo("attributeNames", String[].class.getName(), "The names of the attributes."),
            new MBeanParameterInfo("attributeValues", Object[].class.getName(), "The values of the attributes."),
            new MBeanParameterInfo("sync", Boolean.class.getName(), "If false, aynchronously invoke the operation on the collection and return null, otherwise build and return a collective result."),
            new MBeanParameterInfo("timeout", Long.class.getName(), "The time in milliseconds to wait for all responses to be received. The value should be greater than the request timeout value. The value is ignored if asynchronous execution is specified.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("setAttributes", "Sets attributes on the given component collection. If synchronous execution is requested, returns the return values or exceptions of the individual requests to member of the collection. Returns null if asynchronous execution is requested.",
                                                   mbParamInfos, ICollectiveOpStatus.class.getName(), MBeanOperationInfo.ACTION));

        // domain runtime state
        OPERATION_INFOS.add(new MBeanOperationInfo("getCollectiveState", "Gets the runtime state of the whole domain as recorded by the Agent Manager. Returns an array of IContainerState instances.",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, IState[].class.getName(), MBeanOperationInfo.INFO));

        // collection runtime state
        mbParamInfos = new MBeanParameterInfo[]
        {
        new MBeanParameterInfo("configID", String.class.getName(), "The configuration ID of the collection for which the status is being sought. Returns an array of IContainerState or IComponentState instances that describe the state of the given collection. If an item in the collection no longer represents a known runtime identity then no runtime status will be provided in the returned array.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getCollectiveState", "Gets the runtime state of the given collection as recorded by the Agent Manager.",
                                                   mbParamInfos, IState[].class.getName(), MBeanOperationInfo.INFO));

        // containers runtime state
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("runtimeIds", String[].class.getName(), "The list of the containers runtime identities for which the status is being sought.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getCollectiveState", "Gets the runtime state for the given list of containers as recorded by the Agent Manager. Returns an array of IContainerState instances.",
                                                   mbParamInfos, IState[].class.getName(), MBeanOperationInfo.INFO));

        // relinquish the active role to the current standby
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("seconds", Integer.class.getName(), "The minimum time the active role will be suspended.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("suspendActiveRole", "Relinquish the current active role to the the standby. For an active AM, this operation attempts to relinquish the active role to the current standby. The active AM will wait up to the given number of seconds for a standby to takeover the active role; if a standby does not takeover the active role in this period, the AM will continue its active role.",
                                                   mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));

        // Centralized log
        OPERATION_INFOS.add(new MBeanOperationInfo("clearLogFile", "Clear the centralized log file.",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(), MBeanOperationInfo.ACTION));
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("filename", String.class.getName(), "A valid filename (including path) to save the centralized log file to.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("saveLogFile", "Saves the contents of the centralized log file to the given file name.",
                                                   mbParamInfos, Void.class.getName(), MBeanOperationInfo.ACTION));
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("fromPosition", Long.class.getName(), "The starting position from which to read the log for the given date. If null, then the total log length minus the given readLength is assumed (i.e. the tail of the logging output). The total log length is calculated by summing the size of all log files (current + previous versions."),
            new MBeanParameterInfo("readLength", Long.class.getName(), "The maximum amount of bytes to read from the given (or assumed) starting position. The max size for this value is 1Mb. If null, 200Kb is assumed.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getLogExtract", "Reads the requested bytes from the centralized log file and returns them as a String.",
                                                   mbParamInfos, String.class.getName(), MBeanOperationInfo.INFO));
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("fromPosition", Long.class.getName(), "The starting position from which to read the log for the given date. If null, then the total log length minus the given readLength is assumed (i.e. the tail of the logging output). The total log length is calculated by summing the size of all log files (current + previous versions."),
            new MBeanParameterInfo("readLength", Long.class.getName(), "The maximum amount of bytes to read from the given (or assumed) starting position. The max size for this value is 1Mb. If null, 200Kb is assumed.")
        };
        try
        {
            LOG_COMPOSITE_TYPE_ITEM_NAMES = new String[] { "LogExtract", "LogFileSize" };
            LOG_COMPOSITE_TYPE = new CompositeType("LogExtract", "An extract from and the length of the centralized log",
                                                   LOG_COMPOSITE_TYPE_ITEM_NAMES,
                                                   new String[] { "Log extract String", "Total log size" },
                                                   new OpenType[] { SimpleType.STRING, SimpleType.LONG });
        } catch(OpenDataException e) { e.printStackTrace(); } // should not happen
        OPERATION_INFOS.add(new MBeanOperationInfo("getLogExtractAndLogFileSize", "Reads the requested bytes from the centralized log file and returns them as a String encapsulated with the current log size. The current log size is the size of the current log file plus the sum of all available prior versions. When evaluating log output to be read, the current log file and previous versions (due to rollover activity) will be effectively viewed as a single file.",
                                                   mbParamInfos, CompositeData.class.getName(), MBeanOperationInfo.INFO));
        // when internal usage flag set
        if (IContainer.QA_MODE)
        {
            // trigger container log file rollover attempt
            OPERATION_INFOS.add(new MBeanOperationInfo("attemptLogFileRollover", "Attempt a centralized log file rollover (QA mode only).",
                                                       IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(), MBeanOperationInfo.ACTION));
        }

        //
        // Notifications
        //
        String[] notifTypes = null;

        // startup
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            STARTUP_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, INotification.CLASSNAME, "Agent/Container startup completed."));
        // container unreachable
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            UNREACHABLE_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, INotification.CLASSNAME, "Container unreachable."));
        // FT role change
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY],
            FAILOVER_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, INotification.CLASSNAME, "Standby service has failed over to become the active service."));
        // log failure
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.LOG_SUBCATEGORY],
            LOGFAILURE_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, "Failure to write to the centralized log file."));
        // log threshold
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.LOG_SUBCATEGORY],
            LOGTHRESHOLD_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, "The size of the current centralized log has reached or exceeded (by a 10% increment) the configured threshold."));
    }

    public AgentManager()
    {
        super();

        m_SdfMFTracingIntegration = new SdfMFTracingIntegration();
        m_SdfMFTracingIntegration.register();
    }

    private void makeGlobalComponentUniquenessCall()
    {
        m_uniqunessCallID =  new java.rmi.server.UID().toString();
        m_uniquenessCheckThread = new Thread(IAgentManagerProxy.GLOBAL_ID + " - Uniqueness Checker")
        {
            @Override
            public void run()
            {
               // We want to make sure the uniquness call will get through even if the broker is down.
               // So we keep trying until we get our own message (or a message from another, duplicate, DS
               while (!m_uniquenessCallReceived)
               {
                   // Send a message to any other Agent Manager in this domain
                   // to verify that there only one. globalComponentUniquenessCall
                   // invokes the uniquenessCheck() method of all the Directory Service
                   // components in this domain - see uniquenessCheck() bellow.
                   try
                   {
                       AgentManager.super.m_frameworkContext.makeGlobalComponentUniquenessCall(
                                IAgentManagerProxy.GLOBAL_ID,
                                m_uniqunessCallID,
                                m_isNotFaultTolerant ? null : m_faultToleranceRole);
                   }
                   catch (Exception e)
                   {
                       throw createMFRuntimeException(e);
                   }
                   if (m_uniquenessCallReceived)
                {
                    return;
                }
                else
                   {
                       try
                       {
                           synchronized(m_uniquenessCheckLockObj)
                           {
                               m_uniquenessCheckLockObj.wait(3000);
                           }
                       }
                       catch (InterruptedException e) { }
                   }
               }
            }
         };
         m_uniquenessCheckThread.start();
    }

    @Override
    public void uniquenessCheck(String containerID, String callID)
    {
        String thisContainerName = super.m_frameworkContext.getContainer().getContainerIdentity().getCanonicalName();
        synchronized(m_uniquenessCheckLockObj)
        {
            m_uniquenessCallReceived = true;
            m_uniquenessCheckLockObj.notifyAll();
        }

        if (m_responseSent)
        {
            return;
        }


        if (!containerID.equals(thisContainerName))
        {
            String errMsg = "Domain has 2 active Agent Manager instances (both will be shutdown to allow configuration resolution):" + '\n' + '\n' +
                            '\t' + "One is in this container : " + thisContainerName + '\n' +
                            '\t' + "The other is in container: " + containerID;
            super.m_context.logMessage(errMsg, Level.SEVERE);

            // Makes sure the other container with AM knows about the other AM instance
            try
            {
                Thread.sleep(3000);
                super.m_frameworkContext.makeGlobalComponentUniquenessCall(
                        IAgentManagerProxy.GLOBAL_ID,
                        m_uniqunessCallID,
                        m_isNotFaultTolerant ? null : m_faultToleranceRole);
                m_responseSent = true;
            }
            catch (Exception e)
            {
                throw createMFRuntimeException(e);
            }

            super.m_container.shutdown(IContainerExitCodes.AM_ALREADY_RUNNING_EXIT_CODE);
        }

        if (!callID.equals(m_uniqunessCallID))
        {
            super.m_context.logMessage("There are two active Agent Manager instances in this container.", Level.SEVERE);
            super.m_container.shutdown(IContainerExitCodes.AM_ALREADY_RUNNING_EXIT_CODE);
        }
    }

    @Override
    public String getGlobalID() { return IAgentManagerProxy.GLOBAL_ID; }

    @Override
    public void init(IComponentContext context)
    {
        super.init(context);

        m_connectorServer = super.m_frameworkContext.getContainer().getConnectorServer();
        m_permissionsManager = super.m_frameworkContext.getPermissionsManager();
        m_centralizedLogger = new CentralizedLogger(context);

        readConfiguration(context);

        // needs to be reset, in case it was set prior to initialization
        setTraceMask(super.getTraceMask());

        // create the notification forwarder
        m_stateNotificationForwarder = new NotificationForwarder(super.m_frameworkContext);

        // get all the elements and establish the domain state
        m_domainState = new CurrentDomainState((IElement[])context.getConfigurations("/containers", true));
        m_taskScheduler = super.m_container.getTaskScheduler();

        try
        {
            if (m_isNotFaultTolerant)
            {
                super.m_container.addGlobalComponentSupport(getGlobalID(), null, this);
                super.m_container.addGlobalComponentSupport(getGlobalID(), FAULT_TOLERANCE_ROLE_PRIMARY, this);
            }
            else
            {
                super.m_context.logMessage(IContainer.NEWLINE + IContainer.NEWLINE + '\t' + "Fault tolerant role \"" + m_faultToleranceRole + "\"." + IContainer.NEWLINE, Level.CONFIG);
                super.m_container.addGlobalComponentSupport(getGlobalID(), m_faultToleranceRole, this);
            }
        }
        catch (Exception e)
        {
            String msgPrefix = m_isNotFaultTolerant ? "" : '[' + m_faultToleranceRole + "] ";
            m_context.logMessage(msgPrefix + "Failed communications initialization, trace follows...", e, Level.SEVERE);
            super.m_container.shutdown(IContainerExitCodes.COMMS_FAILURE_EXIT_CODE);
        }

        if (m_isNotFaultTolerant)
        {
            m_faultToleranceStateManager = new StateManager(IFaultTolerantState.STATE_VALUES, IFaultTolerantState.STATE_NOT_FAULT_TOLERANT);
        }
        else
        {
            // configure the state manager
            m_faultToleranceStateManager = new StateManager(IFaultTolerantState.STATE_VALUES, IFaultTolerantState.STATE_WAITING);
            super.m_context.logMessage("Initial state: \"" + IFaultTolerantState.STATE_TEXT[IFaultTolerantState.STATE_WAITING] + '"', Level.INFO);
            m_faultToleranceStateManager.registerStateListener(new StateListener(), null);

            IStateController controller = null;
            controller = new ToActiveStateController();
            m_faultToleranceStateManager.registerStateController(controller, IFaultTolerantState.STATE_WAITING, IFaultTolerantState.STATE_ACTIVE, null);
            m_faultToleranceStateManager.registerStateController(controller, IFaultTolerantState.STATE_STANDBY, IFaultTolerantState.STATE_ACTIVE, null);

            controller = new FromActiveStateController();
            m_faultToleranceStateManager.registerStateController(controller, IFaultTolerantState.STATE_ACTIVE, IFaultTolerantState.STATE_WAITING, null);
            m_faultToleranceStateManager.registerStateController(controller, IFaultTolerantState.STATE_ACTIVE, IFaultTolerantState.STATE_STANDBY, null);

            controller = new WaitingStandbyStateController();
            m_faultToleranceStateManager.registerStateController(controller, IFaultTolerantState.STATE_WAITING, IFaultTolerantState.STATE_STANDBY, null);

            // start the fault detection
            startFaultDetector();
        }

        makeGlobalComponentUniquenessCall();

        // metrics setup
        IMetricInfo[] domainStateMonitorMetricsInfo = DomainStateMonitor.getMetricsInfo();
        IMetricInfo[] infos = new IMetricInfo[domainStateMonitorMetricsInfo.length] ;
        System.arraycopy(domainStateMonitorMetricsInfo, 0, infos, 0, domainStateMonitorMetricsInfo.length);
        m_metricsRegistrar = super.m_context.initMetricsManagement(infos);
        DomainStateMonitor.initMetrics(m_metricsRegistrar);
    }

    @Override
    public MBeanAttributeInfo[] getAttributeInfos() { return (MBeanAttributeInfo[])ATTRIBUTE_INFOS.toArray(IEmptyArray.EMPTY_ATTRIBUTE_INFO_ARRAY); }

    @Override
    public MBeanOperationInfo[] getOperationInfos() { return (MBeanOperationInfo[])OPERATION_INFOS.toArray(IEmptyArray.EMPTY_OPERATION_INFO_ARRAY); }

    @Override
    public MBeanNotificationInfo[] getNotificationInfos() { return (MBeanNotificationInfo[])NOTIFICATION_INFOS.toArray(IEmptyArray.EMPTY_NOTIFICATION_INFO_ARRAY); }

    @Override
    public synchronized void start()
    {
        if (super.m_state == IComponentState.STATE_ONLINE)
        {
            return;
        }

        if (m_isNotFaultTolerant)
        {
            startDomainStateMonitor();
        }

        super.start();
    }

    @Override
    public synchronized void stop()
    {
        if (super.m_state == IComponentState.STATE_OFFLINE)
        {
            return;
        }

        stopDomainStateMonitor();

        super.stop();
    }

    @Override
    public synchronized void enableMetrics(IMetricIdentity[] ids)
    {
        // deal with the metrics that the AgentManager delegates to the task scheduler
        DomainStateMonitor.enableMetrics(ids);
    }

    @Override
    public synchronized void disableMetrics(IMetricIdentity[] ids)
    {
        // deal with the metrics that the AgentManager delegates to the task scheduler
        DomainStateMonitor.disableMetrics(ids);
    }

    @Override
    public void destroy()
    {
        super.m_container.removeGlobalComponentSupport(getGlobalID(), null);

        if (m_isNotFaultTolerant)
        {
            super.m_container.removeGlobalComponentSupport(getGlobalID(), FAULT_TOLERANCE_ROLE_PRIMARY);
        }
        else
        {
            super.m_container.removeGlobalComponentSupport(getGlobalID(), m_faultToleranceRole);
            stopFaultDetector();
            if (m_faultToleranceStateManager.getState(null) == IFaultTolerantState.STATE_ACTIVE)
            {
                releaseLock();
            }
        }

        m_domainState = null;

        if (m_centralizedLogger != null)
        {
            m_centralizedLogger.close();
        }

        super.destroy();
    }

    @Override
    public synchronized void handleElementChange(IElementChange elementChange)
    {
        // we are interested in changes to:
        //  - the list of containers that should be monitored (additions/deletions)
        //  - changes to a containers configuration
        //  - changes to the AM's poll thread min and max counts
        //  - changes to the AM's metrics refresh and collection intervals

        IElementIdentity id = elementChange.getElement().getIdentity();
        String configID = id.getName();

        if (configID.equals(m_primaryConfigID))
        {
            handleAgentManagerElementChange(elementChange);
        }
        else  // container-related changes
        if (configID.equals("/domain/domain"))
        {
            configureCentralizedLogging();
        }
        else  // container-related changes
        if (configID.startsWith("/containers/"))
        {
            handleContainerElementChange(elementChange);
        }
    }

    private void handleAgentManagerElementChange(IElementChange elementChange)
    {
        if (elementChange.getChangeType() == IElementChange.ELEMENT_UPDATED)
        {
            IDeltaElement changeElement = (IDeltaElement)elementChange.getElement();
            IDeltaAttributeSet attrs = (IDeltaAttributeSet)changeElement.getDeltaAttributes();

            String [] mods = attrs.getModifiedAttributesNames();
            handleChangeAgentManagerAttrs(mods, attrs);

            mods = attrs.getNewAttributesNames();
            handleChangeAgentManagerAttrs(mods, attrs);

            mods = attrs.getDeletedAttributesNames();
            handleDeletedAgentManagerAttrs(mods);
        }
    }

    private void handleChangeAgentManagerAttrs(String[] attributeNames, IDeltaAttributeSet modifiedAttrs)
    {
        try
        {
            Integer newMinThreads = null;
            Integer newMaxThreads = null;

            for (int i = 0; i < attributeNames.length; i++)
            {
                if (attributeNames[i].equals(IAgentManagerConstants.MIN_POLL_THREADS_ATTR))
                {
                    newMinThreads = (Integer)modifiedAttrs.getNewValue(IAgentManagerConstants.MIN_POLL_THREADS_ATTR);
                }
                if (attributeNames[i].equals(IAgentManagerConstants.MAX_POLL_THREADS_ATTR))
                {
                    newMaxThreads = (Integer)modifiedAttrs.getNewValue(IAgentManagerConstants.MAX_POLL_THREADS_ATTR);
                }
            }

            // set the new min/max thread count value(s)
            if ((newMaxThreads != null) && (newMinThreads != null))  // both min and max are being changed
            {
                // do it this way so that if new min threads value is larger than
                // old max threads value (or if new max is less than old min),
                // exception will not be thrown...
                m_stateMonitor.setMinThreads(DomainStateMonitor.DEFAULT_MIN_THREADS);
                m_stateMonitor.setMaxThreads(DomainStateMonitor.DEFAULT_MAX_THREADS);
                m_stateMonitor.setMaxThreads(newMaxThreads.shortValue());
                m_stateMonitor.setMinThreads(newMinThreads.shortValue());
            }
            else
            {
                if (newMaxThreads != null)
                {
                    m_stateMonitor.setMaxThreads(newMaxThreads.shortValue());
                }
                if (newMinThreads != null)
                {
                    m_stateMonitor.setMinThreads(newMinThreads.shortValue());
                }
            }
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Error modifying Agent Manager runtime from configuration change, trace follows...", e, Level.WARNING);
        }
    }

    private void handleDeletedAgentManagerAttrs(String[] attributeNames)
    {
        try
        {
            for (int i=0; i< attributeNames.length; i++)
            {
                if (attributeNames[i].equals(IAgentManagerConstants.MIN_POLL_THREADS_ATTR))
                {
                    if (m_stateMonitor != null)
                    {
                        m_stateMonitor.setMinThreads((short)IAgentManagerConstants.MIN_POLL_THREADS_DEFAULT);
                    }
                }
                if (attributeNames[i].equals(IAgentManagerConstants.MAX_POLL_THREADS_ATTR))
                {
                    if (m_stateMonitor != null)
                    {
                        m_stateMonitor.setMaxThreads((short)IAgentManagerConstants.MAX_POLL_THREADS_DEFAULT);
                    }
                }
                if (attributeNames[i].equals(IAgentManagerConstants.REFRESH_INTERVAL_ATTR))
                {
                    if (this.m_metricsRegistrar != null)
                    {
                        this.m_metricsRegistrar.setRefreshInterval((long)IAgentManagerConstants.REFRESH_INTERVAL_DEFAULT * 1000);  //convert to milliseconds
                    }
                }
                if (attributeNames[i].equals(IAgentManagerConstants.COLLECTION_INTERVAL_ATTR))
                {
                    if (this.m_metricsRegistrar != null)
                    {
                        this.m_metricsRegistrar.setCollectionInterval(IAgentManagerConstants.COLLECTION_INTERVAL_DEFAULT * 60000);  //convert to milliseconds
                    }
                }
                if (attributeNames[i].equals(IAgentManagerConstants.REPEAT_ALERT_NOTIFICATIONS_ATTR))
                {
                    if (this.m_metricsRegistrar != null)
                    {
                        this.m_metricsRegistrar.setRepeatMetricAlerts(IAgentManagerConstants.REPEAT_ALERT_NOTIFICATIONS_DEFAULT);
                    }
                }
            }
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Error modifying Agent Manager runtime from configuration change [deletion]", e, Level.SEVERE);
        }
    }

    @Override
    public String getTraceMaskValues()
    {
        return (super.getTraceMaskValues() + "," + MANAGER_TRACE_MASK_VALUES);
    }

    @Override
    public void setTraceMask(Integer traceMask)
    {
        setTraceMask(traceMask, false);
    }

    private void setTraceMask(Integer traceMask, boolean fromSDF)
    {
        // If SDF is used then ignore trace setup from other sources
        // If SDF is not used - update it from other sources
        if (!fromSDF)
        {
            if (m_SdfMFTracingIntegration.wasUpdated())
            {
                return;
            }
            else
            {
                m_SdfMFTracingIntegration.setTraceMask(traceMask);
            }
        }

        super.setTraceMask(traceMask);
        if (m_stateNotificationForwarder != null)
        {
            m_stateNotificationForwarder.setDebug((super.m_traceMask & TRACE_FORWARDED_NOTIFICATIONS) > 0);
        }
    }

    //
    // The management operations we will expose (beyond or overriding that handled by
    // AbstractFrameworkComponent).
    //

    /**
     * Invoke an operation synchronously on the given predefined collection (i.e. those collections known
     * by the DirectoryService).
     *
     * @param configID       The ID of the component collection element (either storage or logical).
     * @param operationName      The name of the operation to invoke on each of the containers/components
     *                           in the collection.
     * @param operationParams    The parameters to the operation.
     * @param operationSignature The signature of the operation.
     * @param sync               If false, aynchronously invoke the operation on the collection and return null,
     *                           otherwise build and return a collective result.
     * @param timeout            The time in milliseconds to wait for all responses to be received.
     *                           The value should be greater than the request timeout value. The value
     *                           is ignored if asynchronous execution is specified.
     *
     * @return If synchronous execution is requested, the return values or exceptions of the individual
     *         requests to member of the collection. Null if asynchronous execution is requested.
     *
     * @exception MFException if there is an undelying communication problem
     *
     * @see com.sonicsw.mf.common.config.IElementIdentity#getName()
     */
    public ICollectiveOpStatus invoke(String configID, String operationName, Object[] operationParams, String[] signature, Boolean sync, Long timeout)
    throws MFException
    {
        super.validateOnline();
        validateActive();

        configID = getStorageIDForCollection(configID);

        final boolean isSync = sync.booleanValue();

        HashMap ids = getCollectionIDs(configID, true);
        Collection canonicalNames = new HashSet(ids.keySet());

        // check permissions for the collection with the Permissions Manager
        // - the result collective status will have an entry for each component on which there was no
        //   permission or some other error was encountered
        ObjectName[] objectNames = new ObjectName[canonicalNames.size()];
        Iterator namesIterator = canonicalNames.iterator();
        int index = 0;
        while (namesIterator.hasNext())
        {
            try
            {
                objectNames[index++] = new ObjectName(((CanonicalName)namesIterator.next()).getCanonicalName());
            }
            catch (MalformedObjectNameException e)
            {
                e.printStackTrace();
            }
        }
        CollectiveOpStatus permissionsManagerResults = m_permissionsManager.managePermissionCheck(objectNames, operationName);

        // work out what components there were no issues with and call the operation on them
        if (permissionsManagerResults != null)
        {
            for (int i = 0; i < permissionsManagerResults.getCount(); i++)
            {
                if (permissionsManagerResults.getThrowable(i) != null)
                {
                    Iterator iterator = canonicalNames.iterator();
                    while (iterator.hasNext())
                    {
                        CanonicalName canonicalName = (CanonicalName)iterator.next();
                        if (canonicalName.getCanonicalName().equals(permissionsManagerResults.getComponentName(i)))
                        {
                            iterator.remove();
                            break;
                        }
                    }
                }
            }
        }

        // short cut if all failed (for one reason or another)
        if (canonicalNames.isEmpty())
        {
            return permissionsManagerResults;
        }

        String[] remainingTargets = new String[canonicalNames.size()];
        Iterator iterator = canonicalNames.iterator();
        for (int i = 0; i < remainingTargets.length; i++)
        {
            remainingTargets[i] = ((CanonicalName)iterator.next()).getCanonicalName();
        }

        try
        {
            ICollectiveOpStatus componentResults = super.m_frameworkContext.invoke(remainingTargets, operationName, operationParams, signature, sync.booleanValue(), timeout.longValue());
            if (permissionsManagerResults == null)
            {
                return isSync ? componentResults : null; // if its async don't return anything
            }
            else
            {
                if (isSync && componentResults != null) // if its async .. don't add actual results
                {
                    for (int i = 0; i < componentResults.getCount(); i++)
                    {
                        permissionsManagerResults.addResult(componentResults.getComponentName(i), componentResults.getReturnValue(i), componentResults.getThrowable(i));
                    }
                }
                return permissionsManagerResults;
            }
        }
        catch(Exception e)
        {
            if (e instanceof MFException)
            {
                throw (MFException)e;
            }
            MFException exception = new MFException("Failed invoke [" + operationName + "] on collection: " + configID);
            exception.setLinkedException(e);
            throw exception;
        }
    }

    private String getStorageIDForCollection(String configID)
    {
        // assumes the given configID could be either a storage or logical name and we want to ensure the caller gets back a storage
        // name
        String logical = m_frameworkContext.storageToLogical(configID);

        // if we got a logical name back, then it is already a storage name
        if (logical != null)
        {
            return configID;
        }

        IElement element = m_frameworkContext.getFSConfiguration(configID, false);
        if (element != null)
        {
            return element.getIdentity().getName();
        }

        // we should at least behave how we used to behave
        return configID;
    }

    /**
     * Sets an attribute synchronously on the given predefined collection (i.e. those collections known
     * by the DirectoryService).
     *
     * @param configID       The ID of the component collection element (either storage or logical).
     * @param attributeName  The name of the attribute
     * @param attributeValue The value of the attribute
     * @param sync           If false, aynchronously invoke the operation on the collection and return null,
     *                       otherwise build and return a collective result.
     * @param timeout        The time in milliseconds to wait for all responses to be received.
     *                       The value should be greater than the request timeout value. The value
     *                       is ignored if asynchronous execution is specified.
     *
     * @return If synchronous execution is requested, the return values or exceptions of the individual
     *         requests to member of the collection. Null if asynchronous execution is requested.
     *
     * @exception MFException if there is an undelying communication problem
     *
     * @see com.sonicsw.mf.common.config.IElementIdentity#getName()
     */
    public ICollectiveOpStatus setAttribute(String configID, String attributeName, Object attributeValue, Boolean sync, Long timeout)
    throws MFException
    {
        super.validateOnline();
        validateActive();

        return internalSetAttributes(false, configID, new String[] { attributeName }, new Object[] { attributeValue }, sync, timeout);
    }

    /**
     * Sets attributes synchronously on the given predefined collection (i.e. those collections known
     * by the DirectoryService).
     *
     * @param configID       The ID of the component collection element (either storage or logical).
     * @param attributeNames  The names of the attributes
     * @param attributeValues The values of the attributes
     * @param sync            If false, aynchronously invoke the operation on the collection and return null,
     *                        otherwise build and return a collective result.
     * @param timeout         The time in milliseconds to wait for all responses to be received.
     *                        The value should be greater than the request timeout value. The value
     *                        is ignored if asynchronous execution is specified.
     *
     * @return If synchronous execution is requested, the return values or exceptions of the individual
     *         requests to member of the collection. Null if asynchronous execution is requested.
     *
     * @exception MFException if there is an undelying communication problem
     *
     * @see com.sonicsw.mf.common.config.IElementIdentity#getName()
     */
    public ICollectiveOpStatus setAttributes(String configID, String[] attributeNames, Object[] attributeValues, Boolean sync, Long timeout)
    throws MFException
    {
        super.validateOnline();
        validateActive();

        return internalSetAttributes(true, configID, attributeNames, attributeValues, sync, timeout);
    }

    /**
     * Get the state of all the containers and their components in the domain.
     *
     * This method is intended for use in getting an initial snapshot of the state of
     * the whole domain. Management applications may call this initially and then subcribe
     * to state notifications generated by the Agent Manager to receive state deltas
     * to this snapshot.
     *
     * @return An array of IContainerState instances that described the aggregated state of the
     *         container and all the components the container is hosting.
     */
    public IState[] getCollectiveState()
    {
        super.validateOnline();
        validateActive();
        return m_domainState.getDomainState();
    }

    /**
     * Get the collective state of all the containers listed in array.
     *
     * @param runtimeIds The list of the containers runtimeID for which the status is being sought.
     *
     * @return An array of IContainerState instances.
     */
    public IState[] getCollectiveState(String [] runtimeID)
    {
        super.validateOnline();
        validateActive();

        ArrayList states = new ArrayList();
        IState[] containerStates = m_domainState.getDomainState();
        for (int i = 0; i < runtimeID.length; i++)
        {
            for (int j = 0; j < containerStates.length; j++)
            {
                if (containerStates[j].getRuntimeIdentity().getCanonicalName().equals(runtimeID[i]))
                {
                    states.add(containerStates[j]);
                    break;
                }
            }
        }
        IState[] runtimeStates = new IState[states.size()];
        return (IState[])states.toArray(runtimeStates);
    }

    /**
     * Get the collective state of all the containers or components in the given collection.
     *
     * This method is intended for use in getting an initial snapshot of the state of
     * a collection.
     *
     * @param configID The configuration ID of the collection for which the status is being sought.
     *
     * @return An array of IContainerState or IComponentState instances that describe the
     *         state of the given collection. If an item in the collection no longer represents
     *         a known runtime identity then no runtime status will be provided in the returned
     *         array.
     */
    public IState[] getCollectiveState(String configID)
    throws MFException
    {
        super.validateOnline();
        validateActive();

        HashMap ids = getCollectionIDs(configID, false);

        IState[] containerStates = m_domainState.getDomainState();
        ArrayList collectiveState = new ArrayList();

        Iterator iterator = ids.entrySet().iterator();
        while (iterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)iterator.next();
            CanonicalName canonicalName = (CanonicalName)entry.getKey();
            String containerID = canonicalName.getDomainName() + '.' + canonicalName.getContainerName();
            IState state = null;
            for (int j = 0; j < containerStates.length; j++)
            {
                // if we get a match on the id then its the whole container status were after
                if (containerStates[j].getRuntimeIdentity().getCanonicalName().equals(canonicalName.getCanonicalName()))
                {
                    state = containerStates[j];
                    break;
                }
                // else well have to see if its the right container
                if (containerStates[j].getRuntimeIdentity().getCanonicalName().equals(containerID))
                {
                    // then loop through the components looking for a match
                    IComponentState[] componentStates = ((IContainerState)containerStates[j]).getComponentStates();
                    for (int k = 0; k < componentStates.length; k++)
                    {
                        if (componentStates[k].getRuntimeIdentity().getCanonicalName().equals(canonicalName.getCanonicalName()))
                        {
                            state = componentStates[k];
                            break;
                        }
                    }
                    break;
                }
            }

            // if we did not find a reported state, create a dummy
            if (state == null)
            {
                IElementIdentity elementID = new ElementIdentity((String)entry.getValue(), null, null);
                if (canonicalName.getComponentName().length() == 0)
                {
                    state = new ContainerState(canonicalName, elementID);
                    ((ContainerState)state).setState(IContainerState.STATE_UNKNOWN);
                }
                else
                {
                    state = new ComponentState(canonicalName, elementID, IComponentState.STATE_UNKNOWN, null, 0, null);
                }
            }

            collectiveState.add(state);
        }

        IState[] runtimeStates = new IState[collectiveState.size()];
        return (IState[])collectiveState.toArray(runtimeStates);
    }

    /**
     * For an active AM, this operation attempts to relinquish the active role to
     * the current standby.  The active AM will wait up the given number of
     * seconds for a standby to takeover the active role.  If a standby does not
     * takeover the active role in this period, the AM will continue its active role.
     * Note: The general use of this operation will be to failback a recovered AM,
     * rather than to force failover to a standby.
     *
     * @param waitSeconds The time in seconds the active AM will wait up to.
     *
     */

    public void suspendActiveRole(final Integer waitSeconds)
    throws MFException
    {
        short currentState = m_faultToleranceStateManager.getState(null);
        if (currentState != IFaultTolerantState.STATE_ACTIVE)
        {
            throw new MFException("Cannot suspend \"" + IFaultTolerantState.STATE_TEXT[IFaultTolerantState.STATE_ACTIVE] +
                                  "\" state when Agent Manager is in a \"" + IFaultTolerantState.STATE_TEXT[currentState] +
                                  "\" state.");
        }

        Runnable suspender = new Runnable()
        {
            @Override
            public void run()
            {
                long waitTime ;
                if (waitSeconds == null || waitSeconds.intValue() == 0)
                {
                    waitTime = m_faultDetectionInterval;
                }
                else
                {
                    waitTime = waitSeconds.intValue() * 1000;
                }

                long startTime = System.currentTimeMillis();

                if (m_faultDetector != null)
                {
                    AgentManager.super.m_context.logMessage("Suspending \"" + IFaultTolerantState.STATE_TEXT[IFaultTolerantState.STATE_ACTIVE] + "\" state for " + waitSeconds + " seconds...", Level.WARNING);
                    m_faultDetector.suspendActiveState();

                    while (true)
                    {
                        try { Thread.sleep(2500); } catch(InterruptedException e) { }

                        // is the container closing in the meantime?
                        if (getContext().getContainer().isClosing())
                        {
                            break;
                        }

                        if (System.currentTimeMillis() - startTime > waitTime)
                        {
                            if (m_faultDetector != null)
                            {
                                AgentManager.super.m_context.logMessage("...suspension complete", Level.INFO);
                                m_faultDetector.resumeAfterSuspend();
                            }
                            break;
                        }
                    }
                }
            }
        };

        super.m_context.scheduleTask(suspender, new Date());
    }

    public String getFaultToleranceRole()
    {
        return this.m_faultToleranceRole;
    }

    public Short getFaultTolerantState()
    {
        return new Short(m_faultToleranceStateManager.getState(null));
    }

    public String getFaultTolerantStateString()
    {
        return IFaultTolerantState.STATE_TEXT[m_faultToleranceStateManager.getState(null)];
    }

    public Boolean getAllowFailover()
    {
        return m_isNotFaultTolerant ? null : new Boolean(m_allowFailover);
    }

    public void setAllowFailover(Boolean allowFailover)
    {
        if (m_isNotFaultTolerant)
        {
            throw new IllegalStateException("Setting of this attribute is not supported for a non-fault tolerant AgentManager");
        }

        boolean allow = allowFailover.booleanValue();
        if (allow != m_allowFailover)
        {
            if (this.m_faultToleranceStateManager.getState(null) != IFaultTolerantState.STATE_ACTIVE)
            {
                super.m_context.logMessage("Failover " + (allow ? "reenabled" : "disabled"), allow ? Level.INFO : Level.WARNING);
            }
            m_allowFailover = allow;
        }
    }

    // settable via AM configuration
    public Integer getMaxPollingThreads()
    {
        // DomainStateMonitor may not yet have been created (e.g. if this AM is serving as the backup, and is in the "waiting" state)
        if (m_stateMonitor != null)
        {
            return new Integer(m_stateMonitor.getMaxThreads());
        }
        else
        {
            // Get the configuration from the configuration element
        	IElement amElement;
        	// if this is a backup AM, the configuration comes from the PRIMARY
            if (m_isBackup)
            {
                amElement = m_context.getConfiguration(m_primaryConfigID, true);  // not interested in updates
            }
            else
            {
                amElement = m_context.getConfiguration(true);
            }
            IAttributeSet amAttributes = amElement.getAttributes();

            Integer maxPollThreads = (Integer) amAttributes.getAttribute(IAgentManagerConstants.MAX_POLL_THREADS_ATTR);
            if (maxPollThreads == null)
            {
                maxPollThreads = new Integer(IAgentManagerConstants.MAX_POLL_THREADS_DEFAULT);
            }
            return maxPollThreads;
        }
    }

    public Integer getMinPollingThreads()
    {
        if (m_stateMonitor != null)
        {
            return new Integer(m_stateMonitor.getMinThreads());
        }
        else  // DomainStateMonitor may not yet have been created (e.g. if this AM is serving as the backup, and is in the "waiting" state)
        {
            // Get the configuration from the configuration element
            IElement amElement;
            //if this is a backup AM, the configuration comes from the PRIMARY
            if (m_isBackup)
            {
                amElement = m_context.getConfiguration(m_primaryConfigID, true);  // not interested in updates
            }
            else
            {
                amElement = m_context.getConfiguration(true);
            }

            IAttributeSet amAttributes = amElement.getAttributes();
            Integer minPollThreads = (Integer) amAttributes.getAttribute(IAgentManagerConstants.MIN_POLL_THREADS_ATTR);
            if (minPollThreads == null)
            {
                minPollThreads = new Integer(IAgentManagerConstants.MIN_POLL_THREADS_DEFAULT);
            }
            return minPollThreads;
        }
    }

    /**
     * Handle internal notifications that are directed at the AgentManager. The AgentManager
     * has to do two things of receipt of such notifications:
     *
     *  - make use of the state changes that may be contained in the notifications to update
     *    the domain state maintained by the Agent Manager
     *  - forward the notifications if it is configured to do so
     *
     * This method is intended for use within the framework and should not be exposed as a
     * management operation (i.e. in the AgentManager's meta-data).
     */
    @Override
    public void handleNotification(INotification notification)
    {
        if (notification.getCategory() == INotification.SYSTEM_CATEGORY)
        {
            handleSystemNotification(notification);
        }
    }

    public String getLogFile()
    {
        return m_centralizedLogger.getLogFile();
    }
    
    public void setLogFile(String path)
    throws Exception
    {
        m_centralizedLogger.setLogFile(path);
    }

    public Long getLogFileSize()
    {
        return new Long(m_centralizedLogger.length());
    }

    public Long getLogFileSizeThreshold()
    {
        return m_centralizedLogger.getLogFileSizeThreshold();
    }

    public void setLogFileSizeThreshold(Long thresholdSize)
    {
        m_centralizedLogger.setLogFileSizeThreshold(thresholdSize);
    }

    public Long getLogFileRolloverSizeThreshold()
    {
        return m_centralizedLogger.getLogFileRolloverSizeThreshold();
    }

    public void setLogFileRolloverSizeThreshold(Long thresholdSize)
    {
        m_centralizedLogger.setLogFileRolloverSizeThreshold(thresholdSize);
    }

    public Integer getLogFileRolloverTimeInterval()
    {
        return m_centralizedLogger.getLogFileRolloverTimeInterval();
    }

    public void setLogFileRolloverTimeInterval(Integer logFileRolloverTimeInterval)
    {
        m_centralizedLogger.setLogFileRolloverTimeInterval(logFileRolloverTimeInterval);
    }

    public synchronized void clearLogFile()
    {
        m_centralizedLogger.clearLogFile();
    }

    public synchronized void saveLogFile(String path)
    {
        m_centralizedLogger.saveLogFile(path);
    }
    
    public void attemptLogFileRollover()
    throws IOException
    {
        m_centralizedLogger.attemptLogFileRollover();
    }

    /**
     * Reads the requested bytes from the log file and returns them as a String.
     * <p>
     * When evaluating log output to be read, the current log file and previous versions (due to
     * rollover activity) will be effectively viewed as a single file.
     *
    * @param fromPosition The starting position from which to read the log for the given date.
     *                     If null, then the total log length minus the given readLength is
     *                     assumed (i.e. the tail of the logging output). The total log length is
     *                     calculated by summing the size of all log files (current + previous
     *                     versions.
     * @param readLength   The maximum amount of bytes to read from the given (or assumed) starting
     *                     position. The max size for this value is 1Mb. If null, 200Kb is assumed.
     *
     * @return A String representation of the actual bytes read (which may be less than the
     *         requested readLength).
     */
    public String getLogExtract(Long fromPosition, Long readLength)
    {
        return m_centralizedLogger.getLogExtract(fromPosition, readLength);
    }

    /**
     * Reads the requested bytes from the log file and returns them as a String encapsulated with
     * the current log size. The current log size is the size of the current log file plus the sum
     * of all available prior versions.
     * <p>
     * When evaluating log output to be read, the current log file and previous versions (due to
     * rollover activity) will be effectively viewed as a single file.
     *
     * @param fromPosition The starting position from which to read the log for the given date.
     *                     If null, then the total log length minus the given readLength is
     *                     assumed (i.e. the tail of the logging output). The total log length is
     *                     calculated by summing the size of all log files (current + previous
     *                     versions.
     * @param readLength   The maximum amount of bytes to read from the given (or assumed) starting
     *                     position. The max size for this value is 1Mb. If null, 200Kb is assumed.
     *
     * @return A javax.management.CompositeData object containing key="LogExtract",
     *         value=<log extract String> and key="LogFileSize", value=<total log size Long>.
     *         The log extract is a String representation of the actual bytes read (which may be
     *         less than the requested readLength).
     */
    public CompositeData getLogExtractAndLogFileSize(Long fromPosition, Long readLength)
    {
        return m_centralizedLogger.getLogExtractAndLogFileSize(fromPosition, readLength);
    }

    // centrally log the given message
    public synchronized void logMessage(String message)
    {
        m_centralizedLogger.logMessage(message);
    }

    public void recordAuditEvent(String source, String logger, String message)
    {
        // don't record it if its from the same container as the AM, because it will get recorded anyways!

        if (new CanonicalName(source).getContainerName().equals(m_context.getComponentName().getContainerName()))
        {
            return;
        }

        super.m_frameworkContext.getAuditManager().recordEvent(logger, message);
    }


    //
    // Internal/package methods
    //

    IContainerState getContainerState(String container, long timeout)
    throws Exception
    {
        if (DEBUG)
        {
            System.out.println("AgentManager.getContainerState: " + container);
        }

        int i = 0;
        while (true) // allow one retry if we get some conflicting info
        {
            long checkTime = System.currentTimeMillis();
            try
            {
                return (IContainerState)super.m_frameworkContext.invoke(container + ":ID=AGENT", "getContainerState", IEmptyArray.EMPTY_OBJECT_ARRAY, IEmptyArray.EMPTY_STRING_ARRAY, true, timeout);
            }
            catch(Exception e)
            {
                if (i < 1 && e instanceof InvokeTimeoutException)
                {
                    // just check whether some conflicting info was recorded in the meantime .. which would
                    // cause us to want to double check the data (but just do this once so we don't end up
                    // in this loop forever)
                    i++;
                    ContainerState containerState = m_domainState.getContainerState(container);
                    if (containerState.getTimeStamp() > checkTime)
                    {
                        continue;
                    }
                }
                sendContainerUnreachableNotification(container, checkTime);
                throw e;
            }
        }
    }

    void sendFailoverNotification()
    {
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY], FAILOVER_NOTIFICATION_TYPE, Level.WARNING);
        notification.setLogType(INotification.WARNING_TYPE);
        notification.setAttribute("FaultToleranceRole", m_faultToleranceRole);
        super.m_context.sendNotification(notification);
    }

    private MFRuntimeException createMFRuntimeException(Exception e)
    {
        return new MFRuntimeException(e.toString());
    }

    private void handleSystemNotification(INotification notification)
    {
        // if this is a state change then we can reflect that in the domain state
        if (notification.getSubCategory().equals(INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY]))
        {
            // update the domain state
            handleSystemStateNotification(notification);

            // is this notification from the AM itself? - if so don't handle
            if (!notification.getSourceIdentity().getCanonicalName().equals(super.m_frameworkContext.getComponentName().getCanonicalName()))
            {
                // forward the notification if required (i.e. its not this component that sent the notification and its not
                // a container startup notification
                if (!notification.getEventName().equals(STARTUP_NOTIFICATION_TYPE))
                {
                    m_stateNotificationForwarder.handleNotification(notification);
                }
            }
        }
    }

    private void handleSystemStateNotification(INotification notification)
    {
        String event = notification.getEventName();
        String type = notification.getType();
        long timestamp = notification.getTimeStamp();
        IComponentIdentity source = notification.getSourceIdentity();
        String sourceContainer = source.getDomainName() + '.' + source.getContainerName();

        if (DEBUG)
        {
            System.out.println("AgentManager.handleSystemStateNotification: " + source.getCanonicalName() + " -> " + event);
        }

        if (event.equals(STARTUP_NOTIFICATION_TYPE))
        {
            // is this startup notification from the AM itself? - if so don't handle
            if (source.getCanonicalName().equals(super.m_frameworkContext.getComponentName().getCanonicalName()))
            {
                return;
            }

            if (m_domainState.updateContainerState(sourceContainer, IContainerState.STATE_ONLINE, timestamp))
            {
                refreshContainerState(sourceContainer);
            }

            sendStartupNotification(sourceContainer);
            return;
        }

        if (type.equals(IAgentProxy.SYSTEM_STATE_SHUTDOWN_NOTIFICATION_ID))
        {
            m_domainState.updateContainerState(sourceContainer, IContainerState.STATE_OFFLINE, timestamp);
            return;
        }

        if (type.equals(IAgentProxy.SYSTEM_STATE_LOAD_NOTIFICATION_ID))
        {
            // ensure the container is online
            m_domainState.updateContainerState(sourceContainer, IContainerState.STATE_ONLINE, timestamp);
            // in this case always refresh the container state
            refreshContainerState(sourceContainer);
            return;
        }

        if (type.equals(IAgentProxy.SYSTEM_STATE_UNLOAD_NOTIFICATION_ID))
        {
            String componentID = (String)notification.getAttributes().get("ID");
            m_domainState.removeComponentState(sourceContainer, componentID, timestamp);
            return;
        }

        if (event.equals(IComponentState.STATE_TEXT[IComponentState.STATE_ONLINE]))
        {
            String lastError = (String)notification.getAttributes().get("LastError");
            Integer lastErrorLevel = new Integer((String)notification.getAttributes().get("LastErrorLevel"));
            if (m_domainState.updateComponentState(sourceContainer, source.getComponentName(), IComponentState.STATE_ONLINE, lastError, lastErrorLevel, timestamp))
            {
                refreshContainerState(sourceContainer);
            }
            return;
        }

        if (event.equals(IComponentState.STATE_TEXT[IComponentState.STATE_OFFLINE]))
        {
            String lastError = (String)notification.getAttributes().get("LastError");
            Integer lastErrorLevel = new Integer((String)notification.getAttributes().get("LastErrorLevel"));
            if (m_domainState.updateComponentState(sourceContainer, source.getComponentName(), IComponentState.STATE_OFFLINE, lastError, lastErrorLevel, timestamp))
            {
                refreshContainerState(sourceContainer);
            }
            return;
        }

        if (type.equals(IAgentProxy.SYSTEM_STATE_CONTAINERSTATE_NOTIFICATION_ID))
        {
            IContainerState containerState = (IContainerState)notification.getAttributes().get("ContainerState");
            this.m_stateMonitor.handleContainerStateNotification(containerState, timestamp);
            return;
        }

        if (type.equals(IActivationDaemonProxy.SYSTEM_STATE_FAILURE_NOTIFICATION_ID))  // handle failure reported from Activation Daemon
        {
            HashMap attributes = notification.getAttributes();
            String childContainer = (String) attributes.get("Container");

            if (childContainer != null)
            {
                m_domainState.updateContainerState(childContainer, IContainerState.STATE_OFFLINE, timestamp);
            }
            return;
        }
    }

    private void refreshContainerState(final String container)
    {
        // state monitor will not get initialized for a standby
        // Agent Manager, therefore, it does not need to refresh
        // the container state.
        if (m_stateMonitor != null)
        {
            m_stateMonitor.schedule(container);
        }
    }

    private void handleContainerElementChange(IElementChange elementChange)
    {
        if (elementChange.getElement() instanceof IElement)
        {
            IElement configuration = (IElement)elementChange.getElement();

            switch(elementChange.getChangeType())
            {
                case IElementChange.ELEMENT_ADDED:
                {
                    if (((IDirElement)configuration).isTemplate())
                    {
                        return;
                    }
                    // add the container to the domain state
                    m_domainState.addContainer(configuration);
                    // get the containers current state
                    IContainerIdentity containerID = new ContainerIdentity(configuration);
                    CanonicalName canonicalName = new CanonicalName(containerID.getDomainName(), containerID.getContainerName(), "");
                    refreshContainerState(canonicalName.getCanonicalName());
                    break;
                }
                case IElementChange.ELEMENT_DELETED:
                {
                    // remove the container from the domain state
                    m_domainState.removeContainer(configuration.getIdentity().getName());
                    break;
                }
                default:
                    return;
            }

            // updates the polling schedule to reflect added/deleted container configuration
            if (m_stateMonitor != null)
            {
                m_stateMonitor.reschedule();
            }
            return;
        }

        if (elementChange.getElement() instanceof IDeltaElement)
        {
            IDeltaElement delta = (IDeltaElement)elementChange.getElement();
            String configID = delta.getIdentity().getName();
            IElement configuration = super.m_context.getConfiguration(configID, true);

            switch(elementChange.getChangeType())
            {
                case IElementChange.ELEMENT_UPDATED:
                {
                    IDeltaAttributeSet attrs = (IDeltaAttributeSet)delta.getDeltaAttributes();

                    String [] mods = attrs.getModifiedAttributesNames();
                    handleChangeContainerAttrs(mods, attrs, configuration);

                    mods = attrs.getNewAttributesNames();
                    handleChangeContainerAttrs(mods, attrs, configuration);

                    mods = attrs.getDeletedAttributesNames();
                    // handleDeletedContainerAttrs(mods); // not required currently

                    return;
                }
                default:
                    return;
            }
        }
    }

    private void handleChangeContainerAttrs(String[] mods, IDeltaAttributeSet attrs, IElement configuration)
    {
        for (int i = 0; i < mods.length; i++)
        {
            if (mods[i].equals(IContainerConstants.MONITORING_ATTR))
            {
                if (m_stateMonitor != null)
                {
                    IContainerIdentity containerID = new ContainerIdentity(configuration);
                    CanonicalName canonicalName = new CanonicalName(containerID.getDomainName(), containerID.getContainerName(), "");
                    m_stateMonitor.reschedule(canonicalName.getCanonicalName());
                }
            }
            else
            if (mods[i].equals(IContainerConstants.CONTAINER_NAME_ATTR))
            {
                // remove the container from the domain state
                m_domainState.removeContainer(configuration.getIdentity().getName());
                // add the container to the domain state
                m_domainState.addContainer(configuration);
                // then reschedule to get the new one in the poll cycle
                if (m_stateMonitor != null)
                {
                    m_stateMonitor.reschedule();
                }
            }
        }
    }

    private void sendStartupNotification(String container)
    {
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY], STARTUP_NOTIFICATION_TYPE, Level.INFO);
        notification.setLogType(INotification.INFORMATION_TYPE);
        notification.setAttribute("Container", container);
        super.m_context.sendNotification(notification);
    }

    private void sendContainerUnreachableNotification(String container, long checkTime)
    {
        if ((super.m_traceMask & TRACE_CONTAINER_UNREACHABLE) > 0)
        {
            super.m_context.logMessage("Container unreachable: " + container, Level.TRACE);
        }

        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.STATE_SUBCATEGORY], UNREACHABLE_NOTIFICATION_TYPE, Level.SEVERE);
        notification.setLogType(INotification.WARNING_TYPE);
        notification.setAttribute("Container", container);
        notification.setAttribute("CheckTime", new Date(checkTime));

        super.m_context.sendNotification(notification);
    }

    /**
     * Get the identities of all the containers or components in the given collection.
     *
     * @param configID       The configuration ID of the collection.
     * @param appendAgentIDs Applies only to container collections - if true converts the container ID
     *                       to the AGENT component within the container.
     */
    private HashMap getCollectionIDs(String configID, boolean appendAgentIDs)
    throws MFException
    {
        if (configID == null || configID.length() == 0)
        {
            throw new MFRuntimeException("Bad config identity: " + configID);
        }

        IElement collectionConfig = super.m_context.getConfiguration(configID, true);
        if (collectionConfig == null)
        {
            throw new MFException("Unknown collection: " + configID);
        }

        boolean isContainerCollection = isContainerCollection(collectionConfig);
        if (!isContainerCollection)
        {
            appendAgentIDs = false;
        }

        IAttributeSet attributes = collectionConfig.getAttributes();
        IAttributeSet idSet = (IAttributeSet)attributes.getAttribute(isContainerCollection ? IContainerCollectionConstants.CONTAINERS_ATTR : IComponentCollectionConstants.COMPONENTS_ATTR);

        HashMap ids = new HashMap();
        Iterator iterator = idSet.getAttributes().values().iterator();
        while (iterator.hasNext())
        {
            IAttributeSet attrs = (IAttributeSet)iterator.next();
            String runtimeID = (String)attrs.getAttribute(isContainerCollection ? IContainerCollectionConstants.CONTAINER_RUNTIME_ID_ATTR : IComponentCollectionConstants.COMPONENT_RUNTIME_ID_ATTR);
            CanonicalName canonicalName = null;
            if (isContainerCollection && appendAgentIDs)
            {
                canonicalName = new CanonicalName(runtimeID + IComponentIdentity.DELIMITED_ID_PREFIX + IAgentProxy.ID);
            }
            else
            {
                canonicalName = new CanonicalName(runtimeID);
            }
            Reference configRef = (Reference)attrs.getAttribute(isContainerCollection ? IContainerCollectionConstants.CONFIG_REF_ATTR : IComponentCollectionConstants.CONFIG_REF_ATTR);
            ids.put(canonicalName, configRef.getElementName());
        }

        return ids;
    }

    static boolean isContainerCollection(IElement collectionConfig)
    {
        return collectionConfig.getIdentity().getType().equals("MF_CONTAINER_COLLECTION");
    }

    static boolean isComponentCollection(IElement collectionConfig)
    {
        return collectionConfig.getIdentity().getType().equals("MF_COMPONENT_COLLECTION");
    }


    /**
     * This method allows setAttribut() and setAttributes() to use common code.
     *
     * @param reportAttributes If true, the attributes that were successfully set are repoerted in the
     *                         collective results (ala. setAttributes()). If false, a null is reported
     *                         (represeting the void return type of setAttribute()).
     */
    private ICollectiveOpStatus internalSetAttributes(final boolean reportAttributes, String configID, final String[] attributeNames, final Object[] attributeValues, Boolean sync, Long timeout)
    throws MFException
    {
        long startTime = System.currentTimeMillis();
        configID = getStorageIDForCollection(configID);

        final HashMap ids = getCollectionIDs(configID, true);
        Collection canonicalNames = new HashSet(ids.keySet());

        final boolean isSync = sync.booleanValue();
        final Hashtable results = isSync ? new Hashtable(ids.size()) : null;

        IPermissionsManager permissionsManager = super.m_frameworkContext.getPermissionsManager();

        // check permissions for the collection with the Permissions Manager
        // - the result collective status will have an entry for each component on which there was no
        //   permission or some other error was encountered
        ObjectName[] objectNames = new ObjectName[canonicalNames.size()];
        Iterator namesIterator = canonicalNames.iterator();
        int index = 0;
        while (namesIterator.hasNext())
        {
            try
            {
                objectNames[index++] = new ObjectName(((CanonicalName)namesIterator.next()).getCanonicalName());
            }
            catch (MalformedObjectNameException e)
            {
                e.printStackTrace();
            }
        }
        CollectiveOpStatus permissionsManagerResults = permissionsManager.managePermissionCheck(objectNames, "setAttributes");

        // work out what components there were no issues with and call the setAttributes() on them
        if (permissionsManagerResults != null)
        {
            for (int i = 0; i < permissionsManagerResults.getCount(); i++)
            {
                if (permissionsManagerResults.getThrowable(i) != null)
                {
                    Iterator iterator = canonicalNames.iterator();
                    while (iterator.hasNext())
                    {
                        CanonicalName canonicalName = (CanonicalName)iterator.next();
                        if (canonicalName.getCanonicalName().equals(permissionsManagerResults.getComponentName(i)))
                        {
                            iterator.remove();
                            break;
                        }
                    }
                }
            }
        }

        Iterator iterator = canonicalNames.iterator();
        while (iterator.hasNext())
        {
            final String canonicalName = ((CanonicalName)iterator.next()).getCanonicalName();
            Runnable invoker = new Runnable()
            {
                @Override
                public void run()
                {
                    Object rtnValue = null;
                    Exception ex = null;
                    try
                    {
                        rtnValue = AgentManager.super.m_frameworkContext.setAttributes(canonicalName, attributeNames, attributeValues, isSync);
                        rtnValue = reportAttributes ? rtnValue : null;
                    }
                    catch (Exception e)
                    {
                        ex = e;
                    }
                    synchronized (results)
                    {
                        
                        if (ex != null)
                        {
                            results.put(canonicalName, new Object[]{ null, ex });
                        }
                        else if (isSync) 
                        {
                            results.put(canonicalName, new Object[]{ rtnValue, null });
                        }
                        if (isSync && results.size() == ids.size())
                        {
                            results.notifyAll();
                        }
                    }
                }
            };
            m_taskScheduler.scheduleTask(invoker, false);
        }

        if (!isSync)
        {
            return permissionsManagerResults;
        }

        synchronized (results)
        {
            try
            {   
                if (timeout <= 0)
                {
                    while (results.size() < ids.size())
                    {
                        results.wait();
                    }
                }
                else
                {
                    long waitTime = timeout - (System.currentTimeMillis() - startTime);
                    while (waitTime > 0 && results.size() < ids.size())
                    {
                        results.wait(waitTime);
                        waitTime = timeout - (System.currentTimeMillis() - startTime);
                    }   
                }
            }
            catch (InterruptedException ie)
            {
            }
            // loop through the results and if there are any missing then add them with a timeout exception
            iterator = ids.keySet().iterator();
            while (iterator.hasNext())
            {
                String canonicalName = ((CanonicalName)iterator.next()).getCanonicalName();
                Object result = results.get(canonicalName);
                if (result == null)
                {
                    results.put(canonicalName, new Object[]
                    { null, new InvokeTimeoutException("Request timeout.") });
                }
            }
        }
        // now build a collective results set (or add to the existing one if there were permissions exceptions)
        Object[] entries = results.entrySet().toArray();
        CollectiveOpStatus collectiveOpStatus = permissionsManagerResults == null ? new CollectiveOpStatus(entries.length) : permissionsManagerResults;
        for (int i = 0; i < entries.length; i++)
        {
            Map.Entry entry = (Map.Entry)entries[i];
            Object[] result = (Object[])entry.getValue();

            collectiveOpStatus.addResult((String)entry.getKey(), result[0], (Throwable)result[1]);
        }
        return collectiveOpStatus;
    }

    private void readConfiguration(IComponentContext context)
    {
        Object tmp = null;

        // Get the configuration from the configuration element
        IElement amElement = context.getConfiguration(true);
        m_configID = amElement.getIdentity().getName();
        IAttributeSet amAttributes = amElement.getAttributes();

        IAttributeSet references = (IAttributeSet)amAttributes.getAttribute(CONFIG_ELEMENT_REFERENCES_ATTR);
        // if there's a backup, this is a PRIMARY config
        if (references != null)
    	{
    		Reference backupRef = (Reference)references.getAttribute(BACKUP_CONFIG_ELEMENT_REF_ATTR);
    		if (backupRef != null)
            {
                m_faultToleranceRole = FAULT_TOLERANCE_ROLE_PRIMARY;
            }
    	}

        if (amElement.getIdentity().getType().equals(BACKUP_AGENT_MANAGER_TYPE))
        {
        	m_faultToleranceRole = FAULT_TOLERANCE_ROLE_BACKUP;
        	IAttributeSet backupRefs = (IAttributeSet)amAttributes.getAttribute(CONFIG_ELEMENT_REFERENCES_ATTR);
        	if (backupRefs == null)
            {
                throw new MFRuntimeException("Backup Agent Manager configuration " + m_configID + " does not reference a primary configuration");
            }
            else
        	{
        		Reference primaryRef = (Reference)backupRefs.getAttribute(PRIMARY_CONFIG_ELEMENT_REF_ATTR);
        		if (primaryRef == null)
                {
                    throw new MFRuntimeException("Backup Agent Manager configuration " + m_configID + " does not reference a primary configuration");
                }
                else
        		{
        			m_primaryConfigID = primaryRef.getElementName();
        			IElement primaryEl = context.getConfiguration(m_primaryConfigID, true);
        			amAttributes = primaryEl.getAttributes();
        		}
        	}
        }
        else
        {
            m_primaryConfigID = m_configID;
        }
        
        m_isNotFaultTolerant = m_faultToleranceRole.equals(FAULT_TOLERANCE_ROLE_DEFAULT);
        m_isPrimary = m_faultToleranceRole.equals(FAULT_TOLERANCE_ROLE_PRIMARY);
        m_isBackup = m_faultToleranceRole.equals(FAULT_TOLERANCE_ROLE_BACKUP);
        tmp = amAttributes.getAttribute(IAgentManagerConstants.FAULT_DETECTION_INTERVAL_ATTR);
        m_faultDetectionInterval = (tmp == null ? IAgentManagerConstants.FAULT_DETECTION_INTERVAL_DEFAULT : ((Integer)tmp).intValue()) * 1000;

        m_centralizedLogger.configure();
    }

    private synchronized void configureCentralizedLogging()
    {
        m_centralizedLogger.configure();
    }

    private void startDomainStateMonitor()
    {
        m_stateMonitor = new DomainStateMonitor(this, m_domainState, super.m_context);
        m_stateMonitor.setDaemon(true);
        // Get the configuration from the configuration element
        try
        {
        	IElement amElement;
            //if this is a backup AM, the configuration comes from the PRIMARY
            if (m_isBackup)
            {
                amElement = m_context.getConfiguration(m_primaryConfigID, true);  // not interested in updates
            }
            else
            {
                amElement = m_context.getConfiguration(true);
            }
            IAttributeSet amAttributes = amElement.getAttributes();
            if (amAttributes != null)
            {
                Integer minThreads = (Integer) amAttributes.getAttribute(IAgentManagerConstants.MIN_POLL_THREADS_ATTR);
                Integer maxThreads = (Integer) amAttributes.getAttribute(IAgentManagerConstants.MAX_POLL_THREADS_ATTR);
                if (maxThreads != null)
                {
                    m_stateMonitor.setMaxThreads(maxThreads.shortValue());
                }
                if (minThreads != null)
                {
                    m_stateMonitor.setMinThreads(minThreads.shortValue());
                }
            }
        }
        catch (Exception e) {}  // defaults will be used

        // start the monitor thread
        m_stateMonitor.start();
    }

    private void stopDomainStateMonitor()
    {
        if (m_stateMonitor != null)
        {
            m_stateMonitor.cleanup();
            m_stateMonitor = null;
        }
    }

    private void startFaultDetector()
    {
        m_faultDetector = new FaultDetector(this);
        m_faultDetector.setTestInterval(m_faultDetectionInterval);
        m_faultDetector.start();
    }

    private void stopFaultDetector()
    {
        if (m_faultDetector != null)
        {
            m_faultDetector.close();
            m_faultDetector = null;
        }
    }

    IStateManager getFaultToleranceStateManager() { return m_faultToleranceStateManager; }

    IFrameworkComponentContext getContext() { return super.m_frameworkContext; }

    // Waiting to Active, Standby to Active
    public class ToActiveStateController
    implements IStateController
    {
        @Override
        public boolean changeState()
        throws NonRecoverableStateChangeException, RecoverableStateChangeException
        {
            AgentManager.this.m_abortFailover = false;
            String msgPrefix = '[' + m_faultToleranceRole + "] ";

            try
            {
                AgentManager.super.m_container.addGlobalComponentSupport(getGlobalID(), null, AgentManager.this);
            }
            catch (Exception e)
            {
                AgentManager.super.m_container.removeGlobalComponentSupport(getGlobalID(), null);

                m_context.logMessage(msgPrefix + "Failed communications initialization, trace follows...", e, Level.SEVERE);
                AgentManager.super.m_container.shutdown(IContainerExitCodes.COMMS_FAILURE_EXIT_CODE);
            }

            if (AgentManager.this.m_abortFailover)
            {
                AgentManager.super.m_container.removeGlobalComponentSupport(getGlobalID(), null);

                String message = msgPrefix + "Failover aborted: an active Agent Manager already exists";
                m_context.logMessage(message, Level.SEVERE);
                throw new RecoverableStateChangeException(message);
            }

            startDomainStateMonitor();
            return true;
        }
    }

    // Active to Waiting, Active to Standby
    public class FromActiveStateController
    implements IStateController
    {
        @Override
        public boolean changeState()
        {
            AgentManager.this.releaseLock();
            AgentManager.super.m_container.removeGlobalComponentSupport(getGlobalID(), null);
            stopDomainStateMonitor();
            return true;
        }
    }

    // Waiting to Standby, Standby to Waiting
    public class WaitingStandbyStateController
    implements IStateController
    {
        @Override
        public boolean changeState()
        {
            return true;
        }
    }

    public class StateListener
    implements IStateListener
    {
        @Override
        public void stateChanging(short currentState, short intendedState)
        {
            //AgentManager.super.m_context.logMessage("Changing from " + IFaultTolerantState.STATE_TEXT[currentState] + " to " + IFaultTolerantState.STATE_TEXT[intendedState], Level.INFO);
        }

        @Override
        public void stateChanged(short previousState, short currentState)
        {
            switch (previousState)
            {
                case IFaultTolerantState.STATE_WAITING:
                {
                    switch (currentState)
                    {
                        case IFaultTolerantState.STATE_ACTIVE:
                        case IFaultTolerantState.STATE_STANDBY:
                        {
                            AgentManager.super.m_context.logMessage("Transition to state: \"" + IFaultTolerantState.STATE_TEXT[currentState] + '"', Level.INFO);
                            break;
                        }
                    }
                    break;
                }
                case IFaultTolerantState.STATE_STANDBY:
                {
                    switch (currentState)
                    {
                        case IFaultTolerantState.STATE_ACTIVE:
                        {
                            AgentManager.super.m_context.logMessage("Failover to state: \"" + IFaultTolerantState.STATE_TEXT[currentState] + '"', Level.WARNING);
                            AgentManager.this.sendFailoverNotification();
                            break;
                        }
                        case IFaultTolerantState.STATE_WAITING:
                        {
                            AgentManager.super.m_context.logMessage("Reverting to state: \"" + IFaultTolerantState.STATE_TEXT[currentState] + '"', Level.WARNING);
                            break;
                        }
                    }
                    break;
                }
                case IFaultTolerantState.STATE_ACTIVE:
                {
                    switch (currentState)
                    {
                        case IFaultTolerantState.STATE_WAITING:
                        {
                            AgentManager.super.m_context.logMessage("Transition to state: \"" + IFaultTolerantState.STATE_TEXT[currentState] + '"', Level.INFO);
                            break;
                        }
                        case IFaultTolerantState.STATE_STANDBY:
                        {
                            AgentManager.super.m_context.logMessage("Reverting to state: \"" + IFaultTolerantState.STATE_TEXT[currentState] + '"', Level.INFO);
                            break;
                        }
                    }
                    break;
                }

            }
            //AgentManager.super.m_context.logMessage("Changed from " + IFaultTolerantState.STATE_TEXT[previousState] + " to " + IFaultTolerantState.STATE_TEXT[currentState], Level.INFO);
        }

        @Override
        public void stateChangeFailed(short currentState, short intendedState)
        {
            AgentManager.super.m_context.logMessage("Failed to change from " + IFaultTolerantState.STATE_TEXT[currentState] + " to " + IFaultTolerantState.STATE_TEXT[intendedState], Level.INFO);
        }
    }


    private class SdfMFTracingIntegration extends com.sonicsw.sdf.AbstractMFComponentTracing
    {
         private boolean m_updateTraceLevelWasCalled;

         SdfMFTracingIntegration()
         {
             super("sonic.mf.am", getTraceMaskValues());
             m_updateTraceLevelWasCalled = false;

             setTraceMask();
         }
         
         private void setTraceMask() {
             setTraceMask(new Integer(AgentManager.this.m_traceMask));
         }

         boolean wasUpdated()
         {
             return m_updateTraceLevelWasCalled;
         }

         @Override
        public void updateTraceLevel(String doiIDNotUsed, HashMap parameters, StringBuffer buffer)
         {
             super.updateTraceLevel(doiIDNotUsed, parameters, buffer);
             m_updateTraceLevelWasCalled = true;
             AgentManager.this.setTraceMask(getCurrentMask(), true);
         }
    }

    private void validateActive()
    {
        if (!m_isNotFaultTolerant && m_faultToleranceStateManager.getState(null) != IFaultTolerantState.STATE_ACTIVE)
        {
            throw new MFServiceNotActiveException('[' + m_faultToleranceRole + "] Agent Manager not active");
        }
    }

    //
    // Used by FaultDetector to acquire the fault tolerance[AM] lock.
    // The lock can be considered acquired for a return value of true; failure to
    // acquire will be signified by a return value of false.
    //
    // "leaseDuration" parameter for leaseLock() operation is calculated by adding following values:
    // 1. FALUT_DETECTION_INTERVAL (ms)
    // 2. a latency time of 5000 ms
    // 3. the connector connection timeout (ms)
    // 4. the adjusted request timeout which is the larger of (25% of the connector request timeout) or 10000ms
    // If the container is configured with a secondary connection URL, then leaseDuration is adjusted further by adding
    // (3 and 4).
    boolean leaseLock() throws Exception
    {
        long connectTimout = m_connectorServer.getConnectTimeout();
        long leaseRequestTimeout = getLeaseRequestTimeout();
        long leaseDuration = m_faultDetectionInterval   // FALUT_DETECTION_INTERVAL
                             + LATENCY_LEASE_LOCK       // latency time for this op
                             + connectTimout            // connector connection timeout
                             + leaseRequestTimeout;     // adjusted request timeout for this op

        return ((Boolean)getContext().invoke(getDSTarget(), "leaseLock",
                                    new Object[]
                                        {
                                        getGlobalID(),
                                        getVMID(),
                                        new Integer((int)(leaseDuration / 1000)),
                                    },
                                    LEASELOCK_SIGNATURE, true, leaseRequestTimeout)).booleanValue()  ;
    }

    long getLeaseRequestTimeout()
    {
        int requestTimeout = ((int)(m_connectorServer.getRequestTimeout())) >> 2;
        long leaseRequestTimeout = (requestTimeout > LEASE_REQUEST_TIMEOUT_MIN ? requestTimeout : LEASE_REQUEST_TIMEOUT_MIN);

        return leaseRequestTimeout;
    }

    //
    // Used by fault tolerant AM to release an acquired lock prior to lease expiration
    //
    void releaseLock()
    {
        Runnable lockReleaser = new Runnable()
        {
            @Override
            public void run()
            {
                try
                {
                    final String[] SIGNATURE =  new String[]
                    {
                        String.class.getName(),
                        String.class.getName()
                    };

                    getContext().invoke(getDSTarget(), "releaseLock",
                                        new Object[]
                                        {
                                            getGlobalID(),
                                            getVMID(),
                                        },
                                        SIGNATURE, true, 0);
                }
                catch(Exception e) { } // eat this as the lock will expire anyway
            }
        };
        super.m_context.scheduleTask(lockReleaser, new Date());
    }

    // returns a Target ID String for invoking operation on DS
    private String getDSTarget()
    {
        StringBuffer target = new StringBuffer();
        target.append(getContext().getComponentName().getDomainName());
        target.append('.');
        target.append(IDirectoryServiceProxy.GLOBAL_ID);
        target.append(':').append(IComponentIdentity.ID_PREFIX);
        target.append(IDirectoryServiceProxy.GLOBAL_ID);

        return target.toString();
    }

    private String getVMID() {return m_vmID;}

    @Override
    public void globalComponentAlreadyExists(String name)
    {
        if (m_isNotFaultTolerant)
        {
            m_context.logMessage("A non-fault tolerant or \"PRIMARY\" Agent Manager appears to be running elsewhere; it must be shutdown prior to starting this container hosting the \"PRIMARY\" Agent Manager", Level.SEVERE);
            abortContainer();
        }
        else
        {
            if (name.startsWith('[' + m_faultToleranceRole + ']'))
            {
                m_context.logMessage("A \"" + m_faultToleranceRole + "\" Agent Manager appears to be running elsewhere; it must be shutdown prior to starting this container hosting the \"" + m_faultToleranceRole + "\" Agent Manager", Level.SEVERE);
                abortContainer();
            }
            else
            {
                // the only other time is when we get a FT dual active scenario; in that case we need to see if were
                // in the middle of a failover or a reconnection after a partition has occured in order to decide
                // how to proceed
                if (m_faultToleranceStateManager.getState(null) == IFaultTolerantState.STATE_STANDBY)
                {
                    // implies failover in process
                    m_abortFailover = true;
                }
                else
                {
                    m_context.logMessage("An \"Active\" Agent Manager appears to be running elsewhere", Level.WARNING);
                    try
                    {
                        m_faultToleranceStateManager.requestStateChange(IFaultTolerantState.STATE_ACTIVE, IFaultTolerantState.STATE_STANDBY, null);
                    }
                    catch (MFException ex)
                    {
                        m_context.logMessage("Failed to revert to \"Standby\"; shutting down this container for manual resolution", Level.SEVERE);
                        abortContainer();
                    }
                }
            }
        }
    }

    private void abortContainer()
    {
        m_context.logMessage("Aborting container", Level.SEVERE);
        m_frameworkContext.getContainer().shutdown(IContainerExitCodes.AM_ALREADY_RUNNING_EXIT_CODE);
    }
}
