package com.sonicsw.mf.framework.manager;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;

import javax.management.openmbean.CompositeData;
import javax.management.openmbean.CompositeDataSupport;
import javax.management.openmbean.CompositeType;
import javax.management.openmbean.OpenDataException;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.util.RollingFileLogger;
import com.sonicsw.mf.mgmtapi.config.constants.IDomainConstants;

class CentralizedLogger
{
    private IComponentContext m_context;
    private String m_path;
    private String m_defaultPath;
    
    private RollingFileLogger m_centralFileLogger;
    private LogWriter m_writer;
    
    private long m_logFileThreshold = IDomainConstants.LOG_FILE_SIZE_THRESHOLD_DEFAULT;
    private long m_logFileRolloverThreshold = IDomainConstants.LOG_FILE_ROLLOVER_SIZE_THRESHOLD_DEFAULT;
    private int m_logFileRolloverTimeInterval = IDomainConstants.LOG_FILE_ROLLOVER_TIME_INTERVAL_DEFAULT;

    private ArrayList m_messageQueue = new ArrayList();
    private Object m_messageQueueSyncObj = new Object();
    
    private static final String FILELOG_SUFFIX = ".log";
    private static int MAX_BUFFERED_LOG_MESSAGES = Integer.parseInt(System.getProperty("sonicsw.mf.manager.maxBufferedLogMessages", "1000"));
    private static int DROP_BUFFERED_LOG_MESSAGES = Integer.parseInt(System.getProperty("sonicsw.mf.manager.dropBufferedLogMessages", "500"));
    private static CompositeType LOG_COMPOSITE_TYPE;
    private static String[] LOG_COMPOSITE_TYPE_ITEM_NAMES;

    private final SimpleDateFormat m_dateTimeFormatter = new SimpleDateFormat("yy/MM/dd HH:mm:ss");

    CentralizedLogger(IComponentContext context)
    {
        m_context = context;
        
        m_defaultPath = context.getComponentName().getDomainName() + FILELOG_SUFFIX;
        m_path = m_defaultPath;
        
        configure();
    }
    
    void logMessage(String message)
    {
        open();
        
        synchronized (m_messageQueueSyncObj)
        {
            if (m_messageQueue.size() >= MAX_BUFFERED_LOG_MESSAGES)
            {
                // do something dramatic because we just can't keep up .. halve the queue, but at least let them know!
                Object[] messages = m_messageQueue.toArray();
                m_messageQueue = new ArrayList(MAX_BUFFERED_LOG_MESSAGES);
                int keepBufferedLogMessages = MAX_BUFFERED_LOG_MESSAGES - DROP_BUFFERED_LOG_MESSAGES;
                for (int i = 0; i < keepBufferedLogMessages; i++)
                {
                    m_messageQueue.add(i, messages[i + keepBufferedLogMessages]);
                }
                
                String warning = createLogMessage("Writing of centralized log messages is too slow; the oldest " + DROP_BUFFERED_LOG_MESSAGES +
                                                  " messages have been discarded, check remote logs", Level.WARNING);
                m_centralFileLogger.logMessage(warning);
            }
            
            m_messageQueue.add(message);
            m_messageQueueSyncObj.notifyAll();
        }
    }
    
    String getLogExtract(Long fromPosition, Long readLength)
    {
        open();

        if (fromPosition == null)
         {
            fromPosition = new Long(-1); // will cause read to be relative to end of file
        }
        if (readLength == null)
         {
            readLength = new Long(-1); // will cause default read length to be used
        }
        
        try
        {
            return new String(m_centralFileLogger.read(fromPosition.longValue(), readLength.longValue()));
        }
        catch (IOException e)
        {
            MFRuntimeException mfe = new MFRuntimeException("Failed to read centralized log file");
            mfe.setLinkedException(e);
            throw mfe;
        }
    }

    CompositeData getLogExtractAndLogFileSize(Long fromPosition, Long readLength)
    {
        try
        {
            Object[] itemValues = new Object[]
            {
                getLogExtract(fromPosition, readLength),
                new Long(m_centralFileLogger.length())
            };
            return new CompositeDataSupport(LOG_COMPOSITE_TYPE, LOG_COMPOSITE_TYPE_ITEM_NAMES, itemValues);
        }
        catch (OpenDataException e)
        {
            m_context.logMessage("Failed to create centralized log extract, trace follows...", e, Level.WARNING);
            return null;
        }
    }

    String getLogFile() { return new File(m_path).getAbsolutePath(); }
    void setLogFile(String path)
    throws Exception
    {
        if (path == null || path.length() == 0)
        {
            m_path = m_defaultPath;
        }
        else
        {
            m_path = path;
        }

        if (m_centralFileLogger != null)
        {
            m_centralFileLogger.resetLogDirectory(m_path);
        }
    }

    Long getLogFileSizeThreshold() { return new Long(m_logFileThreshold); }
    void setLogFileSizeThreshold(Long thresholdSize)
    {
        RollingFileLogger fileLogger = m_centralFileLogger;

        if (thresholdSize.longValue() != m_logFileThreshold && fileLogger != null)
        {
            fileLogger.setSizeThreshold(thresholdSize.longValue());
        }

        m_logFileThreshold = thresholdSize.longValue();
    }

    Long getLogFileRolloverSizeThreshold() { return new Long(m_logFileRolloverThreshold); }
    void setLogFileRolloverSizeThreshold(Long thresholdSize)
    {
        RollingFileLogger fileLogger = m_centralFileLogger;

        if (thresholdSize.longValue() != m_logFileRolloverThreshold && fileLogger != null)
        {
            fileLogger.setRolloverThreshold(thresholdSize.longValue());
        }

        m_logFileRolloverThreshold = thresholdSize.longValue();
    }

    Integer getLogFileRolloverTimeInterval() { return new Integer(m_logFileRolloverTimeInterval); }
    void setLogFileRolloverTimeInterval(Integer intervalValue)
    {
        RollingFileLogger fileLogger = m_centralFileLogger;

        if (intervalValue.intValue() != m_logFileRolloverTimeInterval && fileLogger != null)
        {
            fileLogger.set_LogRolloverTimeInterval(intervalValue.intValue());
        }

        m_logFileRolloverTimeInterval = intervalValue.intValue();
    }

    synchronized void clearLogFile()
    {
        open();
        
        try
        {
            m_centralFileLogger.clearLogFile();
        }
        catch(IOException e)
        {
            MFRuntimeException runtimeException = new MFRuntimeException("Failed to clear centralized log file");
            runtimeException.setLinkedException(e);
            throw runtimeException;
        }
        
        m_context.logMessage("Centralized log file truncated", Level.INFO);
    }

    synchronized void saveLogFile(String path)
    {
        open();
        
        try
        {
            m_centralFileLogger.saveLogFile(path);
        }
        catch(IOException e)
        {
            MFRuntimeException runtimeException = new MFRuntimeException("Failed to save centralized log file to " + path);
            runtimeException.setLinkedException(e);
            throw runtimeException;
        }
        
        m_context.logMessage("Centralized log file copied to " + path, Level.INFO);
    }
    
    void attemptLogFileRollover()
    throws IOException
    {
        open();
        
        RollingFileLogger fileLogger = m_centralFileLogger;

        if (fileLogger == null)
        {
            return;
        }
        
        m_context.logMessage("Manual rollover attempt of centralized log file initiated", Level.TRACE);
        fileLogger.attemptLogFileRollover();
    }

    synchronized final void configure()
    {
        IElement domainElement = m_context.getConfiguration("/domain/domain", true);
        IAttributeSet attributes = domainElement.getAttributes();

        try
        {
            String centralizedLogFile = (String)attributes.getAttribute(IDomainConstants.CENTRALIZED_LOG_FILE_ATTR);
            if (centralizedLogFile == null || centralizedLogFile.length() == 0)
            {
                if (!m_path.equals(m_defaultPath))
                {
                    m_path = m_defaultPath;
                    if (m_centralFileLogger != null)
                    {
                        m_centralFileLogger.resetLogDirectory(m_path);
                    }
                }
            }
            else
            {
                if (!m_path.equals(centralizedLogFile))
                {
                    m_path = centralizedLogFile;
                    if (m_centralFileLogger != null)
                    {
                        m_centralFileLogger.resetLogDirectory(m_path);
                    }
                }
            }

            Long logFileSizeThreshold = (Long)attributes.getAttribute(IDomainConstants.LOG_FILE_SIZE_THRESHOLD_ATTR);
            if (logFileSizeThreshold == null)
            {
                logFileSizeThreshold = new Long(IDomainConstants.LOG_FILE_SIZE_THRESHOLD_DEFAULT);
            }
            setLogFileSizeThreshold(logFileSizeThreshold);
            
            Long logFileRolloverSizeThreshold = (Long)attributes.getAttribute(IDomainConstants.LOG_FILE_ROLLOVER_SIZE_THRESHOLD_ATTR);
            if (logFileRolloverSizeThreshold == null)
            {
                logFileRolloverSizeThreshold = new Long(IDomainConstants.LOG_FILE_ROLLOVER_SIZE_THRESHOLD_DEFAULT);
            }
            setLogFileRolloverSizeThreshold(logFileRolloverSizeThreshold);

            Integer logFileRolloverTimeInterval = (Integer)attributes.getAttribute(IDomainConstants.LOG_FILE_ROLLOVER_TIME_INTERVAL_ATTR);
            if (logFileRolloverTimeInterval == null)
            {
                logFileRolloverTimeInterval = new Integer(IDomainConstants.LOG_FILE_ROLLOVER_TIME_INTERVAL_DEFAULT);
            }
            setLogFileRolloverTimeInterval(logFileRolloverTimeInterval);
        }
        catch (Exception e)
        {
            m_context.logMessage("Failed to configure centralized logging, trace follows...", e, Level.WARNING);
        }
    }

    void close()
    {
        try
        {
            // only need to create the central logger if we have something to do
            if (m_centralFileLogger != null)
            {
                m_centralFileLogger.close();
                m_centralFileLogger = null;
                synchronized (m_messageQueueSyncObj)
                {
                    m_messageQueue.clear();
                    m_messageQueueSyncObj.notifyAll();
                }
            }
        }
        catch (IOException e)
        {
            MFRuntimeException mfe = new MFRuntimeException("Failed to close centralized log file");
            mfe.setLinkedException(e);
            throw mfe;
        }
    }

    long length()
    {
        open();
        
        return m_centralFileLogger.length();
    }

    private void open()
    {
        try
        {
            // only need to create the central logger if we have something to do
            if (m_centralFileLogger == null)
            {
                m_centralFileLogger = new RollingFileLogger(m_context, m_path, m_context.getComponentName().getDomainName(),m_logFileRolloverTimeInterval);
                m_centralFileLogger.setSizeThreshold(m_logFileThreshold);
                m_centralFileLogger.setRolloverThreshold(m_logFileRolloverThreshold);
                m_writer = new LogWriter();
                m_writer.start();
            }
        }
        catch (IOException e)
        {
            MFRuntimeException mfe = new MFRuntimeException("Failed to open centralized log file");
            mfe.setLinkedException(e);
            throw mfe;
        }
    }
    
    private String createLogMessage(String message, int severityLevel)
    {
        StringBuffer timestampedMessage = new StringBuffer();
        timestampedMessage.append('[');
        timestampedMessage.append(m_dateTimeFormatter.format(new Date(System.currentTimeMillis())));
        timestampedMessage.append("]");
        timestampedMessage.append(" (");
        timestampedMessage.append(Level.LEVEL_TEXT[severityLevel]);
        timestampedMessage.append(") ");
        timestampedMessage.append(message);

        return timestampedMessage.toString();
    }

    public void writeMessages()
    {
        synchronized (m_messageQueueSyncObj)
        {
            if (!m_messageQueue.isEmpty())
            {
                String message = (String)m_messageQueue.remove(0);
                m_centralFileLogger.logMessage(message);
            }
            else
            {
                while (m_messageQueue.isEmpty())
                {
                    try
                    {
                        m_messageQueueSyncObj.wait();
                    }
                    catch (InterruptedException e)
                    {
                    } // eat it
                }
            }
        }
    }
    
    private class LogWriter
    extends Thread
    {
        private LogWriter()
        {
            super("Centralized Logging - Log Writer");
            super.setDaemon(true);
        }
        
        @Override
        public void run()
        {
            while (m_centralFileLogger != null)
            {
                try
                {
                    CentralizedLogger.this.writeMessages();
                }
                catch (NullPointerException e) // will happen on close
                {
                    if (m_centralFileLogger == null)
                    {
                        return;
                    }
                    e.printStackTrace();
                }
            }
        }
    }
}