package com.sonicsw.mf.framework.manager;

import java.util.Collection;
import java.util.Enumeration;
import java.util.Hashtable;

import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.IContainerIdentity;
import com.sonicsw.mf.common.runtime.IContainerState;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.common.runtime.impl.ComponentState;
import com.sonicsw.mf.common.runtime.impl.ContainerIdentity;
import com.sonicsw.mf.common.runtime.impl.ContainerState;

/**
 * This class encapsulates the entire runtime state of the domain as established
 * by the AgentManager.
 *
 * It handles new containers/components that come and go, and changes in state
 * of those containers/components.
 */
final class CurrentDomainState
{
    // This hashtable holds a state for each container that is known in the domain.
    // It is indexed by the canonical name - I choose that so that
    // when container configurations are updated we can most easily
    // maintain this table.
    private Hashtable m_containerStates = new Hashtable();

    /**
     * A CurrentDomainState object needs to be seeded with the initial list of
     * containers.
     */
    CurrentDomainState(IElement[] initialContainerList)
    {
        // build an initial list of states with some initial (interim) values
        for (int i = 0; i < initialContainerList.length; i++)
        {
            addContainer((IDirElement)initialContainerList[i]);
        }
    }

    void addContainer(IElement containerConfig)
    {
        // ignore templates
        if (((IDirElement)containerConfig).isTemplate())
        {
            return;
        }

        IContainerIdentity containerID = new ContainerIdentity(containerConfig);
        IElementIdentity containerConfigID = containerID.getConfigIdentity();

        // create the container state
        CanonicalName canonicalName = new CanonicalName(containerID.getDomainName(), containerID.getContainerName(), "");
        ContainerState containerState = new ContainerState(canonicalName, containerConfigID);
        // set a transient state while waiting for the actual state to be set
        containerState.setState(IContainerState.STATE_UNKNOWN);
        containerState.setContainerHost("<UNKNOWN>");
        containerState.setTimestamp(0);

        m_containerStates.put(canonicalName.getCanonicalName(), new Object[] { containerState });
    }

    void removeContainer(String containerConfigID)
    {
        synchronized(m_containerStates)
        {
            Enumeration keys = m_containerStates.keys();
            while (keys.hasMoreElements())
            {
                Object key = keys.nextElement();
                Object[] value = (Object[])m_containerStates.get(key);
                IContainerState containerState = (IContainerState)value[0];

                if (containerState.getRuntimeIdentity().getConfigIdentity().getName().equals(containerConfigID))
                {
                    m_containerStates.remove(key);
                    return;
                }
            }
        }
    }

    /**
     * Get the state as currently recorded by this object.
     */
    IContainerState[] getDomainState()
    {
        synchronized(m_containerStates)
        {
            Collection collection = m_containerStates.values();
            Object[] values = collection.toArray();

            IContainerState[] containerStates = new IContainerState[values.length];
            for (int i = 0; i < values.length; i++)
            {
                containerStates[i] = (IContainerState)((Object[])values[i])[0];
            }
            return containerStates;
        }
    }

    /**
     * Gets the recorded state for the requested container.
     */
    ContainerState getContainerState(String container)
    {
        Object[] value = (Object[])m_containerStates.get(container);

        if (value == null)
        {
            return null;
        }

        return (ContainerState)value[0];
    }

    /**
     * A new container state has been received so update the domain state with
     * the new state information.
     */
    void updateContainerState(String container, IContainerState newContainerState)
    {
        Object[] value = (Object[])m_containerStates.get(container);

        // did the container state get removed on another thread ?
        if (value == null)
        {
            return;
        }

        synchronized(value)
        {
            // we used to check the timestamp to see if it was later than the
            // one we currently have saved, but that led to problems when the
            // containers host clock was behind the AM host clock .. instead
            // now we assume if we got a state from the container then it should
            // always override the existing - danger is if we get a shutdown
            // notification and that gets handled before the last successful poll
            // before the shutdown; the expectation is that this will be unlikely
            // and would get corrected next poll cycle anyway
            value[0] = newContainerState;
        }
    }

    /**
     * A new container state has been received so update the domain state with
     * the new state information.
     *
     * @return True if the up to date container state should be requested from the container.
     */
    boolean updateContainerState(String container, short state, long timestamp)
    {
        Object[] value = (Object[])m_containerStates.get(container);

        // no longer exists (e.g. configuration was removed)
        if (value == null)
        {
            throw new IllegalArgumentException();
        }

        if (value[0] == null)
        {
            return false;
        }

        synchronized(value)
        {
            ContainerState containerState = (ContainerState)value[0];

            if (state == IContainerState.STATE_OFFLINE)
            {
                // if the update is as a result of a notification from the container, but we
                // already have more up to date information, then ignore
                long containerTimestamp = containerState.getTimeStamp();
                if (timestamp == containerTimestamp || (timestamp != 0 && timestamp < containerTimestamp))
                {
                    return false;
                }
                containerState.setTimestamp(timestamp);

                // if its already offline the ignore
                if (containerState.getState() == IContainerState.STATE_OFFLINE)
                {
                    return false;
                }

                // update the container
                containerState.setState(IContainerState.STATE_OFFLINE);

                // update the individual component states
                IComponentState[] componentStates = containerState.getComponentStates();
                for (int i = 0; i < componentStates.length; i++)
                {
                    ComponentState componentState = (ComponentState)componentStates[i];
                    componentState.setState(IComponentState.STATE_OFFLINE);
                    componentState.setLastErrorLevel(Level.UNKNOWN);
                    componentState.setLastErrorDescription("");
                }

                return false;
            }

            if (state == IContainerState.STATE_ONLINE)
            {
                // if the update is as a result of a notification from the container, but we
                // already have more up to date information, then ignore
                if (timestamp <= containerState.getTimeStamp())
                {
                    return false;
                }
                containerState.setTimestamp(timestamp);

                // if its already offline the ignore
                if (containerState.getState() == IContainerState.STATE_ONLINE)
                {
                    return false;
                }

                // update the container
                containerState.setState(IContainerState.STATE_ONLINE);

                return true;
            }
        }

        return false;
    }

    /**
     * A new component state has been received so update the domain state with
     * the new state information.
     *
     * @return True if the up to date container state should be requested from the container.
     */
    boolean updateComponentState(String container, String componentID, short state, String lastError, Integer lastErrorLevel, long timestamp)
    {
        Object[] value = (Object[])m_containerStates.get(container);

        // did the container state get removed on another thread ?
        if (value == null)
        {
            return false;
        }

        synchronized(value)
        {
            ContainerState containerState = (ContainerState)value[0];

            if (containerState.getState() == IContainerState.STATE_ONLINE)
            {
                // if the update is as a result of a notification from the container, but we
                // already have more up to date information, then ignore
                if (timestamp <= containerState.getTimeStamp())
                {
                    return true;
                }
                containerState.setTimestamp(timestamp);

                IComponentState[] componentStates = containerState.getComponentStates();
                for (int i = 0; i < componentStates.length; i++)
                {
                    IComponentIdentity componentIdentity = (IComponentIdentity)componentStates[i].getRuntimeIdentity();
                    if (componentIdentity.getComponentName().equals(componentID))
                    {
                        updateComponentState(componentStates[i], state, lastError, lastErrorLevel);
                        return false;
                    }
                }

                // the component was not found so we should request an update
                return true;
            }
            else // the container state is offline or unknown
            {
                if (state == IComponentState.STATE_ONLINE)
                {
                    // if the update is as a result of a notification from the container, but we
                    // already have more up to date information, then ignore
                    if (timestamp <= containerState.getTimeStamp())
                    {
                        return true;
                    }
                    containerState.setTimestamp(timestamp);

                    // set the state to online - how else would the component got online ?
                    containerState.setState(IContainerState.STATE_ONLINE);

                    IComponentState[] componentStates = containerState.getComponentStates();
                    for (int i = 0; i < componentStates.length; i++)
                    {
                        IComponentIdentity componentIdentity = (IComponentIdentity)componentStates[i].getRuntimeIdentity();
                        if (componentIdentity.getComponentName().equals(componentID))
                        {
                            updateComponentState(componentStates[i], state, lastError, lastErrorLevel);
                        }
                    }
                    return true;
                }
                else
                if (state == IComponentState.STATE_OFFLINE)
                {
                    // if the update is as a result of a notification from the container, but we
                    // already have more up to date information, then ignore
                    if (timestamp <= containerState.getTimeStamp())
                    {
                        return true;
                    }
                    containerState.setTimestamp(timestamp);

                    IComponentState[] componentStates = containerState.getComponentStates();
                    for (int i = 0; i < componentStates.length; i++)
                    {
                        IComponentIdentity componentIdentity = (IComponentIdentity)componentStates[i].getRuntimeIdentity();
                        if (componentIdentity.getComponentName().equals(componentID))
                        {
                            // if we already have the state recorded .. return indicated the full container state should not
                            // be requested
                            if (componentStates[i].getState() == state)
                            {
                                return false;
                            }
                            else
                            {
                                updateComponentState(componentStates[i], state, lastError, lastErrorLevel);
                                return true;
                            }
                        }
                    }
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }
    }

    /**
     * A component has been unloaded and thus should be removed from the containers list
     * if the unload did not happen as part of a shutdown.
     */
    void removeComponentState(String container, String componentID, long timestamp)
    {
        Object[] value = (Object[])m_containerStates.get(container);

        // did the container state get removed on another thread ?
        if (value == null)
        {
            return;
        }

        synchronized(value)
        {
            ContainerState containerState = (ContainerState)value[0];

            // if its already offline the ignore
            if (containerState.getState() == IContainerState.STATE_OFFLINE)
            {
                return;
            }

            // if the update is as a result of a notification from the container, but we
            // already have more up to date information, then ignore
            if (timestamp < containerState.getTimeStamp())
            {
                return;
            }
            else
            {
                containerState.setTimestamp(timestamp);
            }

            containerState.removeComponentState(componentID);
        }
    }

    private void updateComponentState(IComponentState componentState, short state, String lastError, Integer lastErrorLevel)
    {
        ((ComponentState)componentState).setState(state);
        ((ComponentState)componentState).setLastErrorDescription(lastError);
        ((ComponentState)componentState).setLastErrorLevel(lastErrorLevel.intValue());
    }
}
