package com.sonicsw.mf.framework.manager;

import java.util.ArrayList;
import java.util.Date;
import java.util.EmptyStackException;
import java.util.HashSet;
import java.util.Stack;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.IValueType;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.common.metrics.manager.IMetricsRegistrar;
import com.sonicsw.mf.common.metrics.manager.ISampledStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatisticProvider;
import com.sonicsw.mf.common.metrics.manager.StatisticsFactory;
import com.sonicsw.mf.common.runtime.IContainerState;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.runtime.IAgentManagerProxy;

/**
 * This thread is responsible for polling all of the containers in the domain
 * to ascertain their state.
 *
 * As the state of each container is obtained, the domain state object is updated
 * with the information. If a request for a container's state times out, then that
 * container's state will be considered to be "offline" by the AgentManager.
 *
 * Due to the Metrics requirements [that the metrics can be enabled\disabled
 * while in off-line state] and the fact that the AgentManager does not
 * create a DomainStateMonitor instance until its "start" method, this
 * class is now a SINGLETON [it has static members for the Metrics].  If
 * multiple instances of the class are required at some point, then one should
 * look into creating the DomainStateMonitor thread in the AM's "init" method,
 * and starting the DSM instance in the AM's "start" method.  If that is done, of course,
 * code will have to be added to DSM to deal with proper cleanup (in case the
 * DSM thread is created but never started).
 */
final class DomainStateMonitor
extends Thread
{
    private boolean DEBUG_THREADS = false;
    private boolean DEBUG = false;

    private AgentManager m_manager;
    private CurrentDomainState m_domainState;
    private IComponentContext m_context;

    private Integer m_defaultStatusPollInterval = new Integer(60);
    private Integer m_defaultStatusPollTimeout = new Integer(30);

    // When an execution thread is idle on the cache for longer than this
    // duration, the thread can be removed from the thread pool as long as the
    // min pool size is maintained
    private static final long MAX_IDLE_DURATION = 5000;

    // This set holds a list of status checks that must be performed. Each entry
    // holds the time the check should be invoked and the container that the check should
    // be perfomed against. The list is treated as a FIFO queue of tasks by the monitor thread.
    // As each check is completed, a new entry is added to the list with a time of
    // the current time + the poll interval and the completing checks container information.
    private ArrayList m_scheduledStatusChecks = new ArrayList();

    private short m_maxWorkerThreads;
    private short m_minWorkerThreads;
    private short m_workerThreadPoolSize = 0;  //count of worker threads in the thread cache

    private static Stack s_workerThreadPool = new Stack();
    private static int s_executingWorkerThreads = 0;

    // metrics
    private static volatile IMetricsRegistrar s_metricsRegistrar;

    // statistics
    private static volatile IStatistic s_poolSizeStatistic;
    private static volatile IStatistic s_maxPoolSizeStatistic;  // "high water mark"
    private static volatile IStatistic s_poolWaitsStatistic;
    
    // flag to indicate if we should carry on polling
    private boolean m_run = true;
    private Object m_monitorLockObj = new Object();
    
    // static flag to indicate if thread has been started
    private static boolean s_running = false;

    // name to use for
    public static final String TASK_THREAD_NAME = "Container State Poller ";

    // default values for Min and Max threads
    public static final short DEFAULT_MIN_THREADS = 0;
    public static final short DEFAULT_MAX_THREADS = 50;
    public static final short MAX_THREAD_LOWER_BOUND = 20;
    public static final short MAX_THREAD_UPPER_BOUND = 1000;
    
    /**
     * A CurrentDomainState object needs to be seeded with the initial list of
     * containers.
     */
    DomainStateMonitor(AgentManager manager, CurrentDomainState domainState, IComponentContext context)
    {
        super(IAgentManagerProxy.GLOBAL_ID + " - Domain State Monitor");

        m_manager = manager;
        m_domainState = domainState;
        m_context = context;

        m_minWorkerThreads = DomainStateMonitor.DEFAULT_MIN_THREADS;
        m_maxWorkerThreads = DomainStateMonitor.DEFAULT_MAX_THREADS;
    }

    @Override
    public void run()
    {
        // set flag to indicate that thread has been started
        s_running = true;

        // when first started we must do at least one poll of all the containers
        // even if polling is not enabled
        reschedule();

        // begin loop for scheduled checks
        while (m_run)
        {
            Check check = null;

            synchronized (m_monitorLockObj)
            {
                if (DEBUG_THREADS)
                {
                    System.out.println("DomainStateMonitor.run: count of scheduled status checks = " + m_scheduledStatusChecks.size());
                }
                if (!m_scheduledStatusChecks.isEmpty()) // any status checks scheduled?
                {
                    check = (Check) m_scheduledStatusChecks.get(0);
                    if (DEBUG_THREADS)
                    {
                        System.out.println("DomainStateMonitor.run: schedule check for container = " + check.container);
                    }
                    if (check.scheduledTime <= System.currentTimeMillis()) // time for a check...
                    {
                        if (DEBUG_THREADS)
                        {
                            System.out.println("DomainStateMonitor.run: time to check for container = " + check.container);
                        }
                        // we add it back to the end of the queue so that a reschedule between execution
                        // will not add another entry
                        check.scheduledTime = Long.MAX_VALUE;
                        replaceCheck(check);
                        if (DEBUG_THREADS)
                        {
                            System.out.println("DomainStateMonitor.run: ...for container = " + check.container + ", m_run = " + m_run);
                        }
                    }
                    else // wait until it is time to check...
                    {
                        long interval = check.scheduledTime - System.currentTimeMillis();
                        if (DEBUG_THREADS)
                        {
                            System.out.println("DomainStateMonitor.run: wait for interval = " + interval + " before polling...");
                        }
                        try
                        {
                            if (interval > 0)
                            {
                                m_monitorLockObj.wait(interval);
                            }
                            continue;
                        }
                        catch (InterruptedException ie)
                        {
                            if (DEBUG_THREADS)
                            {
                                m_context.logMessage("DomainStateMonitor.run: InterruptedException thrown during wait(interval)...", Level.TRACE);
                                ie.printStackTrace();
                            }
                        }
                    }
                    if (DEBUG_THREADS)
                    {
                        System.out.println("DomainStateMonitor.run: exiting if block for container = " + check.container);
                    }
                }
                else // if no scheduled status checks, wait for one
                {
                    if (DEBUG_THREADS)
                    {
                        System.out.println("DomainStateMonitor.run: no scheduled status checks, sleep for a second...");
                    }
                    try
                    {
                        m_monitorLockObj.wait(1000); // wait for a second...
                        continue;
                    }
                    catch (InterruptedException ie)
                    {
                        if (DEBUG_THREADS)
                        {
                            m_context.logMessage("DomainStateMonitor.run: InterruptedException thrown during wait(1000)...", Level.TRACE);
                            ie.printStackTrace();
                        }
                    }
                }
                if (DEBUG_THREADS)
                {
                    System.out.println("DomainStateMonitor.run: about to release synch lock on DSM for container = " + check.container);
                }
            } //synchronized

            if (DEBUG_THREADS)
            {
                System.out.println("DomainStateMonitor.run: get a worker threead for container = " +
                (check == null ? null : check.container));
            }
            // get a thread with which to execute the check
            assignTask(check);

        }//while (m_run)

        // set flag, before returning, to indicate that thread is no longer running
        s_running = false;
    }

    private void assignTask(Runnable task)
    {
        WorkerThread workerThread = null;
        synchronized (s_workerThreadPool)
        {
            while (workerThread == null)
            {
                try
                {
                    workerThread = (WorkerThread)s_workerThreadPool.pop();
                }
                catch (EmptyStackException e)
                {
                    if (m_workerThreadPoolSize == m_maxWorkerThreads) // wait until we're notified that a thread has become available
                    {
                        if (s_poolWaitsStatistic != null)
                        {
                            s_poolWaitsStatistic.updateValue(1);
                        }
                        try
                        {
                            if (DEBUG_THREADS)
                            {
                                System.out.println("DomainStateMonitor.assignTask: wait on worker thread pool...");
                            }
                            s_workerThreadPool.wait();
                            
                            // in case we were woken up during shutdown
                            if (!m_run)
                            {
                                return;
                            }
                        }
                        catch (InterruptedException ie)
                        {
                            if (DEBUG_THREADS)
                            {
                                m_context.logMessage("DomainStateMonitor.assignTask: InterruptedException thrown during wait on s_workerThreadPool...", Level.TRACE);
                                ie.printStackTrace();
                            }
                            return;
                        }
                    }
                    else // we'll be creating a new one so increment the count in the sync block
                    {
                        m_workerThreadPoolSize++;
                        break;
                    }
                }
            }

            // if there is no execution thread then create one
            if (workerThread == null)
            {
                if (DEBUG_THREADS)
                {
                    System.out.println("DomainStateMonitor.assignTask: assign task to new worker thread: " + task);
                }

                workerThread = new WorkerThread();
                workerThread.task = task;
                workerThread.setDaemon(true);
                workerThread.start();
            }
            else
            {
                if (DEBUG_THREADS)
                {
                    System.out.println("DomainStateMonitor.assignTask: assign task to existing worker thread: " + task);
                }

                // update the execution thread's task and wake it to actually execute the task
                workerThread.task = task;
            }
            s_executingWorkerThreads++; // update count of threads currently in process of executing tasks
            s_workerThreadPool.notifyAll();
        }
    }

    /**
     * Gets the maximum number of task execution threads the task scheduler will use.
     */
    public short getMaxThreads() { return m_maxWorkerThreads; }

    /**
     * Set the maximum number of task threads the task scheduler will use.
     *
     * @param maxThreads Must be >=20, >min threads and <= 1000.
     */
    public void setMaxThreads(short maxThreads)
    {
        synchronized (m_monitorLockObj)
        {
            if (maxThreads > DomainStateMonitor.MAX_THREAD_UPPER_BOUND)
            {
                throw new IllegalArgumentException("Maximum scheduler threads cannot exceed " + DomainStateMonitor.MAX_THREAD_UPPER_BOUND + ".");
            }
            if (maxThreads < m_minWorkerThreads)
            {
                throw new IllegalArgumentException("Maximum scheduler threads cannot be less than the minimum scheduler threads.");
            }
            if (maxThreads < DomainStateMonitor.MAX_THREAD_LOWER_BOUND)
            {
                throw new IllegalArgumentException("Maximum scheduler threads cannot be less than " + DomainStateMonitor.MAX_THREAD_LOWER_BOUND + ".");
            }
            m_maxWorkerThreads = maxThreads;
        }
    }

    /**
     * Gets the minimum number of worker threads the DomainStateMonitor will use
     * (i.e. the minimum thread pool size).
     */
    public short getMinThreads() { return m_minWorkerThreads; }

    /**
     * Set the minimum number of worker threads the DomainStateMonitor will use.
     *
     * @param minThreads Must be >= 0 and < max threads.
     */
    public void setMinThreads(short minThreads)
    {
        synchronized (m_monitorLockObj)
        {
            if (minThreads < (short)0)
            {
                throw new IllegalArgumentException("Minimum number of domain state monitor threads cannot be less than 0.");
            }
            if (minThreads > m_maxWorkerThreads)
            {
                throw new IllegalArgumentException("Minimum number of domain state monitor threads must be less than, or equal to, the maximum number of domain state monitor threads.");
            }
            m_minWorkerThreads = minThreads;
        }
    }


    /**
     * Stops all monitoring activity and cleans up
     */
    void cleanup()
    {
        synchronized (m_monitorLockObj)
        {
            m_scheduledStatusChecks.clear();
            m_run = false;

            // notify the polling thread to wake up
            m_monitorLockObj.notifyAll();

            // any task scheduling waiting of a pool thread
            synchronized (s_workerThreadPool)
            {
                s_workerThreadPool.notifyAll();
            }
        }
    }

    /**
     * Ensures that:
     *  - for every status check to be performed, there is an entry in the current
     *    domain state (if not then the scheduled status check will be removed)
     *  - for every container in the current domain state there is a status check
     *    scheduled in the list of status checks to be performed (if not then a
     *    check is added to the front of the queue
     */
    void reschedule()
    {
        synchronized (m_monitorLockObj)
        {
            IContainerState[] containerStates = m_domainState.getDomainState();

            // build an indexed list of container states
            HashSet containers = new HashSet();
            for (int i = 0; i < containerStates.length; i++)
            {
                containers.add(containerStates[i].getRuntimeIdentity().getCanonicalName());
            }

            if (DEBUG)
            {
                m_context.logMessage("DomainStateMonitor.reschedule(): number of containers=" + containerStates.length, Level.TRACE);
            }

            // check for each scheduled status check there is still a container to be checked
            Object[] priorChecks = m_scheduledStatusChecks.toArray();
            HashSet currentChecks = new HashSet();
            for (int i = 0; i < priorChecks.length; i++)
            {
                if (containers.contains(((Check)priorChecks[i]).container))
                {
                    currentChecks.add(((Check)priorChecks[i]).container);
                }
                else
                {
                    m_scheduledStatusChecks.remove(priorChecks[i]);
                }
            }

            // check for each container to be checked there is a scheduled status check
            for (int i = 0; i < containerStates.length; i++)
            {
                String container = containerStates[i].getRuntimeIdentity().getCanonicalName();
                if (!currentChecks.contains(container))
                {
                    Check newCheck = new Check(0, container);
                    if (!newCheck.updatePollingParams())
                    {
                        continue;
                    }
                    m_scheduledStatusChecks.add(0, newCheck);
                }
            }
        }
    }
    
    void schedule(final String container)
    {
        Runnable task = new Runnable()
        {
            @Override
            public void run()
            {
                try
                {
                    IContainerState containerState = m_manager.getContainerState(container, 0);
                    m_domainState.updateContainerState(container, containerState);
                }
                catch (Exception e) {}
            }
        };
        assignTask(task);
    }

    /**
     * Called when the poll interval changes on the given container
     */
    void reschedule(String container)
    {
        synchronized (m_monitorLockObj)
        {
            int size = m_scheduledStatusChecks.size();
            for (int i = 0; i < size; i++)
            {
                Check check = (Check)m_scheduledStatusChecks.get(i);
                if (check.container.equals(container))
                {
                    if (check.scheduledTime != Long.MAX_VALUE)
                     {
                        return; // the check will reschedule itself with the new value after the current check has run
                    }
                    m_scheduledStatusChecks.remove(i);
                    Check newCheck = new Check(0, check);
                    if (!newCheck.updatePollingParams())
                    {
                        continue;
                    }
                    m_scheduledStatusChecks.add(0, newCheck);
                    return;
                }
            }
        }
    }

    void handleContainerStateNotification(IContainerState containerState, long timestamp)
    {
        String sourceContainer = containerState.getRuntimeIdentity().getCanonicalName();
        DomainStateMonitor.this.m_domainState.updateContainerState(sourceContainer, containerState);

        if (DEBUG)
        {
            m_context.logMessage("DomainStateMonitor.handleContainerStateNotification: received container state notification...", Level.TRACE);
        }
        if ((DomainStateMonitor.this.m_manager.m_traceMask & AgentManager.TRACE_STATUS_POLLING) > 0)
        {
            DomainStateMonitor.this.m_context.logMessage("Container state notification received: " + sourceContainer + ", state=" + IContainerState.STATE_TEXT[containerState.getState()], Level.TRACE);
        }

        Check replacementCheck = null;

        synchronized(m_monitorLockObj)
        {
            // find the existing check if there is one and remove it
            int size = m_scheduledStatusChecks.size();
            for (int i = 0; i < size; i++)
            {
                Check check = (Check)m_scheduledStatusChecks.get(i);
                if (check.container.equals(sourceContainer))
                {
                    replacementCheck = check;
                    break;
                }
            }

            // if there is not an existing Check then create one
            if (replacementCheck == null)
            {
                if (DEBUG)
                {
                    m_context.logMessage("DomainStateMonitor.handleContainerStateNotification: didn't find check for container OR not time to check yet, returning...", Level.TRACE);
                }
                // if we got here then no existing Check was found, so create one
                replacementCheck = new Check(0, sourceContainer); // create a dummy Check just to get the sourceContainer's polling parameters...
                if (!replacementCheck.updatePollingParams())
                {
                    return;
                }
            }

            // update the time we want the check to occur
            replacementCheck.scheduledTime = replacementCheck.pollingParams[0] == 0 ? Long.MAX_VALUE : (System.currentTimeMillis() + replacementCheck.pollingParams[0]);

            // now (re)place in the list of checks
            replaceCheck(replacementCheck);
        }
    }

    private void replaceCheck(Check check)
    {
        synchronized (m_monitorLockObj)
        {
            if (DEBUG)
            {
                m_context.logMessage("DomainStateMonitor.replaceCheck(): container=" + check.container + ", checkTime=" + new Date(check.scheduledTime), Level.TRACE);
            }

            // remove any existing entry first
            int size = m_scheduledStatusChecks.size();
            for (int i = 0; i < size; i++)
            {
                Check existingCheck = (Check)m_scheduledStatusChecks.get(i);
                if (check.container.equals(existingCheck.container))
                {
                    m_scheduledStatusChecks.remove(i);
                    break;
                }
            }

            try
            {
                if (check.scheduledTime == Long.MAX_VALUE)
                {
                    m_scheduledStatusChecks.add(check);
                }
                else
                {
                    size = m_scheduledStatusChecks.size();
                    for (int i = 0;i < size;i++)
                    {
                        Check existingCheck = (Check)m_scheduledStatusChecks.get(i);
                        if (check.scheduledTime < existingCheck.scheduledTime)
                        {
                            m_scheduledStatusChecks.add(i, check);
                            return;
                        }
                    }
                    m_scheduledStatusChecks.add(check);
                }
            }
            finally
            {
                m_monitorLockObj.notifyAll();
            }
        }
    }

    public static IMetricInfo[] getMetricsInfo()
    {
        // create the infos
        IMetricInfo[] infos = new IMetricInfo[3];
        infos[0] = MetricsFactory.createMetricInfo(IAgentManagerProxy.SYSTEM_POLLTHREADS_CURRENTPOOLSIZE_METRIC_ID, IValueType.VALUE,
                                                   "Size of thread pool used to poll for container(s) state.",
                                                   null, false, true, true, false, "pool threads");
        infos[1] = MetricsFactory.createMetricInfo(IAgentManagerProxy.SYSTEM_POLLTHREADS_MAXPOOLSIZE_METRIC_ID, IValueType.VALUE,
                                                   "Maximum size of thread pool used to poll for container(s) state since last metrics reset.",
                                                   null, false, true, false, false, "pool threads");
        infos[2] = MetricsFactory.createMetricInfo(IAgentManagerProxy.SYSTEM_POLLTHREADS_POOLWAITS_METRIC_ID, IValueType.VALUE,
                                                   "Number of times requests for container state had to wait because a pooled thread was not immediately available to service a poll request. Evaluated over the last 30 minutes.",
                                                   null, false, true, true, false, "waits");
        return infos;
    }

    public static void initMetrics(IMetricsRegistrar metricsRegistrar)
    {
        s_metricsRegistrar = metricsRegistrar;

        // memory usage statistic provider and statistics
        IStatisticProvider[] poolSizeProviders = new IStatisticProvider[]
        {
            new IStatisticProvider()
            {
                @Override
                public void updateStatistic(ISampledStatistic statistic)
                {
                    // need to change this to report the number of free threads *plus*
                    // the number of threads in use...
                    if (s_running)
                    {
                        synchronized (s_workerThreadPool)
                        {
                            statistic.updateValue(s_workerThreadPool.size() + s_executingWorkerThreads);
                        }
                    }
                    else
                    {
                        statistic.updateValue((int)0);
                    }
                }
                @Override
                public void resetStatistic(ISampledStatistic statistic) { } // no implementation
            }
        };
        s_poolSizeStatistic = StatisticsFactory.createStatistic(IStatistic.VALUE_MODE, false, poolSizeProviders, (short)0);
        s_maxPoolSizeStatistic = StatisticsFactory.createStatistic(IStatistic.MAXIMUM_MODE, false, poolSizeProviders, (short)0);

        // pool waits
        s_poolWaitsStatistic = StatisticsFactory.createStatistic(IStatistic.COUNTER_MODE, true, null, (short)1);
    }

    public static void enableMetrics(IMetricIdentity[] ids)
    {
        // now enable the ones that the agent deals with directly
        for (int i = 0; i < ids.length; i++)
        {
            if (ids[i].equals(IAgentManagerProxy.SYSTEM_POLLTHREADS_CURRENTPOOLSIZE_METRIC_ID))
            {
                s_metricsRegistrar.registerMetric(IAgentManagerProxy.SYSTEM_POLLTHREADS_CURRENTPOOLSIZE_METRIC_ID, s_poolSizeStatistic);
            }
            else if (ids[i].equals(IAgentManagerProxy.SYSTEM_POLLTHREADS_MAXPOOLSIZE_METRIC_ID))
            {
                s_metricsRegistrar.registerMetric(IAgentManagerProxy.SYSTEM_POLLTHREADS_MAXPOOLSIZE_METRIC_ID, s_maxPoolSizeStatistic);
            }
            else if (ids[i].equals(IAgentManagerProxy.SYSTEM_POLLTHREADS_POOLWAITS_METRIC_ID))
            {
                s_poolWaitsStatistic.reset();
                s_metricsRegistrar.registerMetric(IAgentManagerProxy.SYSTEM_POLLTHREADS_POOLWAITS_METRIC_ID, s_poolWaitsStatistic);
            }
        }
    }

    public static void disableMetrics(IMetricIdentity[] ids)
    {
        // now enable the ones that the agent deals with directly
        for (int i = 0; i < ids.length; i++)
        {
            if (ids[i].equals(IAgentManagerProxy.SYSTEM_POLLTHREADS_CURRENTPOOLSIZE_METRIC_ID))
            {
                s_metricsRegistrar.unregisterMetric(IAgentManagerProxy.SYSTEM_POLLTHREADS_CURRENTPOOLSIZE_METRIC_ID);
            }
            else if (ids[i].equals(IAgentManagerProxy.SYSTEM_POLLTHREADS_MAXPOOLSIZE_METRIC_ID))
            {
                s_metricsRegistrar.unregisterMetric(IAgentManagerProxy.SYSTEM_POLLTHREADS_MAXPOOLSIZE_METRIC_ID);
            }
            else if (ids[i].equals(IAgentManagerProxy.SYSTEM_POLLTHREADS_POOLWAITS_METRIC_ID))
            {
                s_metricsRegistrar.unregisterMetric(IAgentManagerProxy.SYSTEM_POLLTHREADS_POOLWAITS_METRIC_ID);
            }
        }
    }

    private final class Check
    implements Runnable
    {
        String container;
        long scheduledTime;
        long[] pollingParams = new long[2]; // [0] - interval, [1] - timeout

        private Check(long scheduledTime, String container)
        {
            // don't want a scheduled time less than the current time (unless this is a special case of 0)
            long currentTime = System.currentTimeMillis();
            this.scheduledTime = (scheduledTime != 0 && scheduledTime < currentTime) ? currentTime : scheduledTime;
            this.container = container;
        }

        private Check(long scheduledTime, Check lastCheck)
        {
            this(scheduledTime, lastCheck.container);
            this.pollingParams = lastCheck.pollingParams;
        }

        @Override
        public void run()
        {
            Thread.currentThread().setName(TASK_THREAD_NAME + '[' + container + ']');

            // if unable to update the polling params its because the configuration was deleted from under us
            // so get out
            if (!updatePollingParams())
            {
                if (DEBUG_THREADS)
                {
                    m_context.logMessage("DomainStateMonitor.Check.run: UNABLE TO UPDATE POLLING PARAMETERS!!! for container = " + this.container, Level.TRACE);
                }
                synchronized(m_monitorLockObj)
                {
                    m_scheduledStatusChecks.remove(this);
                }
                return;
            }

            long statusCheckStartTime = System.currentTimeMillis();

            if (DEBUG_THREADS)
            {
                m_context.logMessage("DomainStateMonitor.Check.run: container = " + this.container + ", pollingParams[0] = " + this.pollingParams[0], Level.TRACE);
            }

            // a value of 0 means don't poll
            if (this.pollingParams[0] != 0)
            {
                if (DEBUG_THREADS)
                {
                    m_context.logMessage("DomainStateMonitor.Check.run: container = " + this.container + ", pollingParams[1] = " + this.pollingParams[1], Level.TRACE);
                }

                try
                {
                    if ((DomainStateMonitor.this.m_manager.m_traceMask & AgentManager.TRACE_STATUS_POLLING) > 0)
                    {
                        DomainStateMonitor.this.m_context.logMessage("Polling: " + this.container, Level.TRACE);
                    }

                    IContainerState containerState = DomainStateMonitor.this.m_manager.getContainerState(this.container, this.pollingParams[1]);
                    DomainStateMonitor.this.m_domainState.updateContainerState(this.container, containerState);

                    if ((DomainStateMonitor.this.m_manager.m_traceMask & AgentManager.TRACE_STATUS_POLLING) > 0)
                    {
                        DomainStateMonitor.this.m_context.logMessage("Polled container state: " + this.container + ", state=" + IContainerState.STATE_TEXT[containerState.getState()], Level.TRACE);
                    }
                }
                catch(Exception e)
                {
                    if ((DomainStateMonitor.this.m_manager.m_traceMask & AgentManager.TRACE_STATUS_POLLING) > 0)
                    {
                        DomainStateMonitor.this.m_context.logMessage("Polling failed: " + this.container, Level.TRACE);
                    }
                    // if we get here then we failed to get the container state and we should update the
                    // state to reflect an unknown state
                    // NOTE: we use the start of the status request time in case any notifications subsequently
                    //       come in and would otherwise get overwritten
                    try
                    {
                        DomainStateMonitor.this.m_domainState.updateContainerState(this.container, IContainerState.STATE_OFFLINE, statusCheckStartTime);
                    }
                    catch (IllegalArgumentException iae) // thrown when container config has been deleted
                    {
                        synchronized(m_monitorLockObj)
                        {
                            m_scheduledStatusChecks.remove(this);
                        }
                        return;
                    }
                }

                // if unable to update the polling params its because the configuration was deleted from under us
                if (!updatePollingParams())
                {
                    if (DEBUG_THREADS)
                    {
                        m_context.logMessage("DomainStateMonitor.Check.run: UNABLE TO UPDATE POLLING PARAMETERS after poll!!!...for container = " + this.container, Level.TRACE);
                    }
                    synchronized(m_monitorLockObj)
                    {
                        m_scheduledStatusChecks.remove(this);
                    }
                    return;
                }
            }

            // now update the replacement task if it still exists
            synchronized(m_monitorLockObj)
            {
                // reset the scheduled time, add back to the list of scheduled checks
                this.scheduledTime = this.pollingParams[0] == 0 ? Long.MAX_VALUE : (statusCheckStartTime + this.pollingParams[0]);;
                DomainStateMonitor.this.replaceCheck(this);
                m_monitorLockObj.notifyAll();
            }
        }

        private boolean updatePollingParams()
        {
            Integer pollInterval = null;
            Integer pollTimeout = null;

            try
            {
                IContainerState containerState = DomainStateMonitor.this.m_domainState.getContainerState(this.container);
                String configID = containerState.getRuntimeIdentity().getConfigIdentity().getName();
                IElement containerConfig = (IElement)DomainStateMonitor.this.m_context.getConfiguration(configID, true);

                // the config was deleted under us, so return false
                if (containerConfig == null)
                {
                    return false;
                }

                IAttributeSet monitoringAttrs = (IAttributeSet)containerConfig.getAttributes().getAttribute(IContainerConstants.MONITORING_ATTR);
                if (monitoringAttrs != null)
                {
                    pollInterval = (Integer)monitoringAttrs.getAttribute(IContainerConstants.STATUS_POLL_INTERVAL_ATTR);
                    pollTimeout = (Integer)monitoringAttrs.getAttribute(IContainerConstants.STATUS_POLL_TIMEOUT_ATTR);
                }
            }
            catch(Exception e) { }
            finally
            {
                if (pollInterval == null)
                {
                    pollInterval = m_defaultStatusPollInterval;
                }
                if (pollTimeout == null)
                {
                    pollTimeout = m_defaultStatusPollTimeout;
                }

                this.pollingParams[0] = pollInterval.intValue() * 1000;
                this.pollingParams[1] = pollTimeout.intValue() * 1000;
            }

            return true;
        }
        
        @Override
        public String toString()
        {
            return "DomainStateMonitor.Check[container=" + container + ",scheduledTime=" + scheduledTime + ",pollingParams[0]=" + pollingParams[0] + ",pollingParams[1]=" + pollingParams[1] + "]";
        }
    }

    private class WorkerThread
    extends Thread
    {
        String role;
        Runnable task;

        private WorkerThread() { super(TASK_THREAD_NAME); }

        public synchronized void setRole(String role)
        {
            if (this.role != null)
            {
                throw new IllegalStateException("Cannot reset execution role.");
            }
            this.role = role;
        }

        public synchronized String getRole() { return this.role; }

        @Override
        public void run()
        {
            while (true)
            {
                synchronized(DomainStateMonitor.this.s_workerThreadPool)
                {
                    try
                    {
                        if (task == null)
                        {
                            DomainStateMonitor.this.s_workerThreadPool.wait(MAX_IDLE_DURATION);
                        }
                        if (task == null) // not reused within max idle time so see if should be removed
                        {
                            // don't remove the execution thread from the pool if the minimum
                            // pool size is not met
                            if (DomainStateMonitor.this.m_workerThreadPoolSize <= DomainStateMonitor.this.m_minWorkerThreads)
                            {
                                continue;
                            }

                            // don't remove the execution thread from the cache if there are more tasks
                            // waiting to be executed than threads in the cache
                            if (DomainStateMonitor.this.s_workerThreadPool.size() < DomainStateMonitor.this.m_scheduledStatusChecks.size())
                            {
                                continue;
                            }

                            // else remove from the cache
                            DomainStateMonitor.this.m_workerThreadPoolSize--;
                            DomainStateMonitor.this.s_workerThreadPool.remove(this);
                            return;
                        }
                    }
                    catch(InterruptedException ie)
                    {
                        break;
                    }
                }

                try
                {
                    this.task.run();
                }
                catch(Throwable e)
                {
                    // the task should have reported any conditions rather than throw exceptions
                    DomainStateMonitor.this.m_context.logMessage("Unhandled task failure, trace follows...", e, Level.WARNING);
                }
                finally
                {
                    this.setName(TASK_THREAD_NAME + "[idle]");
                    this.role = null;
                    this.task = null;
                }

                synchronized(DomainStateMonitor.this.s_workerThreadPool)
                {
                    DomainStateMonitor.this.s_workerThreadPool.push(this);
                    DomainStateMonitor.this.s_executingWorkerThreads--; // update count of threads currently in process of executing tasks
                    DomainStateMonitor.this.s_workerThreadPool.notifyAll();
                }
            }
        }
    }
}
