package com.sonicsw.mf.framework.manager;

import com.sonicsw.mf.common.IComponent;
import com.sonicsw.mf.common.runtime.IFaultTolerantState;
import com.sonicsw.mf.common.runtime.IStateManager;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.NonRecoverableStateChangeException;
import com.sonicsw.mf.common.runtime.RecoverableStateChangeException;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.mgmtapi.config.constants.IAgentManagerConstants;

/**
 * AM failure detection is performed by the standby AM to determine if it should become active.
 * Active AM failure detection is based on the ability of a standby to acquire a lock from the DS.
 * The lock is specific to the active AM role.  The lock is acquired by requesting the DS to reserve
 * the lock for some lease period and renewing the lease before the original lease expired.
 * During normal running, the active AM will hold the lock by renewing its lease with the DS and
 * the standby will poll in attempts to acquire the lock.  When the active AM fails, the active's
 * will expire and the DS will lease the lock to the standby on its next acquisition attempt.
 */

class FaultDetector
extends Thread
{
    private AgentManager m_am;
    private IFrameworkComponentContext m_context;
    private IStateManager m_stateManager;
    private long m_interval = IAgentManagerConstants.FAULT_DETECTION_INTERVAL_DEFAULT * 1000;
    private int m_traceMask ;
    private boolean m_failoverConditionLogged = false;
    private boolean m_isClosing = false;
    private boolean m_isSuspended = false;

    FaultDetector(AgentManager am)
    {
        super("AM Fault Detector");
        super.setDaemon(true);
        m_am = am;
        m_stateManager = am.getFaultToleranceStateManager();
        m_context = am.getContext();
    }

    @Override
    public void run()
    {
        while (!m_isClosing)
        {
            short prePartnerCheckState = m_stateManager.getState(null);
            try
            {
                if (m_isSuspended)
                {
                    if (m_stateManager.getState(null) == IFaultTolerantState.STATE_ACTIVE)
                    {
                        handleToWaiting(null);
                    }
                }
                else
                {
                    boolean aquiredLock = m_am.leaseLock();
                    if (aquiredLock)
                    {
                        if (m_stateManager.getState(null) == IFaultTolerantState.STATE_STANDBY)
                        {
                            m_am.releaseLock();
                            long backoffDuration = m_interval + m_am.getLeaseRequestTimeout();
                            synchronized(this)
                            {
                                // back off an appropriate amount of time to see if the active could still be active
                                wait(backoffDuration);
                            }
                            if (m_isClosing)
                            {
                                return;
                            }
                            aquiredLock = m_am.leaseLock();
                        }
                        if (aquiredLock)
                        {
                            handleToActive();
                        }
                    }
                    if (!aquiredLock)
                    {
                        handleToStandby(null);
                    }
                }
            }
            catch(RecoverableStateChangeException e)
            {
                if ((m_traceMask & (AgentManager.TRACE_FAULT_DETECTION)) > 0)
                 {
                    m_context.logMessage("Recoverable failure during fault tolerant state change, trace follows...", e, Level.TRACE);
                // wait again and see if the conditions to change state still exist
                }
            }
            catch(NonRecoverableStateChangeException e)
            {
                if ((m_traceMask & (AgentManager.TRACE_FAULT_DETECTION)) > 0)
                {
                    m_context.logMessage("Non-recoverable failure during fault tolerant state change, trace follows...", e, Level.TRACE);
                }
                return;
            }
            catch(Exception e)
            {
                if (m_stateManager.getState(null) != prePartnerCheckState)
                {
                    // then some other thread has caused a state change and we should take another attempt at testing
                    // what should be done (if anything)
                    continue;
                }

                try
                {
                    if (m_isClosing)
                    {
                        return;
                    }
                    handleToStandby(e);
                }
                catch(RecoverableStateChangeException sce)
                {
                    if ((m_traceMask & (AgentManager.TRACE_FAULT_DETECTION)) > 0)
                     {
                        m_context.logMessage("Recoverable failure during fault tolerant state change, trace follows...", e, Level.TRACE);
                    // wait again and see if the conditions to change state still exist
                    }
                }
                catch(NonRecoverableStateChangeException sce)
                {
                    if ((m_traceMask & (AgentManager.TRACE_FAULT_DETECTION)) > 0)
                    {
                        m_context.logMessage("Non-recoverable failure during fault tolerant state change, trace follows...", sce, Level.TRACE);
                    }
                    return;
                }
            }
            if (!m_isClosing)
            {
                synchronized (this)
                {
                    try { wait(m_isSuspended ? 500 : m_interval); } catch (InterruptedException ex) { }
                }
            }
        }
    }

    void setTraceMask(int traceMask)
    {
        m_traceMask = traceMask;
    }

    void setTestInterval(long interval)
    {
        if (interval < 5000)
        {
            throw new IllegalArgumentException("Interval cannot be < 5 seconds");
        }
        if (interval > 60000)
        {
            throw new IllegalArgumentException("Interval cannot be > 60 seconds");
        }

        m_interval = interval;
    }

    synchronized void close()
    {
        m_isClosing = true;
        notifyAll();
    }

    synchronized void suspendActiveState()
    {
        m_isSuspended = true;
        notifyAll();
    }

    synchronized void resumeAfterSuspend()
    {
        m_isSuspended = false;
        notifyAll();
    }

    private synchronized void handleToActive()
    throws NonRecoverableStateChangeException, RecoverableStateChangeException
    {
        if ((m_traceMask & (m_am.TRACE_FAULT_DETECTION | IComponent.TRACE_VERBOSE)) > 0)
        {
            m_context.logMessage("Fault tolerant AgentManager successfully acquired lock from DS", Level.TRACE);
        }
        while (!m_context.getContainer().isClosing())
        {
            short currentState = m_stateManager.getState(null);
            if (!validateCurrentState(currentState))
            {
                return;
            }

            if (currentState == IFaultTolerantState.STATE_ACTIVE)
            {
                return;
            }

            if (currentState == IFaultTolerantState.STATE_WAITING)
            {
                if (m_stateManager.requestStateChange(currentState, IFaultTolerantState.STATE_ACTIVE, null))
                {
                    return;
                }
            }

            if (currentState == IFaultTolerantState.STATE_STANDBY)
            {
                if (m_am.getAllowFailover().booleanValue())
                {
                    if (m_failoverConditionLogged)
                    {
                        m_context.logMessage("Failover condition detected, transition to state: \"" +
                                             IFaultTolerantState.STATE_TEXT[IFaultTolerantState.STATE_ACTIVE] +
                                             "\" is now enabled", Level.INFO);
                        m_failoverConditionLogged = false;
                    }
                    if (m_stateManager.requestStateChange(currentState, IFaultTolerantState.STATE_ACTIVE, null))
                    {
                        return;
                    }
                }
                else
                {
                    if (!m_failoverConditionLogged)
                    {
                        m_context.logMessage("Failover condition detected, however transition to state: \"" +
                                             IFaultTolerantState.STATE_TEXT[IFaultTolerantState.STATE_ACTIVE] +
                                             "\" is not currently enabled", Level.WARNING);
                        m_failoverConditionLogged = true;
                    }
                    return;
                }
            }
        }
    }

    private synchronized void handleToStandby(Exception e)
    throws NonRecoverableStateChangeException, RecoverableStateChangeException
    {
        if ((m_traceMask & (m_am.TRACE_FAULT_DETECTION | IComponent.TRACE_VERBOSE)) > 0)
        {
            if (e == null)
            {
                m_context.logMessage("Fault tolerant AgentManager denied lock from DS", Level.TRACE);
            }
            else
            {
                m_context.logMessage("Fault tolerant AgentManager denied lock from DS, due to...", e, Level.TRACE);
            }
        }

        while (!m_context.getContainer().isClosing())
        {
            short currentState = m_stateManager.getState(null);
            if (!validateCurrentState(currentState))
            {
                return;
            }

            switch (currentState)
            {
                case (IFaultTolerantState.STATE_STANDBY):
                {
                    if (m_failoverConditionLogged)
                    {
                        m_context.logMessage("Failover condition no longer detected", Level.INFO);
                        m_failoverConditionLogged = false;
                    }
                    return;
                }
                case (IFaultTolerantState.STATE_ACTIVE):
                {
                    m_context.logMessage("An \"Active\" Agent Manager appears to be running elsewhere", Level.WARNING);
                    break;
                }
            }
            if (m_stateManager.requestStateChange(currentState, IFaultTolerantState.STATE_STANDBY, null))
            {
                return;
            }
        }
    }

    private synchronized void handleToWaiting(Exception e)
    throws NonRecoverableStateChangeException, RecoverableStateChangeException
    {
        if ((m_traceMask & m_am.TRACE_FAULT_DETECTION) > 0 && !m_isSuspended)
        {
            if (e == null)
            {
                m_context.logMessage("Fault tolerant AgentManager failed to acquire lock from DS", Level.TRACE);
            }
            else
            {
                m_context.logMessage("Fault tolerant AgentManager failed to acquire lock from DS, due to...", e, Level.TRACE);
            }
        }

        while (!m_context.getContainer().isClosing())
        {
            short currentState = m_stateManager.getState(null);
            if (!validateCurrentState(currentState))
            {
                return;
            }

            if (currentState == IFaultTolerantState.STATE_WAITING)
            {
                return;
            }

            if (m_stateManager.requestStateChange(currentState, IFaultTolerantState.STATE_WAITING, null))
            {
                return;
            }
        }
    }

    private boolean validateCurrentState(short currentState)
    {
        switch(currentState)
        {
            case IFaultTolerantState.STATE_STANDBY:
            case IFaultTolerantState.STATE_ACTIVE:
            case IFaultTolerantState.STATE_WAITING:
            {
                return true;
            }
            default:
            {
                m_context.logMessage("Unhandled fault tolerant state: " + currentState, Level.WARNING);
                return false;
            }
        }
    }

 }