// Copyright (c) 2010 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.mf.framework.monitor;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.management.MBeanInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;

import com.sonicsw.mx.util.IEmptyArray;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaAttributeList;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.metrics.IMetric;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.IMetricsData;
import com.sonicsw.mf.common.metrics.IValueType;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.common.runtime.ICanonicalName;
import com.sonicsw.mf.common.runtime.ICollectiveOpStatus;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.INotificationHandler;
import com.sonicsw.mf.framework.monitor.offload.AnalyticsOffloadException;
import com.sonicsw.mf.framework.monitor.storage.StorageException;
import com.sonicsw.mf.jmx.client.MFNotification;
import com.sonicsw.mf.mgmtapi.config.constants.IComponentCollectionConstants;

/**
 * The CollectionMonitor is responsible for performing the monitoring on
 * an individual collection. The CollectionsMonitor component creates a
 * CollectionMonitor for each collection it is configured to monitor.
 */
final class CollectionMonitor
implements Runnable
{
    private String m_collectionConfigID;
    private IFrameworkComponentContext m_frameworkContext;
    private CollectionsMonitor m_parent;

    private long m_metricsRefreshInterval;
    private long m_lastRequestedPollTime;

    private HashSet m_components = new HashSet();
    private HashSet m_slowComponents = new HashSet();
    private HashSet m_forwardNotifications = new HashSet();
    private HashMap m_notificationMonitors = new HashMap();
    private HashSet m_monitorMetrics = new HashSet();

    private BrokerStatusCollector m_brStatusCollector;

    // Broker Components to be monitored
    private HashSet m_brokerComponents = new HashSet();
    private HashSet m_activeComponents = new HashSet();
    private HashSet m_inactiveComponents = new HashSet();

    // Each collection monitor will hold its own complete set of notifications and metric meta data for
    // the components it hosts
    private Hashtable m_metricInfos = new Hashtable();
    private Hashtable m_notificationInfos = new Hashtable();
    private Vector m_componentsWithoutInfos = new Vector(); // a list of components that we still have to get infos for
    private InfoGetter m_infoGetterThread;

    private HashSet m_unknownNotificationInfos = new HashSet();
    private HashMap m_forwardNotificationInfos = new HashMap();
    private HashSet m_unknownMetricInfos = new HashSet();
    private HashMap m_monitoredMetricInfos = new HashMap();
    private HashMap m_metricInstances = new HashMap();
	private boolean m_enterpriseMessageRequiredLogged = false;

    public static final String DEFAULT_DESCRIPTION = "<unknown>";


    /**
     * Numeric constant for replication state
     */
    public static final int WAITING = 1 ;
    public static final int STANDALONE = 2 ;
    public static final int ACTIVE = 3 ;
    public static final int STANDBY = 4 ;
    public static final int ACTIVE_SYNC = 5 ;
    public static final int STANDBY_SYNC = 6 ;

    public static final String MQ_BROKER = "/mq/brokers/";
    public static final String MQ_BACKUPBROKER = "/mq/backupbrokers/";

    public static final  int METRICS_DAMP_FACTOR = 2;

    // notifications related
    public static final String THRESHOLD_NOTIFICATION_TYPE = "Threshold";

    public static final long GET_METRICS_TIMEOUT = 30000;
    public static final long SLOW_POLL_INTERVAL = 60000;
    public static final int MIN_METRICS_REFRESH_INTERVAL = 5;
    public static final Integer DEFAULT_METRICS_REFRESH_INTERVAL = new Integer(60);

    private boolean m_isClosing = false;
    private SlowCollector m_slowCollector;
	private long m_lastRequestedSlowPollTime;
	private long m_lastReqStatusPollTime;
	private long m_metricsSlowRefreshInterval = SLOW_POLL_INTERVAL;

    private static final IMetricInfo[] EMPTY_METRIC_INFO_ARRAY = new IMetricInfo[0];
    private static final MBeanNotificationInfo[] EMPTY_NOTIFICATION_INFO_ARRAY = new MBeanNotificationInfo[0];

    private static final String[] GET_METRICS_DATA_SIGNATURE = new String[]{ IMetricIdentity[].class.getName(), Boolean.class.getName() };
    private static final String[] ENABLE_METRICS_SIGNATURE = new String[]{IMetricIdentity[].class.getName() };

    CollectionMonitor(String configID, IFrameworkComponentContext frameworkContext, CollectionsMonitor parent)
    {
        m_collectionConfigID = configID;
        m_frameworkContext = frameworkContext;

        m_parent = parent;

        // Get the configuration from the configuration element
        IElement collectionElement = frameworkContext.getConfiguration(configID, true);
        IAttributeSet collectionAttributes = collectionElement.getAttributes();

        // get the list of notifications to forward
        IAttributeList forwardNotifications = (IAttributeList)collectionAttributes.getAttribute(IComponentCollectionConstants.FORWARD_NOTIFICATIONS_ATTR);
        if (forwardNotifications != null)
        {
            for (int i = forwardNotifications.getCount() - 1; i >= 0; i--)
            {
                addForwardNotification((String)forwardNotifications.getItem(i));
            }
        }

        // get the list of components and add it to the list of components we will handle
        IAttributeSet componentSet = (IAttributeSet)collectionAttributes.getAttribute(IComponentCollectionConstants.COMPONENTS_ATTR);
        if (componentSet != null)
        {
            Iterator iterator = componentSet.getAttributes().values().iterator();
            while (iterator.hasNext())
            {
                IAttributeSet componentAttrs = (IAttributeSet)iterator.next();
                String compName = (String)componentAttrs.getAttribute(IComponentCollectionConstants.COMPONENT_RUNTIME_ID_ATTR);
                String compType = ((Reference)componentAttrs.getAttribute(IComponentCollectionConstants.CONFIG_REF_ATTR)).getElementName();
                // add the component to broker set if it contains either broker/backupbroker
                if (compType.contains(MQ_BROKER) || compType.contains(MQ_BACKUPBROKER) ){
                    m_brokerComponents.add(compName);
                }
                addComponent(compName);
            }
        }

        // get the set of notification monitors
        IAttributeSet notificationMonitorSet = (IAttributeSet)collectionAttributes.getAttribute(IComponentCollectionConstants.NOTIFICATION_MONITORS_ATTR);
        if (notificationMonitorSet != null)
        {
            HashMap notificationMonitors = (HashMap)notificationMonitorSet.getAttributes();
            Iterator iterator = notificationMonitors.keySet().iterator();
            while (iterator.hasNext())
            {
                String name = (String)iterator.next();
                addNotificationMonitor(name, (IAttributeSet)notificationMonitors.get(name));
            }
        }

        // get the metrics refresh interval
        setMetricsRefreshInterval((Integer)collectionAttributes.getAttribute(IComponentCollectionConstants.METRICS_REFRESH_INTERVAL_SECONDS_ATTR));

        // get the set of metrics to monitor
        handleChangeMonitorMetricsAttr();

        // create the slow collector
        m_slowCollector = new SlowCollector();

        // create Broker Status Collector
        m_brStatusCollector = new BrokerStatusCollector();
        synchronized(this)
        {
            // if there are metrics to monitor then kick off a thread to do it
            scheduleBrokerStatePoll();
            scheduleMetricsPoll();
            scheduleMetricsSlowPoll();
         }
    }

    private synchronized void setMetricsRefreshInterval(Integer interval)
    {
        if (interval == null)
        {
            interval = DEFAULT_METRICS_REFRESH_INTERVAL;
        }

        if (interval.intValue() < MIN_METRICS_REFRESH_INTERVAL)
        {
            m_frameworkContext.logMessage("Invalid metrics refresh interval [" + interval +
                "] for collection [" + m_collectionConfigID + "], defaulting to " +
                DEFAULT_METRICS_REFRESH_INTERVAL + " seconds", Level.INFO);
            interval = DEFAULT_METRICS_REFRESH_INTERVAL;
        }

        m_metricsRefreshInterval = interval.intValue() * 1000;
    }

    private void scheduleMetricsPoll()
    {
        // if there are metrics to monitor then kick off a thread to do it
        synchronized(this)
        {
            if (m_monitorMetrics.isEmpty())
            {
                return;
            }
        }

        if (m_lastRequestedPollTime > 0) // i.e. we've done at least one poll
        {
            // we may have to catch up, by missing intervals
            do
            {
                m_lastRequestedPollTime += m_metricsRefreshInterval;
            }
            while(m_lastRequestedPollTime <= System.currentTimeMillis());
        }
        else
        {
            m_lastRequestedPollTime = System.currentTimeMillis();
        }

        m_frameworkContext.scheduleTask(this, new Date(m_lastRequestedPollTime));
    }

    private void scheduleBrokerStatePoll()
    { 
        // if there are metrics to monitor then kick off a thread to do it
        synchronized (this) {
            if (m_monitorMetrics.isEmpty()) {
                return;
            }
		}
        // i.e. we've done at least one poll
        if ( m_lastReqStatusPollTime > 0 )
        {
            // we may have to catch up, by missing intervals
            do {
               m_lastReqStatusPollTime += ( m_metricsSlowRefreshInterval * METRICS_DAMP_FACTOR );
            } while (m_lastReqStatusPollTime <= System.currentTimeMillis());
        } else {
           m_lastReqStatusPollTime = System.currentTimeMillis();
        }
        m_frameworkContext.scheduleTask(this.m_brStatusCollector , new Date(m_lastReqStatusPollTime));
    }

    private void scheduleMetricsSlowPoll()
    {
		// if there are metrics to monitor then kick off a thread to do it
		synchronized (this) {
			if (m_monitorMetrics.isEmpty()) {
				return;
			}
		}
		if (m_lastRequestedSlowPollTime > 0) // i.e. we've done at least one poll
		{
			// we may have to catch up, by missing intervals
			do {
				m_lastRequestedSlowPollTime += m_metricsSlowRefreshInterval;
			} while (m_lastRequestedSlowPollTime <= System.currentTimeMillis());
		} else {
			m_lastRequestedSlowPollTime = System.currentTimeMillis();
		}
		m_frameworkContext
				.scheduleTask(this.m_slowCollector, new Date(m_lastRequestedSlowPollTime));
    }
    
    /**
     * The monitor is no longer required and this method is called to
     * release any resources used by this monitor.
     */
    synchronized void cleanup()
    {
        m_isClosing = true;
        m_frameworkContext.cancelTask(this);
        m_frameworkContext.cancelTask(m_slowCollector);

        // unsubscribe the forwarded notification subscriptions
        String[] components = (String[])m_components.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        String[] notifications = (String[])m_forwardNotifications.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        for (int i = 0; i < components.length; i++)
        {
            m_parent.m_notificationManager.removeNotificationHandler(components[i], notifications, m_parent.m_notificationForwarder);
        }

        // loop through all the notification monitors cleaning them up
        String[] mons = (String[])m_notificationMonitors.keySet().toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        for (int i = 0; i < mons.length; i++)
        {
            /*
             *  cannot use an iterator to iterate through the monitors, 
             *  as "removeNotificationMonitor" structurally modified the underlying HashMap - 
             *  thus using an iterator could result in a "ConcurrentModificationException"...
             */
            removeNotificationMonitor(mons[i]);
        }

        m_components.clear();
        m_brokerComponents.clear();
        m_activeComponents.clear();
        m_inactiveComponents.clear();
        m_slowComponents.clear();
        m_forwardNotifications.clear();
        m_notificationMonitors.clear();
    }

    /**
     * Gets the list of component identities (canonical form) that this collection monitor
     * is handling.
     */
    synchronized String[] getCollectionComponents()
    {
        return (String[])m_components.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
    }
    
    /**
     * Gets the list of component identities (canonical form) that this collection monitor
     * is handling.
     */
    synchronized String[] getFastCollectionComponents()
    {
        Set fast = (Set)m_components.clone();
        fast.removeAll(m_slowComponents);
        fast.removeAll(m_inactiveComponents);
        return (String[])fast.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
    }

    /**
     * Gets the list of component identities (canonical form) that this collection monitor
     * is handling.
     */
    synchronized String[] getSlowCollectionComponents()
    {
        Set slow = (Set)m_slowComponents.clone();
        slow.removeAll(m_inactiveComponents);
        return (String[])slow.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
    }

    /**
     * Gets the meta-data for the metrics being monitored by this component.
     */
    synchronized IMetricInfo[] getMonitoredMetricsInfo()
    {
        // ensure the info set is up to date
        updateMonitoredMetricInfosAndInstances();

        return (IMetricInfo[])m_monitoredMetricInfos.values().toArray(EMPTY_METRIC_INFO_ARRAY);
    }

    /**
     * Gets the names of instances of the given parent identity that are being
     * monitored by this monitor.
     */
    String[] getInstanceMetricNames(IMetricIdentity id)
    {
        // ensure the instance name sets are up to date
        updateMonitoredMetricInfosAndInstances();

        HashSet set = (HashSet)m_metricInstances.get(id.getName());
        if (set == null || set.isEmpty())
        {
            return IEmptyArray.EMPTY_STRING_ARRAY;
        }

        return (String[])set.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
    }

    private synchronized void updateMonitoredMetricInfosAndInstances()
    {
        // loop through adding infos for those we have
        IMetricIdentity[] unknowns = (IMetricIdentity[])m_unknownMetricInfos.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
        for (int i = 0; i < unknowns.length; i++)
        {
            String unknownName = unknowns[i].getName();

            // find the info object for this identity
            Iterator infos = m_metricInfos.values().iterator();
            while (infos.hasNext())
            {
                IMetricInfo info = (IMetricInfo)infos.next();
                IMetricIdentity id = info.getMetricIdentity();
                String name = id.getName();
                if (unknownName.equals(name))
                {
                    m_monitoredMetricInfos.put(name, info);
                    m_unknownMetricInfos.remove(unknowns[i]);
                    break;
                }
                else // is this an instance of ?
                {
                    if (unknowns[i].isInstanceOf(id))
                    {
                        m_monitoredMetricInfos.put(name, info);
                        String[] nameComponents = unknowns[i].getNameComponents();
                        HashSet instanceNames = (HashSet)m_metricInstances.get(name);
                        if (instanceNames == null)
                        {
                            instanceNames = new HashSet();
                            m_metricInstances.put(name, instanceNames);
                        }
                        instanceNames.add(nameComponents[nameComponents.length - 1]);
                        m_unknownMetricInfos.remove(unknowns[i]);
                        // try to remove from m_monitoredMetricInfos in case a dummy entry was added before
                        // we knew it was an instance metric
                        m_monitoredMetricInfos.remove(unknownName);
                        break;
                    }
                }
            }
            // if its still unknown and there is no dummy entry in the infos table
            // then create a dummy entry
            if (m_unknownMetricInfos.contains(unknowns[i]) && !m_monitoredMetricInfos.containsKey(unknownName))
            {
                IMetricInfo dummy = MetricsFactory.createMetricInfo(unknowns[i], IValueType.VALUE, DEFAULT_DESCRIPTION, null, false, false);
                m_monitoredMetricInfos.put(unknownName, dummy);
            }
        }
    }

    synchronized MBeanNotificationInfo[] getForwardedNotificationsInfo()
    {
        String[] unknownNotificationInfos = (String[])m_unknownNotificationInfos.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        for (int i = 0; i < unknownNotificationInfos.length; i++)
        {
            MBeanNotificationInfo info = (MBeanNotificationInfo)m_notificationInfos.get(unknownNotificationInfos[i]);
            if (info != null)
            {
                m_forwardNotificationInfos.put(unknownNotificationInfos[i], info);
                m_unknownNotificationInfos.remove(unknownNotificationInfos[i]);
            }
        }

        return (MBeanNotificationInfo[])m_forwardNotificationInfos.values().toArray(EMPTY_NOTIFICATION_INFO_ARRAY);
    }

    /**
     * Get the MBean info from the given component
     */
    private void getComponentInfos()
    {
        if (m_componentsWithoutInfos.isEmpty())
        {
            return;
        }

        if (m_infoGetterThread == null)
        {
            m_infoGetterThread = new InfoGetter();
            m_infoGetterThread.start();
        }
        else
        {
            synchronized(m_infoGetterThread.getInforGetterLockObj())
            {
                m_infoGetterThread.getInforGetterLockObj().notifyAll();
            }
        }
    }
    private static class InfoGetterFailure extends Exception
    {
        private static final long serialVersionUID = 1L;
        public InfoGetterFailure(String message)
        {
            super(message);
        }
    }

    private class InfoGetter
    extends Thread
    {
        long retryWait = 4000;
        Random random = new Random();
        Object infoGetterLockObj = new  Object();
        
        private InfoGetter()
        {
            super("CM InfoGetter");
        }

        @Override
        public void run()
        {
            while (!interrupted() && !CollectionMonitor.this.m_isClosing)
            {
                String componentID = null;
                synchronized(CollectionMonitor.this)
                {
                    // if there are no components that we are trying to get info for, then end the thread
                    if (m_componentsWithoutInfos.isEmpty())
                    {
                        m_infoGetterThread = null;
                        return;
                    }

                    // randomly get a component ID from those that are left
                    int posn = 0;
                    try { posn = Math.abs(random.nextInt() % m_componentsWithoutInfos.size()); } catch(Exception e) { }
                    componentID = (String)m_componentsWithoutInfos.get(posn);

                    if (componentID == null)
                    {
                        m_componentsWithoutInfos.remove(posn);
                        continue;
                    }
                }

                // get the meta data for the component
                try
                {
                    getComponentManagementInfo(componentID);
                    m_componentsWithoutInfos.remove(componentID);

                    // If there are still more components we require info for, then loop again immediately.
                    if (!m_componentsWithoutInfos.isEmpty())
                    {
                        continue;
                    }
                }
                catch(InvokeTimeoutException e)
                {
                    // Retry without logging a warning
                }
                catch (InfoGetterFailure e)
                {
                    logWarning(componentID, e.getMessage(), null);
                }
                catch(Exception e)
                {
                    logWarning(componentID, null, e);
                }

                // Failed to get component information or there are no more components left. If there are others we can
                // try, then retry immediately (however since we are randomly picking which component to try it does
                // not ensure a different component will be checked, also we could failed/timedout on more than one
                // component).
                if (m_componentsWithoutInfos.size() > 1)
                {
                    continue;
                }

                synchronized(infoGetterLockObj)
                {
                    try
                    {
                        infoGetterLockObj.wait(retryWait);
                    }
                    catch(InterruptedException e)
                    {
                        m_infoGetterThread = null;
                        return;
                    }
                }
                if (retryWait < 256000)
                {
                    retryWait += retryWait;
                }
            }
        }

        private void logWarning(String componentID, String error, Exception e)
        {
            // only log if its a different container or if its this container and the container has booted
            ICanonicalName cmName = m_frameworkContext.getComponentName();
            ICanonicalName componentName = new CanonicalName(componentID);
            if (!(cmName.getContainerName().equals(componentName.getContainerName())) || m_frameworkContext.getContainer().isBooted())
            {
                boolean traceRequestFailures = (m_parent.m_traceMask & CollectionsMonitor.TRACE_REQUEST_FAILURES) > 0;

                StringBuilder sb = new StringBuilder();
                sb.append("Failed to obtain management information for component [").append(componentID).append("]");
                if (traceRequestFailures && error != null)
                {
                    sb.append(": ").append(error);
                }
                sb.append(", retry will be initiated later");

                if (traceRequestFailures && e != null)
                {
                    m_frameworkContext.logMessage(sb.toString(), e, Level.WARNING);
                }
                else
                {
                    m_frameworkContext.logMessage(sb.toString(), Level.WARNING);
                }
            }
        }

        private IMetricInfo[] getMetricInfo(String componentID, MBeanOperationInfo[] operationInfos)
            throws Exception
        {
            for (MBeanOperationInfo info : operationInfos) // check to see if the component has metrics
            {
                if ("getMetricsInfo".equals(info.getName()))
                {
                    IMetricInfo[] metricInfos =
                        (IMetricInfo[])m_frameworkContext.invoke(componentID, "getMetricsInfo",
                                                                 IEmptyArray.EMPTY_OBJECT_ARRAY,
                                                                 IEmptyArray.EMPTY_STRING_ARRAY, true, 0);
                    if (metricInfos == null)
                    {
                        throw new InfoGetterFailure("metrics information not available");
                    }
                    return metricInfos;
                }
            }
            return EMPTY_METRIC_INFO_ARRAY;
        }

        private void getComponentManagementInfo(String componentID) throws Exception
        {
            MBeanInfo mBeanInfo = (MBeanInfo)m_frameworkContext.invoke(componentID, "getMBeanInfo",
                                                                       IEmptyArray.EMPTY_OBJECT_ARRAY,
                                                                       IEmptyArray.EMPTY_STRING_ARRAY, true, 0);
            if (mBeanInfo == null)
            {
                throw new InfoGetterFailure("management bean information not available");
            }

            MBeanNotificationInfo[] notificationInfos = mBeanInfo.getNotifications();
            IMetricInfo[] metricInfos = getMetricInfo(componentID, mBeanInfo.getOperations());

            for (MBeanNotificationInfo info : notificationInfos)
            {
                String[] notificationTypes = info.getNotifTypes();
                StringBuffer sb = new StringBuffer(notificationTypes[0]);
                if (notificationTypes.length > 1)
                {
                    for (int j = 1; j < notificationTypes.length; j++)
                    {
                        sb.append('.').append(notificationTypes[j]);
                    }
                }
                m_notificationInfos.put(sb.toString(), info);
            }

            for (IMetricInfo info : metricInfos)
            {
                m_metricInfos.put(info.getMetricIdentity(), info);
            }
        }
        
        public Object getInforGetterLockObj() {
            return infoGetterLockObj;
        }
    }

    @Override
    public void run()
    {
        if (m_isClosing)
        {
            return;
        }

        // get the list of components and metrics we're after
        String[] components = getFastCollectionComponents();
        if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_METRIC_POLLING) > 0)
        {
            m_frameworkContext.logMessage("Starting metrics poll on collection [" + m_collectionConfigID + "] for [" + components.length + "] components...", Level.TRACE);
        }

        IMetricIdentity[] metrics = null;
        synchronized(this)
        {
            metrics = (IMetricIdentity[])m_monitorMetrics.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
        }

        ICollectiveOpStatus results = null;
        try
        {
            if ((components != null) && (components.length > 0))
            {
                results = (ICollectiveOpStatus)m_frameworkContext.invoke(components, "getMetricsData", new Object[] { metrics, Boolean.FALSE }, GET_METRICS_DATA_SIGNATURE, true, GET_METRICS_TIMEOUT);
            }
        }
        catch(Exception e)
        {
            m_frameworkContext.logMessage("Failed to poll for metrics on collection [" + m_collectionConfigID + ']', Level.WARNING);
        }

        // go through each of the individual results
        if (results != null)
        {
            for (int i = results.getCount() - 1; i >= 0; i--)
            {
                String component = results.getComponentName(i);
                if (results.operationIsSuccessful(i))
                {
                    // save any results and enable any metrics that were not provided in the results
                    IMetricsData resultMetrics = (IMetricsData) results.getReturnValue(i);
                    if ((resultMetrics != null) && (component != null))
                    {
                        HashSet missingMetrics = storeMetrics(component, resultMetrics);

                        if (!missingMetrics.isEmpty())
                        {
                            enableMetrics(component, missingMetrics);
                        }
                    }
                }
                else
                {
                    Throwable throwable = results.getThrowable(i);
                    boolean relegate = false;
                    if (throwable instanceof InvokeTimeoutException)
                    {
                        if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_METRIC_POLL_TIMEOUTS) > 0)
                        {
                            m_frameworkContext.logMessage("Timeout of poll for metrics on component [" + component + ']', Level.TRACE);
                        }
                        relegate = true;
                    }
                    else if (throwable instanceof NoSuchMethodException)
                    {
                        // ignore it unless tracing is on
                        if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_REQUEST_FAILURES) > 0)
                        {
                            m_frameworkContext.logMessage("Failed to poll for metrics on component [" + component + "]: component does not provide any metrics", Level.TRACE);
                        }
                        relegate = true;
                    }
                    else
                    {
                        // only log if its a different container or if its this container and the container has booted
                        ICanonicalName cmName = m_frameworkContext.getComponentName();
                        ICanonicalName componentName = new CanonicalName(component);
                        if (!(cmName.getContainerName().equals(componentName.getContainerName())) || m_frameworkContext.getContainer().isBooted())
                        {
                            m_frameworkContext.logMessage("Failed to poll for metrics on component [" + component + "]", Level.WARNING);
                            if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_REQUEST_FAILURES) > 0)
                            {
                                m_frameworkContext.logMessage("...trace follows...", throwable, Level.TRACE);
                            }
                        }
                        relegate = true;
                    }
                    if (relegate) {
                    	relegateFastComponent(component);
                    }
                }
            }
        }
        if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_METRIC_POLLING) > 0)
        {
            m_frameworkContext.logMessage("...finished metrics poll on collection [" + m_collectionConfigID + "] for [" + components.length + "] components...", Level.TRACE);
        }

        // schedule next poll
        if (!m_isClosing)
        {
            scheduleMetricsPoll();
        }
    }

    /**
     * Stores each of the returned metrics and return a list of metrics that were expected but not found in the results.
     */
    private HashSet storeMetrics(String component, IMetricsData metricsData)
    {
        HashSet missingMetrics = null;
        synchronized (this)
        {
            missingMetrics = (HashSet)m_monitorMetrics.clone();
        }
        IMetric[] metrics = metricsData.getMetrics();

        try
        {
            for (int i = 0; i < metrics.length; i++)
            {
                Iterator it = missingMetrics.iterator();
                ArrayList al = new ArrayList();
                while (it.hasNext())
                {
                    IMetricIdentity mid = (IMetricIdentity) it.next();
                    Pattern pattern = Pattern.compile(mid.getAbsoluteName());
                    Matcher matcher = pattern.matcher(metrics[i].getMetricIdentity().getAbsoluteName());
                    boolean match = matcher.matches();
                    if (match)
                     {
                        al.add(mid);  // build list of metric identities to remove from list of "missing" metrics
                    }
                }

                it = al.iterator();
                while (it.hasNext())
                 {
                    missingMetrics.remove(it.next());  // remove the metric identify from the list of "missing" metrics
                }
            }

            /** offload or store but not both! */
			if ((m_parent != null) && (m_parent.m_offloader != null)) {
				if (m_parent.isEnterprise()) {
					m_parent.m_offloader.offloadMetrics(metrics, component);
				} else if (!m_enterpriseMessageRequiredLogged) {
					m_frameworkContext.logMessage("Analytics offload is not supported and requires Enterprise Sonic Messaging", Level.SEVERE);
					m_enterpriseMessageRequiredLogged = true;
				}
			} else if ((m_parent != null) && (m_parent.m_store != null)) {
				m_parent.m_store.storeMetrics(metrics, component);
			}
         }
        catch(AnalyticsOffloadException e)
        {
            if (m_parent != null && (m_parent.m_traceMask & CollectionsMonitor.TRACE_OFFLOAD_FAILURES) > 0)
            {
                m_frameworkContext.logMessage("Failed to offload metrics, trace follows...", e, Level.TRACE);
            }
            else
            {
                m_frameworkContext.logMessage("Failed to offload metrics", Level.WARNING);
            }
        }        
        catch(StorageException e)
        {
            if (m_parent != null && (m_parent.m_traceMask & CollectionsMonitor.TRACE_STORAGE_FAILURES) > 0)
            {
                m_frameworkContext.logMessage("Failed to store metrics, trace follows...", e, Level.TRACE);
            }
            else
            {
                m_frameworkContext.logMessage("Failed to store metrics", Level.WARNING);
            }
        }

        return missingMetrics;
    }

    /**
     * Enables the given set of metrics on the given component.
     */
    private void enableMetrics(String component, HashSet enableMetrics)
    {
        try
        {
            IMetricIdentity[] metrics = (IMetricIdentity[])enableMetrics.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
            if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_METRIC_ENABLING) > 0)
            {
                StringBuffer msg = new StringBuffer();
                msg.append("Enabling metrics on component [" + component + "], metrics:");
                for (int i = 0; i < metrics.length; i++)
                {
                    msg.append(IContainer.NEWLINE).append('\t').append(metrics[i].getName());
                }
                m_frameworkContext.logMessage(msg.toString(), Level.TRACE);
            }
            m_frameworkContext.invoke(component, "enableMetrics", new Object[] { metrics }, ENABLE_METRICS_SIGNATURE, true, 0);
        }
        catch(Exception e)
        {
            if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_METRIC_ENABLING) > 0)
            {
                m_frameworkContext.logMessage("Failed to enable metrics on component [" + component + ", trace follows...", e, Level.WARNING);
            }
            else
            {
                m_frameworkContext.logMessage("Failed to enable metrics on component [" + component +']', Level.WARNING);
            }
        }
    }

    /**
     * Handle the configuration changes associated with this monitor's collection.
     */
    synchronized void handleElementChange(IElementChange elementChange)
    {
        // we are interested in changes to:
        //  - the list of members of the collection
        //  - the list of notifications to be forwarded
        //  - the list of metrics to be monitored
        //  - the frequency with which we should poll for monitored metrics
        //  - the list of notifications to be monitored
        //  - changes to the details of a particular notification monitors interval or thresholds

        String configID = elementChange.getElement().getIdentity().getName();

        if (!configID.equals(m_collectionConfigID))
        {
            return;
        }

        try
        {
            if (elementChange.getChangeType() == IElementChange.ELEMENT_UPDATED)
            {
                IDeltaElement changeElement = (IDeltaElement)elementChange.getElement();
                IDeltaAttributeSet attrs = (IDeltaAttributeSet)changeElement.getDeltaAttributes();

                handleNewCollectionAttrs(attrs.getNewAttributesNames(), attrs);
                handleModifiedCollectionAttrs(attrs.getModifiedAttributesNames(), attrs);
                handleDeletedCollectionAttrs(attrs.getDeletedAttributesNames());
            }
        }
        catch (Throwable e)
        {
            m_frameworkContext.logMessage("Error handling configuration change, trace follows...", e, Level.WARNING);
        }
    }

    private void handleNewCollectionAttrs(String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IComponentCollectionConstants.METRICS_REFRESH_INTERVAL_SECONDS_ATTR))
            {
                setMetricsRefreshInterval((Integer)attrs.getNewValue(IComponentCollectionConstants.METRICS_REFRESH_INTERVAL_SECONDS_ATTR));
            }
            if (names[i].equals(IComponentCollectionConstants.COMPONENTS_ATTR))
            {
                handleChangeComponentsAttr();
            }
            if (names[i].equals(IComponentCollectionConstants.FORWARD_NOTIFICATIONS_ATTR))
            {
                handleChangeForwardNotificationsAttr();
            }
            if (names[i].equals(IComponentCollectionConstants.NOTIFICATION_MONITORS_ATTR))
            {
                handleNewNotificationMonitorsAttrs((IAttributeSet)attrs.getNewValue(IComponentCollectionConstants.NOTIFICATION_MONITORS_ATTR));
            }
            if (names[i].equals(IComponentCollectionConstants.MONITOR_METRICS_ATTR))
            {
                handleChangeMonitorMetricsAttr();
            }
        }
    }

    private void handleModifiedCollectionAttrs(String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IComponentCollectionConstants.METRICS_REFRESH_INTERVAL_SECONDS_ATTR))
            {
                setMetricsRefreshInterval((Integer)attrs.getNewValue(IComponentCollectionConstants.METRICS_REFRESH_INTERVAL_SECONDS_ATTR));
            }
            if (names[i].equals(IComponentCollectionConstants.COMPONENTS_ATTR))
            {
                handleChangeComponentsAttr();
            }
            if (names[i].equals(IComponentCollectionConstants.FORWARD_NOTIFICATIONS_ATTR))
            {
                handleChangeForwardNotificationsAttr();
            }
            if (names[i].equals(IComponentCollectionConstants.NOTIFICATION_MONITORS_ATTR))
            {
                handleChangeNotificationMonitorsAttrs((IDeltaAttributeSet)attrs.getNewValue(IComponentCollectionConstants.NOTIFICATION_MONITORS_ATTR));
            }
            if (names[i].equals(IComponentCollectionConstants.MONITOR_METRICS_ATTR))
            {
                handleChangeMonitorMetricsAttr();
            }
        }
    }

    // Runtime attributes go back to their default value if the
    // corresponding configuration values are removed.

    private void handleDeletedCollectionAttrs(String[] names)
    {
        for (int i = 0; i < names.length; i++)
        {
            if (names[i].equals(IComponentCollectionConstants.METRICS_REFRESH_INTERVAL_SECONDS_ATTR))
            {
                setMetricsRefreshInterval(null);
            }
        }
    }

    private void handleChangeComponentsAttr()
    {
        // Get the configuration from the configuration element
        IElement collectionElement = m_frameworkContext.getConfiguration(m_collectionConfigID, true);
        IAttributeSet collectionAttrs = collectionElement.getAttributes();

        // get the list of components to be monitored and those already monitored
        ArrayList componentList = new ArrayList();
        IAttributeSet componentSet = (IAttributeSet)collectionAttrs.getAttribute(IComponentCollectionConstants.COMPONENTS_ATTR);

        Iterator iterator = componentSet.getAttributes().values().iterator();
        while (iterator.hasNext())
        {
            IAttributeSet componentAttrs = (IAttributeSet)iterator.next();

            String compName = (String)componentAttrs.getAttribute(IComponentCollectionConstants.COMPONENT_RUNTIME_ID_ATTR);
            String compType = ((Reference)componentAttrs.getAttribute(IComponentCollectionConstants.CONFIG_REF_ATTR)).getElementName();
            // add the component to broker set if it contains either broker/backupbroker
            if ( compType.contains(MQ_BROKER) || compType.contains(MQ_BACKUPBROKER) ){
                m_brokerComponents.add(compName);
			}
            componentList.add(compName);
        }

        String[] components = (String[])componentList.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        String[] monitoredComponents = (String[])m_components.toArray(IEmptyArray.EMPTY_STRING_ARRAY);

        // evaluate the list of components to be removed
        ArrayList deletedComponents = new ArrayList();
        outerLoop:
        for (int i = 0; i < monitoredComponents.length; i++)
        {
            for (int j = 0; j < components.length; j++)
            {
                if (components[j].equals(monitoredComponents[i]))
                {
                    continue outerLoop;
                }
            }
            deletedComponents.add(monitoredComponents[i]);
        }

        // evaluate the list of components to be added
        ArrayList newComponents = new ArrayList();
        outerLoop:
        for (int i = 0; i < components.length; i++)
        {
            for (int j = 0; j < monitoredComponents.length; j++)
            {
                if (components[i].equals(monitoredComponents[j]))
                {
                    continue outerLoop;
                }
            }
            newComponents.add(components[i]);
        }

        String[] componentIDs = new String[1];

        // loop through removing any deleted components
        for (int i = deletedComponents.size() - 1; i >= 0; i--)
        {
            componentIDs[0] = (String)deletedComponents.get(i);
            // removes the broker if present in the HashSet
            m_brokerComponents.remove(componentIDs[0]);
            removeComponent(componentIDs[0]);
            m_parent.m_notificationManager.unsubscribe(componentIDs);
        }

        // loop through adding new components
        for (int i = newComponents.size() - 1; i >= 0; i--)
        {
            componentIDs[0] = (String)newComponents.get(i);
            addComponent(componentIDs[0]);
            m_parent.m_notificationManager.subscribe(componentIDs);
        }
    }

    private synchronized void addComponent(final String componentID)
    {
        m_components.add(componentID);

        m_componentsWithoutInfos.add(componentID);

        // update notification forwarding
        String[] notifications = (String[])m_forwardNotifications.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        m_parent.m_notificationManager.addNotificationHandler(componentID, notifications, m_parent.m_notificationForwarder);

            // update notification monitors
        Iterator iterator = m_notificationMonitors.values().iterator();
        while (iterator.hasNext())
        {
            NotificationMonitor monitor = (NotificationMonitor)iterator.next();
            monitor.addComponent(componentID);
        }

        // kick off the thread that gets component infos if its not already reading
        getComponentInfos();

        // if metrics are being monitored then ensure they are enabled at the component
        if (!m_monitorMetrics.isEmpty())
        {
            enableMetrics(componentID, m_monitorMetrics);
        }
    }

    private synchronized void removeComponent(String componentID)
    {
        m_components.remove(componentID);
        m_slowComponents.remove(componentID);
        m_componentsWithoutInfos.remove(componentID);

        // update notification forwarding
        String[] notifications = (String[])m_forwardNotifications.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        m_parent.m_notificationManager.removeNotificationHandler(componentID, notifications, m_parent.m_notificationForwarder);

        // update notification monitors
        Iterator iterator = m_notificationMonitors.values().iterator();
        while (iterator.hasNext())
        {
            NotificationMonitor monitor = (NotificationMonitor)iterator.next();
            monitor.removeComponent(componentID);
        }
    }

    private void handleChangeForwardNotificationsAttr()
    {
        // Get the configuration from the configuration element
        IElement collectionElement = m_frameworkContext.getConfiguration(m_collectionConfigID, true);
        IAttributeSet collectionAttrs = collectionElement.getAttributes();

        // get the list of notifications to be forwarded and those already forwarded
        IAttributeList notificationList = (IAttributeList)collectionAttrs.getAttribute(IComponentCollectionConstants.FORWARD_NOTIFICATIONS_ATTR);
        String[] notifications = (String[])notificationList.getItems().toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        String[] forwardedNotifications = null;
        synchronized(this)
        {
            forwardedNotifications = (String[])m_forwardNotifications.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        }

        // evaluate the list of notifications to be removed
        ArrayList deletedNotifications = new ArrayList();
        outerLoop:
        for (int i = 0; i < forwardedNotifications.length; i++)
        {
            for (int j = 0; j < notifications.length; j++)
            {
                if (notifications[j].equals(forwardedNotifications[i]))
                {
                    continue outerLoop;
                }
            }
            deletedNotifications.add(forwardedNotifications[i]);
        }

        // evaluate the list of notifications to be added
        ArrayList newNotifications = new ArrayList();
        outerLoop:
        for (int i = 0; i < notifications.length; i++)
        {
            for (int j = 0; j < forwardedNotifications.length; j++)
            {
                if (notifications[i].equals(forwardedNotifications[j]))
                {
                    continue outerLoop;
                }
            }
            newNotifications.add(notifications[i]);
        }

        String[] componentIDs = getCollectionComponents();

        // loop through removing any deleted notifications
        for (int i = deletedNotifications.size() - 1; i >= 0; i--)
        {
            removeForwardNotification((String)deletedNotifications.get(i));
            // need to ensure the subscription to the correct set of notifications in place
            m_parent.m_notificationManager.subscribe(componentIDs);
        }

        // loop through adding new notifications
        for (int i = newNotifications.size() - 1; i >= 0; i--)
        {
            addForwardNotification((String)newNotifications.get(i));
            // need to ensure the subscription to the correct set of notifications in place
            m_parent.m_notificationManager.subscribe(componentIDs);
        }
    }

    private synchronized void addForwardNotification(String notificationType)
    {
        m_forwardNotifications.add(notificationType);

        // create a default entry for the notification until we can actually get the info
        StringTokenizer st = new StringTokenizer(notificationType, ".");
        // we use "\t" as the delimiter on the last token to ensure we gets whats left in the tokenizer
        // ("\t" is not allowed in notification names and there is a possibility the the remainder could
        // include "." chars)
        String[] notifTypes = new String[] { st.nextToken(), st.nextToken(), st.nextToken("\t").substring(1) };
        MBeanNotificationInfo info = new MBeanNotificationInfo(notifTypes, MFNotification.CLASSNAME, DEFAULT_DESCRIPTION);
        m_forwardNotificationInfos.put(notificationType, info);
        // record the fact that we need the info
        m_unknownNotificationInfos.add(notificationType);

        // update notification forwarding
        String[] componentIDs = (String[])m_components.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        for (int i = 0; i < componentIDs.length; i++)
        {
            m_parent.m_notificationManager.addNotificationHandler(componentIDs[i], new String[]{ notificationType }, m_parent.m_notificationForwarder);
        }
    }

    private synchronized void removeForwardNotification(String notificationType)
    {
        m_forwardNotifications.remove(notificationType);
        m_forwardNotificationInfos.remove(notificationType);
        m_unknownNotificationInfos.remove(notificationType);

        // update notification forwarding
        String[] componentIDs = (String[])m_components.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        for (int i = 0; i < componentIDs.length; i++)
        {
            m_parent.m_notificationManager.removeNotificationHandler(componentIDs[i], new String[]{ notificationType }, m_parent.m_notificationForwarder);
        }
    }

    private void handleChangeNotificationMonitorsAttrs(IDeltaAttributeSet attrs)
    throws Exception
    {
        handleNewNotificationMonitorsAttrs(attrs.getNewAttributesNames(), attrs);
        handleModifiedNotificationMonitorsAttrs(attrs.getModifiedAttributesNames(), attrs);
        handleDeletedNotificationMonitorsAttrs(attrs.getDeletedAttributesNames());
    }

    private void handleNewNotificationMonitorsAttrs(IAttributeSet attrs)
    throws Exception
    {
        Iterator iterator = attrs.getAttributes().entrySet().iterator();
        while (iterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)iterator.next();
            String name = (String)entry.getKey();
            IAttributeSet monitorAttrs = (IAttributeSet)entry.getValue();
            addNotificationMonitor(name, monitorAttrs);
        }
    }

    private void handleNewNotificationMonitorsAttrs(String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            addNotificationMonitor(names[i], (IAttributeSet)attrs.getNewValue(names[i]));
        }
    }

    private void handleModifiedNotificationMonitorsAttrs(String[] names, IDeltaAttributeSet attrs)
    throws Exception
    {
        for (int i = 0; i < names.length; i++)
        {
            NotificationMonitor monitor = null;
            synchronized(this)
            {
                monitor = (NotificationMonitor)m_notificationMonitors.get(names[i]);
            }
            if (monitor != null)
            {
                monitor.handleElementChange((IDeltaAttributeSet)attrs.getNewValue(names[i]));
            }
        }
    }

    private void handleDeletedNotificationMonitorsAttrs(String[] names)
    {
        for (int i = 0; i < names.length; i++)
        {
            this.removeNotificationMonitor(names[i]);
        }
    }

    private synchronized void addNotificationMonitor(String name, IAttributeSet attrs)
    {
        Integer highThreshold = (Integer)attrs.getAttribute(IComponentCollectionConstants.HIGH_THRESHOLD_ATTR);
        Integer lowThreshold = (Integer)attrs.getAttribute(IComponentCollectionConstants.LOW_THRESHOLD_ATTR);
        Integer intervalSeconds = (Integer)attrs.getAttribute(IComponentCollectionConstants.INTERVAL_SECONDS_ATTR);

        IAttributeList monitorNotifications = (IAttributeList)attrs.getAttribute(IComponentCollectionConstants.MONITOR_NOTIFICATIONS_ATTR);
        String[] types = new String[monitorNotifications.getCount()];
        for (int i = 0; i < types.length; i++)
        {
            types[i] = (String)monitorNotifications.getItem(i);
        }

        NotificationMonitor notificationMonitor = new NotificationMonitor(name, highThreshold, lowThreshold, intervalSeconds, types);
        m_notificationMonitors.put(name, notificationMonitor);
    }

    private synchronized void removeNotificationMonitor(String name)
    {
        NotificationMonitor monitor = (NotificationMonitor)m_notificationMonitors.remove(name);
        monitor.cleanup();
    }

    private synchronized void handleChangeMonitorMetricsAttr()
    {
        HashSet monitorMetrics = new HashSet();

        // Get the configuration from the configuration element
        IElement collectionElement = m_frameworkContext.getConfiguration(m_collectionConfigID, true);
        IAttributeSet collectionAttrs = collectionElement.getAttributes();

        m_monitoredMetricInfos.clear();
        m_metricInstances.clear();
        IAttributeList metricList = (IAttributeList)collectionAttrs.getAttribute(IComponentCollectionConstants.MONITOR_METRICS_ATTR);
        if (metricList != null)
        {
            for (int i = metricList.getCount() - 1; i >= 0; i--)
            {
                IMetricIdentity metricID = MetricsFactory.createMetricIdentity((String)metricList.getItem(i));
                monitorMetrics.add(metricID);
                m_unknownMetricInfos.add(metricID);
            }
        }

        synchronized(this)
        {
            int previousMetricCount = m_monitorMetrics.size();

            m_monitorMetrics = monitorMetrics;

            // do we need to reschedule
            if (previousMetricCount == 0)
            {
                m_lastRequestedPollTime = 0;
                scheduleBrokerStatePoll();
                scheduleMetricsPoll();
                scheduleMetricsSlowPoll();
            }
        }
    }

    private class NotificationMonitor
    implements Runnable, INotificationHandler
    {
        String monitorName;
        Integer highThreshold;
        Integer lowThreshold;
        Integer intervalSeconds;
        String[] types = IEmptyArray.EMPTY_STRING_ARRAY;
        boolean isClosing = false;
        Object notificationLockObj = new Object();
        int count;

        long scheduledTime;

        private NotificationMonitor(String monitorName, Integer highThreshold, Integer lowThreshold, Integer intervalSeconds, String[] types)
        {
            this.monitorName = monitorName;
            this.highThreshold = highThreshold;
            this.lowThreshold = lowThreshold;
            this.intervalSeconds = intervalSeconds;
            updateHandledTypes(types);
            this.scheduledTime = System.currentTimeMillis() + (this.intervalSeconds.intValue() * 1000);
            CollectionMonitor.this.m_frameworkContext.scheduleTask(this, new Date(this.scheduledTime));
        }

        private void cleanup()
        {
            isClosing = true;
            CollectionMonitor.this.m_frameworkContext.cancelTask(this);
            updateHandledTypes(IEmptyArray.EMPTY_STRING_ARRAY);
        }

        private void updateHandledTypes(String[] replacementTypes)
        {
            synchronized (notificationLockObj)
            {
                // work out the set to add and create subscriptions
                ArrayList addedTypes = new ArrayList();
                for (int i = 0; i < replacementTypes.length; i++)
                {
                    boolean found = false;
                    for (int j = 0; j < this.types.length; j++)
                    {
                        if (replacementTypes[i].equals(this.types[j]))
                        {
                            found = true;
                            break;
                        }
                    }
                    if (!found)
                    {
                        addedTypes.add(replacementTypes[i]);
                    }
                }

                // work out the set to remove and remove subscriptions
                ArrayList removedTypes = new ArrayList();
                for (int i = 0; i < this.types.length; i++)
                {
                    boolean found = false;
                    for (int j = 0; j < replacementTypes.length; j++)
                    {
                        if (this.types[i].equals(replacementTypes[j]))
                        {
                            found = true;
                            break;
                        }
                    }
                    if (!found)
                    {
                        removedTypes.add(this.types[i]);
                    }
                }

                if (!addedTypes.isEmpty())
                {
                    String[] addTypes = (String[])addedTypes.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
                    String[] components = (String[])CollectionMonitor.this.m_components.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
                    for (int j = 0; j < components.length; j++)
                    {
                        CollectionMonitor.this.m_parent.m_notificationManager.addNotificationHandler(components[j], addTypes, this);
                    }
                }

                // Sonic00036531: In the unusual circumstances that we have both added and removed types, then add a pause between the change in
                //                subscriptions to give a best chance for the add and remove notifications not to conflict due to the async
                //                subscription mechnaism we now use
                if (!addedTypes.isEmpty() && !removedTypes.isEmpty())
                {
                    try 
                    { 
                        long timeout = 2000;
                        long waitTime = timeout;
                        long startTime = System.currentTimeMillis();
                        while (waitTime > 0)
                        {
                            notificationLockObj.wait(waitTime);
                            waitTime = timeout - (System.currentTimeMillis() - startTime);
                        }
                    } catch(Exception e) { }
                    if (m_isClosing || this.isClosing)
                    {
                        return;
                    }
                }

                if (!removedTypes.isEmpty())
                {
                    String[] removeTypes = (String[])removedTypes.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
                    String[] components = (String[])CollectionMonitor.this.m_components.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
                    for (int j = 0; j < components.length; j++)
                    {
                        CollectionMonitor.this.m_parent.m_notificationManager.removeNotificationHandler(components[j], removeTypes, this);
                    }
                }

                this.types = replacementTypes;
            }
        }

        void addComponent(String componentID)
        {
            CollectionMonitor.this.m_parent.m_notificationManager.addNotificationHandler(componentID, this.types, this);
        }

        void removeComponent(String componentID)
        {
            CollectionMonitor.this.m_parent.m_notificationManager.removeNotificationHandler(componentID, this.types, this);
        }

        @Override
        public void run()
        {
            // the task should have been cancelled, but to handle the small window .. just return
            if (m_isClosing || this.isClosing)
            {
                return;
            }

            int lastCount = 0;
            synchronized(notificationLockObj)
            {
                lastCount = this.count;
                this.count = 0;
            }

            INotification notification = null;
            //m_frameworkContext.logMessage("NotificationMonitor.run(): highThreshold=" + highThreshold + ", count=" + lastCount, Level.TRACE);

            if (highThreshold != null && lastCount > highThreshold.intValue())
            {
                notification = CollectionMonitor.this.m_frameworkContext.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.MONITOR_SUBCATEGORY], THRESHOLD_NOTIFICATION_TYPE, Level.WARNING);
                notification.setLogType(INotification.WARNING_TYPE);
                notification.setAttribute("CollectionConfigID", CollectionMonitor.this.m_collectionConfigID);
                notification.setAttribute("NotificationMonitor", this.monitorName);
                notification.setAttribute("HighThreshold", this.highThreshold.toString());
                notification.setAttribute("ActualCount", "" + lastCount);
                CollectionMonitor.this.m_frameworkContext.sendNotification(notification);
            }

            if (lowThreshold != null && lastCount < lowThreshold.intValue())
            {
                notification = CollectionMonitor.this.m_frameworkContext.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.MONITOR_SUBCATEGORY], THRESHOLD_NOTIFICATION_TYPE, Level.WARNING);
                notification.setLogType(INotification.WARNING_TYPE);
                notification.setAttribute("CollectionConfigID", CollectionMonitor.this.m_collectionConfigID);
                notification.setAttribute("NotificationMonitor", this.monitorName);
                notification.setAttribute("LowThreshold", this.lowThreshold.toString());
                notification.setAttribute("ActualCount", "" + lastCount);
                CollectionMonitor.this.m_frameworkContext.sendNotification(notification);
            }

            if (notification != null && m_parent.m_saveThresholdNotifications)
            {
                try
                {
                    /** offload or store but not both! */
                    if (CollectionMonitor.this.m_parent.m_offloader != null) {
                    	if (m_parent.isEnterprise()) {
                    		CollectionMonitor.this.m_parent.m_offloader.offloadNotification(notification);
	    				} else if (!m_enterpriseMessageRequiredLogged) {
	    					m_frameworkContext.logMessage("Analytics offload is not supported and requires Enterprise Sonic Messaging", Level.SEVERE);
	    					m_enterpriseMessageRequiredLogged = true;
	    				}
                    } else if (CollectionMonitor.this.m_parent.m_store != null) {
                    	CollectionMonitor.this.m_parent.m_store.storeNotification(notification);
                    }
                }
                catch(AnalyticsOffloadException e)
                {
                    if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_OFFLOAD_FAILURES) > 0)
                    {
                        CollectionMonitor.this.m_frameworkContext.logMessage("Failed to offload threshold notification, trace follows...", e, Level.TRACE);
                    }
                    else
                    {
                        CollectionMonitor.this.m_frameworkContext.logMessage("Failed to offload threshold notification", Level.WARNING);
                    }
                }           
                catch(StorageException e)
                {
                    if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_STORAGE_FAILURES) > 0)
                    {
                        CollectionMonitor.this.m_frameworkContext.logMessage("Failed to store threshold notification, trace follows...", e, Level.TRACE);
                    }
                    else
                    {
                        CollectionMonitor.this.m_frameworkContext.logMessage("Failed to store threshold notification", Level.WARNING);
                    }
                }
            }

            if (m_isClosing || this.isClosing)
             {
                return; // no need to reschedule
            }

            this.scheduledTime += (this.intervalSeconds.intValue() * 1000);
            CollectionMonitor.this.m_frameworkContext.scheduleTask(this, new Date(this.scheduledTime));
        }

        @Override
        public void handleNotification(INotification notification)
        {
            synchronized(notificationLockObj)
            {
                this.count++;
                //m_frameworkContext.logMessage("NotificationMonitor.handleNotification(): type=" + notification.getType() + ", source=" + notification.getSourceIdentity().getCanonicalName() + ", count=" + this.count, Level.TRACE);
            }

            // save on a separate thread so this method can return quickly
            final INotification saveNotification = notification;
            Runnable saver = new Runnable()
            {
                @Override
                public void run()
                {
                    if (CollectionMonitor.this.m_parent.m_saveMonitoredNotifications)
                    {
                        try
                        {
                            /** offload or store but not both! */
                        	if (CollectionMonitor.this.m_parent.m_offloader != null) {
                        		if (m_parent.isEnterprise()) {
                        			CollectionMonitor.this.m_parent.m_offloader.offloadNotification(saveNotification);
	            				} else if (!m_enterpriseMessageRequiredLogged) {
	            					m_frameworkContext.logMessage("Analytics offload is not supported and requires Enterprise Sonic Messaging", Level.SEVERE);
	            					m_enterpriseMessageRequiredLogged = true;
	            				}
                        	} else if (CollectionMonitor.this.m_parent.m_store != null) {
                        		CollectionMonitor.this.m_parent.m_store.storeNotification(saveNotification);
                        	}
                        }

                        catch(AnalyticsOffloadException e)
                        {
                            if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_OFFLOAD_FAILURES) > 0)
                            {
                                CollectionMonitor.this.m_frameworkContext.logMessage("Failed to offload monitored notification, trace follows...", e, Level.TRACE);
                            }
                            else
                            {
                                CollectionMonitor.this.m_frameworkContext.logMessage("Failed to offload monitored notification", Level.WARNING);
                            }
                        }
                        catch(StorageException e)
                        {
                            if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_STORAGE_FAILURES) > 0)
                            {
                                CollectionMonitor.this.m_frameworkContext.logMessage("Failed to store monitored notification, trace follows...", e, Level.TRACE);
                            }
                            else
                            {
                                CollectionMonitor.this.m_frameworkContext.logMessage("Failed to store monitored notification", Level.WARNING);
                            }
                        }
                    }
                }
            };
            CollectionMonitor.this.m_frameworkContext.scheduleTask(saver, new Date());
        }

        void handleElementChange(IDeltaAttributeSet attrs)
        throws Exception
        {
            handleNewNotificationMonitorsAttrs(attrs.getNewAttributesNames(), attrs);
            handleModifiedNotificationMonitorsAttrs(attrs.getModifiedAttributesNames(), attrs);
            handleDeletedNotificationMonitorsAttrs(attrs.getDeletedAttributesNames());
        }

        private void handleNewNotificationMonitorsAttrs(String[] names, IDeltaAttributeSet attrs)
        throws Exception
        {
            for (int i = 0; i < names.length; i++)
            {
                if (names[i].equals(IComponentCollectionConstants.HIGH_THRESHOLD_ATTR))
                {
                    this.highThreshold = (Integer)attrs.getNewValue(IComponentCollectionConstants.HIGH_THRESHOLD_ATTR);
                }
                if (names[i].equals(IComponentCollectionConstants.LOW_THRESHOLD_ATTR))
                {
                    this.lowThreshold = (Integer)attrs.getNewValue(IComponentCollectionConstants.LOW_THRESHOLD_ATTR);
                }
                if (names[i].equals(IComponentCollectionConstants.INTERVAL_SECONDS_ATTR))
                {
                    this.intervalSeconds = (Integer)attrs.getNewValue(IComponentCollectionConstants.INTERVAL_SECONDS_ATTR);
                }
                if (names[i].equals(IComponentCollectionConstants.MONITOR_NOTIFICATIONS_ATTR))
                {
                    IAttributeList attrList = (IAttributeList)attrs.getNewValue(IComponentCollectionConstants.MONITOR_NOTIFICATIONS_ATTR);
                    String[] monitorNotifications = (String[])attrList.getItems().toArray(IEmptyArray.EMPTY_STRING_ARRAY);
                    updateHandledTypes(monitorNotifications);
                }
            }
        }

        private void handleModifiedNotificationMonitorsAttrs(String[] names, IDeltaAttributeSet attrs)
        throws Exception
        {
            for (int i = 0; i < names.length; i++)
            {
                if (names[i].equals(IComponentCollectionConstants.HIGH_THRESHOLD_ATTR))
                {
                    this.highThreshold = (Integer)attrs.getNewValue(IComponentCollectionConstants.HIGH_THRESHOLD_ATTR);
                }
                if (names[i].equals(IComponentCollectionConstants.LOW_THRESHOLD_ATTR))
                {
                    this.lowThreshold = (Integer)attrs.getNewValue(IComponentCollectionConstants.LOW_THRESHOLD_ATTR);
                }
                if (names[i].equals(IComponentCollectionConstants.INTERVAL_SECONDS_ATTR))
                {
                    this.intervalSeconds = (Integer)attrs.getNewValue(IComponentCollectionConstants.INTERVAL_SECONDS_ATTR);
                }
                if (names[i].equals(IComponentCollectionConstants.MONITOR_NOTIFICATIONS_ATTR))
                {
                    Object list = attrs.getNewValue(IComponentCollectionConstants.MONITOR_NOTIFICATIONS_ATTR);
                    String[] updatedTypes = null;
                    if (list instanceof IAttributeList)
                    {
                        updatedTypes = (String[])((IAttributeList)list).getItems().toArray(IEmptyArray.EMPTY_STRING_ARRAY);
                    }
                    else
                    {
                        IDeltaAttributeList deltaList = (IDeltaAttributeList)list;
                        updatedTypes = (String[])this.types.clone();
                        // handle changed ones
                        HashMap modified = deltaList.getModifiedItems();
                        Iterator iterator = modified.entrySet().iterator();
                        while (iterator.hasNext())
                        {
                            Map.Entry entry = (Map.Entry)iterator.next();
                            updatedTypes[((Integer)entry.getKey()).intValue()] = (String)entry.getValue();
                        }
                        // delete the removed ones
                        int[] removedNotifications = deltaList.getDeletedItemNumbers();
                        if (removedNotifications.length > 0)
                        {
                            for (int j = 0; j < removedNotifications.length; j++)
                            {
                                updatedTypes[removedNotifications[j]] = null;
                            }
                            String[] tempTypes = new String[updatedTypes.length - removedNotifications.length];
                            for (int j = 0, k = 0; j < updatedTypes.length; j++)
                            {
                                if (updatedTypes[j] != null)
                                {
                                    tempTypes[k++] = updatedTypes[j];
                                }
                            }
                            updatedTypes = tempTypes;
                        }
                        // add the new ones
                        Object[] newNotifications = deltaList.getNewItems();
                        if (newNotifications.length > 0)
                        {
                            String[] tempTypes = new String[updatedTypes.length + newNotifications.length];
                            System.arraycopy(updatedTypes, 0, tempTypes, 0, updatedTypes.length);
                            for (int j = 0; j < newNotifications.length; j++)
                            {
                                tempTypes[updatedTypes.length + j] = (String)newNotifications[j];
                            }
                            updatedTypes = tempTypes;
                        }
                    }
                    updateHandledTypes(updatedTypes);
                }
            }
        }

        private void handleDeletedNotificationMonitorsAttrs(String[] names)
        throws Exception
        {
            for (int i = 0; i < names.length; i++)
            {
                if (names[i].equals(IComponentCollectionConstants.HIGH_THRESHOLD_ATTR))
                {
                    this.highThreshold = null;
                }
                if (names[i].equals(IComponentCollectionConstants.LOW_THRESHOLD_ATTR))
                {
                    this.lowThreshold = null;
                }
                if (names[i].equals(IComponentCollectionConstants.INTERVAL_SECONDS_ATTR))
                {
                    this.intervalSeconds = new Integer(IComponentCollectionConstants.METRICS_REFRESH_INTERVAL_SECONDS_DEFAULT);
                }
                if (names[i].equals(IComponentCollectionConstants.MONITOR_NOTIFICATIONS_ATTR))
                {
                    updateHandledTypes(IEmptyArray.EMPTY_STRING_ARRAY);
                }
            }
        }
    }

    private class BrokerStatusCollector implements Runnable {

        @Override
        public void run() {
            if (m_isClosing) {
                return;
            }
            
            // get the list of active and standby broker's
            if ( m_brokerComponents.size() > 0 ) {
            	HashSet activeComponents = new HashSet();
            	HashSet inActiveComponents = new HashSet();
                // populate the Active and Standby broker
                String[] brokerComponents = (String[])m_brokerComponents.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
                for (int i = 0; i < brokerComponents.length; i++)
                {
                    try {
                        Integer nReplStateObj = (Integer) m_frameworkContext.invoke(brokerComponents[i], "getReplicationState",IEmptyArray.EMPTY_OBJECT_ARRAY,IEmptyArray.EMPTY_STRING_ARRAY, true, 0);
                        int nReplState = nReplStateObj.intValue();
                        if( nReplState == STANDALONE   || nReplState ==  ACTIVE || nReplState == ACTIVE_SYNC  ) {
                            activeComponents.add(brokerComponents[i]);
                        } else {
                            inActiveComponents.add(brokerComponents[i]);
                        }
                    }catch(Exception e) {
                        m_frameworkContext.logMessage("Failed to poll for getReplicationState on component ["+ brokerComponents[i] + ']', Level.WARNING);
                    }
                }

                m_activeComponents   = (HashSet)  activeComponents.clone();
                m_inactiveComponents = (HashSet)inActiveComponents.clone();
            }
            // schedule next poll
            if (!m_isClosing) {
                scheduleBrokerStatePoll();
            }
        }
    }

	private class SlowCollector implements Runnable {

		@Override
		public void run() {
			if (m_isClosing) {
				return;
			}

			// get the list of components and metrics we're after
			String[] components = getSlowCollectionComponents();

			if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_METRIC_POLLING) > 0) {
				m_frameworkContext.logMessage(
						"Starting metrics slow poll on collection ["
								+ m_collectionConfigID + "] for [" + components.length + "] components...", Level.TRACE);
			}

			IMetricIdentity[] metrics = null;
			synchronized (this) {
				metrics = (IMetricIdentity[]) m_monitorMetrics
						.toArray(IMetricIdentity.EMPTY_METRIC_IDENTITY_ARRAY);
			}

			ICollectiveOpStatus results = null;
			try {
				if ((components != null) && (components.length > 0)) {
					results = (ICollectiveOpStatus) m_frameworkContext.invoke(
							components, "getMetricsData", new Object[] {
									metrics, Boolean.FALSE },
							GET_METRICS_DATA_SIGNATURE, true,
							GET_METRICS_TIMEOUT);
				}
			} catch (Exception e) {
				m_frameworkContext.logMessage(
						"Failed to slow poll for metrics on collection ["
								+ m_collectionConfigID + ']', Level.WARNING);
			}

			// go through each of the individual results
			if (results != null) {
				for (int i = results.getCount() - 1; i >= 0; i--) {
					String component = results.getComponentName(i);
					if (results.operationIsSuccessful(i)) {
						// save any results and enable any metrics that were not
						// provided in the results
						IMetricsData resultMetrics = (IMetricsData) results
								.getReturnValue(i);
						if ((resultMetrics != null) && (component != null)) {
							HashSet missingMetrics = storeMetrics(component,
									resultMetrics);

							if (!missingMetrics.isEmpty()) {
								enableMetrics(component, missingMetrics);
							}
						}
						promoteSlowComponent(component);
					} else {
						Throwable throwable = results.getThrowable(i);
						if (throwable instanceof InvokeTimeoutException) {
							if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_METRIC_POLL_TIMEOUTS) > 0) {
								m_frameworkContext.logMessage(
										"Timeout of slow poll for metrics on component ["
												+ component + ']', Level.TRACE);
							}
						} else if (throwable instanceof NoSuchMethodException) {
							// ignore it unless tracing is on
							if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_REQUEST_FAILURES) > 0) {
								m_frameworkContext
										.logMessage(
												"Failed to slow poll for metrics on component ["
														+ component
														+ "]: component does not provide any metrics",
												Level.TRACE);
							}
						} else {
							// only log if its a different container or if its
							// this container and the container has booted
							ICanonicalName cmName = m_frameworkContext
									.getComponentName();
							ICanonicalName componentName = new CanonicalName(
									component);
							if (!(cmName.getContainerName()
									.equals(componentName.getContainerName()))
									|| m_frameworkContext.getContainer()
											.isBooted()) {
								m_frameworkContext.logMessage(
										"Failed to slow poll for metrics on component ["
												+ component + "]",
										Level.WARNING);
								if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_REQUEST_FAILURES) > 0) {
									m_frameworkContext.logMessage(
											"...trace follows...", throwable,
											Level.TRACE);
								}
							}
						}
					}
				}
			}
			if ((m_parent.m_traceMask & CollectionsMonitor.TRACE_METRIC_POLLING) > 0) {
				m_frameworkContext.logMessage(
						"...finished metrics slow poll on collection ["
								+ m_collectionConfigID + "] for [" + components.length + "] components...", Level.TRACE);
			}

			// schedule next poll
			if (!m_isClosing) {
				scheduleMetricsSlowPoll();
			}
		}

	}

	public synchronized void promoteSlowComponent(String component) {
		m_slowComponents.remove(component);
	}

	public synchronized void relegateFastComponent(String component) {
		m_slowComponents.add(component);
	}
}
