package com.sonicsw.mf.framework.monitor;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;

import com.sonicsw.mx.util.IEmptyArray;
import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.metrics.IAggregateMetric;
import com.sonicsw.mf.common.metrics.IAggregateMetricsData;
import com.sonicsw.mf.common.metrics.IHistoricalMetric;
import com.sonicsw.mf.common.metrics.IMetric;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.IMetricsData;
import com.sonicsw.mf.common.metrics.IValueType;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.common.metrics.impl.MetricsData;
import com.sonicsw.mf.common.metrics.manager.IMetricsRegistrar;
import com.sonicsw.mf.common.metrics.manager.IStatistic;
import com.sonicsw.mf.common.metrics.manager.IStatisticProvider;
import com.sonicsw.mf.common.metrics.manager.StatisticsFactory;
import com.sonicsw.mf.common.runtime.ICanonicalName;
import com.sonicsw.mf.common.runtime.IComponentState;
import com.sonicsw.mf.common.runtime.IEnterpriseAware;
import com.sonicsw.mf.common.runtime.IEnterpriseStateAccess;
import com.sonicsw.mf.common.runtime.IMonitoredMetrics;
import com.sonicsw.mf.common.runtime.IMonitoredNotifications;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.MonitoredMetrics;
import com.sonicsw.mf.common.runtime.impl.MonitoredNotifications;
import com.sonicsw.mf.framework.AbstractFrameworkComponent;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.INotificationHandler;
import com.sonicsw.mf.framework.monitor.offload.AnalyticsOffloader;
import com.sonicsw.mf.framework.monitor.storage.IHistoryStorage;
import com.sonicsw.mf.framework.monitor.storage.fs.FSStorage;
import com.sonicsw.mf.framework.monitor.storage.jdbc.JDBCStorage;
import com.sonicsw.mf.framework.util.NotificationForwarder;
import com.sonicsw.mf.mgmtapi.config.constants.ICollectionsMonitorConstants;
import com.sonicsw.mf.mgmtapi.runtime.ICollectionsMonitorProxy;

public final class CollectionsMonitor
extends AbstractFrameworkComponent
implements INotificationHandler, IOffloadListener, IHistoryStorageListener, IEnterpriseAware
{
    private String m_configID;
    ICanonicalName m_componentName;
    IHistoryStorage m_store;
    AnalyticsOffloader m_offloader;
    NotificationForwarder m_notificationForwarder;
    NotificationManager m_notificationManager;
    private long m_maxStorageSize;

    private HashMap m_collectionMonitors = new HashMap();

    private IMetricsRegistrar m_metricsRegistrar;
    private SdfMFTracingIntegration m_SdfMFTracingIntegration;

    private Integer m_historyDurationHours = ICollectionsMonitorConstants.HISTORY_DURATION_HOURS_DEFAULT;
    boolean m_saveMonitoredNotifications = ICollectionsMonitorConstants.SAVE_MONITORED_NOTIFICATIONS_DEFAULT;
    boolean m_saveThresholdNotifications = ICollectionsMonitorConstants.SAVE_THRESHOLD_NOTIFICATIONS_DEFAULT;

    public static final String THRESHOLD_NOTIFICATION_TYPE = "Threshold";
    public static final String[] THRESHOLD_NOTIFICATION_TYPES = new String[] { INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY] + '.' + INotification.SUBCATEGORY_TEXT[INotification.MONITOR_SUBCATEGORY] + '.' + THRESHOLD_NOTIFICATION_TYPE };
    private final String[] m_thresholdNotificationSources = new String[1];
	private IEnterpriseStateAccess m_enterpriseStateAccessor;

    private static final String MONITOR_TRACE_MASK_VALUES = "16=received notifications,32=forwarded notifications,64=storage failure details,128=metric poll timeouts,256=metric enabling,512=monitoring request failures,1024=storage activity,2048=metric polling,4096=offload failure details";
    public static final int TRACE_RECEIVED_NOTIFICATIONS = 16;
    public static final int TRACE_FORWARDED_NOTIFICATIONS = 32;
    public static final int TRACE_STORAGE_FAILURES = 64;
    public static final int TRACE_METRIC_POLL_TIMEOUTS = 128;
    public static final int TRACE_METRIC_ENABLING = 256;
    public static final int TRACE_REQUEST_FAILURES = 512;
    public static final int TRACE_STORAGE_ACTIVITY = 1024;
    public static final int TRACE_METRIC_POLLING = 2048;
    public static final int TRACE_OFFLOAD_FAILURES = 4096;

    private static final INotification[] EMPTY_NOTIFICATION_ARRAY = new INotification[0];
    private static final IHistoricalMetric[] EMPTY_HISTORICAL_METRIC_ARRAY = new IHistoricalMetric[0];
    private static final IAggregateMetric[] EMPTY_AGGREGATE_METRIC_ARRAY = new IAggregateMetric[0];
    private static final Reference[] EMPTY_REFERENCE_ARRAY = new Reference[0];

    // metric records written per minute
    public static final IMetricIdentity STORAGE_METRICS_STOREDPERMINUTE_METRIC =
        MetricsFactory.createMetricIdentity(new String[] { "storage", "metrics", "StoredPerMinute" });
    private static IStatistic m_metricsStoredPerMinute;
    // notification records written per minute
    public static final IMetricIdentity STORAGE_NOTIFICATIONS_STOREDPERMINUTE_METRIC =
        MetricsFactory.createMetricIdentity(new String[] { "storage", "notifications", "StoredPerMinute" });
    private static IStatistic m_notificationsStoredPerMinute;
    // average latency(time since metric origination) of collected metrics
	public static final IMetricIdentity MONITOR_METRICS_AVERAGE_METRICLATENCY_METRIC_ID = com.sonicsw.mf.common.metrics.MetricsFactory
			.createMetricIdentity(new String[] { "monitor", "metrics",
					"AverageMetricLatency" });
	private static IStatistic m_averageMetricLatencyStatistic;
    // min latency of collected metrics
	public static final IMetricIdentity MONITOR_METRICS_MIN_METRICLATENCY_METRIC_ID = com.sonicsw.mf.common.metrics.MetricsFactory
			.createMetricIdentity(new String[] { "monitor", "metrics",
					"MinMetricLatency" });
	private static IStatistic m_minMetricLatencyStatistic;
    // max latency of collected metrics
	public static final IMetricIdentity MONITOR_METRICS_MAX_METRICLATENCY_METRIC_ID = com.sonicsw.mf.common.metrics.MetricsFactory
			.createMetricIdentity(new String[] { "monitor", "metrics",
					"MaxMetricLatency" });
	private static IStatistic m_maxMetricLatencyStatistic;
    // metrics collected per second
	public static final IMetricIdentity MONITOR_METRICS_METRICS_PER_SECOND_METRIC_ID = com.sonicsw.mf.common.metrics.MetricsFactory
			.createMetricIdentity(new String[] { "monitor", "metrics",
					"MetricsPerSecond" });
	private static IStatistic m_metricsReceivedPerSecond;
    // notifications collected per second
	public static final IMetricIdentity MONITOR_NOTIFICATIONS_NOTIFICATIONS_PER_SECOND_METRIC_ID = com.sonicsw.mf.common.metrics.MetricsFactory
			.createMetricIdentity(new String[] { "monitor", "notifications",
					"NotificationsPerSecond" });
	private static IStatistic m_notificationsReceivedPerSecond;



    private static final ArrayList ATTRIBUTE_INFOS = new ArrayList();
    private static final ArrayList OPERATION_INFOS = new ArrayList();
    private static final ArrayList NOTIFICATION_INFOS = new ArrayList();

    private static final int MAX_RETRIEVAL_LIMIT = 1048576; // 1Mb

    static
    {
        //
        // Attributes
        //

        // root directory for historical monitoring data
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("HistoryDurationHours", Integer.class.getName(), "Duration (hours) for which historical monitoring data will be held.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("SaveMonitoredNotifications", Boolean.class.getName(), "Flag indicating if a history of monitored notifications should be maintained.", true, true, false));
        ATTRIBUTE_INFOS.add(new MBeanAttributeInfo("SaveThresholdNotifications", Boolean.class.getName(), "Flag indicating if a history of threshold notifications should be maintained.", true, true, false));

        //
        // Operations
        //
        MBeanParameterInfo[] mbParamInfos = null;

        // clear the history
        OPERATION_INFOS.add(new MBeanOperationInfo("clearHistory", "Clears all notification/metric history maintained by this monitor.",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, Void.class.getName(), MBeanOperationInfo.ACTION));

        // gets threshold notifications from the history
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("latest", Long.class.getName(), "The time of the latest threshold notification to be returned."),
            new MBeanParameterInfo("earliest", Long.class.getName(), "The time of the earliest threshold notification to be returned.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getThresholdNotificationHistory", "Gets recorded notifications for the given collection and notification types, between the given dates. Notifications are returned in date order with most recent first.",
                                                   mbParamInfos, INotification[].class.getName(), MBeanOperationInfo.INFO));

        // gets notifications from the history
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("collectionID", String.class.getName(), "The configuration ID to the monitored collection."),
            new MBeanParameterInfo("notificationTypes", String[].class.getName(), "The (canonical) notification types of interest."),
            new MBeanParameterInfo("latest", Long.class.getName(), "The time of the latest monitored notification to be returned."),
            new MBeanParameterInfo("earliest", Long.class.getName(), "The time of the earliest monitored notification to be returned.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getNotificationHistory", "Gets recorded notifications for the given collection and notification types, between the given dates. Notifications are returned in date order with most recent first.",
                                                   mbParamInfos, INotification[].class.getName(), MBeanOperationInfo.INFO));

        // gets aggregated metrics based on the history
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("collectionID", String.class.getName(), "The configuration ID to the monitored collection."),
            new MBeanParameterInfo("metricIDs", IMetricIdentity[].class.getName(), "The metric identities for which aggregate data is sought."),
            new MBeanParameterInfo("latestCurrency", Long.class.getName(), "The time of the latest metric value to be used to compute the aggregate data.."),
            new MBeanParameterInfo("lookbackPeriod", Long.class.getName(), "The period to lookback for metric values.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getAggregateMetricsData", "Gets aggregated metrics for a collection, based from recorded metrics of the given time window.",
                                                   mbParamInfos, IAggregateMetricsData.class.getName(), MBeanOperationInfo.INFO));

        // gets stored metrics for specified Component Collection(s) for specified time period
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("collectionIDs", String[].class.getName(), "The configuration ID(s) of the monitored collection(s)."),
            new MBeanParameterInfo("lastReceivedTimestamp", Long.class.getName(), "The timestamp of the last retrieved metric, or the timr of the last retrieval attempt."),
            new MBeanParameterInfo("lookbackPeriod", Long.class.getName(), "The period to look back for metric values.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getStoredMetrics", "Returns an instance of IMonitoredMetrics containing copies of the stored metrics for the specified component collection(s), for the given time window, up to a total size of approximately 1Mb, and a flag to indicate if there are more metrics to be retrieved.",
                                                   mbParamInfos, IMonitoredMetrics.class.getName(), MBeanOperationInfo.INFO));

        // gets stored notifications for specified Component Collection(s) for specified time period
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("collectionIDs", String[].class.getName(), "The configuration ID(s) of the monitored collection(s)."),
            new MBeanParameterInfo("lastReceivedTimestamp", Long.class.getName(), "The timestamp of the last retrieved notification, or the time of the last retrieval attempt."),
            new MBeanParameterInfo("lookbackPeriod", Long.class.getName(), "The period to look back for notification values.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getStoredNotifications", "Returns an instance of IMonitoredNotifications containing copies of the stored notifications for a collection, for the given time window, up to a total size of approximately 1Mb, and a flag to indicate if there are more notifications to be retrieved.",
                                                   mbParamInfos, IMonitoredNotifications.class.getName(), MBeanOperationInfo.INFO));

        // gets metrics from the history
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("collectionID", String.class.getName(), "The configuration ID to the monitored collection."),
            new MBeanParameterInfo("metricIDs", IMetricIdentity[].class.getName(), "The metric identities of interest."),
            new MBeanParameterInfo("latest", Long.class.getName(), "The time of the latest monitored metric value to be returned."),
            new MBeanParameterInfo("earliest", Long.class.getName(), "The time of the earliest monitored metric value to be returned.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getMetricHistory", "Gets recorded metrics for the given collection and metrics, between the given dates. Metrics are returned in date order with most recent first.",
                                                   mbParamInfos, IMetricsData.class.getName(), MBeanOperationInfo.INFO));

        // gets the list of collections being monitored by this monitor
        OPERATION_INFOS.add(new MBeanOperationInfo("getMonitoredCollections", "Gets the list of collections being monitored by this monitor.",
                                                   IEmptyArray.EMPTY_PARAMETER_INFO_ARRAY, String[].class.getName(), MBeanOperationInfo.INFO));

        // get the info for the notifications being forwarded by this monitor
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("collectionID", String.class.getName(), "The configuration ID to the monitored collection.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getForwardedNotificationsInfo", "Gets the meta-data for the notifications being forwarded by this monitor.",
                                                   mbParamInfos, MBeanNotificationInfo[].class.getName(), MBeanOperationInfo.INFO));

        // get the info for the metrics being monitored by this monitor
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("collectionID", String.class.getName(), "The configuration ID to the monitored collection.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getMonitoredMetricsInfo", "Gets the meta-data for the metrics being forwarded by this monitor.",
                                                   mbParamInfos, IMetricInfo[].class.getName(), MBeanOperationInfo.INFO));

        // get the instance names for instance metrics that are being monitored by this monitor
        mbParamInfos = new MBeanParameterInfo[]
        {
            new MBeanParameterInfo("collectionID", String.class.getName(), "The configuration ID to the monitored collection."),
            new MBeanParameterInfo("id", IMetricIdentity.class.getName(), "The metric identity for which instance names are sought.")
        };
        OPERATION_INFOS.add(new MBeanOperationInfo("getInstanceMetricNames", "Gets the instance metric names for a monitored metric of a specified collection.",
                                                   mbParamInfos, String[].class.getName(), MBeanOperationInfo.INFO));

        //
        // Notifications
        //
        String[] notifTypes = null;

        // threshold broken
        notifTypes = new String[]
        {
            INotification.CATEGORY_TEXT[INotification.SYSTEM_CATEGORY],
            INotification.SUBCATEGORY_TEXT[INotification.MONITOR_SUBCATEGORY],
            THRESHOLD_NOTIFICATION_TYPE
        };
        NOTIFICATION_INFOS.add(new MBeanNotificationInfo(notifTypes, INotification.CLASSNAME, "Sent when a notification monitoring threshold has been broken."));
    }

    @Override
    public MBeanAttributeInfo[] getAttributeInfos() { return (MBeanAttributeInfo[])ATTRIBUTE_INFOS.toArray(IEmptyArray.EMPTY_ATTRIBUTE_INFO_ARRAY); }

    @Override
    public MBeanOperationInfo[] getOperationInfos() { return (MBeanOperationInfo[])OPERATION_INFOS.toArray(IEmptyArray.EMPTY_OPERATION_INFO_ARRAY); }

    @Override
    public MBeanNotificationInfo[] getNotificationInfos() { return (MBeanNotificationInfo[])NOTIFICATION_INFOS.toArray(IEmptyArray.EMPTY_NOTIFICATION_INFO_ARRAY); }

    @Override
    public void init(IComponentContext context)
    {
    	AnalyticsOffloader offloader = AnalyticsOffloader.getInstance();
    	if (offloader.isDeployed()) {
    		m_offloader = offloader;
    		m_offloader.init();
    		m_offloader.setOffloadListener(this);
    	}
        m_notificationForwarder = new NotificationForwarder((IFrameworkComponentContext)context);
        m_notificationManager = new NotificationManager((IFrameworkComponentContext)context);

        // Get the configuration from the configuration element
        IElement cmElement = context.getConfiguration(true);
        m_configID = cmElement.getIdentity().getName();
        IAttributeSet cmAttributes = cmElement.getAttributes();

        // set initial attribute values if defined
        Integer historyDurationHours = (Integer)cmAttributes.getAttribute(ICollectionsMonitorConstants.HISTORY_DURATION_HOURS_ATTR);
        if (historyDurationHours != null)
        {
            setHistoryDurationHours(historyDurationHours);
        }
        // we have hijacked the NOTIFICATION_SUBSCRIPTION_TIMEOUT_ATTR for the renewal interval
        Integer notificationSubscriptionRenewalInterval = (Integer)cmAttributes.getAttribute(ICollectionsMonitorConstants.NOTIFICATION_SUBSCRIPTION_TIMEOUT_ATTR);
        if (notificationSubscriptionRenewalInterval != null)
        {
            setNotificationSubscriptionRenewalInterval(notificationSubscriptionRenewalInterval);
        }
        Boolean saveMonitoredNotifications = (Boolean)cmAttributes.getAttribute(ICollectionsMonitorConstants.SAVE_MONITORED_NOTIFICATIONS_ATTR);
        if (saveMonitoredNotifications != null)
        {
            setSaveMonitoredNotifications(saveMonitoredNotifications);
        }
        Boolean saveThresholdNotifications = (Boolean)cmAttributes.getAttribute(ICollectionsMonitorConstants.SAVE_THRESHOLD_NOTIFICATIONS_ATTR);
        if (saveThresholdNotifications != null)
        {
            setSaveThresholdNotifications(saveThresholdNotifications);
        }
        Long maxStorageSize = (Long) cmAttributes.getAttribute(ICollectionsMonitorConstants.MAX_STORAGE_SIZE_ATTR);
        maxStorageSize = (maxStorageSize != null) ? maxStorageSize : ICollectionsMonitorConstants.MAX_STORAGE_SIZE_DEFAULT;
        setMaxStorageSize(maxStorageSize);

        super.init(context);

        m_componentName = super.m_context.getComponentName();
        m_thresholdNotificationSources[0] = m_componentName.getCanonicalName();

        // create a store instance
        IAttributeSet jdbcStorageAttrs = (IAttributeSet)cmAttributes.getAttribute(ICollectionsMonitorConstants.JDBC_HISTORY_STORAGE_ATTR);
        if (m_offloader  != null) {
        // NOSONAR
        } else if (jdbcStorageAttrs == null) {
            m_store = getFSStorage((IAttributeSet)cmAttributes.getAttribute(ICollectionsMonitorConstants.FS_HISTORY_STORAGE_ATTR));
            m_store.setMaxStorageSize(m_maxStorageSize);
            m_store.setHistoryStorageListener(this);
        }
        else
        {
            m_store = getJDBCStorage(jdbcStorageAttrs);
            // set the storage size
            m_store.setMaxStorageSize(m_maxStorageSize);
            m_store.setHistoryStorageListener(this);
       }

        // needs to be reset, in case it was set prior to initialization
        setTraceMask(super.getTraceMask());

        // metrics
        m_metricsRegistrar = super.m_context.initMetricsManagement(getMetricsInfo());
    }

    @Override
    public synchronized void enableMetrics(IMetricIdentity[] ids)
    {
    	/**
    	 * Statistic is offset by the value at the start of the last refresh interval
    	 */
		final boolean INTERVAL_MODE = true;
		
		final short NO_HISTORY = 0;
		final short REFRESHED_VALUES_OVER_COLLECTION_INTERVAL_HISTORY = 1;
		final short REFRESHED_VALUES_AND_UPDATE_COUNTS_OVER_COLLECTION_INTERVAL_HISTORY = 2;
        for (int i = 0; i < ids.length; i++) {
            if (ids[i].equals(ICollectionsMonitorProxy.STORAGE_METRICS_STOREDPERMINUTE_METRIC_ID)) {
                m_metricsStoredPerMinute = StatisticsFactory.createStatistic(IStatistic.COUNTER_MODE, INTERVAL_MODE, null, REFRESHED_VALUES_OVER_COLLECTION_INTERVAL_HISTORY);
                m_metricsRegistrar.registerMetric(ICollectionsMonitorProxy.STORAGE_METRICS_STOREDPERMINUTE_METRIC_ID, m_metricsStoredPerMinute);
            } else if (ids[i].equals(ICollectionsMonitorProxy.STORAGE_NOTIFICATIONS_STOREDPERMINUTE_METRIC_ID))
            {
                m_notificationsStoredPerMinute = StatisticsFactory.createStatistic(IStatistic.COUNTER_MODE, INTERVAL_MODE, null, REFRESHED_VALUES_OVER_COLLECTION_INTERVAL_HISTORY);
                m_metricsRegistrar.registerMetric(ICollectionsMonitorProxy.STORAGE_NOTIFICATIONS_STOREDPERMINUTE_METRIC_ID, m_notificationsStoredPerMinute);
            } else if (ids[i]
					.equals(ICollectionsMonitorProxy.MONITOR_METRICS_AVERAGE_METRICLATENCY_METRIC_ID)
					&& m_averageMetricLatencyStatistic == null) {
				m_averageMetricLatencyStatistic = StatisticsFactory
						.createStatistic(IStatistic.COUNTER_MODE,
								INTERVAL_MODE, (IStatisticProvider[]) null,
								REFRESHED_VALUES_AND_UPDATE_COUNTS_OVER_COLLECTION_INTERVAL_HISTORY);
				m_metricsRegistrar.registerMetric(
						ICollectionsMonitorProxy.MONITOR_METRICS_AVERAGE_METRICLATENCY_METRIC_ID,
						m_averageMetricLatencyStatistic);
			} else if (ids[i]
					.equals(ICollectionsMonitorProxy.MONITOR_METRICS_MAX_METRICLATENCY_METRIC_ID)
					&& m_maxMetricLatencyStatistic == null) {
				m_maxMetricLatencyStatistic = StatisticsFactory
						.createStatistic(IStatistic.MAXIMUM_MODE,
								!INTERVAL_MODE, (IStatisticProvider[]) null,
								REFRESHED_VALUES_OVER_COLLECTION_INTERVAL_HISTORY);
				m_maxMetricLatencyStatistic.setInitialValue(0);
				m_metricsRegistrar
						.registerMetric(
								ICollectionsMonitorProxy.MONITOR_METRICS_MAX_METRICLATENCY_METRIC_ID,
								m_maxMetricLatencyStatistic);
			} else if (ids[i]
					.equals(ICollectionsMonitorProxy.MONITOR_METRICS_MIN_METRICLATENCY_METRIC_ID)
					&& m_minMetricLatencyStatistic == null) {
				m_minMetricLatencyStatistic = StatisticsFactory
						.createStatistic(IStatistic.MINIMUM_MODE,
								!INTERVAL_MODE, (IStatisticProvider[]) null,
								REFRESHED_VALUES_OVER_COLLECTION_INTERVAL_HISTORY);
				m_minMetricLatencyStatistic.setInitialValue(60000);
				m_metricsRegistrar
						.registerMetric(
								ICollectionsMonitorProxy.MONITOR_METRICS_MIN_METRICLATENCY_METRIC_ID,
								m_minMetricLatencyStatistic);
			} else if (ids[i]
					.equals(ICollectionsMonitorProxy.MONITOR_METRICS_METRICS_PER_SECOND_METRIC_ID)
					&& m_metricsReceivedPerSecond == null) {
				m_metricsReceivedPerSecond = StatisticsFactory.createStatistic(
						IStatistic.COUNTER_MODE, INTERVAL_MODE,
						(IStatisticProvider[]) null,
						REFRESHED_VALUES_OVER_COLLECTION_INTERVAL_HISTORY);
				m_metricsRegistrar.registerMetric(
						ICollectionsMonitorProxy.MONITOR_METRICS_METRICS_PER_SECOND_METRIC_ID,
						m_metricsReceivedPerSecond);
			} else if (ids[i]
					.equals(ICollectionsMonitorProxy.MONITOR_NOTIFICATIONS_NOTIFICATIONS_PER_SECOND_METRIC_ID)
					&& m_notificationsReceivedPerSecond == null) {
				m_notificationsReceivedPerSecond = StatisticsFactory.createStatistic(
						IStatistic.COUNTER_MODE, INTERVAL_MODE,
						(IStatisticProvider[]) null,
						REFRESHED_VALUES_OVER_COLLECTION_INTERVAL_HISTORY);
				m_metricsRegistrar.registerMetric(
						ICollectionsMonitorProxy.MONITOR_NOTIFICATIONS_NOTIFICATIONS_PER_SECOND_METRIC_ID,
						m_notificationsReceivedPerSecond);
			}
        }
    }

    @Override
    public synchronized void disableMetrics(IMetricIdentity[] ids)
    {
        for (int i = 0; i < ids.length; i++)
        {
            if (ids[i].equals(ICollectionsMonitorProxy.STORAGE_METRICS_STOREDPERMINUTE_METRIC_ID))
            {
                m_metricsRegistrar.unregisterMetric(ICollectionsMonitorProxy.STORAGE_METRICS_STOREDPERMINUTE_METRIC_ID);
                m_metricsStoredPerMinute = null;
            }
            else if (ids[i].equals(ICollectionsMonitorProxy.STORAGE_NOTIFICATIONS_STOREDPERMINUTE_METRIC_ID))
            {
                m_metricsRegistrar.unregisterMetric(ICollectionsMonitorProxy.STORAGE_NOTIFICATIONS_STOREDPERMINUTE_METRIC_ID);
                m_notificationsStoredPerMinute = null;
            }
			if (ids[i]
					.equals(MONITOR_METRICS_AVERAGE_METRICLATENCY_METRIC_ID)
					&& m_averageMetricLatencyStatistic != null) {
				m_metricsRegistrar
						.unregisterMetric(
								MONITOR_METRICS_AVERAGE_METRICLATENCY_METRIC_ID,
								m_averageMetricLatencyStatistic);
				m_averageMetricLatencyStatistic = null;
			} else if (ids[i]
					.equals(MONITOR_METRICS_MAX_METRICLATENCY_METRIC_ID)
					&& m_maxMetricLatencyStatistic != null) {
				m_metricsRegistrar
						.unregisterMetric(
								MONITOR_METRICS_MAX_METRICLATENCY_METRIC_ID,
								m_maxMetricLatencyStatistic);
				m_maxMetricLatencyStatistic = null;
			} else if (ids[i]
					.equals(MONITOR_METRICS_MIN_METRICLATENCY_METRIC_ID)
					&& m_minMetricLatencyStatistic != null) {
				m_metricsRegistrar
						.unregisterMetric(
								MONITOR_METRICS_MIN_METRICLATENCY_METRIC_ID,
								m_minMetricLatencyStatistic);
				m_minMetricLatencyStatistic = null;
			} else if (ids[i]
					.equals(MONITOR_METRICS_METRICS_PER_SECOND_METRIC_ID)
					&& m_metricsReceivedPerSecond != null) {
				m_metricsRegistrar.unregisterMetric(
						MONITOR_METRICS_METRICS_PER_SECOND_METRIC_ID,
						m_metricsReceivedPerSecond);
				m_metricsReceivedPerSecond = null;
			} else if (ids[i]
					.equals(MONITOR_NOTIFICATIONS_NOTIFICATIONS_PER_SECOND_METRIC_ID)
					&& m_notificationsReceivedPerSecond != null) {
				m_metricsRegistrar.unregisterMetric(
						MONITOR_NOTIFICATIONS_NOTIFICATIONS_PER_SECOND_METRIC_ID,
						m_notificationsReceivedPerSecond);
				m_notificationsReceivedPerSecond = null;
			}
        }
    }

    private IHistoryStorage getFSStorage(IAttributeSet fsStorageAttrs)
    {
        String rootDirectory = ".";
        if (fsStorageAttrs != null)
        {
            String directory = (String)fsStorageAttrs.getAttribute(ICollectionsMonitorConstants.DIRECTORY_ATTR);
            if (directory != null)
            {
                rootDirectory = directory;
            }
        }

        return new FSStorage(rootDirectory, m_componentName.getComponentName(), super.m_context);
    }

    private IHistoryStorage getJDBCStorage(IAttributeSet jdbcStorageAttrs)
    {
        String driver = (String)jdbcStorageAttrs.getAttribute(ICollectionsMonitorConstants.DRIVER_ATTR);
        String url = (String)jdbcStorageAttrs.getAttribute(ICollectionsMonitorConstants.URL_ATTR);
        String user = (String)jdbcStorageAttrs.getAttribute(ICollectionsMonitorConstants.USER_ATTR);
        String password = (String)jdbcStorageAttrs.getAttribute(ICollectionsMonitorConstants.PASSWORD_ATTR);
        String blobSQLType = (String)jdbcStorageAttrs.getAttribute(ICollectionsMonitorConstants.BLOB_SQL_TYPE_ATTR);
        String tnfError = (String)jdbcStorageAttrs.getAttribute(ICollectionsMonitorConstants.TABLE_NOT_FOUND_ERROR_ATTR);
        String ucError = (String)jdbcStorageAttrs.getAttribute(ICollectionsMonitorConstants.UNIQUE_CONSTRAINT_ERROR_ATTR);

        return new JDBCStorage(driver, url, user, password, blobSQLType, tnfError, ucError, m_componentName.getComponentName(), super.m_context);
    }

    @Override
    public synchronized void start()
    {
        m_SdfMFTracingIntegration = new SdfMFTracingIntegration();
        m_SdfMFTracingIntegration.register();

        if (super.m_state == IComponentState.STATE_ONLINE)
        {
            return;
        }

        if (m_store != null) {
	        // set the history duration and open the persistent store
	        m_store.setExpireAfter(m_historyDurationHours.intValue() * 60 * 60 * 1000);
	        m_store.open();
        }

        if (m_offloader != null) {
        	m_offloader.start();
        }
        updateMonitoredCollections();

        // we don't want to start subscribing until the local container has finished startup
        // in case there are any components in the local container that is a sources of notifications
        // that we are interested in
        Runnable subscriber = new Runnable()
        {
            @Override
            public void run()
            {
                CollectionsMonitor.this.waitForContainerBoot();
                if (CollectionsMonitor.super.m_container.isClosing())
                {
                    return;
                }
                CollectionsMonitor.this.m_notificationManager.subscribe(null);
            }
        };
        super.m_frameworkContext.scheduleTask(subscriber, new Date(System.currentTimeMillis()));

        super.start();
    }

    @Override
    public synchronized void stop()
    {
        if (super.m_state == IComponentState.STATE_OFFLINE)
        {
            return;
        }

        Object[] collections = m_collectionMonitors.keySet().toArray();
        for (int i = 0; i < collections.length; i++)
        {
            removeCollectionMonitor((String)collections[i]);
        }

        m_collectionMonitors.clear();
        m_notificationManager.cleanup();
        if (m_offloader != null) {
        	m_offloader.stop();
        }
        if (m_store != null) {
        	m_store.close();
        }

        super.stop();
    }

    @Override
    public void destroy()
    {
        m_notificationManager = null;
        m_store = null;
        if (m_offloader != null) {
        	m_offloader.destroy();
        	m_offloader = null;
        }
        super.destroy();
    }

    @Override
    public synchronized void handleElementChange(IElementChange elementChange)
    {
        // we are interested in changes to:
        //  - the list of collections being monitored
        //  - the details of the collections being monitored (which we we will
        //    pass on to the associated CollectionMonitor)
        //  - duration we maintain history for
        //  - the flags that determine what notifications should be stored in the history

        String configID = elementChange.getElement().getIdentity().getName();

        if (configID.equals(m_configID))
        {
            // this is a change to the CollectionsMonitor's configuration
            handleCollectionsMonitorElementChange(elementChange);
        }
        else // this is a change to one of the collection monitoring definitions handled by this monitor
        {
            CollectionMonitor monitor = (CollectionMonitor)m_collectionMonitors.get(configID);
            if (monitor != null)
            {
                // i.e. we are still monitoring this collection
                monitor.handleElementChange(elementChange);
            }
        }
    }

    @Override
    public String getTraceMaskValues()
    {
        return (super.getTraceMaskValues() + "," + MONITOR_TRACE_MASK_VALUES);
    }


    @Override
    public void setTraceMask(Integer traceMask)
    {
        setTraceMask(traceMask, false);
    }

    private void setTraceMask(Integer traceMask, boolean fromSDF)
    {
        // If SDF is used then ignore trace setup from other sources
        // If SDF is not used - update it from other sources
        if (!fromSDF && m_SdfMFTracingIntegration != null)
        {
            if (m_SdfMFTracingIntegration.wasUpdated())
            {
                return;
            }
            else
            {
                m_SdfMFTracingIntegration.setTraceMask(traceMask);
            }
        }

        super.setTraceMask(traceMask);
        if (m_notificationForwarder != null)
        {
            m_notificationForwarder.setDebug((super.m_traceMask & TRACE_FORWARDED_NOTIFICATIONS) > 0);
        }
        if (m_notificationManager != null)
        {
            m_notificationManager.setDebug((super.m_traceMask & this.TRACE_RECEIVED_NOTIFICATIONS) > 0);
        }
        FSStorage.setTraceMask(traceMask);
    }

    public void clearHistory()
    {
        if (m_store != null) {
        	m_store.clear();
        }
    }

    public INotification[] getThresholdNotificationHistory(Long latest, Long earliest)
    {
        if (m_store == null) {
        	return new INotification[0];
        }
    	Iterator iterator = m_store.getNotifications(THRESHOLD_NOTIFICATION_TYPES, m_thresholdNotificationSources, latest.longValue(), earliest.longValue());

        ArrayList list = new ArrayList();
        while (iterator.hasNext())
        {
            //list.add(iterator.next());
            IEventHolder holder = (IEventHolder) iterator.next();
            INotification notification = (INotification) holder.getEvent();
            list.add(notification);
        }

        return (INotification[])list.toArray(EMPTY_NOTIFICATION_ARRAY);
    }

    public INotification[] getNotificationHistory(String collectionID, String[] notificationTypes, Long latest, Long earliest)
    {
        if (m_store == null) {
        	return new INotification[0];
        }

    	Iterator iterator = m_store.getNotifications(notificationTypes, getComponentNames(collectionID), latest.longValue(), earliest.longValue());
        ArrayList list = new ArrayList();
        while (iterator.hasNext())
        {
            //list.add(iterator.next());
            IEventHolder holder = (IEventHolder) iterator.next();
            INotification notification = (INotification) holder.getEvent();
            list.add(notification);
        }

        return (INotification[])list.toArray(EMPTY_NOTIFICATION_ARRAY);
    }

    public IAggregateMetricsData getAggregateMetricsData(String collectionID, IMetricIdentity[] metricIDs, Long latestCurrency, Long lookbackPeriod)
    {
        if (m_store == null) {
        	return new MetricsData();
        }

    	String[] components = getComponentNames(collectionID);
        long latest = latestCurrency.longValue();
        long earliest = latest - lookbackPeriod.longValue();

        // we just want to go to the store once
        Iterator iterator = m_store.getMetrics(metricIDs, getComponentNames(collectionID), latest, earliest);
        ArrayList list = new ArrayList();
        while (iterator.hasNext())
        {
            //list.add(iterator.next());
            IEventHolder holder = (IEventHolder) iterator.next();
            IHistoricalMetric metric = (IHistoricalMetric) holder.getEvent();
            list.add(metric);
        }
        IHistoricalMetric[] recordedMetrics = (IHistoricalMetric[])list.toArray(EMPTY_HISTORICAL_METRIC_ARRAY);
        ArrayList aggregatedMetrics = new ArrayList();

        HashMap useableMetrics = new HashMap();
        HashSet sources = new HashSet();
        for (int i = 0; i < metricIDs.length; i++)
        {
            // clear out from last usage
            useableMetrics.clear();
            sources.clear();

            long[] aggregateValues = new long[] { 0, 0, Long.MIN_VALUE, Long.MAX_VALUE };

            long aggregateCurrency = 0;
            int contributerCount = 0;
            IMetricIdentity aggregatedID = null;

            // extract the useable metrics
            for (int j = 0; j < recordedMetrics.length; j++)
            {
                IMetricIdentity id = recordedMetrics[j].getMetricIdentity();

                if (id.equals(metricIDs[i]))
                {
                    String source = recordedMetrics[j].getSource();
                    if (sources.contains(source))
                    {
                        continue;
                    }

                    // else we need to use this metric in the aggregation
                    sources.add(source);
                    aggregatedID = id;

                    long value = recordedMetrics[j].getValue();
                    aggregateValues[IAggregateMetric.TOTAL_AGGREGATION_TYPE] += value;
                    aggregateValues[IAggregateMetric.AVERAGE_AGGREGATION_TYPE] += value;
                    if (aggregateValues[IAggregateMetric.MAXIMUM_AGGREGATION_TYPE] < value)
                    {
                        aggregateValues[IAggregateMetric.MAXIMUM_AGGREGATION_TYPE] = value;
                    }
                    if (aggregateValues[IAggregateMetric.MINIMUM_AGGREGATION_TYPE] > value)
                    {
                        aggregateValues[IAggregateMetric.MINIMUM_AGGREGATION_TYPE] = value;
                    }

                    aggregateCurrency = Math.round((((double)aggregateCurrency * (double)contributerCount) + recordedMetrics[j].getCurrencyTimestamp()) / ++contributerCount);
                }
            }

            // if there were no contributers then we won't even provide an aggregated metric
            if (contributerCount == 0)
            {
                continue;
            }

            // for an average we then divide by the number of values we have
            aggregateValues[IAggregateMetric.AVERAGE_AGGREGATION_TYPE] = Math.round((double)aggregateValues[IAggregateMetric.AVERAGE_AGGREGATION_TYPE] / (double)contributerCount);

            // create the aggegrate metric
            aggregatedMetrics.add(MetricsFactory.createMetric((String[])sources.toArray(IEmptyArray.EMPTY_STRING_ARRAY), aggregatedID, aggregateValues, aggregateCurrency));
        }

        // build the metrics data
        IAggregateMetric[] metrics = (IAggregateMetric[])aggregatedMetrics.toArray(EMPTY_AGGREGATE_METRIC_ARRAY);
        return MetricsFactory.createMetricsData(metrics, 0);
    }

    /**
     * Obtain the stored notifications for the specified set of Component Collection for the specified time period.
     *
     * If the lastObtainedTimestamp argument is null, stored notifications will be retrieved for the specified lookback period,
     * "looking back" from the current time.
     *
     * If the lastObtainedTimestamp argument is non-null, stored notifications from the current time back to the lastObtainedTimestamp
     * will be retrieved, provided that the period does not exceed the "maxLookbackPeriod" [if specified]
     *
     * @param collectionIDs          array of Collection Identifiers for which stored notifications will be retrieved
     * @param lastObtainedTimestamp  timestamp of last notification retrieved by caller; used to determine starting point for retrieval [if non-null]
     * @param maxLookbackPeriod      maximum interval (in milliseconds) for which notifications will be retrieved, before and up to current time
     * @return object containing retrieved notifications, timestamp of last retrieved notification, and flag indicating if there are more notifications to be retrieved for the specified time period; this return value will never be null
     */
    public IMonitoredNotifications getStoredNotifications(java.lang.String[] collectionIDs, Long lastObtainedTimestamp, Long maxLookbackPeriod)
    {
        if (m_store == null) {
        	return new MonitoredNotifications();
        	
        }
        // get the current time
        long currentTime = System.currentTimeMillis();

        // calculate the appropriate "start time" to pass to the store for retrieval.  Stored events with timestamps
        // equal to or later than the "start time", and earlier or equal to the current time, will be retrieved.
        long startTime = determineStartTimeForRetrieval(currentTime, lastObtainedTimestamp, maxLookbackPeriod);

        // create an object to hold the retrieved data
        MonitoredNotifications monitoredNotifications = new MonitoredNotifications();

        // get difference between provided timestamp and current time
        long deltaTime = currentTime - startTime;

        // if there is no time difference, return an object that contains
        // only a new timestamp (the current time) and a flag to indicate
        // that there is no more data to return
        if (deltaTime <= 0)
        {
            monitoredNotifications.setLatestTimestamp(currentTime);
            monitoredNotifications.setMoreDataToRetrieve(false);
            return monitoredNotifications;
        }

        // if there are no specified component collections, return an object that contains
        // only a new timestamp (the current time) and a flag to indicate that there is no data to return
        if ( (collectionIDs == null) || (collectionIDs.length == 0) )
        {
            monitoredNotifications.setLatestTimestamp(currentTime);
            monitoredNotifications.setMoreDataToRetrieve(false);
            return monitoredNotifications;
        }

        // build a list of all the component names based on all of the input CollectionIDs
        Vector compNamesVec = buildComponentVectorFromCollectionIDs(collectionIDs);

        // initialize the timestamp to return to the caller...
        long lastRetrievedTimestamp = currentTime;

        // initialize the total size counter
        int totalSize = 0;  // 0 bytes to start

        // initialize the flag used to indicate if a follow-up invocation of this method will be required
        // [in the event that the size of stored notifications exceeds the specified max specified size]
        boolean additionalDataToReturn = false;  // assume that all notifications can be returned in one invocation...

        // initialize the flag used to indicate if the retrieved data size exceeds the maximum specified size [to return in one "chunk"]
        boolean maxExceeded = false;

        // retrieve notifications for specified time interval [in order from earliest-to-latest]
        Iterator it = m_store.getNotifications(((String[])compNamesVec.toArray(IEmptyArray.EMPTY_STRING_ARRAY)), currentTime, startTime);
        if (it != null) // it would be null if the collections monitor is in the process of closing
        {
            while (it.hasNext())
            {
                //INotification notification = (INotification) it.next();
                IEventHolder holder = (IEventHolder) it.next();
                INotification notification = (INotification) holder.getEvent();

                //long timestamp = notification.getTimeStamp();
                long timestamp = holder.getStorageTimestamp();

                if ( (maxExceeded) && (timestamp > lastRetrievedTimestamp) )
                {
                    additionalDataToReturn = true;  // indicate that another invocation will be necessary in order to obtain all of the requested data...
                    break;
                }

                // record the timestamp of the most recently retrieved notification
                lastRetrievedTimestamp = timestamp;

                // add the notification to the return object
                monitoredNotifications.addNotification(notification);

                // update the size of the array list of returned notifications
                int notificationLength = notification.toString().getBytes().length;
                totalSize += notificationLength;

                // check if retrieved notifications have reached max allowable size size
                if (totalSize >= MAX_RETRIEVAL_LIMIT)
                {
                    maxExceeded = true;
                }
            } //"for ..."
        }

        // in the event that another invocation is required in order to return all of the requested data, update
        // the last retrieved timestamp value [so that the next invocation won't retrieve any of the notifications
        // that are about to be returned from the current invocation].
        lastRetrievedTimestamp += 1;   // timestamp of last retrieved notification plus one millisecond

        // set the updated last retrieval timestamp and the flag to indicate if there are more notifications [for the specified period] to return
        monitoredNotifications.setLatestTimestamp(lastRetrievedTimestamp);
        monitoredNotifications.setMoreDataToRetrieve(additionalDataToReturn);

        // return the data
        return (IMonitoredNotifications) monitoredNotifications;
    }

    /**
     * Obtain the stored metrics for the specified set of Component Collection for the specified time period.
     *
     *  If the lastObtainedTimestamp argument is null, stored metrics will be retrieved for the specified lookback period,
     * "looking back" from the current time.
     *
     * If the lastObtainedTimestamp argument is non-null, stored metrics from the current time back to the lastObtainedTimestamp
     * will be retrieved, provided that the period does not exceed the "maxLookbackPeriod" [if specified]
     *
     * @param collectionIDs          array of Collection Identifiers for which stored metrics will be retrieved
     * @param lastObtainedTimestamp  timestamp of last metric retrieved by caller; used to determine starting point for retrieval [if non-null]
     * @param maxLookbackPeriod      maximum interval (in milliseconds) for which metrics will be retrieved, before and up to current time
     * @return object containing retrieved metrics, timestamp of last retrieved metric, and flag indicating if there are more metrics to be retrieved for the specified time period; this return value will never be null
     */
    public IMonitoredMetrics getStoredMetrics(java.lang.String[] collectionIDs, Long lastObtainedTimestamp, Long maxLookbackPeriod)
    {
        if (m_store == null ) {
        	return new MonitoredMetrics();
        }
        // get the current time
        long currentTime = System.currentTimeMillis();

        // calculate the appropriate "start time" to pass to the store for retrieval.  Stored events with timestamps
        // equal to or later than the "start time", and earlier or equal to the current time, will be retrieved.
        long startTime = determineStartTimeForRetrieval(currentTime, lastObtainedTimestamp, maxLookbackPeriod);

        // create an object to hold the retrieved data, last retrieved timestamp and flag to indicate if an subsequent invocation is required to retrieve all data
        MonitoredMetrics monitoredMetrics = new MonitoredMetrics();

        // get difference between provided timestamp and current time
        long deltaTime = currentTime - startTime;

        // if there is no time difference, return an object that contains
        // only a new timestamp (the current time) and a flag to indicate
        // that there is no more data to return
        if (deltaTime <= 0)
        {
            monitoredMetrics.setLatestTimestamp(currentTime);
            monitoredMetrics.setMoreDataToRetrieve(false);
            return monitoredMetrics;
        }

        // if there are no specified component collections, return an object that contains
        // only a new timestamp (the current time) and a flag to indicate that there is no data to return
        if ( (collectionIDs == null) || (collectionIDs.length == 0) )
        {
            monitoredMetrics.setLatestTimestamp(currentTime);
            monitoredMetrics.setMoreDataToRetrieve(false);
            return monitoredMetrics;
        }

        // build a list of all the component names based on all of the input CollectionIDs
        Vector compNamesVec = buildComponentVectorFromCollectionIDs(collectionIDs);

        // initialize the timestamp to return to the caller...
        long lastRetrievedTimestamp = currentTime;

        // initializae the total size counter
        int totalSize = 0;  // 0 bytes to start

        // initialize the flag used to indicate if a follow-up invocation of this method will be required
        // [in the event that the size of stored metrics exceeds the specified max specified size]
        boolean additionalDataToReturn = false;  // assume that all metrics can be returned in one invocation...

        // initialize the flag used to indicate if the retrieved data size exceeds the maximum specified size [to return in one "chunk"]
        boolean maxExceeded = false;

        // retrieve any stored metrics for the specified time interval
        Iterator it = m_store.getMetrics(((String[])compNamesVec.toArray(IEmptyArray.EMPTY_STRING_ARRAY)), currentTime, startTime);
        if (it != null) // it would be null if the collections monitor is in the process of closing
        {
            while (it.hasNext())
            {
                //IHistoricalMetric metric = (IHistoricalMetric) it.next();
                IEventHolder holder = (IEventHolder) it.next();
                IHistoricalMetric metric = (IHistoricalMetric) holder.getEvent();

                //long timestamp = metric.getCurrencyTimestamp();
                long timestamp = holder.getStorageTimestamp();

                if ( (maxExceeded) && (timestamp > lastRetrievedTimestamp) )
                {
                    additionalDataToReturn = true;  // indicate that another invocation will be necessary in order to obtain all of the requested data...
                    break;
                }

                // record the timestamp of the most recently retrieved metric
                lastRetrievedTimestamp = timestamp;

                // add the metric to the return object
                monitoredMetrics.addMetric(metric);

                // update the combined size of metrics to be returned
                int metricLength = metric.toString().getBytes().length;
                totalSize += metricLength;                // add the metric's size to the total size of retrieved metric data
                totalSize += metric.getSource().length(); // add the component's name length to the total size of retrieved metric data

                // check if retrieved metrics have reached max allowable size size
                if (totalSize >= MAX_RETRIEVAL_LIMIT)
                {
                    maxExceeded = true;
                }
            } //"for ..."
        }

        // in the event that another invocation is required in order to return all of the requested data, update
        // the last retrieved timestamp value [so that the next invocation won't retrieve any of the metrics
        // that are to be returned from the current invocation].
        lastRetrievedTimestamp += 1;  // timestamp of last retrieved metric plus one millisecond...

        // update the last retrieved timestamp and set the flag indicating if there are more metrics [for the specified time period] to retrieve.
        monitoredMetrics.setLatestTimestamp(lastRetrievedTimestamp);
        monitoredMetrics.setMoreDataToRetrieve(additionalDataToReturn);

        // return the data
        return (IMonitoredMetrics) monitoredMetrics;
    }

    public static IMetricInfo[] getMetricsInfo()
    {
        // create the infos
        IMetricInfo[] infos = new IMetricInfo[7];
		final boolean INSTANCE = true;
		final String extendedData = null;
		final boolean DYNAMIC = true; // permits runtime enablement
		final boolean LOWALERTS = true;
		final boolean HIGHALERTS = true;
		infos[0] = MetricsFactory.createMetricInfo(ICollectionsMonitorProxy.STORAGE_METRICS_STOREDPERMINUTE_METRIC_ID, IValueType.PER_MINUTE_RATE,
                                                   "Number of monitored metric values stored per minute.",
                                                   extendedData, !INSTANCE, DYNAMIC, HIGHALERTS, false, "metrics stored per minute");
        infos[1] = MetricsFactory.createMetricInfo(ICollectionsMonitorProxy.STORAGE_NOTIFICATIONS_STOREDPERMINUTE_METRIC_ID, IValueType.PER_MINUTE_RATE,
                                                   "Number of monitored notifications stored per minute.",
                                                   extendedData, !INSTANCE, DYNAMIC, HIGHALERTS, false, "notifications stored per minute");
		infos[2] = MetricsFactory
				.createMetricInfo(
						ICollectionsMonitorProxy.MONITOR_METRICS_AVERAGE_METRICLATENCY_METRIC_ID,
						IValueType.AVERAGE,
						"Average latency in metrics reporting over collection interval.",
						extendedData, !INSTANCE, DYNAMIC, HIGHALERTS,
						LOWALERTS, "milliseconds");
		infos[3] = MetricsFactory.createMetricInfo(
				ICollectionsMonitorProxy.MONITOR_METRICS_MIN_METRICLATENCY_METRIC_ID,
				IValueType.MINIMUM,
				"Min latency in metrics reporting over collection interval.",
				extendedData, !INSTANCE, DYNAMIC, !HIGHALERTS, !LOWALERTS,
				"millisecond");
		infos[4] = MetricsFactory.createMetricInfo(
				ICollectionsMonitorProxy.MONITOR_METRICS_MAX_METRICLATENCY_METRIC_ID,
				IValueType.MAXIMUM,
				"Max latency in metrics reporting over collection interval.",
				extendedData, !INSTANCE, DYNAMIC, HIGHALERTS, LOWALERTS,
				"milliseconds");
		infos[5] = MetricsFactory.createMetricInfo(
				ICollectionsMonitorProxy.MONITOR_METRICS_METRICS_PER_SECOND_METRIC_ID,
				IValueType.PER_SECOND_RATE,
				"Metrics per second over collection interval.", extendedData,
				!INSTANCE, DYNAMIC, HIGHALERTS, LOWALERTS, "per second");
		infos[6] = MetricsFactory.createMetricInfo(
				ICollectionsMonitorProxy.MONITOR_NOTIFICATIONS_NOTIFICATIONS_PER_SECOND_METRIC_ID,
				IValueType.PER_SECOND_RATE,
				"Notifications per second over collection interval.", extendedData,
				!INSTANCE, DYNAMIC, HIGHALERTS, LOWALERTS, "per second");
		return infos;
    }

    public IMetricsData getMetricHistory(String collectionID, IMetricIdentity[] metricIDs, Long latest, Long earliest)
    {
        if (m_store == null) {
        	return new MetricsData();
        }

    	Iterator iterator = m_store.getMetrics(metricIDs, getComponentNames(collectionID), latest.longValue(), earliest.longValue());
        ArrayList list = new ArrayList();
        while (iterator.hasNext())
        {
            //list.add(iterator.next());
            IEventHolder holder = (IEventHolder) iterator.next();
            IHistoricalMetric metric = (IHistoricalMetric) holder.getEvent();
            list.add(metric);
        }

        return MetricsFactory.createMetricsData((IHistoricalMetric[])list.toArray(EMPTY_HISTORICAL_METRIC_ARRAY), 0);
    }

    public String[] getMonitoredCollections()
    {
        String[] monitoredCollections = (String[])m_collectionMonitors.keySet().toArray(IEmptyArray.EMPTY_STRING_ARRAY);
        return monitoredCollections;
    }

    public IMetricInfo[] getMonitoredMetricsInfo(String collectionID)
    {
        CollectionMonitor monitor = (CollectionMonitor)m_collectionMonitors.get(collectionID);
        return monitor.getMonitoredMetricsInfo();
    }

    public String[] getInstanceMetricNames(String collectionID, IMetricIdentity id)
    {
        CollectionMonitor monitor = (CollectionMonitor)m_collectionMonitors.get(collectionID);
        return monitor.getInstanceMetricNames(id);
    }

    public MBeanNotificationInfo[] getForwardedNotificationsInfo(String collectionID)
    {
        CollectionMonitor monitor = (CollectionMonitor)m_collectionMonitors.get(collectionID);
        return monitor.getForwardedNotificationsInfo();
    }

    //
    // The management attributes we will expose (beyond or overriding that handled by
    // AbstractFrameworkComponent).
    //

    /**
     * The CollectionsMonitor can be configured to persist a limited amount of
     * the data that was used in the monitoring of notifications and aggregation
     * of metrics. A value of 0 indicates that no historical data will be perisisted.
     *
     * @return Returns the duration that historical data will be persisted.
     *
     * @see #setHistoryDurationHours(Integer)
     */
    public Integer getHistoryDurationHours() { return m_historyDurationHours; }

    /**
     * The CollectionsMonitor can be configured to persist a limited amount of
     * the data that was used in the monitoring of notifications and aggregation
     * of metrics. A value of 0 indicates that no historical data will be perisisted.
     *
     * @param hours Sets the duration (hours) that historical data will be persisted.
     *
     * @see #getHistoryDurationHours()
     */
    public synchronized void setHistoryDurationHours(Integer hours)
    {
        if (hours.intValue() < 0)
        {
            throw new IllegalArgumentException("History duration (hours) must be >= 0");
        }
        m_historyDurationHours = hours;
    }

    private synchronized void setNotificationSubscriptionRenewalInterval(Integer interval)
    {
        if (interval != null)
         {
            m_notificationManager.setNotificationSubscriptionRenewalInterval(interval.longValue() * 1000);  // convert from seconds to milliseconds...
        }
    }

    /**
     * The CollectionsMonitor can be configured to persist monitored notifications
     * as they arrive.
     *
     * @return Returns a flag to indicate if monitored notifications will be persisted.
     *
     * @see #getHistoryDurationHours()
     */
    public Boolean getSaveMonitoredNotifications() { return new Boolean(m_saveMonitoredNotifications); }

    /**
     * The CollectionsMonitor can be configured to persist monitored notifications
     * as they arrive.
     *
     * @param save If true, saves subsequent monitored notifications.
     *
     * @see #getHistoryDurationHours()
     */
    public synchronized void setSaveMonitoredNotifications(Boolean save) { m_saveMonitoredNotifications = save.booleanValue(); }

    /**
     * The CollectionsMonitor can be configured to persist threshold notifications
     * as they are generated.
     *
     * @return Returns a flag to indicate if monitored notifications will be persisted.
     *
     * @see #getHistoryDurationHours()
     */
    public Boolean getSaveThresholdNotifications() { return new Boolean(m_saveThresholdNotifications); }

    /**
     * The CollectionsMonitor can be configured to persist monitored notifications
     * as they arrive.
     *
     * @param save If true, saves subsequent threshold notifications will be saved.
     *
     * @see #getHistoryDurationHours()
     */
    public synchronized void setSaveThresholdNotifications(Boolean save) { m_saveThresholdNotifications = save.booleanValue(); }

    /**
     * The size to which the store used by the CollectionsMonitor to persist monitored
     * notifications and metrics is allowed to grow.
     *
     * @param maxStorageSize maximum size, in megabytes, that store may grow to before older saved notifications and/or metrics are flushed from the store
     *
     */
    public synchronized Long getMaxStorageSize() { return new Long(m_maxStorageSize); }

    /**
     * The size to which the store used by the CollectionsMonitor to persist monitored
     * notifications and metrics may be constrained.
     *
     * @param maxStorageSize maximum size, in megabytes, that store may grow to before older saved notifications and/or metrics are flushed from the store
     *
     */
    public synchronized void setMaxStorageSize(Long maxStorageSize) { m_maxStorageSize = maxStorageSize.longValue(); }

    //
    // The management operations we will expose (beyond or overriding that handled by
    // AbstractFrameworkComponent).
    //

    /**
     * Handle notifications that are directed at the CollectionsMonitor. The CollectionsMonitor
     * has to do two things of receipt of such notifications:
     *
     *  - forward the notifications if it is configured to do so
     *  - apply monitored notifications to the notification monitors to which they apply
     *
     * This method is intended for use within the framework and should not be exposed as a
     * management operation (i.e. in the CollectionsMonitor's meta-data).
     */
    @Override
    public void handleNotification(INotification notification)
    {
        m_notificationManager.handleNotification(notification);
    }


    //
    // Internal/package methods
    //

    void waitForContainerBoot()
    {
        synchronized(m_container)
        {
            try
            {
                while (!(super.m_container.isBooted() || super.m_container.isClosing()))
                {
                    super.m_container.wait(1000);
                }
            } catch(InterruptedException ie) { return; } // can't think that this would ever occur
        }
    }

    private void updateMonitoredCollections()
    {
        // Get the configuration from the configuration element
        IElement cmElement = super.m_context.getConfiguration(true);
        IAttributeSet cmAttributes = cmElement.getAttributes();

        // get the list of collections to be monitored
        IAttributeList collectionsList = (IAttributeList)cmAttributes.getAttribute(ICollectionsMonitorConstants.COLLECTIONS_ATTR);
        Reference[] collectionIDs = (Reference[])collectionsList.getItems().toArray(EMPTY_REFERENCE_ARRAY);

        String[] monitoredCollections = (String[])m_collectionMonitors.keySet().toArray(IEmptyArray.EMPTY_STRING_ARRAY);

        // evaluate the list of collections to be removed
        ArrayList deletedCollections = new ArrayList();
        outerLoop:
        for (int i = 0; i < monitoredCollections.length; i++)
        {
            for (int j = 0; j < collectionIDs.length; j++)
            {
                if (collectionIDs[j].getElementName().equals(monitoredCollections[i]))
                {
                    continue outerLoop;
                }
            }
            deletedCollections.add(monitoredCollections[i]);
        }

        // evaluate the list of collections to be added
        ArrayList newCollections = new ArrayList();
        outerLoop:
        for (int i = 0; i < collectionIDs.length; i++)
        {
            for (int j = 0; j < monitoredCollections.length; j++)
            {
                if (collectionIDs[i].getElementName().equals(monitoredCollections[j]))
                {
                    continue outerLoop;
                }
            }
            newCollections.add(collectionIDs[i].getElementName());
        }

        // loop through removing any deleted collections
        for (int i = deletedCollections.size() - 1; i >= 0; i--)
        {
            removeCollectionMonitor((String)deletedCollections.get(i));
        }

        // loop through creating an individual monitor for the collection
        for (int i = newCollections.size() - 1; i >= 0; i--)
        {
            addCollectionMonitor((String)newCollections.get(i));
        }
    }

    private void handleCollectionsMonitorElementChange(IElementChange elementChange)
    {
        if (elementChange.getChangeType() == IElementChange.ELEMENT_UPDATED)
        {
            IDeltaElement changeElement = (IDeltaElement)elementChange.getElement();
            IDeltaAttributeSet attrs = (IDeltaAttributeSet)changeElement.getDeltaAttributes();
            String[] mods = attrs.getModifiedAttributesNames();
            updateRuntimeWithChangedConfig(mods, attrs);

            mods = attrs.getNewAttributesNames();
            updateRuntimeWithChangedConfig(mods, attrs);

            mods = attrs.getDeletedAttributesNames();
            updateRuntimeWithDeletedConfig(mods);
        }
    }

    private void updateRuntimeWithChangedConfig(String[] attributeNames, IDeltaAttributeSet attrs)
    {
        try
        {
            for (int i = 0; i < attributeNames.length; i++)
            {
                if (attributeNames[i].compareTo(ICollectionsMonitorConstants.HISTORY_DURATION_HOURS_ATTR) == 0)
                {
                    setHistoryDurationHours((Integer)attrs.getNewValue(ICollectionsMonitorConstants.HISTORY_DURATION_HOURS_ATTR));
                }
                if (attributeNames[i].compareTo(ICollectionsMonitorConstants.SAVE_MONITORED_NOTIFICATIONS_ATTR) == 0)
                {
                    setSaveMonitoredNotifications((Boolean)attrs.getNewValue(ICollectionsMonitorConstants.SAVE_MONITORED_NOTIFICATIONS_ATTR));
                }
                if (attributeNames[i].compareTo(ICollectionsMonitorConstants.SAVE_THRESHOLD_NOTIFICATIONS_ATTR) == 0)
                {
                    setSaveThresholdNotifications((Boolean)attrs.getNewValue(ICollectionsMonitorConstants.SAVE_THRESHOLD_NOTIFICATIONS_ATTR));
                }
                if (attributeNames[i].compareTo(ICollectionsMonitorConstants.COLLECTIONS_ATTR) == 0)
                {
                    updateMonitoredCollections();
                }
                // we have hijacked the NOTIFICATION_SUBSCRIPTION_TIMEOUT_ATTR for the renewal interval
                if (attributeNames[i].equals(ICollectionsMonitorConstants.NOTIFICATION_SUBSCRIPTION_TIMEOUT_ATTR))
                {
                    setNotificationSubscriptionRenewalInterval((Integer)attrs.getNewValue(ICollectionsMonitorConstants.NOTIFICATION_SUBSCRIPTION_TIMEOUT_ATTR));
                }
            }
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Error modifying attributes from configuration change", e, Level.WARNING);
        }
    }

    // Runtime attributes go back to their default value if the
    // corresponding configuration values are removed.

    private void updateRuntimeWithDeletedConfig(String[] attributeNames)
    {
        try
        {
            for (int i = 0; i < attributeNames.length; i++)
            {
                if (attributeNames[i].compareTo(ICollectionsMonitorConstants.HISTORY_DURATION_HOURS_ATTR) == 0)
                {
                    setHistoryDurationHours(ICollectionsMonitorConstants.HISTORY_DURATION_HOURS_DEFAULT);
                }
                if (attributeNames[i].compareTo(ICollectionsMonitorConstants.SAVE_MONITORED_NOTIFICATIONS_ATTR) == 0)
                {
                    setSaveMonitoredNotifications(ICollectionsMonitorConstants.SAVE_MONITORED_NOTIFICATIONS_DEFAULT);
                }
                if (attributeNames[i].compareTo(ICollectionsMonitorConstants.SAVE_THRESHOLD_NOTIFICATIONS_ATTR) == 0)
                {
                    setSaveThresholdNotifications(ICollectionsMonitorConstants.SAVE_THRESHOLD_NOTIFICATIONS_DEFAULT);
                }
                // we have hijacked the NOTIFICATION_SUBSCRIPTION_TIMEOUT_ATTR for the renewal interval
                if (attributeNames[i].compareTo(ICollectionsMonitorConstants.NOTIFICATION_SUBSCRIPTION_TIMEOUT_ATTR) == 0)
                {
                    setNotificationSubscriptionRenewalInterval(new Integer(ICollectionsMonitorConstants.NOTIFICATION_SUBSCRIPTION_TIMEOUT_DEFAULT));
                }
            }
        }
        catch (Exception e)
        {
            super.m_context.logMessage("Error modifying attributes from configuration change", e, Level.WARNING);
        }
    }

    private synchronized void addCollectionMonitor(String collectionConfigID)
    {
        CollectionMonitor collectionMonitor =
            new CollectionMonitor(collectionConfigID, super.m_frameworkContext, this);
        m_collectionMonitors.put(collectionConfigID, collectionMonitor);
    }

    private synchronized void removeCollectionMonitor(String collectionConfigID)
    {
        CollectionMonitor collectionMonitor = (CollectionMonitor)m_collectionMonitors.remove(collectionConfigID);
        collectionMonitor.cleanup();
    }

    private String[] getComponentNames(String collectionID)
    {
        CollectionMonitor monitor = (CollectionMonitor)m_collectionMonitors.get(collectionID);

        if (monitor == null)
        {
            throw new IllegalArgumentException("Unknown collection: collectionID = " + collectionID);
        }

        return monitor.getCollectionComponents();
    }

    /**
     * calculate the appropriate "start time" to pass to the store for retrieval.  Stored events with timestamps
     * equal to or later than the "start time", and earlier or equal to the "end time", will be retrieved.
     *
     * if a non-null "lastRetrievedTimestamp" value has been passed in, that value will be used as the start time UNLESS
     * the time span between that value and the "endTime" exceeds the specified maximum lookback period.
     * The exception to this last rule is the case where the input maximum lookback period is null.
     *
     * @param endTime                value representing the time up to which events will be retrieved from the store
     * @param lastObtainedTimestamp  object containing timestamp of last retrieved event,  and used
     *                               in conjunction with "endTime" to determine the start time from
     *                               which events will be retrieved from the store; the argument may be null,
     *                               and the units are milliseconds (representing a specific time)
     * @param maxLookbackPeriod      object containing maximum period over which events will be retrieved
     *                               from the store; the argument may be null, and the units are milliseconds
     *                               (representing a time period)
     * @return the start time calculated from either the input lastObtainedTimestamp or the maxLookbackPeriod, and the input endTime
     */
    private long determineStartTimeForRetrieval(final long endTime, Long lastObtainedTimestamp, Long maxLookbackPeriod)
    {
        // initialize the start time to be the input value of the "end time"
        long startTime = endTime;

        if (lastObtainedTimestamp != null)
        {
            long lastTimestamp = Math.abs(lastObtainedTimestamp.longValue());  // handle a negative input value for the last received timestamp...
            if (lastTimestamp < endTime)
            {
                // provided that a maximum lookback period was input, check that the calculated lookback period does
                // not exceed the input maximum lookback period
                if (maxLookbackPeriod != null)
                {
                    long earliestLookbackTime = endTime - Math.abs(maxLookbackPeriod.longValue());  // handle negative input value for max lookback period
                    if ( lastTimestamp < earliestLookbackTime )
                    {
                        // if the last timestamp predates the max lookback period, use a start time based on the max lookback period...
                        startTime = earliestLookbackTime;
                    }
                    else
                    {
                        // input max lookback argument was null...
                        startTime = lastTimestamp;
                    }
                }
                else {
                    startTime = lastTimestamp;  // no choice but to use the last received timestamp
                }

            }
            else  // deal with either a "bad" input value for the timestamp (greater than the "end time", or with a timestamp that is equivalent to the "end time"...
            {
                // if a max lookback period was input, use that to determine the start time
                if (maxLookbackPeriod != null)
                {
                    startTime = endTime - Math.abs(maxLookbackPeriod.longValue());
                }

                // otherwise, "start time" will remain equal to "end time"
            }
        }
        else  // input lastObtainedTimestamp argument was null...
        {
            if (maxLookbackPeriod != null)
             {
                startTime = endTime - Math.abs(maxLookbackPeriod.longValue());  // use the absolute value of the input lookback period, just in case caller provided a negative number...
            }

            // note that if the input lookbackPeriod is null, in combination with the input null lastObtainedTimestamp,
            // the start time will be equal to the "end time" - in that case no metrics will be returned, and the returned
            // timestamp value will be equal to the "end time"
        }

        return startTime;
    }

    /**
     * Builds a vector of component names from all specified component collections; each component name will appear
     * once and only once in the returned vector
     *
     * @param collectionIDs  non-null reference to array of component collection identifiers
     * @return vector of all component names in input component collections [will not contain duplicates of component names]
     * @throws IllegalArgumentException if collectionIDs == null
     */
    private Vector buildComponentVectorFromCollectionIDs(String[] collectionIDs)
    {
        // enforce the precondition
        if (collectionIDs == null)
        {
            throw new IllegalArgumentException("Illegal input array of collectionIDs: array reference must not be null");
        }

        // build a list of all the component names based on all of the input CollectionIDs
        Vector compNamesVec = new Vector();
        for (int i = 0, m = collectionIDs.length; i < m; i++)
        {
            String[] componentNames = null;
            try
            {
                 // get the names of the component in the collection
                 componentNames = getComponentNames(collectionIDs[i]);

                 // add the names to the vector [skip over name if it is already in the vector]
                 for (int j = 0, n = componentNames.length; j < n; j++)
                 {
                     if (!compNamesVec.contains(componentNames[j]))
                    {
                        compNamesVec.add(componentNames[j]);
                    }
                 }
            }
            catch (IllegalArgumentException iae)
            {
                if (super.m_state == IComponentState.STATE_ONLINE)
                {
                    // don't log if the component is stopping or is stopped
                    super.m_context.logMessage("Unknown component collection encountered: collectionID = " + collectionIDs[i] + " - it will be ignored.", Level.WARNING);
                }
            }
        }

        return compNamesVec;
    }

    /**
     * Builds an ordered list of notifications, from earliest to latest, for the specified time period and returns an Iterator for the list.
     *
     * It is assumed that:
     *   (a) the input array of names is non-null
     *   (b) the input "lastest" time is LARGER THAN the input "earliest" time
     *
     * The return parameter will never be null, though the returned Iterator may not contain any values
     *
     * @param componentNames  names of components for which notifications will be retrieved [assumed that names are unique, i.e. a component name does not appear in the array more than once]
     * @param latest          time of latest notification to retrieve
     * @param earliest        time of earliest notification to retrieve
     * @return  an Iterator of retrieved notifications for the specified time period, in order from earliest to latest [will never be null, but may be empty]
     */
    private Iterator getOrderedNotifications(String[] componentNames, long latest, long earliest)
    {
        //assert (latest > earliest);
        //assert (componentNames != null);

        // create an array list to hold the retrieved notifications
        ArrayList notificationList = new ArrayList();

        // retrieve the notifications for the specified time period from the store
        Iterator it = m_store.getNotifications(componentNames,latest, earliest);

        // add the notifications to the array list
        while (it.hasNext())
        {
            // append the notification...
            notificationList.add(it.next());
        }

        // determine if list needs to be reversed; if so, reverse the order of the entries in the list
        if (notificationList.size() > 1)
        {
            INotification first = (INotification) notificationList.get(0);
            INotification second = (INotification) notificationList.get(1);

            long firstTimestamp = first.getTimeStamp();
            long secondTimestamp = second.getTimeStamp();

            if (firstTimestamp > secondTimestamp)  // list needs to be reversed to get earliest-to-latest order
            {
                Collections.reverse(notificationList);
            }
        }

        // return the orderer list (return parameter will never be null)
        return notificationList.iterator();
    }

    /**
     * Builds an ordered list of metrics, from earliest to latest, for the specified time period and returns an Iterator for the list.
     *
     * It is assumed that:
     *   (a) the input array of names is non-null
     *   (b) the input "lastest" time is LARGER THAN the input "earliest" time
     *
     * The return parameter will never be null, though the returned Iterator may not contain any values
     *
     * @param componentNames  names of components for which metricss will be retrieved [assumed that names are unique, i.e. a component name does not appear in the array more than once]
     * @param latest          time of latest metric to retrieve
     * @param earliest        time of earliest metric to retrieve
     * @return  an Iterator of retrieved metrics for the specified time period, in order from earliest to latest [will never be null, but may be empty]
     */
    private Iterator getOrderedMetrics(String[] componentNames, long latest, long earliest)
    {
        //assert (latest > earliest);
        //assert (componentNames != null);

        // create an array list to hold the retrieved metrics
        ArrayList metricList = new ArrayList();

        // retrieve the metrics for the specified time period from the store
        Iterator it = m_store.getMetrics(componentNames,latest, earliest);

        // add the metrics to the array list
        while (it.hasNext())
        {
            // append the metric...
            metricList.add(it.next());
        }

        // determine if list needs to be reversed; if so, reverse the order of the entries in the list
        if (metricList.size() > 1)
        {
            IMetric first = (IMetric) metricList.get(0);
            IMetric second = (IMetric) metricList.get(1);

            long firstTimestamp = first.getCurrencyTimestamp();
            long secondTimestamp = second.getCurrencyTimestamp();

            if (firstTimestamp > secondTimestamp)  // list needs to be reversed to get earliest-to-latest order
            {
                Collections.reverse(metricList);
            }
        }

        // return the orderer list (return parameter will never be null)
        return metricList.iterator();
    }

    private class SdfMFTracingIntegration extends com.sonicsw.sdf.AbstractMFComponentTracing
    {
         private boolean m_updateTraceLevelWasCalled;

         SdfMFTracingIntegration()
         {
             super("sonic.mf.monitor." + m_context.getComponentName().getComponentName().replace(' ', '_'), getTraceMaskValues());
             m_updateTraceLevelWasCalled = false;

             setTraceMask();
         }
         
         private void setTraceMask() {
             setTraceMask(new Integer(CollectionsMonitor.this.m_traceMask));
         }

         boolean wasUpdated()
         {
             return m_updateTraceLevelWasCalled;
         }

         @Override
        public void updateTraceLevel(String doiIDNotUsed, HashMap parameters, StringBuffer buffer)
         {
             super.updateTraceLevel(doiIDNotUsed, parameters, buffer);
             m_updateTraceLevelWasCalled = true;
             CollectionsMonitor.this.setTraceMask(getCurrentMask(), true);
         }
    }

	@Override
	public void onMetricStored(IMetric metric) {
		IStatistic stat = m_metricsStoredPerMinute;
		if ( stat != null) {
			stat.updateValue(1);
		}
		updateCollectorMetrics(metric);
	}
	
	private void updateCollectorMetrics(IMetric metric) {
		long latency = System.currentTimeMillis() - metric.getCurrencyTimestamp();
		IStatistic stat = m_minMetricLatencyStatistic;
		if (stat != null) {
			stat.updateValue(latency);
		}
		stat = m_maxMetricLatencyStatistic;
		if (stat != null) {
			stat.updateValue(latency);
		}
		stat = m_averageMetricLatencyStatistic;
		if (stat != null) {
			stat.updateValue(latency);
		}
		stat = m_metricsReceivedPerSecond;
		if (stat != null) {
			stat.updateValue(1);
		}
	}

	@Override
	public void onNotificationStored(INotification notification) {
		IStatistic stat = m_notificationsStoredPerMinute;
		if (stat != null) {
			stat.updateValue(1);
		}
		stat = m_notificationsReceivedPerSecond;
		if (stat != null) {
			stat.updateValue(1);
		}
	}

	@Override
	public void onMetricsOffloaded(IMetric[] metric) {
		long time = System.currentTimeMillis();
		for (int i = 0; i < metric.length; i++) {
			updateCollectorMetrics(metric[i]);
		}
	}

	@Override
	public void onNotificationOffloaded(INotification notification) {
		IStatistic stat = m_notificationsReceivedPerSecond;
		if (stat != null) {
			stat.updateValue(1);
		}
	}

	@Override
	public void setEnterpriseStateAccess(IEnterpriseStateAccess s) {
		m_enterpriseStateAccessor = s;		
	}

	public boolean isEnterprise() {
		return m_enterpriseStateAccessor.isEnterprise();	
	}	
}
