package com.sonicsw.mf.framework.monitor;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.INotificationHandler;
import com.sonicsw.mf.framework.util.NotificationForwarder;
import com.sonicsw.mf.jmx.client.MFNotification;
import com.sonicsw.mf.mgmtapi.config.constants.ICollectionsMonitorConstants;

/**
 * The NotificationManager is responsible for managing notification subscriptions
 * for the notifications handled by a CollectionsMonitor and dispatching those
 * notifications to one or more handlers.
 *
 * Each CollectionsMonitor creates an instance of a NotificationManager to service
 * the CollectionMonitor(s) they create.
 */
final class NotificationManager
implements INotificationHandler
{
    private IFrameworkComponentContext m_context;
    private HashMap m_idMap = new HashMap();
    private long m_notificationSubscriptionRenewalInterval = (long)(ICollectionsMonitorConstants.NOTIFICATION_SUBSCRIPTION_TIMEOUT_DEFAULT * 1000);

    private boolean m_debug = false;

    private static final String[] EMPTY_STRING_ARRAY = IEmptyArray.EMPTY_STRING_ARRAY;

    NotificationManager(IFrameworkComponentContext context)
    {
        m_context = context;
    }

    /**
     * The manager is no longer required and this method is called to
     * release any resources used by this manager.
     */
    void cleanup()
    {
        unsubscribe(null);
        m_idMap.clear();
    }

    /**
     * Add a notification handler.
     *
     * @param notificationSource The canonical name of the component whose notification should
     *                           be subscribed to.
     * @param notificationTypes  The notification types (e.g. "system.state.Online").
     * @param handler            The handler to which received notifications should be dispatched.
     */
    void addNotificationHandler(String notificationSource, String[] notificationTypes, INotificationHandler handler)
    {
        boolean subscribe = false;
        
        synchronized(m_idMap)
        {
            HashMap notificationTypeMap = (HashMap)m_idMap.get(notificationSource);
            if (notificationTypeMap == null)
            {
                notificationTypeMap = new HashMap();
                m_idMap.put(notificationSource, notificationTypeMap);
            }

            for (int i = 0; i < notificationTypes.length; i++)
            {
                HashSet handlerSet = (HashSet)notificationTypeMap.get(notificationTypes[i]);
                if (handlerSet == null)
                {
                    handlerSet = new HashSet();
                    notificationTypeMap.put(notificationTypes[i], handlerSet);
                }
    
                if (!handlerSet.contains(handler))
                {
                    handlerSet.add(handler);
                    subscribe = true;
                }
            }
        }
        
        if (subscribe)
        {
            subscribe(new String[] { notificationSource });
        }
    }

    /**
     * Remove a notification handler.
     *
     * @param notificationSource The canonical name of the component whose notification should
     *                           be subscribed to.
     * @param notificationTypes  The notification types (e.g. "system.state.online").
     * @param handler            The handler to which received notifications should be dispatched.
     */
    synchronized void removeNotificationHandler(String notificationSource, String[] notificationTypes, INotificationHandler handler)
    {
        boolean unsubscribe = false;
        
        synchronized(m_idMap)
        {
            HashMap notificationTypeMap = (HashMap)m_idMap.get(notificationSource);
            if (notificationTypeMap == null)
            {
                return;
            }

            for (int i = 0; i < notificationTypes.length; i++)
            {
                HashSet handlerSet = (HashSet)notificationTypeMap.get(notificationTypes[i]);
                if (handlerSet == null)
                {
                    return;
                }
    
                handlerSet.remove(handler);
                if (handlerSet.isEmpty())
                {
                    notificationTypeMap.remove(notificationTypes[i]);
                    if (notificationTypeMap.isEmpty())
                    {
                        m_idMap.remove(notificationSource);
                        unsubscribe = true;
                    }
                }
            }
        }
        
        if (unsubscribe)
        {
            unsubscribe(new String[] { notificationSource });
        }
    }

    /**
     * Creates or updates the subscriptions with all the subscription sources.
     *
     * This will be called by the CollectionsMonitor once during startup, then by
     * individual CollectionMonitor's as they handle configuration changes that
     * would change the list of notification sources or notifications that should be
     * subscribed to.
     *
     * @param notificationSources A list of notification sources whose notifications
     *                            should be subscribed to. If null, then all sources
     *                            known by this manager will have their subscriptions
     *                            made.
     */
    void subscribe(String[] notificationSources)
    {
        if (notificationSources == null)
        {
            notificationSources = getAllNotificationSources();
        }

        // set the notification subscription timeout on the IFrameworkComponentContext
        m_context.setNotificationSubscriptionRenewalInterval(new Long(m_notificationSubscriptionRenewalInterval));

        // loop through all the requested subscription sources making the subscriptions
        for (int i = 0; i < notificationSources.length; i++)
        {
            HashMap notificationTypeMap = (HashMap)m_idMap.get(notificationSources[i]);
            String[] notificationTypes =
                notificationTypeMap == null ? EMPTY_STRING_ARRAY : (String[])notificationTypeMap.keySet().toArray(EMPTY_STRING_ARRAY);
            if (notificationTypes.length == 0)
            {
                //TBD, 10/18/06: looks like if there are no notification types specified for the specified notification source, the handler for that notification source is removed from the map...
                // Should probably add a comment to that effect and explain why...
                m_context.removeNotificationHandler(notificationSources[i], this);
            }
            else
            {
                //System.out.println("NotificationManager.subscribe(): " + new java.util.Date() + ", notificationSource=" + notificationSources[i] + ", notificationType=" + notificationTypes[0]);
                m_context.addNotificationHandler(notificationSources[i], this, notificationTypes);
            }
        }
    }

    /**
     * Removes the subscriptions from all the subscription sources.
     *
     * This will be called by the CollectionsMonitor once during stop (via cleanup(),
     * or by individual CollectionMonitor's as they handle configuration changes that
     * would change the list of notification sources or notifications that should be
     * subscribed to.
     *
     * @param notificationSources A list of notification sources whose notifications
     *                            should be unsubscribed from. If null, then all sources
     *                            known by this manager will have their subscriptions
     *                            removed.
     */
    void unsubscribe(String[] notificationSources)
    {
        if (notificationSources == null)
        {
            notificationSources = getAllNotificationSources();
        }

        // loop through all the requested subscription sources removing the subscriptions
        for (int i = 0; i < notificationSources.length; i++)
        {
            m_context.removeNotificationHandler(notificationSources[i], this);
        }
    }

    private String[] getAllNotificationSources()
    {
        return (String[])m_idMap.keySet().toArray(EMPTY_STRING_ARRAY);
    }

    synchronized void setNotificationSubscriptionRenewalInterval(long notificationSubscriptionRenewalInterval)
    {
        m_notificationSubscriptionRenewalInterval = notificationSubscriptionRenewalInterval;
        m_context.setNotificationSubscriptionRenewalInterval(new Long(notificationSubscriptionRenewalInterval));
    }

    @Override
    public void handleNotification(INotification notification)
    {
        if (m_debug)
        {
            m_context.logMessage("Notification [" + notification.getType() + "] received from " + notification.getSourceIdentity().getCanonicalName(), Level.TRACE);
        }

        HashMap notificationTypeMap = (HashMap)m_idMap.get(notification.getSourceIdentity().getCanonicalName());

        if (notificationTypeMap == null)
        {
            // the handler(s) were removed
            return;
        }

        HashSet handlerSet = (HashSet)notificationTypeMap.get(notification.getType());
        
        if (handlerSet == null) // check for wildcards
        {
            String notificationType = notification.getType();
            Set keys = notificationTypeMap.keySet();
            Iterator it = keys.iterator();
            while (it.hasNext())
            {
                String key = (String)it.next();
                if (key.endsWith("*"))
                {
                    int index = key.indexOf("*");
                    String typeSubstring = key.substring(0, index - 1);
                    if (notificationType.length() >= index) // only need to check for a match if the length of the input notification type is at least as long as the length of the registered notification type
                    {
                        String notificationTypeSubstring = notificationType.substring(0, index - 1);
                        if (typeSubstring.equals(notificationTypeSubstring))
                        {
                            handlerSet = (HashSet)notificationTypeMap.get(key);
                            break;  // only going to get the first handlerSet that matches...
                        }
                    }
                }
            }
        }
       
        if (handlerSet == null)
        {
            // the handler was removed
            return;
        }

        Object[] handlers = null;
        synchronized(handlerSet)
        {
            handlers = handlerSet.toArray();
        }

        // pass the notification to all the handlers
        for (int i = 0; i < handlers.length; i++)
        {
            if ((handlers[i] instanceof NotificationForwarder) && (notification instanceof MFNotification))
            {
                // clone the notification, and send the clone
                INotification clonedNotification = (INotification) (new MFNotification(((MFNotification)notification)));

                ((INotificationHandler)handlers[i]).handleNotification(clonedNotification);
            }
            else
            {
                ((INotificationHandler)handlers[i]).handleNotification(notification);
            }
        }
    }

    void setDebug(boolean debug) { m_debug = debug; }
}
