package com.sonicsw.mf.framework.monitor.storage;

import java.util.Iterator;

import com.sonicsw.mf.common.metrics.IMetric;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.manager.IStatistic;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.framework.monitor.IHistoryStorageListener;

/**
 * This interface describes a persistent store that the CollectionsMonitor can use
 * for all its historical storage and retrieval requirements.
 */
public interface IHistoryStorage
{
    /**
     * Open the persistent store. If the store does not exist, then the
     * store should be created. Subsequent calls should reopen the store if
     * the store was closed (e.g. due to some error condition that has now
     * been cleared).
     */
    public void open()
    throws StorageException;

    /**
     * Close the persistent store and release all resources associated
     * with the store.
     */
    public void close()
    throws StorageException;

    /**
     * Sets the delta length of time data should be stored before it should
     * be expired.
     */
    public void setExpireAfter(long expireAfter);

    /**
     * Gets the delta length of time data should be stored before it should
     * be expired.
     */
    public long getExpireAfter();

    /**
     * Sets the maximum size to which the sotre is allowed to grow.
     */
    public void setMaxStorageSize(long maxStorageSize);

    /**
     * Gets the maximum size to which the store is allowed to grow.
     */
    public long getMaxStorageSize();

    /**
     * Clears all historical data from the store.
     */
    public void clear()
    throws StorageException;

    /**
     * Stores the given notification if it has not already expired.
     */
    public void storeNotification(INotification notification)
    throws StorageException;

    /**
     * Stores the given metrics if not already stored.
     */
    public void storeMetrics(IMetric[] metrics, String source)
    throws StorageException;

    /**
     * Get notifications that satisfy the given parameters. Notifications will be
     * ordered from most recent to least recent generation time.
     *
     * @param notificationTypes   The (canonical) notification types.
     * @param notificationSources A list of notification sources (the canonical form of component
     *                            runtime identities)
     * @param latest              The most recent notification should have a timestamp <= to this value
     * @param earliest            The least recent notification should have a timestamp >= to this value
     *
     * @return An iterator to a sorted/qualified set of notifications
     */
    public Iterator getNotifications(String[] notificationTypes, String[] notificationSources, long latest, long earliest)
    throws StorageException;

    /**
     * Get notifications that satisfy the given parameters. Notifications will be
     * ordered from most recent to least recent generation time.
     *
     * @param notificationSources A list of notification sources (the canonical form of component runtime
     *                            identities)
     * @param latest     The most recent notification should have a timestamp <= to this value
     * @param earliest   The least recent notification should have a timestamp >= to this value
     *
     * @return An iterator to a sorted/qualified set of notifications
     */
    public Iterator getNotifications(String[] notificationSources, long latest, long earliest)
    throws StorageException;

    /**
     * Get metrics that satisfy the given parameters. Metrics will be
     * ordered from most recent to least recent currency time.
     *
     * @param metricIDs           The metric identities.
     * @param componentIdentities A list of metric sources (the canonical form of component runtime
     *                            identities)
     * @param latest              The most recent metric should have a currency <= to this value
     * @param earliest            The least recent metric should have a timestamp >= to this value
     *
     * @return An iterator to a sorted/qualified set of notifications
     */
    public Iterator getMetrics(IMetricIdentity[] metricIDs, String[] componentIdentities, long latest, long earliest)
    throws StorageException;

    /**
     * Get all metrics that satisfy the given parameters. Metrics will be
     * ordered from most recent to least recent currency time.
     *
     * @param componentIdentities A list of metric sources (the canonical form of component runtime
     *                            identities)
     * @param latest              The most recent metric should have a currency <= to this value
     * @param earliest            The least recent metric should have a timestamp >= to this value
     *
     * @return An iterator to a sorted/qualified set of notifications
     */
    public Iterator getMetrics(String[] componentIdentities, long latest, long earliest)
    throws StorageException;
    
    /**
     * Sets the storage listener
     * 
     * @param listener The listener 
     */
    public void setHistoryStorageListener(IHistoryStorageListener listener);
}
