package com.sonicsw.mf.framework.monitor.storage.fs;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import com.odi.Database;
import com.odi.DatabaseNotFoundException;
import com.odi.DatabaseNotOpenException;
import com.odi.DatabaseRootNotFoundException;
import com.odi.NoTransactionInProgressException;
import com.odi.ObjectStore;
import com.odi.ObjectStoreConstants;
import com.odi.Session;
import com.odi.Transaction;
import com.odi.util.DuplicateIndexException;
import com.odi.util.IndexedCollection;
import com.odi.util.OSTreeSet;
import com.odi.util.query.Query;
import com.sonicsw.mx.util.FIFO;
import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.metrics.IMetric;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.manager.IStatistic;
import com.sonicsw.mf.common.runtime.ICanonicalName;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.monitor.CollectionsMonitor;
import com.sonicsw.mf.framework.monitor.IEventHolder;
import com.sonicsw.mf.framework.monitor.IHistoryStorageListener;
import com.sonicsw.mf.framework.monitor.storage.EventHolder;
import com.sonicsw.mf.framework.monitor.storage.IHistoryStorage;
import com.sonicsw.mf.framework.monitor.storage.StorageException;

/**
 * The default file system storage makes use of the default SonicMQ
 * data store (but we don't expose that /*store).
 */
public final class FSStorage
implements IHistoryStorage
{
    // constants
    public static final long DEFAULT_EXPIRE_AFTER = 48 * 60 * 60 * 1000;
    public static final long MIN_EXPIRE_AFTER = 1 * 60 * 60 * 1000;
    private static final int EXPIRE_BATCH_SIZE = 1000;

    public static final long BYTES_PER_MEGABYTE = (long) 1048576;

    private static final String NOTIFICATIONS = "Notifications";
    private static final String METRICS = "Metrics";
    private static final String NOTIFICATIONS_AND_METRICS_A = "NotificationsAndMetricsA";
    private static final String NOTIFICATIONS_AND_METRICS_B = "NotificationsAndMetricsB";
    private long m_expireAfter = DEFAULT_EXPIRE_AFTER;
    
    public static final int MAX_COMPONENTS_PER_QUERY = Integer.getInteger("sonicsw.mf.monitor.componentsPerQuery", 600).intValue();

    private IComponentContext m_context;
    private File m_directory;
    private Thread m_writerThread;
    private Thread m_expirerThread;
    private long m_maxStorageSize;

    private boolean m_isClosing = false;
    private boolean m_needToRecreateStorage = false;
    
    private Object m_expirerThreadLockObj = new Object();
    private Object m_writerThreadLockObj = new Object();
    
    // declare array to hold Store objects
    private Store[] m_stores = new Store[2];

    private static final int MAX_SIZE = 250;
    private HashSet[] m_metricsRecentlyStored = new HashSet[] { new HashSet(MAX_SIZE), new HashSet(MAX_SIZE) };
    private FIFO m_metricsToBeStored = new FIFO();
    private FIFO m_notificationsToBeStored = new FIFO();

    private static ClassLoader m_ctxClassLoader = FSStorage.class.getClassLoader();
    private static int m_traceMask;
    private IHistoryStorageListener m_historyStorageListener;

    public FSStorage(String rootDirectory, String instanceName, IComponentContext context)
    throws StorageException
    {
        m_context = context;
        ICanonicalName name = context.getComponentName();

        m_directory = new File(rootDirectory + File.separatorChar + name.getDomainName() + '.' + name.getContainerName() + '.' + name.getComponentName());

        if (m_directory.exists())
        {
            if (!m_directory.isDirectory())
            {
                throw new StorageException("Failed storage initialization: bad storage directory [" + m_directory.getPath() + ']');
            }
        }
        else
        {
            if (!m_directory.mkdirs())
            {
                throw new StorageException("Failed storage initialization: unable to create storage directory [" + m_directory.getPath() + ']');
            }
        }

        // create the Store objects...
        m_stores[0] = new Store();
        m_stores[1] = new Store();
    }

    //
    // IHistoryStorage interface implementation
    //
    @Override
    public void open()
    throws StorageException
    {
        ClassLoader ctxClassLoader = Thread.currentThread().getContextClassLoader();
        m_isClosing = false;

        // establish the store
        try
        {
            Thread.currentThread().setContextClassLoader(m_ctxClassLoader);

            // set the db file names
            m_stores[0].setDbName(NOTIFICATIONS_AND_METRICS_A);
            m_stores[1].setDbName(NOTIFICATIONS_AND_METRICS_B);

            // create the notifications/metrics session and set up the notifications/metrics indexed collection
            m_stores[0].open();

            // create writer and expirer threads
            m_writerThread = new Writer();
            m_expirerThread = new Expirer();

            m_writerThread.start();
            m_expirerThread.start();
        }
        catch(Throwable e)
        {
            // try to cleanup as best we can
            cleanup();
            throw new StorageException("Failed to open storage.", e);
        }
        finally
        {
            Thread.currentThread().setContextClassLoader(ctxClassLoader);
        }
    }

    @Override
    public synchronized void close()
    throws StorageException
    {
        cleanup();
    }

    @Override
    public synchronized void setExpireAfter(long expireAfter)
    {
        if (expireAfter < MIN_EXPIRE_AFTER)
        {
            throw new IllegalArgumentException("Can use expiration > " + MIN_EXPIRE_AFTER + " milliseconds");
        }

        m_expireAfter = expireAfter;
    }

    @Override
    public synchronized long getExpireAfter() { return m_expireAfter; }

    /**
     * Sets the maximum size to which the store is allowed to grow.
     */
    @Override
    public void setMaxStorageSize(long maxStorageSize)
    {
        // convert from MB to bytes
        if ( maxStorageSize > (Long.MAX_VALUE/BYTES_PER_MEGABYTE))
        {
            // make sure storage capacity of Long is not exceeded
            m_maxStorageSize = (long) Long.MAX_VALUE;
        }
        else
        {
            m_maxStorageSize = maxStorageSize * BYTES_PER_MEGABYTE;
        }
    }

    /**
     * Gets the maximum size to which the store is allowed to grow.
     */
    @Override
    public long getMaxStorageSize()
    {
        return (m_maxStorageSize/BYTES_PER_MEGABYTE);  // convert back to megabytes
    }

    @Override
    public void clear()
    throws StorageException
    {
        if (m_stores != null)
        {
            clearPSEObjects(NOTIFICATIONS);
            clearPSEObjects(METRICS);
        }
    }

    @Override
    public void finalize() // try to be good about cleaning up !
    {
        cleanup();
    }

    @Override
    public void storeNotification(INotification notification)
    throws StorageException
    {
        synchronized(m_writerThreadLockObj)
        {
            m_writerThreadLockObj.notifyAll();
        }

        synchronized(m_notificationsToBeStored)
        {
            try
            {
                m_notificationsToBeStored.add(new PSENotification(notification));
            }
            catch(Exception e)
            {
                throw new StorageException("Failed to store notification: " + notification.getType(), e);
            }
        }
    }

    @Override
    public void storeMetrics(IMetric[] metrics, String source)
    throws StorageException
    {
        synchronized(m_writerThreadLockObj)
        {
            m_writerThreadLockObj.notifyAll();
        }

        synchronized(m_metricsToBeStored)
        {
            for (int i = 0; i < metrics.length; i++)
            {
                try
                {
                    m_metricsToBeStored.add(new PSEMetric(metrics[i], source));
                }
                catch(Exception e)
                {
                    throw new StorageException("Failed to store metric: " + metrics[i].getMetricIdentity().getName(), e);
                }
            }
        }
    }

    @Override
    public Iterator getNotifications(String[] notificationTypes, String[] notificationSources, long latest, long earliest)
    throws StorageException
    {
        // build base expression
        StringBuffer expression = new StringBuffer();
        expression.append("getTimestamp()<=").append((0 - earliest)).append('L');
        expression.append("&&");
        expression.append("getTimestamp()>=").append((0 - latest)).append('L');
        expression.append("&&(");
        for (int i = 0; i < notificationTypes.length ; i++)
        {
            if (i > 0)
            {
                expression.append("||");
            }
            expression.append("getType().equals(\"").append(notificationTypes[i]).append("\")");
        }
        expression.append(')');

        // run query using base expression and list of component identities
        return aggregateQuerySources(PSENotification.class, expression.toString(), notificationSources, NOTIFICATIONS);
    }

    @Override
    public Iterator getNotifications(String[] notificationSources, long latest, long earliest)
    throws StorageException
    {
        // build base expression
        StringBuffer expression = new StringBuffer();
        expression.append("getStorageTimestamp()>=").append(earliest).append('L');
        expression.append("&&");
        expression.append("getStorageTimestamp()<=").append(latest).append('L');

        // run query using base expression and list of component identities
        return aggregateQuerySources(PSENotification.class, expression.toString(), notificationSources, NOTIFICATIONS);
    }

    @Override
    public Iterator getMetrics(IMetricIdentity[] metricIDs, String[] componentIdentities, long latest, long earliest)
    throws StorageException
    {
        // build base expression
        StringBuffer expression = new StringBuffer();
        expression.append("getTimestamp()<=").append((0 - earliest)).append('L');
        expression.append("&&");
        expression.append("getTimestamp()>=").append((0 - latest)).append('L');
        expression.append("&&(");
        for (int i = 0; i < metricIDs.length ; i++)
        {
            if (i > 0)
            {
                expression.append("||");
            }
            expression.append("getName().equals(\"").append(metricIDs[i].getName()).append("\")");
        }
        expression.append(')');

        // run query using base expression and list of component identities
        return aggregateQuerySources(PSEMetric.class, expression.toString(), componentIdentities, METRICS);
    }

    @Override
    public Iterator getMetrics(String[] componentIdentities, long latest, long earliest)
    throws StorageException
    {
        // build base expression
        StringBuffer expression = new StringBuffer();
        expression.append("getStorageTimestamp()>=").append(earliest).append('L');
        expression.append("&&");
        expression.append("getStorageTimestamp()<=").append(latest).append('L');

        // run query using base expression and list of component identities
        return aggregateQuerySources(PSEMetric.class, expression.toString(), componentIdentities, METRICS);
    }
    
    private Iterator aggregateQuerySources(Class storageClass, String baseExpression, String[] sources, String type)
    {
        List result = new ArrayList();
        int idx = 0;
        
        while (idx < sources.length)
        {
            StringBuffer expression = new StringBuffer(baseExpression);
            expression.append("&&(");
            
            for (int i = 0; i < MAX_COMPONENTS_PER_QUERY && idx < sources.length; i++)
            {
                if (i > 0)
                {
                    expression.append("||");
                }
                expression.append("getSource().equals(\"").append(sources[idx]).append("\")");
                idx++;
            }
            expression.append(')');   

            List partResult = getPSEObjects(storageClass, expression.toString(), type);

            result.addAll(partResult);
        }
        
        return result.iterator();
    }

    //
    // Internal methods
    //
    private synchronized void cleanup()
    {
        m_isClosing = true;

        if (m_expirerThread != null)
        {
            synchronized(m_expirerThreadLockObj)
            {
                m_expirerThreadLockObj.notifyAll();
            }
        }

        if (m_writerThread != null)
        {
            synchronized(m_writerThreadLockObj)
            {
                m_writerThreadLockObj.notifyAll();
            }
        }

        synchronized(m_stores)
        {
            for (int index = 0; index < 2; index++)
            {
                m_stores[index].close();
            }
        }
    }

    private void expireData()
    throws Exception
    {
        long olderThan = System.currentTimeMillis() - m_expireAfter;

        Query query = null;

        // preparatory work
        StringBuffer expression = new StringBuffer();
        expression.append("getTimestamp()>").append(0 - olderThan).append('L');

        query = new Query(PSENotification.class, expression.toString());
        if (m_stores != null)
        {
            expirePSEObjects(query, NOTIFICATIONS);
        }

        query = new Query(PSEMetric.class, expression.toString());
        if (m_stores != null)
        {
            expirePSEObjects(query, METRICS);
        }
    }

    private void expirePSEObjects(Query query, String type)
    {
        long elapsedTime = System.currentTimeMillis();

        synchronized(m_stores)
        {
            if (m_isClosing)
            {
                return;
            }

            ClassLoader ctxClassLoader = Thread.currentThread().getContextClassLoader();
            try
            {
                Thread.currentThread().setContextClassLoader(m_ctxClassLoader);
            }
            catch (Exception e)
            {
                throw new StorageException("Failed to expire/clear history", e);
            }

            for (int index = 0; index < 2; index++)
            {
                if (m_stores[index] == null || m_stores[index].m_session == null)
                {
                    continue;
                }

                try
                {
                    m_stores[index].m_session.join();
                    Transaction txn = null;
                    try
                    {
                        txn = m_stores[index].m_session.currentTransaction();
                    }
                    catch (NoTransactionInProgressException e)
                    {
                        txn = Transaction.begin(ObjectStore.UPDATE);
                    }

                    Set set = (Set)m_stores[index].m_database.getRoot(type);

                    // Query and expire in batches, since we cannot remove
                    // entries from the set while we are iterating through
                    // a query on the set.  Concurrent modifications via
                    // set.remove() can corrupt the query/index iterator,
                    // causing incorrect results or exceptions (Sonic00020288).
                    while (true)
                    {
                        Iterator iterator = query.iterator(set);
                        Object objsToBeExpired[] = new Object[EXPIRE_BATCH_SIZE];
                        int count = 0;
                        while (iterator.hasNext() && (count < EXPIRE_BATCH_SIZE))
                        {
                            objsToBeExpired[count] = iterator.next();
                            count++;
                        }
                        for (int i = 0; i < count; i++)
                        {
                            set.remove(objsToBeExpired[i]);
                            ObjectStore.destroy(objsToBeExpired[i]);
                            objsToBeExpired[i] = null;
                        }
                        if (count < EXPIRE_BATCH_SIZE)
                        {
                            break;
                        }
                    }

                    txn.commit();
                    Transaction.begin(ObjectStore.UPDATE);

                    Session.leave();
                }
                catch (Exception e)
                {
                    Session.leave();
                    Thread.currentThread().setContextClassLoader(ctxClassLoader);
                    m_stores.notifyAll();
                    throw new StorageException("Failed to expire/clear history", e);
                }
            }

            try
            {
                Thread.currentThread().setContextClassLoader(ctxClassLoader);
            }
            catch (Exception e)
            {
                throw new StorageException("Failed to expire/clear history", e);
            }
        }

        if ((m_traceMask & CollectionsMonitor.TRACE_STORAGE_ACTIVITY) > 0)
        {
            StringBuffer traceMessage = new StringBuffer("Storage expire: ");
            traceMessage.append("duration (millis)=").append((System.currentTimeMillis() - elapsedTime));
            traceMessage.append(", type=").append(type);
            m_context.logMessage(traceMessage.toString(), Level.TRACE);
        }
    }

    private void clearPSEObjects(String type)
    {
        synchronized(m_stores)
        {
            if (m_isClosing)
            {
                return;
            }

            ClassLoader ctxClassLoader = Thread.currentThread().getContextClassLoader();
            try
            {
                Thread.currentThread().setContextClassLoader(m_ctxClassLoader);
            }
            catch (Exception e)
            {
                throw new StorageException("Failed to expire/clear history", e);
            }

            for (int index = 0; index < 2; index++)
            {
                if (m_stores[index] == null || m_stores[index].m_session == null)
                {
                    continue;
                }

                Store store = m_stores[index];

                try
                {
                    store.m_session.join();
                    Transaction txn = null;
                    try
                    {
                        txn = store.m_session.currentTransaction();
                    }
                    catch (NoTransactionInProgressException e)
                    {
                        txn = Transaction.begin(ObjectStore.UPDATE);
                    }

                    Set set = null;
                    if (type.equals(NOTIFICATIONS))
                    {
                        set = store.m_notifications;
                    }
                    else
                    {
                        set = store.m_metrics;
                    }

                    Iterator iterator = set.iterator();
                    while (iterator.hasNext())
                    {
                        ObjectStore.destroy(iterator.next());
                    }
                    set.clear();

                    txn.commit();
                    Transaction.begin(ObjectStore.UPDATE);

                    Session.leave();
                }
                catch (Exception e)
                {
                    Session.leave();
                    Thread.currentThread().setContextClassLoader(ctxClassLoader);
                    m_stores.notifyAll();
                    throw new StorageException("Failed to clear history", e);
                }
            }

            try
            {
                Thread.currentThread().setContextClassLoader(ctxClassLoader);
            }
            catch (Exception e)
            {
                throw new StorageException("Failed to expire/clear history", e);
            }
        }
    }

    private void writeQueuedObjects()
    throws StorageException
    {
        int metricsToBeStored = 0;
        int metricsStored = 0;
        int notificationsToBeStored = 0;
        int notificationsStored = 0;

        long elapsedTime = 0;

        synchronized(m_stores)
        {
            if (m_isClosing)
            {
                return;
            }

            // determine if there is enough room in the "active" database to store
            // another metric or notification; if there is not enough room, swap the stores
            if (m_stores[0].m_database.getSizeInBytes() >= m_maxStorageSize)
            {
                swapStores();
            }

            // objects are only *written* to the "active" store, m_stores[0]
            Store store = m_stores[0];

            synchronized(m_metricsToBeStored)
            {
                metricsToBeStored = m_metricsToBeStored.size();
            }
            synchronized(m_notificationsToBeStored)
            {
                notificationsToBeStored = m_notificationsToBeStored.size();
            }
            
            // even if some metrics or notifications got added in the meantime, we will
            // pick them up next cycle
            if (metricsToBeStored == 0 && notificationsToBeStored == 0)
            {
                return;
            }
                    
            ClassLoader ctxClassLoader = Thread.currentThread().getContextClassLoader();

            try
            {
                Thread.currentThread().setContextClassLoader(m_ctxClassLoader);

                Transaction txn = null;
                elapsedTime = System.currentTimeMillis();
                store.m_session.join();
                try
                {
                    txn = store.m_session.currentTransaction();
                }
                catch (NoTransactionInProgressException e)
                {
                    txn = Transaction.begin(ObjectStore.UPDATE);
                }

                Set set = null;

                // Metrics
                synchronized(m_metricsToBeStored)
                {
                    metricsToBeStored = m_metricsToBeStored.size();
                    if (metricsToBeStored > 0)
                    {
                        set = (Set)store.m_database.getRoot(METRICS);

                        Iterator metricObjects = m_metricsToBeStored.iterator();
                        while (metricObjects.hasNext())
                        {
                            PSEMetric pseMetric = (PSEMetric)metricObjects.next();

                            // if its a metric, check to see if an equivalent metric already exists ..  no point writing if its already there!
                            String pseMetricID = pseMetric.getTimestamp() + pseMetric.getName() + pseMetric.getSource();
                            if (!m_metricsRecentlyStored[0].contains(pseMetricID) && !m_metricsRecentlyStored[1].contains(pseMetricID))
                            {
                                if (m_metricsRecentlyStored[0].size() >= MAX_SIZE)
                                {
                                    HashSet hashSet = m_metricsRecentlyStored[1];
                                    m_metricsRecentlyStored[1] = m_metricsRecentlyStored[0];
                                    m_metricsRecentlyStored[0] = hashSet;
                                    m_metricsRecentlyStored[0].clear();
                                }
                                m_metricsRecentlyStored[0].add(pseMetricID);
                                set.add(pseMetric);
                                if (m_historyStorageListener != null)
                                {
                                	m_historyStorageListener.onMetricStored(pseMetric.getMetric());
                                }
                                metricsStored++;
                            }
                        }

                        m_metricsToBeStored.clear();
                    }
                }

                // Notifications
                synchronized(m_notificationsToBeStored)
                {
                    notificationsToBeStored = m_notificationsToBeStored.size();
                    if (notificationsToBeStored > 0)
                    {
                        set = (Set)store.m_database.getRoot(NOTIFICATIONS);

                        Iterator notificationObjects = m_notificationsToBeStored.iterator();
                        while (notificationObjects.hasNext())
                        {
                            PSENotification notification = (PSENotification)notificationObjects.next();
                        	set.add(notification);
                             if (m_historyStorageListener != null)
                            {
                            	 m_historyStorageListener.onNotificationStored(notification.getNotification());
                            }
                            notificationsStored++;
                        }

                        m_notificationsToBeStored.clear();
                    }
                }

                txn.commit(ObjectStore.RETAIN_HOLLOW);
                elapsedTime = System.currentTimeMillis() - elapsedTime;

                if (metricsToBeStored > 0 || notificationsToBeStored > 0)
                {
                    if ((m_traceMask & CollectionsMonitor.TRACE_STORAGE_ACTIVITY) > 0)
                    {
                        StringBuffer traceMessage = new StringBuffer("Storage write: ");
                        traceMessage.append("duration (millis)=").append(elapsedTime);
                        if (metricsToBeStored > 0)
                        {
                            traceMessage.append(", metrics to be stored=").append(metricsToBeStored).append(", metrics actually stored=").append(metricsStored);
                        }
                        if (notificationsToBeStored > 0)
                        {
                            traceMessage.append(", notifications to be stored=").append(notificationsToBeStored).append(", notifications actually stored=").append(notificationsStored);
                        }
                        m_context.logMessage(traceMessage.toString(), Level.TRACE);
                    }
                }

                Transaction.begin(ObjectStore.UPDATE);
                m_needToRecreateStorage = false;
            }
            catch (Exception e)
            {
                throw new StorageException("Failed to store history", e);
            }
            finally
            {
                Session.leave();
                Thread.currentThread().setContextClassLoader(ctxClassLoader);
                m_stores.notifyAll();
            }
        }
    }

    private List getPSEObjects(Class storageClass, String expression, String type)
    {
        Query query = new Query(storageClass, expression);
        List results = new ArrayList();
        Set set = null;

        long elapsedTime = System.currentTimeMillis();

        synchronized(m_stores)
        {
            if (m_isClosing)
            {
                return null;
            }

            ClassLoader ctxClassLoader = Thread.currentThread().getContextClassLoader();
            try
            {
                Thread.currentThread().setContextClassLoader(m_ctxClassLoader);
            }
            catch (Exception e)
            {
                throw new StorageException("Failed to retrieve history: " + expression.toString(), e);
            }

            for (int index = 0; index < 2; index++)
            {
                if (m_stores[index] == null || m_stores[index].m_session == null)
                {
                    continue;
                }

                Store store = m_stores[index];

                try
                {
                    store.m_session.join();
                    set = (Set)store.m_database.getRoot(type);

                    Iterator iterator = query.iterator(set);
                    while (iterator.hasNext())
                    {
                        PSEObject obj = (PSEObject)iterator.next();
                        IEventHolder holder = new EventHolder(obj.getObject(), obj.getStorageTimestamp());
                        results.add(holder);
                    }
                    Session.leave();
                }
                catch (Exception e)
                {
                    Session.leave();
                    Thread.currentThread().setContextClassLoader(ctxClassLoader);
                    m_stores.notifyAll();
                    throw new StorageException("Failed to retrieve history: " + expression.toString(), e);
                }
            }

            try
            {
                Thread.currentThread().setContextClassLoader(ctxClassLoader);
            }
            catch (Exception e)
            {
                throw new StorageException("Failed to retrieve history: " + expression.toString(), e);
            }
        }

        if ((m_traceMask & CollectionsMonitor.TRACE_STORAGE_ACTIVITY) > 0)
        {
            StringBuffer traceMessage = new StringBuffer("Storage read: ");
            traceMessage.append("duration (millis)=").append((System.currentTimeMillis() - elapsedTime));
            traceMessage.append(", type=").append(type);
            traceMessage.append(", query expression='").append(expression).append("'");
            m_context.logMessage(traceMessage.toString(), Level.TRACE);
        }

        return results;
    }


    private void swapStores() throws StorageException
    {
        long elapsedTime = System.currentTimeMillis();
        synchronized(m_stores)
        {
            Object temp = null;

            // swap the stores...
            temp = m_stores[0];
            m_stores[0] = m_stores[1];
            m_stores[1] = (Store) temp;

            m_stores[0].close();
            m_stores[0].removeDbFile();

            try
            {
                m_stores[0].open();
            }
            catch(Throwable e)
            {
                cleanup();
                throw new StorageException("Failed to open storage.", e);
            }

            if ((m_traceMask & CollectionsMonitor.TRACE_STORAGE_ACTIVITY) > 0)
            {
                StringBuffer traceMessage = new StringBuffer("Storage swap: ");
                traceMessage.append("duration (millis)=").append((System.currentTimeMillis() - elapsedTime));
                m_context.logMessage(traceMessage.toString(), Level.TRACE);
            }
        }
    }

    private void clearQueues()
    {
        synchronized(m_notificationsToBeStored)
        {
            m_notificationsToBeStored.clear();
        }

        synchronized(m_metricsToBeStored)
        {
            m_metricsToBeStored.clear();
        }
    }
    private void recoverActiveStore() throws StorageException
    {
        clearQueues();
        synchronized(m_stores)
        {
            if (m_needToRecreateStorage || !reopenActiveStore())
            {
                recreateActiveStore();
            }
        }
    }

    private void recreateActiveStore() throws StorageException
    {
        m_stores[0].forceClose();
        m_stores[0].removeDbFile();

        try
        {
            m_stores[0].open();
            FSStorage.this.m_context.logMessage("Recreated storage", Level.INFO);
        }
        catch(Throwable e)
        {
            cleanup();
            throw new StorageException("recreateActiveStore: Failed to recreate storage.", e);
        }
    }

    private boolean reopenActiveStore()
    {
        m_stores[0].forceClose();

        try
        {
            m_stores[0].open();
            m_needToRecreateStorage = true; // Will be true until we write to the storage and prove it's ok
            FSStorage.this.m_context.logMessage("Recovered and reopened storage", Level.INFO);
            return true;
        }
        catch(Throwable e)
        {
            FSStorage.this.m_context.logMessage("recoverActiveStore: Failed to recover storage", e, Level.WARNING);
            return false;
        }
    }

    static byte[] toBytes(Serializable object)
    throws IOException
    {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(baos);

        oos.writeObject(object);
        return baos.toByteArray();
    }

    static Object fromBytes(byte[] bytes)
    throws Exception
    {
        ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        ObjectInputStream ois = new ObjectInputStream(bais);

        return ois.readObject();
    }

    private class Store implements ObjectStoreConstants
    {
        Session m_session;
        Database m_database;
        String m_dbFileName;
        Set m_notifications;
        Set m_metrics;

        void setDbName( String dbFileName )
        {
            m_dbFileName = dbFileName;
        }

        void removeDbFile()
        {
            // remove the existing physical db file
            File dbFile = new File(m_directory, m_dbFileName + ".odb");

            if (dbFile.exists())
            {
                //PSE2's storage is a directory - we have to remove it and its content
                if (dbFile.isDirectory())
                {
                    removePSE2Storage(dbFile);
                    return;
                }

                if (!dbFile.delete())
                {
                    throw new StorageException("Failed to remove database file: " + m_dbFileName + ".odb");
                }
            }

            // remove the existing physical log file
            File logFile = new File(m_directory, m_dbFileName + ".log");

            if (logFile.exists())
            {
                if (!logFile.delete())
                {
                    throw new StorageException("Failed to remove log file: " + m_dbFileName + ".log");
                }
            }
        }

        void setupIndexedCollection(Database database, Class storageClass, String[] indexMemberNames, String rootName)
        throws IOException
        {
            IndexedCollection indexedCollection = null;
            try
            {
                indexedCollection = (IndexedCollection)database.getRoot(rootName);
                return; // the indexed collection already exists
            }
            catch (DatabaseRootNotFoundException de)
            {
                // the indexed collection doesn't exist yet, fall through to create it
            }

            indexedCollection = new OSTreeSet(database);
            database.createRoot(rootName, indexedCollection);

            // add the indexes (only the timestamp needs to be ordered)
            for (int i = 0; i < indexMemberNames.length; i++)
            {
                try
                {
                    indexedCollection.addIndex(storageClass, indexMemberNames[i], (i == 0), true);
                }
                catch(DuplicateIndexException e) {  } // this would occur if the last open aborted and is ok to ignore
            }
        }

        void open()
        {
            Properties props = new Properties();

            // disable String pooling
            props.setProperty("com.odi.stringPoolSize", "0");

            m_session = Session.create(null, props);
            try
            {
                // perform leave and then join, to ensure current thread is joined to current session...
                Session.leave();
                m_session.join();

                Transaction.setDefaultRetain(RETAIN_HOLLOW);

                File dbFile = new File(m_directory, m_dbFileName + ".odb");
                if (dbFile.exists() && !dbFile.isDirectory())
                {
                    throw new StorageException(m_directory.getCanonicalPath() + " is obsolete. It must be removed before the Collection Monitored is started");
                }
                try
                {
                    m_database = Database.open(dbFile.getCanonicalPath(), UPDATE);
                }
                catch (DatabaseNotFoundException e)
                {
                    m_database = Database.create(dbFile.getCanonicalPath(), ALL_READ | ALL_WRITE);
                }

                Transaction.begin(ObjectStore.UPDATE);
                setupIndexedCollection(m_database, PSENotification.class, new String[]{ "getTimestamp()", "getStorageTimestamp()", "getType()", "getSource()" }, NOTIFICATIONS);
                m_notifications = (Set) m_database.getRoot(NOTIFICATIONS);
                try { m_session.currentTransaction().commit(); } catch(Exception e) { }

                Transaction.begin(ObjectStore.UPDATE);
                setupIndexedCollection(m_database, PSEMetric.class, new String[]{ "getTimestamp()", "getStorageTimestamp()", "getName()", "getSource()" }, METRICS);
                m_metrics = (Set) m_database.getRoot(METRICS);
                try { m_session.currentTransaction().commit(); } catch(Exception e) { }

                // create the next open transaction
                Transaction.begin(ObjectStore.UPDATE);
            }
            catch (StorageException e)
            {
                throw e;
            }
            catch (Exception e)
            {
                throw new StorageException("Failed storage initialization: unable to open store", e);
            }
            finally
            {
                Session.leave();
            }
        }

        void close()
        {
            if (m_session != null)
            {
                try
                {
                    // first, perform a blind "leave" on the session (in case an implicit "join" was made prior to invoking this method)
                    Session.leave();

                    // now, join the session
                    m_session.join();
                    try
                    {
                        Transaction txn = m_session.currentTransaction();
                        txn.commit();
                    }
                    catch (NoTransactionInProgressException e){}

                    try
                    {
                        if (m_database != null)
                        {
                            m_database.close();
                        }
                    }
                    catch (DatabaseNotOpenException e){}
                }
                finally
                {
                    Session.leave();
                    m_session.terminate();
                    m_session = null;
                    m_database = null;
                    m_notifications = null;
                    m_metrics = null;
                    // m_dbFileName intentionally remains set for next use of this Store object...
                }
            }
        }

        void forceClose()
        {
            if (m_session != null)
            {
                try
                {
                    // first, perform a blind "leave" on the session (in case an implicit "join" was made prior to invoking this method)
                    Session.leave();

                    try
                    {
                        // now, join the session
                        m_session.join();
                        Transaction txn = m_session.currentTransaction();
                        txn.commit();
                    }
                    catch (Throwable e){}

                    try
                    {
                        if (m_database != null)
                        {
                            m_database.close();
                        }
                    }
                    catch (Throwable e){}
                }
                finally
                {
                    Session.leave();
                    m_session.terminate();
                    m_session = null;
                    m_database = null;
                    m_notifications = null;
                    m_metrics = null;
                    // m_dbFileName intentionally remains set for next use of this Store object...
                }
            }
        }

        private void removePSE2Storage(File dir)
        {
            String[] fileList = dir.list();
            for (int i = 0; i < fileList.length; i++)
            {
                File crntFile = new File(dir, fileList[i]);
                if (crntFile.isDirectory())
                {
                    removePSE2Storage(crntFile);
                }
                else if (!crntFile.delete())
                {
                    throw new StorageException("Could not delete  the database file: " +  crntFile.getAbsolutePath());
                }
            }
            if (!dir.delete())
            {
                throw new StorageException("Could not delete the database directory  " +  dir.getAbsolutePath());
            }
        }

    }  //class Store

    private class Writer
    extends Thread
    {
        private static final long WRITER_BATCHING_PERIOD = 1000;

        private Writer()
        {
            super("CollectionsMonitor Storage Writer");
            super.setDaemon(true);
        }

        @Override
        public void run()
        {
            while (true)
            {
                if (m_isClosing)
                {
                    return;
                }

                synchronized(m_writerThreadLockObj)
                {
                    try
                    {
                        // the thread will get woken prematurely under two conditions:
                        // - the store is closing
                        // - something will be queued for writing
                        m_writerThreadLockObj.wait(1000);
                    }
                    catch (InterruptedException e) { }
                }

                if (m_isClosing)
                {
                    return;
                }

                // now sleep a short period to potentially multiple objects to be queued for writing to
                // intoduce some efficiency in the storage (through batching of writes into larger transactions)
                try
                {
                    Thread.sleep(WRITER_BATCHING_PERIOD);
                }
                catch (InterruptedException e) { }

                if (m_isClosing)
                {
                    return;
                }
                try
                {
                    FSStorage.this.writeQueuedObjects();
                }
                catch(Exception e)
                {
                    FSStorage.this.m_context.logMessage("Failed to write history to storage", e, Level.WARNING);
                    try
                    {
                        FSStorage.this.recoverActiveStore();
                    }
                    catch (StorageException se)
                    {
                        FSStorage.this.m_context.logMessage("Failed to recover storage", se, Level.SEVERE);
                        throw se;
                    }
                }
            }
        }
    }

    private class Expirer
    extends Thread
    {
        private static final long EXPIRATION_PERIOD = 30000;

        private Expirer()
        {
            super("CollectionsMonitor Storage Expirer");
            super.setDaemon(true);
        }

        @Override
        public void run()
        {
            while (true)
            {

                if (m_isClosing)
                {
                    return;
                }

                synchronized(m_expirerThreadLockObj)
                {
                    try
                    {
                        m_expirerThreadLockObj.wait(EXPIRATION_PERIOD);
                    }
                    catch (InterruptedException e) { }
                }

                if (m_isClosing)
                {
                    return;
                }

                try
                {
                    FSStorage.this.expireData();
                    m_needToRecreateStorage = false;
                }
                catch(Exception e)
                {
                    FSStorage.this.m_context.logMessage("Failed to delete expired history", e, Level.WARNING);

                    try
                    {
                        FSStorage.this.recoverActiveStore();
                    }
                    catch (StorageException se)
                    {
                        FSStorage.this.m_context.logMessage("Failed to recover storage", se, Level.SEVERE);
                    }

                }
            }
        }
    }

    public static void setTraceMask(Integer traceMask)
    {
        m_traceMask = traceMask.intValue();
    }

	@Override
	public void setHistoryStorageListener(IHistoryStorageListener listener) {
		m_historyStorageListener = listener;
	}
}

