package com.sonicsw.mf.framework.monitor.storage.jdbc;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.metrics.IHistoricalMetric;
import com.sonicsw.mf.common.metrics.IMetric;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.MetricsFactory;
import com.sonicsw.mf.common.metrics.manager.IStatistic;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.monitor.IEventHolder;
import com.sonicsw.mf.framework.monitor.IHistoryStorageListener;
import com.sonicsw.mf.framework.monitor.storage.EventHolder;
import com.sonicsw.mf.framework.monitor.storage.IHistoryStorage;
import com.sonicsw.mf.framework.monitor.storage.StorageException;

/**
 * The default file system storage makes use of the default SonicMQ
 * relational data store (but we don't expose that relational store).
 */
public final class JDBCStorage
implements IHistoryStorage
{
    // constants
    public static final long DEFAULT_EXPIRE_AFTER = 48 * 60 * 60 * 1000;
    public static final long MIN_EXPIRE_AFTER = 1 * 60 * 60 * 1000;

    private static final String JDBC_DRIVER = "com.borland.datastore.jdbc.DataStoreDriver";
    private static final String JDBC_URL_PREFIX = "jdbc:borland:dslocal:";

    private long m_expireAfter = DEFAULT_EXPIRE_AFTER;

    private String m_url;
    private String m_user;
    private String m_password;
    private String m_blobSQLDatatype;
    private String m_tableNotFound;
    private String m_uniqueConstraint;
    private String m_tablePrefix;

    private Connection m_notificationConnection;
    private Connection m_metricConnection;

    private IComponentContext m_context;
    private Runnable m_expirationTask;
	private IHistoryStorageListener m_listener;

    public JDBCStorage(String driver, String url, String user, String password, String blobSQLDatatype, String tableNotFound, String uniqueConstraint, String instanceName, IComponentContext context)
    throws StorageException
    {
        m_context = context;
        m_url = url;
        m_user = user;
        m_password = password;

        m_blobSQLDatatype = blobSQLDatatype;
        m_tableNotFound = tableNotFound;
        m_uniqueConstraint = uniqueConstraint;

        m_tablePrefix = instanceName.toUpperCase();

        // establish the store
        try
        {
            // JDBC connection setup
            Class.forName(driver);
        }
        catch(Exception e)
        {
            // try to cleanup as best we can
            cleanup();
            throw new StorageException("Failed stoarge initialization", e);
        }
    }

    //
    // IHistoryStorage interface implementation
    //

    @Override
    public void open()
    throws StorageException
    {
        // cleanup in case this is a subsequent (re)open .. e.g. the connection failed and
        // an administrator has correction the situation and want to get the CM running again
        cleanup();

        // establish the store
        try
        {
            // JDBC connection setup
            m_notificationConnection = DriverManager.getConnection(m_url, m_user, m_password);
            m_notificationConnection.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
            m_notificationConnection.setAutoCommit(true);
            m_metricConnection = DriverManager.getConnection(m_url, m_user, m_password);
            m_metricConnection.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
            m_metricConnection.setAutoCommit(true);

            // JDBC table setup if required
            if (isNewStore())
            {
                m_context.logMessage("Existing history storage not found, creating new tables", Level.INFO);
                setupNewStore();
            }

            // create an expiration task and schedule it
            m_expirationTask = new Runnable()
            {
                @Override
                public void run()
                {
                    try
                    {
                        if (!m_notificationConnection.isClosed())
                        {
                            expireData(System.currentTimeMillis() - m_expireAfter);
                            JDBCStorage.this.m_context.scheduleTask(JDBCStorage.this.m_expirationTask, new java.util.Date(System.currentTimeMillis() + 30000));
                        }
                    }
                    catch(Exception e)
                    {
                        JDBCStorage.this.m_context.logMessage("Failed to delete expired history", e, Level.WARNING);
                    }
                }
            };
            // schedule it a first time, after that it will reschedule itself
            m_context.scheduleTask(m_expirationTask, new java.util.Date(System.currentTimeMillis() + 30000));
        }
        catch(Exception e)
        {
            // try to cleanup as best we can
            cleanup();
            throw new StorageException("Failed to open storage.", e);
        }
    }

    @Override
    public synchronized void close()
    throws StorageException
    {
        cleanup();
    }

    @Override
    public synchronized void setExpireAfter(long expireAfter)
    {
        if (expireAfter < MIN_EXPIRE_AFTER)
        {
            throw new IllegalArgumentException("Can use expiration > " + MIN_EXPIRE_AFTER + " milliseconds");
        }

        m_expireAfter = expireAfter;
    }

    @Override
    public synchronized long getExpireAfter() { return m_expireAfter; }

    /**
     * Sets the maximum size to which the sotre is allowed to grow.
     */
    @Override
    public void setMaxStorageSize(long maxStorageSize)
    {
        throw new RuntimeException("A maximum storage size may not be set for a JDBC store.");
    }

    /**
     * Gets the maximum size to which the store is allowed to grow.
     */
    @Override
    public long getMaxStorageSize()
    {
        throw new RuntimeException("A JDBC store may not have a maximum storage size");
    }

    @Override
    public void clear()
    throws StorageException
    {
        try
        {
            expireData(Long.MAX_VALUE);
        }
        catch(Exception e)
        {
            throw new StorageException("Storage clear failure.", e);
        }
    }

    @Override
    public void finalize() // try to be good about cleaning up !
    {
        cleanup();
    }

    @Override
    public void storeNotification(INotification notification)
    throws StorageException
    {
        synchronized(m_notificationConnection)
        {
            PreparedStatement stmt = null;
            try
            {
                StringBuilder sbQuery = new StringBuilder("INSERT INTO ");
                sbQuery.append(m_tablePrefix).append("notifications VALUES (?, ?, ?, ?)");
                
                stmt = m_notificationConnection.prepareStatement(sbQuery.toString());
                stmt.setLong(1, notification.getTimeStamp());
                stmt.setString(2, notification.getType());
                stmt.setString(3, notification.getSourceIdentity().getCanonicalName());
                stmt.setBytes(4, toBytes(notification));
                stmt.executeUpdate();
            }
            catch(Exception e)
            {
                if (e instanceof SQLException)
                {
                    if (((SQLException)e).getSQLState().equals(this.m_uniqueConstraint))
                    {
                        return;
                    }
                }
                throw new StorageException("Storage insertion failure.", e);
            }
            finally
            {
                try { stmt.close(); } catch(Exception e) { }
            }
        }
        if (m_listener != null)
        {
            m_listener.onNotificationStored(notification);
        }
    }

    @Override
    public void storeMetrics(IMetric[] metrics, String source)
    throws StorageException
    {
        synchronized(m_metricConnection)
        {
            for (int i = 0; i < metrics.length; i++)
            {
                PreparedStatement stmt = null;
                try
                {
                    StringBuilder sbQuery = new StringBuilder("INSERT INTO ");
                    sbQuery.append(m_tablePrefix).append("metrics VALUES (?, ?, ?, ?)");
                    
                    stmt = m_metricConnection.prepareStatement(sbQuery.toString());
                    stmt.setLong(1, metrics[i].getCurrencyTimestamp());
                    stmt.setString(2, metrics[i].getMetricIdentity().getName());
                    stmt.setString(3, source);
                    stmt.setBytes(4, toBytes(MetricsFactory.createMetric(source, metrics[i])));
                    stmt.executeUpdate();
                }
                catch(Exception e)
                {
                    if (e instanceof SQLException)
                    {
                        if (((SQLException)e).getSQLState().equals(this.m_uniqueConstraint))
                        {
                            return;
                        }
                    }
                    throw new StorageException("Storage insertion failure.", e);
                }
                finally
                {
                    try { stmt.close(); } catch(Exception e) { }
                }
                if (m_listener != null)
                {
                    m_listener.onMetricStored(metrics[i]);
                }
            }
        }
    }

    @Override
    public Iterator getNotifications(String[] notificationTypes, String[] notificationSources, long latest, long earliest)
    throws StorageException
    {
        StringBuffer query = new StringBuffer();
        // SELECT clause
        query.append("SELECT notification FROM " + m_tablePrefix + "notifications");
        // WHERE clause
        query.append(" WHERE");
        query.append(" currency <= " + latest + " AND");
        query.append(" currency >= " + earliest + " AND");
        query.append(" (");
        for (int i = notificationTypes.length - 1; i >= 0; i--)
        {
            query.append("type = '" + notificationTypes[i] + '\'');
            if (i > 0)
            {
                query.append(" OR ");
            }
        }
        query.append(")");
        query.append(" AND");
        query.append(" (");
        for (int i = notificationSources.length - 1; i >= 0; i--)
        {
            query.append("source = '" + notificationSources[i] + '\'');
            if (i > 0)
            {
                query.append(" OR ");
            }
        }
        query.append(")");
        // ORDER BY clause
        query.append(" ORDER BY currency DESC");

        PreparedStatement stmt = null;
        try
        {
            synchronized(m_notificationConnection)
            {
                stmt = prepareStatement(m_notificationConnection, query.toString());
                ResultSet results = stmt.executeQuery();

                ArrayList notifications = new ArrayList();
                while (results.next())
                {
                    byte[] bytes = results.getBytes(1);
                    try
                    {
                        //notifications.add(fromBytes(bytes));
                        INotification notification = (INotification) fromBytes(bytes);
                        IEventHolder holder = new EventHolder(notification, notification.getTimeStamp());
                        notifications.add(holder);
                    }
                    catch(Exception e)
                    {
                        throw new StorageException("Storage retrieval failure.", e);
                    }
                }
                results.close();

                return notifications.iterator();
            }
        }
        catch(SQLException e)
        {
            throw new StorageException("Storage retrieval failure.", e);
        }
        finally
        {
            try { stmt.close(); } catch(Exception e) { }
        }
    }

    @Override
    public Iterator getNotifications(String[] notificationSources, long latest, long earliest)
    throws StorageException
    {
        StringBuffer query = new StringBuffer();
        // SELECT clause
        query.append("SELECT notification FROM " + m_tablePrefix + "notifications");
        // WHERE clause
        query.append(" WHERE");
        query.append(" currency <= " + latest + " AND");
        query.append(" currency >= " + earliest + " AND");
        query.append("type = *");
        query.append(" AND");
        query.append(" (");
        for (int i = notificationSources.length - 1; i >= 0; i--)
        {
            query.append("source = '" + notificationSources[i] + '\'');
            if (i > 0)
            {
                query.append(" OR ");
            }
        }
        query.append(")");

        // ORDER BY clause
        query.append(" ORDER BY currency DESC");

        PreparedStatement stmt = null;
        try
        {
            synchronized(m_notificationConnection)
            {
                stmt = prepareStatement(m_notificationConnection, query.toString());
                ResultSet results = stmt.executeQuery();

                ArrayList notifications = new ArrayList();
                while (results.next())
                {
                    byte[] bytes = results.getBytes(1);
                    try
                    {
                        //notifications.add(fromBytes(bytes));
                        INotification notification = (INotification) fromBytes(bytes);
                        IEventHolder holder = new EventHolder(notification, notification.getTimeStamp());
                        notifications.add(holder);
                    }
                    catch(Exception e)
                    {
                        throw new StorageException("Storage retrieval failure.", e);
                    }
                }
                results.close();

                return notifications.iterator();
            }
        }
        catch(SQLException e)
        {
            throw new StorageException("Storage retrieval failure.", e);
        }
        finally
        {
            try { stmt.close(); } catch(Exception e) { }
        }
    }

    @Override
    public Iterator getMetrics(IMetricIdentity[] metricIDs, String[] componentIdentities, long latest, long earliest)
    throws StorageException
    {
        StringBuffer query = new StringBuffer();
        // SELECT clause
        query.append("SELECT metric FROM " + m_tablePrefix + "metrics");
        // WHERE clause
        query.append(" WHERE");
        query.append(" currency <= " + latest + " AND");
        query.append(" currency >= " + earliest + " AND");
        query.append(" (");
        for (int i = metricIDs.length - 1; i >= 0; i--)
        {
            query.append("name = '" + metricIDs[i].getName() + '\'');
            if (i > 0)
            {
                query.append(" OR ");
            }
        }
        query.append(")");
        query.append(" AND");
        query.append(" (");
        for (int i = componentIdentities.length - 1; i >= 0; i--)
        {
            query.append("source = '" + componentIdentities[i] + '\'');
            if (i > 0)
            {
                query.append(" OR ");
            }
        }
        query.append(")");
        // ORDER BY clause
        query.append(" ORDER BY currency DESC");

        PreparedStatement stmt = null;
        try
        {
            synchronized(m_metricConnection)
            {
                stmt = prepareStatement(m_metricConnection, query.toString());
                ResultSet results = stmt.executeQuery();

                ArrayList metrics = new ArrayList();
                while (results.next())
                {
                    byte[] bytes = results.getBytes(1);
                    try
                    {
                        //metrics.add(fromBytes(bytes));
                        IHistoricalMetric metric = (IHistoricalMetric) fromBytes(bytes);
                        IEventHolder holder = new EventHolder(metric, metric.getCurrencyTimestamp());
                        metrics.add(holder);
                    }
                    catch(Exception e)
                    {
                        throw new StorageException("Storage retrieval failure.", e);
                    }
                }
                results.close();

                return metrics.iterator();
            }
        }
        catch(SQLException e)
        {
            throw new StorageException("Storage retrieval failure.", e);
        }
        finally
        {
            try { stmt.close(); } catch(Exception e) { }
        }
    }

    @Override
    public Iterator getMetrics(String[] componentIdentities, long latest, long earliest)
    throws StorageException
    {
        StringBuffer query = new StringBuffer();
        // SELECT clause
        query.append("SELECT metric FROM " + m_tablePrefix + "metrics");
        // WHERE clause
        query.append(" WHERE");
        query.append(" currency <= " + latest + " AND");
        query.append(" currency >= " + earliest + " AND");
        query.append("name = *");
        query.append(" AND");
        query.append(" (");
        for (int i = componentIdentities.length - 1; i >= 0; i--)
        {
            query.append("source = '" + componentIdentities[i] + '\'');
            if (i > 0)
            {
                query.append(" OR ");
            }
        }
        query.append(")");
        // ORDER BY clause
        query.append(" ORDER BY currency DESC");

        PreparedStatement stmt = null;
        try
        {
            synchronized(m_metricConnection)
            {
                stmt = prepareStatement(m_metricConnection, query.toString());
                ResultSet results = stmt.executeQuery();

                ArrayList metrics = new ArrayList();
                while (results.next())
                {
                    byte[] bytes = results.getBytes(1);
                    try
                    {
                        //metrics.add(fromBytes(bytes));
                        IHistoricalMetric metric = (IHistoricalMetric) fromBytes(bytes);
                        IEventHolder holder = new EventHolder(metric, metric.getCurrencyTimestamp());
                        metrics.add(holder);
                    }
                    catch(Exception e)
                    {
                        throw new StorageException("Storage retrieval failure.", e);
                    }
                }
                results.close();

                return metrics.iterator();
            }
        }
        catch(SQLException e)
        {
            throw new StorageException("Storage retrieval failure.", e);
        }
        finally
        {
            try { stmt.close(); } catch(Exception e) { }
        }
    }

    //
    // Internal methods
    //

    private void cleanup()
    {
        if (m_expirationTask != null)
        {
            m_context.cancelTask(m_expirationTask);
        }
        if (m_notificationConnection != null)
        {
            try { m_notificationConnection.close(); } catch(Throwable e) { }
        }
        if (m_metricConnection != null)
        {
            try { m_metricConnection.close(); } catch(Throwable e) { }
        }
    }

    /**
     * Implements a crude method to detect if the database has been setup
     */
    private boolean isNewStore()
    throws Exception
    {
        PreparedStatement stmt = null;

        // create the notifications table and add its index
        try
        {
            stmt = prepareStatement(m_notificationConnection, 
                                    "SELECT currency FROM " + m_tablePrefix + "notifications WHERE currency > " + Long.MAX_VALUE);
            stmt.executeUpdate();
            return false;
        }
        catch(SQLException e)
        {
            return !(e.getSQLState().equals(m_tableNotFound));
        }
        finally
        {
            try {
            	if (stmt != null) {
            		stmt.close();	
            	}            	 
            } catch(Exception e) { }
        }
    }

    private void setupNewStore()
    throws Exception
    {
        PreparedStatement stmt_notif_tab = null;
        PreparedStatement stmt_notif_idx = null;
        PreparedStatement stmt_metrics_tab = null;
        PreparedStatement stmt_metrics_idx = null;
        try
        {
            // these are here to aid development when we have a problem that leaves the db in a bad state
            //try { stmt.executeUpdate("DROP TABLE " + m_tablePrefix + "notifications"); } catch(Exception e) { }
            //try { stmt.executeUpdate("DROP TABLE " + m_tablePrefix + "metrics"); } catch(Exception e) { }

            // create the notifications table and add its index
            StringBuilder query = new StringBuilder("CREATE TABLE ");
            query.append(m_tablePrefix).append("notifications ").append("(currency VARCHAR(24) NOT NULL,");
            query.append(" type VARCHAR(128) NOT NULL,").append(" source VARCHAR(128) NOT NULL,");
            query.append(" notification ").append(m_blobSQLDatatype).append(" NOT NULL,");
            query.append(" PRIMARY KEY (currency, type, source))");
            String createTableQuery = query.toString();
            
            query = new StringBuilder("CREATE INDEX notificationsIdx ON ");
            query.append(m_tablePrefix).append("notifications ").append("(currency DESC, type, source)");
            String createIndexQuery = query.toString();
            
            stmt_notif_tab = prepareStatement(m_notificationConnection, createTableQuery);
            stmt_notif_tab.executeUpdate();
            stmt_notif_idx = prepareStatement(m_notificationConnection, createIndexQuery);
            stmt_notif_idx.executeUpdate();

            // create the metrics table and add its index
            query = new StringBuilder("CREATE TABLE ");
            query.append(m_tablePrefix).append("metrics ").append("(currency VARCHAR(24) NOT NULL,");
            query.append(" name VARCHAR(256) NOT NULL,").append(" source VARCHAR(128) NOT NULL,");
            query.append(" metric ").append(m_blobSQLDatatype).append(" NOT NULL,");
            query.append(" PRIMARY KEY (currency, name, source))");
            createTableQuery = query.toString();

            query = new StringBuilder("CREATE INDEX metricsIdx ON ");
            query.append(m_tablePrefix).append("metrics ").append("(currency DESC, name, source)");
            createIndexQuery = query.toString();
            
            stmt_metrics_tab = prepareStatement(m_metricConnection, createTableQuery);
            stmt_metrics_tab.executeUpdate();
            stmt_metrics_idx = prepareStatement(m_metricConnection, createIndexQuery);
            stmt_metrics_idx.executeUpdate();
        }
        finally
        {
            try { stmt_notif_tab.close(); } catch(Exception e) { }
            try { stmt_notif_idx.close(); } catch(Exception e) { }
            try { stmt_metrics_tab.close(); } catch(Exception e) { }
            try { stmt_metrics_idx.close(); } catch(Exception e) { }
        }
    }

    private void expireData(long olderThan)
    throws Exception
    {
        PreparedStatement stmt_notif = null;
        PreparedStatement stmt_metrics = null;
        try
        {
            stmt_notif = prepareStatement(m_notificationConnection,
                                          "DELETE FROM " + m_tablePrefix + "notifications WHERE currency < " + olderThan);
            stmt_notif.executeUpdate();
            stmt_metrics = prepareStatement(m_metricConnection,
                                            "DELETE FROM " + m_tablePrefix + "metrics WHERE currency < " + olderThan);
            stmt_notif.executeUpdate();
        }
        finally
        {
            try { stmt_notif.close(); } catch(Exception e) { }
            try { stmt_metrics.close(); } catch(Exception e) { }
        }
    }

    private byte[] toBytes(Serializable object)
    throws Exception
    {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(baos);

        oos.writeObject(object);
        return baos.toByteArray();
    }

    private Object fromBytes(byte[] bytes)
    throws Exception
    {
        ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        ObjectInputStream ois = new ObjectInputStream(bais);

        return ois.readObject();
    }
    
    private PreparedStatement prepareStatement(Connection conn, String sql) throws SQLException {
        return conn.prepareStatement(sql);
    }

	@Override
	public void setHistoryStorageListener(IHistoryStorageListener listener) {
		m_listener = listener;
		
	}
}
