package com.sonicsw.mf.framework.security;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;

import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.NotModifiedAttException;
import com.sonicsw.mf.common.dirconfig.DirectoryDoesNotExistException;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.IPermissionsManager;

abstract class AbstractPermissionsMap
extends HashMap
{
    private static final boolean DEBUG = System.getProperty("com.sonicsw.debugFGS") != null;
    
    AuthenticationPrincipalMap m_authenticationPrincipalMap;
    
    IFrameworkComponentContext m_context;
    private String m_configID;
    
    static final int ALLOW = 1;
    static final int DENY = -1;
    static final int NONE = 0;

    AbstractPermissionsMap(IFrameworkComponentContext context, AuthenticationPrincipalMap authenticationPrincipalMap, String configID)
    {
        m_context = context;
        m_authenticationPrincipalMap = authenticationPrincipalMap;
        m_configID = configID;
        
        IElement permissions = null;
        try
        {
            permissions = context.getConfiguration(configID, true);
        }
        catch(MFRuntimeException e)
        {
            if (e.getLinkedException() == null || !(e.getLinkedException() instanceof DirectoryDoesNotExistException))
             {
                throw e;
            // otherwise fall through ... permissions will be null
            }
        }
        
        if (permissions == null)
        {
            return;
        }
        
        initPermissionsMap(permissions);
    }
    
    private void initPermissionsMap(IElement permissions)
    {
        IAttributeSet attributes = permissions.getAttributes();
        HashMap pathMap = attributes.getAttributes();
        
        Iterator pathIterator = pathMap.entrySet().iterator();
        while (pathIterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)pathIterator.next();
            addPermissions((String)entry.getKey(), (IAttributeSet)entry.getValue());
        }
    }
    
    public synchronized void handleElementChange(final IElementChange elementChange)
    {
        IBasicElement element = elementChange.getElement();
        
        // only handle changes to the element this map maps on to
        if (!element.getIdentity().getName().equals(m_configID))
        {
            return;
        }

        switch (elementChange.getChangeType())
        {
            case IElementChange.ELEMENT_UPDATED:
            {
                // can only be a change in password so ignore
                IDeltaElement changeElement = (IDeltaElement)element;
                IDeltaAttributeSet changes = (IDeltaAttributeSet)changeElement.getDeltaAttributes();
                handleNewPermissions(changes);
                handleModifiedPermissions(changes);
                handleDeletedPermissions(changes);
                break;
            }
            case IElementChange.ELEMENT_ADDED:
            {
                // this would occur when permissions checking has just been enabled and the first set
                // of permissions (e.g. default permissions are being added)
                initPermissionsMap((IElement)element);
                break;
            }
            case IElementChange.ELEMENT_DELETED:
            {
                // this should not occur, but defensive programming
                break;
            }
            case IElementChange.ELEMENT_REPLACED:
            {
                // will never occur
                break;
            }
        }
    }
    
    private void handleNewPermissions(IDeltaAttributeSet changes)
    {
        String[] newPermissions = changes.getNewAttributesNames();
        
        for (int i = 0; i < newPermissions.length; i++)
        {
            try
            {
                IAttributeSet pathPermissionsAttrs = (IAttributeSet)changes.getNewValue(newPermissions[i]);
                addPermissions(newPermissions[i], pathPermissionsAttrs);
            }
            catch (NotModifiedAttException e)
            {
                // should not happen!!!
                e.printStackTrace();
            }
        }
    }
    
    private void handleModifiedPermissions(IDeltaAttributeSet changes)
    {
        String[] modifiedPermissions = changes.getModifiedAttributesNames();
        
        for (int i = 0; i < modifiedPermissions.length; i++)
        {
            try
            {
                IDeltaAttributeSet pathPermissionsAttrs = (IDeltaAttributeSet)changes.getNewValue(modifiedPermissions[i]);
                handleNewPermission(modifiedPermissions[i], pathPermissionsAttrs);
                handleModifiedPermission(modifiedPermissions[i], pathPermissionsAttrs);
                handleDeletedPermission(modifiedPermissions[i], pathPermissionsAttrs);
            }
            catch (NotModifiedAttException e)
            {
                // should not happen!!!
                e.printStackTrace();
            }
        }
    }
        
    private void handleDeletedPermissions(IDeltaAttributeSet changes)
    {
        String[] deletedPermissions = changes.getDeletedAttributesNames();
        
        for (int i = 0; i < deletedPermissions.length; i++)
        {
            deletePermissions(deletedPermissions[i]);
        }
    }
    
    private void handleNewPermission(String path, IDeltaAttributeSet changes)
    {
        String[] newPermissions = changes.getNewAttributesNames();
        
        for (int i = 0; i < newPermissions.length; i++)
        {
            try
            {
                addPermission(path, newPermissions[i], (IAttributeSet)changes.getNewValue(newPermissions[i]));
            }
            catch (NotModifiedAttException e)
            {
                // should not happen!!!
                e.printStackTrace();
            }
        }
    }
    
    private void handleModifiedPermission(String path, IDeltaAttributeSet changes)
    {
        String[] modifiedPermissions = changes.getModifiedAttributesNames();
        
        for (int i = 0; i < modifiedPermissions.length; i++)
        {
            try
            {
                Object permissionAttrs = changes.getNewValue(modifiedPermissions[i]);
                if (DEBUG)
                {
                    System.out.println("AbstractPermissionsMap.handleModifiedPermission(): [use deletePermission() & addPermission()]");
                }
                deletePermission(path, modifiedPermissions[i]);
                if (permissionAttrs instanceof IAttributeSet)
                {
                    addPermission(path, modifiedPermissions[i], (IAttributeSet)permissionAttrs);
                }
                else
                {
                    addPermission(path, modifiedPermissions[i], (IDeltaAttributeSet)permissionAttrs);
                }
            }
            catch (NotModifiedAttException e)
            {
                // should not happen!!!
                e.printStackTrace();
            }
        }
    }
        
    private void handleDeletedPermission(String path, IDeltaAttributeSet changes)
    {
        String[] deletedPermissions = changes.getDeletedAttributesNames();
        
        for (int i = 0; i < deletedPermissions.length; i++)
        {
            deletePermission(path, deletedPermissions[i]);
        }
    }

    private synchronized void addPermissions(String escapedPath, IAttributeSet permissionsSet)
    {
        HashMap permissionMap = (HashMap)permissionsSet.getAttributes();
        
        if (permissionMap.isEmpty()) // no policies defined for this path
        {
            super.remove(escapedPath);
            return;
        }
        
        Iterator permissionIterator = permissionMap.entrySet().iterator();
        while (permissionIterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)permissionIterator.next();
            Object value = entry.getValue();
            if (value instanceof IAttributeSet)
            {
                addPermission(escapedPath, (String)entry.getKey(), (IAttributeSet)value);
            }
        }
    }

    private synchronized void addPermission(String escapedPath, String principal, IAttributeSet permissionAttributes)
    {
        addPermission(escapedPath, principal, (String)permissionAttributes.getAttribute("VALUE"));
    }
    
    private synchronized void addPermission(String escapedPath, String principal, IDeltaAttributeSet permissionAttributes)
    throws NotModifiedAttException
    {
        addPermission(escapedPath, principal, (String)permissionAttributes.getNewValue("VALUE"));
    }
    
    private synchronized void addPermission(String escapedPath, String principal, String delimitedValue)
    {
        HashMap permissions = (HashMap)super.get(escapedPath);
        if (permissions == null)
        {
            permissions = new HashMap();
            super.put(escapedPath, permissions);
        }
        
        Permission permission = new Permission(delimitedValue);
        permissions.put(principal, permission);
        
        if (DEBUG)
        {
            System.out.println("AbstractPermissionsMap.addPermission(): path=" + unescapePath(escapedPath) + ", principal=" + principal + ", scope=" + permission.permissionScope + ", permission=" + permission.permissionValue);
        }
    }

    private synchronized void deletePermissions(String escapedPath)
    {
        super.remove(escapedPath);
        
        if (DEBUG)
        {
            System.out.println("AbstractPermissionsMap.deletePermissions(): path=" + unescapePath(escapedPath));
        }
    }
    
    private synchronized void deletePermission(String escapedPath, String principal)
    {
        HashMap permissions = (HashMap)super.get(escapedPath);
        if (permissions == null)
         {
            return; // but should not happen
        }
        
        permissions.remove(principal);
        
        if (DEBUG)
        {
            System.out.println("AbstractPermissionsMap.deletePermission(): path=" + unescapePath(escapedPath) + ", principal=" + principal);
        }
    }
    
    boolean hasPermission(String principal, String path, int requiredPermission, int scope)
    {
        // get the list of groups a user belongs to
        String[] groups = m_authenticationPrincipalMap.getGroups(principal);
        
        return hasPermission(principal, path, groups, requiredPermission, scope);
    }
    
    boolean hasPermission(String principal, String path, String[] groups, int requiredPermission, int scope)
    {
        int result = DENY;
        while (true)
        {
            result = checkPermission(principal, path, groups, requiredPermission, scope);
            if (result == ALLOW)
            {
                return true;
            }
            if (result == DENY)
            {
                return false;
            }
            if (path.equals("/"))
            {
                break;
            }
            path = getParentPath(path);
        }
        
        // default is no permission
        return false;
    }
    
    
    int checkPermission(String principal, String path, String[] groups, int requiredPermission, int scope)
    {
        int result = NONE;
        
        try
        {
            HashMap permissions = (HashMap)super.get(escapePath(path));
            if (permissions != null && !permissions.isEmpty())
            {
                Permission permission =  null;
    
                // look for a user based permission first
                permission = (Permission)permissions.get(principal);
                if (permission != null)
                {
                    result = checkPermission(permission, scope, requiredPermission);
                    if (result != NONE)
                    {
                        return result;
                    }
                }
                
                // else look for group based permissions that the user belongs to
                boolean foundDeny = false;
                for (int i = 0; i < groups.length; i++)
                {
                    permission = (Permission)permissions.get(groups[i]);
                    if (permission == null)
                    {
                        continue;
                    }
                    result = checkPermission(permission, scope, requiredPermission);
                    if (result == ALLOW)
                    {
                        return ALLOW;
                    }
                    if (result == DENY)
                    {
                        foundDeny = true;
                    }
                }
                if (foundDeny)
                {
                    return DENY;
                }
            }
            
            return result;
        }
        finally
        {
            if (DEBUG)
            {
                System.out.println("AbstractPermissionsMap.checkPermission(): principal=" + principal +
                                   ", path=" + path +
                                   ", requiredPermission=" + requiredPermission +
                                   ", scope=" + scope +
                                   ", result=" + result);
            }
        }
    }
    
    /**
     * 
     * @param permission
     * @param scope
     * @param requiredPermission
     * @return -1 deny, 0 no permission defined, 1 allow
     */
    int checkPermission(Permission permission, int scope, int requiredPermission)
    {
        if ((permission.permissionScope & scope) != scope)
         {
            return NONE; // no matching scope on the permission
        }
        
        if ((permission.permissionValue & requiredPermission) > 0)
         {
            return ALLOW; // we have a matching allow permission
        }
        
        if ((permission.permissionValue & (requiredPermission << 1)) > 0)
         {
            return DENY; // we have a matching deny permission
        }
        
        return NONE;
    }
    
    /**
     * Utility method to get the parent folder for the given logical storage name. The logical storage
     * name could be a configuration (no trailing '/') or a folder (with a trailing '/').
     */
    String getParentPath(String path)
    {
        if (path.endsWith("/"))
        {
            path = path.substring(0, path.length() - 1);
        }

        return path.substring(0, path.lastIndexOf('/') + 1);
    }
    
    final class Permission
    {
        int permissionValue;
        int permissionScope;
        
        private Permission(String delimitedValue)
        {
            StringTokenizer tokenizer = new StringTokenizer(delimitedValue, ":");
            
            try
            {
                permissionScope = Integer.parseInt(tokenizer.nextToken());
            }
            catch(NumberFormatException e) { } // default 0
            
            try
            {
                permissionValue = Integer.parseInt(tokenizer.nextToken());
            }
            catch(NumberFormatException e) { } // default 0
        }
    }

    void cleanup()
    {
        try
        {
            m_context.getConfiguration(m_configID, false);
        }
        catch(MFRuntimeException e)
        {
            if (e.getLinkedException() == null && !(e.getLinkedException() instanceof DirectoryDoesNotExistException))
             {
                throw e;
            // else no permissions were ever configured!
            }
        }
        super.clear();
    }
    
    private String escapePath(String path)
    {
        path = path.replaceAll(IPermissionsManager.PATH_DELIMITER, IPermissionsManager.ESCAPED_PATH_DELIMITER);
        path = path.replaceAll(IPermissionsManager.SPACE_CHARACTER, IPermissionsManager.ESCAPED_SPACE_CHARACTER);
        return path;
    }
    
    private String unescapePath(String escapedPath)
    {
        escapedPath = escapedPath.replaceAll(IPermissionsManager.ESCAPED_PATH_DELIMITER, IPermissionsManager.PATH_DELIMITER);
        escapedPath = escapedPath.replaceAll(IPermissionsManager.ESCAPED_SPACE_CHARACTER, IPermissionsManager.SPACE_CHARACTER);
        return escapedPath;
    }
}
