package com.sonicsw.mf.framework.security;

import java.lang.reflect.Array;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import javax.management.Attribute;
import javax.management.MBeanInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectInstance;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.xml.DOMConfigurator;
import org.apache.xerces.impl.dv.util.Base64;

import com.sonicsw.mf.common.IXMLFragment;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IFSElementChange;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.security.IConfigurePermissionBits;
import com.sonicsw.mf.common.security.IManagePermissionBits;
import com.sonicsw.mf.common.util.ObjectNameHelper;
import com.sonicsw.mf.framework.IAuditManager;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.agent.TaskScheduler;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDomainConstants;
import com.sonicsw.mf.mgmtapi.runtime.IDirectoryServiceProxy;

public class AuditManager
implements IAuditManager
{
    private static final boolean DEBUG = System.getProperty("sonicsw.mf.debugFGS") != null;

    // use the system property "com.sonicsw.auditExt" to tweak auditing behavior (not all MBeans expose action operations as ACTION or ACTION_INFO
    // and some
    private static final String EXTENDED_AUDIT_BEHAVIOR_PROPERTY = "sonicsw.mf.auditExt";
    private static final HashMap<String, ArrayList<String>> EXTENDED_AUDIT_BEHAVIOR_MAP = new HashMap<String, ArrayList<String>>();
    
    private static final String DOMAIN_CONFIG_ID = "/domain/domain";
    private static final String CONFIGURE_LOGGER = "sonicmf.audit.configure";
    private static final String MANAGE_LOGGER = "sonicmf.audit.manage";
    
    private IFrameworkComponentContext m_context;
    private MBeanServer m_mBeanServer;
    private String m_containerConfigID;
    
    private boolean m_auditConfigureEvents = false;
    private boolean m_auditManageEvents = false;
    private boolean m_enableCentralizedAudit = false;
    
    private static final int MAX_EVENT_QUEUE_SIZE = 250;
    private ArrayList<Runnable> m_configureEventQueue = new ArrayList<Runnable>();
    private ArrayList<Runnable> m_manageEventQueue = new ArrayList<Runnable>();
    private Thread m_configureEventLogger;
    private Thread m_manageEventLogger;
    
    private String m_defaultAuditConfig;
    private String m_localAuditConfig;
    private String m_lastAuditConfig = "";
    
    private static final String[] CONFIGURE_ACTIONS = new String[] { "create", "update", "delete", "rename" };
    public static final int CREATE_ACTION = 0;
    public static final int UPDATE_ACTION = 1;
    public static final int DELETE_ACTION = 2;
    public static final int RENAME_ACTION = 3;
    
    private static final String ANONYMOUS_USER = "anonymous";
    
    private String m_agentManager;

    private static final String RECORD_AUDIT_EVENT_OPERATION_NAME = "recordAuditEvent";
    private static final String[] RECORD_AUDIT_EVENT_OPERATION_SIGNATURE = new String[] { String.class.getName(), String.class.getName(), String.class.getName() };

    private final SimpleDateFormat m_dateTimeFormatter = new SimpleDateFormat("yy/MM/dd HH:mm:ss");
    static
    {
        String auditExt = System.getProperty(EXTENDED_AUDIT_BEHAVIOR_PROPERTY);
        if (auditExt != null)
        {
            StringTokenizer nameAuditingPairs = new StringTokenizer(auditExt, ";");
            while (nameAuditingPairs.hasMoreTokens())
            {
                StringTokenizer nameAuditingPair = new StringTokenizer(nameAuditingPairs.nextToken(), "|");
                String name = nameAuditingPair.nextToken();
                if (name.length() > 0)
                {
                    String auditingString = nameAuditingPair.nextToken();
                    if (auditingString.length() > 0)
                    {
                        ArrayList<String> auditableItems = new ArrayList<String>();
                        StringTokenizer auditingItems = new StringTokenizer(auditingString, ",");
                        while (auditingItems.hasMoreTokens())
                        {
                            auditableItems.add(auditingItems.nextToken());
                        }
                        EXTENDED_AUDIT_BEHAVIOR_MAP.put(name, auditableItems);
                    }
                }
            }
        }
    }

    public AuditManager(IFrameworkComponentContext context, MBeanServer mBeanServer)
    {
        m_context = context;
        m_mBeanServer = mBeanServer;
        
        m_agentManager = m_context.getComponentName().getDomainName() + '.' + "AGENT MANAGER" + IComponentIdentity.DELIMITED_ID_PREFIX + "AGENT MANAGER";
        
        initAuditManager();
    }

    private void initAuditManager()
    {
        IElement domainConfiguration = null;
        try
        {
            domainConfiguration = m_context.getConfiguration(DOMAIN_CONFIG_ID, true);
        }
        catch(MFRuntimeException e) { } // thrown when the directory is not found
        if (domainConfiguration == null)
        {
            return;
        }
            
        IAttributeSet domainAttributes = domainConfiguration.getAttributes();
        
        Boolean auditConfigureEvents = (Boolean)domainAttributes.getAttribute(IDomainConstants.AUDIT_CONFIGURE_EVENTS_ATTR);
        m_auditConfigureEvents = auditConfigureEvents == null ? false : auditConfigureEvents.booleanValue();
        if (m_auditConfigureEvents && m_configureEventLogger == null)
        {
            m_configureEventLogger = new AsyncEventLogger("Configure", m_configureEventQueue);
            m_configureEventLogger.start();
            if (m_context.getContainer().isHostingComponent(IDirectoryServiceProxy.GLOBAL_ID))
            {
                m_context.getContainer().logMessage(null, "Auditing of configure events enabled", Level.CONFIG);
            }
        }
        if (!m_auditConfigureEvents && m_configureEventLogger != null)
        {
            m_configureEventLogger.interrupt();
            m_configureEventLogger = null;
            if (m_context.getContainer().isHostingComponent(IDirectoryServiceProxy.GLOBAL_ID))
            {
                m_context.getContainer().logMessage(null, "Auditing of configure events disabled", Level.CONFIG);
            }
        }
        
        Boolean auditManageEvents = (Boolean)domainAttributes.getAttribute(IDomainConstants.AUDIT_MANAGE_EVENTS_ATTR);
        m_auditManageEvents = auditManageEvents == null ? false : auditManageEvents.booleanValue();
        if (m_auditManageEvents && m_manageEventLogger == null)
        {
            m_manageEventLogger = new AsyncEventLogger("Manage", m_manageEventQueue);
            m_manageEventLogger.start();
            m_context.getContainer().logMessage(null, "Auditing of manage events enabled", Level.CONFIG);
        }
        if (!m_auditManageEvents && m_manageEventLogger != null)
        {
            m_manageEventLogger.interrupt();
            m_manageEventLogger = null;
            m_context.getContainer().logMessage(null, "Auditing of manage events disabled", Level.CONFIG);
        }
        
        Boolean enableCentralizedAudit = (Boolean)domainAttributes.getAttribute(IDomainConstants.ENABLE_CENTRALIZED_AUDIT_ATTR);
        m_enableCentralizedAudit = enableCentralizedAudit == null ? IDomainConstants.ENABLE_CENTRALIZED_AUDIT_DEFAULT : enableCentralizedAudit.booleanValue();
        
        m_defaultAuditConfig = (String)domainAttributes.getAttribute(IDomainConstants.DEFAULT_MANAGEMENT_AUDIT_CONFIG_ATTR);
        if (m_defaultAuditConfig != null && m_defaultAuditConfig.length() == 0)
        {
            m_defaultAuditConfig = null;
        }
        
        // now get the local containers audit destination configuration
        IElement containerConfiguration = m_context.getConfiguration(true);
        m_containerConfigID = containerConfiguration.getIdentity().getName();
        IAttributeSet containerAttrs = containerConfiguration.getAttributes();
        m_localAuditConfig = (String)containerAttrs.getAttribute(IContainerConstants.MANAGEMENT_AUDIT_CONFIG_ATTR);
        if (m_localAuditConfig != null && m_localAuditConfig.length() == 0)
        {
            m_localAuditConfig = null;
        }
        
        String currentAuditConfig = m_localAuditConfig == null || m_localAuditConfig.length() == 0 ? m_defaultAuditConfig : m_localAuditConfig;
        initLog4j(currentAuditConfig);
    }
    
    private void initLog4j(String auditConfig)
    {
        if (!m_auditConfigureEvents && !m_auditManageEvents)
        {
            return;
        }
        
        if (auditConfig == null)
        {
            if (m_lastAuditConfig != null)
            {
                m_context.getContainer().logMessage(null, "Default audit destination(s) configuration will be used", Level.CONFIG);
            }
        }
        else
        {
            if (auditConfig.equals(m_lastAuditConfig))
             {
                return; // no change
            }
            
            if (m_lastAuditConfig == null || !auditConfig.equals(m_lastAuditConfig))
            {
                m_context.getContainer().logMessage(null, "Audit destination(s) configuration file used: " + auditConfig, Level.CONFIG);
            }
        }
        
        // unregister any previous interest
        if (m_lastAuditConfig != null && m_lastAuditConfig.length() > 0 && !m_lastAuditConfig.equals(auditConfig))
        {
            URL url = null;
            try
            {
                url = new URL(m_lastAuditConfig);
                if (url.getProtocol().equals("sonicfs"))
                {
                    m_context.unregisterFileChangeInterest(m_lastAuditConfig);
                }
            }
            catch (Throwable e) { }
        }
        
        m_lastAuditConfig = auditConfig;
        
        if (auditConfig == null || auditConfig.length() == 0)
        {
            PropertyConfigurator.configure(createDefaultAuditConfig());
            return;
        }
        
        URL url = null;
        try
        {
            url = new URL(auditConfig);
        }
        catch (Throwable e)
        {
            m_context.getContainer().logMessage(null, "Exception occurred while attempting to load log4j configuration file " + auditConfig + ", trace follows...", e, Level.SEVERE);
        }

        // register interest in changes to the file if its a sonicfs:/// URL
        if (null != url && "sonicfs".equals(url.getProtocol()))
        {
            m_context.registerFileChangeInterest(auditConfig);
        }
        
        try
        {
            if (auditConfig.endsWith(".xml"))
            {
                DOMConfigurator.configure(url);
            }
            else
            {
                PropertyConfigurator.configure(url);
            }
        }
        catch (Throwable e)
        {
            m_context.getContainer().logMessage(null, "Exception occurred while attempting to load log4j configuration file " + auditConfig + ", trace follows...", e, Level.SEVERE);
            initLog4j(null);
        }
        return;
    }

    public synchronized void handleElementChange(IElementChange elementChange)
    {
        IElementIdentity id = elementChange.getElement().getIdentity();
        String configID = id.getName();
        
        if (configID.startsWith(DOMAIN_CONFIG_ID) || elementChange.getElement().getIdentity().getName().equals(m_containerConfigID) ) // domain or container changes?
        {
            if (elementChange.getChangeType() == IElementChange.ELEMENT_UPDATED)
            {
                initAuditManager();
            }
        }
    }
    
    public synchronized void handleFileChange(IFSElementChange fileChange)
    {
        if (m_lastAuditConfig != null && m_lastAuditConfig.length() > 0)
        {
            URL url = null;
            try
            {
                url = new URL(m_lastAuditConfig);
            }
            catch (Throwable e)
            {
                m_context.getContainer().logMessage(null, "Exception occurred while handling a dynamic configuration change " + m_lastAuditConfig + ", trace follows...", e, Level.SEVERE);
            }
            
            if (fileChange.getChangeType() == IFSElementChange.ELEMENT_DELETED)
            {
                m_context.getContainer().logMessage(null, "Audit configuration file " + m_lastAuditConfig + " has been deleted; correct before container restart", Level.WARNING);
                return;
            }
            
            if (url != null && fileChange.getLogicalPath().equals(url.getPath()))
            {
                initAuditManager();
            }
        }
    }

    private void appendPath(StringBuilder sb, String path, String type)
    {
        if ((path != null) && (path.length() > 0))
        {
            sb.append("<audit:path type=\"").append(type).append("\">").append(path).append("</audit:path>");
        }
    }
    
    @Override
    public void recordConfigureEvent(String[] logicalPaths, String[] storagePaths, int[] actions, String[] details)
    {
        if (!m_auditConfigureEvents)
        {
            return;
        }
        
        final StringBuilder sb = new StringBuilder();
        
        addEventElementHeader(sb, "info");
        
        for (int i = 0; i < logicalPaths.length; i++)
        {
            sb.append("<audit:configure>");
            appendPath(sb, logicalPaths[i], "logical");
            appendPath(sb, storagePaths[i], "storage");
            sb.append("<audit:action>").append(CONFIGURE_ACTIONS[actions[i]]).append("</audit:action>");
            sb.append("<audit:details>").append(details[i]).append("</audit:details>");
            sb.append("</audit:configure>");
        }
        
        addEventElementTrailer(sb);
        
        recordEvent(CONFIGURE_LOGGER, true, m_configureEventQueue, sb.toString());
    }
    
    @Override
    public void recordConfigurePermissionDeniedEvent(String logicalPath, String storagePath, int requiredPermission)
    {
        if (!m_auditConfigureEvents)
        {
            return;
        }
        
        final StringBuilder sb = new StringBuilder();
        
        addEventElementHeader(sb, "warning");
        
        sb.append("<audit:configurePermissionDenied").append(" requiredPermission=\"").append(getConfigurePermissionDescription(requiredPermission)).append("\">");
        appendPath(sb, logicalPath, "logical");
        appendPath(sb, storagePath, "storage");
        sb.append("</audit:configurePermissionDenied>");
        
        addEventElementTrailer(sb);
        
        recordEvent(CONFIGURE_LOGGER, true, m_configureEventQueue, sb.toString());
    }
    
    @Override
    public void recordManageEvent(ObjectName objectName, MBeanInfo mBeanInfo, String operationName, Object[] params, Object returnValue)
    {
        if (!m_auditManageEvents)
        {
            return;
        }
        
        if (mBeanInfo == null)
        {
            if (DEBUG)
            {
                System.out.println("AuditManager.recordManageEvent(): Called with MBeanInfo, stack dump follows...");
                Thread.dumpStack();
            }
        }

        // only record actions
        if (!isAuditableAction(objectName, mBeanInfo, operationName))
        {
            return;
        }
            
        StringBuilder sb = new StringBuilder();
        
        addEventElementHeader(sb, "info");
        
        sb.append("<audit:manage>");
        
        sb.append("<audit:target>").append(objectName== null ? "[MBeanServer]" : objectName.getCanonicalName()).append("</audit:target>");
        sb.append("<audit:operation>").append(operationName).append("</audit:operation>");
        
        for (int i = 0; i < params.length; i++)
        {
            sb.append("<audit:param index=\"").append(i).append("\">");
            appendXMLFragment(sb, params[i]);
            sb.append("</audit:param>");
        }

        if (returnValue != null)
        {
            sb.append("<audit:return>");
            appendXMLFragment(sb, returnValue);
            sb.append("</audit:return>");
        }
        
        sb.append("</audit:manage>");
        
        addEventElementTrailer(sb);
        
        recordEvent(MANAGE_LOGGER, false, m_manageEventQueue, sb.toString());
    }

    @Override
    public void recordManagePermissionDeniedEvent(ObjectName objectName, String operationName, int requiredPermission)
    {
        if (!m_auditManageEvents)
        {
            return;
        }
        
        final StringBuilder sb = new StringBuilder();
        
        addEventElementHeader(sb, "warning");
        
        sb.append("<audit:managePermissionDenied").append(" requiredPermission=\"").append(getManagePermissionDescription(requiredPermission)).append("\">");
        sb.append("<audit:target>").append(objectName== null ? "[MBeanServer]" : objectName.getCanonicalName()).append("</audit:target>");
        sb.append("<audit:operation>").append(operationName).append("</audit:operation>");
        sb.append("</audit:managePermissionDenied>");
        
        addEventElementTrailer(sb);
        
        recordEvent(MANAGE_LOGGER, false, m_manageEventQueue, sb.toString());
    }

    private Properties createDefaultAuditConfig()
    {
        String auditFilePrefix = m_context.getComponentName().getDomainName() + '.' + m_context.getComponentName().getContainerName() + '.';
        
        Properties log4jProperties = new Properties();
        
        // null appender for root
        log4jProperties.setProperty("log4j.rootLogger", "DEBUG, ROOT");
        log4jProperties.setProperty("log4j.appender.ROOT", "org.apache.log4j.varia.NullAppender");
        log4jProperties.setProperty("log4j.appender.ROOT.layout", "org.apache.log4j.PatternLayout");
        log4jProperties.setProperty("log4j.appender.ROOT.layout.ConversionPattern", "%n");

        log4jProperties.setProperty("log4j.logger.sonicmf.audit.configure", "DEBUG, CONFIGURE");
        log4jProperties.setProperty("log4j.appender.CONFIGURE", "org.apache.log4j.DailyRollingFileAppender");
        log4jProperties.setProperty("log4j.appender.CONFIGURE.layout", "com.sonicsw.mf.framework.security.AuditXMLLayout");
        log4jProperties.setProperty("log4j.appender.CONFIGURE.layout.PrettyXML", "false");
        log4jProperties.setProperty("log4j.appender.CONFIGURE.File", auditFilePrefix + "configure.audit");
        log4jProperties.setProperty("log4j.appender.CONFIGURE.BufferedIO", "false");
        log4jProperties.setProperty("log4j.appender.CONFIGURE.DatePattern", "'.'yyyy-MM-dd");
        log4jProperties.setProperty("log4j.appender.CONFIGURE.ImmediateFlush", "true");
        log4jProperties.setProperty("log4j.additivity.CONFIGURE", "true");

        log4jProperties.setProperty("log4j.logger.sonicmf.audit.manage", " DEBUG, MANAGE");
        log4jProperties.setProperty("log4j.appender.MANAGE", "org.apache.log4j.DailyRollingFileAppender");
        log4jProperties.setProperty("log4j.appender.MANAGE.layout", "com.sonicsw.mf.framework.security.AuditXMLLayout");
        log4jProperties.setProperty("log4j.appender.MANAGE.layout.PrettyXML", "false");
        log4jProperties.setProperty("log4j.appender.MANAGE.File", auditFilePrefix + "manage.audit");
        log4jProperties.setProperty("log4j.appender.MANAGE.BufferedIO", "false");
        log4jProperties.setProperty("log4j.appender.MANAGE.DatePattern", "'.'yyyy-MM-dd");
        log4jProperties.setProperty("log4j.appender.MANAGE.ImmediateFlush", "true");
        log4jProperties.setProperty("log4j.additivity.MANAGE", "true");
        
        return log4jProperties;
    }

    private String getConfigurePermissionDescription(int requiredPermission)
    {
        switch(requiredPermission)
        {
            case IConfigurePermissionBits.ALLOW_READ:
                return "Read";
            case IConfigurePermissionBits.ALLOW_WRITE:
                return "Write";
            case IConfigurePermissionBits.ALLOW_DELETE:
                return "Delete";
            case IConfigurePermissionBits.ALLOW_SET_PERMISSIONS:
                return "Set permissions";
        }
        
        return "Unknown";
    }

    private String getManagePermissionDescription(int requiredPermission)
    {
        switch(requiredPermission)
        {
            case IManagePermissionBits.ALLOW_LIFE_CYCLE_CONTROL:
                return "Life cycle control";
            case IManagePermissionBits.ALLOW_ENABLE_DISABLE_METRICS:
                return "Enable/disable metrics";
            case IManagePermissionBits.ALLOW_NOTIFICATION_SUBSCRIPTION:
                return "Subscribe to notifications";
            case IManagePermissionBits.ALLOW_SET_ATTRIBUTES:
                return "Set attributes";
            case IManagePermissionBits.ALLOW_PERFORM_ACTIONS:
                return "Other actions";
            case IManagePermissionBits.ALLOW_GET_INFORMATION:
                return "Get information";
        }
        
        return "Unknown";
    }

    private boolean isAuditableAction(ObjectName objectName, MBeanInfo mBeanInfo, String operationName)
    {
        if (objectName == null) // MBeanServer call
        {
            if (CheckedAPI.isMBeanServerAPI(operationName))
            {
                return true;
            }
            return false;
        }
        if (CheckedAPI.isLifeCycleAPI(operationName))
        {
            return true;
        }
        if (CheckedAPI.isEnableDisableMetricsAPI(operationName))
        {
            return true;
        }
        if (CheckedAPI.isSetAttributesAPI(operationName))
        {
            return true;
        }
        
        boolean isMFComponent = ObjectNameHelper.isMFComponentName(objectName);

        if (isMFComponent)
        {
            if (CheckedAPI.isMBeanActionAPI(mBeanInfo, operationName))
            {
                if (ObjectNameHelper.getMFComponentName(objectName).equals("DIRECTORY SERVICE"))
                {
                    if (CheckedAPI.isCheckedDSAPI(operationName))
                    {
                        return true;
                    }
                }
                else
                {
                    return true;
                }
            }
        }
        
        // did not fit any of the standard checks, so now try any custom checks
        Iterator<Map.Entry<String, ArrayList<String>>> iterator = EXTENDED_AUDIT_BEHAVIOR_MAP.entrySet().iterator();
        while (iterator.hasNext())
        {
            Map.Entry<String, ArrayList<String>> entry = iterator.next();
            ObjectName name;
            try
            {
                name = new ObjectName(entry.getKey());
            }
            catch (MalformedObjectNameException e)
            {
                m_context.logMessage("Invalid JMX ObjectName specified in " + EXTENDED_AUDIT_BEHAVIOR_PROPERTY + " property, trace follows...", e, Level.WARNING);
                continue;
            }
            ArrayList<String> auditableItems = entry.getValue();
            Iterator mBeanInstances = m_mBeanServer.queryMBeans(name, null).iterator();
            while (mBeanInstances.hasNext())
            {
                ObjectInstance mBeanInstance = (ObjectInstance)mBeanInstances.next();
                if (mBeanInstance.getObjectName().equals(objectName)) // got an exact match so examine the auditable items
                {
                    // is there an explicit match on the operation name
                    if (auditableItems.contains(operationName))
                    {
                        return true;
                    }
                    
                    // else get the operation impact and see if that was specified
                    String impact = null;
                    if (mBeanInfo != null)
                    {
                        MBeanOperationInfo[] operationInfos = mBeanInfo.getOperations();
                        for (int i = 0; i < operationInfos.length; i++)
                        {
                            if (operationInfos[i].getName().equals(operationName))
                            {
                                if (operationInfos[i].getImpact() == MBeanOperationInfo.ACTION)
                                {
                                    impact = "ACTION";
                                }
                                if (operationInfos[i].getImpact() == MBeanOperationInfo.ACTION_INFO)
                                {
                                    impact = "ACTION_INFO";
                                }
                                if (operationInfos[i].getImpact() == MBeanOperationInfo.INFO)
                                {
                                    impact = "INFO";
                                }
                                if (operationInfos[i].getImpact() == MBeanOperationInfo.UNKNOWN)
                                {
                                    impact = "UNKNOWN";
                                }
                                break;
                            }
                        }
                        
                        if (impact != null && auditableItems.contains(impact))
                        {
                            return true;
                        }
                    }
                    
                }
            }
        }
        
        return false;
    }
    
    private StringBuilder addEventElementHeader(StringBuilder sb, String level)
    {
        String dateTime = null;
        dateTime = m_dateTimeFormatter.format(new Date(System.currentTimeMillis()));
            
        String user = TaskScheduler.getCurrentUserID();
        if (user == null || user.length() == 0)
        {
            user = ANONYMOUS_USER;
        }
        
        sb.append("<audit:event");
        sb.append(" dateTime=\"").append(dateTime);
        sb.append("\" level=\"").append(level);
        sb.append("\" user=\"").append(user);
        sb.append("\">");
        
        return sb;
    }
    
    private StringBuilder addEventElementTrailer(StringBuilder sb)
    {
        sb.append("</audit:event>");
        
        return sb;
    }
    
    private void appendXMLFragment(StringBuilder sb, Object object)
    {
        if (object == null)
        {
            sb.append("null");
            return;
        }
        
        if (object instanceof IXMLFragment)
        {
            sb.append(((IXMLFragment)object).toXMLFragment());
            return;
        }
        
        if (object instanceof Collection)
        {
            object = ((Collection)object).toArray();
        }
        
        if (object.getClass().isArray())
        {
            Class<?> componentType = object.getClass().getComponentType();
            if (!componentType.isPrimitive())
            {
                Object[] array = (Object[])object;
                sb.append("<list>");
                for (Object element : array)
                {
                    sb.append("<item>");
                    appendXMLFragment(sb, element);
                    sb.append("</item>");
                }
                sb.append("</list>");
            }
            else
            {
                // arrays of primitive types, used e.g. in the host manager API for 
                // file transfer or stdout/stderr (byte[]).
                // TODO: is it really useful to audit all of the byte[]?
                if (object instanceof byte[])
                {
                    sb.append("<bytes>");
                    sb.append(Base64.encode((byte[])object));
                    sb.append("</bytes>");
                }
                else
                {
                    sb.append("<array length=\"");
                    sb.append(Array.getLength(object));
                    sb.append("\" type=\"");
                    sb.append(object.getClass().getComponentType().getName());
                    sb.append("\"/>");
                }
            }
            return;
        }
        
        if (object instanceof Map)
        {
            sb.append("<map>");
            Iterator iterator = ((Map)object).entrySet().iterator();
            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry)iterator.next();
                
                sb.append("<item>");
                sb.append("<key>");
                appendXMLFragment(sb, entry.getKey());
                sb.append("</key>");
                sb.append("<value>");
                appendXMLFragment(sb, entry.getValue());
                sb.append("</value>");
                sb.append("</item>");
            }
            sb.append("</map>");
            return;
        }
        
        if (object instanceof Attribute)
        {
            sb.append("<attribute name=\"").append(((Attribute)object).getName()).append("\">");
            sb.append(escapeReservedXMLCharacters(((Attribute)object).getValue().toString()));
            sb.append("</attribute>");
            return;
        }

        sb.append(escapeReservedXMLCharacters(object.toString()));
    }
    
    private String escapeReservedXMLCharacters(String string)
    {
        return string.replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll(">", "&gt;").replaceAll("'", "&apos;").replaceAll("\"", "&quot;");
    }
    
    @Override
    public void recordEvent(String loggerName, String eventMessage)
    {
        if (loggerName.equals(CONFIGURE_LOGGER))
        {
            recordEvent(loggerName, true, m_configureEventQueue, eventMessage);
        }
        else if (loggerName.equals(MANAGE_LOGGER))
        {
            recordEvent(loggerName, false, m_manageEventQueue, eventMessage);
        }
    }
    
    private void recordEvent(final String loggerName, final boolean isConfigureEvent, ArrayList<Runnable> eventQueue, final String eventMessage)
    {
        if (m_enableCentralizedAudit)
        {
            try
            {
                m_context.invoke(m_agentManager, RECORD_AUDIT_EVENT_OPERATION_NAME, new Object[] { m_context.getComponentName().getCanonicalName(), loggerName, eventMessage }, RECORD_AUDIT_EVENT_OPERATION_SIGNATURE, false, 0);
            }
            catch (Exception e)
            {
                m_context.getContainer().logMessage(null, "Failed to record audit event at Domain Manager, trace follows...", e, Level.SEVERE);
            }
        }
            
        Runnable logTask = new Runnable()
        {
            @Override
            public void run()
            {
                // evaluate again at the last possible time if the auditing for the type of event is still valid
                if (isConfigureEvent)
                {
                    if (!m_auditConfigureEvents)
                    {
                        return;
                    }
                }
                else
                {
                    if (!m_auditManageEvents)
                    {
                        return;
                    }
                }
                Logger logger = Logger.getLogger(loggerName);
                logger.info(eventMessage);
            }
        };
        synchronized(eventQueue)
        {
            eventQueue.add(logTask);
            eventQueue.notifyAll();
        }
    }
    
    @Override
    public boolean configureAuditingEnabled()
    {
        return m_auditConfigureEvents;
    }
    
    private final class AsyncEventLogger
    extends Thread
    {
        private ArrayList<Runnable> eventQueue;
        
        private AsyncEventLogger(String type, ArrayList<Runnable> eventQueue)
        {
            super("Async " + type + " Audit Event Logger");
            super.setDaemon(true);
            this.eventQueue = eventQueue;
        }
        
        @Override
        public void run()
        {
            while (!interrupted())
            {
                Runnable logTask = null;
                synchronized(this.eventQueue)
                {
                    while (this.eventQueue.size() > MAX_EVENT_QUEUE_SIZE)
                    {
                        this.eventQueue.remove(0);
                    }
                    if (!this.eventQueue.isEmpty())
                    {
                        logTask = this.eventQueue.remove(0);
                    }
                }
                if (interrupted())
                {
                    return;
                }
                if (logTask != null)
                {
                    try
                    {
                        logTask.run();
                    }
                    catch(Throwable e)
                    {
                        AuditManager.this.m_context.getContainer().logMessage(null, "Failed to record audit event, trace follows...", e, Level.SEVERE);
                    }
                }
                else
                {
                    synchronized(this.eventQueue)
                    {
                        if (interrupted())
                        {
                            return;
                        }
                        if (this.eventQueue.isEmpty())
                        {
                            try
                            {
                                this.eventQueue.wait();
                            }
                            catch (InterruptedException e)
                            {
                                return;
                            }
                        }
                    }
                }
            }
        }
    }
}
