package com.sonicsw.mf.framework.security;

import java.util.*;
import java.util.Map.Entry;
import java.util.StringTokenizer;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.comm.InvokeTimeoutCommsException;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.NotModifiedAttException;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.security.IManagementPermission;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.mgmtapi.config.constants.IAuthenticationDomainConstants;

public class AuthenticationPrincipalMap
extends HashMap
{
    private HashMap m_usersByConfigIDMap = new HashMap();
    private HashMap m_groupsByConfigIDMap = new HashMap();
    private HashMap m_externalGroupsToInternalGroups = new HashMap();
    
    private IComponentContext m_context;
    
    private String m_descriptorPath;
    private String m_usersDir;
    private String m_externalUsersDir;
    private String m_groupsDir;
    private String m_externalGroupsDir;
    private boolean m_externalDomain = false;
    
    private static final boolean DEBUG = System.getProperty("com.sonicsw.debugFGS") != null;

    protected AuthenticationPrincipalMap(IComponentContext context, String authenticationDomainDirectory, boolean external)
    {
        m_context = context;
        m_externalDomain = external;
        m_descriptorPath = authenticationDomainDirectory + '/' + "_MFDomainDescriptor";
        updateGroupMap();
        
        m_usersDir = authenticationDomainDirectory + '/' + "_MFUsers";
        IElement[] userElements = getElements(m_usersDir);
        for (int i = 0; i < userElements.length; i++)
        {
            addUser(userElements[i]);
        }
        
        m_externalUsersDir = authenticationDomainDirectory + "/external/" + "_MFUsers";
        if (m_externalDomain)
        {
            IElement[] externalUserElements = getElements(m_externalUsersDir);
            for (int i = 0; i < externalUserElements.length; i++)
            {
                addUser(externalUserElements[i]);
            }
        }
        
        m_groupsDir = authenticationDomainDirectory + '/' + "_MFGroups";
        IElement[] groupElements = getElements(m_groupsDir);
        for (int i = 0; i < groupElements.length; i++)
        {
            addGroup(groupElements[i]);
        }
        
        m_externalGroupsDir = authenticationDomainDirectory + "/external/" + "_MFGroups";
        if (m_externalDomain)
        {
            IElement[] externalGroupElements = getElements(m_externalGroupsDir);
            for (int i = 0; i < externalGroupElements.length; i++)
            {
                addGroup(externalGroupElements[i]);
            }
        }
    }
    
    private IElement[] getElements(String dir)
    {
        boolean loggedFailure = false;
        while (true)
        {
            try
            {
                IElement[] elements = m_context.getConfigurations(dir, true);
                if (loggedFailure)
                {
                    m_context.logMessage("...security principal information retrieved", Level.INFO);
                }
                return elements;
            }
            catch(MFRuntimeException e)
            {
                Throwable linkedException = e.getLinkedException();
                
                if (linkedException == null)
                {
                    throw e;
                }
                if (linkedException instanceof InvokeTimeoutCommsException)
                {
                    throw e;
                }
                if (linkedException instanceof InvokeTimeoutException)
                {
                    if (((IFrameworkComponentContext)m_context).getContainer().isClosing())
                    {
                        throw e;
                    }
                    if (!loggedFailure)
                    {
                        m_context.logMessage("Timeout while retrieving security principal information, retrying...", Level.WARNING);
                        loggedFailure = true;
                    }
                    continue;
                }
                throw e;
            }
        }
    }

    public synchronized void handleElementChange(IElementChange elementChange)
    {
        IBasicElement element = elementChange.getElement();
        String configID = element.getIdentity().getName();
        if (configID.startsWith(m_usersDir) || configID.startsWith(m_externalUsersDir))
        {
            switch (elementChange.getChangeType())
            {
                case IElementChange.ELEMENT_ADDED:
                {
                    addUser((IElement)element);
                    break;
                }
                case IElementChange.ELEMENT_DELETED:
                {
                    removeUser(configID);
                    break;
                }
                case IElementChange.ELEMENT_UPDATED:
                {
                    // can only be a change in password so ignore
                    break;
                }
                case IElementChange.ELEMENT_REPLACED:
                {
                    String user = (String)m_usersByConfigIDMap.remove(configID);
                    ArrayList groupList = (ArrayList)super.get(user);
                    removeUser(configID);
                    IElement replacementElement = m_context.getConfiguration(configID, true);
                    addUserForElementReplacedAction(replacementElement,groupList);
                    break;
                }
            }
        }
        else if (configID.startsWith(m_groupsDir) || configID.startsWith(m_externalGroupsDir))
        {
            switch (elementChange.getChangeType())
            {
                case IElementChange.ELEMENT_ADDED:
                {
                    addGroup((IElement)element);
                    break;
                }
                case IElementChange.ELEMENT_DELETED:
                {
                    removeGroup(configID);
                    break;
                }
                case IElementChange.ELEMENT_UPDATED:
                {
                    updateGroup((IDeltaElement)element);
                    break;
                }
                case IElementChange.ELEMENT_REPLACED:
                {
                    removeGroup(configID);
                    IElement replacementElement = m_context.getConfiguration(configID, true);
                    addGroup(replacementElement);
                    break;
                }
            }
        }
        else if (configID.equals(m_descriptorPath))
        {
            updateGroupMap();
        }
    }
    
    /**
     * Gets the principal list for the given user. The principal list will include
     * all the groups the identity belongs to plus the user itself.
     */
    public synchronized String[] getGroups(String user)
    {
        List groupList = (List)super.get(user);
        if (groupList == null)
        {
            if (DEBUG)
            {
                System.out.println("get groups: user " + user + " not found (or associated group info' missing)!");
            }
            
            return IEmptyArray.EMPTY_STRING_ARRAY;
        }

        HashSet groupSetIncludingMappings = new HashSet(groupList);
        
        Iterator groups = groupList.iterator();
        while (groups.hasNext())
        {
            String group = (String)groups.next();
            HashSet mappedGroups = (HashSet)m_externalGroupsToInternalGroups.get(group);
            if (mappedGroups != null)
            {
                Iterator internalGroups = mappedGroups.iterator();
                while (internalGroups.hasNext())
                {
                    groupSetIncludingMappings.add(internalGroups.next());
                }
            }
        }
            
        return (String[])groupSetIncludingMappings.toArray(IEmptyArray.EMPTY_STRING_ARRAY);
    }

    /**
     * Gets the princiapl type (unknown, user or group)
     */
    public short getPrincipalType(String principal)
    {
        if (super.containsKey(principal))
        {
            return IManagementPermission.USER_PRINCIPAL_TYPE;
        }
        
        if (m_groupsByConfigIDMap.containsValue(principal))
        {
            return IManagementPermission.GROUP_PRINCIPAL_TYPE;
        }
        
        return IManagementPermission.UNKNOWN_PRINCIPAL_TYPE;
    }

    private void addUser(IElement userElement)
    {
        IAttributeSet userAttributes = userElement.getAttributes();
        
        String user = (String)userAttributes.getAttribute("USER_NAME");
        
        m_usersByConfigIDMap.put(userElement.getIdentity().getName(), user);
        
        ArrayList groupList = (ArrayList)super.get(user);
        if (groupList == null)
        {
            groupList = new ArrayList();
            super.put(user, groupList);
        }

        if (DEBUG)
        {
            System.out.println("add user (1): " + user);
        }
    }

    //MQ-35229: SIP PRD2 : Issues with DS-Restore (Permission Exceptions)
	//The user and group lists in this class are populated and associated properly only in a particular order when the users are first configured and then the groups.
    //This is what happens in the constructor.
    //But when the element change notification of type ELEMENT_REPLACED is fired for the user configuration the user configuration is removed an then added, In this process
    //the associated groups list for that user is lost. This is happening because when element change is fired groups are populated before the users.
    //These element change notifications were triggered from the TaskSchedular and I did not find any way to schedule those events to fire in a particular desired manner.
    //Hence introduced this new method which will read and store the groups temporarily from the new element. AFter removing the user and then while adding it back the temporarily stored groups are assigned to the user.
    private void addUserForElementReplacedAction(IElement userElement, ArrayList groupListReplaceElement)
    {
        IAttributeSet userAttributes = userElement.getAttributes();

        String user = (String)userAttributes.getAttribute("USER_NAME");

        m_usersByConfigIDMap.put(userElement.getIdentity().getName(), user);

        ArrayList groupList = (ArrayList)super.get(user);
        if(groupList == null)
        {
            groupList = new ArrayList();
        }

        if (groupListReplaceElement == null )
        {
            groupListReplaceElement = new ArrayList();
        }

        groupList.addAll(groupListReplaceElement);
        Set tempSet = new HashSet(groupList);
        groupList.clear();
        groupList.addAll(tempSet);

        super.put(user, groupList);

        if (DEBUG) 
		{
			System.out.println("add user (1): " + user);
		}
    }
    
    private void addGroup(IElement groupElement)
    {
        IAttributeSet groupAttributes = groupElement.getAttributes();
        
        String group = (String)groupAttributes.getAttribute("GROUP_NAME");
        m_groupsByConfigIDMap.put(groupElement.getIdentity().getName(), group);
        if (DEBUG)
        {
            System.out.println("add group: " + group + " [" + groupElement.getIdentity().getName() + "]");
        }
        
        // this code assumes the only members are users (as is assumed elsewhere) .. this could change
        // sometime in the future
        IAttributeSet groupMembers = (IAttributeSet)groupAttributes.getAttribute("GROUP_MEMBERS");
        
        // defensive programming .. if no users in group and attr set is null
        if (groupMembers == null)
        {
            return;
        }
        
        Iterator usersIterator = groupMembers.getAttributes().keySet().iterator();
        while (usersIterator.hasNext())
        {
            String user = (String)usersIterator.next();

            ArrayList groupList = (ArrayList)super.get(user);
            if (groupList == null)
            {
                groupList = new ArrayList();
                super.put(user, groupList);
            }
            if (DEBUG)
            {
                System.out.println("add group: " + group + ", user: " + user);
            }
            if (!groupList.contains(group))
            {
                groupList.add(group);
            }
        }
    }
    
    private void removeUser(String configID)
    {
        String user = (String)m_usersByConfigIDMap.remove(configID);
        if (DEBUG)
        {
            System.out.println("remove user: " + user);
        }
        super.remove(user);
    }
    
    private void removeGroup(String configID)
    {
        String group = (String)m_groupsByConfigIDMap.remove(configID);
        if (DEBUG)
        {
            System.out.println("remove group: " + group + " [" + configID + "]");
        }
        
        Iterator usersIterator = super.entrySet().iterator();
        while (usersIterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)usersIterator.next();
            ArrayList groupList = (ArrayList)entry.getValue();
            if (DEBUG)
            {
                System.out.println("remove group: " + group + ", user: " + entry.getKey());
            }
            if (groupList.contains(group))
            {
                groupList.remove(group);
            }
        }
    }
    
    private void updateGroup(IDeltaElement deltaElement)
    {
        String group = (String)m_groupsByConfigIDMap.get(deltaElement.getIdentity().getName());
        
        IDeltaAttributeSet deltaAttributes = (IDeltaAttributeSet)deltaElement.getDeltaAttributes();

        String[] modifiedAttributeNames = deltaAttributes.getModifiedAttributesNames();
        for (int i = 0; i < modifiedAttributeNames.length; i++)
        {
            if (modifiedAttributeNames[i].equals("GROUP_MEMBERS"))
            {
                try
                {
                    IDeltaAttributeSet membersDelta = (IDeltaAttributeSet)deltaAttributes.getNewValue("GROUP_MEMBERS");
                    String[] users = null;
                    
                    users = membersDelta.getDeletedAttributesNames();
                    for (int j = 0; j < users.length; j++)
                    {
                        ArrayList groupList = (ArrayList)super.get(users[j]);
                        if (groupList != null && groupList.contains(group))
                        {
                            groupList.remove(group);
                        }
                        if (DEBUG)
                        {
                            System.out.println("removed user: "+ users[j] + " from group: " + group + " [" + deltaElement.getIdentity().getName() + "]");
                        }
                    }
                    
                    users = membersDelta.getModifiedAttributesNames();
                    for (int j = 0; j < users.length; j++)
                    {
                        if (DEBUG)
                        {
                            System.out.println("modified user: "+ users[j] + " in group: " + group + " [" + deltaElement.getIdentity().getName() + "]");
                        }
                    }

                    users = membersDelta.getNewAttributesNames();
                    for (int j = 0; j < users.length; j++)
                    {
                        ArrayList groupList = (ArrayList)super.get(users[j]);
                        if (groupList == null)
                        {
                            groupList = new ArrayList();
                            super.put(users[j], groupList);
                        }
                        if (!groupList.contains(group))
                        {
                            groupList.add(group);
                        }
                        if (DEBUG)
                        {
                            System.out.println("new user: "+ users[j] + " in group: " + group + " [" + deltaElement.getIdentity().getName() + "]");
                        }
                    }
                }
                catch (NotModifiedAttException e)
                {
                    e.printStackTrace(); // should not happen
                }
                return;
            }
        }

        String[] newAttributeNames = deltaAttributes.getNewAttributesNames();
        for (int i = 0; i < newAttributeNames.length; i++)
        {
            if (newAttributeNames[i].equals("GROUP_MEMBERS"))
            {
                try
                {
                    Iterator users = ((IAttributeSet)deltaAttributes.getNewValue("GROUP_MEMBERS")).getAttributes().keySet().iterator();
                    while (users.hasNext())
                    {
                        String user = (String)users.next();
                        ArrayList groupList = (ArrayList)super.get(user);
                        if (groupList == null)
                        {
                            groupList = new ArrayList();
                            super.put(user, groupList);
                        }
                        if (!groupList.contains(group))
                        {
                            groupList.add(group);
                        }
                        if (DEBUG)
                        {
                            System.out.println("new user: "+ user + " in group: " + group + " [" + deltaElement.getIdentity().getName() + "]");
                        }
                    }
                }
                catch (NotModifiedAttException e)
                {
                    e.printStackTrace(); // should not happen
                }
                return;
            }
        }
    }

    private void updateGroupMap()
    {
        m_externalGroupsToInternalGroups.clear();
        
        IElement authenticationDomainDescriptor = m_context.getConfiguration(m_descriptorPath, true);
        IAttributeSet descriptorAttributes = authenticationDomainDescriptor.getAttributes();
        IAttributeSet groupMap = (IAttributeSet)descriptorAttributes.getAttribute(IAuthenticationDomainConstants.GROUP_MAP_ATTR);
        
        if (groupMap != null)
        {
            Iterator groupMappings = groupMap.getAttributes().entrySet().iterator();
            while (groupMappings.hasNext())
            {
                Map.Entry groupMapping = (Entry)groupMappings.next();
                String internalGroup = (String)groupMapping.getKey();
                StringTokenizer externalGroupList = new StringTokenizer((String)groupMapping.getValue(), ",");
                while (externalGroupList.hasMoreTokens())
                {
                    String externalGroup = externalGroupList.nextToken();
                    HashSet internalGroups = (HashSet)m_externalGroupsToInternalGroups.get(externalGroup);
                    if (internalGroups == null)
                    {
                        internalGroups = new HashSet();
                        m_externalGroupsToInternalGroups.put(externalGroup, internalGroups);
                    }
                    internalGroups.add(internalGroup);
                }
            }
        }
    }

    void cleanup()
    {
        m_context.getConfigurations(m_usersDir, false);
        if (m_externalDomain)
        {
            m_context.getConfigurations(m_externalUsersDir, false);
        }
        m_context.getConfigurations(m_groupsDir, false);
        if (m_externalDomain)
        {
            m_context.getConfigurations(m_externalGroupsDir, false);
        }
        m_usersByConfigIDMap.clear();
        m_groupsByConfigIDMap.clear();
        super.clear();
    }
}
