package com.sonicsw.mf.framework.security;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.management.ObjectName;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.security.IManageScopeBits;
import com.sonicsw.mf.common.util.ObjectNameHelper;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;

public class ManagePermissionsMap
extends AbstractPermissionsMap
{
    private HashMap m_containerNameToConfigIDMap = new HashMap();
    
    private static final String MANAGE_PERMISSIONS_CONFIG_ID = "/permissions/manage";

    public ManagePermissionsMap(IFrameworkComponentContext context, AuthenticationPrincipalMap authenticationPrincipalMap)
    {
        super(context, authenticationPrincipalMap, MANAGE_PERMISSIONS_CONFIG_ID);
        
        IElement containerConfiguration = context.getConfiguration(true);
        mapContainerNameToConfigID(containerConfiguration);
    }
    
    @Override
    public synchronized void handleElementChange(final IElementChange elementChange)
    {
        IBasicElement element = elementChange.getElement();
        
        // only handle changes to the element this map maps on to
        if (element.getIdentity().getType().equals("MF_CONTAINER"))
        {
            switch (elementChange.getChangeType())
            {
                case IElementChange.ELEMENT_UPDATED:
                {
                    // to keep it simple, just get the latest copy and map it again
                    element = super.m_context.getConfiguration(element.getIdentity().getName(), true);
                    mapContainerNameToConfigID((IElement)element);
                    break;
                }
                case IElementChange.ELEMENT_ADDED:
                {
                    mapContainerNameToConfigID((IElement)element);
                    break;
                }
                case IElementChange.ELEMENT_DELETED:
                {
                    unmapContainerNameToConfigID(element.getIdentity().getName());
                    break;
                }
                case IElementChange.ELEMENT_REPLACED:
                {
                    // will never occur
                    break;
                }
            }
        }
        else
        {
            super.handleElementChange(elementChange);
        }
    }
    
    private synchronized void mapContainerNameToConfigID(IElement containerConfiguration)
    {
        String containerConfigID = containerConfiguration.getIdentity().getName();
        IAttributeSet containerAttrs = containerConfiguration.getAttributes();
        String containerName = (String)containerAttrs.getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);
        
        // if this is a name change, we need to remove the old entry first
        unmapContainerNameToConfigID(containerConfigID);
        
        m_containerNameToConfigIDMap.put(containerName, containerConfigID);
    }
    
    private synchronized void unmapContainerNameToConfigID(String containerConfigID)
    {
        // if this is a name change, we need to remove the old entry first
        Iterator containers = m_containerNameToConfigIDMap.entrySet().iterator();
        while (containers.hasNext())
        {
            Map.Entry entry = (Map.Entry)containers.next();
            if (entry.getValue().equals(containerConfigID))
            {
                containers.remove();
                break;
            }
        }
    }
    
    private synchronized void getOtherContainerConfigIDs()
    {
        IElement[] containerConfigs = super.m_context.getConfigurations("/containers", true);
        for (int i = 0; i < containerConfigs.length; i++)
        {
            mapContainerNameToConfigID(containerConfigs[i]);
        }
    }

    boolean hasPermission(String principal, ObjectName objectName, int requiredPermission)
    {
        String containerName = ObjectNameHelper.getMFContainerName(objectName);
        String containerConfigID = null;
        
        synchronized(this)
        {
            containerConfigID = (String)m_containerNameToConfigIDMap.get(containerName);
            
            if (containerConfigID == null)
            {
                getOtherContainerConfigIDs();
                containerConfigID = (String)m_containerNameToConfigIDMap.get(containerName);
            }
        }
        
        // get the current logical path for the container (each time in case it changes)
        String logicalPath = super.m_context.storageToLogical(containerConfigID);
        
        // get the underlying component name
        String componentName = ObjectNameHelper.getMFComponentName(objectName);
        
        // get the list of groups a user belongs to
        String[] groups = super.m_authenticationPrincipalMap.getGroups(principal);
        
        // the path in this case will be the object name and we may need to determine if
        // the MBean is the AGENT/container
        boolean isContainer = componentName.equals(IAgentProxy.ID);
        
        int scope = IManageScopeBits.THIS_COMPONENT_SCOPE;
        int result = DENY;
        
        if (isContainer)
        {
            scope = IManageScopeBits.THIS_CONTAINER_SCOPE;
        }
        else // if this is not the AGENT component [container], then check the component path first
        {
            scope = IManageScopeBits.THIS_COMPONENT_SCOPE;
            String hybridPath = logicalPath + IComponentIdentity.DELIMITED_ID_PREFIX + componentName;
            result = checkPermission(principal, hybridPath, groups, requiredPermission, scope);
            if (result == ALLOW)
            {
                return true;
            }
            if (result == DENY)
            {
                return false;
            }
            scope = IManageScopeBits.ALL_COMPONENTS_SCOPE;
        }
        
        // check the container
        result = checkPermission(principal, logicalPath, groups, requiredPermission, scope);
        if (result == ALLOW)
        {
            return true;
        }
        if (result == DENY)
        {
            return false;
        }
        
        // if can't find anything under the compone and/or contaoner look in the parent folders
        // associated with the logical configuration name of the container
        String parentPath = getParentPath(logicalPath);
        scope = isContainer ? IManageScopeBits.ALL_CONTAINERS_SCOPE : IManageScopeBits.ALL_COMPONENTS_SCOPE;
        result = checkPermission(principal, parentPath, groups, requiredPermission, scope);
        if (result == ALLOW)
        {
            return true;
        }
        if (result == DENY)
        {
            return false;
        }

        if (parentPath.equals("/"))
         {
            return false; // since the root folder's permissions have already been checked
        }
        
        // nothing? .. then check in the parent's parents
        scope = scope | IManageScopeBits.ALL_FOLDERS_SCOPE;
        parentPath = getParentPath(parentPath);
        
        return super.hasPermission(principal, parentPath, groups, requiredPermission, scope);
    }
}
