// Copyright (c) 2007 Progress Software Corporation. All Rights Reserved.
package com.sonicsw.mf.framework.security;

import java.util.ArrayList;

import javax.management.MBeanInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementChange;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.runtime.ICollectiveOpStatus;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.CanonicalName;
import com.sonicsw.mf.common.runtime.impl.CollectiveOpStatus;
import com.sonicsw.mf.common.runtime.impl.ComponentIdentity;
import com.sonicsw.mf.common.security.ConfigurePermissionDeniedException;
import com.sonicsw.mf.common.security.IConfigurePermissionBits;
import com.sonicsw.mf.common.security.IManagePermissionBits;
import com.sonicsw.mf.common.security.IManagementPermission;
import com.sonicsw.mf.common.security.ManagePermissionDeniedException;
import com.sonicsw.mf.common.security.ManagementPermissionDeniedException;
import com.sonicsw.mf.common.util.ObjectNameHelper;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.IPermissionsManager;
import com.sonicsw.mf.framework.agent.Agent;
import com.sonicsw.mf.framework.agent.TaskScheduler;
import com.sonicsw.mf.framework.directory.DSComponent;
import com.sonicsw.mf.jmx.client.MFNotification;
import com.sonicsw.mf.mgmtapi.config.constants.IAuthenticationDomainConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDomainConstants;
import com.sonicsw.mf.mgmtapi.runtime.IAgentManagerProxy;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;
import com.sonicsw.mf.mgmtapi.runtime.IDirectoryServiceProxy;

public class PermissionsManager
implements IPermissionsManager
{
    private static final boolean DEBUG = System.getProperty("com.sonicsw.debugFGS") != null;
    
    private static final String DOMAIN_CONFIG_ID = "/domain/domain";

    private IFrameworkComponentContext m_context;
    
    private String m_authenticationDomain;
    
    private AuthenticationPrincipalMap m_authenticationPrincipalMap;
    private ManagePermissionsMap m_managePolicyMap;
    private ConfigurePermissionsMap m_configurePolicyMap;
    
    private IComponentIdentity m_dsID;
    
    private Object m_lock = new Object();
    private int m_checkers = 0;

    private int m_traceMask;
    
    public PermissionsManager(IFrameworkComponentContext context)
    {
        m_context = context;
        
        configurePermissionsManager();
    }
    
    @Override
    public boolean isPermissionsCheckingEnabled()
    {
        return m_authenticationPrincipalMap != null;
    }
    
    private void configurePermissionsManager()
    {
        IElement domainConfiguration = null;
        try
        {
            domainConfiguration = m_context.getConfiguration(DOMAIN_CONFIG_ID, true);
        }
        catch(MFRuntimeException e) { } // thrown when the directory is not found
        if (domainConfiguration == null)
        {
            return;
        }
            
        synchronized(m_lock)
        {
            while (m_checkers > 0) // give current checkers time to finish
            {
                try
                {
                    m_lock.wait(10000); // use some large time in case we are hung in this loop and shutdown is occuring
                    if (m_context.getContainer().isClosing())
                    {
                        return;
                    }
                }
                catch (InterruptedException e)
                {
                }
            }
            
            try
            {
                IAttributeSet domainAttributes = domainConfiguration.getAttributes();
                
                Reference authenticationDomainReference = (Reference)domainAttributes.getAttribute(IDomainConstants.AUTHENTICATION_DOMAIN_ATTR);
                if (authenticationDomainReference == null || authenticationDomainReference.getElementName() == null || authenticationDomainReference.getElementName().length() == 0)
                {
                    // check if it was previously enabled, if so disable
                    if (m_authenticationPrincipalMap != null)
                    {
                        disablePermissionsManager();
                    }
                }
                else
                {
                    enablePermissionsManager(authenticationDomainReference);
                }
            }
            finally
            {
                m_lock.notifyAll();
            }
        }
    }
    
    private void enablePermissionsManager(Reference authenticationDomainReference)
    {
        String authenticationDomain = authenticationDomainReference.getElementName();
        
        if (m_authenticationDomain != null)
        {
            if (authenticationDomain.equals(m_authenticationDomain))
            {
                return; // already enabled
            }
            else
            {
                m_context.getContainer().logMessage(null, "Management permissions checking will be temporarily disabled while it is reconfigured", Level.WARNING);
                disablePermissionsManager();
            }
        }
        else
        {
            if (authenticationDomain == null || authenticationDomain.length() == 0)
            {
                return;
            }
        }
        
        initPrincipalMap(authenticationDomainReference);
        initConfigurePermissionsMap();
        initManagePermissionsMap();
        
        m_authenticationDomain = authenticationDomain;
        
        m_context.getContainer().logMessage(null, "Management permissions checking enabled", Level.CONFIG);
    }
    
    private void disablePermissionsManager()
    {
        m_authenticationPrincipalMap.cleanup();
        m_authenticationPrincipalMap = null;
        if (m_configurePolicyMap != null)
        {
            m_configurePolicyMap.cleanup();
            m_configurePolicyMap = null;
        }
        m_managePolicyMap.cleanup();
        m_managePolicyMap = null;
        m_authenticationDomain = null;
        m_context.getContainer().logMessage(null, "Management permissions checking disabled", Level.CONFIG);
    }
    
    private void initPrincipalMap(Reference authenticationDomain)
    {
        String authenticationDomainDir = authenticationDomain.getElementName();
        IElement domain = m_context.getConfiguration(authenticationDomainDir, true);
        // find out if its an external domain
        IAttributeSet domainSet = domain.getAttributes();
        Boolean external = (Boolean)domainSet.getAttribute(IAuthenticationDomainConstants.EXTERNAL_ATTR);
        boolean externalValue = false;
        if (external != null)
        {
            externalValue = external.booleanValue();
        }
        // remove the _MFDomainDescriptor
        authenticationDomainDir = authenticationDomainDir.substring(0, authenticationDomainDir.indexOf("_MFDomainDescriptor") - 1);
        m_authenticationPrincipalMap = new AuthenticationPrincipalMap(m_context, authenticationDomainDir, externalValue);
    }

    private void initConfigurePermissionsMap()
    {
        IElement containerConfiguration = m_context.getConfiguration(true);
        IAttributeSet containerAttributes = containerConfiguration.getAttributes();
        
        // while were getting the container config work out the initial trace mask as well 
        Integer traceMask = (Integer)containerAttributes.getAttribute(IContainerConstants.TRACE_MASK_ATTR);
        if (traceMask != null)
        {
            setTraceMask(traceMask.intValue());
        }

        // determine if this container hosts the directory service
        Boolean hostsDS = (Boolean)containerAttributes.getAttribute(IContainerConstants.HOSTS_DIRECTORY_SERVICE_ATTR);
        
        // if it does host the DS then we can create the configure permissions map
        if (hostsDS != null && hostsDS.booleanValue())
        {
            m_configurePolicyMap = new ConfigurePermissionsMap(m_context, m_authenticationPrincipalMap);
            CanonicalName dsName = new CanonicalName(m_context.getComponentName().getDomainName() + "." + m_context.getComponentName().getContainerName() + IComponentIdentity.DELIMITED_ID_PREFIX + IDirectoryServiceProxy.GLOBAL_ID);
            m_dsID = new ComponentIdentity(dsName, null); 
        }
    }

    private void initManagePermissionsMap()
    {
        m_managePolicyMap = new ManagePermissionsMap(m_context, m_authenticationPrincipalMap);
    }

    public synchronized void handleElementChange(IElementChange elementChange)
    {
        IElementIdentity id = elementChange.getElement().getIdentity();
        String configID = id.getName();

        if (configID.startsWith(DOMAIN_CONFIG_ID))
        {
            handleDomainChanges(elementChange);
        }
        else
        {
            if (m_authenticationPrincipalMap != null) // is permissions checking enabled?
            {
                m_authenticationPrincipalMap.handleElementChange(elementChange);
                if (m_managePolicyMap != null)
                {
                    m_managePolicyMap.handleElementChange(elementChange);
                }
                if (m_configurePolicyMap != null)
                {
                    m_configurePolicyMap.handleElementChange(elementChange);
                }
            }
        }
    }
    
    private void handleDomainChanges(IElementChange elementChange)
    {
        if (elementChange.getChangeType() != IElementChange.ELEMENT_UPDATED)
         {
            return; // don't support the add or delete of this element
        }

        configurePermissionsManager();
    }

    @Override
    public void configurePermissionCheck(IComponentContext context, String path, boolean isLogicalPath, int requiredPermission)
    throws ManagementPermissionDeniedException
    {
        if (!isLogicalPath && path.startsWith("/_MFContext"))
         {
            return; // we won't deal with JNDI tree in this release
        }
        
        synchronized(m_lock) // ensure the configuration doesn't get change in the middle of checking
        {
            m_checkers++;
        }

        try
        {
            if (m_authenticationPrincipalMap == null)
            {
                return;
            }
            
            if (!TaskScheduler.isExecutionThread())
            {
                if (DEBUG)
                {
                    System.out.println("PermissionsManager.configurePermissionCheck(): Called from non-task thread, stack dump follows...");
                    Thread.dumpStack();
                }
                return;
            }
            
            String principal = TaskScheduler.getCurrentUserID();
                        
            if (principal == null || principal.length() == 0)
            {
                if (DEBUG)
                {
                    System.out.println("PermissionsManager.configurePermissionCheck(): No user identity associated with configure request: principal=" + principal +
                                       ", path=" + path +
                                       ": stack dump follows...");
                    Thread.dumpStack();
                }
                if ((m_traceMask & Agent.TRACE_PERMISSION_DENIED) > 0)
                {
                    m_context.getContainer().logMessage(IDirectoryServiceProxy.GLOBAL_ID, "Permission denied: no user identity associated with mangement request", Level.TRACE);
                }
                throw new ManagementPermissionDeniedException("No user identity associated with mangement request, check management security settings");
            }
            
            if (principal.equals(IManagementPermission.SUPER_USER_NAME))
            {
                // super user "Administrator" can do anything!
                return;
            }
        
            if (!m_configurePolicyMap.hasPermission(principal, path, requiredPermission, isLogicalPath))
            {
                if (isLogicalPath)
                {
                    recordConfigurePermissionDenied(context, principal, path, null, requiredPermission);
                }
                else
                {
                    recordConfigurePermissionDenied(context, principal, null, path, requiredPermission);
                }
                throw new ConfigurePermissionDeniedException("Permission denied: user identity=" + principal + ", path=" + path + ", isLogicalPath=" + isLogicalPath + ", permission=" + requiredPermission,
                                                             isLogicalPath ? path : null, requiredPermission);
            }
        }
        finally
        {
            synchronized(m_lock)
            {
                m_checkers--;
                m_lock.notifyAll();
            }
        }
    }

    @Override
    public void managePermissionCheck(ObjectName objectName, String operationName, MBeanInfo mBeanInfo)
    throws ManagementPermissionDeniedException
    {
        synchronized(m_lock) // ensure the configuration doesn't get change in the middle of checking
        {
            m_checkers++;
        }

        try
        {
            if (m_authenticationPrincipalMap == null)
            {
                return;
            }
            
            if (!TaskScheduler.isExecutionThread())
            {
                if (DEBUG)
                {
                    System.out.println("PermissionsManager.managePermissionCheck(): Called from non-task thread: " + objectName.getCanonicalName() +
                                       ", operationName=" + operationName +
                                       ": stack dump follows...");
                    Thread.dumpStack();
                }
                return;
            }
            
            String principal = TaskScheduler.getCurrentUserID();
            
            boolean isMFComponent = objectName != null && ObjectNameHelper.isMFComponentName(objectName);
            
            if (principal == null || principal.length() == 0)
            {
                if (DEBUG)
                {
                    System.out.println("PermissionsManager.managePermissionCheck(): No user identity associated with mangement request: principal=" + principal +
                                       ", " + objectName == null ? "MBeanServer" : objectName.getCanonicalName() +
                                       ", operationName=" + operationName +
                                       ": stack dump follows...");
                    Thread.dumpStack();
                }
                if ((m_traceMask & Agent.TRACE_PERMISSION_DENIED) > 0)
                {
                    m_context.getContainer().logMessage(isMFComponent ? objectName.getKeyProperty("ID") : null, "Permission denied: no user identity associated with mangement request", Level.TRACE);
                }
                throw new ManagementPermissionDeniedException("No user identity associated with mangement request, check management security settings");
            }
            
            if (principal.equals(IManagementPermission.SUPER_USER_NAME))
            {
                // super user "Administrator" can do anything!
                return;
            }
            
            // determine what permission is required for the given operation
            int requiredPermission = IManagePermissionBits.ALLOW_GET_INFORMATION;
            
            ObjectName checkName = objectName;
            if (!isMFComponent)
            {
                // Permissions set on the AGENT component (container) will also apply to calls directly
                // targeted at the container's MBeanServer .. so in this case fixup the name first
                try
                {
                    checkName = new ObjectName(m_context.getComponentName().getCanonicalName());
                }
                catch (MalformedObjectNameException e) { e.printStackTrace(); } // should never occur
            }
            
            // test first for whether this is a call on the MBeanServer (vs. an MF component)
            if (objectName == null)
            {
                if (CheckedAPI.isMBeanServerAPI(operationName))
                {
                    requiredPermission = IManagePermissionBits.ALLOW_PERFORM_ACTIONS;
                }
                else
                if (CheckedAPI.isMBeanServerInfo(operationName))
                {
                    requiredPermission = IManagePermissionBits.ALLOW_GET_INFORMATION;
                }
            }
            else
            if (CheckedAPI.isLifeCycleAPI(operationName))
            {
                requiredPermission = IManagePermissionBits.ALLOW_LIFE_CYCLE_CONTROL;
            }
            else
            if (CheckedAPI.isEnableDisableMetricsAPI(operationName))
            {
                requiredPermission = IManagePermissionBits.ALLOW_ENABLE_DISABLE_METRICS;
            }
            else
            if (CheckedAPI.isSetAttributesAPI(operationName))
            {
                requiredPermission = IManagePermissionBits.ALLOW_SET_ATTRIBUTES;
            }
            else
            if (CheckedAPI.isNotificationSubscriptionAPI(operationName))
            {
                requiredPermission = IManagePermissionBits.ALLOW_NOTIFICATION_SUBSCRIPTION;
            }
            else
            {
                // special treatment for calls to the DS since some of those calls will be governed by
                // configure permissions and do not require checking here
                if (isMFComponent && objectName.getKeyProperty("ID").equals("DIRECTORY SERVICE") && !CheckedAPI.isCheckedDSAPI(operationName))
                {
                    return;
                }

                if (mBeanInfo == null) // when called from the AM during collection delegation, this will be null
                {
                    try
                    {
                        mBeanInfo = (MBeanInfo)m_context.invoke(objectName.getCanonicalName(), "getMBeanInfo", IEmptyArray.EMPTY_OBJECT_ARRAY, IEmptyArray.EMPTY_STRING_ARRAY, true, -1);
                    }
                    catch (Exception e)
                    {
                        if (e instanceof RuntimeException)
                        {
                            throw (RuntimeException)e;
                        }
                        
                        // manufacture a runtime exception
                        MFRuntimeException mfre = new MFRuntimeException();
                        mfre.setLinkedException(e);
                        throw mfre;
                    }
                }
                
                // is this an action impact operation
                if (CheckedAPI.isMBeanActionAPI(mBeanInfo, operationName))
                {
                    requiredPermission = IManagePermissionBits.ALLOW_PERFORM_ACTIONS;
                }
            }
                
            if (!m_managePolicyMap.hasPermission(principal, checkName, requiredPermission))
            {
                recordManagePermissionDenied(IAgentProxy.ID, principal, objectName, operationName, requiredPermission);
                throw new ManagePermissionDeniedException("Permission denied: user identity=" + principal + ", target=" + (objectName == null ? "[MBeanServer]" : objectName.getCanonicalName()) + ", operation=" + operationName + ", permission=" + requiredPermission,
                                                          objectName == null ? "[MBeanServer]" : objectName.getCanonicalName(), requiredPermission);
            }
        }
        finally
        {
            synchronized(m_lock)
            {
                m_checkers--;
                m_lock.notifyAll();
            }
        }
    }
    
    @Override
    public CollectiveOpStatus managePermissionCheck(ObjectName[] objectNames, String operationName)
    {
        synchronized(m_lock) // ensure the configuration doesn't get change in the middle of checking
        {
            m_checkers++;
        }

        try
        {
            if (m_authenticationPrincipalMap == null)
            {
                return null;
            }
            
            if (!TaskScheduler.isExecutionThread())
            {
                if (DEBUG)
                {
                    System.out.println("PermissionsManager.managePermissionCheck(): Called from non-task thread: operationName=" + operationName +
                                       ": stack dump follows...");
                    Thread.dumpStack();
                }
                return null;
            }
            
            String principal = TaskScheduler.getCurrentUserID();
            
            if (principal == null || principal.length() == 0)
            {
                if (DEBUG)
                {
                    System.out.println("PermissionsManager.managePermissionCheck(): No user identity associated with mangement request: principal=" + principal +
                                       ", operationName=" + operationName +
                                       ": stack dump follows...");
                    Thread.dumpStack();
                }
                if ((m_traceMask & Agent.TRACE_PERMISSION_DENIED) > 0)
                {
                    m_context.getContainer().logMessage("AGENT MANAGER", "Permission denied: no user identity associated with mangement request", Level.TRACE);
                }
                throw new ManagePermissionDeniedException("No user identity associated with mangement request, check management security settings", null, 0);
            }
            
            if (principal.equals(IManagementPermission.SUPER_USER_NAME))
            {
                // super user "Administrator" can do anything!
                return null;
            }
            
            CollectiveOpStatus collectiveResults = new CollectiveOpStatus(objectNames.length);
            
            // determine what permission is required for the given operation
            int requiredPermission = IManagePermissionBits.ALLOW_GET_INFORMATION;
            
            if (CheckedAPI.isLifeCycleAPI(operationName))
            {
                requiredPermission = IManagePermissionBits.ALLOW_LIFE_CYCLE_CONTROL;
            }
            else
            if (CheckedAPI.isEnableDisableMetricsAPI(operationName))
            {
                requiredPermission = IManagePermissionBits.ALLOW_ENABLE_DISABLE_METRICS;
            }
            else
            if (CheckedAPI.isSetAttributesAPI(operationName))
            {
                requiredPermission = IManagePermissionBits.ALLOW_SET_ATTRIBUTES;
            }
            else
            if (CheckedAPI.isNotificationSubscriptionAPI(operationName))
            {
                requiredPermission = IManagePermissionBits.ALLOW_NOTIFICATION_SUBSCRIPTION;
            }
            else
            {
                ArrayList targets = new ArrayList();
                for (int i = 0; i < objectNames.length; i++)
                {
                    String target = objectNames[i].getCanonicalName();
                    if (target.endsWith("DIRECTORY SERVICE") && !CheckedAPI.isCheckedDSAPI(operationName))
                    {
                        continue;
                    }
                    targets.add(target);
                }
                
                // get MBeanInfo for each component in the collection
                ICollectiveOpStatus collectiveOpStatus = null;
                if (!targets.isEmpty())
                {
                    try
                    {
                        collectiveOpStatus = (ICollectiveOpStatus)m_context.invoke((String[])targets.toArray(), "getMBeanInfo", IEmptyArray.EMPTY_OBJECT_ARRAY, IEmptyArray.EMPTY_STRING_ARRAY, true, -1);
                    }
                    catch(Exception e)
                    {
                        MFRuntimeException exception = new MFRuntimeException("Failed invoke [" + operationName + "] on collection");
                        exception.setLinkedException(e);
                        throw exception;
                    }
                }
                
                // iterate through those that we have MBeanInfo for and determine if the caller has permission
                for (int i = 0; i < collectiveOpStatus.getCount(); i++)
                {
                    if (collectiveOpStatus.operationIsSuccessful(i))
                    {
                        // it makes no sense because the operation in all the components in the collection
                        // *should* have the same impact, but do it just in case of version descrepencies
                        requiredPermission = IManagePermissionBits.ALLOW_GET_INFORMATION;
                        
                        MBeanInfo mBeanInfo = (MBeanInfo)collectiveOpStatus.getReturnValue(i);
                        
                        // work out if this is an action operation
                        MBeanOperationInfo[] operationInfos = mBeanInfo.getOperations();
                        for (int j = 0; j < operationInfos.length; j++)
                        {
                            if (operationInfos[j].getName().equals(operationName))
                            {
                                if (operationInfos[j].getImpact() == MBeanOperationInfo.ACTION || operationInfos[j].getImpact() == MBeanOperationInfo.ACTION_INFO)
                                {
                                    requiredPermission = IManagePermissionBits.ALLOW_PERFORM_ACTIONS;
                                }
                                // else its already been set to get information permission
                                break;
                            }
                        }
                        
                        ObjectName objectName = null;
                        try
                        {
                            objectName = new ObjectName(collectiveOpStatus.getComponentName(i));
                        }
                        catch (MalformedObjectNameException e)
                        {
                            // TODO Auto-generated catch block
                            e.printStackTrace();
                        }
                        if (!m_managePolicyMap.hasPermission(principal, objectName, requiredPermission))
                        {
                            recordManagePermissionDenied(IAgentManagerProxy.GLOBAL_ID, principal, objectName, operationName, requiredPermission);
                            ManagePermissionDeniedException exception = new ManagePermissionDeniedException("Permission denied: user identity=" + principal + 
                                                                                                            ", target=" + (objectName == null ? null : objectName.getCanonicalName()) + 
                                                                                                            ", operation=" + operationName + ", permission=" + requiredPermission, 
                                                                                                            (objectName == null ? null : objectName.getCanonicalName()), requiredPermission);
                            collectiveResults.addResult(collectiveOpStatus.getComponentName(i), null, exception);
                        }
                    }
                    else
                    {
                        collectiveResults.addResult(collectiveOpStatus.getComponentName(i), collectiveOpStatus.getReturnValue(i), collectiveOpStatus.getThrowable(i));
                    }
                }
                
                return collectiveResults;
            }
            
            // if we fall through here, we were not checking based on MBeanInfo, so now check each component
            // in the collection
            
            for (int i = 0; i < objectNames.length; i++)
            {
                if (!m_managePolicyMap.hasPermission(principal, objectNames[i], requiredPermission))
                {
                    recordManagePermissionDenied(IAgentManagerProxy.GLOBAL_ID, principal, objectNames[i], operationName, requiredPermission);
                    ManagePermissionDeniedException exception = new ManagePermissionDeniedException("Permission denied: user identity=" + principal + ", target=" + objectNames[i].getCanonicalName() + ", operation=" + operationName + ", permission=" + requiredPermission,
                                                                                                    objectNames[i].getCanonicalName(), requiredPermission);
                    collectiveResults.addResult(objectNames[i].getCanonicalName(), null, exception);
                }
            }
            
            return collectiveResults;
        }
        finally
        {
            synchronized(m_lock)
            {
                m_checkers--;
                m_lock.notifyAll();
            }
        }
    }

    /**
     * Checks if the given user is a user in the Authentication Domain used for
     * management security.
     */
    @Override
    public short getPrincipalType(String principal)
    {
        if (m_authenticationPrincipalMap == null)
        {
            return IManagementPermission.UNKNOWN_PRINCIPAL_TYPE;
        }
        
        return m_authenticationPrincipalMap.getPrincipalType(principal);
    }
    
    /**
     * Checks if the user associated with the current thread is considered an adminstrative
     * super-user.
     */
    @Override
    public boolean isSuperUser()
    {
        if (TaskScheduler.isExecutionThread())
        {
            String userID = TaskScheduler.getCurrentUserID();
            if (userID != null && userID.equals(IManagementPermission.SUPER_USER_NAME))
            {
                return true;
            }
        }
        
        return false;
    }

    public int getTraceMask()
    {
        return m_traceMask;
    }
    
    public void setTraceMask(int traceMask)
    {
        m_traceMask = traceMask;
    }
    
    // we potentially record the denied event in several ways: container log (if tracing enabled), a notification and the audit trail if that is enabled
    private void recordConfigurePermissionDenied(IComponentContext context, String principal, String logicalPath, String storagePath, int requiredPermission)
    {
        if ((m_traceMask & Agent.TRACE_PERMISSION_DENIED) > 0)
        {
            StringBuffer sb = new StringBuffer();
            sb.append("Configure permission denied: user identity=").append(principal);
            if ((logicalPath != null) && (logicalPath.length() > 0))
            {
                sb.append(", logicalPath=").append(logicalPath);
            }
            if ((storagePath != null) && (storagePath.length() > 0))
            {
                sb.append(", storagePath=").append(storagePath);
            }
            sb.append(", permission=").append(requiredPermission);
            m_context.getContainer().logMessage(IDirectoryServiceProxy.GLOBAL_ID, sb.toString(), Level.TRACE);
        }
        
        // we do not send a notification or audit read denied events
        if (!(requiredPermission == IConfigurePermissionBits.ALLOW_READ))
        {
            sendConfigurePermissionDeniedNotification(context, principal, logicalPath, storagePath, requiredPermission);
            m_context.getAuditManager().recordConfigurePermissionDeniedEvent(logicalPath, storagePath, requiredPermission);
        }
    }

    // we potentially record the denied event in several ways: container log (if tracing enabled), a notification and the audit trail if that is enabled
    private void recordManagePermissionDenied(String sourceID, String principal, ObjectName objectName, String operationName, int requiredPermission)
    {
        if ((m_traceMask & Agent.TRACE_PERMISSION_DENIED) > 0)
        {
            StringBuffer sb = new StringBuffer();
            sb.append("Manage permission denied: user identity=").append(principal);
            sb.append(", target=").append(objectName== null ? "[MBeanServer]" : objectName.getCanonicalName());
            sb.append(", operation=").append(operationName);
            sb.append(", permission=").append(requiredPermission);
            m_context.getContainer().logMessage(sourceID, sb.toString(), Level.TRACE);
        }
        sendManagePermissionDeniedNotification(principal, objectName== null ? "[MBeanServer]" : objectName.getCanonicalName(), operationName, requiredPermission);
        m_context.getAuditManager().recordManagePermissionDeniedEvent(objectName, operationName, requiredPermission);
    }

    private void sendConfigurePermissionDeniedNotification(IComponentContext context, String principal, String logicalPath, String storagePath, int requiredPermission)
    {
        INotification notification = context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.SECURITY_SUBCATEGORY], DSComponent.CONFIGURE_PERMISSION_DENIED_NOTIFICATION_TYPE, Level.WARNING);
        ((MFNotification)notification).setSourceIdentity(m_dsID);
        notification.setLogType(INotification.FAILURE_AUDIT_TYPE);
        notification.setAttribute("User", principal);
        if ((logicalPath != null) && (logicalPath.length() > 0))
        {
            notification.setAttribute("LogicalPath", logicalPath);
        }
        if ((storagePath != null) && (storagePath.length() > 0))
        {
            notification.setAttribute("StoragePath", storagePath);
        }
        notification.setAttribute("RequiredPermission", new Integer(requiredPermission));
        context.sendNotification(notification);
    }

    private void sendManagePermissionDeniedNotification(String principal, String target, String operationName, int requiredPermission)
    {
        INotification notification = m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.SECURITY_SUBCATEGORY], Agent.MANAGE_PERMISSION_DENIED_NOTIFICATION_TYPE, Level.WARNING);
        notification.setLogType(INotification.FAILURE_AUDIT_TYPE);
        notification.setAttribute("User", principal);
        notification.setAttribute("Target", target);
        notification.setAttribute("Operation", operationName);
        notification.setAttribute("RequiredPermission", new Integer(requiredPermission));
        m_context.sendNotification(notification);
    }
}
