// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.mf.framework.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.RandomAccessFile;
import java.text.SimpleDateFormat;
import java.util.Date;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.runtime.Level;

/**
 * Containers with file based logging enabled, log their (and their components) messages to a rolling log. The
 * RollingLog class manages one or more LogFile instances; a current LogFile and potentially multiple archive
 * instances that are the result of log file rollover.
 * <p>
 * The AM implements a centralized log that uses the same RollingLog and LogFile classes.
 * <p>
 * The LogFile class provides methods to:
 *  - write log messages
 *  - read log file contents
 *  - rollover
 *  - set the rollover threshold
 *  - get the current file length
 */
public class LogFile
{
    private static final int MAX_TAIL_LENGTH = 204800; // 200K
    private static final long DEFAULT_ROLLOVER_SIZE = 1048576; // 1.0Mb
    private static final long MIN_ROLLOVER_SIZE = DEFAULT_ROLLOVER_SIZE >> 1; // 0.5Mb
    private static final long MAX_ROLLOVER_SIZE = DEFAULT_ROLLOVER_SIZE << 11; // 1.0Gb
    
    private static final boolean QA_BEHAVIOR = System.getProperty("sonicsw.mf.qa.abort") != null;
    
    private String m_path;
    private File m_file;
    private RandomAccessFile m_rwFile;
    private boolean m_isReadOnly = true;
    private boolean m_lastAccessWasRead = false;
    
    private IComponentContext m_context;

    private long m_rolloverSize = DEFAULT_ROLLOVER_SIZE;
    
    // We will hold the tail of the log file in memory for faster access for default readers (those that want to read
    // the last 200K of the log file)
    private Buffer m_tail;

    private final SimpleDateFormat m_dateFormatter = new SimpleDateFormat("yyyy-MM-dd");
    private final SimpleDateFormat m_rolloverDateTimeFormatter = new SimpleDateFormat("yyyy-MM-dd.HH-mm");

    LogFile(String path, boolean isReadOnly, IComponentContext context)
    throws IOException
    {
        m_path = path;
        m_isReadOnly = isReadOnly;
        
        m_context = context;
        
        init();
    }
    
    private void init()
    throws IOException
    {
        m_file = new File(m_path);
        m_path = m_file.getCanonicalPath();
        m_rwFile = new RandomAccessFile(m_path, m_isReadOnly ? "r" : "rw");
        if (!m_isReadOnly)
        {
            m_tail = new Buffer();
            byte[] bytes = read((long)-1, (long)MAX_TAIL_LENGTH);
            m_tail.appendBytes(bytes);
        }
    }
    
    synchronized void close()
    throws IOException
    {
        m_rwFile.close();
        m_tail = null;
        m_rwFile = null;
    }
    
    public void clearLogFile()
    throws IOException
    {
        m_rwFile.setLength(0);
    }

    public void saveLogFile(String path)
    throws IOException
    {
        close();
        
        File newFile = new File(path);
        String canonicalPath = newFile.getCanonicalPath();
        newFile = new File(canonicalPath);
        
        byte[] buf = new byte[1024];
        FileInputStream in = new FileInputStream(m_file);
        FileOutputStream out = new FileOutputStream(newFile);
        while (true)
        {
            int bytesRead = in.read(buf);
            if (bytesRead == -1)
            {
                break;
            }
            out.write(buf, 0, bytesRead);
        }
        in.close();
        out.close();
        
        init();
    }
    
    String getLogPath()
    {
        return m_path;
    }
    
    synchronized void write(String logMessage)
    throws IOException
    {
        if (m_rwFile == null)
        {
            return;
        }
        
        if (m_isReadOnly)
        {
            throw new IOException("Log file is read-only: " + m_path);
        }
        
        InterruptedIOException interruptedIOException = null;
        
        try
        {
            if (m_lastAccessWasRead)
            {
                long writeOffset = length();
                m_rwFile.seek(writeOffset);
            }
            m_rwFile.write(logMessage.getBytes());
        }
        catch(InterruptedIOException e)
        {
            interruptedIOException = e;
            
            // reset the pointer and try writing again
            if (m_lastAccessWasRead)
            {
                long writeOffset = length();
                m_rwFile.seek(writeOffset);
            }
            m_rwFile.write(logMessage.getBytes());
        }

        if (m_tail != null)
        {
            m_tail.appendBytes(logMessage.getBytes());
        }
        
        // rethrow the interrupt if there was one
        if (interruptedIOException != null)
        {
            throw interruptedIOException;
        }
        
        m_lastAccessWasRead = false;
    }
    
    synchronized byte[] read(long fromPosn, long readLength)
    throws IOException
    {
        if (m_rwFile == null)
        {
            return new byte[0];
        }
        
        m_lastAccessWasRead = true;
        
        long fileLength = length();
        
        if (fromPosn == -1)
        {
            fromPosn = fileLength - readLength;
        }
        if (fromPosn < 0)
        {
            fromPosn = 0;
        }
        
        if (fromPosn + readLength > fileLength)
        {
            readLength = fileLength - fromPosn;
        }
        
        if (m_tail != null && readLength == MAX_TAIL_LENGTH && (fromPosn == -1 || fromPosn > fileLength - MAX_TAIL_LENGTH))
        {
            return m_tail.getBytes();
        }

        m_rwFile.seek(fromPosn);
        
        byte[] bytes = new byte[(int)readLength];
        m_rwFile.read(bytes);
        
        return bytes;
    }
    
    synchronized File rollover()
    throws IOException
    {
        if (m_rwFile == null)
        {
            return null;
        }
        
        if (length() >= m_rolloverSize)
        {
            File currentFile = new File(m_path);

            File archiveFile;

            // MQ-33653 - Configurable log rollover check interval
            boolean hourGranularity = RollingFileLogger.CALCULATE_NEXT_ROLLOVER_TIME_IN == RollingFileLogger.CALC_IN_HOURS_FROM_SYSTEM_PROPERTY;
            boolean dailyRollover = hourGranularity ? RollingFileLogger.LOG_ROLLOVER_CHECK_INTERVAL == 24 : RollingFileLogger.LOG_ROLLOVER_CHECK_INTERVAL == 1440;
            if (dailyRollover)
            {
                // preserve original behaviour/labelling
                Date yesterday = new Date(System.currentTimeMillis() - 86400000L);
                archiveFile = new File(m_path + '.' + m_dateFormatter.format(yesterday));

                if (archiveFile.exists()) // we already have one or more archive file(s) for yesterday so create one for today
                {
                    Date today = new Date(System.currentTimeMillis());
                    archiveFile = new File(m_path + '.' + m_dateFormatter.format(today));
                }
            }
            else
            {
                // include the rollover time if the check interval has been modified
                Date rolloverTime = new Date(System.currentTimeMillis());
                archiveFile = new File(m_path + '.' + m_rolloverDateTimeFormatter.format(rolloverTime));
            }

            if (archiveFile.exists())
            {
                int suffix = 0;
                while (true)
                {
                    File versionedArchiveFile = new File(archiveFile.getPath() + '.' + ++suffix);
                    if (!versionedArchiveFile.exists())
                    {
                        archiveFile = versionedArchiveFile;
                        break;
                    }
                }
            }

            if (m_context != null)
            {
                m_context.logMessage("Log file rollover initiated...", Level.INFO);
            }
            
            m_rwFile.close();

            currentFile.renameTo(archiveFile);
            m_rwFile = new RandomAccessFile(currentFile.getPath(), "rw");

            if (m_context != null)
            {
                m_context.logMessage("...rollover complete, archived log: " + archiveFile.getPath(), Level.INFO);
            }
            
            return archiveFile;
        }
        
        return null;
    }
    
    synchronized long length()
    {
        try
        {
            return m_rwFile.length();
        }
        catch (IOException e) { }
        return m_file.length();
    }
    
    void setRolloverThreshold(long rolloverThreshold)
    {
        if (!QA_BEHAVIOR) // don't check if were in QA mode
        {
            if (rolloverThreshold < MIN_ROLLOVER_SIZE)
            {
                throw new IllegalArgumentException("Rollover threshold cannot be less than 0.5Mb");
            }
            if (rolloverThreshold > MAX_ROLLOVER_SIZE)
            {
                throw new IllegalArgumentException("Rollover threshold cannot be greater than 1.0Gb");
            }
        }

        m_rolloverSize = rolloverThreshold;
    }
    
    private final class Buffer
    {
        private byte[] tailBytes = new byte[MAX_TAIL_LENGTH];
        private int tailLength = 0; // length from start position in circular buffer (may wrap)
        private int tailEnd = 0; // end position in the circular buffer
        
        private void appendBytes(byte[] appendBytes)
        {
            int start1 = 0;
            int length1 = 0;
            int start2 = 0;
            int length2 = 0;
            
            if (appendBytes.length > MAX_TAIL_LENGTH)
            {
                start1 = appendBytes.length - tailBytes.length;
                length1 = MAX_TAIL_LENGTH - tailEnd;
            }
            else
            {
                int remainder = MAX_TAIL_LENGTH - tailEnd;
                length1 = appendBytes.length <= remainder ? appendBytes.length : remainder;
            }
            
            if (appendBytes.length - length1 > 0)
            {
                start2 = start1 + length1;
                length2 = appendBytes.length - start1 - length1;
            }
            
            if (length1 > 0)
            {
                System.arraycopy(appendBytes, start1, tailBytes, tailEnd, length1);
                tailEnd += length1;
                if (tailEnd >= MAX_TAIL_LENGTH)
                {
                    tailEnd = 0;
                }
            }
            
            if (length2 > 0)
            {
                System.arraycopy(appendBytes, start2, tailBytes, 0, length2);
                tailEnd = length2;
            }
            
            tailLength += appendBytes.length;
            if (tailLength > MAX_TAIL_LENGTH)
            {
                tailLength = MAX_TAIL_LENGTH;
            }
        }
        
        private byte[] getBytes()
        {
            byte[] bytes = new byte[tailLength];
            
            int start = tailLength < MAX_TAIL_LENGTH ? 0 : tailEnd;
            int length = tailLength == MAX_TAIL_LENGTH ? tailLength - tailEnd : tailEnd;
            
            System.arraycopy(tailBytes, start, bytes, 0, length);
            if (start > 0)
            {
                System.arraycopy(tailBytes, 0, bytes, length, tailLength - length);
            }
            
            return bytes;
        }
    }

    /*
    public static void main(String[] args)
    {
        String NEWLINE = System.getProperty("line.separator");

        try
        {
            LogFile logFile = new LogFile("temp.log", false);
            
            for (int i = 0; i < args.length; i++)
            {
                System.out.println("Size before: " + logFile.length());
                logFile.write(args[i] + NEWLINE);
                System.out.println("Size after : " + logFile.length());
                
                LogFile archiveFile = logFile.rollover();
                if (archiveFile != null)
                    System.out.println("Rollover to: " + archiveFile.m_path);
            }
        }
        catch(Exception e) { e.printStackTrace(); }
    }
    */
}
