package com.sonicsw.mf.framework.util;

import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IFrameworkComponentContext;
import com.sonicsw.mf.framework.INotificationHandler;
import com.sonicsw.mf.framework.ITaskScheduler;

/**
 * The NotificationForwarder is used by framework components to pass through
 * (forward) notifications they receive or are subscribed to without modifying
 * the source of the notification.
 */
public final class NotificationForwarder
implements INotificationHandler
{
    private IFrameworkComponentContext m_context;
    private ITaskScheduler m_taskScheduler;

    private boolean m_debug = false;

    public NotificationForwarder(IFrameworkComponentContext context)
    {
        m_context = context;
        m_taskScheduler = m_context.getContainer().getTaskScheduler();
    }

    /**
     * Handle a notification.
     *
     * @param notification The notification to be handled.
     */
    @Override
    public void handleNotification(final INotification notification)
    {
        Runnable invoker = new Runnable()
        {
            @Override
            public void run()
            {
                try
                {
                    if (m_debug)
                    {
                        NotificationForwarder.this.m_context.logMessage("Forwarding notification [" + notification.getType() + "] received from " + notification.getSourceIdentity().getCanonicalName(), Level.TRACE);
                    }
                    // when were forwarding tack who forwarded the notification into the notification attributes
                    notification.setAttribute("ForwardedBy", NotificationForwarder.this.m_context.getComponentName().getCanonicalName());
                    NotificationForwarder.this.m_context.sendNotification(notification);
                }
                catch(Exception e)
                {
                    String message = "Failed to forward notification: "
                                + notification.getType()
                                + " [source="
                                + notification.getSourceIdentity().getCanonicalName()
                                + "], trace follows...";
                    NotificationForwarder.this.m_context.logMessage(message, e, Level.WARNING);
                }
            }
        };
        m_taskScheduler.scheduleTask(invoker, false);
    }

    public void setDebug(boolean debug) { m_debug = debug; }
}
