// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.mf.framework.util;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;

import com.sonicsw.mx.util.Sorter;

import com.sonicsw.mf.common.IComponentContext;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.IMessageLogger;

/**
 * Containers with file based logging enabled, log their (and their components) messages to a rolling log. The
 * RollingLog class manages one or more LogFile instances; a current LogFile and potentially multiple archive
 * instances that are the result of log file rollover.
 * <p>
 * The AM implements a centralized log that uses the same RollingLog and LogFile classes.
 * <p>
 * The RollingLog class provides methods to:
 *  - write log messages
 *  - read log file contents (including from archived versions)
 *  - set the rollover threshold
 *  - get the current log length (the aggregate for current and archived versions)
 */
public class RollingFileLogger
implements IMessageLogger
{
    private static final boolean DEBUG = false;

    private IComponentContext m_context;

    private String m_logDirectoryPath;
    private String m_baseLogFileName;
    
    private LogFile m_currentLogFile;
    private long m_sizeThreshold; // overall initial size threshold (current + archived) .. when broken send a notification
    private int m_logRolloverTimeInterval;
    private long m_currentSizeThreshold; // overall initial size threshold (current + archived) .. when broken send a notification
    private long m_lastRecordedLength; // we don't check the length every call, just ever PHYSICAL_CHECK_FREQUENCY;
    private int m_physicalCheckCount = RollingFileLogger.PHYSICAL_CHECK_FREQUENCY; // to force a first check
    private static final int PHYSICAL_CHECK_FREQUENCY = 10;

    static int LOG_ROLLOVER_CHECK_INTERVAL = 1440;
    static final int CALC_IN_MINS_FROM_SMC = -1;
    static final int CALC_IN_MINS_FROM_SYSTEM_PROPERTY = 0;
    static final int CALC_IN_HOURS_FROM_SYSTEM_PROPERTY = 1;
    static int CALCULATE_NEXT_ROLLOVER_TIME_IN = CALC_IN_MINS_FROM_SMC;

    private RolloverTask m_rolloverTask = null;
    
    private IOException m_lastIOException = null;
    
    private final Object m_lock = new Object();


    private static final long MAX_READ_LENGTH = 1048576; // 1.0Mb
    private static final long DEFAULT_READ_LENGTH = 204800; // 200Kb
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");

    private static final String LOGFAILURE_NOTIFICATION_TYPE = "Failure";
    private static final String LOGTHRESHOLD_NOTIFICATION_TYPE = "Threshold";
    
    private static final boolean QA_MODE = System.getProperty(IContainer.MF_QA_ABORT_PROPERTY) != null;

    public RollingFileLogger(String logDirectoryPath, String logFileNamePrefix)
    throws IOException
    {
        this(null, logDirectoryPath, logFileNamePrefix, LOG_ROLLOVER_CHECK_INTERVAL);
    }

    // constructor for use when starting in the container (not the first phase launcher)
    public RollingFileLogger(IComponentContext context, String logDirectoryPath, String logFileNamePrefix, int logRolloverTimeInterval)
    throws IOException
    {
        this.m_logRolloverTimeInterval = logRolloverTimeInterval;

        int interval = 1440;
        try
        {
            interval = Integer.getInteger("sonicsw.mf.logRolloverCheckIntervalMinutes");
            interval = (interval > 0 && interval <= 1440) ? interval : 1440;
            CALCULATE_NEXT_ROLLOVER_TIME_IN = CALC_IN_MINS_FROM_SYSTEM_PROPERTY;
        }
        catch (Exception e)
        {
            try
            {
                interval = Integer.getInteger("sonicsw.mf.logRolloverCheckIntervalHours");
                interval = (interval > 0 && interval <= 24) ? interval : 24;
                CALCULATE_NEXT_ROLLOVER_TIME_IN = CALC_IN_HOURS_FROM_SYSTEM_PROPERTY;
            }
            catch (Exception e1)
            {
                interval = this.m_logRolloverTimeInterval;
            }
        }
        LOG_ROLLOVER_CHECK_INTERVAL = interval;

        // pre-v7.6 compatibility
        File logFile = new File(logDirectoryPath);
        if (logFile.exists() && logFile.isFile())
        {
            System.out.println("Renaming existing container log file \"" + logFile.getPath() + "\" to \"" + new File(logDirectoryPath + ".old\""));
            File renamedLogFile = new File(logDirectoryPath + ".old"); 
            logFile.renameTo(renamedLogFile);
            if (logFile.exists())
            {
                System.out.println("Failed to rename existing container log file");
            }
        }
        
        m_baseLogFileName = logFileNamePrefix + ".log";

        m_context = context;
        
        resetLogDirectory(logDirectoryPath);
        
        if (m_context != null)
        {
            m_rolloverTask = new RolloverTask();
            m_rolloverTask.schedule(); // to do the initial schedule of the rollover task
        }
    }
    
    public void close()
    throws IOException
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                if (m_rolloverTask != null)
                {
                    m_context.cancelTask(m_rolloverTask);
                }
                m_currentLogFile.close();
                m_currentLogFile = null;
            }
        }
    }
    
    /**
     * This method is provided for backwards compatibility; its implementation will clear the current
     * log file and all archive versions.
     * @throws IOException
     */
    public void clearLogFile()
    throws IOException
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                if (m_currentLogFile ==  null)
                {
                    return;
                }

                File[] archiveLogFiles = getArchivedLogFiles(true);
                for (int i = 0; i < archiveLogFiles.length; i++)
                {
                    archiveLogFiles[i].delete();
                }

                m_currentLogFile.clearLogFile();
                
                // this will force a physical check of the log file size the next time we
                // log a message
                m_physicalCheckCount = RollingFileLogger.PHYSICAL_CHECK_FREQUENCY;
            }
        }
    }
    
    /**
     * This method is provided for backwards compatibility; its implementation will only save the current log file
     * and not archived versions.
     * @throws IOException
     */
    public void saveLogFile(String path)
    throws IOException
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                if (m_currentLogFile == null)
                {
                    return;
                }

                m_currentLogFile.saveLogFile(path);
            }
        }
    }
    
    @Override
    public void logMessage(String logMessage)
    {
        logMessage += LINE_SEPARATOR;
        
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                if (m_currentLogFile == null)
                {
                    return;
                }

                try
                {
                    m_currentLogFile.write(logMessage);
                    checkSizeThreshold(logMessage.length());

                    m_lastIOException = null; // clear any prior IOException when we have successfully logged
                }
                catch(InterruptedIOException e) { } // we have to eat it in order not to break MQ (Sonic00040198)
                catch(IOException e)
                {
                    handleIOExceptionOnWrite(e);
                }
            }
        }
    }

    // these two methods are for support of components that are using unsupported
    // writes to stdout/stderr rather than logMessage()
    public void write(int b)
    throws InterruptedIOException
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                if (m_currentLogFile == null)
                {
                    return;
                }

                try
                {
                    m_currentLogFile.write(new String(new byte[] { (byte)b }));
                    //checkSizeThreshold(); // can't afford to do this for every byte!
                }
                catch(InterruptedIOException e)
                {
                    throw e;
                }
                catch(IOException e)
                {
                    handleIOExceptionOnWrite(e);
                }
            }
        }
    }
    
    public void write(byte[] bytes, int offset, int length)
    throws InterruptedIOException
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                if (m_currentLogFile == null)
                {
                    return;
                }

                try
                {
                    m_currentLogFile.write(new String(bytes, offset, length));
                    checkSizeThreshold(length);
                }
                catch(InterruptedIOException e)
                {
                    throw e;
                }
                catch(IOException e)
                {
                    handleIOExceptionOnWrite(e);
                }
            }
        }
    }
    
    public long length()
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                if (m_currentLogFile == null)
                {
                    return 0L;
                }

                File[] archiveLogFiles = getArchivedLogFiles(false);

                long totalLength = 0;
                for (int i = 0; i < archiveLogFiles.length; i++)
                {
                    totalLength += archiveLogFiles[i].length();
                }

                totalLength += m_currentLogFile.length();

                return totalLength;
            }
        }
    }
    
    public final void resetLogDirectory(String logDirectoryPath)
    throws IOException
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                File logDirectory = new File(logDirectoryPath);
                if (logDirectory.exists() && !logDirectory.isDirectory())
                {
                    throw new IOException("Log path is not a directory: " + logDirectoryPath);
                }

                if (!logDirectory.exists())
                {
                    if (!logDirectory.mkdirs())
                    {
                        throw new IOException("Unable to create log directory: " + logDirectoryPath);
                    }
                }

                File newLogFilePath = new File(logDirectory, m_baseLogFileName);
                LogFile newLogFile = new LogFile(newLogFilePath.getPath(), false, m_context);

                m_logDirectoryPath = logDirectoryPath;

                if (m_currentLogFile != null)
                {
                    m_context.logMessage("Logging path changed to \"" + newLogFile.getLogPath() + '"', Level.INFO);
                    m_currentLogFile.close();
                    LogFile oldLogFile = m_currentLogFile;
                    m_currentLogFile = newLogFile;
                    m_context.logMessage("Logging path changed from \"" + oldLogFile.getLogPath() + '"', Level.INFO);
                }
                else
                {
                    m_currentLogFile = newLogFile;
                }
            }
        }
    }
    
    public void setRolloverThreshold(long rolloverThreshold)
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                if (m_currentLogFile == null)
                {
                    return;
                }

                m_currentLogFile.setRolloverThreshold(rolloverThreshold);
            }
        }
    }
    
    public long getSizeThreshold()
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                return m_sizeThreshold;
            }
        }
    }

    public void setSizeThreshold(long sizeThreshold)
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                m_sizeThreshold = sizeThreshold;
                m_currentSizeThreshold = sizeThreshold;
                if (m_context != null && length() > sizeThreshold)
                {
                    sendLogThresholdNotification(m_sizeThreshold, length());
                }
            }
        }
    }

    private void checkSizeThreshold(int messageLength)
    {
        if (m_context == null)
        {
            // don't check during first phase startup .. only when this log has been instantiated within the second phase [container]
            return;
        }

        // this logic is to limit the times we do a physical check on the size (which can take a significant amount
        // of time, especially is there are multiple archived log files) .. clearly this logic is an approximation,
        // an archive could be removed in the meantime
        if (++m_physicalCheckCount > RollingFileLogger.PHYSICAL_CHECK_FREQUENCY)
        {
            m_lastRecordedLength = length();
            m_physicalCheckCount = 0;
        }
        else
        {
            m_lastRecordedLength += messageLength;
        }

        if (m_lastRecordedLength > m_currentSizeThreshold)
        {
            m_currentSizeThreshold = (long)(m_currentSizeThreshold * 1.1); // reset the next threshold to be 10% greater than the last
            if (m_lastRecordedLength > m_currentSizeThreshold)
            {
                m_currentSizeThreshold = (long)(m_lastRecordedLength * 1.1);
            }
            m_context.logMessage("Container log size threshold exceeded, threshold=" + m_sizeThreshold + ", actual=" + m_lastRecordedLength, Level.WARNING);
            sendLogThresholdNotification(m_sizeThreshold, m_lastRecordedLength);
        }
        else
        {
            if (m_lastRecordedLength < m_sizeThreshold)
            {
                // possible the log file was truncated or one of the archives was removed reducing the overall size
                m_currentSizeThreshold = m_sizeThreshold;
            }
        }
    }

    private void handleIOExceptionOnWrite(IOException e)
    {
        // don't report anything if its the same exception as we got on the last write
        if (m_lastIOException != null && m_lastIOException.getMessage().equals(e.getMessage()))
        {
            return;
        }

        m_lastIOException = e;

        if (m_context == null)
        {
            System.err.println("Failed to write to log file: " + m_currentLogFile.getLogPath() + ", trace follows...");
            e.printStackTrace();
        }
        else
        {
            m_context.logMessage("Failed to write to log file: " + m_currentLogFile.getLogPath() + ", trace follows...", e, Level.SEVERE);
            sendLogFailureNotification(length());
        }
    }

    public byte[] read(long fromPosn, long readLength)
    throws IOException
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                if (m_currentLogFile == null)
                {
                    return "<log to file disabled>".getBytes();
                }

                if (readLength > MAX_READ_LENGTH)
                {
                    throw new IllegalArgumentException("Cannot read more than 1.0Mb");
                }
                if (readLength == -1)
                {
                    readLength = DEFAULT_READ_LENGTH;
                }

                long remainderToBeRead = readLength;
                byte[] bytesRead = null;

                long currentLength = length();
                if (fromPosn > currentLength)
                {
                    return new byte[0];
                }

                // if were trying to read starting from a position back from the EOF
                // then we can take some shortcuts
                if (fromPosn == -1)
                {
                    bytesRead = m_currentLogFile.read(fromPosn, readLength);

                    remainderToBeRead -= bytesRead.length;

                    // if we got enough data from the current log then look no further
                    if (remainderToBeRead == 0)
                    {
                        return bytesRead;
                    }
                }

                ArrayList bytesReadList = new ArrayList();
                long totalBytesRead = 0;
                if (bytesRead != null && bytesRead.length > 0)
                {
                    bytesReadList.add(bytesRead);
                    totalBytesRead += bytesRead.length;
                }

                File[] archiveLogFiles = getArchivedLogFiles(true);

                if (fromPosn == -1) // then read backwards
                {
                    for (int i = archiveLogFiles.length - 1; i >= 0; i--) // see reverse order
                    {
                        long archiveLogFileLength = archiveLogFiles[i].length();
                        long archiveFromPosn = 0;
                        long archiveReadLength = remainderToBeRead;

                        if (archiveLogFileLength < remainderToBeRead)
                        {
                            archiveReadLength = archiveLogFileLength; // well be reading the whole thing
                        }
                        else {
                            archiveFromPosn = archiveLogFileLength - remainderToBeRead; // well be doing a partial read from some position
                        }

                        LogFile archiveLogFile = new LogFile(archiveLogFiles[i].getPath(), true, m_context);
                        byte[] archiveBytes = archiveLogFile.read(archiveFromPosn, archiveReadLength);
                        archiveLogFile.close();

                        if (archiveBytes.length > 0)
                        {
                            bytesReadList.add(0, archiveBytes);
                            totalBytesRead += archiveBytes.length;
                            remainderToBeRead -= archiveBytes.length;
                        }
                    }
                }
                else // read forwards from given position
                {
                    long remainderToBeSkipped = fromPosn;
                    for (int i = 0; i < archiveLogFiles.length; i++)
                    {
                        long archiveLogFileLength = archiveLogFiles[i].length();
                        if (remainderToBeSkipped >= archiveLogFileLength)
                        {
                            remainderToBeSkipped -= archiveLogFileLength;
                            continue;
                        }

                        long archiveFromPosn = remainderToBeSkipped;
                        long archiveReadLength = archiveLogFileLength - remainderToBeSkipped;

                        if (archiveLogFileLength - remainderToBeSkipped > remainderToBeRead)
                         {
                            archiveReadLength = remainderToBeRead; // well be reading the whole thing
                        }

                        remainderToBeSkipped = 0; 

                        LogFile archiveLogFile = new LogFile(archiveLogFiles[i].getPath(), true, m_context);
                        byte[] archiveBytes = archiveLogFile.read(archiveFromPosn, archiveReadLength);
                        archiveLogFile.close();

                        if (archiveBytes.length > 0)
                        {
                            bytesReadList.add(archiveBytes);
                            totalBytesRead += archiveBytes.length;
                            remainderToBeRead -= archiveBytes.length;
                        }

                        if (remainderToBeRead == 0)
                        {
                            break;
                        }
                    }

                    if (remainderToBeRead > 0) // add in content from the current log file
                    {
                        byte[] currentLogBytes = m_currentLogFile.read(remainderToBeSkipped, remainderToBeRead);
                        bytesReadList.add(currentLogBytes);
                        totalBytesRead += currentLogBytes.length;
                    }
                }

                // aggregate all the read bytes together in one array
                bytesRead = new byte[(int)totalBytesRead];
                int offset = 0;
                for (int i = 0; i < bytesReadList.size(); i++)
                {
                    byte[] bytes = (byte[])bytesReadList.get(i);
                    System.arraycopy(bytes, 0, bytesRead, offset, bytes.length);
                    offset += bytes.length;
                }

                return bytesRead;
            }
        }
    }

    private void sendLogThresholdNotification(long thresholdSize, long actualSize)
    {
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.LOG_SUBCATEGORY], LOGTHRESHOLD_NOTIFICATION_TYPE, Level.WARNING);
        notification.setLogType(INotification.INFORMATION_TYPE);
        notification.setAttribute("Directory", m_logDirectoryPath);
        notification.setAttribute("BaseFilename", m_baseLogFileName);
        notification.setAttribute("ThresholdSize", new Long(thresholdSize));
        notification.setAttribute("ActualSize", new Long(actualSize));
        m_context.sendNotification(notification);
    }

    private void sendLogFailureNotification(long sizeAtFailure)
    {
        INotification notification =
            m_context.createNotification(INotification.SYSTEM_CATEGORY, INotification.SUBCATEGORY_TEXT[INotification.LOG_SUBCATEGORY], LOGFAILURE_NOTIFICATION_TYPE, Level.SEVERE);
        notification.setLogType(INotification.ERROR_TYPE);
        notification.setAttribute("SizeAtFailure", new Long(sizeAtFailure));
        m_context.sendNotification(notification);
    }

    private File[] getArchivedLogFiles(boolean sort)
    {
        File logDirectoryPath = new File(m_logDirectoryPath);
        final String archiveFilenamePrefix = m_baseLogFileName + '.';

        FilenameFilter archiveFilenameFilter = new FilenameFilter()
        {
            @Override
            public boolean accept(File directory, String filename)
            {
                return filename.startsWith(archiveFilenamePrefix);
            }
        };

        File[] archivedLogFiles = logDirectoryPath.listFiles(archiveFilenameFilter);

        // Sonic00040366 .. this will return null if there is an IO issue while listing the directory (naff java!)
        // so give the caller something back they can handle (even if we can't report something sensible!)
        if (archivedLogFiles == null)
        {
            return new File[0];
        }

        return sort ? (File[])Sorter.sort(archivedLogFiles) : archivedLogFiles;
    }

    public void attemptLogFileRollover()
    throws IOException
    {
        if (!IContainer.QA_MODE)
        {
            throw new IllegalAccessError("Manual log file rollover attempt only permissible in QA mode!");
        }

        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                m_currentLogFile.rollover();
            }
        }
    }

    public int get_LogRolloverTimeInterval()
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                return m_logRolloverTimeInterval;
            }
        }
    }

    public void set_LogRolloverTimeInterval(int logRolloverTimeInterval)
    {
        synchronized (System.out)
        {
            synchronized (m_lock)
            {
                if (m_currentLogFile == null)
                {
                    return;
                }

                this.m_logRolloverTimeInterval = logRolloverTimeInterval;
            }
        }
    }

    // used to schedule and run the rollover
    private final class RolloverTask
    implements Runnable
    {
        @Override
        public void run()
        {
            synchronized(System.out)
            {
                synchronized(RollingFileLogger.this.m_lock)
                {
                    try
                    {
                        if (RollingFileLogger.this.m_currentLogFile != null) // so we don't rollover every time we startup
                        {
                            if (DEBUG)
                            {
                                RollingFileLogger.this.m_context.logMessage("Attempting log file rollover", Level.TRACE);
                            }

                            RollingFileLogger.this.m_currentLogFile.rollover();
                        }
                    }
                    catch (IOException e)
                    {
                        e.printStackTrace();
                    }
                    finally
                    {
                        schedule();
                    }
                }
            }
        }

        public void schedule()
        {
            final Calendar nextRolloverAttemptTime = Calendar.getInstance();
            int nextRolloverInMillis = 0;

            if(CALCULATE_NEXT_ROLLOVER_TIME_IN == CALC_IN_HOURS_FROM_SYSTEM_PROPERTY)
            {
                //Backward Compatibility : Calculating for the hours configured by
                // system property sonicsw.mf.logRolloverCheckIntervalHours
                // MQ-33653 Scheduling the next rollover interval based on the rollover check interval system property.
                nextRolloverInMillis = getNextRolloverInMillisForHours(nextRolloverAttemptTime, LOG_ROLLOVER_CHECK_INTERVAL);
                calcNextRolloverTime(nextRolloverAttemptTime, nextRolloverInMillis);
            }
            else if(CALCULATE_NEXT_ROLLOVER_TIME_IN == CALC_IN_MINS_FROM_SYSTEM_PROPERTY )
            {
                //Backward Compatibility : Calculating for the minutes configured by
                // system property sonicsw.mf.logRolloverCheckIntervalMinutes
                nextRolloverInMillis = getRolloverInMillisForMins(nextRolloverAttemptTime, LOG_ROLLOVER_CHECK_INTERVAL);
                calcNextRolloverTime(nextRolloverAttemptTime, nextRolloverInMillis);
            }
            else
            {
                //Calculating for the minutes from the user configured value on Container configuration.
                nextRolloverInMillis = getRolloverInMillisForMins(nextRolloverAttemptTime, get_LogRolloverTimeInterval());
                calcNextRolloverTime(nextRolloverAttemptTime, nextRolloverInMillis);
            }

            if (DEBUG)
            {
                SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm");
                RollingFileLogger.this.m_context.logMessage("Next rollover scheduled for " + format.format(nextRolloverAttemptTime.getTime()), Level.TRACE);
            }
            RollingFileLogger.this.m_context.scheduleTask(this, nextRolloverAttemptTime.getTime());
        }

        /**
         * Method schedules the next rollover time interval either in hour of the day Or every midnight
         * i.e. if set as 3, next hour would be 3, 6, 9, 12, 15, 18, 21, 24.
         * else
         * schedules the next rollover time interval in minutes of the day
         * i.e. 0.5, 0.10, 0.15, 0.20, etc minutes of the day.
         * But not from the contianer start time.
         */
        private void calcNextRolloverTime(Calendar nextRolloverAttemptTime, int nextRolloverInMillis)
        {
            final int dstDiff = nextRolloverAttemptTime.get(Calendar.DST_OFFSET);

            // MQ-35039 - Post DSTchanges , high cpu usage from 11pm - 12am on Sonic JVMs
            nextRolloverAttemptTime.add(Calendar.SECOND, -nextRolloverAttemptTime.get(Calendar.SECOND));
            nextRolloverAttemptTime.add(Calendar.MINUTE, -nextRolloverAttemptTime.get(Calendar.MINUTE));
            nextRolloverAttemptTime.add(Calendar.HOUR_OF_DAY, -nextRolloverAttemptTime.get(Calendar.HOUR_OF_DAY));

            nextRolloverAttemptTime.add(Calendar.MILLISECOND, nextRolloverInMillis);

            // That covers DST transition ('Spring forward' / 'Fall back')
            nextRolloverAttemptTime.add(Calendar.MILLISECOND, dstDiff - nextRolloverAttemptTime.get(Calendar.DST_OFFSET));
        }

        private int getRolloverInMillisForMins(Calendar nextRolloverAttemptTime, int logRolloverCheckInterval)
        {
            return  (((((nextRolloverAttemptTime.get(Calendar.HOUR_OF_DAY) * 60) + nextRolloverAttemptTime.get(Calendar.MINUTE)) / logRolloverCheckInterval) + 1)
                    * logRolloverCheckInterval) * 60 * 1000;
        }

        private int getNextRolloverInMillisForHours(Calendar nextRolloverAttemptTime, int logRolloverCheckInterval)
        {
            return (((nextRolloverAttemptTime.get(Calendar.HOUR_OF_DAY) / logRolloverCheckInterval) + 1)
                    * logRolloverCheckInterval) * 60 * 60 * 1000;
        }
    }
}
