// Copyright (c) 2009 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.mf.framework.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;

import com.sonicsw.mf.common.runtime.IStateController;
import com.sonicsw.mf.common.runtime.IStateListener;
import com.sonicsw.mf.common.runtime.IStateManager;
import com.sonicsw.mf.common.runtime.NonRecoverableStateChangeException;
import com.sonicsw.mf.common.runtime.RecoverableStateChangeException;

/**
 * The StateManager class acts as a central point for managing a state such as the
 * fault tolerant state of the DS or AM.
 * <p>
 * The state manager is configured with a set of states with which it can work.
 * <p>
 * State controllers (implementers of IStateController) may be registered to effect
 * the behavioral adjustments associated with a particular state transition.
 * <p>
 * State listeners (implementors of IStateListener) may be added to be informed when
 * a state transitions starts and whether the transition succeeds or fails.
 * <p>
 * A developer should not attempt to request a state change from either a state
 * controller or a state listener.
 */
public class StateManager
implements IStateManager
{
    private short[] m_managedStates;
    
    private short m_currentState;
    private Object m_currentStateLock = new Object();

    private HashMap m_stateControllers = new HashMap();
    private HashSet m_stateListeners = new HashSet();

    /**
     * The StateManager is constructed with a set of valid state (expected
     * to be 0 thru n) and an initial state code that is the value of one
     * of the states.
     */
    public StateManager(short[] managedStates, short initialState)
    {
        m_managedStates = managedStates;

        if (!isValidState(initialState))
        {
            throw new IllegalArgumentException("Invalid state: " + initialState);
        }

        m_currentState = initialState;
    }

    @Override
    public short getState(Object obj)
    {
        // Since this concrete implementation of IStateManager manages a single
        // object, the input argument can be ignored.

        // return the current state
        synchronized(m_currentStateLock)
        {
            return m_currentState;
        }
    }

    /**
     * The thread that detects the condition that requires a state change will call this
     * method to effect the state change.
     * <p>
     * The calling thread is expected to have previously obtained the current state and
     * to have made the decision as to what is the next state that should be applied. To
     * ensure synchronization between calls, the caller is required to provide the expected
     * current state so that it can possibly retry if another thread changed the state
     * between the callers calls.
     *
     * @param expectedCurrentState The current state obtained immediately before calling
     *                             this method
     * @param desiredState         The state the caller wishes to transition to
     *
     * @return Return true if the state transition was successful, false if not.
     */
    @Override
    public synchronized boolean requestStateChange(short expectedCurrentState, short desiredState, Object obj)
    throws NonRecoverableStateChangeException, RecoverableStateChangeException
    {
        // Since this concrete implementation of IStateManager manages a single
        // object, the input argument can be ignored.

        // make sure that the desired state is permissible
        if (!isValidState(desiredState))
        {
            throw new IllegalArgumentException("Invalid state: " + desiredState);
        }

        // the caller of this method should have called get state first and provide that
        // state as the first parameter - the caller provides the state he got to this
        // call and if the state has changed in between getting the state and calling this method,
        // this method will return false (the state change did not occur) .. in which case the
        // caller should reevaluate if it still desires to change the state
        if (expectedCurrentState != m_currentState)
        {
            return false;
        }

        boolean stateChanged = false;
        final short preCallState = m_currentState;
       
        doStateChange(desiredState);
        
        try
        {
            // if there is a controller for the change in state then call it
            String key = preCallState + "," + desiredState;
            IStateController controller = (IStateController)m_stateControllers.get(key);
            stateChanged = controller == null ? true : controller.changeState();            
        }
        finally
        {
            super.notifyAll(); // we already have the mutex - this is a synchronized method

            // let all the listeners know if the state changed succeeded or not
            Iterator iterator = m_stateListeners.iterator();
            while (iterator.hasNext())
            {
                if (stateChanged)
                {
                    ((IStateListener)iterator.next()).stateChanged(preCallState, m_currentState);
                }
                else
                {
                    ((IStateListener)iterator.next()).stateChangeFailed(m_currentState, desiredState);
                }
            }
        }

        return stateChanged;
    }

    /**
     * Does the work of changing state to the given state
     * including informing all listeners of the change
     * 
     * @param targetState
     */
    private void doStateChange(final short targetState) 
    {
        // let all listeners know that a state is changing

        final short oldState = m_currentState;

        synchronized(m_currentStateLock) // this is the only place after initial setup where we change the state
        {
            m_currentState = targetState;
        }

        Iterator iterator = m_stateListeners.iterator();
        while (iterator.hasNext())
        {
            ((IStateListener)iterator.next()).stateChanging(oldState, m_currentState);
        }      

    }

    /**
     * Register a state controller that will cause the behvioral changes associated
     * with the transition from one specified state to another specified state.
     */
    @Override
    public synchronized void registerStateController(IStateController stateController, short fromState, short toState, Object obj)
    {
        // Since this concrete implementation of IStateManager manages a single
        // object, the input argument can be ignored.

        // Make sure that the input states are valid
        if (!isValidState(fromState))
        {
            throw new IllegalArgumentException("Invalid state: " + fromState);
        }
        if (!isValidState(toState))
        {
            throw new IllegalArgumentException("Invalid state: " + toState);
        }

        String key = fromState + "," + toState;
        if (m_stateControllers.containsKey(key))
        {
            throw new IllegalStateException("State controller for specified state transition already registered");
        }

        m_stateControllers.put(key, stateController);
    }

    @Override
    public synchronized void registerStateListener(IStateListener stateListener, Object obj)
    {
        // Since this concrete implementation of IStateManager manages a single
        // object, the input argument can be ignored.

        // add the listener to the listener set
        m_stateListeners.add(stateListener);
    }

    private boolean isValidState(short state)
    {
        for (int i = 0; i < m_managedStates.length; i++)
        {
            if (m_managedStates[i] == state)
            {
                return true;
            }
        }

        return false;
    }
}