/* Copyright (c) 2009 Progress Software Corporation.  All Rights Reserved. */

package com.sonicsw.mf.interceptor.actional.soniclog;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;

import com.actional.lg.interceptor.sdk.ILogEntry;
import com.actional.lg.interceptor.sdk.LogEntryFactory;
import com.actional.lg.interceptor.sdk.UplinkConfig;

public final class SonicLogInterceptor
{
    public static final String MAX_MSG_SIZE_ATTR = "com.progress.actional.interceptor.soniclog.maxMsgSize";
    public static final String DISABLE_LOG_OUTPUT_ATTR = "com.progress.actional.interceptor.soniclog.disableLogOutput";
    
    private static int MAX_MSG_SIZE_VALUE = -1;
    private static boolean DISABLE_LOG_OUTPUT_VALUE = false;

    private SonicLogInterceptor()
    {
        // This class provides static methods, and is not meant for instantiation.
    }

    /**
     * Method to allow for updates to properties used by the logging interceptor.
     * 
     * @param name the name of the property to be updated
     * @param value the value of the property
     */
    public static void updateLoggingProperty(String name, Object value) 
    { 
        if (MAX_MSG_SIZE_ATTR.equals(name)) {
            // Handle case where the property is not set, assign default value.
            if (value == null)
            {
                MAX_MSG_SIZE_VALUE = -1;
            }
            else
            {
                MAX_MSG_SIZE_VALUE = ((Integer)value).intValue();
            }
        }
        else if (DISABLE_LOG_OUTPUT_ATTR.equals(name))
        {
            if (value == null)
            {
                DISABLE_LOG_OUTPUT_VALUE = false;
            }
            else
            {
                DISABLE_LOG_OUTPUT_VALUE = ((Boolean)value).booleanValue();
            }
        }
    }
    
    /**
     * Method to be executed before MF Container logMessage() methods.
     * 
     * @param componentId the MF Component Id that is logging the message
     * @param message the message to log
     * @param severity the MF severity level of the log
     * @return <code>true</code> if log should not be propagated to the actual MF container log, <code>false</code>
     *         otherwise.
     */
    public static boolean beforeLogMessage(final String componentId, final String message, final int severity)
    {
        ILogEntry logEntry = LogEntryFactory.beginLogEntry();

        if (logEntry == null)
        {
            return false;
        }

        SonicLoggingEvent loggingEvent = new SonicLoggingEvent(componentId, message, severity);
        String className = loggingEvent.getClassName();
        if (className.startsWith("org.apache.log4j"))
        {
            // This is a log4j event, e.g. logged by the XQLogAppender
            // We must ignore it, since it is already reported the Log4j interceptor
            return false;
        }

        String logMessage = loggingEvent.getMessage();
        if (MAX_MSG_SIZE_VALUE > 0 && logMessage.length() > MAX_MSG_SIZE_VALUE)
        {
            logMessage = logMessage.substring(0, MAX_MSG_SIZE_VALUE).concat("...");
        }

        logEntry.setLogMessage(logMessage);
        logEntry.setLoggerName(loggingEvent.getComponentId());
        logEntry.setLogLevel(loggingEvent.getLevel());
        logEntry.setLogClassName(className);
        logEntry.setLogMethodName(loggingEvent.getMethodName());
        logEntry.setLogThreadID(loggingEvent.getThreadId());
        logEntry.setLogThreadName(loggingEvent.getThreadName());
        logEntry.setLogTime(loggingEvent.getTimestamp());

        boolean processed = logEntry.end();

        // only skip console output if the log event was successfully processed and the analyzer is not down
        if (DISABLE_LOG_OUTPUT_VALUE && processed && UplinkConfig.getCurrent().getAlive())
        {
            return true;
        }

        return false;
    }

    /**
     * Utility method to convert a Throwable to a String representation of its stack trace.
     * 
     * @param th the Throwable to convert
     * @return the String representation of the Throwable stack trace
     */
    public static String throwableToString(final Throwable th)
    {
        ByteArrayOutputStream bstream = new ByteArrayOutputStream();
        PrintStream pstream = new PrintStream(bstream);
        th.printStackTrace(pstream);
        return bstream.toString();
    }
}
