/* Copyright (c) 2009 Progress Software Corporation.  All Rights Reserved. */

package com.sonicsw.mf.interceptor.actional.soniclog;

import com.actional.lg.interceptor.sdk.LogLevel;

import com.sonicsw.mf.common.runtime.Level;

public class SonicLoggingEvent
{
    private LogLevel level_;
    private String componentId_;
    private String message_;
    private String className_;
    private String methodName_;
    private String threadId_;
    private String threadName_;
    private long timestamp_;

    /**
     * Instantiates a new LoggingEvent, out of the parameters passed to MF Container log methods.
     * 
     * @param _componentId Id of the MF Component which is logging to the container log.
     * @param _message the logged message
     * @param _level the severity level
     */
    public SonicLoggingEvent(final String _componentId, final String _message, final int _level)
    {
        componentId_ = _componentId;
        message_ = _message;
        level_ = mapLevel(_level);
        threadId_ = Long.toString(Thread.currentThread().getId());
        threadName_ = Thread.currentThread().getName();
        analyzeStack();
        timestamp_ = System.currentTimeMillis();
    }

    /**
     * Maps MF Container severity level to canonical Log Interceptor API severity levels.
     * 
     * @param _level the container level to be mapped
     * @return the canonical level
     */
    private LogLevel mapLevel(final int _level)
    {
        switch (_level)
        {
            case Level.TRACE:
                return LogLevel.TRACE;
            case Level.FINER:
                return LogLevel.DEBUG;
            case Level.FINE:
                return LogLevel.DEBUG;
            case Level.INFO:
                return LogLevel.INFO;
            case Level.WARNING:
                return LogLevel.WARNING;
            case Level.SEVERE:
                return LogLevel.ERROR;
            case Level.CONFIG:
                return LogLevel.CONFIG;
            default:
                return LogLevel.UNKNOWN;
        }

    }

    private void analyzeStack()
    {
        Throwable th = new Throwable();

        StackTraceElement[] currentProc = null;
        try
        {
            currentProc = th.getStackTrace();
        }
        catch (Exception e)
        {
            e.printStackTrace();
            return;
        }

        int stackDepth = currentProc.length;

        for (int i = 1; i < stackDepth; i++)
        {
            String currentClassName = currentProc[i].getClassName();
            if (!currentClassName.startsWith("com.sonicsw.mf.interceptor.actional.soniclog") && !currentClassName.startsWith("com.sonicsw.mf.framework.agent") && !currentClassName.startsWith("com.sonicsw.xqimpl"))
            {
                className_ = currentProc[i].getClassName();
                methodName_ = currentProc[i].getMethodName();
                break;
            }
        }
    }

    /**
     * @return the name of the application class that has logged the message
     */
    public final String getClassName()
    {
        return className_;
    }

    /**
     * @return the canonical severity level of this log event
     */
    public final LogLevel getLevel()
    {
        return level_;
    }

    /**
     * @return the logged message
     */
    public final String getMessage()
    {
        return message_;
    }

    /**
     * @return the application method that has logged this event
     */
    public final String getMethodName()
    {
        return methodName_;
    }

    /**
     * @return the timestamp of creation of this event
     */
    public final long getTimestamp()
    {
        return timestamp_;
    }

    /**
     * @return the Sonic MF Component Id that is logging this event
     */
    public final String getComponentId()
    {
        return componentId_;
    }

    /**
     * @return the Java Thread Id logging this event
     */
    public final String getThreadId()
    {
        return threadId_;
    }

    /**
     * @return the Java Thread Name logging this event
     */
    public final String getThreadName()
    {
        return threadName_;
    }
}
