package com.sonicsw.mf.jmx.client;

import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.management.Notification;
import javax.management.NotificationFilter;

import com.sonicsw.mf.common.runtime.INotification;

/**
 * This implementation of the javax.management.NotificationFilter interface provides
 * a filter that uses regular expressions to determine which notifications should be
 * sent to associated listeners. The filter is a powerful alternative to the JMX
 * supplied NotificationFilterSupport class, providing the ability to filter on both
 * the notification type and the attribute values of <a href="../../common/runtime/INotification.html">Sonic notifications</a>.
 * <p>
 * The filter allows specification of a single expression for the notification type and a single
 * expression per attribute. When a notification is tested against the filter (see
 * <a href="#isNotificationEnabled(javax.management.Notification)">isNotificationEnabled</a>)
 * the result of testing each expression are AND'd together to form an aggregate result. For
 * example if a type expression such as "application\.session\.Subscribe" and attribute
 * expressions for the attributes "TopicName" and "UserName" of "App1|App2" and "Guest"
 * respectively are specified, then the listener would be sent only
 * <a href="../../../../../../mgmt_monitoring/BrokerNotifications.html#application.session.Subscribe"><code>application.session.Subscribe</code></a>
 * notifications that are associated with JMS subscriptions to topics "App1" or "App2" and
 * where the subscribing client's user name is "Guest".
 * <p>
 * Notifications will not be sent to associated listeners if an attribute expression is
 * specified for an attribute that does not exist in the notification or the attribute
 * exists but its value is null.
 * <p>
 * All filter expressions are case sensitive.
 * <p>
 * This filter uses the Java regular expression library when available (J2SE 1.4 or greater)
 * or the GNU regular expression library if not. The libraries must appear on the classpath
 * of the Sonic container hosting the source MBean, but are not generally required on the
 * classpath of the listening management client.
 */
public class ExpressionBasedNotificationFilter
implements NotificationFilter, Serializable
{
    private static final long serialVersionUID = 0L;

    // data members
    private String m_typeExpression;
    private HashMap m_attributeExpressions = new HashMap(); // indexed by attribute name

    private static boolean m_TestedForRegExParser = false;
    private static boolean m_useJavaRegEx = true;
    private static boolean m_useGNURegEx = true;
    private static volatile Constructor m_reConstructor;
    private static volatile Method m_matchTestMethod;
    
    @Override
    public synchronized boolean isNotificationEnabled(Notification notification)
    {
        // if a type expression has been defined then check that
        if (m_typeExpression != null)
        {
            if (!isMatch(m_typeExpression, notification.getType()))
            {
                return false;
            }
        }

        // loop through all enabled attributes, checking for match
        synchronized(m_attributeExpressions)
        {
            // if there are no attribute expressions to check then we've got a amatch
            if (m_attributeExpressions.isEmpty())
            {
                return true;
            }

            // if there are attribute expressions to check and the notification is not an
            // MF notification then we assume there is not match
            if (!(notification instanceof INotification))
            {
                return false;
            }

            HashMap attributes = ((INotification)notification).getAttributes();

            Iterator iterator = m_attributeExpressions.entrySet().iterator();
            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry)iterator.next();
                String attributeName = (String)entry.getKey();

                // if they have specified an expression for an attribute that does not
                // exist in this notification then of course the notification will
                // not match the filter!
                if (!attributes.containsKey(attributeName))
                {
                    return false;
                }

                Object attributeValue = attributes.get(attributeName);

                // if there is no value for the named attribute then we can't possibly have a match
                if (attributeValue == null)
                {
                    return false;
                }

                // regex only works with string input, so...
                if (!isMatch((String)entry.getValue(), attributeValue.toString()))
                {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Enable notifications the type of which matches the specified expression (and any
     * defined attribute expressions) to be sent to the listener. Setting the expression to
     * null clears a previously set expression.
     */
    public synchronized void setTypeExpression(String expression)
    {
        m_typeExpression = expression;
    }

    /**
     * Gets a previously set name expression (or null if a name expression has not be set).
     */
    public synchronized String getTypeExpression()
    {
        return m_typeExpression;
    }

    /**
     * Enable notifications with an attribute that matches the specified expression
     * (and any defined type or other attribute expressions) to be sent to the listener.
     * Setting the expression to null clears a previously set expression for the given
     * attribute name.
     */
    public synchronized void setAttributeExpression(String attributeName, String expression)
    {
        if (attributeName == null || attributeName.length() == 0)
        {
            throw new IllegalArgumentException("Attribute name null or zero length");
        }

        if (expression == null)
        {
            m_attributeExpressions.remove(attributeName);
        }
        else
        {
            m_attributeExpressions.put(attributeName, expression);
        }
    }

    /**
     * Gets previously set attribute expressions keyed by attribute name.
     */
    public synchronized Map getAttributeExpressions()
    {
        return (Map)m_attributeExpressions.clone();
    }

    /**
     * Clear all previously defined attribute name expressions.
     */
    public synchronized void clearAllAttributeExpressions()
    {
        m_attributeExpressions.clear();
    }
    
    /**
     * Compares to see if the given object matches the notification filter.
     * <p>
     * A match will occur if the given object is an ExpressionBasedNotificationFilter with
     * the same type expression and same set of attribute expressions.
     * 
     * @param obj The reference object with which to compare.
     * @return true if this object is the same as the obj argument; false otherwise.
     */
    @Override
    public boolean equals(Object obj)
    {
        if (obj == null || this.getClass() != obj.getClass())
        {
            return false;
        }        
        ExpressionBasedNotificationFilter otherFilter = (ExpressionBasedNotificationFilter)obj;
        
        if (otherFilter.m_typeExpression == null && m_typeExpression != null)
        {
            return false;
        }
        if (otherFilter.m_typeExpression != null && m_typeExpression == null)
        {
            return false;
        }
        
        if (otherFilter.m_attributeExpressions.size() != m_attributeExpressions.size())
        {
            return false;
        }
        
        Iterator attributeExpressions = otherFilter.m_attributeExpressions.entrySet().iterator();
        while (attributeExpressions.hasNext())
        {
            Map.Entry attributeExpressionEntry = (Map.Entry)attributeExpressions.next();
            Object expression = m_attributeExpressions.get(attributeExpressionEntry.getKey());
            if (expression == null)
            {
                // null expressions are not allowed, therefor if we get null is because there is no equivalent
                return false;
            }
            
            if (!attributeExpressionEntry.getValue().equals(expression))
            {
                return false;
            }
        }
        
        return true;
    }

    @Override
    public int hashCode() {
        List<Object> values = new ArrayList<Object>();
        values.add(m_typeExpression);
        if(m_attributeExpressions != null && ! m_attributeExpressions.isEmpty()) {
            values.addAll(m_attributeExpressions.values());
        }
        return Objects.hash(values.toArray());        
    }
    
    private boolean isMatch(String regex, String input)
    {
        if (!m_TestedForRegExParser)
        {
            testForRegExParser();
        }

        if (m_useJavaRegEx)
        {
            return isJavaRegexMatch(regex, input);
        }
        else
        if (m_useGNURegEx)
        {
            return isGNURegexMatch(regex, input);
        }
        else
        {
            throw new IllegalStateException("Regular expression parser (Java or GNU) not available on classpath");
        }
    }

    private boolean isJavaRegexMatch(String regex, String input)
    {
        try
        {
            Boolean isMatch = (Boolean)m_matchTestMethod.invoke(null, new Object[]{regex, input});
            return isMatch.booleanValue();
        }
        catch(InvocationTargetException e)
        {
            Throwable target = e.getTargetException();
            if (target instanceof RuntimeException)
            {
                throw (RuntimeException)target;
            }
            e.printStackTrace(); // since this should never occur
            return false;
        }
        catch(IllegalAccessException e)
        {
            e.printStackTrace(); // since this should never occur
            return false;
        }
    }

    private boolean isGNURegexMatch(String regex, String input)
    {
        try
        {
            Object re = m_reConstructor.newInstance(new Object[]{regex});
            Boolean isMatch = (Boolean)m_matchTestMethod.invoke(re, new Object[]{input});
            return isMatch.booleanValue();
        }
        catch(InvocationTargetException e)
        {
            Throwable target = e.getTargetException();
            if (target instanceof RuntimeException)
            {
                throw (RuntimeException)target;
            }
            e.printStackTrace(); // since this should never occur
            return false;
        }
        catch(IllegalAccessException e)
        {
            e.printStackTrace(); // since this should never occur
            return false;
        }
        catch(InstantiationException e)
        {
            e.printStackTrace(); // since this should never occur
            return false;
        }
    }

    private void testForRegExParser()
    {
        // is java regex available?
        try
        {
            Class patternClass = Class.forName("java.util.regex.Pattern");
            Class charSequenceClass = Class.forName("java.lang.CharSequence");
            m_matchTestMethod = patternClass.getMethod("matches", new Class[]{String.class, charSequenceClass});
        }
        catch(ClassNotFoundException e)
        {
            m_useJavaRegEx = false;
        }
        catch(NoSuchMethodException e)
        {
            m_useJavaRegEx = false;
        }

        if (!m_useJavaRegEx)
        {
            // is GNU regex available?
            try
            {
                Class reClass = Class.forName("gnu.regexp.RE");
                m_reConstructor = reClass.getConstructor(new Class[]{Object.class});
                m_matchTestMethod = reClass.getMethod("isMatch", new Class[]{Object.class});
            }
            catch(ClassNotFoundException e)
            {
                m_useGNURegEx = false;
            }
            catch(NoSuchMethodException e)
            {
                m_useGNURegEx = false;
            }
        }

        m_TestedForRegExParser = true;
    }
}