package com.sonicsw.mf.jmx.client;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;

/**
 * Defines the JMS connector address. This object is used by the connector client to check that the address
 * used as the argument of its connect method is of the appropriate type.
 * <p>
 * No defaults are made.
 *
 * @see JMSConnectorClient
 */
public final class JMSConnectorAddress
{
    private Hashtable m_env;
    private String m_managementNode;
    private String m_serverIdentity;

    /**
     * Construct a JMS connector address using the given environment.
     * <p>
     * The environment (specified as a hashtable of key/value pairs) should
     * contain sufficient information from which a SonicMQ implementation
     * of a JMS TopicConnectionFactory may be created. Each key should map to a
     * java bean style attribute name of such a factory. The typical set of
     * key [attribute] names will be:
     * <pre>
     *           ConnectionURLs
     *           DefaultUser
     *           DefaultPassword
     * </pre>
     */
    public JMSConnectorAddress(Hashtable env)
    {
        m_env = env;
    }

    /*
     * If management communications are required to span routing nodes
     * the routing node through which the container(s) hosting the Directory Service
     * and Agent Manager components will communicate must be specified. If not
     * the local routing node will be assumed.
     */
    public void setManagementNode(String node) { m_managementNode = node; }

    /*
     * @see #setManagementNode(String)
     */
    public String getManagementNode() { return m_managementNode; }

    /**
     * If the connector client is to be dedicated to a particular Sonic container,
     * the identity of that container must be provided prior to connecting. Each
     * Sonic container contains a JMX MBeanServer whose default domain equals the
     * container identity and all Sonic compliant component will be MBeans within
     * that JMX default domain.
     * <p>
     * If the server identity is not set, remote MBeanServer methods from which the
     * server identity cannot be derived, will fail.
     *
     * @param identity A tokenized string of the form "&lt;mf domain&gt;.&lt;mf container name&gt;", where:
     * <pre>
     *           &lt;mf domain&gt;
     *              - is the management framework domain name for the
     *                deployment to which we are trying to connect
     *           &lt;mf container name&gt;
     *              - is the management framework container name for the
     *                particular container (and hence MBeanServer instance)
     *                to which we are trying to connect.
     * </pre>
     * <p>
     *                 A value of null is a valid value, unsetting a previously set identity.
     *
     * @see JMSConnectorClient#connect(JMSConnectorAddress)
     */
    public void setServerIdentity(String identity)
    {
        if (identity == null)
        {
            m_serverIdentity = identity;
            return;
        }

        StringTokenizer st = new StringTokenizer(identity, ".", true);

        if (st.countTokens() != 3)
        {
            throw new IllegalArgumentException("Identity not of format \"<domain>.<container>\": " + identity);
        }

        String firstToken = st.nextToken();
        String secondToken = st.nextToken();
        String thirdToken = st.nextToken();
        if (firstToken.length() == 0 || !secondToken.equals(".") || thirdToken.length() == 0)
        {
            throw new IllegalArgumentException("Identity does not supply values of the format \"<domain>.<container>\": " + identity);
        }

        m_serverIdentity = identity;
    }

    /**
     * Where the address implies a dedicated communication channel to a particular connector
     * server the server identity will not be null.
     *
     * @return The server identity or null if not set.
     *
     * @see #setServerIdentity(String)
     */
    public String getServerIdentity() { return m_serverIdentity; }


    //
    // IConnectorAddress implementation
    //

    /**
     * @return The environment provided when constructing the connector.
     *
     * @see #JMSConnectorAddress(Hashtable)
     */
    public Hashtable getEnv()
    {
        return m_env;
    }

    /**
     * Compares the connector address with another connector address.
     * <p>
     * Two connector addresses will be equal if they have an environment
     * with matching key/values pairs.
     *
     * @param object A connector adddress to test for equality.
     * @return true if equal, false if not equal
     */
    @Override
    public boolean equals(Object object)
    {
        if (!(object instanceof JMSConnectorAddress))
        {
            return false;
        }

        JMSConnectorAddress address = (JMSConnectorAddress)object;
        Hashtable otherEnv = address.getEnv();

        if (m_env.size() != otherEnv.size())
        {
            return false;
        }

        Enumeration keys = m_env.keys();
        while (keys.hasMoreElements())
        {
            Object key = keys.nextElement();
            Object thisValue = m_env.get(key);
            Object otherValue = otherEnv.get(key);

            if (otherValue == null)
            {
                return false;
            }
            if (!thisValue.equals(otherValue))
            {
                return false;
            }
        }

        return true;
    }

    /**
     * Sets the secondary connection URLs, if applicable (Fault Tolerant-enabled)
     * <p>
     * No-op as of Sonic Management v7.5
     *
     * @param secondaryURLs  A comma-delimited set of SonicMQ URLs
     * 
     * @deprecated As of Sonic Management v7.5
     */
    public void setSecondaryConnectionURLs(String secondaryURLs)
    {
    }

    /**
     * Gets the secondary connection URLs, if applicable (Fault Tolerant-enabled)
     *
     * @return null
     * 
     * @deprecated As of Sonic Management v7.5
     */
    public String getSecondaryConnectionURLs()
    {
        return null;
    }

    /**
     * Sets the secondary connection node, if applicable (Fault Tolerant-enabled)
     * <p>
     * No-op as of Sonic Management v7.5
     *
     * @param secondaryNode  The SonicMQ DRA node name of the secondary management node
     * 
     * @deprecated As of Sonic Management v7.5
     */
    public void setSecondaryNode(String secondaryNode)
    {
    }

    /**
     * Gets the secondary connection node, if applicable (Fault Tolerant-enabled)
     *
     * @return null
     * 
     * @deprecated As of Sonic Management v7.5
     */
    public String getSecondaryNode()
    {
        return null;
    }

    /**
     * Sets the flag that indicates if a backup DS or AM should be tried
     * when a request to either fails due to a timeout.
     * @deprecated
     */
    public void setTryDSAndAMBackupsOnFailure(Boolean tryDSAndAMBackupsOnFailure)
    {
    }

    /**
     * Gets the flag that indicates if a backup DS or AM should be tried
     * when a request to either fails due to a timeout.
     * @deprecated
     */
    public Boolean getTryDSAndAMBackupsOnFailure()
    {
        return new Boolean(false);
    }

    //
    // Other public methods
    //

    @Override
    public int hashCode() { return 0; } // return a common hashcode so only the equals method makes a difference for keying

}
