package com.sonicsw.mf.jmx.client;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.InvalidAttributeValueException;
import javax.management.JMRuntimeException;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanRegistrationException;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.QueryExp;
import javax.management.ReflectionException;
import javax.management.RuntimeOperationsException;

import com.sonicsw.mf.comm.IRetryCallback;
import com.sonicsw.mf.comm.InvokeTimeoutCommsException;
import com.sonicsw.mf.comm.jms.ConnectorClient;
import com.sonicsw.mf.comm.jms.DurableConnector;
import com.sonicsw.mf.common.util.ObjectNameHelper;

/**
 * A JMS Connector uses the SonicMQ JMS implementation to provide JMX Connector
 * functionality. Since the JMS Connector enables a single connection point for
 * management communications to multiple JMX servers, a JMS address can indicate
 * either the connector should be dedicated to a particular MBeanServer ("bound") or capable
 * of addressing multiple servers ("unbound") (acheived through JMX domain & ObjectName patterns).
 * <p>
 * JMS Connector Clients are designed to be used exclusively with the Sonic Management containers
 * and components which expose themselves using JMX and the Sonic Management
 * JMX domain & ObjectName patterns. In an unbounded mode, JMX ObjectNames must reflect the said
 * patterns otherwise remote calls will timeout.
 * <p>
 * Sonic Management JMX ObjectName patterns are of the form
 * "&lt;SM&nbsp;domain&gt;.&lt;SM&nbsp;container&nbsp;name&gt;:ID=&lt;SM&nbsp;component&nbsp;name&gt;".
 * <p>
 * The JMS Connector client exposes MBean server methods suitable for remoting. If the
 * JMS Connector is not dedicated to a particular MBean server ("unbound"), MBean server methods
 * that are MBean server specific (e.g. such as getDefaultDomain()) will throw an exception when
 * called. Remote support is indicated in the documentation for each method.
 * <p>
 * The JMSConnectorClient wraps all unreported exceptions/error in a CommunicationException.
 * <p>
 * Unlike other JMX connectors, the default connector connect behavior is to wait for a connection to be
 * established. Using the <a href="#connect(JMSConnectorAddress)"><code>connect(JMSConnectorAddress)</code></a>
 * the connector will wait until a connection can be established or a terminal condition (e.g. inauthentic client)
 * occurs. This connection behavior can be modified to timeout after a specified period (see
 * <a href="#connect(JMSConnectorAddress, long)"><code>connect(JMSConnectorAddress, long)</code></a>).
 * <p>
 * The connector remote request behavior can also be customized; by default requests will timeout after 30 seconds
 * and this timeout may encapsulate waiting for a (re)connect or a response to arrive. It is possible that requests
 * will timeout due to heavily loaded systems and large amounts of data. In such cases the default request timeout
 * may be modified using <a href="#setRequestTimeout(long)"><code>setRequestTimeout(long)</code></a>. Also since
 * the default behavior is to span short term disconnects, the connector behavior can be modified to fail requests
 * immediately if the connector is not actaully connected at the time the request is made; this done using
 * <a href="#setFailWhenDisconnected(boolean)"><code>setFailWhenDisconnected(boolean)</code></a>. This specific
 * modified behavior does not apply to the add/remove of notification subscriptions.
 * <p>
 * The JMS Connector implements its own reconnect logic that includes reestablishing context. This allows the
 * connector to handle short-term network failures and alleviates API programmers from implementing their own
 * reconnect logic.
 * <p>
 * NOTE: JMX (as of v1.1) does not formalize the remote access to JMX MBean servers. Sun
 * have provided sample connectors on which this JMS connector has been loosely modelled.
 * When remote access has been formalized in the JMX specification, this JMS Connector
 * will be made conformant to the specification.
 *
 * @see JMSConnectorAddress
 * @see CommunicationException
 * @see javax.management.MBeanServer
 */
public class JMSConnectorClient
implements IRemoteMBeanServer
{
    private volatile boolean m_isClosing = false;

    // the underlying JMS connector
    protected ConnectorClient m_connector;

    // don't process requests if the connector is in a transient disconnect state
    private boolean m_failWhenDisconnected = false;

    // the prior connector address and server identity (if any)
    private JMSConnectorAddress m_address;
    private String m_serverIdentity;

    // a table that maps MBean object names to actual listeners
    private HashMap m_mBeanToLocalListeners = new HashMap();
    // a table that maps MBean object names to subscription renewal threads
    private HashMap<String,SubscriptionRenewalThread> m_mBeanToSubscriptionRenewalThreads = new HashMap<String,SubscriptionRenewalThread>();
    private HashMap m_notificationSourceUsages = new HashMap();

    // failure list so we only report once between successes
    private HashSet m_renewalFailures = new HashSet();
    
    private boolean m_needsRenewal;

    // timeout value for connection setup attempt
    private long m_connectTimeout;

    // timeout value for socket connection setup attempt (each URL in the connection URL list)
    private long m_socketConnectTimeout;
    
    // timeout value for client requests
    private long m_requestTimeout = ConnectorClient.REQUEST_TIMEOUT_DEFAULT;

    // notification subscription timeout
    private long m_notificationSubscriptionTimeout = DurableConnector.DURABLE_SUBSCRIPTION_TTL;  // default is same as durable subscription timeout
    // notification subscription renewal interval
    private long m_notificationSubscriptionRenewalInterval = 30000;
    
    private boolean m_useOnewaySubscriptionRequests = false;

    private static final String ADD_NOTIFICATION_LISTENER_METHOD_NAME = "addNotificationListener";

    static final char COMPOUND_HANDBACK_ID_DELIMITER = (char)8;

    // constants for params and signatures
    // java
    private static final Object[] EMPTY_PARAMS_ARRAY = new Object[0];
    private static final String[] EMPTY_SIGNATURE_ARRAY = new String[0];
    private static final String OBJECT_CLASSNAME = Object.class.getName();
    private static final String STRING_CLASSNAME = String.class.getName();
    private static final String OBJECT_ARRAY_CLASSNAME = Object[].class.getName();
    private static final String STRING_ARRAY_CLASSNAME = String[].class.getName();
    private static final String LONG_CLASSNAME = Long.class.getName();
    // jmx
    private static final String OBJECTNAME_CLASSNAME = ObjectName.class.getName();
    private static final String ATTRIBUTE_CLASSNAME = Attribute.class.getName();
    private static final String ATTRIBUTELIST_CLASSNAME = AttributeList.class.getName();
    private static final String NOTIFICATIONLISTENER_CLASSNAME = NotificationListener.class.getName();
    private static final String NOTIFICATIONFILTER_CLASSNAME = NotificationFilter.class.getName();
    private static final String QUERYEXP_CLASSNAME = QueryExp.class.getName();
    private static final String[] ADDREMOVE_NOTIFICATION_LISTENER_SIGNATURE = new String[]
    {
        OBJECTNAME_CLASSNAME,
        NOTIFICATIONLISTENER_CLASSNAME,
        NOTIFICATIONFILTER_CLASSNAME,
        OBJECT_CLASSNAME
    };
    private static final String[] ADDREMOVE_NOTIFICATION_LISTENER_WITH_TIMEOUT_SIGNATURE = new String[]
    {
        OBJECTNAME_CLASSNAME,
        NOTIFICATIONLISTENER_CLASSNAME,
        NOTIFICATIONFILTER_CLASSNAME,
        OBJECT_CLASSNAME,
        LONG_CLASSNAME
    };

    // listener to connection changes including permanent failures
    private com.sonicsw.mf.jmx.client.IConnectionListener m_connectionListener;
    // listener to permanent connection failures (deprecated)
    private com.sonicsw.mf.jmx.client.IExceptionListener m_exceptionListener;

    //
    // As per Sun's JDMK RemoteMBeanServer interface
    //

    /**
     * Connects the connector to the underlying JMS transport.
     * <p>
     * NOTE: connect() will block until either the connection is established, a non-recoverable
     *       condition was determined (in which case a runtime exception will be thrown) or
     *       the thread invoking connect() is interrupted. Recoverable conditions include those
     *       consitions that might reasonably be expected, such as when the SonicMQ broker designated
     *       to carry management communications comes on line. Non-recoverable conditions include
     *       more permanent condition like the inability to resolve an IP address or a security
     *       failure such as an inauthentic client.
     *
     * @param address The attributes describing how to connect to the underlying JMS transport.
     * @return If the connector address dedicates the connector to a particular MBeanServer, the
     *         MBeanServer identity is returned, otherwise null is returned.
     *
     * @see #disconnect()
     */
    @Override
    public synchronized String connect(JMSConnectorAddress address)
    {
        return connect(address, 0);
    }

    /**
     * Connects the connector to the underlying JMS transport.
     * <p>
     * NOTE: connect() will block until either the connection is established, a non-recoverable
     *       condition was determined (in which case a runtime exception will be thrown) or
     *       the thread invoking connect() is interrupted. Recoverable conditions include those
     *       consitions that might reasonably be expected, such as when the SonicMQ broker designated
     *       to carry management communications comes on line. Non-recoverable conditions include
     *       more permanent condition like the inability to resolve an IP address or a security
     *       failure such as an inauthentic client.
     *
     * @param address The attributes describing how to connect to the underlying JMS transport.
     * @param timeout The number of milliseconds within which the initial connection must be established.
     * @return If the connector address dedicates the connector to a particular MBeanServer, the
     *         MBeanServer identity is returned, otherwise null is returned.
     *
     * @see #disconnect()
     */
    public synchronized String connect(JMSConnectorAddress address, long timeout)
    {
        if (m_address != null && !address.equals(m_address))
        {
            throw new IllegalArgumentException("Cannot connect with a different address.");
        }

        if (m_connector != null)
        {
            throw new java.lang.IllegalStateException("Already connected");
        }

        m_connector = new ConnectorClient("JMXCLIENT");
        m_connector.setRequestTimeout(m_requestTimeout);
        m_connector.setConnectTimeout(m_connectTimeout);
        m_connector.setSocketConnectTimeout(m_socketConnectTimeout);

        String managementNode = address.getManagementNode();
        if (managementNode != null && managementNode.length() > 0)
        {
            m_connector.setManagementNode(managementNode);
        }
        m_connector.setConnectionListener(new com.sonicsw.mf.comm.IConnectionListener()
        {
            public String lrn;
            @Override
            public void onFailure(Exception e)
            {
                if (JMSConnectorClient.this.m_connectionListener != null)
                {
                    JMSConnectorClient.this.m_connectionListener.onFailure(e);
                }
                // the following listener is deprecated
                if (JMSConnectorClient.this.m_exceptionListener != null)
                {
                    JMSConnectorClient.this.m_exceptionListener.onException(e);
                }
            }

            @Override
            public void onReconnect(String localRoutingNode)
            {
                // To avoid potential deadlock, commented out below section,
                // and instead added code to create new thread to invoke
                // connection listener's "onReconnect" method...
                /* synchronized(JMSConnectorClient.this)
                {
                    JMSConnectorClient.this.restartSubscriptionRenewalThreads();
                    if (JMSConnectorClient.this.m_connectionListener != null)
                        JMSConnectorClient.this.m_connectionListener.onReconnect(localRoutingNode);
                }
                */

                lrn = localRoutingNode;
                Thread reconnectHandler = new Thread("JMSConnectorClient - Reconnect Handler")
                {
                    @Override
                    public void run()
                    {
                        synchronized(JMSConnectorClient.this)
                        {
                            JMSConnectorClient.this.restartSubscriptionRenewalThreads();
                            if (JMSConnectorClient.this.m_connectionListener != null)
                            {
                                JMSConnectorClient.this.m_connectionListener.onReconnect(lrn);
                            }
                        }
                    }
                };
                reconnectHandler.setDaemon(true);
                reconnectHandler.start();
            }

            @Override
            public void onDisconnect()
            {
                synchronized(JMSConnectorClient.this)
                {
                    JMSConnectorClient.this.stopSubscriptionRenewalThreads();
                    if (JMSConnectorClient.this.m_connectionListener != null)
                    {
                        JMSConnectorClient.this.m_connectionListener.onDisconnect();
                    }
                }
            }
        });

        try
        {
            m_connector.connect(address.getEnv(), timeout);
        }
        catch(Exception e)
        {
            if (m_connector != null)
            {
                m_connector.close();
                m_connector = null;
            }

            // set the cause through reflection in case this is not J2SE >= 1.4
            JMRuntimeException jmre = new JMRuntimeException(e.getMessage());
            try
            {
                Method method = jmre.getClass().getMethod("initCause", new Class[] { Throwable.class});
                method.invoke(jmre, new Object[] { e });
            }
            catch(Exception ex) { }
            throw jmre;
        }

        m_address = address;
        m_serverIdentity = address.getServerIdentity();
        return m_serverIdentity;
    }

    /**
     * Gets the (singleton) exception listener for the connector.
     *
     * @deprecated IConnectionListener superseeds IExceptionListener
     *
     * @return Reference to an instance of a registered IExceptionListener implementation
     * @see #getConnectionListener(IConnectionListener)
     * @see #setConnectionListener(IConnectionListener)
     */
    public IExceptionListener getExceptionListener() { return m_exceptionListener; }

    /**
     * Sets the (singleton) exception listener for the connector.
     *
     * @deprecated IConnectionListener supercedes IExceptionListener
     *
     * @param listener Reference to an IExceptionListener implementation
     * @see #getConnectionListener(IConnectionListener)
     * @see #setConnectionListener(IConnectionListener)
     */
    public synchronized void setExceptionListener(IExceptionListener listener) { m_exceptionListener = listener; }

    /**
     * Gets the (singleton) connection listener for the connector.
     *
     * @return Reference to a registered IConnectionListener implementation
     */
    public IConnectionListener getConnectionListener() { return m_connectionListener; }

    /**
     * Sets the (singleton) connection listener for the connector.
     *
     * @param listener Reference to an instance of an IConnectionListener implementation
     */
    public synchronized void setConnectionListener(IConnectionListener listener) { m_connectionListener = listener; }

    /**
     * Gets the (singleton) orphaned reply listener for the connector.
     *
     * @return Reference to a registered IOrphanedReplyListener instance
     */
    public IOrphanedReplyListener getOrphanedReplyListener() { return (IOrphanedReplyListener)m_connector.getOrphanedReplyListener(); }

    /**
     * Sets the (singleton) orphaned reply listener for the connector.
     *
     * @param listener Reference to an instance of an IOrphanedReplyListener implementation
     */
    public synchronized void setOrphanedReplyListener(IOrphanedReplyListener listener) { m_connector.setOrphanedReplyListener(listener); }

    /**
     * Disconnect the connector from the underlying JMS transport.
     *
     * @see #connect(JMSConnectorAddress)
     */
    @Override
    public synchronized void disconnect()
    {
        if (m_connector == null)
        {
            return;
        }

        m_isClosing = true;
        m_connector.closePending();

        // remove any registered notification listeners
        synchronized(m_mBeanToLocalListeners)
        {
            // iterate through the component(Mbean)->listeners map (key=component canonical name,value=map of listeners)
            Iterator componentMbeans = m_mBeanToLocalListeners.keySet().iterator();
            while (componentMbeans.hasNext())
            {
                String componentCanonicalName = (String) componentMbeans.next();
                HashMap listenerDelegateMap = (HashMap) m_mBeanToLocalListeners.get(componentCanonicalName);
                Iterator listeners = listenerDelegateMap.keySet().iterator();
                while (listeners.hasNext())
                {
                    NotificationListener listener = (NotificationListener) listeners.next();
                    NotificationListenerDelegate delegate = (NotificationListenerDelegate) listenerDelegateMap.get(listener);
                    // send a request to remove the listener to the server-side
                    // (we don't really care if the request succeeds, as the notification subscription
                    // for the listener will expire at some point anyhow).
                    try
                    {
                        ObjectName objectName = new ObjectName(componentCanonicalName);
                        m_connector.invokeOneway(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), componentCanonicalName, "removeNotificationListener",
                                                 new Object[] { objectName, delegate },
                                                 new String[] { OBJECTNAME_CLASSNAME, NOTIFICATIONLISTENER_CLASSNAME },
                                                 m_requestTimeout);
                    }
                    catch (Exception e) {}

                    // close the delegate
                    delegate.close();

                } // end of "while (listeners.hasNext())"
            } // end of "while (componentsMbean.hasNext())"
            /* for debugging */
            /*
            Iterator iterator = m_mBeanToLocalListeners.values().iterator();
            while (iterator.hasNext())
            {
                Iterator remoteListeners = ((HashMap)iterator.next()).values().iterator();
                while (remoteListeners.hasNext())
                    ((NotificationListenerDelegate)remoteListeners.next()).close();
            }
            */
            m_mBeanToLocalListeners.clear();

            Iterator<SubscriptionRenewalThread> threads = m_mBeanToSubscriptionRenewalThreads.values().iterator();
            while (threads.hasNext())
            {
                SubscriptionRenewalThread thread = threads.next();
                synchronized(thread.getThreadLockObject())
                {
                    m_needsRenewal = true;
                    thread.getThreadLockObject().notifyAll();
                }
            }
            m_mBeanToSubscriptionRenewalThreads.clear();
        }

        m_connector.close();
        m_connector = null;
    }

    /**
     * Sets the default timeout the connector waits for responses from remote requests. The minimum is
     * 10000 milliseconds (silently enforced).
     *
     * @param milliseconds Timeout value in milliseconds
     */
    public void setRequestTimeout(long milliseconds)
    {
        m_requestTimeout = (milliseconds >= ConnectorClient.REQUEST_TIMEOUT_MINIMUM) ? milliseconds : ConnectorClient.REQUEST_TIMEOUT_MINIMUM;

        if (m_connector != null)
        {
            m_connector.setRequestTimeout(m_requestTimeout);
        }
    }

    /**
     * Gets the default timeout the connector waits for responses from remote requests.
     *
     * @return Current request timeout value in milliseconds
     */
    public long getRequestTimeout()
    {
        return m_requestTimeout;
    }

    /**
     * Sets the duration, in seconds, that a notification subscription will "live"
     * before it must be renewed
     */
    public void setNotificationSubscriptionTimeout(long seconds)
    {
        m_notificationSubscriptionTimeout = seconds * 1000;
    }

    /**
     * Gets the duration, in seconds, a notification subscription will "live"
     * before it must be renewed
     *
     * @return Current notification subscription timeout value in seconds
     */
    public long getNotificationSubscriptionTimeout()
    {
        return m_notificationSubscriptionTimeout / 1000;
    }

    /**
     * Sets the interval, in seconds, that notification subscriptions will be renewed
     */
    public void setNotificationSubscriptionRenewalInterval(long interval)
    {
        m_notificationSubscriptionRenewalInterval = interval * 1000;
    }

    /**
     * Gets the interval, in seconds, that notification subscriptions will be renewed
     *
     * @return Current notification subscription renewal interval value in seconds
     */
    public long getNotificationSubscriptionRenewalInterval()
    {
        return m_notificationSubscriptionRenewalInterval / 1000;
    }

    /**
     * When true, causes requests (invoke, setAttribute, etc.) to fail immediately if the connector is in a
     * transient disconnected state.
     *
     * @param failWhenDisconnected Flag to indicate if immediate failure should occur when in disconnected state
     */
    public void setFailWhenDisconnected(boolean failWhenDisconnected) { m_failWhenDisconnected = failWhenDisconnected; }

    /**
     * @return Current setting of "failWhenDisconnected" flag
     * @see #setFailWhenDisconnected(boolean)
     */
    public boolean getFailWhenDisconnected() {  return m_failWhenDisconnected; }

    /**
     * Supported connector modes: bounded
     *
     * @see javax.management.MBeanServer#createMBean(String, ObjectName)
     */
    @Override
    public ObjectInstance createMBean(String className, ObjectName objectName)
    throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException,
           MBeanException, NotCompliantMBeanException
    {
        return createMBean(className, objectName, EMPTY_PARAMS_ARRAY, EMPTY_SIGNATURE_ARRAY);
    }

    /**
     * Supported connector modes: bounded
     *
     * @see javax.management.MBeanServer#createMBean(String, ObjectName, Object[], String[])
     */
    @Override
    public ObjectInstance createMBean(String className, ObjectName objectName, Object[] params, String[] signature)
    throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException,
           MBeanException, NotCompliantMBeanException
    {
        checkNotNull("className", className);
        checkNotNull("objectName", objectName);
        checkNotNull("params", params);
        checkNotNull("signature", signature);

        testOperationSupported();
        testConnector();
        testFailWhenDisconnected();

        ObjectInstance returnValue = null;

        try
        {
            returnValue = (ObjectInstance)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, m_serverIdentity, null, "createMBean", new Object[] { className, objectName, params, signature }, new String[] { STRING_CLASSNAME, OBJECTNAME_CLASSNAME, OBJECT_ARRAY_CLASSNAME, STRING_ARRAY_CLASSNAME });
        }
        catch(ReflectionException re) { throw re; }
        catch(InstanceAlreadyExistsException iaee) { throw iaee; }
        catch(MBeanRegistrationException mre) { throw mre; }
        catch(MBeanException mbe) { throw mbe; }
        catch(NotCompliantMBeanException ncme) { throw ncme; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed createMBean");
        }

        return returnValue;
    }

    /**
     * Supported connector modes: bounded
     *
     * @see javax.management.MBeanServer#createMBean(String, ObjectName, ObjectName)
     */
    @Override
    public ObjectInstance createMBean(String className, ObjectName objectName, ObjectName loaderName)
    throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException,
           MBeanException, NotCompliantMBeanException, InstanceNotFoundException
    {
        return createMBean(className, objectName, loaderName, EMPTY_PARAMS_ARRAY, EMPTY_SIGNATURE_ARRAY);
    }

    /**
     * Supported connector modes: bounded
     *
     * @see javax.management.MBeanServer#createMBean(String, ObjectName, ObjectName, Object[], String[])
     */
    @Override
    public ObjectInstance createMBean(String className, ObjectName objectName, ObjectName loaderName, Object[] params, String[] signature)
    throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException,
           MBeanException, NotCompliantMBeanException, InstanceNotFoundException
    {
        checkNotNull("className", className);
        checkNotNull("objectName", objectName);
        checkNotNull("params", params);
        checkNotNull("signature", signature);

        testOperationSupported();
        testConnector();
        testFailWhenDisconnected();

        ObjectInstance returnValue = null;

        try
        {
            returnValue = (ObjectInstance)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, m_serverIdentity, null, "createMBean", new Object[] { className, objectName, loaderName, params, signature }, new String[] { STRING_CLASSNAME, OBJECTNAME_CLASSNAME, OBJECTNAME_CLASSNAME, OBJECT_ARRAY_CLASSNAME, STRING_ARRAY_CLASSNAME });
        }
        catch(ReflectionException re) { throw re; }
        catch(InstanceAlreadyExistsException iaee) { throw iaee; }
        catch(MBeanRegistrationException mre) { throw mre; }
        catch(MBeanException mbe) { throw mbe; }
        catch(NotCompliantMBeanException ncme) { throw ncme; }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed createMBean");
        }

        return returnValue;
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#getAttribute(ObjectName, String)
     */
    @Override
    public Object getAttribute(ObjectName objectName, String attribute)
    throws MBeanException, AttributeNotFoundException, InstanceNotFoundException, ReflectionException
    {
        checkNotNull("objectName", objectName);
        checkNotNull("attribute", attribute);

        testConnector();
        testFailWhenDisconnected();

        Object returnValue = null;

        try
        {
            returnValue = m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), objectName.getCanonicalName(), "getAttribute", new Object[] { objectName, attribute }, new String[] { OBJECTNAME_CLASSNAME, STRING_CLASSNAME });
        }
        catch(MBeanException mbe) { throw mbe; }
        catch(AttributeNotFoundException anfe) { throw anfe; }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(ReflectionException re) { throw re; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed getAttribute");
        }

        return returnValue;
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#getAttributes(ObjectName, String[])
     */
    @Override
    public AttributeList getAttributes(ObjectName objectName, String[] attributes)
    throws InstanceNotFoundException, ReflectionException
    {
        checkNotNull("objectName", objectName);
        checkNotNull("attributes", attributes);

        testConnector();
        testFailWhenDisconnected();

        AttributeList returnValue = null;

        try
        {
            returnValue = (AttributeList)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), objectName.getCanonicalName(), "getAttributes", new Object[] { objectName, attributes }, new String[] { OBJECTNAME_CLASSNAME, STRING_ARRAY_CLASSNAME });
        }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(ReflectionException re) { throw re; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed getAttributes");
        }

        return returnValue;
    }

    /**
     * Supported connector modes: bounded
     *
     * @see javax.management.MBeanServer#getDefaultDomain()
     */
    @Override
    public String getDefaultDomain()
    {
        testOperationSupported();
        testConnector();
        testFailWhenDisconnected();

        String returnValue = null;

        try
        {
            returnValue = (String)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, m_serverIdentity, null, "getDefaultDomain", EMPTY_PARAMS_ARRAY, EMPTY_SIGNATURE_ARRAY);
        }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed getDefaultDomain");
        }

        return returnValue;
    }

    @Override
    public String[] getDomains()
    throws IOException
    {
        testOperationSupported();
        testConnector();
        testFailWhenDisconnected();

        String[] returnValue = null;

        try
        {
            returnValue = (String[])m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, m_serverIdentity, null, "getDomains", EMPTY_PARAMS_ARRAY, EMPTY_SIGNATURE_ARRAY);
        }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed getDomains");
        }

        return returnValue;
    }

    /**
     * Supported connector modes: bounded
     *
     * @see javax.management.MBeanServer#getMBeanCount()
     */
    @Override
    public Integer getMBeanCount()
    {
        testOperationSupported();
        testConnector();
        testFailWhenDisconnected();

        Integer returnValue = null;

        try
        {
            returnValue = (Integer)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, m_serverIdentity, null, "getMBeanCount", EMPTY_PARAMS_ARRAY, EMPTY_SIGNATURE_ARRAY);
        }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed getMBeanCount");
        }

        return returnValue;
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#getMBeanInfo(ObjectName)
     */
    @Override
    public MBeanInfo getMBeanInfo(ObjectName objectName)
    throws InstanceNotFoundException, IntrospectionException, ReflectionException
    {
        checkNotNull("objectName", objectName);

        testConnector();
        testFailWhenDisconnected();

        MBeanInfo returnValue = null;

        try
        {
            returnValue = (MBeanInfo)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), objectName.getCanonicalName(), "getMBeanInfo", new Object[] { objectName }, new String[] { OBJECTNAME_CLASSNAME });
        }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(IntrospectionException ie) { throw ie; }
        catch(ReflectionException re) { throw re; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed getMBeanInfo");
        }

        return returnValue;
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#getObjectInstance(ObjectName)
     */
    @Override
    public ObjectInstance getObjectInstance(ObjectName objectName)
    throws InstanceNotFoundException
    {
        checkNotNull("objectName", objectName);

        testConnector();
        testFailWhenDisconnected();

        ObjectInstance returnValue = null;

        try
        {
            returnValue = (ObjectInstance)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), null, "getObjectInstance", new Object[] { objectName }, new String[] { OBJECTNAME_CLASSNAME });
        }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed getObjectInstance");
        }

        return returnValue;
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#invoke(ObjectName, String, Object[], String[])
     */
    @Override
    public Object invoke(ObjectName objectName, String operationName, Object[] params, String[] signature)
    throws InstanceNotFoundException, MBeanException, ReflectionException
    {
        return invoke(objectName, operationName, params, signature, null);
    }

    /**
     * This method allows remote invocation of the given operations with a particular timeout.
     *
     * Supported connector modes: bounded, unbounded
     *
     * @see #setRequestTimeout(long)
     * @see javax.management.MBeanServer#invoke(ObjectName, String, Object[], String[])
     */
    public Object invoke(ObjectName objectName, String operationName, Object[] params, String[] signature, long timeout)
    throws InstanceNotFoundException, MBeanException, ReflectionException
    {
        return invoke(objectName, operationName, params, signature, timeout, null);
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#invoke(ObjectName, String, Object[], String[])
     */
    public Object invoke(ObjectName objectName, String operationName, Object[] params, String[] signature, ClassLoader loader)
    throws InstanceNotFoundException, MBeanException, ReflectionException
    {
        checkNotNull("objectName", objectName);
        checkNotNull("operationName", operationName);
        checkNotNull("params", params);
        checkNotNull("signature", signature);

        testConnector();
        testFailWhenDisconnected();

        Object returnValue = null;
        ClassLoader existingContextLoader = null;

        try
        {
            if (loader != null)
            {
                existingContextLoader = Thread.currentThread().getContextClassLoader();
                Thread.currentThread().setContextClassLoader(loader);
            }
            returnValue = m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), objectName.getCanonicalName(), "invoke", new Object[] { objectName, operationName, params, signature }, new String[] { OBJECTNAME_CLASSNAME, STRING_CLASSNAME, OBJECT_ARRAY_CLASSNAME, STRING_ARRAY_CLASSNAME });
        }
        catch(MBeanException mbe) { throw mbe; }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(ReflectionException re) { throw re; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed invoke");
        }
        finally
        {
            if (loader != null)
            {
                Thread.currentThread().setContextClassLoader(existingContextLoader);
            }
        }

        return returnValue;
    }

    /**
     * This method allows remote invocation of the given operations with a particular timeout.
     *
     * Supported connector modes: bounded, unbounded
     *
     * @see #setRequestTimeout(long)
     * @see javax.management.MBeanServer#invoke(ObjectName, String, Object[], String[])
     */
    public Object invoke(ObjectName objectName, String operationName, Object[] params, String[] signature, long timeout, ClassLoader loader)
    throws InstanceNotFoundException, MBeanException, ReflectionException
    {
        checkNotNull("objectName", objectName);
        checkNotNull("operationName", operationName);
        checkNotNull("params", params);
        checkNotNull("signature", signature);

        testConnector();
        testFailWhenDisconnected();

        Object returnValue = null;
        ClassLoader existingContextLoader = null;

        try
        {
            if (loader != null)
            {
                existingContextLoader = Thread.currentThread().getContextClassLoader();
                Thread.currentThread().setContextClassLoader(loader);
            }
            returnValue = m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), objectName.getCanonicalName(), "invoke", new Object[] { objectName, operationName, params, signature }, new String[] { OBJECTNAME_CLASSNAME, STRING_CLASSNAME, OBJECT_ARRAY_CLASSNAME, STRING_ARRAY_CLASSNAME }, timeout);
        }
        catch(MBeanException mbe) { throw mbe; }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(ReflectionException re) { throw re; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed invoke");
        }
        finally
        {
            if (loader != null)
            {
                Thread.currentThread().setContextClassLoader(existingContextLoader);
            }
        }

        return returnValue;
    }

    /**
     * @return Returns true if the connector has a live connection to the underlying transport.
     *
     * Gets the connection state of the connector.
     */
    @Override
    public boolean isConnected()
    {
        if (m_connector == null)
        {
            return false;
        }

        try
        {
            return m_connector.isConnected();
        }
        catch(NullPointerException e)
        {
            return false;
        }
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#isInstanceOf(ObjectName, String)
     */
    @Override
    public boolean isInstanceOf(ObjectName objectName, String className)
    throws InstanceNotFoundException
    {
        checkNotNull("objectName", objectName);
        checkNotNull("className", className);

        testConnector();
        testFailWhenDisconnected();

        Boolean returnValue = null;

        try
        {
            returnValue = (Boolean)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), null, "isInstanceOf", new Object[] { objectName, className }, new String[] { OBJECTNAME_CLASSNAME, STRING_CLASSNAME });
        }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed isInstanceOf");
        }

        return returnValue.booleanValue();
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#isRegistered(ObjectName)
     */
    @Override
    public boolean isRegistered(ObjectName objectName)
    {
        checkNotNull("objectName", objectName);

        testConnector();
        testFailWhenDisconnected();

        Boolean returnValue = null;

        try
        {
            returnValue = (Boolean)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), null, "isRegistered", new Object[] { objectName }, new String[] { OBJECTNAME_CLASSNAME });
        }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed isRegistered");
        }

        return returnValue.booleanValue();
    }

    /**
     * Supported connector modes: bounded
     *
     * @see javax.management.MBeanServer#queryMBeans(ObjectName, QueryExp)
     */
    @Override
    public Set queryMBeans(ObjectName objectName, QueryExp query)
    {
        testOperationSupported();
        testConnector();
        testFailWhenDisconnected();

        Set returnValue = null;

        try
        {
            returnValue = (Set)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, m_serverIdentity, null, "queryMBeans", new Object[] { objectName, query }, new String[] { OBJECTNAME_CLASSNAME, QUERYEXP_CLASSNAME });
        }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed queryMBeans");
        }

        return returnValue;
    }

    /**
     * Supported connector modes: bounded
     *
     * @see javax.management.MBeanServer#queryNames(ObjectName, QueryExp)
     */
    @Override
    public Set queryNames(ObjectName objectName, QueryExp query)
    {
        testOperationSupported();
        testConnector();
        testFailWhenDisconnected();

        Set returnValue = null;

        try
        {
            returnValue = (Set)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, m_serverIdentity, null, "queryNames", new Object[] { objectName, query }, new String[] { OBJECTNAME_CLASSNAME, QUERYEXP_CLASSNAME });
        }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed queryNames");
        }

        return returnValue;
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#setAttribute(ObjectName, Attribute)
     */
    @Override
    public void setAttribute(ObjectName objectName, Attribute attribute)
    throws InstanceNotFoundException, AttributeNotFoundException, InvalidAttributeValueException,
           MBeanException, ReflectionException
    {
        checkNotNull("objectName", objectName);
        checkNotNull("attribute", attribute);

        testConnector();
        testFailWhenDisconnected();

        try
        {
            m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), objectName.getCanonicalName(), "setAttribute", new Object[] { objectName, attribute }, new String[] { OBJECTNAME_CLASSNAME, ATTRIBUTE_CLASSNAME });
        }
        catch(MBeanException mbe) { throw mbe; }
        catch(AttributeNotFoundException anfe) { throw anfe; }
        catch(InvalidAttributeValueException iave) { throw iave; }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(ReflectionException re) { throw re; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed setAttribute");
        }
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#setAttributes(ObjectName, AttributeList)
     */
    @Override
    public AttributeList setAttributes(ObjectName objectName, AttributeList attributes)
    throws InstanceNotFoundException, ReflectionException
    {
        checkNotNull("objectName", objectName);
        checkNotNull("attributes", attributes);

        testConnector();
        testFailWhenDisconnected();

        AttributeList returnValue = null;

        try
        {
            returnValue = (AttributeList)m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), objectName.getCanonicalName(), "setAttributes", new Object[] { objectName, attributes }, new String[] { OBJECTNAME_CLASSNAME, ATTRIBUTELIST_CLASSNAME });
        }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(ReflectionException re) { throw re; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed setAttributes");
        }

        return returnValue;
    }

    /**
     * Supported connector modes: bounded
     *
     * @see javax.management.MBeanServer#unregisterMBean(ObjectName objectName)
     */
    @Override
    public void unregisterMBean(ObjectName objectName)
    throws InstanceNotFoundException, MBeanRegistrationException
    {
        checkNotNull("objectName", objectName);

        testOperationSupported();
        testConnector();
        testFailWhenDisconnected();

        try
        {
            m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, m_serverIdentity, null, "unregisterMBean", new Object[] { objectName }, new String[] { OBJECTNAME_CLASSNAME });
        }
        catch(InstanceNotFoundException infe) { throw infe; }
        catch(MBeanRegistrationException mre) { throw mre; }
        catch(Exception e)
        {
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed unregisterMBean");
        }
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#addNotificationListener(ObjectName, NotificationListener, NotificationFilter, Object)
     */
    @Override
    public void addNotificationListener(ObjectName objectName, NotificationListener listener, NotificationFilter filter, Object handback)
    throws InstanceNotFoundException
    {
        addNotificationListener(objectName, listener, filter, handback, m_notificationSubscriptionTimeout);
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#addNotificationListener(ObjectName, NotificationListener, NotificationFilter, Object)
     */
    public void addNotificationListener(ObjectName objectName, NotificationListener listener, NotificationFilter filter, Object handback, long timeout)
    throws InstanceNotFoundException
    {
        checkNotNull("objectName", objectName);
        checkNotNull("listener", listener);

        testConnector();

        String canonicalName = objectName.getCanonicalName();

        synchronized(m_mBeanToLocalListeners)
        {
            HashMap localToRemoteListeners = (HashMap)m_mBeanToLocalListeners.get(canonicalName);
            if (localToRemoteListeners == null)
            {
                localToRemoteListeners = new HashMap();
                m_mBeanToLocalListeners.put(canonicalName, localToRemoteListeners);
            }
            NotificationListenerDelegate delegate = (NotificationListenerDelegate)localToRemoteListeners.get(listener);
            if (delegate == null)
            {
                delegate = new NotificationListenerDelegate(m_connector, listener);
                localToRemoteListeners.put(listener, delegate);
            }
            String handbackID = delegate.addSubscription(filter, handback);
            delegate.setNotificationSubscriptionTimeout(timeout);  //set the timeout to be used for the notification subscription expiration...
            renewSubscription(objectName, delegate, filter, handbackID, timeout, false);
            startSubscriptionRenewalThread(objectName, false);
        }
    }

    /**
     * Unsupported
     *
     * @see javax.management.MBeanServer#addNotificationListener(ObjectName, ObjectName, NotificationFilter, Object)
     */
    @Override
    public void addNotificationListener(ObjectName name, ObjectName listener, NotificationFilter filter, Object handback)
    throws InstanceNotFoundException
    {
        throw new UnsupportedOperationException();
    }

    private void startSubscriptionRenewalThread(ObjectName objectName, boolean renewImmediately)
    {
        String canonicalName = objectName.getCanonicalName();

        synchronized(m_mBeanToLocalListeners)
        {
            SubscriptionRenewalThread subscriptionRenewalThread = m_mBeanToSubscriptionRenewalThreads.get(canonicalName);
            if (subscriptionRenewalThread == null)
            {
                subscriptionRenewalThread = new SubscriptionRenewalThread(objectName, renewImmediately);
                m_mBeanToSubscriptionRenewalThreads.put(canonicalName, subscriptionRenewalThread);
                subscriptionRenewalThread.setDaemon(true);
                subscriptionRenewalThread.start();
            }
        }
    }

    private void restartSubscriptionRenewalThreads()
    {
        synchronized (m_mBeanToLocalListeners)
        {
            // restart all the subscription renewal threads
            Iterator iterator = m_mBeanToLocalListeners.keySet().iterator();
            while (iterator.hasNext())
            {
                try
                {
                    ObjectName objectName = new ObjectName((String)iterator.next());
                    startSubscriptionRenewalThread(objectName, true);
                }
                catch (MalformedObjectNameException e)
                {
                    e.printStackTrace(); // can't not happen
                }
            }
        }
    }

    private void stopSubscriptionRenewalThreads()
    {
        synchronized (JMSConnectorClient.this.m_mBeanToLocalListeners)
        {
            // stop all the subscription renewal threads
            Iterator<SubscriptionRenewalThread> iterator = JMSConnectorClient.this.m_mBeanToSubscriptionRenewalThreads.values().iterator();
            while (iterator.hasNext())
            {
                iterator.next().interrupt();
            }
            JMSConnectorClient.this.m_mBeanToSubscriptionRenewalThreads.clear();
        }
    }

    private boolean renewSubscriptions(ObjectName objectName, boolean renewImmediately, Object threadLockObject)
    {
        synchronized (threadLockObject)
        {
            try
            {
                if (!m_mBeanToSubscriptionRenewalThreads.containsValue(Thread.currentThread()))
                {
                    return false;
                }
                long timeout = renewImmediately ? 1000 : m_notificationSubscriptionRenewalInterval;
                long startTime = System.currentTimeMillis();
                m_needsRenewal = false;
                if (timeout <= 0)
                {
                    while (!m_needsRenewal)
                    {
                        threadLockObject.wait();
                    }
                }
                else
                {
                    long waitTime = timeout;
                    while (waitTime > 0 && !m_needsRenewal)
                    {
                        threadLockObject.wait(waitTime);
                        waitTime = timeout - (System.currentTimeMillis() - startTime);
                    }
                }
            }
            catch (InterruptedException e)
            {
                return false;
            }
        }
        String canonicalName = objectName.getCanonicalName();

        HashMap localToRemoteListeners = null;
        Object[] entries = null;

        synchronized (m_mBeanToLocalListeners)
        {
            if (m_isClosing)
            {
                return false;
            }
            if (!m_mBeanToSubscriptionRenewalThreads.containsValue(Thread.currentThread()))
            {
                return false;
            }
            localToRemoteListeners = (HashMap)m_mBeanToLocalListeners.get(canonicalName);
            if (localToRemoteListeners == null)
            {
                m_mBeanToSubscriptionRenewalThreads.remove(canonicalName);
                return false; // removed
            }
            entries = localToRemoteListeners.entrySet().toArray();
        }

        for (int i = 0;i < entries.length;i++)
        {
            Map.Entry entry = (Map.Entry)entries[i];
            Object localListener = entry.getKey();
            NotificationListenerDelegate remoteListener = (NotificationListenerDelegate)entry.getValue();

            ArrayList subscriptionRenewalPairs = remoteListener.getSubscriptionRenewalPairs();
            Iterator iterator = subscriptionRenewalPairs.iterator();
            while (iterator.hasNext())
            {
                Object[] subscriptionRenewalPair = (Object[])iterator.next();
                // for each renewal make best attempt to check the renewal attempt should still occur
                if (!m_mBeanToSubscriptionRenewalThreads.containsValue(Thread.currentThread()))
                {
                    return false;
                }
                if (Thread.currentThread().isInterrupted())
                {
                    return true;
                }
                if (m_isClosing)
                {
                    return false;
                }
                if (!m_mBeanToLocalListeners.containsKey(canonicalName))
                {
                    return true;
                }
                if (!localToRemoteListeners.containsKey(localListener))
                {
                    continue;
                }
                renewSubscription(objectName, remoteListener, (NotificationFilter)subscriptionRenewalPair[0], (String)subscriptionRenewalPair[1], m_notificationSubscriptionTimeout, true);
            }
        }

        return true;
    }

    private void renewSubscription(ObjectName objectName, NotificationListenerDelegate delegate, NotificationFilter filter, String handbackID, long notificationSubscriptionTimeout, boolean renewal)
    {
        // NOTE: the server side of the connector has special handling for the "addNotificationListener"
        //       operation as it needs to create its own proxy listener
        StringBuffer sb = new StringBuffer();
        sb.append(objectName.getCanonicalName().hashCode()).append('.');
        sb.append(delegate.hashCode()).append('.');
        sb.append(filter == null ? 0 : delegate.hashCode()).append('.');
        sb.append(handbackID == null ? 0 : handbackID.hashCode());
        try
        {
            if (!m_isClosing)
            {
                if (!m_connector.isConnected())
                {
                    throw new InvokeTimeoutCommsException("Not connected");
                }
                Object[] params = new Object[]
                {
                    objectName,
                    delegate,
                    filter,
                    buildCompoundHandbackID(filter, handbackID),
                    new Long(notificationSubscriptionTimeout)
                };
                if (m_useOnewaySubscriptionRequests)
                {
                    m_connector.invokeOneway(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), objectName.getCanonicalName(), ADD_NOTIFICATION_LISTENER_METHOD_NAME, params, ADDREMOVE_NOTIFICATION_LISTENER_WITH_TIMEOUT_SIGNATURE,0); // supply notification subscription timeout value...
                }
                else {
                    m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), objectName.getCanonicalName(), ADD_NOTIFICATION_LISTENER_METHOD_NAME, params, ADDREMOVE_NOTIFICATION_LISTENER_WITH_TIMEOUT_SIGNATURE,0); // supply notification subscription timeout value...
                }
            }
            m_renewalFailures.remove(sb.toString());
        }
        catch(Exception e)
        {
            if (m_isClosing)
            {
                return;
            }
            if (m_renewalFailures.contains(sb.toString()))
             {
                return; // since its already been reported
            }
            m_renewalFailures.add(sb.toString());
            if (m_connectionListener != null)
            {
                m_connectionListener.onNotificationListenerRenewalFailure(e);
            }
        }
    }

    /**
     * A compound handback ID is a string that encapsulates both an identifier for the filter and original
     * user's handback object. This method creates a tokenized string of the form:
     * <p>
     *   [delimiter][delimiter][filter instance hashcode][delimiter][handbackID]
     * <p>
     * The leading 2 delimiters allow consumers of the compound ID to identify that this is actually
     * a compound, rather than a regular handback object that happens to be a string, and the subsequent
     * delimiter allows the actual handback identifier to be stripped from the compound ID (this is
     * done in ContainerImpl).
     * <p>
     * To make it very, very unliklely that a MGMT 2.0 JMX client could provide a handback object that
     * was a string of a similar format, the delimiter used is the bell character (char)8.
     * <p>
     * This method builds the compound ID using the given filiter and handback identity (an identity
     * already established for the clients original handback object). It is allowed that no filter be
     * supplied.
     */
    private String buildCompoundHandbackID(NotificationFilter filter, String handbackID)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(COMPOUND_HANDBACK_ID_DELIMITER);
        sb.append(COMPOUND_HANDBACK_ID_DELIMITER);
        sb.append(filter == null ? 0 : filter.hashCode());
        sb.append(COMPOUND_HANDBACK_ID_DELIMITER);
        sb.append(handbackID);

        return sb.toString();
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#registerRetryCallback(IRetryCallback)
     */
    public void registerRetryCallback(IRetryCallback rcb)
    {
        this.m_connector.registerRetryCallback(rcb);
    }

    /**
     * @see com.sonicsw.mf.comm.IConnectorClient#deregisterRetryCallback()
     */
    public void deregisterRetryCallback()
    {
        this.m_connector.deregisterRetryCallback();
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#removeNotificationListener(ObjectName, NotificationListener)
     */
    @Override
    public void removeNotificationListener(ObjectName objectName, NotificationListener listener)
    throws InstanceNotFoundException, ListenerNotFoundException
    {
        checkNotNull("objectName", objectName);
        checkNotNull("listener", listener);

        testConnector();

        String canonicalName = objectName.getCanonicalName();

        NotificationListenerDelegate delegate = null;
        synchronized(m_mBeanToLocalListeners)
        {
            SubscriptionRenewalThread subscriptionRenewalThread = m_mBeanToSubscriptionRenewalThreads.get(canonicalName);
            if (subscriptionRenewalThread != null)
            {
                synchronized(subscriptionRenewalThread.getThreadLockObject())
                {
                    m_needsRenewal = true;
                    subscriptionRenewalThread.getThreadLockObject().notifyAll();
                }
            }

            HashMap localToRemoteListeners = (HashMap)m_mBeanToLocalListeners.get(canonicalName);
            if (localToRemoteListeners == null)
             {
                return; // already removed
            }

            delegate = (NotificationListenerDelegate)localToRemoteListeners.remove(listener);
            if (delegate == null)
             {
                return; // already removed
            }

            // cleanup if no more listeners for this object name
            if (localToRemoteListeners.isEmpty())
            {
                m_mBeanToLocalListeners.remove(canonicalName);
                synchronized(m_notificationSourceUsages)
                {
                    m_notificationSourceUsages.remove(canonicalName);
                }
            }
        }

        // cleanup the delegate
        delegate.close();

        // now unregister interest in the notifications via invoke
        // NOTE: the server side of the connector has special handling for the "removeNotificationListener"
        //       operation as it needs to create its own proxy listener
        try
        {
            m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), objectName.getCanonicalName(), "removeNotificationListener",
                               new Object[] { objectName, delegate },
                               new String[] { OBJECTNAME_CLASSNAME, NOTIFICATIONLISTENER_CLASSNAME });
        }
        catch(InstanceNotFoundException e)
        {
            if (m_isClosing)
            {
                return;
            }
            throw e;
        }
        catch(ListenerNotFoundException e)
        {
            if (m_isClosing)
            {
                return;
            }
            throw e;
        }
        catch(Exception e)
        {
            if (m_isClosing)
            {
                return;
            }
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed removeNotificationListener");
        }
    }

    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#removeNotificationListener(ObjectName, ObjectName)
     */
    @Override
    public void removeNotificationListener(ObjectName objectName, ObjectName listener)
    throws InstanceNotFoundException, ListenerNotFoundException
    {
        throw new UnsupportedOperationException();
    }
    
    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#removeNotificationListener(ObjectName, NotificationListener, NotificationFilter, Object)
     */
    @Override
    public void removeNotificationListener(ObjectName objectName, NotificationListener listener, NotificationFilter filter, Object handback)
    throws InstanceNotFoundException, ListenerNotFoundException
    {
        checkNotNull("objectName", objectName);
        checkNotNull("listener", listener);

        testConnector();

        String canonicalName = objectName.getCanonicalName();

        String handbackID = null;

        NotificationListenerDelegate delegate = null;
        synchronized(m_mBeanToLocalListeners)
        {
            SubscriptionRenewalThread subscriptionRenewalThread = m_mBeanToSubscriptionRenewalThreads.get(canonicalName);
            if (subscriptionRenewalThread != null)
            {
                synchronized(subscriptionRenewalThread.getThreadLockObject())
                {
                    m_needsRenewal = true;
                    subscriptionRenewalThread.getThreadLockObject().notifyAll();
                }
            }

            HashMap localToRemoteListeners = (HashMap)m_mBeanToLocalListeners.get(canonicalName);
            if (localToRemoteListeners == null)
             {
                return; // already removed
            }

            delegate = (NotificationListenerDelegate)localToRemoteListeners.get(listener);
            if (delegate == null)
             {
                return; // already removed
            }

            handbackID = delegate.removeSubscription(filter, handback);

            // if the delegate has no more renewal pairs then we can dispose of it
            if (delegate.getSubscriptionRenewalPairs().size() == 0)
            {
                delegate.close();
                localToRemoteListeners.remove(listener);
            }

            // cleanup if no more listeners for this object name
            if (localToRemoteListeners.isEmpty())
            {
                m_mBeanToLocalListeners.remove(canonicalName);
                synchronized(m_notificationSourceUsages)
                {
                    m_notificationSourceUsages.remove(canonicalName);
                }
            }
        }

        // now unregister interest in the notifications via invoke
        // NOTE: the server side of the connector has special handling for the "removeNotificationListener"
        //       operation as it needs to create its own proxy listener
        try
        {
            if (handbackID != null)
            {
                m_connector.invoke(JMSConstants.JMX_COMMS_TYPE, getMFNamespace(objectName), objectName.getCanonicalName(), "removeNotificationListener",
                                   new Object[] { objectName, delegate, filter, buildCompoundHandbackID(filter, handbackID) },
                                   ADDREMOVE_NOTIFICATION_LISTENER_SIGNATURE);
            }
        }
        catch(InstanceNotFoundException e)
        {
            if (m_isClosing)
            {
                return;
            }
            throw e;
        }
        catch(ListenerNotFoundException e)
        {
            if (m_isClosing)
            {
                return;
            }
            throw e;
        }
        catch(Exception e)
        {
            if (m_isClosing)
            {
                return;
            }
            if (e instanceof RuntimeException)
            {
                throw (RuntimeException)e;
            }
            throw CommunicationException.create(e, "Failed removeNotificationListener");
        }
    }
    
    /**
     * Supported connector modes: bounded, unbounded
     *
     * @see javax.management.MBeanServer#removeNotificationListener(ObjectName, ObjectName, NotificationFilter, Object)
     */
    @Override
    public void removeNotificationListener(ObjectName objectName, ObjectName listener, NotificationFilter filter, Object handback)
    throws InstanceNotFoundException, ListenerNotFoundException
    {
        throw new UnsupportedOperationException();
    }

    /**
     * Sets System.out runtime debug tracing mask. Values are as per JMS CONNECTOR
     * values.
     */
    public void setTraceMask(int maskValue) { m_connector.setTraceMask(maskValue); }

    /**
     * Gets System.out runtime debug tracing mask. Values are as per JMS CONNECTOR
     * values.
     */
    public int getTraceMask() { return m_connector.getTraceMask(); }

    /**
     * Sets the connect timeout in milliseconds.
     * This limits the total time allowed per JMS connection (or FT reconnect) attempt.
     * The connection/reconnect attempt may involve several URLs if multiple connection URLs are provided.
     * Subsequent connection attempts may be made if the timeout passed to connect() allows for this.
     * 
     * @see #setSocketConnectTimeout(long)
     * @see #connect(JMSConnectorAddress, long)
     */
    public void setConnectTimeout(long milliseconds)
    {
        m_connectTimeout = (milliseconds >= ConnectorClient.CONNECT_TIMEOUT_MINIMUM) ? milliseconds : ConnectorClient.CONNECT_TIMEOUT_MINIMUM;

        if (m_connector != null)
        {
            m_connector.setConnectTimeout(m_connectTimeout);
        }
    }

    /**
     * Gets the connect timeout in milliseconds.
     * 
     * @see #setConnectTimeout(long)
     */
    public long getConnectTimeout()
    {
        return m_connectTimeout;
    }

    /**
     * Sets the socket connect timeout in milliseconds.
     * This limits the connection time spent per-URL when multiple connection URLs are provided.
     * 
     * @see #setConnectTimeout(long)
     */
    public void setSocketConnectTimeout(long milliseconds)
    {
        m_socketConnectTimeout = (milliseconds >= ConnectorClient.SOCKET_CONNECT_TIMEOUT_MINIMUM) ? milliseconds : ConnectorClient.SOCKET_CONNECT_TIMEOUT_MINIMUM;

        if (m_connector != null)
        {
            m_connector.setSocketConnectTimeout(m_socketConnectTimeout);
        }
    }

    /**
     * Gets the socket connect timeout in milliseconds.
     * 
     * @see #setSocketConnectTimeout(long)
     */
    public long getSocketConnectTimeout()
    {
        return m_socketConnectTimeout;
    }
    
    /**
     * By default JMX notification subscriptions (and renewals) are made using a request/reply
     * mechanism that ensures successful is made or the user is notified via a registered callback.
     * <p>
     * By using a oneway subscription request, the client will never be notified that the subscription
     * request failed. This is useful if there are a large volume of notification sources to which a
     * subsciption has been made and such sources are often not running when the subscription is made
     * or renewed.
     * 
     * @param useOneway When true, oneway notification subscription requests will be made.
     * 
     * @see com.sonicsw.mf.jmx.client.IConnectionListener#onNotificationListenerRenewalFailure(Exception)
     */
    public void setUseOnewaySubscriptionRequests(boolean useOneway)
    {
        m_useOnewaySubscriptionRequests = useOneway;
    }

    /**
     * By default JMX notification subscriptions (and renewals) are made using a request/reply
     * mechanism that ensures successful is made or the user is notified via a registered callback.
     * <p>
     * By using a oneway subscription request, the client will never be notified that the subscription
     * request failed. This is useful if there are a large volume of notification sources to which a
     * subsciption has been made and such sources are often not running when the subscription is made
     * or renewed.
     * 
     * @return When true, oneway notification subscription requests will be made.
     * 
     * @see com.sonicsw.mf.jmx.client.IConnectionListener#onNotificationListenerRenewalFailure(Exception)
     */
    public boolean getUseOnewaySubscriptionRequests()
    {
        return m_useOnewaySubscriptionRequests;
    }

    private void checkNotNull(String paramName, Object object)
    {
        if (object == null)
        {
            IllegalArgumentException e = new IllegalArgumentException(paramName + " must not be null");
            throw new RuntimeOperationsException(e);
        }
    }

    private final class SubscriptionRenewalThread
    extends Thread
    {
        private ObjectName objectName;
        private boolean renewImmediately;
        private Object threadLockObj = new Object();

        private SubscriptionRenewalThread(ObjectName objectName, boolean renewImmediately)
        {
            super("Notification Subscription Renewer - " + objectName.getCanonicalName());
            this.objectName = objectName;
            this.renewImmediately = renewImmediately;
        }

        @Override
        public void run()
        {
            while (JMSConnectorClient.this.renewSubscriptions(this.objectName, this.renewImmediately, threadLockObj))
            {
                this.renewImmediately = false;
            }
        }
        
        public Object getThreadLockObject() 
        {
            return threadLockObj;
        }
    }

    private void testOperationSupported()
    {
        if (m_serverIdentity == null)
        {
            throw new JMRuntimeException("Operation unsupported for unbounded client connector.");
        }
    }

    private void testConnector()
    {
        if (m_connector == null)
        {
            throw new java.lang.IllegalStateException("Not connected");
        }
    }

    private void testFailWhenDisconnected()
    {
        if (m_failWhenDisconnected && !isConnected())
        {
            throw CommunicationException.create("Not currently connected");
        }
    }
    
    private String getMFNamespace(ObjectName objectName)
    {
        String namespace = ObjectNameHelper.getMFNamespace(objectName);
        
        if (namespace == null)
        {
            namespace = m_serverIdentity;
        }
        
        return namespace;
    }
}
