package com.sonicsw.mf.jmx.client;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.management.MalformedObjectNameException;
import javax.management.Notification;
import javax.management.ObjectName;

import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.runtime.IComponentIdentity;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.common.runtime.Level;
import com.sonicsw.mf.common.runtime.impl.ComponentIdentity;

public class MFNotification
extends Notification
implements INotification
{
    private static final long serialVersionUID = 6407583935938698016L;
    private static final short m_serialVersion = 0;

    private short m_category;
    private String m_subCategory;
    private String m_eventName;
    private int m_severityLevel;
    private short m_logType;
    private IComponentIdentity m_sourceIdentity;
    public static final String HOST = getHostIP();
    private String m_sourceHost;
    private HashMap m_attributes;

    private static String getHostIP() {
        try
        {
            return InetAddress.getLocalHost().getHostName();
        }
        catch(UnknownHostException e)
        {
            return "127.0.0.1"; // NOSONAR (MUTE SONAR RULE "IP addresses should not be hardcoded")
        }
    }

    /**
     * Components should not directly reference MFNotification. This method should
     * only be used by the current SMC.
     *
     * @deprecated
     */
    public static String getType(String[] notificationTypes)
    {
        StringBuffer sb = new StringBuffer(notificationTypes[0]);
        if (notificationTypes.length > 1)
        {
            for (int i = 1; i < notificationTypes.length; i++)
            {
                sb.append('.').append(notificationTypes[i]);
            }
        }

        return sb.toString();
    }

    /**
     * Constructor to create a generic MF notification. Such notifications should be created via the
     * NotificationFactory.
     *
     * @param source        The source of the notification.
     * @param category      Each notification belongs to a major category. Permissible values are
     *                      defined in INotification (see <>_CATEGORY constants).
     * @param subCategory   Each notification has a minor category.
     * @param eventName     Each notification type is unique by category/sub category/event name. Event names
     *                      are descriptive of what event generated the notification.
     * @param severityLevel Each notification has a severity associated with it. Severities are used by
     *                      client applications to prioritize multiple notifications. Additionally the severity
     *                      of certain notifications will be used to derive an aggregated state for a container.
     *                      Permissible values are defined in INotification (see <>_SEVERITY constants).
     *
     * @see com.sonicsw.mf.common.NotificationFactory#createNotification(short, String, String, short)
     */
    public MFNotification(ObjectName source, short category, String subCategory, String eventName, int severityLevel)
    {
        super(INotification.CATEGORY_TEXT[category] + '.' + subCategory + '.' + eventName, source, 0);
        switch(severityLevel)
        {
            case Level.SEVERE:
                m_logType = INotification.ERROR_TYPE;
                break;
            case Level.WARNING:
                m_logType = INotification.WARNING_TYPE;
                break;
            default:
                m_logType = INotification.INFORMATION_TYPE;
                break;
        }

        m_category = category;
        m_subCategory = subCategory;
        m_eventName = eventName;
        m_severityLevel = severityLevel;
        m_sourceHost = HOST;
    }
    
    //
    // Copy constructor (safer than "clone" method)
    //
    public MFNotification(MFNotification notification)
    {
        super(INotification.CATEGORY_TEXT[notification.getCategory()] + '.' + notification.getSubCategory() + '.' + notification.getEventName(), (ObjectName)notification.getSource(), 0);

        switch(notification.getSeverity())
        {
            case Level.SEVERE:
                m_logType = INotification.ERROR_TYPE;
                break;
            case Level.WARNING:
                m_logType = INotification.WARNING_TYPE;
                break;
            default:
                m_logType = INotification.INFORMATION_TYPE;
                break;
        }

        m_category = notification.getCategory();
        m_subCategory = notification.getSubCategory();
        m_eventName = notification.getEventName();
        m_severityLevel = notification.getSeverity();
        m_sourceHost = notification.getSourceHost();
        
        // It is sufficient to use a direct reference to the original notification's
        // IComponentIdentity object, as the sole concrete implementation, ComponentIdentity, 
        // is a "final" class.  Just to be safe, however, an "instanceof" check will be made here
        // to be certain that the source identity of the original notification is an
        // instance of "ComponentIdentity" before we set the clone's m_sourceIdentity reference.
        // TODO: how to properly handle case where a reference to an object that implements
        //       IComponentIdentity but is not an instance of the concrete ComponentIdentity class
        //       get returned from the original notification's "getSourceIdentity()"?
        IComponentIdentity cid = notification.getSourceIdentity();


        if (cid != null)
        { 
            if (cid instanceof ComponentIdentity)
            {
                m_sourceIdentity = cid;
            }
            else
            {
                throw new MFRuntimeException("Attempt to set notification source ID reference of cloned notification to something other than an instance of ComponentIdentity");
            }
        }

        setSequenceNumber(notification);

        // copy the attributes
        HashMap attributes = notification.getAttributes();
        if (attributes != null)
        {
            // the attributes HashMap (for the cloned copy) is only created when necessary
            if (m_attributes == null)
            {
                m_attributes = new HashMap();
            }
            
            Set entries = attributes.entrySet();
            if (entries != null)
            {
                Iterator it = entries.iterator();
                while (it.hasNext())
                {
                    Map.Entry entry = (Map.Entry) it.next();
                    if (entry != null)
                    {
                        Object key = entry.getKey();
                        Object value = entry.getValue();
                        m_attributes.put(key,value);
                    }
                }
            }
        }
    }

    private void setSequenceNumber(MFNotification notification) {
        // set the sequence number of the cloned instance, based on that of the original notification....
        super.setSequenceNumber(notification.getSequenceNumber());
    }
    
    // Need to override getType()
    @Override
    public String getType() { return INotification.CATEGORY_TEXT[m_category] + '.' + m_subCategory + '.' + m_eventName; }

    //
    // INotification implementation
    //

    /**
     * @see com.sonicsw.mf.common.INotification#getCategory()
     */
    @Override
    public short getCategory() { return m_category; }

    /**
     * @see com.sonicsw.mf.common.INotification#subCategory()
     */
    @Override
    public String getSubCategory() { return m_subCategory; }

    /**
     * @see com.sonicsw.mf.common.INotification#getEventName()
     */
    @Override
    public String getEventName() { return m_eventName; }

    /**
     * @see com.sonicsw.mf.common.INotification#getSeverity()
     */
    @Override
    public int getSeverity() { return m_severityLevel; }

    /**
     * @see com.sonicsw.mf.common.INotification#setLogType(short)
     */
    @Override
    public void setLogType(short logType)
    {
        if (m_logType < 0 || m_logType > 4)
        {
            throw new IllegalArgumentException();
        }

        m_logType = logType;
    }

    /**
     * @see com.sonicsw.mf.common.INotification#getLogType()
     */
    @Override
    public short getLogType() { return m_logType; }

    /**
     * @see javax.management.Notification#getSource()
     */
    @Override
    public Object getSource()
    {
        // we have to override the getSource() of the JMX Notification as forwarded notifications
        // will have had their source reset to the forwarder - yet we should still return the
        // original source here
        ObjectName source = null;
        try
        {
            source = new ObjectName(m_sourceIdentity.getCanonicalName());
        }
        catch(MalformedObjectNameException mone) { mone.printStackTrace(); } // should never happen
        return source;
    }

    /**
     * @see com.sonicsw.mf.common.INotification#getSourceIdentity()
     */
    @Override
    public IComponentIdentity getSourceIdentity() { return m_sourceIdentity; }

    /**
     * @see com.sonicsw.mf.common.INotification#setSourceIdentity(IComponentIdentity)
     */
    @Override
    public void setSourceIdentity(IComponentIdentity sourceIdentity)
    {
        m_sourceIdentity = sourceIdentity;
    }

    /**
     * @see com.sonicsw.mf.common.INotification#getSourceHost()
     */
    @Override
    public String getSourceHost() { return m_sourceHost; }

    /**
     * @see com.sonicsw.mf.common.INotification#getSequenceNumber()
     */
    @Override
    public long getSequenceNumber() { return super.getSequenceNumber(); }

    /**
     * @see com.sonicsw.mf.common.INotification#getTimeStamp()
     */
    @Override
    public long getTimeStamp() { return super.getTimeStamp(); }

    /**
     * @see com.sonicsw.mf.common.INotification#setAttribute(String, Object)
     */
    @Override
    public synchronized void setAttribute(String name, Object value)
    {
        // we only create it when needed
        if (m_attributes == null)
        {
            m_attributes = new HashMap();
        }

        m_attributes.put(name, value);
    }

    /**
     * @see com.sonicsw.mf.common.INotification#getAttributes()
     */
    @Override
    public synchronized HashMap getAttributes() 
    {
        HashMap copy = null;
        if (m_attributes != null)
        {
            copy = (HashMap) m_attributes.clone();
        }
        return copy; 
    }
    
    //
    // Serialization
    //

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        // we know how many fields we will write
        stream.writeInt(12);

        // MFNotification specific fields

        stream.writeUTF("serialVersion");
        stream.writeObject(new Short(m_serialVersion));

        stream.writeUTF("category");
        stream.writeObject(new Short(m_category));

        stream.writeUTF("subCategory");
        stream.writeObject(m_subCategory);

        stream.writeUTF("eventName");
        stream.writeObject(m_eventName);

        stream.writeUTF("severityLevel");
        stream.writeObject(new Integer(m_severityLevel));

        stream.writeUTF("logType");
        stream.writeObject(new Short(m_logType));

        stream.writeUTF("sourceIdentity");
        stream.writeObject(m_sourceIdentity);

        stream.writeUTF("sourceHost");
        stream.writeObject(m_sourceHost);

        stream.writeUTF("attributes");
        
        
        synchronized(this)
        {
            // in order to avoid the possibility of a ConcurrentModificationException,
            // which can realistically happen, need to synchronize on the notification
            // while writing the attributes hashmap to the stream...
            stream.writeObject(m_attributes);
        }
        
        // JMX Notification specific fields

        stream.writeUTF("source");
        stream.writeObject(super.getSource());

        stream.writeUTF("sequenceNumber");
        stream.writeObject(new Long(super.getSequenceNumber()));

        stream.writeUTF("timeStamp");
        stream.writeObject(new Long(super.getTimeStamp()));

        // add JMX message and user data if required
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        int numValues = stream.readInt();
        HashMap map = new HashMap(numValues);
        for (int i = 0; i < numValues; i++)
        {
            map.put(stream.readUTF(), stream.readObject());
        }

        // now map them back into the local variables based on the version
        // Note: As a general rule need to catch exceptions and either do something to set a default
        //       value or ignore

        switch(((Short)map.get("serialVersion")).shortValue())
        {
            // case olderVersion<n> ...
            // case olderVersion<n> ...
            default: // the current version or newer versions
            {
                // MFNotification fields
                try { m_category = ((Short)map.get("category")).shortValue(); } catch(Exception e) {}
                try { m_subCategory = (String)map.get("subCategory"); } catch(Exception e) {}
                try { m_eventName = (String)map.get("eventName"); } catch(Exception e) {}
                try { m_severityLevel = ((Integer)map.get("severityLevel")).intValue(); } catch(Exception e) {}
                try { m_logType = ((Short)map.get("logType")).shortValue(); } catch(Exception e) {}
                try { m_sourceIdentity = (IComponentIdentity)map.get("sourceIdentity"); } catch(Exception e) {}
                try { m_sourceHost = (String)map.get("sourceHost"); } catch(Exception e) {}
                try { m_attributes = (HashMap)map.get("attributes"); } catch(Exception e) {}

                // JMX Notification specific fields
                try { super.setSource(map.get("source")); } catch(Exception e) {}
                try { super.setSequenceNumber(((Long)map.get("sequenceNumber")).longValue()); } catch(Exception e) {}
                try { super.setTimeStamp(((Long)map.get("timeStamp")).longValue()); } catch(Exception e) {}
                break;
            }
        }
    }

}
